/*!****************************************************************************
 *! @file sensor_data_logging.c                                               *
 *!                                                                           *
 *! @brief Logging of received GNSS and PoS data from SCC.                    *
 *!                                                                           *
 *! @author Sanjay Gurugubelli(RBEI/ECF1)                                     *
 *!                                                                           *
 *! @date 2018-Jan-18 sga5kor  initial version.                               *
 *!                                                                           *
 *! Copyright (c) Robert Bosch GmbH, 2018. All rights reserved.               *
 *!****************************************************************************
 */

/*!****************************************************************************
 *!                             header files                                  *
 *!****************************************************************************
 */
#define OSAL_S_IMPORT_INTERFACE_TYPES
#define OSAL_S_IMPORT_INTERFACE_THREADING
#include "osal_if.h"
#include "osansi.h"
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/statvfs.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include "dev_gnss_types.h"
#include "sensor_dispatcher_types.h"
#include "sensor_data_logging.h"

/*!****************************************************************************
 *!                           macro definitions                               *
 *!****************************************************************************
 */
#define POSGNSS_POS_DATA_LOG_FILE_NAME      \
                                   "/var/opt/bosch/dynamic/pos_data.log"

#define POSGNSS_GNSS_DATA_LOG_FILE_NAME     \
                                   "/var/opt/bosch/dynamic/gnss_data.log"

#define POSGNSS_DATA_LOG_TRIGGER_FILE_NAME  \
                                   "/var/opt/bosch/dynamic/log_sensor_data.cfg"
                                
#define POSGNSS_DISK_PARTATION               \
                                   "/var/opt/bosch/dynamic/"

#define POSGNSS_STRING_MAX_FILE_SIZE        "MAX_SIZE"

/*!****************************************************************************
 *!                             derived types                                 *
 *!****************************************************************************
 */

typedef struct
{
   
   int   GnssLogFileFd;       /*! gnss log file descriptor */
   int   PosLogFileFd;        /*! pos log file descriptor  */
   tU32  u32GnssLogFileSize;  /*! current size of gnss log file */
   tU32  u32PosLogFileSize;   /*! current size of pos log file  */
   tU32  u32MaxLogFileSize;   /*! max log file size of either gnss or pos   */
   tBool bIsInitDone;         /*! flag to indicate status of initialization */
}PosGnss_LogType;

static PosGnss_LogType rLog = { .GnssLogFileFd      = 0,
                                .PosLogFileFd       = 0,
                                .u32GnssLogFileSize = 0,
                                .u32PosLogFileSize  = 0,
                                .u32MaxLogFileSize  = 0,
                                .bIsInitDone        = FALSE};

/*!****************************************************************************
 *!                            global variables                               *
 *!****************************************************************************
 */
/*! flag to represent logging status */
tBool PosGnss_IsDataLoggingEnabled = FALSE;

/*!****************************************************************************
 *!                          external functions                               *
 *!****************************************************************************
 */
extern tVoid SenDisp_vTraceOut( TR_tenTraceLevel enTraceLevel,
                                          const tChar *pcFormatString,...);

/*!****************************************************************************
 *!                          global functions                                 *
 *!****************************************************************************
 */

/*!****************************************************************************
 *! @fn      : PosGnss_WritePosDataToLogFile()                                *
 *! @brief   : writes received data to pos data log file.                     *
 *! @param   : data_ptr : pointer to the data which is to be written.         *
 *!            size : size of the data                                        *
 *! @return  : void                                                           *
 *!****************************************************************************
 */
void PosGnss_WritePosDataToLogFile(const unsigned char* data_ptr,
                             unsigned int size, PoSGnss_enDataDirection enDir)
{
   static char data_msg[SEN_DISP_MAX_PACKET_SIZE];
   const unsigned char *end_ptr = data_ptr + size;
   int iter = 0,bytes, append_new_line = 0;

   if (!data_ptr ||
       -1 == rLog.PosLogFileFd ||
       rLog.u32PosLogFileSize >= rLog.u32MaxLogFileSize)
   {
      return;
   }

   memset(data_msg,'\0', sizeof(data_msg));

   if (MSG_ID_SCC_SENSOR_R_DATA == data_ptr[SEN_DISP_OFFSET_MSG_ID])
   {
      iter = sprintf(data_msg, "%s%s",
                enDir==POSGNSS_EN_DATA_IN?"\n[IN]":"\n[OUT]","[DATA_MSG]\n" );
   }
   else
   {
      iter = sprintf(data_msg, "%s%s",
            enDir==POSGNSS_EN_DATA_IN?"\n[IN]":"\n[OUT]","[NON_DATA_MSG]\n" );
   }

   while (data_ptr <= end_ptr && iter+5 < (int)sizeof(data_msg))
   {
      /*! tens place */
      data_msg[iter] = *data_ptr/16;
      data_msg[iter] = data_msg[iter]<10?data_msg[iter]+'0':data_msg[iter]+55;
      iter++;
      /*! ones place */
      data_msg[iter] = *data_ptr%16;
      data_msg[iter] = data_msg[iter]<10?data_msg[iter]+'0':data_msg[iter]+55;
      iter++;
      data_ptr++;

      if( ++append_new_line%40 == 0 )
      {
         data_msg[iter] = '\n';
      }
      else
      {
         data_msg[iter] = ' ';
      }
      iter++;
   }

   data_msg[iter++] = '\n';
   data_msg[iter] = '\0';

   bytes = strlen(data_msg);

   if (bytes != write(rLog.PosLogFileFd,(const void*)data_msg, bytes))
   {
      SenDisp_vTraceOut(TR_LEVEL_ERRORS,
                  "writing time stamp to file failed errno %d", errno );
   }
}

/*!****************************************************************************
 *! @fn      : PosGnss_WriteGnssDataToLogFile()                               *
 *! @brief   : Writes the received data to gnss data log file.                *
 *! @param   : data_ptr : pointer to the data which is to be written.         *
 *!            size : size of the data                                        *
 *! @return  : void                                                           *
 *!****************************************************************************
 */
void PosGnss_WriteGnssDataToLogFile(const unsigned char* data_ptr,
                              unsigned int size,PoSGnss_enDataDirection enDir)
{
   static char buf[2048] = {0};
   unsigned int ts   = 0;
   char ts_ascii[32] = {0};
   int  bytes;

   /*! sanity check */
   if( !data_ptr ||
       -1 == rLog.GnssLogFileFd ||
       rLog.u32GnssLogFileSize >= rLog.u32MaxLogFileSize )
   {
      return;
   }

   memset(buf,0,sizeof(buf));

   /*! direction of data */
   (void)strcat(buf, enDir==POSGNSS_EN_DATA_IN?"\n[IN]":"\n[OUT]");

   /*! concatenate message type? */
   if (GNSS_PROXY_SCC_R_DATA_START_MSGID == data_ptr[GNSS_PROXY_OFFSET_MSG_ID])
   {
      strcat(buf,"[DATA_MSG]");
      /*! time stamp starts from index 1*/
      ts |= (unsigned int)data_ptr[1] << 0;
      ts |= (unsigned int)data_ptr[2] << 8;
      ts |= (unsigned int)data_ptr[3] << 16;
      ts |= (unsigned int)data_ptr[4] << 24;

      (void)sprintf(ts_ascii, "[TIMESTAMP-%u]\n", ts);
      (void)strcat(buf,ts_ascii);
   }
   else
   {
      strcat(buf,"[NON_DATA_MSG]\n");
   }

   bytes = strlen((const char*)buf);
   if (bytes != write(rLog.GnssLogFileFd,(const void*)buf,(unsigned int)bytes))
   {
      SenDisp_vTraceOut(TR_LEVEL_ERRORS,
      "updating message info to file failed");
   }

   /*! is it data message? */
   if (GNSS_PROXY_SCC_R_DATA_START_MSGID == data_ptr[GNSS_PROXY_OFFSET_MSG_ID])
   {
      /*! write gnss data to the file */
      /*! size-6 because to omit NULL at the end of the message */
      bytes = write(rLog.GnssLogFileFd, (const void*)&data_ptr[5], size-6);
      if (size-6 != (unsigned int)bytes)
      {
         SenDisp_vTraceOut(TR_LEVEL_ERRORS,
               "writing gnss data to file failed errno %d", errno);
      }
   }
   else
   {
      const unsigned char   *end_ptr = data_ptr + size;
      unsigned int iter = 0;

      /*! copy each byte in the received data to the file after
       *! converting to ascii */
      while (data_ptr <= end_ptr && iter < sizeof(buf)-2)
      {
         /*! tens place */
         buf[iter] = *data_ptr/16;
         buf[iter] = buf[iter]<10?buf[iter]+'0':buf[iter]+55;
         iter++;

         /*! ones place */
         buf[iter] = *data_ptr%16;
         buf[iter] = buf[iter]<10?buf[iter]+'0':buf[iter]+55;
         iter++;
         buf[iter++] = ' ';
         data_ptr++;
      }
      buf[iter++] = '\n';
      buf[iter]   = '\0';

      /*! update to file */
      write(rLog.GnssLogFileFd, buf, strlen(buf));
   }
}

/*!****************************************************************************
 *! @fn      : PosGnss_UpdateDataLogFileConfigWithDefaultParam()              *
 *! @brief   : updates global structure with data log file configuration.     *
 *!            for now configuration contains only data log file size.        *
 *! @param   : void                                                           *
 *! @return  : void                                                           *
 *!****************************************************************************
 */
static void PosGnss_UpdateDataLogFileConfigWithDefaultParam( void )
{
   struct statvfs vfs_attr;

   /*! check the disk space available in the vfs partation /var/opt/bosch/dyna-
    *! mic. use quarter of the available free disk space for gnss data logging.
    */
   if (0 != statvfs(POSGNSS_DISK_PARTATION, &vfs_attr))
   {
      rLog.u32MaxLogFileSize = 0;
      SenDisp_vTraceOut(TR_LEVEL_ERRORS,
             "statvfs on /var/opt/bosch/dynamic/ failed!,err %d"
             "max gnss data log file size is set to zero\n",errno);
   }
   else
   {
      rLog.u32MaxLogFileSize = vfs_attr.f_bfree*vfs_attr.f_bsize/4;
      SenDisp_vTraceOut(TR_LEVEL_USER_4,
             "statvfs on /var/opt/bosch/dynamic/ success,"
             "available free space is %u bytes,"
             "gnss data log file size is set to %u bytes,"
             "partation is %s\n",
             vfs_attr.f_bfree*vfs_attr.f_bsize, rLog.u32MaxLogFileSize,
             (vfs_attr.f_flag&ST_RDONLY)?"readonly":"writable");
   }
}

/*!****************************************************************************
 *! @fn      : PoSGnss_UpdateDataLogFileConfigToGlobalStruct()                *
 *! @brief   : updates global structure with data log file configuration.     *
 *!            for now configuration contains only data log file size.        *
 *! @param   : void                                                           *
 *! @return  : void                                                           *
 *!****************************************************************************
 */
static void PoSGnss_UpdateDataLogFileConfigToGlobalStruct( void )
{
   tBool bUpdateDefaultParam = TRUE;
   char Buf[81] = {0}, *SearchPtr;
   int  fd, size;

   if (-1 == (fd = open(POSGNSS_DATA_LOG_TRIGGER_FILE_NAME,O_RDONLY)))
   {
      SenDisp_vTraceOut(TR_LEVEL_ERRORS,
      "opening data log trigger file failed with error %d", errno);
   }
   /*! read configuration info */
   else if (-1 == (size = read(fd, Buf, sizeof(Buf)-1)))
   {
      SenDisp_vTraceOut(TR_LEVEL_ERRORS,
      "reading config from data log trigger file failed with error %d",errno);
   }
   /*! check the size of read data */
   else if (0 == size)
   {
      SenDisp_vTraceOut(TR_LEVEL_ERRORS,
                    "no config info in data log trigger file");
   }
   else
   {
      Buf[80] = '\0';  /*! coverity: NULL terminate */
      /*! is max file size mentioned in configuration file */
      if (NULL==(SearchPtr = strstr(Buf, POSGNSS_STRING_MAX_FILE_SIZE)))
      {
         SenDisp_vTraceOut(TR_LEVEL_ERRORS,
                 "max data log file size is not mentioned in data trig file");
      }
      else if(NULL==(SearchPtr = strstr( SearchPtr, "=" )))
      {
         SenDisp_vTraceOut(TR_LEVEL_ERRORS,
                 "error, didn't find = separator");
      }
      else
      {
         int retval;
         errno = 0;
         retval = strtoul((const char*)++SearchPtr, NULL, 10 );
         if( errno != 0 )
         {
           SenDisp_vTraceOut(TR_LEVEL_ERRORS,
           "strtoul conversion failed with error %d", errno);
         }
         else
         {
           bUpdateDefaultParam = FALSE; /*! don't use default size */
           rLog.u32MaxLogFileSize = (tU32)retval;
           SenDisp_vTraceOut(TR_LEVEL_USER_4,
                 "using data log size provided in the trigger file %d",
                 rLog.u32MaxLogFileSize);
         }
      }
   }

   if (-1 != fd)
   {
      (void)close(fd);
   }

   if (TRUE == bUpdateDefaultParam)
   {
      PosGnss_UpdateDataLogFileConfigWithDefaultParam();
   }
}

/*!****************************************************************************
 *! @fn      : PoSGnss_IsTriggerFileAvailable()                               *
 *! @brief   : checks the availability of trigger file for logging.           *
 *! @param   : void                                                           *
 *! @return  : TRUE - if trigger file exists.                                 *
 *!            FALSE - if trigger file does not exist.                        *
 *!****************************************************************************
 */
static tBool PoSGnss_IsTriggerFileAvailable(void)
{
   int fd; tBool ret;

   if (-1 != (fd = open(POSGNSS_DATA_LOG_TRIGGER_FILE_NAME,O_RDONLY)))
   {
      ret = TRUE;
   }
   else if (errno != ENOENT)
   {
      /*! file exists but opening failed due to some other reason */
      ret = TRUE;
      SenDisp_vTraceOut( TR_LEVEL_ERRORS,
      "Opening data log config file failed due to error %d\n", errno);
   }
   else
   {
      ret = FALSE;  /*! file does not exist */
   }

   if (-1 != fd)
   {
      (void)close(fd);
   }

   return ret;
}

/*!****************************************************************************
 *! @fn      : PoSGnss_CreateDataLogFiles()                                   *
 *! @brief   : creates a file to record received gnss data from SCC.          *
 *! @param   : void                                                           *
 *! @return  : void                                                           *
 *!****************************************************************************
 */
static void PoSGnss_CreateGnssDataLogFile( void )
{
   struct stat buf;

   /*! open gnss data log file */
   rLog.GnssLogFileFd = open( POSGNSS_GNSS_DATA_LOG_FILE_NAME,
                                    O_WRONLY|O_CREAT|O_APPEND,
                                    0600 );
   if (-1 == rLog.GnssLogFileFd)
   {
      SenDisp_vTraceOut( TR_LEVEL_ERRORS,
      "Opening/creating data log file \"%s\" failed with error %d\n",
      POSGNSS_GNSS_DATA_LOG_FILE_NAME, errno);
   }
   else
   {
      /*! get the file size */
      if (-1 != stat((const char *)POSGNSS_GNSS_DATA_LOG_FILE_NAME, &buf))
      {
         rLog.u32GnssLogFileSize = (tU32)buf.st_size;
         SenDisp_vTraceOut( TR_LEVEL_ERRORS,
                        "Current log file size %u", rLog.u32GnssLogFileSize);

         /*! update time and date to og file */
         if (rLog.u32GnssLogFileSize < rLog.u32MaxLogFileSize)
         {
            char CharDate[512] = {0};
            time_t t = time(NULL);
            struct tm tm = *localtime(&t);
            sprintf(CharDate,
                  "\n-----------------------------------------------------"
                  "\nDATA LOGGING STARTED AT %04d/%02d/%02d-%02d:%02d:%02d"
                  "\n-----------------------------------------------------",
                  tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, tm.tm_hour,
                  tm.tm_min, tm.tm_sec);
            write( rLog.GnssLogFileFd, CharDate, strlen(CharDate));
         }
      }
      else
      {
         /*! unable to get size of the file, better not to write any more data
          *! to the file */
         rLog.u32GnssLogFileSize = rLog.u32MaxLogFileSize;
         SenDisp_vTraceOut( TR_LEVEL_ERRORS,
                     "stat on the file \"%s\" failed with error %d"
                     "not writing any more data to the file",
                     POSGNSS_GNSS_DATA_LOG_FILE_NAME, errno );
      }
   }
}

/*!****************************************************************************
 *! @fn      : PoSGnss_CreatePosDataLogFile()                                 *
 *! @brief   : Creates a file to record received pos data from SCC.           *
 *! @param   : void                                                           *
 *! @return  : void                                                           *
 *!****************************************************************************
 */
static void PoSGnss_CreatePosDataLogFile(void)
{
   struct stat buf;

   /*! open gnss data log file */
   rLog.PosLogFileFd = open( POSGNSS_POS_DATA_LOG_FILE_NAME,
                             O_WRONLY|O_CREAT|O_APPEND,
                             0600 );
   if (-1 == rLog.PosLogFileFd)
   {
      SenDisp_vTraceOut( TR_LEVEL_ERRORS,
            "Opening/creating data log file \"%s\" failed with error %d\n",
            POSGNSS_POS_DATA_LOG_FILE_NAME, errno);
   }
   else
   {
      /*! get the file size */
      if (-1 != stat((const char *)POSGNSS_POS_DATA_LOG_FILE_NAME, &buf))
      {
         rLog.u32PosLogFileSize = (tU32)buf.st_size;
         SenDisp_vTraceOut( TR_LEVEL_ERRORS,
         "Current log file size %u", rLog.u32PosLogFileSize);
         /*! update time and date to log file */
         if (rLog.u32PosLogFileSize < rLog.u32MaxLogFileSize)
         {
            char CharDate[512] = {0};
            time_t t = time(NULL);
            struct tm tm = *localtime(&t);

            sprintf(CharDate,
                    "\n-----------------------------------------------------"
                    "\nDATA LOGGING STARTED AT %04d/%02d/%02d-%02d:%02d:%02d"
                    "\n-----------------------------------------------------",
                    tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, tm.tm_hour,
                    tm.tm_min, tm.tm_sec);

            write( rLog.PosLogFileFd, CharDate, strlen(CharDate));
         }
      }
      else
      {
         /*! unable to get size of the file, better not to write any
          *! more data to the file */
         rLog.u32PosLogFileSize = rLog.u32MaxLogFileSize;
         SenDisp_vTraceOut( TR_LEVEL_ERRORS,
                   "stat on the file \"%s\" failed with error %d"
                   "not writing any more data to the file",
                   POSGNSS_POS_DATA_LOG_FILE_NAME, errno );
      }
   }
}

/*!****************************************************************************
 *! @fn      : PosGnss_vInitDataLogging()                                     *
 *! @brief   : Initializes data logging                                       *
 *! @param   : void                                                           *
 *! @return  : void                                                           *
 *!****************************************************************************
 */
void PosGnss_vInitDataLogging( void )
{
   /*! CAUTION:
    *  Race condition:
    *  This function is invoked from both sensor proxy and gnss.
    *  Since VDSenor opens sensor proxy in INIT where as gnss
    *  driver at NORMAL state, so this race does not have any impact
    *  since after after first initialization no global are updated.
    */
   if (TRUE == rLog.bIsInitDone)
   {
      return;
   }

   rLog.bIsInitDone = TRUE;

   /*! check for the availability of data log trigger file */
   if (FALSE == PoSGnss_IsTriggerFileAvailable())
   {
      SenDisp_vTraceOut(TR_LEVEL_COMPONENT,
      "sensor data logging is disabled as trigger file \"%s\" is unavailable",
      POSGNSS_DATA_LOG_TRIGGER_FILE_NAME);
   }
   else
   {
      /*! trigger file exists, enable data logging */
      PosGnss_IsDataLoggingEnabled = TRUE;

      SenDisp_vTraceOut(TR_LEVEL_COMPONENT,
      "sensor data logging to file is enabled!!!\n");

      /*! get the configuration of data log file */
      PoSGnss_UpdateDataLogFileConfigToGlobalStruct();

	   /*! create log files for gnss and pos */
      PoSGnss_CreateGnssDataLogFile();
      PoSGnss_CreatePosDataLogFile();
   }
}

/*!****************************************************************************
 *! @fn      : PosGnss_vDeInitDataLogging()                                   *
 *! @brief   : closes the files opened                                        *
 *! @param   : void                                                           *
 *! @return  : void                                                           *
 *!****************************************************************************
 */
void PosGnss_vDeInitDataLogging(void)
{
   if (FALSE == rLog.bIsInitDone)
   {
      return;
   }

   if (-1 != rLog.GnssLogFileFd)
   {
      (void)close(rLog.GnssLogFileFd);
   }

   if (-1 != rLog.PosLogFileFd)
   {
      (void)close(rLog.PosLogFileFd);
   }

   PosGnss_IsDataLoggingEnabled = FALSE;
   rLog.bIsInitDone = FALSE;
}

/*!****************************************************************************
 *!                             end of file                                   *
 *!****************************************************************************
 */
