/***************************************************************************
*FILE        : inc2soc_testfunctions.cpp   
*
*SW-COMPONENT : Google Testing framework
*
*DESCRIPTION  : This file implements the inc2soc Google@target test cases for Gnss and POS
*               devices.
*
*AUTHOR       : Akshatha Nagaraju (RBEI/ECF1)
*
*COPYRIGHT    : (C) COPYRIGHT RBEI - All Rights Reserved
******************************************************************************/
#include "inc2soc_testfunctions.h"

using namespace std;
using namespace testing;


/*****************************************************************************
* Global variables
*****************************************************************************/
struct sockaddr_un rClientPos;
struct sockaddr_un rClientGnss;

class Inc2SocTestFunctionsPos : public ::testing ::Test
{
   protected:
   OSAL_tIODescriptor hDevGyro;
   OSAL_tIODescriptor hDevAcc;
   OSAL_tIODescriptor hDevOdo;

   public:
   virtual void SetUp()
   {
      hDevAcc = OSAL_IOOpen (OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READONLY);
      hDevOdo = OSAL_IOOpen (OSAL_C_STRING_DEVICE_ODOMETER, OSAL_EN_READONLY);
   }

   virtual void TearDown()
   {
      OSAL_s32IOClose(hDevAcc);
      OSAL_s32IOClose(hDevOdo);

      if(hDevGyro != NULL)
      {
         OSAL_s32IOClose(hDevGyro);
      }
      /* Close and Delete the Message queue */
      if(OSAL_NULL != hldMsgQueuePos)
      {
         if(OSAL_OK !=OSAL_s32MessageQueueClose(hldMsgQueuePos))
         {
            cout<<"MQ close failed, Error code: "<<OSAL_u32ErrorCode()<<endl;
         }
         else if ( OSAL_OK != OSAL_s32MessageQueueDelete(CLIENT_MSG_QUE_POS))
         {
            cout<<"MQ delete failed, Error code: "<<OSAL_u32ErrorCode()<<endl;
         }
      }
   }
};


class Inc2SocTestFunctionsGnss : public ::testing ::Test
{
   protected:
   OSAL_tIODescriptor hDevGnss;

   public:
   virtual void SetUp()
   {

   }

   virtual void TearDown()
   {
      if(hDevGnss != NULL)
      {
         OSAL_s32IOClose(hDevGnss);
      }
      /* Close and Delete the Message queue */
      if(OSAL_NULL != hldMsgQueueGnss)
      {
         if(OSAL_OK !=OSAL_s32MessageQueueClose(hldMsgQueueGnss))
         {
            cout<<"MQ close failed, Error code: "<<OSAL_u32ErrorCode()<<endl;
         }
         else if ( OSAL_OK != OSAL_s32MessageQueueDelete(CLIENT_MSG_QUE_GNSS))
         {
            cout<<"MQ delete failed, Error code: "<<OSAL_u32ErrorCode()<<endl;
         }
      }
   }
};



/*****************************************************************************
* TEST CASE      : Inc2SocBasicReadPos
*
* DESCRIPTION   : This test case creates client application for Inc2soc 
*                  to test the basic read functionality for POS.

*****************************************************************************/
TEST_F(Inc2SocTestFunctionsPos, Inc2SocBasicReadPos)
{
   pid_t Pid;
   uid_t UsrId = OSAL_NULL;
   gid_t GrpId = OSAL_NULL;

   tU32 u32GrpStatus;
   tU32 u32UsrStatus;
   tS32 s32Sock;
   tS32 s32Mode;

   tS32 s32RetVal;
   tU8 u8StrRecv[MAX_DATA+1];
   tS32 s32SockAddrLen = sizeof(struct sockaddr_un);
   rClientPos.sun_family = AF_UNIX;
   strncpy(rClientPos.sun_path, SOCKET_PATH_POS, sizeof(rClientPos.sun_path)-1);


   /* Get the NonRoot user's Id and eco_sensors group id */
   struct passwd *pNonRoot;
   struct group *pSensGroup;

   if((pNonRoot = getpwnam("nobody")) == NULL)
      cout<<"Error in fetching the User Id"<<endl;
   else
      UsrId=pNonRoot->pw_uid;
      
   if((pSensGroup = getgrnam("eco_sensors")) == NULL)
      cout<<"Error in fetching the Group Id"<<endl;
   else
      GrpId=pSensGroup->gr_gid;

   ASSERT_NE (OSAL_ERROR,(hDevGyro = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GYRO,
                           OSAL_EN_READONLY )))<< "GyroOpen Fail, OSAL Error code :" <<hDevGyro<<" "<< OSAL_u32ErrorCode();

   /* Create Message Queue to post received number of bytes from Client process to Parent process */
   ASSERT_NE (OSAL_ERROR,(OSAL_s32MessageQueueCreate((tCString)CLIENT_MSG_QUE_POS,
                           MAX_CLIENT_MESSAGE, sizeof(s32RetVal), (OSAL_tenAccess)OSAL_EN_READWRITE,
                           &hldMsgQueuePos)))<<"Error in message queue creation: "<<OSAL_u32ErrorCode();

   /* Create a new process to mock the Client environment to read from the socket */
   Pid = fork();
   if (Pid==0)
   {
      /* Change the permissions of the Client process */
      if((GrpId!=OSAL_NULL)&&(UsrId!=OSAL_NULL))
      {
         u32GrpStatus=setgid(GrpId);
         u32UsrStatus=setuid(UsrId);
      }

      /* Create Client Socket, connect to server end */
      if((s32Sock = socket(AF_UNIX, SOCK_SEQPACKET,0)) == ERROR_VALUE)
      {
         cout<<"Error during socket creation: "<<s32Sock;
         s32RetVal=ERROR_VALUE;
      }

      /* Give read & write permissions to the Group users of socket */
      else if((s32Mode = fchmod(s32Sock, S_IRWXU|S_IRWXG))==ERROR_VALUE)
      {
         cout<<"fchmod failed "<<s32Mode;
         s32RetVal=ERROR_VALUE;
      }

      else if((s32RetVal = connect(s32Sock, (struct sockaddr *) &rClientPos, s32SockAddrLen)) == ERROR_VALUE)
      {
         cout<<"Connection failed";
         s32RetVal=ERROR_VALUE;
      }

      /* Receive data from Server/Parent Socket.
         On success returns the number of bytes, on error -1 */
      else if((s32RetVal = recv(s32Sock, u8StrRecv, MAX_DATA,0)) > 0)
      {
            cout<<"DATA Received: "<<s32RetVal<<endl;
      }
      else
      {
         s32RetVal=ERROR_VALUE;
      }
      
      /* Open the Message Queue */
      EXPECT_EQ(OSAL_OK,(OSAL_s32MessageQueueOpen((tCString)CLIENT_MSG_QUE_POS,
                           (OSAL_tenAccess)OSAL_EN_READWRITE, &hldClientMsgQueuePos)))
                           <<"Error in opening message queue, Error code: "<<OSAL_u32ErrorCode();

      /* Post the Message Queue along with the structure and close the handler */
      EXPECT_EQ(OSAL_OK,(OSAL_s32MessageQueuePost(hldClientMsgQueuePos,
                           (tPCU8)&s32RetVal, sizeof(s32RetVal),CLIENT_MSG_PRIO)))
                           <<"Error in mq post: "<<OSAL_u32ErrorCode();

      EXPECT_EQ(OSAL_OK,(OSAL_s32MessageQueueClose(hldClientMsgQueuePos)))
                           <<"MQH close Failed, Error code: "<<OSAL_u32ErrorCode();

      /* Close the Socket */
      if(s32Sock!=ERROR_VALUE)
         {
            close(s32Sock);
         }
      _exit(0);

   }
   else if (Pid==-1)
   {
      cout<<"Error in Fork creation"<<endl;
      _exit(0);
   }
   else
   {
      /* Wait until the Client process ends */
      wait(NULL);

      /* Receive the Message on Message queue and copy it to buffer */
      EXPECT_NE(OSAL_ERROR,(OSAL_s32MessageQueueWait(hldMsgQueuePos,(tPU8)&s32RetVal,
                              sizeof(s32RetVal), OSAL_NULL, OSAL_CLIENT_WAITTIME)))
                              <<"Error in MQ_WAIT: "<<OSAL_u32ErrorCode();

      EXPECT_LT(0,s32RetVal)
           <<"TEST FAILED";
   }
}


/*****************************************************************************
* TEST CASE      : Inc2SocBasicReadGnss
*
* DESCRIPTION   : This test case creates client application for Inc2soc 
*                  to test the basic read functionality for GNSS.

*****************************************************************************/
TEST_F(Inc2SocTestFunctionsGnss, Inc2SocBasicReadGnss)
{
   pid_t Pid;
   uid_t UsrId = OSAL_NULL;
   gid_t GrpId = OSAL_NULL;

   tU32 u32GrpStatus;
   tU32 u32UsrStatus;
   tS32 s32Sock;
   tS32 s32Mode;

   tS32 s32RetVal;
   tU8 u8StrRecv[MAX_DATA+1];
   tS32 s32SockAddrLen = sizeof(struct sockaddr_un);
   rClientGnss.sun_family = AF_UNIX;
   strncpy(rClientGnss.sun_path, SOCKET_PATH_GNSS, sizeof(rClientGnss.sun_path)-1);

   /* Get the NonRoot user's Id and eco_sensors group id */
   struct passwd *pNonRoot;
   struct group *pSensGroup;

   if ((pNonRoot = getpwnam("nobody")) == NULL)
      cout<<"Error in fetching the User Id"<<endl;
   else
      UsrId=pNonRoot->pw_uid;

   if ((pSensGroup = getgrnam("eco_sensors")) == NULL)
      cout<<"Error in fetching the Group Id"<<endl;
   else
      GrpId=pSensGroup->gr_gid;

   ASSERT_NE (OSAL_ERROR,hDevGnss = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GNSS,
                           OSAL_EN_READONLY))<<"Gnss Open failed, Error code: "<<OSAL_u32ErrorCode();

   /* Create Message Queue to post received number of bytes from Client process to Parent process */
   ASSERT_NE (OSAL_ERROR,(OSAL_s32MessageQueueCreate((tCString)CLIENT_MSG_QUE_GNSS,
                           MAX_CLIENT_MESSAGE, sizeof(s32RetVal), (OSAL_tenAccess)OSAL_EN_READWRITE,
                           &hldMsgQueueGnss)))<<"Error in Message queue creation, Error code: "<<OSAL_u32ErrorCode();

   /* Flush the POS sensor data */
   ASSERT_NE (OSAL_ERROR,OSAL_s32IOControl( hDevGnss, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0))
                           <<"Failed to Flush POS sensor data"<<" Error code="<< OSAL_u32ErrorCode();

   /* Create a new process to mock the Client environment to read from the socket */
   Pid = fork();
   if (Pid==0)
   {
      /* Change the permissions of the Client process */
      if((GrpId!=OSAL_NULL)&&(UsrId!=OSAL_NULL))
      {
          u32GrpStatus=setgid(GrpId);
          u32UsrStatus=setuid(UsrId);
      }

      /* Create Client Socket, connect to server end */      
      if((s32Sock = socket(AF_UNIX, SOCK_SEQPACKET,0)) == ERROR_VALUE)
      {
         cout<<"Error during socket creation: "<<s32Sock<<endl;
         s32RetVal=ERROR_VALUE;
      }

      /* Give read & write permissions to the Group users of socket */	 
      else if((s32Mode = fchmod(s32Sock, S_IRWXU|S_IRWXG)) == ERROR_VALUE)
      {
         cout<<"fchmod failed "<<s32Mode<<endl;
         s32RetVal=ERROR_VALUE;
      }

      else if((s32RetVal = connect(s32Sock, (struct sockaddr *) &rClientGnss, s32SockAddrLen)) == ERROR_VALUE)
      {
         cout<<"Connection Failed"<<endl;
         s32RetVal=ERROR_VALUE;
      }

      /* Receive data from Server/Parent Socket.
         On success returns the number of bytes, on error -1 */	 
      else if ((s32RetVal = recv(s32Sock, u8StrRecv, MAX_DATA,0)) > 0)
      {
            cout<<"DATA Received: "<<s32RetVal<<endl;
      }
      else
      {
         s32RetVal=ERROR_VALUE;
      }

      /* Open the Message Queue */
      ASSERT_EQ(OSAL_OK,(OSAL_s32MessageQueueOpen((tCString)CLIENT_MSG_QUE_GNSS,
                           (OSAL_tenAccess)OSAL_EN_READWRITE, &hldClientMsgQueueGnss)))
                           <<"Error in opening message queue, Error code: "<<OSAL_u32ErrorCode();
    
      /* Post the Message Queue along with the structure and close the handler */
      ASSERT_EQ(OSAL_OK,(OSAL_s32MessageQueuePost(hldClientMsgQueueGnss,
                           (tPCU8)&s32RetVal, sizeof(s32RetVal),CLIENT_MSG_PRIO)))
                           <<"Error in mq post: "<<OSAL_u32ErrorCode();

      ASSERT_EQ(OSAL_OK,(OSAL_s32MessageQueueClose(hldClientMsgQueueGnss)))
                           <<"MQH close failed, Error code: "<<OSAL_u32ErrorCode();

      /* Close the Socket */
      if(s32Sock!=ERROR_VALUE)
         {
            close(s32Sock);
         }
      _exit(0);
   }
   else if(Pid==-1)
   {
      cout<<"Error in Fork creation"<<endl;
      _exit(0);
   }
   else
   {
      /* Wait until the Client process ends */
      wait(NULL);

      /* Receive the Message on Message queue and copy it to buffer */
      EXPECT_NE(OSAL_ERROR,(OSAL_s32MessageQueueWait(hldMsgQueueGnss,(tPU8)&s32RetVal,
                              sizeof(s32RetVal), OSAL_NULL, OSAL_CLIENT_WAITTIME)))
                              <<"Error in MQ_WAIT: "<<OSAL_u32ErrorCode();

      EXPECT_LT(0,s32RetVal)
                  <<"TEST FAILED";
   }
}
