/******************************************************************************
*FILE         : dev_gyro_TestFunctions.cpp
*
*SW-COMPONENT : Google Testing framework
*
*DESCRIPTION  : This file implements the Google@target test cases for the Gyro
*               device.
*
*AUTHOR       : Abhimanyu Pandit (RBEI/ECF12)
*
*COPYRIGHT    : (C) COPYRIGHT RBEI - All Rights Reserved
******************************************************************************/
#include "dev_gyro_TestFunctions.h"

using namespace testing;

/****************************************************************
Function prototypes
*****************************************************************/
static void vGyroThread(void* pvArg);
static void vOdoThread(void* pvArg);
static void vAccThread(void* pvArg);
static tU32 u32GyroTimestampValidationCallback(tU32);
static void GYROOedt_vPlotdata(const OSAL_trIOCtrl3dGyroData *prIOCtrlGyroData);

/****************************************************************
Global variables
*****************************************************************/
OSAL_tEventHandle MthrdEvent_handler;

class GyroTestFunctions : public ::testing::Test
{
   protected :
   OSAL_tIODescriptor hDevGyro;
   OSAL_tIODescriptor hDevAcc;
   OSAL_tIODescriptor hDevOdo;

   public:
   virtual void SetUp()
   {
      hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READONLY );
      hDevOdo = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ODOMETER, OSAL_EN_READONLY );
   }
   virtual void TearDown()
   {
      OSAL_s32IOClose( hDevAcc );
      OSAL_s32IOClose( hDevOdo );

      if(hDevGyro != OSAL_ERROR )
      {
      	 OSAL_s32IOClose ( hDevGyro );
      }
   }

   tU32 u32GyroCheckGYROValuesSeq(char);
};

class GyroMultiThreadTestFunctions : public ::testing::Test
{
   protected:
   OSAL_trThreadAttribute rThreadAttr;
   OSAL_tEventMask ResultMask;
   public:
   virtual void SetUp()
   {

   }

   virtual void TearDown()
   {

   }
   tU32 u32ThreadAttributes(char);
};


/***************************************************************************
||FUNCTION    :  u32GyroTimestampValidationCallback()
||
||DESCRIPTION :  Validates if the gyro data timestamp difference is 50ms
||
||PARAMETER   :  u32SamplesLogged
||
||RETURNVALUE :  tU32 error code
||
***************************************************************************/
tU32 u32GyroTimestampValidationCallback(tU32 u32SamplesLogged)
{
   tU32 u32PreviousTimeStamp;
   tU32 u32CurrentTimeStamp;
   tU32 u32ReturnValue = GYRO_ERROR_NONE;
   tS32 s32TimeDiff;
   FILE *fDescriptor1;
   tU32 u32ReadBuffer;
   tU32 u32NumOfTimestampErrors = 0;

   if((fDescriptor1 = fopen(GYRO_TIMESTAMP_RECORD, "r")) != NULL)   // open the file in read mode
   {
	  if ( 1 == fread(&u32ReadBuffer,sizeof(u32ReadBuffer),1,fDescriptor1))  // read from the file
      {
         u32CurrentTimeStamp =  u32ReadBuffer;
         u32SamplesLogged--;

         while(u32SamplesLogged > 0)
         {
        	if ( 1 == fread(&u32ReadBuffer,sizeof(u32ReadBuffer),1,fDescriptor1))
            {
               u32PreviousTimeStamp = u32CurrentTimeStamp;
               u32CurrentTimeStamp = u32ReadBuffer;
               s32TimeDiff = u32CurrentTimeStamp - u32PreviousTimeStamp;
               u32SamplesLogged--;

               if(s32TimeDiff >= GYRO_TIMESTAMP_DIFFERENCE_LOWER_LIMIT  && s32TimeDiff <= GYRO_TIMESTAMP_DIFFERENCE_UPPER_LIMIT)
               {
                  u32ReturnValue |= GYRO_ERROR_NONE;
               }
               else
               {
                  u32NumOfTimestampErrors++;
                  u32ReturnValue |= GYRO_ERROR_TIMESTAMP;
               }
            }
         }
      }
	  else
	  {
	     u32ReturnValue = GYRO_ERROR_TIMESTAMP_RECORD_READ;
	  }

	  fclose(fDescriptor1);
   }
   else
   {
      u32ReturnValue = GYRO_ERROR_TIMESTAMP_RECORD_OPEN;
   }

   return u32ReturnValue;
 }

/***************************************************************************
||FUNCTION    :  GYROOedt_vPlotdata()
||
||DESCRIPTION :  Plots the gyro R-S-T data graphically on terminal
||
||PARAMETER   :  prIOCtrlGyroData
||
||RETURNVALUE :  tU32 error code
||
***************************************************************************/
static void GYROOedt_vPlotdata(const OSAL_trIOCtrl3dGyroData *prIOCtrlGyroData)
{
   tU8 au8PlotData[GYRO_OEDT_TOTAL_PLOT_WIDTH + 1];
   tU16 u16R_Index, u16S_Index, u16T_Index;

   if (OSAL_NULL != prIOCtrlGyroData)
   {
      OSAL_pvMemorySet( au8PlotData, '_', GYRO_OEDT_TOTAL_PLOT_WIDTH);
      u16R_Index = (tU16)(prIOCtrlGyroData->u16Gyro_r / GYRO_OEDT_DATA_SCALE_FACTOR);
      u16S_Index = (tU16)(prIOCtrlGyroData->u16Gyro_s / GYRO_OEDT_DATA_SCALE_FACTOR);
      u16T_Index = (tU16)(prIOCtrlGyroData->u16Gyro_t / GYRO_OEDT_DATA_SCALE_FACTOR);

      au8PlotData[u16R_Index + GYRO_OEDT_R_PLOT_OFFSET] = 'R';
      au8PlotData[u16S_Index + GYRO_OEDT_S_PLOT_OFFSET] = 'S';
      au8PlotData[u16T_Index + GYRO_OEDT_T_PLOT_OFFSET] = 'T';

      au8PlotData[GYRO_OEDT_TOTAL_PLOT_WIDTH] = '\0';
      std::cout<< " "<< au8PlotData<<"\n";
   }
}

/***************************************************************************
||FUNCTION    :  u32GyroCheckGyroValuesSeq()
||
||DESCRIPTION :  Reads 100 data records from the driver,
||               Checks the variation in axis data is within 0.5%
||               of full scale when the target is still(not moving).
||
||PARAMETER   :  cAxis
||
||RETURNVALUE :  tU32 error code
||
***************************************************************************/
tU32 GyroTestFunctions:: u32GyroCheckGYROValuesSeq(char cAxis)
{
   OSAL_trIOCtrl3dGyroData rData = {0,0,0,0,0};
   tU32 u32NoOfRecsForRead = 100;
   tU8 u8Index = 0;
   tU16 au16GyroReadVal_rst[100] = {0};
   tU32 u32LowVal = 0;
   tU32 u32HighVal = 0;
   tU32 u32Sum = 0;
   tU32 u32AvgVal = 0;
   tF32 f32PercentageOfRange = 0.5;
   tU16 u16Gyro_s_varience_higher_limit = 0;
   tU16 u16Gyro_s_variance_lower_limit = 0;
   tU16 u16BaseAvgVal = 0;
   tU32 u32RetVal = GYRO_ERROR_NONE;

   for( u8Index= 0; ( u8Index < u32NoOfRecsForRead ) && ( u32RetVal != GYRO_ERROR_IOREAD ); u8Index++)
   {
	  if( OSAL_ERROR != OSAL_s32IORead( hDevGyro, (tPS8 )&rData,sizeof(rData)) )
      {
         if(cAxis == 'r')
            au16GyroReadVal_rst[u8Index] = rData.u16Gyro_r;
         else if(cAxis == 's')
            au16GyroReadVal_rst[u8Index] = rData.u16Gyro_s;
         else if(cAxis == 't')
            au16GyroReadVal_rst[u8Index] = rData.u16Gyro_t;
      }

	  else
	  {
		  u32RetVal = GYRO_ERROR_IOREAD;
	  }
   }

   if( u32RetVal == GYRO_ERROR_NONE )
   {
      for(u8Index= 0; u8Index < NO_OF_RECORDS; u8Index++)
      {
         u32Sum  +=  au16GyroReadVal_rst[u8Index];
      }

      u32AvgVal =(tU16) (u32Sum / NO_OF_RECORDS);
      //Calculate the 0.5% of average value
      u16BaseAvgVal = (tU16)((u32AvgVal * 0.5)/100);
      u16Gyro_s_varience_higher_limit = (tU16)(u32AvgVal + u16BaseAvgVal);
      u16Gyro_s_variance_lower_limit = (tU16)(u32AvgVal - u16BaseAvgVal);

      u32Sum = 0;
      u32AvgVal = 0;

      for( u8Index= 0; u8Index < u32NoOfRecsForRead; u8Index++)
      {
         u32Sum  +=  au16GyroReadVal_rst[u8Index];
         if( au16GyroReadVal_rst[u8Index] > u32HighVal )
            u32HighVal = au16GyroReadVal_rst[u8Index];
      }

      u32LowVal = GYRO_SENSOR_DATA_RESOLUTION_RANGE ;

      for( u8Index= 0; u8Index < u32NoOfRecsForRead; u8Index++)
      {
         if( au16GyroReadVal_rst[u8Index] <  u32LowVal )
            u32LowVal = au16GyroReadVal_rst[u8Index];
      }

      if( (u32HighVal - u32LowVal) > ((tU32)(f32PercentageOfRange*GYRO_SENSOR_DATA_RESOLUTION_RANGE)/100))
      {
         u32RetVal = GYRO_ERROR_VALUE_RANGE;
      }
   }

   if( u32RetVal == GYRO_ERROR_NONE )
   {
      u32AvgVal = u32Sum / u32NoOfRecsForRead;
      if( (u32AvgVal < u16Gyro_s_variance_lower_limit) || (u32AvgVal > u16Gyro_s_varience_higher_limit) )
      {
         u32RetVal=GYRO_ERROR_MID_VALUE;
      }
   }

   return  u32RetVal;
}

/***************************************************************************
||FUNCTION    :  u32ThreadAttributes()
||
||DESCRIPTION :  Defines attributes for gyro,acc and odo threads
||
||PARAMETER   :  cArg
||
||RETURNVALUE :  tU32 error code
||
***************************************************************************/
tU32 GyroMultiThreadTestFunctions::u32ThreadAttributes(char cArg)
{
   //Update thread attributes
   if(cArg == 'G')
   {
      rThreadAttr.szName  = "Gyro thread";
      rThreadAttr.pfEntry = vGyroThread;
   }
   else if(cArg == 'A')
   {
      rThreadAttr.szName  = "ACC thread";
      rThreadAttr.pfEntry = vAccThread;
   }
   else if(cArg == 'O')
   {
      rThreadAttr.szName  = "ODO thread";
      rThreadAttr.pfEntry = vOdoThread;
   }
   rThreadAttr.u32Priority   = OSAL_C_U32_THREAD_PRIORITY_NORMAL;
   rThreadAttr.s32StackSize  = 2048;
   rThreadAttr.pvArg         = OSAL_NULL;

   if(OSAL_ERROR == OSAL_ThreadSpawn(&rThreadAttr) )
      return GYRO_ERROR_THREAD_SPAWN;
   else
      return GYRO_ERROR_NONE;
}

/*******************************************/
//              GYROMETER THREAD
/*******************************************/
static void vGyroThread( tPVoid pvArg)
{
   OSAL_tIODescriptor hDevice ;
   OSAL_trIOCtrl3dGyroData rGyroData[10] = {0,0,0,0,0};
   (void)pvArg;//to remove lint //

   hDevice = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE );
   if ( hDevice == OSAL_ERROR)
   {
      std::cout<<"Gyro IO open failed\n";
   }

   else
   {
      /*Just Opened Gyro Driver.Just Wait Till Ring Buffer
      Is Filled With Some Entries */
      OSAL_s32ThreadWait(6000);
      //Read Data
      if(OSAL_s32IORead(hDevice,(tPS8 )rGyroData,
              NO_OF_ELEMENTS_TO_READ * sizeof( OSAL_trIOCtrl3dGyroData )) == OSAL_ERROR )
      {
         OSAL_s32IOClose ( hDevice );
         std::cout<<"Gyro IO read failed\n";
      }

      if ( OSAL_s32IOClose ( hDevice ) == OSAL_ERROR )
      {
         std::cout<<"Gyro IO close failed\n";
      }
   }

   if(OSAL_ERROR==OSAL_s32EventPost(MthrdEvent_handler,GYRO_MASK_VALUE,OSAL_EN_EVENTMASK_OR))
   {
      std::cout<<"Event post failed in Gyro Thread\n";
   }

   OSAL_vThreadExit();
}

/*******************************************/
//            ACCELEROMETER THREAD
/*******************************************/
static void vAccThread( tPVoid pvArg)
{
   OSAL_tIODescriptor hDevice ;
   OSAL_trIOCtrlAccData rAccData[10] ={0,0,0,0,0};
   (void)pvArg;//to remove lint//

   //Open Accelerometer Device
   hDevice = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE );
   if ( hDevice == OSAL_ERROR)
   {
      std::cout<<"Acc IO open failed\n";
   }
   else
   {
      OSAL_s32ThreadWait(6000);

      if(OSAL_s32IORead (hDevice,(tS8 *)rAccData,
                sizeof(OSAL_trIOCtrlAccData)*NO_OF_ELEMENTS_TO_READ) == OSAL_ERROR)
      {
         (tVoid)OSAL_s32IOClose ( hDevice );
         std::cout<<"Acc IO read failed\n";
      }

      if ( OSAL_s32IOClose ( hDevice ) == OSAL_ERROR )
      {
         std::cout<<"Acc IO close failed\n";
      }
   }
   if(OSAL_ERROR==OSAL_s32EventPost(MthrdEvent_handler,ACC_MASK_VALUE,OSAL_EN_EVENTMASK_OR))
   {
      std::cout<<"Event post failed in Acc\n";
   }

   OSAL_vThreadExit();
}

/*******************************************/
//              ODOMETER THREAD
/*******************************************/
static void vOdoThread( tPVoid pvArg)
{
   OSAL_tIODescriptor hDevice;
   OSAL_trIOCtrlOdometerData rOdoData[10]={0};
   (void)pvArg;//to remove lint//

   //Open Odometer Device
   hDevice = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ODOMETER, OSAL_EN_READWRITE );
   if ( hDevice == OSAL_ERROR)
   {
      std::cout<<"ODO  IO open failed\n";
   }
   else
   {
      OSAL_s32ThreadWait(6000);
      //Sequential read

      if(OSAL_s32IORead (hDevice,(tS8 *)rOdoData,
           sizeof(OSAL_trIOCtrlOdometerData)*NO_OF_ELEMENTS_TO_READ) == OSAL_ERROR)
      {
         OSAL_s32IOClose ( hDevice );
         std::cout<<"ODO  IO read failed\n";
      }

      if ( OSAL_s32IOClose ( hDevice ) == OSAL_ERROR )
      {
         std::cout<<"ODO  IO close failed\n";
      }
   }

   if(OSAL_ERROR==OSAL_s32EventPost(MthrdEvent_handler,ODO_MASK_VALUE,OSAL_EN_EVENTMASK_OR))
   {
      std::cout<<"Event post failed in Odo Thread\n";
   }

   OSAL_vThreadExit();
}

/*****************************************************************
* TEST CASE   : Gyro_IOOpenClose_Test
* DESCRIPTION : Tests device open and close.
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_IOOpenClose_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Gyro_IOReOpen_Test
* DESCRIPTION : Tests device re-open. Driver should not allow re-open
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_IOReOpen_Test)
{
   OSAL_tIODescriptor hDevGyro2;

   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_EQ( OSAL_ERROR, hDevGyro2 = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroReOpen Test Fail, OSAL Error code :" << OSAL_u32ErrorCode(),OSAL_s32IOClose( hDevGyro2 );

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}
/*****************************************************************
* TEST CASE   : Gyro_IOCloseAlreadyClosed_Test
* DESCRIPTION : Tests device re-close. Driver should not allow re-close
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_IOCloseAlreadyClosed_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_EQ( OSAL_ERROR, OSAL_s32IOClose( hDevGyro ) )
          << "GyroIOReclose Test Failed, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Gyro_IOOpenCloseRepeatedly_Test
* DESCRIPTION : Tests driver response for repeated open and close calls.
                Full project autosar must be flashed .
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_IOOpenCloseRepeatedly_Test)
{
   tU8 u8Index = 0;

   while(( u8Index < 20 ) && (hDevGyro != OSAL_ERROR))
   {
      ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
             << "GyroOpen Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

      ASSERT_NE( OSAL_ERROR, OSAL_s32IOClose( hDevGyro ) )
             << "GyroClose Fail, OSAL Error code :"<<OSAL_u32ErrorCode();

      u8Index++;
   }
}

/*****************************************************************
* TEST CASE   : Gyro_Self_Test
* DESCRIPTION : Tests Gyro self test functionality.
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_Self_Test)
{
   tS32 s32SelfTestResult = -1;

   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl (hDevGyro,OSAL_C_S32_IOCTRL_GYRO_SELF_TEST,(tLong) &s32SelfTestResult) )
          << "GyroSelfTest Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_EQ(1,s32SelfTestResult) << "GyroSelfTest Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Gyro_GetCount_Test
* DESCRIPTION : Tests Gyro get count functionality.
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_GetCount_Test)
{
   tU32 u32NumOfSamplesInBuff = 0;

   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   OSAL_s32ThreadWait(500);

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_GYRO_GETCNT,(tLong)&u32NumOfSamplesInBuff ) )
          << "GyroGetCount Test Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Gyro_GetCycleTime_Test
* DESCRIPTION : Tests Gyro get cycle time functionality.
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_GetCycleTime_Test)
{
   tU32 u32GyroCycleTimeInNS;

   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_GYRO_GETCYCLETIME,(tLong)&u32GyroCycleTimeInNS ) )
          << "GyroCycleTime Test Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_EQ( u32GyroCycleTimeInNS, GYRO_DEFAULT_TIME_INTERVAL )
          << "GYRO_CYCLE_TIME_VALUE_MISMATCH_ERROR"; //lint !e774

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Gyro_GetTemperature_Test
* DESCRIPTION : Tests Gyro get temperature functionality.
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_GetTemperature_Test)
{
   tF32 f32Gyrotemperature = 0;
   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
	      << "GyroOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   std::cout<<"waiting one min to get Gyro temperature\n";

   OSAL_s32ThreadWait(60000);
   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_GYRO_GET_TEMP,(tLong)&f32Gyrotemperature ) )
          << "GyroGetTemp Test Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Gyro_BasicRead_Test
* DESCRIPTION : Checks the number of records available in the buffer
                and reads them.
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_BasicRead_Test)
{
   tU32 u32NumOfSamplesInBuff = 0;
   tU32 u32RecSize = sizeof(OSAL_trIOCtrl3dGyroData);
   OSAL_trIOCtrl3dGyroData *DataBufferPointer = NULL;
   OSAL_trIOCtrl3dGyroData *NullPointer = NULL;
   tS32 s32GyroDataRead;

   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   OSAL_s32ThreadWait(500);

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_GYRO_GETCNT,(tLong)&u32NumOfSamplesInBuff ) )
          << "GyroGetCount Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   ASSERT_NE( NullPointer , DataBufferPointer = (OSAL_trIOCtrl3dGyroData*)OSAL_pvMemoryAllocate( u32NumOfSamplesInBuff * u32RecSize ) )
          << "GyroMemAlloc Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, s32GyroDataRead = OSAL_s32IORead( hDevGyro, (tPS8)DataBufferPointer, u32NumOfSamplesInBuff * u32RecSize ) )
          << "GyroRead Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_EQ( u32NumOfSamplesInBuff*u32RecSize, s32GyroDataRead )
          << "Read lesser records than available" ;

   OSAL_vMemoryFree(DataBufferPointer);

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Gyro_ReadPassingNullBuffer_Test
* DESCRIPTION : Tests Gyro driver response for the read functionality
                by passing null buffer.
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_ReadPassingNullBuffer_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( OSAL_ERROR, OSAL_s32IORead( hDevGyro, OSAL_NULL, 1 ) )
          << "GyroReadPassingNullBuffer Fail, OSAL Error code :"<<OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Gyro_IOCRTLPassingNullBuffer_Test
* DESCRIPTION : Tests Gyro driver response for IO control functionality
                by passing null buffer.
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_IOCRTLPassingNullBuffer_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( OSAL_ERROR, OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_VERSION,(tLong)NULL ) )
          << "GyroDriverVersionTest, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( OSAL_ERROR, OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_GYRO_GETCNT,(tLong)NULL ) )
          << "GyroGetCountTest Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( OSAL_ERROR, OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_GYRO_GETCYCLETIME,(tLong)NULL ) )
          << "GyroGetCycleTimeTest  Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Gyro_InterfaceCheckSeq_Test
* DESCRIPTION : Tests all the IO Control interface provided by the
                Gyro Driver.
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_InterfaceCheckSeq_Test)
{
   OSAL_trIOCtrlAccData rData;
   tU32 u32Version, u32CycleTimeInNs, u32NumOfSamplesInBuff;

   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_VERSION,(tLong)&u32Version ) )
          << "GyroDriverVersionTest, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_GYRO_GETCYCLETIME,(tLong)&u32CycleTimeInNs ) )
          << "GyroGetCycleTime Fail, OSAL Error code :"<<OSAL_u32ErrorCode();

   OSAL_s32ThreadWait(500);

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_GYRO_GETCNT,(tLong)&u32NumOfSamplesInBuff ) )
          << "GyroGetCount Fail, OSAL Error code :"<<OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IORead( hDevGyro, (tPS8) &rData, sizeof(rData) ) )
          << "GyroRead Fail-1, OSAL Error code :"<< OSAL_u32ErrorCode();//lint !e774

   EXPECT_NE( OSAL_ERROR, OSAL_s32IORead( hDevGyro, (tPS8) &rData, sizeof(rData) ) )
          << "GyroRead Fail-2, OSAL Error code :" << OSAL_u32ErrorCode();//lint !e774

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Gyro_GetHwInfo_Test
* DESCRIPTION : Tests Gyro get hardware info functionality.
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_GetHwInfo_Test)
{
   OSAL_trIOCtrlHwInfo sGyroHwInfo;

   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_GYRO_GET_HW_INFO,(tLong)&sGyroHwInfo ) )
          << "GyroGetHwInfoTest Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : GyroValuesSeq_r_Test
* DESCRIPTION : Reads 100 data records from the driver,
                Checks the variation in R-axis data is within 0.5%
                of full scale when the target is still(not moving).
******************************************************************/
TEST_F(GyroTestFunctions, GyroValuesSeq_r_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( GYRO_ERROR_NONE, u32GyroCheckGYROValuesSeq('r') )
          << "GyroValuesSeq_r Fail,OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : GyroValuesSeq_s_Test
* DESCRIPTION : Reads 100 data records from the driver,
                Checks the variation in S-axis data is within 0.5%
                of full scale when the target is still(not moving).
******************************************************************/
TEST_F(GyroTestFunctions, GyroValuesSeq_s_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( GYRO_ERROR_NONE,u32GyroCheckGYROValuesSeq('s') )
          << "GyroValuesSeq_s Fail,OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : GyroValuesSeq_t_Test
* DESCRIPTION : Reads 100 data records from the driver,
                Checks the variation in T-axis data is within 0.5%
                of full scale when the target is still(not moving).
******************************************************************/
TEST_F(GyroTestFunctions, GyroValuesSeq_t_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( GYRO_ERROR_NONE,u32GyroCheckGYROValuesSeq('z') )
          << "GyroValuesSeq_t Fail,OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : GyroOedt_u32ValuesPlot_Test
* DESCRIPTION : Plot gyro data on console . This test is disabled.
******************************************************************/
TEST_F(GyroTestFunctions, GYROOedt_u32ValuesPlot)
{
   OSAL_trIOCtrl3dGyroData rIOCtrlGyroData;
   tBool bTestEndFlag = FALSE;
   tU32 u32GyroPrevPlotTS = 0;
   tU32 u32TestEndTime = OSAL_ClockGetElapsedTime() + GYRO_OEDT_TEST_DURATION_MS;

   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   while ( TRUE != bTestEndFlag )
   {
      if( OSAL_ERROR != OSAL_s32IORead( hDevGyro,(tPS8) &rIOCtrlGyroData,sizeof(OSAL_trIOCtrl3dGyroData)) )
      {
         if ( (rIOCtrlGyroData.u32TimeStamp - u32GyroPrevPlotTS) > GYRO_OEDT_PLOT_DATA_INTERVALL )
         {
            GYROOedt_vPlotdata(&rIOCtrlGyroData);
            u32GyroPrevPlotTS = rIOCtrlGyroData.u32TimeStamp;
         }
      }

      if ( OSAL_ClockGetElapsedTime() > u32TestEndTime )
      {
         bTestEndFlag = TRUE;
      }
   }

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Gyro_TimestampValidation_Test
* DESCRIPTION : Checks whether the timestamp differences between consecutive
                Gyro data is in between GYRO_TIMESTAMP_DIFFERENCE_LOWER_LIMIT
                and GYRO_TIMESTAMP_DIFFERENCE_UPPER_LIMIT, for 5 minuntes.
******************************************************************/
TEST_F(GyroTestFunctions, Gyro_TimestampValidation_Test)
{
   tU32 u32NumOfSamplesLogged = 0,u32NumOfSamplesInBuff;
   OSAL_trIOCtrl3dGyroData *DataBufferPointer = NULL;
   tU32 u32TimestampData;
   OSAL_trIOCtrl3dGyroData *NullPointer = NULL;
   tBool bTimeoutFlag = FALSE;
   tU32 u32Index;
   FILE *fDescriptor1 = NULL;
   FILE *NullPtr = NULL;
   tU32 u32TestEndTime = OSAL_ClockGetElapsedTime() + GYRO_SAMPLE_READ_TIME_IN_MS;

   ASSERT_NE( OSAL_ERROR, hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READWRITE ) )
          << "GyroOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   ASSERT_NE(NullPtr,(fDescriptor1 = fopen(GYRO_TIMESTAMP_RECORD, "w+")))
          << "GYRO_TIMESTAMP_RECORD Open Fail, OSAL Error code :" << OSAL_u32ErrorCode();// open the file in write/append mode

/* Since initial chunk of gyro data (pre-sampled data) has timestamp difference of 150ms between the successive samples,
 * this chunk of data is ignored for timestamp validation.
 */
   OSAL_s32ThreadWait(1000);
   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_GYRO_GETCNT,(tLong)&u32NumOfSamplesInBuff ) )
          << "GyroGetCount Fail, OSAL Error code :"<<OSAL_u32ErrorCode();

   ASSERT_NE( NullPointer , DataBufferPointer = (OSAL_trIOCtrl3dGyroData*)OSAL_pvMemoryAllocate( u32NumOfSamplesInBuff * sizeof(OSAL_trIOCtrl3dGyroData) ) )
          << "GyroMemAlloc Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IORead( hDevGyro, (tPS8)DataBufferPointer, (tU32)(u32NumOfSamplesInBuff * sizeof(OSAL_trIOCtrl3dGyroData )) ) )
          << "GyroRead Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   std::cout<<"this test runs for 5mins\n";

   OSAL_s32ThreadWait(250);
   while(!bTimeoutFlag)
   {
	  if( OSAL_ERROR != OSAL_s32IOControl( hDevGyro, OSAL_C_S32_IOCTRL_GYRO_GETCNT,(tLong)&u32NumOfSamplesInBuff ) )
	  {
         if( u32NumOfSamplesInBuff > 0 )
         {
    	    if (OSAL_ERROR != OSAL_s32IORead( hDevGyro, (tPS8)DataBufferPointer, (tU32)(u32NumOfSamplesInBuff * sizeof(OSAL_trIOCtrl3dGyroData)) ) )
    	    {
               for(u32Index = 0; u32Index < u32NumOfSamplesInBuff;  u32Index++)
               {
                  u32TimestampData = DataBufferPointer[u32Index].u32TimeStamp;
                  if(1 == (fwrite(&u32TimestampData, sizeof(tU32),1,fDescriptor1)))
                     u32NumOfSamplesLogged ++;
               }
    	    }
         }
	  }
      else
      {
         OSAL_s32ThreadWait(250);
      }

      if ( OSAL_ClockGetElapsedTime() > u32TestEndTime )
      {
         bTimeoutFlag = TRUE;
      }
      u32NumOfSamplesInBuff = 0;
   }

   OSAL_vMemoryFree(DataBufferPointer);

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevGyro ) )
          << "GyroClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
   EXPECT_EQ(0,fclose(fDescriptor1))
          << "GyroLog file close Fail, OSAL Error code :"<<OSAL_u32ErrorCode();  //close the file

   EXPECT_EQ(GYRO_ERROR_NONE, u32GyroTimestampValidationCallback(u32NumOfSamplesLogged))
          << "Gyro timestamp validationCallback Fail, OSAL Error code :"<<OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : MultiThreadBasicRead_Test
* DESCRIPTION : Reads the data from Gyro ,Acc and Odo.
******************************************************************/
TEST_F(GyroMultiThreadTestFunctions,MultiThreadBasicRead_Test)
{
   tU32 u32RetVal;
   u32RetVal = OSAL_s32EventCreate(MULTI_THREAD_EVENT,&MthrdEvent_handler);
   ASSERT_NE(OSAL_ERROR,u32RetVal);

   EXPECT_EQ(GYRO_ERROR_NONE,u32ThreadAttributes('G'))<< "THREAD_SPAWN_ERROR, OSAL Error code :"<<OSAL_u32ErrorCode(); // parameter 'G' is for Gyro, 'A' is for Acc and 'O' is for Odo.

   EXPECT_EQ(GYRO_ERROR_NONE,u32ThreadAttributes('A'))<< "THREAD_SPAWN_ERROR, OSAL Error code :"<<OSAL_u32ErrorCode();

   EXPECT_EQ(GYRO_ERROR_NONE,u32ThreadAttributes('O'))<< "THREAD_SPAWN_ERROR, OSAL Error code :"<<OSAL_u32ErrorCode();

   u32RetVal = OSAL_s32EventWait(MthrdEvent_handler,GYRO_MASK_VALUE|ACC_MASK_VALUE|ODO_MASK_VALUE,OSAL_EN_EVENTMASK_AND,OSAL_MULTITHREAD_WAIT_TIMEOUT ,&ResultMask);
   EXPECT_NE(OSAL_ERROR,u32RetVal);

   u32RetVal = OSAL_s32EventPost(MthrdEvent_handler,~ResultMask,OSAL_EN_EVENTMASK_AND);
   EXPECT_NE(OSAL_ERROR,u32RetVal);

   u32RetVal = OSAL_s32EventClose(MthrdEvent_handler);
   EXPECT_NE(OSAL_ERROR,u32RetVal);

   u32RetVal = OSAL_s32EventDelete(MULTI_THREAD_EVENT);
   EXPECT_NE(OSAL_ERROR,u32RetVal);
}

//EOF
