/************************************************************************
 | $Revision: 1.0 $
 | $Date: 2012/01/01 $
 |************************************************************************
 | FILE:         cdaudio_cdtext.c
 | PROJECT:      GEN3
 | SW-COMPONENT: OSAL I/O Device
 |------------------------------------------------------------------------
 | DESCRIPTION:
 |  This file contains the /dev/cdaudio device implementation
 |------------------------------------------------------------------------
 | Date      | Modification                      | Author
 | 2013      | GEN3 VW MIB                       | srt2hi
 | 2-6-2017  | Correction of logic for reading CD|
             | text from Raw CD data             | boc7kor
 |************************************************************************
 |************************************************************************/

/************************************************************************
 | includes of component-internal interfaces
 | (scope: component-local)
 |-----------------------------------------------------------------------*/
/* Interface for variable number of arguments */
// srt2hi migrate #include <extension/dbgspt.h>
/* Basic OSAL includes */
#include "OsalConf.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#include "Linux_osal.h"
#include "ostrace.h"
#include "cd_main.h"
#include "cd_cache.h"
#include "cdaudio.h"
#include "cd_funcenum.h"
#include "cdaudio_cdtext.h"
#include "cdaudio_trace.h"
#include "cd_scsi_if.h"

#ifdef __cplusplus
extern "C"
{
#endif

/************************************************************************
 |defines and macros (scope: module-local)
 |-----------------------------------------------------------------------*/
#define CDAUDIO_ATABYTE(pu8,row,col) (pu8\
                                     [(CDAUDIO_CDTEXT_ATA_BUFFER_BYTES_PER_LINE\
                                       * (tU32)(row)) + (tU32)(col)])

/************************************************************************
 |typedefs (scope: module-local)
 |----------------------------------------------------------------------*/

/************************************************************************
 | variable definition (scope: global)
 |-----------------------------------------------------------------------*/

/************************************************************************
 | variable inclusion  (scope: global)
 |-----------------------------------------------------------------------*/

/************************************************************************
 | variable definition (scope: module-local)
 |-----------------------------------------------------------------------*/

/************************************************************************
 |function prototype (scope: module-local)
 |-----------------------------------------------------------------------*/

  /************************************************************************
 |function implementation (scope: module-local)
 |-----------------------------------------------------------------------*/

/*****************************************************************************
 * FUNCTION:     vTraceHex
 * PARAMETER:    some needed values
 * RETURNVALUE:  result
 * DESCRIPTION:  prints hex dump

 * HISTORY:
 ******************************************************************************/
static tVoid vTraceHex(tDevCDData *pShMem, const tU8 *pu8Buf, tInt iBufLen,
                       tInt iBytesPerLine)
{
  tInt iBpl = iBytesPerLine;
  tInt iL, iC;
  char c;
  char szTmp[32];
  char szTxt[CDAUDIO_CDTEXT_TEXT_BUFFER_LEN];
  if(NULL == pu8Buf)
  {
    return;
  } //if(NULL == pu8Buf)

  CD_CHECK_SHARED_MEM_VOID(pShMem);

  for(iL = 0; iL < iBufLen; iL += iBpl)
  {
    sprintf(szTxt, "%04X:", (unsigned int)(iL));
    for(iC = 0; iC < iBpl; iC++)
    {
      sprintf(szTmp, " %02X", (unsigned int)pu8Buf[iL + iC]);
      strcat(szTxt, szTmp);
    } //for(iC = 0 ; iC < iBpl ; iC++)

    strcat(szTxt, " ");
    for(iC = 0; iC < iBpl; iC++)
    {
      c = (char)pu8Buf[iL + iC];
      if((c < 0x20) || (c >= 0x80))
      {
        c = '.';
      }
      sprintf(szTmp, "%c", (unsigned char)c);

      strcat(szTxt, szTmp);
    } //for(iC = 0 ; iC < iBpl ; iC++)
    CDAUDIO_PRINTF_U3("%s", szTxt);
  } //for(i = 0 ; i < iBufLen ; i += iBpl)
}

/******************************************************************************
 *FUNCTION      :vGetTrackText
 *
 *DESCRIPTION   :Fills up the buffer sent by the caller with the name of
 *               the artist/track requested. The function ensures that
 *               the buffer is null terminated.
 *
 *
 *PARAMETER     :pu8OutText    Pointer to buffer to store text, contains "\000" if
 *                          no text of the requested type is available for the
 *                          track
 *               pU8InBuff     Pointer to buffer containing Raw CD Text
 *               u32DataLength Size of InBuff in bytes
 *               u8PackType   Type of packet to read, e.g. 0x80 for title,
 *                          0x81 for artist... see ATAPI spec
 *               Track      Number of track to retrieve text for
 *
 *RETURNVALUE   :none
 *
 *HISTORY:      :Ported from Gen2 by sgo1cob
 *               Date      | Modification                      | Author
 *               2-6-2017  | Correction of logic for reading CD|
 *                         | text from Raw CD data             | boc7kor
 *****************************************************************************/
static void vGetTrackText(tU8* pu8OutText, tU8* pU8InBuff,
                tU32 u32DataLength, tU8 u8PackType, tU8 u8Track)
{
    tU32    iu32Off = 0;
    tU32    ctu32N;
    tU8*    pU8TextDesc;
    tU8     iu8Index=0;
    tU8     u8TempTrack;
    tU8     u8CharPos;

    /*set all the bytes of the buffer to zero to avoid
      the need of null terminating*/
    memset(pu8OutText, 0, CDAUDIO_CDTEXT_TEXT_BUFFER_LEN);

    /*traverse to the packet which contains the track information*/
    for(ctu32N = CDAUDIO_CDTEXT_ATA_BUFFER_TEXTDATA_OFFSET
        ; ctu32N < u32DataLength + CDAUDIO_CDTEXT_SIZE_BYTE_OFFSET
                   ; ctu32N += CDAUDIO_CDTEXT_ATA_BUFFER_BYTES_PER_LINE)
    {
        /*Point to the first byte of a packet*/
        pU8TextDesc = pU8InBuff + ctu32N;
        if((pU8TextDesc[0] == u8PackType)
        && (pU8TextDesc[1] & CDAUDIO_CDTEXT_ATA_BUFFER_TRACKNO_MASK) >= u8Track)
        {
            break;
        }
    }

    if(ctu32N  >= u32DataLength + CDAUDIO_CDTEXT_SIZE_BYTE_OFFSET)
    {/*if Track information not present return*/
        return;
    }

    if(u8Track == pU8TextDesc[1] & CDAUDIO_CDTEXT_ATA_BUFFER_TRACKNO_MASK)
    {/*if the packet explicitly starts with the track of interest*/
        if((u8CharPos = pU8TextDesc[3] & 
                CDAUDIO_CDTEXT_ATA_BUFFER_CHARPOS_MASK) !=0)
        {/*if the track has some data in previous packet, copy it */
            memcpy(&pu8OutText[0],pU8InBuff + (ctu32N-pU8TextDesc[3]-2)
                      ,u8CharPos);
            iu32Off = u8CharPos;
        }

        while(iu8Index < CDAUDIO_CDTEXT_ATA_BUFFER_TEXT_PER_LINE)
        {/*collect Track text from current packet*/
            if(pU8TextDesc[CDAUDIO_CDTEXT_PACKET_TEXT_OFFSET + iu8Index] == 0)
            {
                break;
            }

            if(iu32Off < CDAUDIO_CDTEXT_TEXT_BUFFER_LEN - 1)
            {
                pu8OutText[iu32Off++] 
                 = pU8TextDesc[CDAUDIO_CDTEXT_PACKET_TEXT_OFFSET + iu8Index++];
            }
            else
            {
                break;
            }
            
            /* 
             * if the end of packet reached and next packet
             * also contains current track's text then continue collecting.
             * The 2nd header which contain track information is at an
             * offset of 3 bytes from current position (1st CRC byte)
             */
            if(iu8Index == CDAUDIO_CDTEXT_ATA_BUFFER_TEXT_PER_LINE
                 && pU8TextDesc[CDAUDIO_CDTEXT_PACKET_TEXT_OFFSET + iu8Index + 3]
                        == u8Track)
            {
                pU8TextDesc += CDAUDIO_CDTEXT_ATA_BUFFER_BYTES_PER_LINE;
                iu8Index=0;
            }
         }
    }
    else
    {/*else search in the previous packet*/

        /*point to the previous packet*/
        pU8TextDesc = pU8TextDesc - CDAUDIO_CDTEXT_ATA_BUFFER_BYTES_PER_LINE;
        /*start with the first track in the packet*/
        u8TempTrack=pU8TextDesc[1] & CDAUDIO_CDTEXT_ATA_BUFFER_TRACKNO_MASK;

        while(u8TempTrack < u8Track)
        {/*Traverse to the track of interest*/
            if(iu8Index >= CDAUDIO_CDTEXT_ATA_BUFFER_TEXT_PER_LINE)
            {/*if packet ended*/
                break;
            }   

            if(pU8TextDesc[CDAUDIO_CDTEXT_PACKET_TEXT_OFFSET + iu8Index++] 
                == 0)
            {
                u8TempTrack++;
            }
        }
        if(u8TempTrack == u8Track)
        {
            while(iu8Index < CDAUDIO_CDTEXT_ATA_BUFFER_TEXT_PER_LINE)
            {
                if(pU8TextDesc[CDAUDIO_CDTEXT_PACKET_TEXT_OFFSET + iu8Index]
                            == 0)
                {/*if null character found string ends*/
                    break;
                }

                if(iu32Off < CDAUDIO_CDTEXT_TEXT_BUFFER_LEN - 1)
                {
                    pu8OutText[iu32Off++] 
                        = pU8TextDesc[CDAUDIO_CDTEXT_PACKET_TEXT_OFFSET + iu8Index++];
                }
                else
                {
                    break;
                }
            }
            /* if null not encountered before packet ends and the
             * next packet contains information about different track
             * The 2nd header which contain track information is at an
             * offset of 3 bytes from current position (1st CRC byte)
             */
            if(iu8Index == CDAUDIO_CDTEXT_ATA_BUFFER_TEXT_PER_LINE
                 && pU8TextDesc[CDAUDIO_CDTEXT_PACKET_TEXT_OFFSET + iu8Index + 3]
                        != u8Track)
            {
                pu8OutText[0]=0;
            }

         }
    }

    /*Last byte made 0 to ensure null termination of string*/
    pu8OutText[CDAUDIO_CDTEXT_TEXT_BUFFER_LEN-1] = 0;
}


/*****************************************************************************
 * FUNCTION:     u32ConvertUglyATA2NiceCDText
 * PARAMETER:    some needed values
 * RETURNVALUE:  Osal error code
 * DESCRIPTION:  makes osal-style cd.text from ugly ATA format

 * HISTORY:Date      | Modification                      | Author
 *         2-6-2017  | Correction of logic for reading CD|
 *                   | text from Raw CD data             | boc7kor
 ******************************************************************************/
static tU32 u32ConvertUglyATA2NiceCDText(tDevCDData *pShMem,
                                         CDAUDIO_sCDText_type *prCDTxt,
                                         tU32 u32ATACDTextLen)
{
  tU32 u32Ret = OSAL_E_NOERROR;

  CD_CHECK_SHARED_MEM(pShMem, OSAL_E_INVALIDVALUE);

  CDAUDIO_TRACE_ENTER_U4(CDID_u32ConvertUglyATA2NiceCDText, prCDTxt,
                         u32ATACDTextLen, 0, 0);

  if(NULL != prCDTxt)
  {
    tU8 au8Txt[CDAUDIO_CDTEXT_TEXT_BUFFER_LEN] = { 0};
    tU8 u8Len;
    tU8 u8MaxTrack = 0;
    tU8 iu8Index;


    u8MaxTrack = prCDTxt->au8ATARawCDTextBuffer[3];
    //clear old content
    memset(prCDTxt->saTrackText, 0, sizeof(CDAUDIO_sTrackCDText_type));


    for(iu8Index=0; iu8Index <= u8MaxTrack; iu8Index++)
    {
        /*Get Track Name*/
        vGetTrackText(au8Txt, prCDTxt->au8ATARawCDTextBuffer, 
                       u32ATACDTextLen, enCDAUDIO_CDTEXT_TYPE_TITLE, iu8Index);
        u8Len=strlen(au8Txt);
        u8Len = CD_MIN(u8Len, (CDAUDIO_CDTEXT_TEXT_BUFFER_LEN-1));
        memcpy(prCDTxt->saTrackText[iu8Index].szTitle, au8Txt, u8Len);
        prCDTxt->saTrackText[iu8Index].szTitle[u8Len] = 0x00;
        prCDTxt->saTrackText[iu8Index].iTitleLen = u8Len;

        /*Get Artist Name*/
        vGetTrackText(au8Txt, prCDTxt->au8ATARawCDTextBuffer, 
                       u32ATACDTextLen, enCDAUDIO_CDTEXT_TYPE_ARTIST, iu8Index);
        u8Len=strlen(au8Txt);
        u8Len = CD_MIN(u8Len, (CDAUDIO_CDTEXT_TEXT_BUFFER_LEN-1));
        memcpy(prCDTxt->saTrackText[iu8Index].szArtist, au8Txt, u8Len);
        prCDTxt->saTrackText[iu8Index].szArtist[u8Len] = 0x00;
        prCDTxt->saTrackText[iu8Index].iArtistLen = u8Len;
    }
  }
  else //if(NULL != prCDTxt)
  {
    u32Ret = OSAL_E_INVALIDVALUE;
  } //else //if(NULL != prCDTxt)

  CDAUDIO_TRACE_LEAVE_U4(CDID_u32ConvertUglyATA2NiceCDText, u32Ret,
                                                          0, 0, 0, 0);
  return u32Ret;
}

/*****************************************************************************
 * FUNCTION:     CDAUDIO_vCDTEXTDebug
 * PARAMETER:    void
 * RETURNVALUE:  void
 * DESCRIPTION:  printf CD-Text
 ******************************************************************************/
tVoid CDAUDIO_vCDTEXTDebug(tDevCDData *pShMem)
{
  CD_CHECK_SHARED_MEM_VOID(pShMem);

  if(enCDAUDIO_CDTEXT_VALID == pShMem->rCDText.enValid)
  {
    tU8 u8T;
    tInt iALen, iTLen;
    tU32 u32DataLen;
    const char *pcszA, *pcszT;

    u32DataLen = CD_8TO16(pShMem->rCDText.au8ATARawCDTextBuffer[0],
                          pShMem->rCDText.au8ATARawCDTextBuffer[1]);
    vTraceHex(pShMem, pShMem->rCDText.au8ATARawCDTextBuffer, 4, 4);
    if(u32DataLen > 0)
    {
      vTraceHex(pShMem, &pShMem->rCDText.au8ATARawCDTextBuffer[4],
                (tInt)(u32DataLen - 4), 18);
    }

    for(u8T = 0; u8T < CD_TRACK_ARRAY_SIZE; u8T++)
    {
      iALen = pShMem->rCDText.saTrackText[u8T].iArtistLen;
      iTLen = pShMem->rCDText.saTrackText[u8T].iTitleLen;
      pcszA = pShMem->rCDText.saTrackText[u8T].szArtist;
      pcszT = pShMem->rCDText.saTrackText[u8T].szTitle;
      CDAUDIO_PRINTF_FORCED("Track%02u Title [%3d] [%s]", (unsigned int)u8T,
                            iTLen, pcszT);
      CDAUDIO_PRINTF_FORCED("        Artist [%3d] [%s]", iALen, pcszA);
    } //for(u8T = 0; u8T < CD_TRACK_ARRAY_SIZE; u8T++)
  }
  else //if(enCDAUDIO_CDTEXT_VALID == pShMem->rCDText.enValid)
  {
    CDAUDIO_PRINTF_FORCED("NO CD-TEXT available");
  } //else //if(enCDAUDIO_CDTEXT_VALID == pShMem->rCDText.enValid)
}

/*****************************************************************************
 * FUNCTION:     CDAUDIO_u32GetText
 * PARAMETER:    IN:drive index
 *               OUT: min track
 *               OUT: max track
 * RETURNVALUE:  OSAL Error
 * DESCRIPTION:  get track info from TOC
 ******************************************************************************/
tU32 CDAUDIO_u32GetText(tDevCDData *pShMem, OSAL_tSemHandle hSem, tU8 u8Track,
                        tU8 *pu8Title, tU8 *pu8Artist, tU32 u32TextBufferSize)
{
  tU32 u32Ret = OSAL_E_NOERROR;
  tU32 u32AtaCDTextSize;

  CD_CHECK_SHARED_MEM(pShMem, OSAL_E_INVALIDVALUE);

  CDAUDIO_TRACE_ENTER_U4(CDID_CDAUDIO_u32GetText, 0, pu8Title, pu8Artist, 0);

  /*check if cd-text is valid*/
  if(enCDAUDIO_CDTEXT_INVALID == pShMem->rCDText.enValid)
  { //try to read cd-text from drive
    u32AtaCDTextSize = 0;
    u32Ret = CD_SCSI_IF_u32ReadCDText(pShMem,
                                      pShMem->rCDText.au8ATARawCDTextBuffer,
                                      CDAUDIO_ATA_CDTEXT_RAW_BUFFER_LEN,
                                      &u32AtaCDTextSize);

    if((u32AtaCDTextSize >= (2 + CDAUDIO_CDTEXT_ATA_BUFFER_BYTES_PER_LINE))
       && (u32Ret == OSAL_E_NOERROR))
    { //Get CD-text from uglyATA-buffer
      CDAUDIO_PRINTF_U3("Raw-CDText-Size %u bytes",
                        (unsigned int)u32AtaCDTextSize);

      u32Ret = u32ConvertUglyATA2NiceCDText(pShMem, &pShMem->rCDText,
                                            u32AtaCDTextSize);
      if(OSAL_E_NOERROR == u32Ret)
      {
        pShMem->rCDText.enValid = enCDAUDIO_CDTEXT_VALID;
      } //if(OSAL_E_NOERROR == u32Ret)
    } //if(u32Ret == OSAL_E_NOERROR)
  } //if(enCDAUDIO_CDTEXT_INVALID == pShMem->rCDText.enValid)

  if(enCDAUDIO_CDTEXT_VALID == pShMem->rCDText.enValid)
  {
    tU32 u32Len;
    tU8 u8MinTrack = 0;
    tU8 u8MaxTrack = 0;
    (void)CD_u32CacheGetCDInfo(pShMem, hSem, &u8MinTrack, &u8MaxTrack, NULL);
    if((u8Track == 0) || ((u8Track >= u8MinTrack) && (u8Track <= u8MaxTrack)))
    {
      if(NULL != pu8Title)
      {
        u32Len = CD_MIN(u32TextBufferSize,
                        (tU32)pShMem->rCDText.saTrackText[u8Track].iTitleLen);
        memset(pu8Title, 0, u32TextBufferSize);
        memcpy(pu8Title, pShMem->rCDText.saTrackText[u8Track].szTitle, u32Len);
      } //if(NULL != pu8Title)

      if(NULL != pu8Artist)
      {
        u32Len = CD_MIN(u32TextBufferSize,
                        (tU32)pShMem->rCDText.saTrackText[u8Track].iArtistLen);
        memset(pu8Artist, 0, u32TextBufferSize);
        memcpy(pu8Artist, pShMem->rCDText.saTrackText[u8Track].szArtist,
               u32Len);
      } //if(NULL != pu8Artist)
    }
    else //if((u8Track >= u8MinTrack) && (u8Track <= u8MaxTrack))
    {
      u32Ret = OSAL_E_INVALIDVALUE;
    } //else //if((u8Track >= u8MinTrack) && (u8Track <= u8MaxTrack))
  }
  else //if(enCDAUDIO_CDTEXT_VALID == pShMem->rCDText.enValid)
  {
    u32Ret = OSAL_E_DOESNOTEXIST;
  } //else //if(enCDAUDIO_CDTEXT_VALID == pShMem->rCDText.enValid)

  CDAUDIO_TRACE_LEAVE_U4(CDID_CDAUDIO_u32GetText, u32Ret, 0, 0, 0, 0);

  return u32Ret;
}

/*****************************************************************************
 * FUNCTION:     CDAUDIO_vCDTEXTClear
 * PARAMETER:    drive index
 * RETURNVALUE:  void
 * DESCRIPTION:  deletes all cached information about a cd
 ******************************************************************************/
tVoid CDAUDIO_vCDTEXTClear(tDevCDData *pShMem)
{

  CD_CHECK_SHARED_MEM_VOID(pShMem);

  CDAUDIO_TRACE_ENTER_U4(CDID_CDAUDIO_vCDTEXTClear, 0, 0, 0, 0);

  pShMem->rCDText.enValid = enCDAUDIO_CDTEXT_INVALID;

  //clear entire CD-Text (RAW+converted)
  memset(&pShMem->rCDText, 0, sizeof(CDAUDIO_sCDText_type));

  CDAUDIO_TRACE_LEAVE_U4(CDID_CDAUDIO_vCDTEXTClear, 0, 0, 0, 0, 0);
}

/*****************************************************************************
 * FUNCTION:     CDAUDIO_u32CDTextInit
 * PARAMETER:    void
 * RETURNVALUE:  OSAL Error code
 * DESCRIPTION:  initialises CDTEXT
 ******************************************************************************/
tU32 CDAUDIO_u32CDTextInit(tDevCDData *pShMem)
{
  tU32 u32Result = OSAL_E_NOERROR;
  CD_CHECK_SHARED_MEM(pShMem, OSAL_E_INVALIDVALUE);
  CDAUDIO_TRACE_ENTER_U4(CDID_CDAUDIO_u32CDTextInit, 0, 0, 0, 0);
  // set startup values
  CDAUDIO_vCDTEXTClear(pShMem);
  CDAUDIO_TRACE_LEAVE_U4(CDID_CDAUDIO_u32CDTextInit, u32Result, 0, 0, 0, 0);

  return u32Result;
}

/*****************************************************************************
 * FUNCTION:     CDAUDIO_vCDTextDestroy
 * PARAMETER:    void
 * RETURNVALUE:  void
 * DESCRIPTION:  initialises CDTEXT
 ******************************************************************************/
tVoid CDAUDIO_vCDTextDestroy(tDevCDData *pShMem)
{
  CD_CHECK_SHARED_MEM_VOID(pShMem);
}

#ifdef __cplusplus
}
#endif
/************************************************************************
 |end of file
 |-----------------------------------------------------------------------*/
