/************************************************************* +MHDR * .MA *

 .FILE             [ ossema.cpp  ]

 .SW-COMPONENT      osal

 .DESCRIPTION
  This file contains the implementation of the class OSAL_tclMutex.

 .COPYRIGHT            [ (c) 2001 Robert Bosch GmbH, Hildesheim           ]
 .AUTHOR               [ U. Walkling                                      ]

 .HISTORY

  Date     | Name            Modification
 ----------|---------------------------------------------------------------
  18.01.02 | CM-DI/CF31      New
************************************************************** -MHDR * .ME */
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
/* Constructor of this class. */
OSAL_tclMutex::OSAL_tclMutex()
{
   /*
      Initialisation of the member variables of the class.
   */
   szName = OSAL_NULL;
   hMutexHandle = OSAL_C_INVALID_HANDLE;
}
/* Destructor of this class. */
OSAL_tclMutex::~OSAL_tclMutex()
{
   this->s32Close();
   if (OSAL_NULL != this->szName)
   {
      /*
         Delete the name string.
      */
      OSAL_vMemoryFree( (tPVoid)this->szName);
      this->szName = OSAL_NULL;
   }
}
/*
   This methode creates the Mutex. 
   If the creation of the Mutex has finished without
   an error the s32Open methode is called afterwards.
   (For more detail see OSAL Refernzhandbuch Kapitel Mutex.)

   Additional Errors or meaning:
   OSAL_E_UNKNOWN: A handle already stored in this object.
   OSAL_E_NOSPACE: There is no space to save the name.
*/
tS32 OSAL_tclMutex::s32Create( tCString coszName    /* I */,
                                   tU32 u32Option   /* I */)
{
   tS32 s32ErrorValue = OSAL_OK;
   /*
      This member stores the temporary handle of the Mutex.
   */
    OSAL_tMtxHandle hTempMutexHandle = OSAL_C_INVALID_HANDLE;
   do
   {
      if (OSAL_NULL == coszName)
      {
         /*
            There is a NULL pointer as name specified.
         */
         s32ErrorValue = OSAL_ERROR;
         OSAL_vSetErrorCode( OSAL_E_INVALIDVALUE );
         break;
      }
      if (OSAL_C_INVALID_HANDLE != this->hMutexHandle)
      {
         /*
            There is a handle already stored in this object.
         */
         s32ErrorValue = OSAL_ERROR;
         OSAL_vSetErrorCode( OSAL_E_UNKNOWN );
         break;
      }
      if (OSAL_NULL != this->szName)
      {
         /*
            Delete the name string.
         */
         OSAL_vMemoryFree( (tPVoid)this->szName);
         this->szName = OSAL_NULL;
      }
      s32ErrorValue = this->s32SaveName(coszName, this->szName );
      if (OSAL_OK == s32ErrorValue)
      {
         /*
            Call the OSAL function to create the Mutex.
         */
         s32ErrorValue = OSAL_s32MutexCreate( coszName, &hTempMutexHandle, 0);
                                                  
                                                  
      }
      else
      {
         break;
      }
      if (OSAL_ERROR == s32ErrorValue)
      {
         /*
            Delete the name string.
         */
         OSAL_vMemoryFree( (tPVoid)this->szName);
         this->szName = OSAL_NULL;
      }
      /*
         Save the object internal Values.
      */
      this->hMutexHandle = hTempMutexHandle;
   }
   while (FALSE);
   return s32ErrorValue;
}
/*
   This methode opens a Mutex.
   (For more detail see OSAL Refernzhandbuch Kapitel Mutex.)

   Additional Errors or meaning:
   OSAL_E_ALREADYEXISTS: There exists already an opened Mutex.
   OSAL_E_NOSPACE: There is no space to save the name.
*/
tS32 OSAL_tclMutex::s32Open( tCString coszName /* (I) */ )
{
   tS32 s32ErrorValue = OSAL_OK;
   /*
      This member stores the temporary handle of the Mutex.
   */
    OSAL_tMtxHandle hTempMutexHandle = OSAL_C_INVALID_HANDLE;
   /*
      This member stores temporary the name of the Mutex.
   */
   tCString szTempName = this->szName;
   do
   {
      if (OSAL_NULL == coszName)
      {
         /*
            There is a NULL pointer as name specified.
         */
         s32ErrorValue = OSAL_ERROR;
         OSAL_vSetErrorCode( OSAL_E_INVALIDVALUE );
         break;
      }
      /*
         Check whether there is already a handle saved in the object.
      */
      if (OSAL_C_INVALID_HANDLE != this->hMutexHandle)
      {
         /*
            Set the error to tell the user there exists already an 
            opened Mutex.
         */
         s32ErrorValue = OSAL_ERROR;
         OSAL_vSetErrorCode( OSAL_E_ALREADYEXISTS );
         break;
      }
      s32ErrorValue = this->s32SaveName( coszName, this->szName );
      /*
         Check whether there is already a name saved in the object.
      */
      if (OSAL_OK == s32ErrorValue)
      {
         s32ErrorValue = OSAL_s32MutexOpen( coszName, 
                                                &hTempMutexHandle );
         if (OSAL_OK == s32ErrorValue)
         {
            /*
               Save the object internal Values.
            */
            this->hMutexHandle = hTempMutexHandle;
            /*
               Delete the new name string.
            */
            OSAL_vMemoryFree( (tPVoid)szTempName );
         }
         else
         {
            /*
               Delete the new name string.
            */
            OSAL_vMemoryFree( (tPVoid)this->szName );
            /*
               Restore the last name.
            */
            this->szName = szTempName;
         }
      }
      else
      {
         /*
            Restore the last name.
         */
         this->szName = szTempName;
      }
   }
   while (FALSE);
   return s32ErrorValue;
}
/*
   This methode opens a Mutex with the name stored
   in this object.
   (This is an additional method!)

   Additional Errors or meaning:
   OSAL_E_ALREADYEXISTS: There exists already an opened Mutex.
   OSAL_E_UNKNOWN: There is no name saved in the object. 
                   (Use s32Open( tCString coszName ) )
*/
tS32 OSAL_tclMutex::s32Open()
{
   tS32 s32ErrorValue = OSAL_OK;
   do
   {
      /*
         Check whether there is already a handle saved in the object.
      */
      if (OSAL_C_INVALID_HANDLE != this->hMutexHandle)
      {
         /*
            Set the error to tell the user there exists already an 
            opened Mutex.
         */
         s32ErrorValue = OSAL_ERROR;
         OSAL_vSetErrorCode( OSAL_E_ALREADYEXISTS );
         break;
      }
      if (OSAL_NULL == this->szName)
      {
         /*
            Set the error to tell the user there is no name specified.
         */
         s32ErrorValue = OSAL_ERROR;
         OSAL_vSetErrorCode( OSAL_E_UNKNOWN );
         break;
      }
      s32ErrorValue = OSAL_s32MutexOpen( this->szName, 
                                             &this->hMutexHandle );
      if (OSAL_OK == s32ErrorValue)
      {
         /*
            Save the object internal Values.
         */
         this->hMutexHandle = OSAL_C_INVALID_HANDLE;
      }
   }
   while (FALSE);
   return s32ErrorValue;
}
/*
   This methode closes a Mutex.
   (For more detail see OSAL Refernzhandbuch Kapitel Mutex.)
*/
tS32 OSAL_tclMutex::s32Close( )
{
   tS32 s32ErrorValue = OSAL_OK;
   if (OSAL_C_INVALID_HANDLE != this->hMutexHandle)
   {
      s32ErrorValue = OSAL_s32MutexClose( this->hMutexHandle );
      if (OSAL_OK == s32ErrorValue)
      {
         this->hMutexHandle = OSAL_C_INVALID_HANDLE;
      }
   }
   return s32ErrorValue;
}
/*
   This methode deletes the Mutex. 
   Before deleting the Mutex the methode s32Close will be
   performed to close the Mutex. After performing this 
   methode this Mutex object provides no Mutex 
   functionality but it is possible to create a new Mutex 
   in the same object.
   (For more detail see OSAL Refernzhandbuch Kapitel Mutex.)
*/
tS32 OSAL_tclMutex::s32Delete( )
{
   tS32 s32ErrorValue = OSAL_OK;
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   do
   {
      if (OSAL_C_INVALID_HANDLE != this->hMutexHandle)
      {
         s32ErrorValue = this->s32Close();
      }
      if (OSAL_OK == s32ErrorValue)
      {
         /*
            Reset the value of the handel.
         */
         this->hMutexHandle = OSAL_C_INVALID_HANDLE;
      }
      else
      {
         break;
      }
      s32ErrorValue = OSAL_s32MutexDelete( this->szName );
      u32ErrorCode = OSAL_u32ErrorCode();
      if (( OSAL_OK == s32ErrorValue )
          ||
          (
             ( OSAL_ERROR == s32ErrorValue )
             &&
             ( OSAL_E_INVALIDVALUE == u32ErrorCode)
          ))
      {
         /*
            Delete the name string.
         */
         OSAL_vMemoryFree( (tPVoid)this->szName);
         this->szName = OSAL_NULL;
      }
   }
   while (FALSE);
   return s32ErrorValue;
}

/*
   This methode increments the value of the Mutex.
   (For more detail see OSAL Refernzhandbuch Kapitel Mutex.)
*/
tS32 OSAL_tclMutex::s32Unlock()
{
   return OSAL_s32MutexUnLock( this->hMutexHandle );   
}
/*
   This methode decrements the value of the Mutex and the thread
   will be blocked in case of a Mutex value 0.
   (For more detail see OSAL Refernzhandbuch Kapitel Mutex.)
*/
tS32 OSAL_tclMutex::s32Lock( OSAL_tMSecond msec )
{
   return OSAL_s32MutexLock( this->hMutexHandle,
                                 msec);
}
/* 
   The following methods are new and/or with modification 
   to the OSAL specification. 
*/
/*
   To get the name of the created Mutex this methode can be used.
   After deletion the Mutex there is no name available furthermore.
*/
tCString OSAL_tclMutex::pszGetName()
{
   return this->szName;
}
