/****************************************************************************
* FILE                      [ $Workfile:   system_kds_def.h  $ ]
* PROJECT (1st APPLICATION) [ ELeNa ]
* ---------------------------------------------------------------------------
* EXPLANATION
* Definition of KDS-Entries
*
*  Define the Entries for common usage of KDS
*  The Entries are of the Data-Type "tEntry", defined above.
*  Following structure has to be used for generating the Entry-Id
*  ==> ((unsigned char Target)<<8) + (unsigned char Type)
*  "Target" describes if the Entry is for a Module, an PCB or the whole Sample
*  and the "Type" is the Value-Type like serial number, production date and
*  so on.
*
* ---------------------------------------------------------------------------
* Last Review:
* ---------------------------------------------------------------------------
* COPYRIGHT (c) 2003 Blaupunkt GmbH, Hildesheim
* AUTHOR    [ CM-CR/EES4 Kania ]
* ---------------------------------------------------------------------------
* HISTORY AT THE END OF THE FILE
****************************************************************************/
#ifndef SYSTEM_KDS_DEF_HEADER
   #define SYSTEM_KDS_DEF_HEADER

/**************************************************************************
** #include
**************************************************************************/

/**************************************************************************
** typedef
**************************************************************************/
typedef unsigned char      tMember;
typedef unsigned short     tEntry;

/* Common Datatypes */
typedef unsigned char      tDate[6];
typedef unsigned char      tCounter[2];
typedef unsigned char      tFailure[3];
typedef unsigned short     tVersion;
typedef unsigned char      tLongString[20];
typedef unsigned char      tString10[10];
typedef unsigned char      tString13[13];
typedef unsigned char      tString14[14];

/*************************************************************************
* Type number                                                            *
* This is the Blaupunkt-device-number, which identifies the Radio device.*
* This number has 10-digits and is stored in the alpha-numeric, also     *
* ASCII-format (10 Byte).                                                *
* nnnnnnnnnn                                                             *
*                                                                        *
* ATTENTION!                                                             *
* This number should only be programmed once during startup of the radio *
* and not be overwritten in production process.                          *
*************************************************************************/
typedef unsigned char      tTypeNumber[10];

/*******************************************************************************
* Serial number                                                                *
* Each device receives a 7-digit, BCD-coded, serial number (7 Byte).           *
* 0sssssss                                                                     *
*                                                                              *
* There is an alphanumerical interpretion too, this has a size of 14 bytes     *
*                                                                              *
* This number is entered in the production in the PDB and is equal to the      *
* serial number on the barcode-sticker of the radio (is to be seen in          *
*    conjunction with the production date).                                    *
*                                                                              *
* ATTENTION!                                                                   *
* This number should only be programmed once during startup of the radio       *
* and not be overwritten in production process.                                *
*                                                                              *
* The "Production serial number" is incremented every day starting from 0'.   *
* At the moment, according to the details from the production, between 2,000   *
* and 2,700 devices are going on the assembly line every day in the location   *
* Hildesheim, and between 15,000 and 20,000 devices are being manufactured per *
* day in the location Portugal.                                                *
*******************************************************************************/
typedef unsigned char      tSerialNumber[7];
typedef unsigned char      tSerialNumberAlphanum[14];

/*********************************************************************************
* Production date of the pcb                                                     *
* At this place, the Production date is stored. This detail is written during    *
* the production of the device.                                                  *
* yymmdd                                                                         *
*                                                                                *
* The date is stored as ASCII coded digits. It is the date when the programming  *
* of Flash and EEPROM was done and not the date when the final-test of the device*
* was performed. Therefore it cannot be used as the production date requested by *
* the customer.                                                                  *
*********************************************************************************/
typedef tDate tPcbDate;

/***********************************************************************************
* Production shift                                                                 *
* This number identifies the working-shift in the production. The detail is        *
* BCD coded and occupies 1 Byte.                                                   *
* xx                                                                               *
*                                                                                  *
* The productions shift is coded in following way:                                 *
* Factory (first byte)    Shift (second byte)                                      *
* 0X  BpW                  X0  Morning-Shift                                       *
* 1X  BrgW                 X1  Afternoon-Shift                                     *
* 2X  RBMA                 X2  Night-Shift                                         *
*                                                                                  *
* Hereby the shift is ment, when the programming of the Flash and EEPROM was done. *
***********************************************************************************/
typedef unsigned char      tProdShift;

/**************************************************************************
* Count of programming cycles                                             *
* This variable is incremented during the programming (the content is     *
* updated by the production processor at the testing station). The detail *
* is BCD coded and occupies 2Byte memory space.                           *
* cc                                                                      *
*                                                                         *
**************************************************************************/
typedef tCounter           tProgCycles;

/**************************************************************************
* Count of checks                                                         *
* This variable is incremented during the checks (the content is           *
* updated by the production processor at the testing station). The detail *
* is BCD coded and occupies 2Byte memory space.                           *
* cc                                                                      *
*                                                                         *
**************************************************************************/
typedef tCounter           tCheckCycles;

/***************************************************************************
* Count of recognized failures                                             *
* This variable is incremented on every detection of error (the content is *
* updated by the production processor at the testing station). The detail  *
* is BCD coded and occupies 1Byte.                                         *
* gg                                                                       *
*                                                                          *
* This value is written in production at each testplace in following way:  *
* At the beginning of the testprogramm:                                    *
* - read content "gg" from IDB via MAUS,                                   *
* - increment value of variable "gg",                                      *
* - writte value "gg" to IDB via MAUS ... perform all test                 *
* - if all tests have been performed without any errors write the original *
*   (not incremented) value "gg" to the IDB.                               *
* With this method is possible to have the correct value even if the       *
* communication is lost during the test.                                   *
***************************************************************************/
typedef tCounter           tFailureCount;

/****************************************************************************
* Kind of last Test/Result                                                  *
* This variable stores the coded name of the error type, which had last     *
* occurred or registered (the content is coded by the production processor  *
* at the testing station). The specification of error coding is to be fixed *
* by the pro-duction. Since while drafting this specification, no other     *
* details regarding the format of these variables were known, it has been   *
* assumed that the variable is coded hexadecimal. For this data, 3 Bytes of *
* memory space have been provided.                                          *
* kkkk                                                                      *
*                                                                           *
****************************************************************************/
typedef tFailure           tLastTestFailure;

/*************************************************************************
* Count of repairs                                                       *
* This variable stores the number of repairs executed (the content is    *
* updated by the production). The detail is BCD coded and occupies 2Byte *
* memory space.                                                          *
* rr                                                                     *
*                                                                        *
* This value has to be incremented by the repair-software after a        *
* successful repair.                                                     *
*************************************************************************/
typedef tCounter           tRepairCount;

/***************************************************************************
* Kind of last repair                                                      *
* This variable stores the coded name of the last repair type (the content *
* is coded by the production at the repair station). The specification for *
* the repair coding is to be laid down by the production. At the time of   *
* preparing this specification, no details were available with regard to   *
* the format of these variables, therefore it has been assumed that the    *
* variable has been hexadecimal-coded. For this detail, 2Byte-memory space *
* is provided.                                                             *
* aaaa                                                                     *
*                                                                          *
* This value has to be written by the repair-software after a succesful    *
* repair. The value should be coded cor-responding to the list of          *
* repaircodes in each factory.                                             *
***************************************************************************/
typedef tFailure           tLastRepairKind;

/*****************************************************************************
* Version of hardware                                                        *
* This variable stores the Hardware version number of                        *
* the device.  The hardware status can be coded project-specifically by a    *
* resistor chain on the PCB.  If the hardware coding is done, the Version    *
* number is automatically read once and stored during the first read request *
* by the IDB-Manager.  The repeat read process can be initiated by filling   *
* the variables with the character 0xAA.  But if no project-specific         *
* HW-coding is provided, this variable is to be either filled with the       *
* character 0xAA (not known) or written by the production processor.  The    *
* detail is BCD coded and occupies 2Byte-memory space.                       *
* hhhh                                                                       *
*                                                                            *
* ATTENTION!                                                                 *
* While supplying software, this variable should be filled generally with    *
* the character 0xAA.                                                        *
*****************************************************************************/
typedef tVersion           tHwVersion;

/**************************************************************************
* Version of download file                                                *
* The variable stores the Version number of the download file, with which *
* the last virgin/service update was done.                                *
* dddd                                                                    *
*                                                                         *
* The date is stored BCD coded.                                           *
**************************************************************************/
typedef tVersion           tDlVersion;

/****************************************************************************
* Download file date                                                        *
* The date on which the download file was created, is stored at this place. *
* This detail is updated during or after the download by service/virgin     *
* download.                                                                 *
* yymmdd                                                                    *
*                                                                           *
* The date is stored as ASCII coded. digits                                 *
****************************************************************************/
typedef tDate              tDlDate;

/*************************************************************************
* Version of programming software                                        *
* The variable stores the version number of the software, with which the *
* programming was done.                                                  *
* vv                                                                     *
*                                                                        *
* The detail is BCD coded. This value is directly written by the         *
* programming tool /.DLL/.EXE corresponding to the following table:      *
* 0x00...0x1F BpW/TEF Loader .DLL                                        *
* 0x20...0x3F BrgW/TEF Loader .EXE                                       *
* 0x40...0x5F CM/ESD DLL                                                 *
* 0x60...0x7F VKD Tool                                                   *
* 0x80...0x9F FlashPro-Tool                                              *
* 0xA0...0xFF reserved                                                   *
*************************************************************************/
typedef tVersion           tPsVersion;

/***************************************************************************
* Module Number                                                            *
* This byte stores the number of the production place. The details are BCD *
* coded.                                                                   *
* l/l                                                                      *
*                                                                          *
***************************************************************************/
typedef unsigned char      tModuleNumber[2];

/***************************************************************************
* Line/Program place                                                       *
* This byte stores the number of the production place. The details are BCD *
* coded.                                                                   *
* l/l                                                                      *
*                                                                          *
* This value is written after startup-Programming with the ID of the       *
* programming place. Number has to be defined by each factory              *
* corresponding to the used systematic of station-ID.                      *
***************************************************************************/
typedef unsigned char      tProductionPlace[2];

/***************************************************************************
* Code                                                                     *
* This is an 8-digit hexadecimal coded number, i.e. 8 Bytes long, which is *
* calculated individually in the producti-on for every radio device.       *
* Xxxxxxxx                                                                 *
***************************************************************************/
typedef unsigned long      tCode;

/***************************************************************************
* GPS Calibration Data                                                     *
* This structure stores the frequency measured at the stored temperature   *
* The Type for storing this data is KDS_TYPE_GPS_FREQ_TEMP                 *
***************************************************************************/
typedef struct
{
   unsigned long  u32Frequency;
   unsigned short u16Temperature_In_K;
} tGpsFreqTemp;

/***************************************************************************
* GPS Calibration Data                                                     *
* This structure stores the four Cx Values to calibrate the GPS            *
* The Values are stored in the Format "s32Fix * 10^s32Float"               *
* The Type for storing this data is KDS_TYPE_GPS_C_VALUES                  *
***************************************************************************/
typedef struct
{
   signed long  s32Fix;
   signed long  s32Float;
} tGps_C_Value;
typedef tGps_C_Value tGps_Cn_Value[4];


/***************************************************************************
* GSM control data                                                         *
* This value must be 0xff after the KDS initialisation or after repairing. *
* So the application read the GSM parameters new and write it into         *
* the KDS. In the next step the application set the value up to 0x01       *
***************************************************************************/
typedef unsigned char   tGsmValid;

/***************************************************************************
* GSM module serial number IMEI                                            *
***************************************************************************/
typedef tLongString  tGsmImeiNumber;

/***************************************************************************
* GSM SIM net identification number IMSI                                   *
***************************************************************************/
typedef tLongString  tGsmImsiNumber;

/***************************************************************************
* GSM SIM serial number                                                    *
***************************************************************************/
typedef tLongString  tGsmSimNumber;

/***************************************************************************
* GSM Phone model                                                          *
***************************************************************************/
typedef tLongString  tGsmPhoneModel;

/***************************************************************************
* GSM SIM PIN                                                              *
***************************************************************************/
typedef tDate        tGsmSimPIN;

/**************************************************************************
* Version of application file                                             *
* The variable stores the Version number of the application file.         *
* e.g. TUCC3.0v5                                                          *
* ASCII                                                                   *
**************************************************************************/
typedef unsigned char      tSwVersion[13];

/****************************************************************************
* Application file date                                                     *
* The date on which the application file was created,is stored at this place*
* yymmddhhmm                                                                *
* ASCII                                                                     *
****************************************************************************/
typedef unsigned char      tSwDate[10];


/**************************************************************************
** Macro #define
**************************************************************************/
/* Generate the whole Entry-Id with this Macro */
   #define M_KDS_ENTRY(Target,Type) ((tEntry)((((tMember)(Target))<<8) + ((tMember)(Type))))

/**************************************************************************
** #define TARGET
**************************************************************************/

/* This is common for all products */
   #define KDS_TARGET_RESERVED         (0x00)   /* USED BY SYSTEM */
   #define KDS_TARGET_DEVICE           (0x01)   /* Use for whole device info */
   #define KDS_TARGET_MAIN             (0x02)   /* Use for main pcb info */
   #define KDS_TARGET_HMI              (0x03)   /* Use for hmi device info */
   #define KDS_TARGET_TUNER            (0x04)   /* Use for tuner device info */
   #define KDS_TARGET_GPS              (0x05)   /* Use for gps device info */
   #define KDS_TARGET_PANEL            (0x06)   /* Use for panel device info */
   #define KDS_TARGET_GYRO             (0x07)   /* Use for gyro device info */
   #define KDS_TARGET_CDROM            (0x08)   /* Use for cd-drive device info */
   #define KDS_TARGET_BACKPLANE        (0x09)   /* Use for backplane device info */
   #define KDS_TARGET_GSM              (0x0A)   /* Use for GSM device info */
   #define KDS_TARGET_BLUETOOTH        (0x0B)   /* Use for bluetooth device info */
   #define KDS_TARGET_NAVIGATION       (0x0C)   /* Use for navi device info */
   #define KDS_TARGET_CUSTOMER         (0x0D)   /* Use for customer device info */

   #define KDS_TARGET_WELCOME_SCREEN   (0x0E)   /* Use for welcome-screen-data  */
   #define KDS_TARGET_ODOMETER         (0x0F)   /* Use for odometer device      */
   #define KDS_TARGET_RADIOLIB         (0x10)   /* Use for RadioLib - generalized*/  
   #define KDS_TARGET_RADIOLIB_TUN     (0x11)   /* Use for RadioLib (Tun) */
   #define KDS_TARGET_SECURITY         (0x12)   /* Use for security access parameters */
   #define KDS_TARGET_HEATCTRL         (0x13)   /* Use for heat control parameters */
   #define KDS_TARGET_DIAGNOSE         (0xA0)   /* Use for diagnose             */
   
   #define KDS_TARGET_RADIOLIB_ADR     KDS_TARGET_RADIOLIB   /* Use for RadioLib - keep it backward compatible*/

/* to be continued... */
/* Reserved space for future common Target-ID's */

/* Now comes the space for product specific data */
   #define KDS_TARGET_KONFIGUARTION_START    (0x80)

/* This is an section for special secured data. Use this section
   if the entry should not be accessible in iromer.exe */
   #define KDS_TARGET_SECURE_DATA_START      (0xF0)
   #define KDS_TARGET_SECURE_DATA_END        (0xF8)

/* to be continued in projects... */

/**************************************************************************
** #define TYPE
**************************************************************************/


   /*********************************/
   /* KDS_TARGET_DEVICE (0x01)      */
   /*********************************/
   #define KDS_TYPE_SW_VERSION                           (0x00)  /* Version of SW */
   #define KDS_TYPE_SW_DATE                              (0x01)  /* Date of SW */
   #define KDS_TYPE_SW_NUMBER                            (0x02)  /* Type number of SW */
   #define KDS_TYPE_PART_NUMBER                          (0x03)  /* Module number */
   #define KDS_TYPE_CODE                                 (0x04)  /* Security Code */
   #define KDS_TYPE_TYP_NUMBER                           (0x05)  /* Type number of part */
   #define KDS_TYPE_SERIAL_NUMBER                        (0x06)  /* Serial number of device */
   #define KDS_TYPE_PROD_DATE                            (0x07)  /* Production date */
   #define KDS_TYPE_PROD_SHIFT                           (0x08)  /* Production shift */
   #define KDS_TYPE_COUNT_CHECKS                         (0x09)  /* Count of checks */
   #define KDS_TYPE_REC_FAILURE                          (0x0A)  /* Count of recognized failures */
   #define KDS_TYPE_LAST_FAILURE                         (0x0B)  /* Kind of last failure */
   #define KDS_TYPE_COUNT_REPAIR                         (0x0C)  /* Count of repairs */
   #define KDS_TYPE_LAST_REPAIR                          (0x0D)  /* Kind of last repair */
   #define KDS_TYPE_HW_VERSION                           (0x0E)  /* Version of HW */
   #define KDS_TYPE_VER_DOWNLOAD                         (0x0F)  /* Version of download program */
   #define KDS_TYPE_DOWNLOAD_DATE                        (0x10)  /* Date of download */
   #define KDS_TYPE_NOSD                (0x11)  /* Number of service downloads */
   #define KDS_TYPE_VER_PROG_SW                          (0x12)  /* Version programmed SW */
   #define KDS_TYPE_LINE_PLACE                           (0x13)  /* Production line */
   #define KDS_TYPE_ADC_REFERENCE_VOL                    (0x14)  /* ADC Reference Voltage for Diagnostic (Diaglog) */

   #define KDS_TYPE_BURN_IN_FLAG                         (0xA0)  /* Flag to start the burn in test after next power cycle */
   #define KDS_TYPE_BURN_IN_RUNS                         (0xA1)  /* Number of max. runs for the next burn in test */

   /*********************************/
   /* KDS_TARGET_MAIN (0x02)        */
   /*********************************/


   /*********************************/
   /* KDS_TARGET_HMI (0x03)         */
   /*********************************/


   /*********************************/
   /* KDS_TARGET_TUNER (0x04)       */
   /*********************************/


   /*********************************/
   /* KDS_TARGET_GPS (0x05)         */
   /*********************************/
   #define KDS_TYPE_GPS_FREQ_TEMP                        (0x14)  /* GPS Frequency/Temperature cal. values */
   #define KDS_TYPE_GPS_C_VALUES                         (0x15)  /* GPS Cn cal. values */
   #define KDS_TYPE_GPS_ANTENNA                          (0x24)  /* Antenna current calculation coeffs and limits */
   #define KDS_TYPE_GPS_CONF_1                           (0x25)  /* GPS configuration data part 1*/
   #define KDS_TYPE_GPS_CONF_2                           (0x26)  /* GPS configuration data part 2*/

   /*********************************/
   /* KDS_TARGET_PANEL (0x06)       */
   /*********************************/

   #define KDS_TYPE_CONTRAST_BASE                        (0xA2)  /* Contrast Base values */
   #define KDS_TYPE_CONTRAST_BASE_DIAG                   (0xA3)  /* Contrast Base values seperated values store for Diagnosis*/

   /*********************************/
   /* KDS_TARGET_GYRO (0x07)        */
   /*********************************/


   /*********************************/
   /* KDS_TARGET_CDROM (0x08)       */
   /*********************************/


   /*********************************/
   /* KDS_TARGET_BACKPLANE (0x09)   */
   /*********************************/


   /*********************************/
   /* KDS_TARGET_GSM (0x0A)         */
   /*********************************/
   #define KDS_TYPE_GSM_SIM_PIN                          (0x16)  /* SIM card PIN */
   #define KDS_TYPE_GSM_IMEI                             (0x17)  /* GSM Modul Seriennummer IMEI */
   #define KDS_TYPE_GSM_IMSI                             (0x18)  /* SIM card net identification number IMSI */
   #define KDS_TYPE_GSM_SIM_NUMBER                       (0x19)  /* SIM phone number */
   #define KDS_TYPE_GSM_SCID                             (0x1A)  /* SIM card ID */


   /*********************************/
   /* KDS_TARGET_CUSTOMER (0x0D)    */
   /*********************************/

   #define KDS_TYPE_SPEEDLOCK                            (0x21)
   #define KDS_TYPE_ECU_DIAGNOSTIC_ADDRESS               (0xC0)  /* Diag Customer Data */
   #define KDS_TYPE_ADG2_MCU_VERSION                     (0xC1)
   #define KDS_TYPE_ADG2_EEPROM_VERSION                  (0xC2)
   #define KDS_TYPE_SYSTEM_SUPPLIER_ID                   (0xC3)
   #define KDS_TYPE_SYSTEM_NAME                          (0xC4)
   #define KDS_TYPE_DIAGNOSTIC_DATA_IDENTIFIER           (0xC5)
   #define KDS_TYPE_END_MODEL_PART_NUMBER                (0xC6)
   #define KDS_TYPE_BASE_MODEL_PART_NUMBER               (0xC7)
   #define KDS_TYPE_END_MODEL_PART_NUMBER_ALPHA_CODE     (0xC8)
   #define KDS_TYPE_BASE_MODEL_PART_NUMBER_ALPHA_CODE    (0xC9)

   #define KDS_TYPE_CUST_DATA_PROJECT                    (0xCA)  /* Diag Customer Data - project specific */
   /* reserverd 0xD0...0xFF for KDSEntryLength.h */


   /*********************************/
   /* KDS_TARGET_ODOMETER (0x0F)    */
   /*********************************/
   #define KDS_TYPE_ODOM_CHANNEL                         (0x23)  /* Channel: OvC, OvL */


   /*********************************/
   /* KDS_TARGET_RADIOLIB (0x10)*/
   /*********************************/
   #define KDS_TYPE_ADR_MAN_PROT1                        (0x00)  /* 32 Bytes gro (als Byte-Array) - ADR-Chip*/ 
   #define KDS_TYPE_ADR_MAN_PROT2                        (0x01)  /* 96 Bytes gro (als Byte-Array) - ADR-Chip*/  

   #define KDS_TYPE_HIT_ALIGNMENT                        (0x03)  /* 16 Bytes gro (als Byte-Array) - HIT-CHIP*/ 


   /*********************************/
   /* KDS_TARGET_RADIOLIB_TUN (0x11)*/
   /*********************************/

   #define KDS_TYPE_TUN_PHANTOM_ALIGNMENT (0x00) /* C0, C1 and C2 and current limits for OPEN, OVERCURRENT and SHORT */
   #define KDS_TYPE_TUN_PHANTOM_THRESHOLDS (0x01) /* Thresholds for phantom current OPEN, OVERCURRENT and SHORT */
   #define KDS_TYPE_DAB_PHANTOM_ALIGNMENT (0x02) /* approximation coefficient C0, C1 and C2 of DAB antenna circuit alignment */
   #define KDS_TYPE_DAB_PHANTOM_THRESHOLDS (0x03) /* Thresholds for phantom current OPEN, OVERCURRENT and SHORT of DAB antenna */
   
   /*********************************/
   /* KDS_TARGET_SECURITY (0x12)      */
   /*********************************/

   #define KDS_TYPE_DEVICE_THEFTPROT_PIN_CERTIFICATE     (0x01)
   #define KDS_TYPE_DEVICE_THEFTPROT_AES_KEY             (0x02)
   #define KDS_TYPE_DEVICE_THEFTPROT_PIN_CODE            (0x03)

   #define KDS_TYPE_SDCARD_THEFTPROT_CID_0               (0x20)
   #define KDS_TYPE_SDCARD_THEFTPROT_CID_1               (0x21)
   #define KDS_TYPE_SDCARD_THEFTPROT_CID_2               (0x22)

   #define KDS_TYPE_ECC_PUBLIC_KEY                       (0x14)
   #define KDS_TYPE_AES_KEY                              (0x15)
   #define KDS_TYPE_NAVTEQ_AES_KEY                       (0x16)
   #define KDS_TYPE_RBKEY                                (0x17)

   /*********************************/
   /* KDS_TARGET_HEATCTRL (0x13)    */
   /*********************************/
   #define KDS_TYPE_HEAT_TEMP_SHUTDOWN                   (0x01) // 16 Bit Temp where device should shutdown
   #define KDS_TYPE_HEAT_FAN_ON                          (0x02) // 16 Bit Temp where fan should start rotating on lowest speed
   #define KDS_TYPE_HEAT_FAN_OFF                         (0x03) // 16 Bit Temp where fan should stop rotating
   #define KDS_TYPE_HEAT_FAN_FULL_SPEED                  (0x04) // 16 Bit Temp where fan should rotate on full speed


   /*********************************/
   /* KDS_TARGET_DIAGNOSE (0xA0)    */
   /*********************************/
//gib2hi+: commented out the following multiple defined defines, because they are already defined under TARGET_DEVICE
//   #define KDS_TYPE_SW_VERSION                           (0x00)  /* Version of SW */
//   #define KDS_TYPE_SW_DATE                              (0x01)  /* Date of SW */
//   #define KDS_TYPE_SW_NUMBER                            (0x02)  /* Type number of SW */
//   #define KDS_TYPE_PART_NUMBER                          (0x03)  /* Module number */
//   #define KDS_TYPE_CODE                                 (0x04)  /* Security Code */
//   #define KDS_TYPE_TYP_NUMBER                           (0x05)  /* Type number of part */
//   #define KDS_TYPE_SERIAL_NUMBER                        (0x06)  /* Serial number of device */
//   #define KDS_TYPE_PROD_DATE                            (0x07)  /* Production date */
//   #define KDS_TYPE_PROD_SHIFT                           (0x08)  /* Production shift */
//   #define KDS_TYPE_COUNT_CHECKS                         (0x09)  /* Count of checks */
//   #define KDS_TYPE_REC_FAILURE                          (0x0A)  /* Count of recognized failures */
//   #define KDS_TYPE_LAST_FAILURE                         (0x0B)  /* Kind of last failure */
//   #define KDS_TYPE_COUNT_REPAIR                         (0x0C)  /* Count of repairs */
//   #define KDS_TYPE_LAST_REPAIR                          (0x0D)  /* Kind of last repair */
//   #define KDS_TYPE_HW_VERSION                           (0x0E)  /* Version of HW */
//   #define KDS_TYPE_VER_DOWNLOAD                         (0x0F)  /* Version of download program */
//   #define KDS_TYPE_DOWNLOAD_DATE                        (0x10)  /* Date of download */
//   #define KDS_TYPE_NOSD                                 (0x11)  /* Number of service downloads */
//   #define KDS_TYPE_VER_PROG_SW                          (0x12)  /* Version programmed SW */
//   #define KDS_TYPE_LINE_PLACE                           (0x13)  /* Production line */
  
//gib2hi-

   #define KDS_TYPE_SERIAL_NUMBER_ALPHANUM               (0xB0)  /* Diag Produktion Data (14)*/
   
   #define KDS_TYPE_PROD_DATA_PROJECT                    (0xB1)  /* Diag Produktion Data - project specific */
   /* reserverd 0xD0...0xFF for KDSEntryLength.h */


   /*********************************/
   /* UNKNOWN                       */
   /*********************************/

   #define KDS_TYPE_DEVICE                               (0x20)  /* device E1, E2, ... */
   #define KDS_TYPE_DEVELOPER_DEVICE                     (0x22)
   #define KDS_TYPE_CARTYPE                              (0xA4)  /* Car type */
   #define KDS_TYPE_HW_ID                                (0xA5)  /* HW variante required for SPI */




/**************************************************************************
** #define TYPE for: KDS_TARGET_WELCOME_SCREEN
**************************************************************************/
   #define KDS_TYPE_WELCOME_SCREEN_DATA0 (0x00) /* data package 0  */
   #define KDS_TYPE_WELCOME_SCREEN_DATA1 (0x01) /* data package 1  */

#endif

/****************************************************************************
* FILE                      [ $Workfile:   system_kds_def.h  $ ]
* ---------------------------------------------------------------------------
* GENERAL :
* $Header:   //hi230124/projects/Vasco/swnavi/archives/products/system/system_kds_def.h-arc   1.11   Jul 26 2004 11:01:54   prw2hi  $
* ---------------------------------------------------------------------------
* HISTORY
* REV.  DATE        TIME                   AUTHOR
* COMMENT
* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
* $Revision:   1.11  $  $Date:   Jul 26 2004 11:01:54  $  $Author:   prw2hi  $
* $Log:   //hi230124/projects/Vasco/swnavi/archives/products/system/system_kds_def.h-arc  $
 * 
 *    Rev 1.11   Jul 26 2004 11:01:54   prw2hi
 * Add define KDS_TARGET_DIAGNOSE 
 * and the diagnosis defines:
 * KDS_TYPE_BURN_IN_FLAG
 * KDS_TYPE_BURN_IN_RUNS
 * KDS_TYPE_CONTRAST_BASE
 * KDS_TYPE_CONTRAST_BASE_DIAG
 * 
 *    Rev 1.10   Jul 15 2004 14:14:20   xub2hi
 * added entry for odovcan
 * 
 *    Rev 1.9   May 07 2004 15:44:02   RAT2HI
 * Changed entry for TUCC
 * 
 *    Rev 1.8   Mar 16 2004 17:25:58   PEC2HI
 * added developer_device
 * 
 *    Rev 1.7   Mar 04 2004 15:17:10   KAN2HI
 * - New "secured" KDS-Section. Access with IROMER.EXE is locked by default.
 * - Some TUCC-Entries added from TM.
 *
 *    Rev 1.6   Feb 24 2004 16:47:44   PEC2HI
 * added speedlock
 *
 *    Rev 1.5   Feb 10 2004 14:40:02   KAN2HI
 * New define for Target-Version
 *
 *    Rev 1.4   Dec 08 2003 09:53:30   SJC2HI
 * some new typdefs for SwVersion and SwDate (TUCC project)
 *
 *    Rev 1.3   03 Nov 2003 16:51:52   IR62HI
 * TUCC Feature Varianten
 *
 *    Rev 1.2   Oct 28 2003 14:55:36   SJC2HI
 * some new defince for TUCC / GSM parameters
 *
 *    Rev 1.1   Sep 02 2003 13:32:50   pkd2hi
 * add defines for WelcomeScreen
 *
 *    Rev 1.0   Jun 19 2003 11:35:46   TND2HI
 * Initial revision.
 *
 *    Rev 1.3   Jun 16 2003 11:08:26   KAN2HI
 * Typedefs changed for better matching with mcnet
 *
 *    Rev 1.2   May 16 2003 16:42:18   KAN2HI
 * own datatypes changed to ansi types
 *
 *    Rev 1.1   May 09 2003 17:32:30   KAN2HI
 * New definitions for GPS
 *
 *    Rev 1.0   Mar 26 2003 15:16:58   KAN2HI
 * Initial revision.
 *
 *    Rev 1.1   Feb 12 2003 14:53:54   kan2hi
 * Common KDS-Definitions added
 *
 *    Rev 1.0   Feb 03 2003 15:55:16   kan2hi
 * Initial revision.
****************************************************************************/

