/******************************************************************************
* \file               AGW_SPS_DBusHandler.cpp
*******************************************************************************
\verbatim
PROJECT:        Seamless pairing
SW-COMPONENT:   Seamless pairing server
DESCRIPTION:    AGW dBushandler
COPYRIGHT:      &copy; RBEI
HISTORY:
Date       | Author                   | Modifications
27.12.2017 | svs7kor                  | Initial Version
\endverbatim
******************************************************************************/

#include "AGW_SPS_DBusHandler.h"
#include "sps_connection.h"
#include "App_MessageHandler.h"
#include "JSON_MessageHandler.h"

DLT_DECLARE_CONTEXT(AGWDBus);

#define AGW_SPS_BUS        "com.bosch.AutomotiveProxy"
#define AGW_SPS_INTERFACE  "com.bosch.AutomotiveProxy.SeamlessPairingServer"
#define AGW_SPS_OBJPATH    "/com/bosch/AutomotiveProxy/SeamlessPairingServer" 

static guint g_watcher_id = 0;
static GDBusProxy *pProxyObject = NULL;
static GDBusConnection* g_connection = NULL;

/******************************************************************************
** FUNCTION   : cb_start_spserver
*******************************************************************************
* \fn     cb_start_spserver
* \brief  Call back on signal start_spserver
* \param  GObject *object, GParamSpec *param, gpointer user_data
* \retval void.
******************************************************************************/
void cb_start_spserver( GDBusConnection *connection,
                        const gchar *sender_name,
                        const gchar *object_path,
                        const gchar *interface_name,
                        const gchar *signal_name,
                        GVariant *parameters,
                        gpointer user_data )

                        
{
      DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
      if(!parameters)
      {
          DLT_LOG(AGWDBus, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                 DLT_STRING("(): Null params"));
          vUpdateSPConnectionState(SPS_START_FAILED);
          return;
      }
      GVariant *connInfo = NULL;
      g_variant_get(parameters, "(@a{sv})", &connInfo);
      if(!connInfo)
      {
          DLT_LOG(AGWDBus, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                 DLT_STRING("(): Null connection info"));
          vUpdateSPConnectionState(SPS_START_FAILED);
          return;
      } 

      GVariantIter l_iter;
      GVariant *value;
      gchar *key;
      gsize len = 0;
      guint32 port = 0;
      gchar* ipAddr = NULL;

      g_variant_iter_init (&l_iter, connInfo);

      while (g_variant_iter_next (&l_iter, "{sv}", &key, &value))
      {
            gsize l_len = 0;

            if(!g_strcmp0 (key, "ip_address"))
            {
                  ipAddr = g_variant_get_string(value, &l_len);
                  DLT_LOG(AGWDBus,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                         DLT_STRING("():IP Address: "), DLT_STRING(ipAddr));
            }
            else if(!g_strcmp0 (key, "port"))
            {
                  port = g_variant_get_uint32(value);
                  DLT_LOG(AGWDBus, DLT_LOG_INFO,DLT_STRING(__FUNCTION__), 
                        DLT_STRING("():Port: "),  DLT_UINT(port));
            }
            g_variant_unref (value);
            g_free (key);
      }  
      g_variant_unref(connInfo);
 
      if( (ipAddr != NULL) && (port) && (bOpenSPConnection(port)))
      {
           vUpdateSPConnectionState(SPS_START_SUCCESS);
      }
      else
      {
            DLT_LOG(AGWDBus, DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                     DLT_STRING("(): Failed starting the server"));
            vUpdateSPConnectionState(SPS_START_FAILED);   
      }         
      
      DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("OUT "),DLT_STRING(__FUNCTION__)); 
}

/******************************************************************************
** FUNCTION   : cb_stop_spserver
*******************************************************************************
* \fn     cb_stop_spserver
* \brief  Call back on signal stop_spserver
* \param  GObject *object, GParamSpec *param, gpointer user_data
* \retval void.
******************************************************************************/
void cb_stop_spserver( GDBusConnection *connection,
                       const gchar *sender_name,
                       const gchar *object_path,
                       const gchar *interface_name,
                       const gchar *signal_name,
                       GVariant *parameters,
                       gpointer user_data )
{
      DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
      //stop listeining over secure port
      vCloseSPConnection();

      guint16 pairStatus; //0 :success ,1:failure
      gchar*  certPath;  
      if(!parameters)
      {
          DLT_LOG(AGWDBus, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                 DLT_STRING("(): Null params"));
          return;
      }
      g_variant_get(parameters, "(qs)", &pairStatus, &certPath );
		
	DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("Pair status: "),DLT_UINT(pairStatus));
    DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("Cert path: "),DLT_STRING(certPath)); 
	if((!certPath) || (pairStatus))
	{
		DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("certpath is empty or pairstatus is failed")); 
		return;
	}
	else{

		DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("set smartphone credentials to MQTT")); 
		vSet_SphoneMQTT_credentials(certPath);
	}	
	if(certPath)	
      g_free(certPath);
  
    DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("OUT "),DLT_STRING(__FUNCTION__));	
}

/******************************************************************************
** FUNCTION   : cb_sendMessage_toApp
*******************************************************************************
* \fn     cb_sendMessage_toApp
* \brief  Call back on signal send-message-to-app
* \param  GObject *object, GParamSpec *param, gpointer user_data
* \retval void.
*******************************************************************************/
void cb_sendMessage_toApp( GDBusConnection *connection,
                           const gchar *sender_name,
                           const gchar *object_path,
                           const gchar *interface_name,
                           const gchar *signal_name,
                           GVariant *parameters,
                           gpointer user_data )
{
      DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
      if(!parameters)
      {
           DLT_LOG(AGWDBus, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                    DLT_STRING("(): Null params"));
           return;
      }
      GVariant *message = NULL;
      g_variant_get(parameters, "(@a{ss})", &message);
      if(!message)
            DLT_LOG(AGWDBus, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
                 DLT_STRING("(): NULL message"));

      else
      {
          if(!bHandleMsgToAppWrapper(message)) 
            DLT_LOG(AGWDBus, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                 DLT_STRING("(): Failed sending msg to App"));
          g_variant_unref(message);
      }

      //Once we send out a msg, keep looking for reply from App
     // vReadMsgFromApp(); 
      DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("OUT "),DLT_STRING(__FUNCTION__));
}

/******************************************************************************
** FUNCTION   : vForwardMsgFromApp
*******************************************************************************
* \fn     vForwardMsgFromApp
* \brief  Forwards the app message to the seamless midw
* \param  key, value
* \retval void
******************************************************************************/
void vForwardMsgFromApp( gchar* key, GVariant* value)
{
      DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("IN "),
                                 DLT_STRING(__FUNCTION__));
       
      if(pProxyObject == NULL)
      {
           DLT_LOG(AGWDBus,DLT_LOG_INFO,  DLT_STRING(__FUNCTION__),
               DLT_STRING("():Null proxy"));
      }
      else if((!key) || (!value))
      {
           DLT_LOG(AGWDBus,DLT_LOG_INFO,  DLT_STRING(__FUNCTION__),
              DLT_STRING("():Null input(s)"));
      }
      else
      {          
                 g_dbus_proxy_call( pProxyObject,
                                    "ProccessAppMessage",
                                    g_variant_new ("(s@a(y))",
                                         key, value),
                                    G_DBUS_CALL_FLAGS_NONE,
                                    -1,
                                    NULL,
                                    NULL,
                                    NULL
                                  );
  
      }
      DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("OUT "),
                                 DLT_STRING(__FUNCTION__));
}

/******************************************************************************
** FUNCTION   : vUpdateSPConnectionState
*******************************************************************************
* \fn     vUpdateSPConnectionState
* \brief  Updates the SP Server connection state to SP middleware
* \param  connState
* \retval void
******************************************************************************/
void vUpdateSPConnectionState( SPServerConnectionState connState )
{
      DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("IN "),
                                 DLT_STRING(__FUNCTION__));

      if(pProxyObject == NULL)
      {
           DLT_LOG(AGWDBus,DLT_LOG_INFO,  DLT_STRING(__FUNCTION__),
               DLT_STRING("():Null proxy"));
      }
      else
      {
            g_dbus_proxy_call( pProxyObject,
                               "SPSConnectionState",
                               g_variant_new ("(q)",
                                     (guint16)connState),
                               G_DBUS_CALL_FLAGS_NONE,
                               -1,
                               NULL,
                               NULL,
                               NULL
                             );

      }
      DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("OUT "),
                                 DLT_STRING(__FUNCTION__));
}

/******************************************************************************
** FUNCTION   : vGatewaySPSProxyCB
*******************************************************************************
* \fn     vGatewaySPSProxyCB
* \brief  Callback for Automotive Gateway SPS Proxy
* \param  source_object, result, user_data
* \retval void
******************************************************************************/     
void vGatewaySPSProxyCB( GObject *source_object, 
                        GAsyncResult *res,
                        gpointer user_data )
{
    DLT_LOG(AGWDBus, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
    GError *error = NULL;
    pProxyObject = g_dbus_proxy_new_finish (res, &error);

    if(!pProxyObject)
    {
        DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                   DLT_STRING("(): ProxyObject NULL"));
        if(error)
        {
             DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                   DLT_STRING("(): error = "), DLT_STRING(
                           error->message ? error->message : ""));
             g_clear_error(&error);
        } 
    }
    else 
    {
          DLT_LOG(AGWDBus,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                   DLT_STRING("(): Proxy created"));

          g_dbus_connection_signal_subscribe( g_connection,
                                              AGW_SPS_BUS,
                                              AGW_SPS_INTERFACE,
                                              "StartSPServer",
                                              AGW_SPS_OBJPATH,
                                              NULL,
                                              G_DBUS_SIGNAL_FLAGS_NONE,
                                              cb_start_spserver,
                                              NULL,
                                              NULL ); 

          g_dbus_connection_signal_subscribe( g_connection,
                                              AGW_SPS_BUS,
                                              AGW_SPS_INTERFACE,
                                              "StopSPServer",
                                              AGW_SPS_OBJPATH,
                                              NULL,
                                              G_DBUS_SIGNAL_FLAGS_NONE,
                                              cb_stop_spserver,
                                              NULL,
                                              NULL );

         g_dbus_connection_signal_subscribe(  g_connection,
                                              AGW_SPS_BUS,
                                              AGW_SPS_INTERFACE,
                                              "SendMessageToApp",
                                              AGW_SPS_OBJPATH,
                                              NULL,
                                              G_DBUS_SIGNAL_FLAGS_NONE,
                                              cb_sendMessage_toApp,
                                              NULL,
                                              NULL );

    }
    DLT_LOG(AGWDBus, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
}  
     
/******************************************************************************
** FUNCTION   : on_name_appeared
*******************************************************************************
* \fn     on_name_appeared
* \brief  Call back on DBus name appeared
* \param  GDBusConnection *connection,gchar *bus name,gchar *name_owner,
*              gpointer user_data.
* \retval void.
******************************************************************************/
void on_name_appeared(GDBusConnection *connection,
                      const gchar *name,
                      const gchar *name_owner,
                      gpointer user_data)                           
{
	DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
        DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("Bus name::"),DLT_STRING(name));

       g_dbus_proxy_new (  connection,
                           G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
                           NULL,
                           AGW_SPS_BUS,
                           AGW_SPS_OBJPATH,
                           AGW_SPS_INTERFACE,
                           NULL,
                           vGatewaySPSProxyCB,
                           NULL );

        g_connection = g_object_ref(connection);
        DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("OUT "),DLT_STRING(__FUNCTION__));
}

/******************************************************************************
** FUNCTION   : on_name_vanished
*******************************************************************************
* \fn     on_name_vanished
* \brief  Call back on DBus name vanished
* \param  GDBusConnection *connection,gchar *bus_name,
               gpointer user_data.
* \retval void.
******************************************************************************/
void on_name_vanished(GDBusConnection *connection,
                      const gchar     *name,
                      gpointer        user_data)                                
{
	DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("IN "),
                                       DLT_STRING(__FUNCTION__));
        if(pProxyObject)
        {
           g_clear_object(&pProxyObject);
        }
        if(g_connection)
        {
            g_object_unref(g_connection);
            g_connection = NULL;
        }   
	DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("OUT "),
                                       DLT_STRING(__FUNCTION__));
}

/******************************************************************************
** FUNCTION   : bStartAGWDBusHandler
*******************************************************************************
* \fn     bStartAGWDBusHandler
* \brief  Function to watch for AGW bus
* \param  None.
* \retval gboolean.
******************************************************************************/
gboolean bStartAGWDBusHandler()
{
	DLT_REGISTER_CONTEXT(AGWDBus,"AGWD","AGW Dbus handler");
        DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(
                                                     __FUNCTION__));
  //  g_type_init();
     // bOpenSPConnection(8170);
        g_watcher_id = g_bus_watch_name( G_BUS_TYPE_SESSION,
                                   AGW_SPS_BUS,    
                                   G_BUS_NAME_WATCHER_FLAGS_NONE,
                                   on_name_appeared,
                                   on_name_vanished,
                                   NULL,
                                   NULL 
                                 );

        if(g_watcher_id != 0)
        {
            DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING(
                 "Started watching bus com.bosch.AutomotiveProxy"));
            return TRUE;
        }
        else
        {
           DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING(
                  "Failed to watch bus com.bosch.AutomotiveProxy"));
           return FALSE;
        } 
}

/******************************************************************************
** FUNCTION   : vStopAGWDBusHandler
*******************************************************************************
* \fn     bStopAGWDBusHandler
* \brief  Function to stop AGWDBusHandler
* \param  None.
* \retval void
******************************************************************************/
void vStopAGWDBusHandler()
{
    DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
    if(g_watcher_id)
    {
 	 g_bus_unwatch_name(g_watcher_id);
	 //pProxyObject should be unrefed 
	 DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("No active watcher for AGW service"));
         g_watcher_id = 0;     
    }
    if(pProxyObject)
    {
        g_clear_object(&pProxyObject);
    }
    else
    {
        DLT_LOG(AGWDBus,DLT_LOG_INFO,DLT_STRING("Seamless server dbus listerner" 
                                " is alreadystopped"));
    }
    if(g_connection)
    {
          g_object_unref(g_connection);
          g_connection = NULL;
    }
    vCloseSPConnection();
}
