/*!
*******************************************************************************
* @file             : seamless_settings.c
*******************************************************************************
*  - PROJECT:       : Seamless pairing Middleware
*  - SW-COMPONENT   : SPMA
*  - DESCRIPTION    : Implementation to read seamless settings from .conf
*  - COPYRIGHT      : &copy; 2018 Robert Bosch Engineering & Business Solutions
*  - Documents      : NA
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|-----------------------
* 04.09.2018 | ipd5kor      | 1.0.0 | Intial implementation
******************************************************************************/
#include <seamless_settings.h>
#include <string.h>
#include <stdlib.h>
#include "dlt/dlt.h"

DLT_IMPORT_CONTEXT(SPM_SPL);

#define CONFIG_FILE "/etc/seamless_pair.conf"
static GKeyFile *g_keyfile = NULL;

/*********************************************************************
 * FUNCTION:    bLoadConfig
 * DESCRIPTION: Function to load key file from seamless_pair.conf 
 * PARAMETERS:  void
 * RETURNS:     TRUE, if key file loaded successfuly
                FALSE, if failed to load key file
**********************************************************************/ 
gboolean bLoadConfig()
{
    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(
                "+"),DLT_STRING( __FUNCTION__));
    GKeyFileFlags flags;
    GError *error = NULL;

    // Create a new GKeyFile object
    g_keyfile = g_key_file_new ();
    if(NULL == g_keyfile)
    {
        DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("keyfile is NULL"));
        return FALSE;
    }
    flags = G_KEY_FILE_KEEP_COMMENTS | G_KEY_FILE_KEEP_TRANSLATIONS;
    // Load the GKeyFile from keyfile.conf or return.
    if (!g_key_file_load_from_file (g_keyfile, CONFIG_FILE, flags, &error))
    {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR,
                DLT_STRING("error in loading config file: "),
                DLT_STRING(error->message));
        g_clear_error (&error);
        return FALSE;
    }
    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(
                "-"),DLT_STRING( __FUNCTION__));
    
    return TRUE;
}

/*********************************************************************
 * FUNCTION:    bGetSPMIntegerSetting
 * DESCRIPTION: Method to get a Integer type setiing from conf
 * PARAMETERS:  param, value
 * RETURNS:     TRUE,  if value fetched successfuly
                FALSE, if failed to load conf / fetch value
**********************************************************************/
gboolean bGetSPMIntegerSetting( gchar* param,
                                gint*  value)
{
    if(!param)
    {
       DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): Null param"));
       return FALSE;
    } 
    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                       DLT_STRING("():"), DLT_STRING(param));
   
    if( (g_keyfile) || (bLoadConfig()) )
    {
        GError* error = NULL;
        *value = g_key_file_get_integer( g_keyfile,
                                         "spmidw_config",
                                         param,
                                         &error 
                                       );
        if(error)
        {
           DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
             DLT_STRING("(): Failed to read setting, error:"),
               DLT_STRING(error->message ? error->message : ""));
           return FALSE;
        } 
        return TRUE;
    } 
    DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): failed to load config"));
    return FALSE;
}         

/**************************************************************************
 * FUNCTION:    bGetSPMStringSetting
 * DESCRIPTION: Method to get a string type setiing from conf
                The method will not take care to allocate or 
                deallocate memory to its paramaters. Calling function
                shall take care of assigning enough size for the 'value'
               'arg'. MAX_CONF_ITEM_SIZE is recommended size. 
 * PARAMETERS:  param, value
                Memory shall be pre-assigned to 'value' arg
 * RETURNS:     TRUE,  if value fetched successfuly
                FALSE, if failed to load conf / fetch value
***************************************************************************/                 
gboolean bGetSPMStringSetting( gchar* param,
                               gchar* value )
{
    if(!param || !value)
    {
       DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): Null param"));
       return FALSE;
    }
    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                       DLT_STRING("():"), DLT_STRING(param));

    if( (g_keyfile) || (bLoadConfig()) )
    {
        GError* error = NULL;
        gchar*  tempVal = NULL;
        tempVal = g_key_file_get_string( g_keyfile,
                                         "spmidw_config",
                                         param,
                                         &error
                                       );
        if(error) 
        {
           DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
             DLT_STRING("(): Failed to read setting, error:"),
               DLT_STRING(error->message ? error->message : ""));
           if(tempVal)
              free(tempVal);  
           return FALSE;
        }
        if(!tempVal)
        {
            DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("():Null value received"));
            return FALSE;
        }

        strcpy(value, tempVal);
        free(tempVal);
        return TRUE;
    }
    DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): failed to load config"));
    return FALSE;
}         

/*********************************************************************
 * FUNCTION:    vUnloadConfig
 * DESCRIPTION: Function to unload key file 
 * PARAMETERS:  void
 * RETURNS:     void
**********************************************************************/
void vUnloadConfig()
{
    DLT_LOG(SPM_SPL, DLT_LOG_INFO,DLT_STRING(
                "+"),DLT_STRING( __FUNCTION__));
    if(g_keyfile != NULL)
    {
        g_key_file_free (g_keyfile);
        g_keyfile = NULL;
    }
    DLT_LOG(SPM_SPL, DLT_LOG_INFO,DLT_STRING(
                "-"),DLT_STRING( __FUNCTION__));
    
}

    
