/****************************************************************************
* FILE:          seamless_pairing.c 
* PROJECT:       Seamless Pairing For IPCM
* DESCRIPTION:   Seamless pairing session control logic
* AUTHOR:        IPD5KOR
* COPYRIGHT:     (c) 2018 Robert Bosch GmbH, Hildesheim
****************************************************************************/
#include "seamless_aes_sessionID.h"
#include "seamless_controller.h"
#include "Seamless_WBL_DBusHandler.h"
#include "SeamlessMidw_SPS_DBusHandler.h"
#include "SeamlessMidw_HMI_enums.h"
#include "SeamlessMidw_HMI_DBusHandler.h"
#include "seamless_data.h"
#include "seamless_config_interface.h"
#include "TrustedDeviceList_Handler.h"
#include "dlt/dlt.h"
#include "qrcode.h"
#include "certificateHandler.h"
#include "seamless_settings.h"
#include "BTSeamlessControllerWrapper.h"
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <glib.h>
#include <gio/gio.h>
#include <fcntl.h>
#include <time.h>
#include <ctype.h>
#include <dirent.h>

DLT_IMPORT_CONTEXT(SPM_SPL);

#define SP_IPV4_ADDRESS "10.19.229.1"
#define CERT_HASH_ALGORITHM "SHA256"
#define SP_ALL_PARAMS_RECEIVED   0x011111
#define SP_TRUSTED_DEVICE_ADDED  0x111111

#define STUNNEL_PORT "8170"
#define TRUSTED_CERT_PATH "/var/opt/bosch/dynamic/seamlesspairing/TrustedDeviceCerts"
#define MAX_CERT_NAME_LEN   (MAX_CONF_SIZE + 32)
#define MAX_BT_ADDR_LEN 64
#define MAX_BT_NAME_LEN 255
#define MAX_QR_FILENAME_LEN 100

extern char* arc4random_buf(unsigned char* input, size_t length);
extern  gboolean  prepareQrBmp(QRCode qrcode, char* fileName);

static gboolean g_bPairingInProgress = FALSE;   
static QRData   g_qr_data;
static CurrentDeviceInfo g_current_device;
static guint    g_uTimerID = 0;
static gboolean g_bQRGenerated = FALSE;
static gboolean g_bCertExtracted = FALSE;
static gboolean g_bSmartPhoneConnected = FALSE;
static unsigned  long int g_pairInfoState = 0x000000;
static gboolean g_bFinalizePairing = FALSE;
static gboolean g_bPhoneAckSent = FALSE;
static gboolean g_bSeamlessBTInfoSent = FALSE;
static SPDeviceUpdateStatus g_seamlessPairStatus = SP_COMPLETED;
static SPDeviceUpdateErrCode g_seamlessPairErrorCode = SP_ERR_UNKNOWN_ERROR;
static WiFiAPState g_iCurrentWiFiState = WIFI_AP_INACTIVE;

/*****************************************************************************
* Function:    vStartSeamlessPairing
* Description: This method intitiates the seamless pairing mechanism
* Parameters:  void
* Returns:     void
****************************************************************************/
void vStartSeamlessPairing()
{
     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

     vResetData();
     g_bPairingInProgress = TRUE;

     if(!bGetHUID(g_qr_data.huid))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
             DLT_STRING("():Error fteching HUID"));
         vCloseSeamlessSession(SP_COMPLETED, SP_ERR_INTERNAL_ERROR);
         return ;
     }
          
     if(!bGenerateSessionID(g_qr_data.hex_session_id))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
              DLT_STRING("():Error generating session ID"));
         vCloseSeamlessSession(SP_COMPLETED, SP_ERR_INTERNAL_ERROR);
         return ;
     } 

     if(!bGenerateWPAKey())
     {
         DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
             DLT_STRING("():Error generating wpa key"));
         vCloseSeamlessSession(SP_COMPLETED, SP_ERR_INTERNAL_ERROR);
         return ;
     }
 
     if(!bGenerateSSID())
     {
         DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("():Error generating SSID"));
         vCloseSeamlessSession(SP_COMPLETED, SP_ERR_INTERNAL_ERROR);
         return ;
     }
     
     if(!g_bCertExtracted)
         g_bCertExtracted= bExtractCertInfo();

     if((!g_bCertExtracted) || (!bFetchCertInfo()))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("():Error fetching cert info"));
         vCloseSeamlessSession(SP_COMPLETED, SP_ERR_INTERNAL_ERROR);
         return ;
     }         
    
     gchar port[6] = { '\0' }; 
     gchar address[MAX_CONF_SIZE] = { '\0' };
     if(!bGetSPMStringSetting("stunnel_port", port))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
             DLT_STRING("():Failed fetching port from conf"));
         // use default port
         strcpy(port, STUNNEL_PORT); 
     }         

     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
             DLT_STRING("():port = "), DLT_STRING(port));

     if(!bGetSPMStringSetting("sp_network_address", address))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
             DLT_STRING("():Failed fetching address from conf"));
         //use default
         strcpy(address, SP_IPV4_ADDRESS);   
     }

     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
             DLT_STRING("():IP = "), DLT_STRING(address));


     strcpy(g_qr_data.ipv4, SP_IPV4_ADDRESS);
     strcpy(g_qr_data.port, port); 
     strcpy(g_qr_data.hashAlgorithm, CERT_HASH_ALGORITHM);

     if(!bCreateTempWiFiSetup(g_qr_data.ssid, g_qr_data.wpa_key))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
             DLT_STRING("():Error requesting temp AP"));
         vCloseSeamlessSession(SP_COMPLETED, SP_ERR_WIFI_ERROR);
         return ;
     }

     GVariant* connInfo = pGetServerConnInfo();       

     if(!bStartSeamlessPairingServer(connInfo))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("():Error starting the seamless server"));
         vCloseSeamlessSession(SP_COMPLETED, SP_ERR_SERVER_ERROR);
         return ;
     }        
     vStartSeamlessTimer();

     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
     return ;
}

/*****************************************************************************
* Function:    vResetData
* Description: This method initializes the seamless pairing
               data items
* Parameters:  void
* Returns:     void
****************************************************************************/
void vResetData()
{
     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 

     memset(g_qr_data.wpa_key, '\0', sizeof(g_qr_data.wpa_key));
     memset(g_qr_data.hex_session_id, '\0', sizeof(g_qr_data.hex_session_id));
     memset(g_qr_data.ssid, '\0', sizeof(g_qr_data.ssid));
     memset(g_qr_data.huid, '\0', sizeof(g_qr_data.huid));  
     memset(g_qr_data.hashAlgorithm, '\0' , sizeof(g_qr_data.hashAlgorithm));
     memset(g_qr_data.issuerKeyHash, '\0' , sizeof(g_qr_data.issuerKeyHash));
     memset(g_qr_data.issuerNameHash, '\0' , sizeof(g_qr_data.issuerKeyHash));
     memset(g_qr_data.certificateHash, '\0' , sizeof(g_qr_data.certificateHash));
     memset(g_qr_data.serialNumber, '\0' , sizeof(g_qr_data.serialNumber));
     memset(g_qr_data.ipv4, '\0', sizeof(g_qr_data.ipv4));
     memset(g_qr_data.port, '\0', sizeof(g_qr_data.port));
     
     g_qr_data.serialNumberLen = 0; 
     g_pairInfoState = 0x000000;
     g_bPairingInProgress = FALSE;
     g_bSmartPhoneConnected = FALSE;
     g_bFinalizePairing = FALSE;
     g_bQRGenerated = FALSE;
     g_bPhoneAckSent = FALSE;
     g_bSeamlessBTInfoSent = FALSE;

     if(g_uTimerID)
        g_source_remove(g_uTimerID);
     g_uTimerID = 0;
 

     memset(g_current_device.device_name, '\0',
                     	   sizeof(g_current_device.device_name));
     memset(g_current_device.psk, '\0', sizeof(g_current_device.psk)); 
     memset(g_current_device.ssid, '\0', sizeof(g_current_device.ssid));
     memset(g_current_device.bssid, '\0', sizeof(g_current_device.bssid));
     memset(g_current_device.pair_time, '\0', 
                                    sizeof(g_current_device.pair_time)); 
    memset(g_current_device.serialnumber, '\0',
           sizeof(g_current_device.serialnumber));
     if(g_current_device.cert)
         free(g_current_device.cert);
     g_current_device.cert = NULL; 
     g_current_device.certLength = 0; 

     if(g_current_device.stunnelCertName)
          free(g_current_device.stunnelCertName);
     g_current_device.stunnelCertName = NULL;
                         
     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/************************************************************************
* Function:    bIsPairingInProgress
* Description: This method tells whether pairing
               is in progress or idle 
* Parameters:  void
* Return:      gboolean
*************************************************************************/
gboolean bIsPairingInProgress()
{
     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
         DLT_STRING(g_bPairingInProgress ?  "(): Pairing in progress" :
                           "No pairing ongoing, idle"));
     return g_bPairingInProgress;
}

/***************************************************************************
* Function:    bFetchCertInfo
* Description: This method fetches the certificate details
               for QR code generation
* Parameters:  void
* Return:      gboolean
****************************************************************************/
gboolean bFetchCertInfo()
{
     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
     unsigned char hashData[HASH_LEN] = { 0 };
     if(!bGetCertificateHash(hashData))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
           DLT_STRING("(): bGetCertificateHash failed"));
         return FALSE;
     }
     
     bByteToHexStrConvert(hashData, (HASH_LEN / 2), g_qr_data.certificateHash);
     memset(hashData, 0, HASH_LEN);
                                 
     if(!bGetKeyHash(hashData))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
           DLT_STRING("(): bGetKeyHash failed"));
         return FALSE;
     }
     bByteToHexStrConvert(hashData, (HASH_LEN / 2), g_qr_data.issuerKeyHash);
     memset(hashData, 0, HASH_LEN);
 
     if(!bGetIssuerNameHash(hashData))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
           DLT_STRING("(): bGetKeyHash failed"));
         return FALSE;
     }
     bByteToHexStrConvert(hashData, (HASH_LEN / 2), g_qr_data.issuerNameHash);
     memset(hashData, 0, HASH_LEN);

     if(!bGetSerialNumber(hashData, &g_qr_data.serialNumberLen))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
           DLT_STRING("(): bGetKeyHash failed"));
         return FALSE;
     }
     bByteToHexStrConvert(hashData, g_qr_data.serialNumberLen,
                                       g_qr_data.serialNumber);

     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
     return TRUE;
}
            
/***************************************************************************
* Function:    bGenerateWPAKey
* Description: This method generates the wpa key for Wi-Fi AP
* Parameters:  void
* Return:      gboolean
****************************************************************************/
gboolean bGenerateWPAKey()
{
     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
     unsigned char buf[46] = { '\0' };

     arc4random_buf(buf, 45);
     memset(g_qr_data.wpa_key, '\0', WPA_KEY_LENGTH+1);       

     strncpy(g_qr_data.wpa_key , g_base64_encode(buf, 48), WPA_KEY_LENGTH);
     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
     return TRUE;
}

/***************************************************************************
* Function:    bGenerateSSID
* Description: This method generates SSID of length 32 
* Parameters:  void
* Return:      gboolean
*****************************************************************************/
gboolean bGenerateSSID()
{
     DLT_LOG(SPM_SPL, DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING(__FUNCTION__));

     memset(g_qr_data.ssid, '\0', SSID_LENGTH+1);

     int randomData = open("/dev/urandom", O_RDONLY);
     if (randomData < 0)
     {
          DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
                 DLT_STRING("(): SSID generation failed"));
          return FALSE;
     }

     int result = read(randomData, g_qr_data.ssid, SSID_LENGTH);
     if (result < 0)
     {
         DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): Failed to generate random ssid"));
         close(randomData);
         return FALSE;
     }
     int i = 0;
     for(i = 0; i < SSID_LENGTH; i++)
     {
         g_qr_data.ssid[i] = (char)(((int)g_qr_data.ssid[i] % 45) + 48);
     }
     g_qr_data.ssid[SSID_LENGTH] = '\0';
     close(randomData);

     DLT_LOG(SPM_SPL, DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__));
     return TRUE;
}   
     
/***********************************************************************
* Function:    vHandleActiveWiFiUpdate
* Description: This method generates SSID of length 32
* Parameters:  ssid, wpa_key, WiFiAPState 
* Return:      None
*************************************************************************/
void vHandleActiveWiFiUpdate( gchar* ssid,
                              gchar* wpa_key,
                              WiFiAPState wifiAPMode )
{
     DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING(__FUNCTION__));
     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Wifi AP state = "), DLT_UINT(wifiAPMode));     

     g_iCurrentWiFiState = wifiAPMode;

     if(!g_bPairingInProgress)
     {
         DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
          DLT_STRING("(): Pairing not in progress, ignore WBL update"));
         return;
     }

     if(g_bFinalizePairing)
     {
         if(wifiAPMode == WIFI_AP_INACTIVE) 
         {
              DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                      DLT_STRING("(): Finalize pairing"));     
              vFinalizePairingSession();
         }
     }

     else if(wifiAPMode == WIFI_AP_ACTIVE)
     {
         gboolean isValidTempAP = bValidateTempWiFiSetup(ssid, wpa_key);
         if((isValidTempAP) && (!g_bQRGenerated))
         {
              DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                 DLT_STRING("(): Temp AP Available, proceed with QR"));
              if(!bGenerateQRCode())
              {
                  DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                         DLT_STRING("():bGenerateQRCode failed"));
                  vCloseSeamlessSession( SP_COMPLETED,
                                         SP_ERR_QR_GEN_ERROR);
              }
         }
   
         else if((!isValidTempAP) && (g_bQRGenerated))
         {
              DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Error in temp AP, Terminate seamless pair"));
              vCloseSeamlessSession(SP_COMPLETED, SP_ERR_WIFI_ERROR);  
         } 
     }

     else if(g_bQRGenerated)
     {
             DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): AP Disconnected during seamless pair"));
             vCloseSeamlessSession(SP_COMPLETED, SP_ERR_WIFI_ERROR);
     } 
     DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
}  

/***********************************************************************
* Function:    bValidateTempWiFiSetup
* Description: This function validates the temporary 
               wifi AP setup
* Parameters:  ssid, wpa_key
* Returns:     gboolean
*************************************************************************/
gboolean bValidateTempWiFiSetup( char* ssid,
                                 char* wpa_key )
{
     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
     if((!ssid) || (!wpa_key))
     {
           DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                 DLT_STRING("(): NULL input"));
          return FALSE;
     }

     if(strcmp(ssid, g_qr_data.ssid) != 0)
     { 
           DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
                          DLT_STRING("(): SSID mismatch"));
           return FALSE;
     }

     if(strcmp(wpa_key, g_qr_data.wpa_key) != 0)
     {
           DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                         DLT_STRING("(): WPA2 Key mismatch"));
          return FALSE;
     }
     return TRUE;
     DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}
         
/************************************************************************
* Function:    bGenerateQRCode
* Description: This function generates the QR code with
               the current seamless session data  
* Parameters:  void
* Returns:     gboolean
*************************************************************************/
gboolean bGenerateQRCode()
{
     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
     QRCode qrcode;
     int qrVersion = 13;
     int eccLevel = 0;
                
     if(!bGetSPMIntegerSetting("qrVersion" , &qrVersion))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Failed to get QR code version"));
         return FALSE; 
     }

     if(!bGetSPMIntegerSetting("eccLevel" , &eccLevel))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Failed to ecc level for QR Code"));
         return FALSE;
     }

     if((qrVersion < 0) || (qrVersion > 40))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Invalid qrVersion, use default"));
         qrVersion = 17;
     }

     if((eccLevel < 0) || (eccLevel > 3))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Invalid eccLevel, use default"));       
         eccLevel = 0;
     }
            
     guint8 qrcodeBytes[qrcode_getBufferSize(qrVersion)];
     guint8 *qrData;
     gsize   qrDataSize = 0;
     gsize   i =0, j =0;
     GVariant* qrImageBuffer = NULL;
 
     qrDataSize =  strlen(g_qr_data.ssid)
                    + strlen(g_qr_data.wpa_key)
                      + strlen(g_qr_data.hex_session_id)
                        + strlen(g_qr_data.huid) 
                         + strlen(g_qr_data.ipv4) 
                          + strlen(g_qr_data.port)
                            + strlen(g_qr_data.hashAlgorithm)
                              + (2*g_qr_data.serialNumberLen)
                                + (3 * HASH_LEN) + 13;

     qrData =  malloc((sizeof(guint8)) * qrDataSize); 

     memset(qrData,  0, ((sizeof(guint8)) * qrDataSize));
     if(!qrData)
     {
         DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__), 
            DLT_STRING("(): malloc failed" ));
         return FALSE;
     }

     qrData[i++] = '#';
     for(j = 0; j < strlen(g_qr_data.ssid); j++)
     {
            qrData[i++] = g_qr_data.ssid[j];
     }

     qrData[i++] = '#';
     for(j = 0; j < strlen(g_qr_data.wpa_key); j++)
     {
         qrData[i++] = g_qr_data.wpa_key[j];
     }

     qrData[i++] = '#';
     for(j = 0; j < strlen(g_qr_data.huid); j++)
     {
         qrData[i++] = g_qr_data.huid[j];
     }

     qrData[i++] = '#';
     for(j = 0; j < strlen(g_qr_data.ipv4); j++)
     {
         qrData[i++] = g_qr_data.ipv4[j];
     }

     qrData[i++] = '#';
     for(j = 0; j < strlen(g_qr_data.port); j++)
     {
         qrData[i++] = g_qr_data.port[j];
     }

     qrData[i++] = '#';
     for(j = 0; j < strlen(g_qr_data.hex_session_id); j++)
     {
         qrData[i++] = g_qr_data.hex_session_id[j];
     }

     qrData[i++] = '#';
     for(j = 0; j < strlen(g_qr_data.hashAlgorithm); j++)
     {
         qrData[i++] = g_qr_data.hashAlgorithm[j];
     }
 
     qrData[i++] = '#';
     for(j = 0; j < HASH_LEN; j++)
     {
         qrData[i++] = tolower(g_qr_data.issuerNameHash[j]);
     }

     qrData[i++] = '#';
     for(j = 0; j < HASH_LEN; j++)
     {
         qrData[i++] = tolower(g_qr_data.issuerKeyHash[j]);
     }

     qrData[i++] = '#';
     for(j = 0; j < ( g_qr_data.serialNumberLen * 2); j++)
     {
         qrData[i++] = g_qr_data.serialNumber[j];
     }

     qrData[i++] = '#';
     for(j = 0; j < HASH_LEN; j++)
     {
         qrData[i++] = tolower(g_qr_data.certificateHash[j]);
     }

     qrData[i++] = '#';

     DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
             DLT_STRING("(): QR Code = "), DLT_STRING(qrData));
         
     qrcode_initBytes(&qrcode, qrcodeBytes, qrVersion, (guint8) eccLevel, qrData,  strlen(qrData));
     free(qrData);

     char qrFileName[MAX_QR_FILENAME_LEN+1] = { '\0' };
     strcpy(qrFileName, "/tmp/qrcode.bmp");
     if(!bGetSPMStringSetting("qrImageFileName", qrFileName))
     {
         DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Failed to get qr filename"));
     }
     
      
     if(!prepareQrBmp(qrcode, qrFileName))
     {
         DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("(): prepareQrBmp failed"));
         return FALSE;
     }

     if(!bEmitQRCodeUpdate(qrFileName))
     {
         DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("(): bEmitQRCodeUpdate failed"));
         return FALSE;
     }

     g_bQRGenerated = TRUE;

     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("-"),
                                DLT_STRING(__FUNCTION__));
     return TRUE;
}


/************************************************************************
* Function:    vProcessAppMsg
* Description: This function proccesses the Msg data sent by SP App
* Parameters:  key, value
* Returns:     void
*************************************************************************/
void vProcessAppMsg(gchar* key, GVariant *value)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    gsize size = 0;
    gchar* strVal = NULL;
    GVariant* msg = NULL;
    GVariantBuilder builder;

    if((g_bFinalizePairing) || (!g_bPairingInProgress))
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
         DLT_STRING("(): No pairing / Pairing finalized, ignore app msg"));
        return;
    }
	
    gconstpointer cvByteArray =  g_variant_get_fixed_array ( value,
                                                             &size,
                                                             sizeof(gchar)
                                                           );
														  
 
    if((cvByteArray == NULL) || (size == 0))
    {
         DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
             DLT_STRING("(): Invalid byte array"));
         return;
    } 
	DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__), 
	       DLT_STRING("(): Variant size= "), DLT_UINT(size));                     
    strVal = malloc(size+1);    
	
    if(!strVal)
    {
         DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): malloc failed"));
         vCloseSeamlessSession(SP_COMPLETED, SP_ERR_INTERNAL_ERROR);
         return;
    } 
	
    strncpy(strVal, (char*) cvByteArray, size);
    strVal[size] = '\0'; 
    
	
    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): key is "), DLT_STRING(key));

    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): value is "), DLT_STRING(strVal));

    if(strcmp(key, "cert") == 0) 
    {
        g_current_device.cert = malloc(size);
        if(!g_current_device.cert)
        {
             DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                 DLT_STRING("(): malloc failed for device cert"));
             vCloseSeamlessSession( SP_COMPLETED, 
                                    SP_ERR_INTERNAL_ERROR );
        }        
        else
        {
            memcpy(g_current_device.cert, cvByteArray, size);
            g_current_device.certLength = size;
            if(bCheckTrustedCertExistence())
            {
                DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): Cert exists already"));
                vCloseSeamlessSession( SP_COMPLETED,
                                       SP_ERR_CERT_ALREADY_EXIST);

            }                                  
        }   
    }
    else if(strcmp(key, "signSid") == 0)
    {
        if(bProcessSignedSessionID( strVal, 
		                           g_current_device.cert, 
								   g_current_device.certLength) ) 
        {
            g_variant_builder_init(&builder,G_VARIANT_TYPE("a{ss}"));
            g_variant_builder_add(&builder, "{ss}", "result","certificate accepted");
            msg = g_variant_builder_end(&builder); 
            bSendMessageToApp( msg );
        }
        else
        {
             DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                 DLT_STRING("(): session id verification failed"));
             vCloseSeamlessSession( SP_COMPLETED,
                                    SP_ERR_CLIENT_AUTH_ERROR );
        }        
        g_pairInfoState |= 0x000001;        
    }
    else if(strcmp(key, "sp_wifi_ssid") == 0)
    {
         vUpdateCurrentDeviceInfo("ssid", strVal);
         g_pairInfoState |= 0x000010;
    }
    else if(strcmp(key, "sp_wifi_psk") == 0)
    {
         vUpdateCurrentDeviceInfo("wpa_key", strVal);
         g_pairInfoState |= 0x000100;
    }
    else if(strcmp(key, "sp_wifi_bssid") == 0)
    {
         vUpdateCurrentDeviceInfo("bssid", strVal);
         g_pairInfoState |= 0x001000;
    }
    else if(strcmp(key, "sp_name") == 0)
    {
          vUpdateCurrentDeviceInfo("device_name", strVal);
          g_pairInfoState |= 0x010000;
    }
    else if(strcmp(key , "status") == 0)
    {
         if(strcmp(strVal, "success") == 0)
         {
              
             if(SP_TRUSTED_DEVICE_ADDED == g_pairInfoState) 
             {
                if(g_uTimerID)
				{	
                   g_source_remove(g_uTimerID);
                   g_uTimerID = 0;
				}
                   vCloseSeamlessSession( SP_COMPLETED,
                                        SP_ERR_NO_ERROR );
             } 
             else
             {
                 DLT_LOG(SPM_SPL, DLT_LOG_INFO, 
                   DLT_STRING(__FUNCTION__), DLT_STRING(
                     "(): Device not yet added, ignore"));
             }
         }   
    }
    else
    {  
         DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                    DLT_STRING("():Unknown App message"));
    }
 
    if((SP_ALL_PARAMS_RECEIVED == g_pairInfoState) && (!g_bPhoneAckSent))
    {
         if(!bStoreTrustedDeviceCert())
         {
             DLT_LOG(SPM_SPL,DLT_LOG_ERROR,
                 DLT_STRING(__FUNCTION__),DLT_STRING(
                      "():Failed to store device cert"));
             vCloseSeamlessSession(
                       SP_COMPLETED,
              SP_ERR_INTERNAL_ERROR
                                  );
         }
         else
         {
             vUpdatePairTime();
            vUpdateCertificateSerialnumber();
             bUpdateTrustedDevice();
             g_pairInfoState |= 0x100000;
             g_bPhoneAckSent = TRUE; 
             vSendBTInfoAndAckToApp(); 
         }
    }   
    free(strVal);
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}  

/**********************************************************************
* Function:    vSendBTInfoAndAckToApp
* Description: This function updates sends the success acknowledgement
               and the Bluetooth pairing info to the companion App  
* Parameters:  void
* Returns:     void
***********************************************************************/
void vSendBTInfoAndAckToApp()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),
                            DLT_STRING(__FUNCTION__));

     GVariant* msg = NULL;
     GVariantBuilder builder;
     char bluetoothAddr[MAX_BT_ADDR_LEN] = { '\0' };
     char bluetoothName[MAX_BT_NAME_LEN] = { '\0' };

     g_variant_builder_init(&builder,G_VARIANT_TYPE("a{ss}"));
     g_variant_builder_add(&builder, "{ss}", "status","success");

     if((iFetchSeamlessBTNameWrapper(bluetoothName, sizeof(bluetoothName))) 
       && (iFetchSeamlessBTAddressWrapper(bluetoothAddr, sizeof(bluetoothAddr)))
       )
     {
         GVariant* deviceInfo = pGetCurrentDeviceInfo(); 
         g_variant_builder_add(&builder, "{ss}", "sp_bd_addr", bluetoothAddr);
         g_variant_builder_add(&builder, "{ss}", "sp_bluetooth_friendly_name",
                                                              bluetoothName);
         g_bSeamlessBTInfoSent = TRUE;       
     }
     
     msg = g_variant_builder_end(&builder);
     bSendMessageToApp(msg);
}    

/**********************************************************************
* Function:    vUpdateCertificate_Serialnumber
* Description: This function updates smart phone's certificate serialNumber
* Parameters:  void
* Returns:     void
***********************************************************************/
void vUpdateCertificateSerialnumber()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),
            DLT_STRING(__FUNCTION__));

    char strcertFileName[MAX_CERT_NAME_LEN+1] = { '\0' };
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("g_current_device.bssid"),
            DLT_STRING(g_current_device.bssid));
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("TRUSTED_CERT_PATH"),
            DLT_STRING(TRUSTED_CERT_PATH));
    sprintf(strcertFileName, "%s/%s.crt", TRUSTED_CERT_PATH, g_current_device.bssid);
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("smartphone cert file path:"),
            DLT_STRING(strcertFileName));
			
    ASN1_INTEGER *asn1_serial = NULL;
    BIO              *certbio = NULL;
    X509                *cert = NULL;
    int iReturn;
    unsigned char g_serialNumber[HASH_SIZE+1];

    /* ---------------------------------------------------------- *
       * Create the Input/Output BIO's.                             *
       * ---------------------------------------------------------- */
    certbio = BIO_new(BIO_s_file());

    /* ---------------------------------------------------------- *
       * Load the certificate from file (PEM).                      *
       * ---------------------------------------------------------- */
    iReturn = BIO_read_filename(certbio, strcertFileName);
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("BIO_read_filename ret:"),DLT_INT(iReturn));
    if(!iReturn)
    {
           DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("Error reading cert to bio"));
           BIO_free_all(certbio);
           return;
    }
    if (! (cert = PEM_read_bio_X509(certbio, NULL, 0, NULL)))
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("Error loading cert into memory"));
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("Error string: "),
                                    DLT_STRING(ERR_error_string(ERR_get_error(), NULL)));
        return;
    }
    /* ---------------------------------------------------------- *
       * Extract the certificate's serial number.                   *
       * ---------------------------------------------------------- */
    asn1_serial = X509_get_serialNumber(cert);
    if (asn1_serial == NULL)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("Error getting serial number from certificate"));
        return;
    }else
    {

        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                DLT_STRING("(): SerialNumber: "),
                DLT_RAW(asn1_serial->data,asn1_serial->length));
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                DLT_STRING("():Serial Length: "),
                DLT_INT(asn1_serial->length));
    }
   int retVal=  bByteToHexStrConvert(asn1_serial->data, asn1_serial->length,
                                      g_serialNumber);
   g_serialNumber[asn1_serial->length*2] ='\0';

   DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("bByteToHexStrConvert retVal:"),
            DLT_INT(retVal));
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("sp serialnumber:"),
            DLT_STRING(g_serialNumber));
		
	if(!retVal)
	{
		 DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("bByteToHexStrConvert failed"));
			X509_free(cert);
			BIO_free_all(certbio);
			return;
	}

    X509_free(cert);
    BIO_free_all(certbio);

    vUpdateCurrentDeviceInfo("serial_number", g_serialNumber);

    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),
            DLT_STRING(__FUNCTION__));
}

/**********************************************************************
* Function:    vUpdatePairTime
* Description: This function updates the device pair time in
               format 'yyyy:mm:dd hh:mm:ss'
* Parameters:  void
* Returns:     void
***********************************************************************/
void vUpdatePairTime()
{
     DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),
                              DLT_STRING(__FUNCTION__));
     char timeStr[21] = { '\0' };
     time_t t = time(NULL);
     struct tm td = *localtime(&t);
    
     snprintf(timeStr, 20, "%04d:%02d:%02d:%02d:%02d:%02d", 
                  td.tm_year+1900, td.tm_mon+1, td.tm_mday, 
                           td.tm_hour, td.tm_min, td.tm_sec );

     DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
           DLT_STRING("(): Pair time is "), DLT_STRING(timeStr));

     vUpdateCurrentDeviceInfo("pair_time", timeStr);
     DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),
                              DLT_STRING(__FUNCTION__));
}

/**********************************************************************
* Function:    vUpdateSPConnectedStatus
* Description: This function updates the SP Connected status
* Parameters:  status
* Returns:     void
***********************************************************************/
void vUpdateSPConnectedStatus(gboolean status) 
{
   g_bSmartPhoneConnected = status;
}
  
/**********************************************************************
* Function:    bUpdateTrustedDevice
* Description: This function requests for trusted device list update
* Parameters:  void
* Returns:     gboolean
*************************************************************************/
gboolean bUpdateTrustedDevice()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),
                            DLT_STRING(__FUNCTION__));      
    GVariant* deviceInfo = pGetCurrentDeviceInfo();
    
    if(deviceInfo == NULL)
    {
       DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
               DLT_STRING("(): pGetCurrentDeviceInfo failed"));
       return FALSE;
    }
    return bUpdateTrustedDeviceListInfo( deviceInfo, ADD_DEVICE);
}        

/************************************************************************
* Function:    bStoreTrustedDeviceCert
* Description: This method stores the Trusted device cert
               on the root file system
* Parameters:  void
* Returns:     void
*************************************************************************/
gboolean bStoreTrustedDeviceCert()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),
                            DLT_STRING(__FUNCTION__));
    char trustedCertPath[MAX_CONF_SIZE] = { '\0' };
    char certFileName[MAX_CERT_NAME_LEN+1] = { '\0' };

    if(!g_current_device.cert)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): cert is NULL"));
        return FALSE; 
    }
    
    if((!bGetTrustedCertFileName(g_current_device.bssid, certFileName))
          || (!strlen(certFileName)))
    {
         DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Failed to get cert file name"));
         return FALSE;
    }
    
    FILE *pCertFile = fopen(certFileName, "wb");
    if(!pCertFile)
    {
         DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Failed to open cert file")); 
         return FALSE;
    }
    fwrite(g_current_device.cert, 1, g_current_device.certLength, pCertFile);
    fclose(pCertFile);

    return (bRehashTrustedCert(certFileName, g_current_device.bssid)); 
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),
                            DLT_STRING(__FUNCTION__));
}

/*********************************************************************
* Function:    bCheckTrustedCertExistence    
* Description: This method checks if current cert is
               already available in the trusted list
* Parameters:  void
* Returns:     gboolean
               TRUE, if cert already exists or some failure
               FALSE, if cert doesn`t exist 
***********************************************************************/
gboolean bCheckTrustedCertExistence()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),
                            DLT_STRING(__FUNCTION__));
    gboolean bRetVal = TRUE;
    char hashFileName[MAX_CERT_NAME_LEN+1] = { '\0' };
    char trustedCertPath[MAX_CONF_SIZE] = { '\0' };   
    BIO *bio = NULL;
    X509* x_509 = NULL;
    struct stat  statbuf;
    unsigned long int hash = 0;

    if(!g_current_device.cert)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): g_current_device.cert is NULL")); 
        return bRetVal;         
    }

    if(!bGetSPMStringSetting("trustedCertificatePath", trustedCertPath))
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Failed to get trustedCertPath"));
       //use default path
        strcpy( trustedCertPath,
            "/var/opt/bosch/dynamic/seamlesspairing/TrustedDeviceCerts");
    }

    bio = BIO_new_mem_buf((void*)g_current_device.cert, g_current_device.certLength);
    if(!bio)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): BIO_new_mem_buf failed"));
        return bRetVal;
    }

    PEM_read_bio_X509(bio, &x_509, NULL, NULL);
    hash = X509_subject_name_hash(x_509);
    sprintf(hashFileName, "%s/%08lx.0", trustedCertPath, hash);
        DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): hashFileName = "), DLT_STRING(hashFileName));    
    if(lstat(hashFileName, &statbuf) != 0)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Cert doesn`t exist"));   
        bRetVal = FALSE;
    }
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),
                            DLT_STRING(__FUNCTION__));        
    return bRetVal;
}    

/************************************************************************
* Function:    bRehashTrustedCert
* Description: This method rehashes the Trusted device cert
               on the stunnel cert verification path 
* Parameters:  certFileName, bssid
* Returns:     void
*************************************************************************/
gboolean bRehashTrustedCert(gchar* certFileName, gchar* bssid)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    char rehashFileName[MAX_CERT_NAME_LEN+1] = { '\0' };
    char stunnelCAPath[MAX_CONF_SIZE] = { '\0' };
    char certSymLinkName[MAX_CERT_NAME_LEN+1] = { '\0' };

    if(!bGetRehashCertFileName(certFileName, rehashFileName))
    {
         DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Failed to get hash file name"));
         return FALSE;
    }

    if(!bGetSPMStringSetting("stunnelCAPath", stunnelCAPath))
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Failed to get stunnelCAPath"));
        //use default path
        strcpy( stunnelCAPath,
            "/etc/SPTrustedCerts");
    }
    
    sprintf(certSymLinkName, "%s/%s.crt", stunnelCAPath, bssid);

    if(0 != symlink(certSymLinkName, rehashFileName))
    {
         DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Failed to create rehash certificate"));
         return FALSE;
    }

    g_current_device.stunnelCertName = malloc(strlen(certSymLinkName) + 1);
    if(g_current_device.stunnelCertName)
        strcpy(g_current_device.stunnelCertName, certSymLinkName); 

    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__));
    return TRUE;
}        

/***********************************************************************
* Function:    bGetTrustedCertFileName
* Description: This method gives the file name for the
               trusted smartphone certificate
* Parameters:  bssid, certFileName
* Returns:     gboolean
*************************************************************************/
gboolean bGetTrustedCertFileName( gchar* bssid,
                                  gchar* certFileName) 
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    char trustedCertPath[MAX_CONF_SIZE] = { '\0' };

    if((!bssid) || (!certFileName))
    {
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Invalid input(s)"));
        return FALSE;
    }

    if(!bGetSPMStringSetting("trustedCertificatePath", trustedCertPath))
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Failed to get trustedCertPath"));
       //use default path
        strcpy( trustedCertPath, TRUSTED_CERT_PATH );
    }

    sprintf(certFileName, "%s/%s.crt", trustedCertPath, bssid);
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__));
    return TRUE;
}

/***********************************************************************
* Function:    bGetRehashCertFileName
* Description: This method gives the file name for the
               rehash certificate
* Parameters:  Original cert file name, hash file name
* Returns:     gboolean
*************************************************************************/   
gboolean bGetRehashCertFileName(gchar* certFileName, gchar* hashFileName)
{          
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING(__FUNCTION__));

    FILE* pCert = NULL;
    X509* x509 = NULL;
    char certVerifyPath[MAX_CONF_SIZE] = { '\0' };
    unsigned long int hash = 0;

    if((!certFileName) || (!strlen(certFileName)) || (!hashFileName))
    {
         DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Improper input(s)"));
         return FALSE;
    }

    if(!bGetSPMStringSetting("trustedCertificatePath", certVerifyPath))
    {
         DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Failed to get trustedCertPath"));
         return FALSE;
    }

    if(!strlen(certVerifyPath))
    {
         DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Improper trustedCertificatePath"));
         return FALSE;
    }

    pCert = fopen(certFileName, "rb");
    if(!pCert)
    {
         DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Failed to open cert file"));
         return FALSE;
    }

    x509 = PEM_read_X509(pCert, NULL, NULL, NULL);
    if(!x509)
    {
         DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): PEM_read_X509 failed"));
         fclose(pCert);
         return FALSE;
    }
    hash = X509_subject_name_hash(x509);
    sprintf(hashFileName, "%s/%08lx.0", certVerifyPath, hash);
    fclose(pCert);

    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__));
    return TRUE;
}

/***********************************************************************
* Function:    bRemoveTrustedDevice
* Description: This method removes a device from trusted list
* Parameters:  deviceInfo
* Returns:     gboolean
************************************************************************/
gboolean bRemoveTrustedDevice(GVariant* deviceInfo)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    char certFileName[MAX_CERT_NAME_LEN+1] = { '\0' };
    char rehashFileName[MAX_CERT_NAME_LEN+1] = { '\0' };
    GVariantIter iter;    
    if(!deviceInfo)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Null device info"));
        return FALSE;
    }
 
    const gchar* varType = g_variant_get_type_string(deviceInfo);

    if (!(g_strcmp0(varType, "a{sv}")))
    {
        GVariant *value;
        gchar *key;

        g_variant_iter_init (&iter, deviceInfo);

        while (g_variant_iter_next (&iter, "{sv}", &key, &value))
        {
            if(!g_strcmp0 (key, "sp_wifi_bssid"))
            {
                 const char* bssid = g_variant_get_string(value, NULL);
                 DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                     DLT_STRING("(): bssid:"), DLT_STRING(bssid));
                 if(!bRemoveTrustedCert(bssid))
                 {
                     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                            DLT_STRING("(): Failed to remove cert"));
                     g_variant_unref (value);
                     g_free (key);
                     return FALSE;
                 }
                 else
                 {
                     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                          DLT_STRING("(): Trusted cert removed successfuly"));                          
                     if(bUpdateTrustedDeviceListInfo(deviceInfo, REMOVE_DEVICE))
                     {
                          DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                               DLT_STRING("(): Trusted device removed successfuly"));
                          g_variant_unref (value);
                          g_free (key);
                          return TRUE;
                     }
                     else
                     {
                          DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                               DLT_STRING("(): bUpdateTrustedDeviceListInfo failed"));
                          g_variant_unref (value);
                          g_free (key);
                          return FALSE;
                     }  
                 }
            }
            g_variant_unref (value);
            g_free (key);
        }   
     }
     else
     {
          DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
           DLT_STRING("(): Invalid variant type for deviceInfo"));
          return FALSE;
     }  
     //We are here, means failed to spot the devcie 
     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): Unknown error"));
     return FALSE; 
}

/************************************************************************
* Function:    bRemoveAllTrustedDevices
* Description: This method deletes all the
               devices in the trusted list
* Parameters:  TrustedDeviceList
* Returns:     gboolean
*************************************************************************/
gboolean bRemoveAllTrustedDevices(GVariant* trustedList)
{
     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),
                            DLT_STRING(__FUNCTION__));

     struct dirent *de;
     gchar trustedCertPath[MAX_CONF_SIZE+1] = { '\0' };
     gchar fileName[MAX_CERT_NAME_LEN+1] = { '\0' };

     if(!bGetSPMStringSetting("trustedCertificatePath", trustedCertPath))
     {
         DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Failed to get trustedCertPath"));
        //use default path
         strcpy( trustedCertPath, TRUSTED_CERT_PATH);
     }

     DIR *dr = opendir(trustedCertPath);
     while ((de = readdir(dr)) != NULL)
     {
         if( (strcmp(de->d_name, ".") != 0) &&
               (strcmp(de->d_name, "..") != 0) )
         {
             sprintf(fileName, "%s/%s", trustedCertPath, de->d_name);
             if(remove(fileName) != 0)
             {
                 DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                     DLT_STRING("(): Failed to remove file: "),
                                         DLT_STRING(fileName));
                 closedir(dr);
                 return FALSE;
             } 
         }
     }
     closedir(dr);

     if(!bUpdateTrustedDeviceListInfo(trustedList, REMOVE_ALL))
     {
         DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
           DLT_STRING("():bUpdateTrustedDeviceListInfo failed for REMOVE_ALL"));
         return FALSE;
     }
     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("-"),
                            DLT_STRING(__FUNCTION__));
     return TRUE;
}

/************************************************************************
* Function:    bRemoveTrustedCert
* Description: This method deletes the cert of
               a given trusted device (bssid)
* Parameters:  bssid
* Returns:     gboolean
*************************************************************************/
gboolean bRemoveTrustedCert(const char* bssid)
{
     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"), 
                              DLT_STRING(__FUNCTION__));      

     char certFileName[MAX_CERT_NAME_LEN+1] = { '\0' };
     char rehashFileName[MAX_CERT_NAME_LEN+1] = { '\0' };

     if(!bGetTrustedCertFileName(bssid, certFileName))
     {
          DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                     DLT_STRING("(): Failed getting cert name"));
          return FALSE;
     }

     if(!bGetRehashCertFileName(certFileName, rehashFileName))
     {
          DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Failed getting rehashed cert name"));
          return FALSE;
     }

     if((!remove(certFileName)) && (!remove(rehashFileName)))
     {
          DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Certs removed successfuly"));
          return TRUE;
     }
     else
     {
          DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                   DLT_STRING("(): Failed to remove  certs"));
          return FALSE;
     }

     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("-"),
                          DLT_STRING(__FUNCTION__));
}

   
/************************************************************************
* Function:    vStartSeamlessTimer
* Description: This function starts the seamless pairing timer
* Parameters:  void
* Returns:     void
*************************************************************************/
void vStartSeamlessTimer()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING(__FUNCTION__));   
    int timeout = 0;
    if(!bGetSPMIntegerSetting("sessionTimeout", &timeout))
    {
       DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Failed to get session timeout"));
       // set to default time out
       timeout = 120;
    }    
    timeout *= 1000;
    g_uTimerID = g_timeout_add( timeout, 
	                        bSeamlessTimerExpiredCB,
                                NULL);

    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__));  
}

/***************************************************************************
* Function:    bSeamlessTimerExpiredCB
* Description: Terminates the seamless pairing sesison
* Parameters:  gpointer
* Return:      gboolean
*****************************************************************************/
gboolean bSeamlessTimerExpiredCB(gpointer data)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    if(g_bPairingInProgress)
    {
         if(SP_TRUSTED_DEVICE_ADDED == g_pairInfoState )
             vCloseSeamlessSession( SP_COMPLETED,
                                   ( g_bSeamlessBTInfoSent?
                                     SP_ERR_NO_ERROR :
                                     SP_ERR_BT_ERROR ));
         else
             vCloseSeamlessSession(SP_TIMEOUT, SP_ERR_SESSION_TIMEOUT);
    }
    else
    {
      DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
        DLT_STRING("(): Seamless pair not in progress, ignore Timeout CB"));
    } 
    if(g_uTimerID)
    {          
        g_source_remove(g_uTimerID);
        g_uTimerID = 0;
     }
         
     DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
     return TRUE;
}
      
/*************************************************************************
* Function:    vCloseSeamlessSession
* Description: This function triggers termination
               for the current seamless pairing session.  
* Parameters:  SPDeviceUpdateStatus, SPDeviceUpdateErrCode
* Returns:     void   
**************************************************************************/
void vCloseSeamlessSession( SPDeviceUpdateStatus  status, 
                            SPDeviceUpdateErrCode errCode )
{
     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 

     if(!g_bPairingInProgress)
     {
         DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): pair not in progress"));
         vResetData();
         return;
     }
     if(g_bFinalizePairing)
     {
         DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Session finalize in progress"));
         return;
     } 

     if(g_uTimerID)
           g_source_remove(g_uTimerID); 
     g_uTimerID = 0;

     SP_Status pairStatus = PAIR_FAILED;
     if( (status == SP_COMPLETED) && 
         (errCode == SP_ERR_NO_ERROR))
        pairStatus = PAIR_SUCCESS;
 
     vStopSeamlessPairingServer( (g_current_device.stunnelCertName ? 
                                  g_current_device.stunnelCertName : ""),
                                  pairStatus 
                               );

     g_seamlessPairStatus = status;
     g_seamlessPairErrorCode = errCode;

     if(( SP_ERR_WIFI_ERROR == errCode) || 
         (g_iCurrentWiFiState != WIFI_AP_ACTIVE))
     {
         vFinalizePairingSession(); 
     }     
     else
     {
         g_bFinalizePairing = TRUE;
         vDeActivateTempAP(); 
     } 

     DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/*************************************************************************
* Function:    vFinalizePairingSession
* Description: This function finalize the seamless pairing
               session status and triggers HMI update 
               seamless pairing session
* Parameters:  void
* Returns:     void
**************************************************************************/
void vFinalizePairingSession()
{
     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));    
     GVariant* deviceInfo = pGetCurrentDeviceInfo();

     vEmitSeamlessPairingStatus( (guint16)g_seamlessPairStatus,
                                 (guint16)g_seamlessPairErrorCode,
                                 deviceInfo );
     vResetData();
     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
} 
           
    /**************************************************************************
    * Function:    vUpdateCurrentDeviceInfo
    * Description: This function updates the current device info
    * Parameters:  device_name, ssid, psk, bssid
    * Returns:     void
    *************************************************************************
    void vUpdateCurrentDeviceInfo(   gchar* device_name,
                                     gchar* ssid,
                                     gchar* psk,
                                     gchar* bssid 
                                   )
    {
          DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING(__FUNCTION__));
          gsize len = 0; 
          if( (!ssid) || (!psk) || (!bssid) || (!bssid))
          {
             DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                                       DLT_STRING("(): NULL input(s)"));     
             return;
          }
          if((len = strlen(device_name)) <= MAX_DEVICE_NAME) 
          { 
             strncpy(g_current_device.device_name, device_name, len);
             g_current_device.device_name[len] = '\0';
          }
          else
             DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                                       DLT_STRING("(): Invalid device name"));

          if((len = strlen(ssid)) <= SSID_LENGTH)
          {
             strncpy(g_current_device.ssid, ssid, len);
             g_current_device.ssid[len] = '\0';
          }
          else
             DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                                       DLT_STRING("(): Invalid SSID"));
  
          if((len = strlen(psk)) <= WPA_KEY_LENGTH)
          {
             strncpy(g_current_device.psk, psk, len);
             g_current_device.psk[len] = '\0';
          }
          else
             DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                                       DLT_STRING("(): Invalid psk"));

          if((len = strlen(bssid)) <= MAX_MAC_ADDR)
          {
             strncpy(g_current_device.bssid, bssid, len);
             g_current_device.bssid[len] = '\0';
          }
          else
             DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                                       DLT_STRING("(): Invalid bssid"));
          
          DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__));
    }
*/

/************************************************************************
* Function:    vUpdateCurrentDeviceInfo
* Description: This function updates the current device info
* Parameters:  key, value
* Returns:     void
*************************************************************************/
void vUpdateCurrentDeviceInfo( gchar*  key,
                               gchar*  value )
{
     DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),
                             DLT_STRING(__FUNCTION__));
     gsize len = 0;

     if((!key) || (!value))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                                    DLT_STRING("(): NULL input(s)"));
         return;
     }

     if((strcmp(key, "device_name") == 0) && 
              ( (len = strlen(value)) <= MAX_DEVICE_NAME))
     {
         strncpy(g_current_device.device_name, value, len);
         g_current_device.device_name[len] = '\0';
     }
     else if((strcmp(key, "ssid") == 0) &&
                 ((len = strlen(value)) <= SSID_LENGTH))
     {
         strncpy(g_current_device.ssid, value, len);
         g_current_device.ssid[len] = '\0';
     }
     else if((strcmp(key, "wpa_key") == 0) &&
                 ((len = strlen(value)) <= WPA_KEY_LENGTH))
     {
         strncpy(g_current_device.psk, value, len);
         g_current_device.psk[len] = '\0';
     }

     else if((strcmp(key, "bssid") == 0) && 
                  ((len = strlen(value)) <= MAX_MAC_ADDR))
     {
         strncpy(g_current_device.bssid, value, len);
         g_current_device.bssid[len] = '\0';
     }
     else if((strcmp(key, "pair_time") == 0) &&
              ((len = strlen(value)) <= TIME_LEN))
     {
         strncpy(g_current_device.pair_time, value, len);
         g_current_device.pair_time[len] = '\0';
     }
    else if((strcmp(key, "serial_number") == 0) &&
            ((len = strlen(value)) <= HASH_LEN))
    {
         DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("serial num len"),
                                DLT_INT(len));
        strncpy(g_current_device.serialnumber, value, len);
        g_current_device.serialnumber[len] = '\0';
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(" g_current_device.serialnumber:"),
                DLT_STRING( g_current_device.serialnumber));
    }
     else
         DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                                DLT_STRING("(): invalid bssid"));

     DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"),
                                DLT_STRING(__FUNCTION__));
}

/************************************************************************
* Function:    pGetCurrentDeviceInfo
* Description: This function provides the info of
               current device (smart phone)
* Parameters:  void
* Returns:     GVariant*
*************************************************************************/
GVariant*  pGetCurrentDeviceInfo()
{
     DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING(__FUNCTION__));

     GVariantBuilder builder;
     GVariant *device_info = NULL;
     g_variant_builder_init(&builder,G_VARIANT_TYPE("a{sv}"));

     g_variant_builder_add(&builder, "{sv}", "sp_name", 
                     g_variant_new_string(g_current_device.device_name));

     g_variant_builder_add(&builder, "{sv}", "sp_wifi_ssid", 
                     g_variant_new_string(g_current_device.ssid));

     g_variant_builder_add(&builder, "{sv}", "sp_wifi_psk",
                     g_variant_new_string(g_current_device.psk));

     g_variant_builder_add(&builder, "{sv}", "sp_wifi_bssid",
                     g_variant_new_string(g_current_device.bssid));        
          
    g_variant_builder_add(&builder, "{sv}", "sp_pair_time",
                    g_variant_new_string(g_current_device.pair_time));

    g_variant_builder_add(&builder, "{sv}", "sp_serial_number",
                          g_variant_new_string(g_current_device.serialnumber));

    DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("sp_serial_number"),
                             DLT_STRING(g_current_device.serialnumber));

     device_info = g_variant_builder_end(&builder);
         
     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("-"),
                              DLT_STRING(__FUNCTION__)); 
     return device_info;
}

/***********************************************************************
* Function:    pGetServerConnInfo
* Description: This function provides the info
               for server connection
* Parameters:  void
* Returns:     GVariant*
*************************************************************************/
GVariant*  pGetServerConnInfo()
{
     DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
     GVariantBuilder builder;
     GVariant *connInfo = NULL;
     int port = 8171; //default port
     if(!bGetSPMIntegerSetting("sps_port", &port))
     {
         DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
             DLT_STRING("():Failed fetching port from conf"));
     }

     //info to be passed to finalised, currently passing empty
     g_variant_builder_init(&builder, G_VARIANT_TYPE("a{sv}"));
     g_variant_builder_add(&builder, "{sv}", "ip_address", g_variant_new_string(SP_IPV4_ADDRESS) );
     g_variant_builder_add(&builder, "{sv}", "port", g_variant_new_uint32(port) ); 

     connInfo = g_variant_builder_end(&builder);
     DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));  
     return connInfo;  
}  

/**********************************************************************
* Function:    bProcessSignedSessionID
* Description: This function is for processing Session ID
* Parameters:  unsigned char *arg_signedSessionID, unsigned char *arg_pSPCert
* Returns:     gboolean
***********************************************************************/
gboolean bProcessSignedSessionID(char *arg_signedSessionID, 
                                 unsigned char *arg_pSPCert,
                                 size_t certLength )
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    BIO *l_pCertBio = NULL;
    X509 *l_pCert = NULL;
    EVP_PKEY *l_pPubkey = NULL;
    int l_iResult = 0;
    unsigned char* l_strDecodedSignature = NULL;
    int l_signatureLength = 0;
    unsigned char* l_strSessionIDHash = NULL;
    int l_iSessionIDHashLength = 0;
    RSA *l_pRsaKey = NULL;

    DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
      DLT_STRING("(): arg_signedSessionID= "), DLT_STRING(arg_signedSessionID));
	
	DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
	              DLT_STRING("(): arg_pSPCert= "),  DLT_STRING(arg_pSPCert));
				  
    DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
	        DLT_STRING("(): certLength = "), DLT_UINT(certLength));
	
    do
    {
        l_pCertBio = BIO_new(BIO_s_mem());
        BIO_write(l_pCertBio, arg_pSPCert, (int)certLength);

        if(!(l_pCert = PEM_read_bio_X509(l_pCertBio, NULL, 0, NULL)))
        {
            //ERROR
            DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
			            DLT_STRING("(): Failed in PEM_read_bio_X509"));        
            break;
        }
        else
        {
            DLT_LOG(SPM_SPL, DLT_LOG_DEBUG, DLT_STRING(__FUNCTION__),
                         DLT_STRING("(): PEM_read_bio_X509 success"));
        }

        if(!(l_pPubkey = X509_get_pubkey(l_pCert)))
        {
            //ERROR
            DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                          DLT_STRING("(): Failed in X509_get_pubkey"));
            break;
        }
        else
        {
            DLT_LOG(SPM_SPL, DLT_LOG_DEBUG, DLT_STRING(__FUNCTION__),
                          DLT_STRING("(): X509_get_pubkey successful"));
        }

        l_strDecodedSignature = g_base64_decode( arg_signedSessionID,
                                        		 &l_signatureLength);


  	    if(l_strDecodedSignature == NULL)
        {
            DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                          DLT_STRING("(): Failed in g_base64_decode"));
            break;
        }
        else
        {
            DLT_LOG(SPM_SPL, DLT_LOG_DEBUG, DLT_STRING(__FUNCTION__),
                            DLT_STRING("g_base64_decode successful"));
            DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("base64 decoded signature:"),
                                DLT_STRING(l_strDecodedSignature));		
            DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("decodeded signature LEN= "),
                    DLT_INT(l_signatureLength));								
        }

        //Compute the Hash of Session ID
        //And then call RSA_Verify to verify the session ID signature
        if(bCalculateMessageDigest( g_qr_data.hex_session_id,
                                    strlen(g_qr_data.hex_session_id),
                                    &l_strSessionIDHash,
                                    &l_iSessionIDHashLength))
        {
            DLT_LOG(SPM_SPL, DLT_LOG_DEBUG, DLT_STRING(__FUNCTION__),
                    DLT_STRING("(): SessionID hash calculated"));

            DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("Session ID hash: "),
                    DLT_RAW(l_strSessionIDHash, l_iSessionIDHashLength));
            DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("Session ID hash length: "),
                    DLT_INT(l_iSessionIDHashLength));

            l_pRsaKey = EVP_PKEY_get1_RSA(l_pPubkey);
            if(l_pRsaKey == NULL)
            {
                DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                        DLT_STRING("(): RSA Key is NULL"));
                break;
            }
            else
            {
                DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                        DLT_STRING("(): EVP_PKEY_get1_RSA Successful"));
            }

            DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("RSA key size: "),
                    DLT_INT(RSA_size(l_pRsaKey)));


            DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("RSA key size bits: "),
                            DLT_INT(BN_num_bits(l_pRsaKey->n)));
        
							
            l_iResult = RSA_verify(NID_sha256,
                                l_strSessionIDHash,
                                l_iSessionIDHashLength,
                                l_strDecodedSignature,
                                l_signatureLength,
                                l_pRsaKey);
					

            DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("RSA Verify result: "),
                                                         DLT_INT(l_iResult));
			if(l_iResult != 1)
		    {		 
														 
               DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING("RSA_verify error: "),
                          DLT_STRING(ERR_error_string(ERR_get_error(), NULL)));
					
               DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING("RSA_verify error code: "),
                                                    DLT_INT(ERR_get_error()));
			}										 
        }
        else
        {
            DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                    DLT_STRING("(): Error in CalculateMessageDigest"));
        }
    }while(0);

    if(l_pCert != NULL)
        X509_free(l_pCert);

    if(l_pCertBio != NULL)
        BIO_free_all(l_pCertBio);

    if(l_pPubkey != NULL)
        free(l_pPubkey);

    if(l_strDecodedSignature != NULL)
		 
        g_free(l_strDecodedSignature);

    if(l_strSessionIDHash != NULL)
        free(l_strSessionIDHash);

    if(l_iResult == 1) {
        DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                DLT_STRING("Verified signature"));
        return TRUE;
    } else {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
          DLT_STRING("(): Failed to verify signature, return code: "),
                                                 DLT_INT(l_iResult));  
        return FALSE;
    }
}

