/****************************************************************************
* FILE          : certificateHandler.c
* PROJECT       : Seamless Pairing For IPCM
* DESCRIPTION   : Certificate data Handling logic
* AUTHOR        : RVH5KOR
* COPYRIGHT     : (c) 2018 Robert Bosch GmbH, Hildesheim
****************************************************************************/

#include "certificateHandler.h"
#include "adit-components/sdc_op_conv.h"
#include "adit-components/sdc_perm.h"
#include "seamless_settings.h"
#include "dlt/dlt.h"

#define DEVICE_CERT_PATH "/var/opt/bosch/dynamic/ecm/server.crt.wrap"

DLT_IMPORT_CONTEXT(SPM_SPL);

static unsigned char * g_pUnwrappedCert = NULL;

static unsigned char *g_certHash = NULL;
static unsigned char *g_pubKeyHash = NULL;
static unsigned char *g_issuerNameHash = NULL;
static unsigned char *g_sPublicKeyCorrected = NULL;
static unsigned char g_serialNumber[HASH_SIZE+1];
static int g_iSerialSize;

/*************************************************************************
 * FUNCTION     : vCorrectPublicKeyString
 * DESCRIPTION  : This method corrects the public key string
 * PARAMETER    : unsigned char *arg_sPublicKey
 * RETURNVALUE  : void
*************************************************************************/
void vCorrectPublicKeyString(unsigned char *arg_sPublicKey)
{
    /*This method is to truncate the public key string so that
     * BEGIN PUBLIC KEY and END PUBLIC KEY occurances are removed
     * from the public key string*/

    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    unsigned char **l_sToken = NULL;

    l_sToken = g_strsplit(arg_sPublicKey,"\n",-1);

    g_sPublicKeyCorrected = g_strjoinv (NULL,l_sToken);

    g_sPublicKeyCorrected = g_sPublicKeyCorrected + CERT_BEGIN_TRUNCATION_INDEX;
    g_sPublicKeyCorrected[CERT_END_TRUNCATION_INDEX] = '\0';

    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+vCorrectPublicKeyString: "),
            DLT_STRING(g_sPublicKeyCorrected));

    g_strfreev(l_sToken);
}

/*************************************************************************
 * FUNCTION     : bCalculateMessageDigest
 * DESCRIPTION  : This method calculates the hash of input message
 * PARAMETER    : unsigned char *arg_pCertHash
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bCalculateMessageDigest(unsigned char *arg_sMessage,
                                 int arg_iMessageLen,
                                 unsigned char **arg_sDigest,
                                 unsigned int *arg_iDigestLen)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__));
    EVP_MD_CTX *l_pMdctx;
    gboolean l_bResult = TRUE;

    if((l_pMdctx = EVP_MD_CTX_create()) == NULL)
    {
        printf("Error in EVP_DigestInit_ex\n");
        l_bResult = FALSE;
    }

    if(1 != EVP_DigestInit_ex(l_pMdctx, EVP_sha256(), NULL))
    {
        printf("Error in EVP_DigestInit_ex\n");
        l_bResult = FALSE;
    }

    if(1 != EVP_DigestUpdate(l_pMdctx, arg_sMessage, arg_iMessageLen))
    {
        printf("Error in EVP_DigestUpdate\n");
        l_bResult = FALSE;
    }

    if((*arg_sDigest = (unsigned char *)OPENSSL_malloc(EVP_MD_size(EVP_sha256()))) == NULL)
    {
        printf("Error in OPENSSL_malloc\n");
        l_bResult = FALSE;
    }

    if(1 != EVP_DigestFinal_ex(l_pMdctx, *arg_sDigest, arg_iDigestLen))
    {
        printf("Error in EVP_DigestFinal_ex\n");
        l_bResult = FALSE;
    }

    EVP_MD_CTX_destroy(l_pMdctx);
    return l_bResult;
}

/*************************************************************************
 * FUNCTION     : bGetCertificateHash
 * DESCRIPTION  : This is a Get function for Cert hash
 * PARAMETER    : unsigned char *arg_pCertHash
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bGetCertificateHash(unsigned char *arg_pCertHash)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    memcpy(arg_pCertHash,g_certHash,HASH_SIZE);

    if(arg_pCertHash != NULL)
        return TRUE;
    return FALSE;
}

/*************************************************************************
 * FUNCTION     : bGetKeyHash
 * DESCRIPTION  : This is a Get function for Key hash
 * PARAMETER    : unsigned char *arg_pKeyHash
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bGetKeyHash(unsigned char *arg_pKeyHash)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    memcpy(arg_pKeyHash,g_pubKeyHash,HASH_SIZE);

    if(arg_pKeyHash != NULL)
        return TRUE;
    return FALSE;
}

/*************************************************************************
 * FUNCTION     : bGetIssuerNameHash
 * DESCRIPTION  : This is a Get function for Cert hash
 * PARAMETER    : unsigned char *arg_pIssuerNameHash
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bGetIssuerNameHash(unsigned char *arg_pIssuerNameHash)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    memcpy(arg_pIssuerNameHash,g_issuerNameHash,HASH_SIZE);

    if(arg_pIssuerNameHash != NULL)
        return TRUE;
    return FALSE;
}

/*************************************************************************
 * FUNCTION     : bGetSerialNumber
 * DESCRIPTION  : This is a Get function for Cert hash
 * PARAMETER    : unsigned char *arg_pSerialNumber, int *arg_iSerialNumberLength
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bGetSerialNumber(unsigned char *arg_pSerialNumber, int *arg_iSerialNumberLength)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    memcpy(arg_pSerialNumber,g_serialNumber,g_iSerialSize);
    *arg_iSerialNumberLength = g_iSerialSize;

    if(arg_pSerialNumber != NULL)
        return TRUE;
    return FALSE;
}

/*************************************************************************
 * FUNCTION     : bFreeUpCertMemory
 * DESCRIPTION  : This function frees up memory used by Cert data
 * PARAMETER    : void
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bFreeUpCertMemory()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    if(g_pUnwrappedCert != NULL)
        free(g_pUnwrappedCert);

    if(g_certHash != NULL)
        free(g_certHash);
    if(g_pubKeyHash != NULL)
        free(g_pubKeyHash);
    if(g_issuerNameHash != NULL)
        free(g_issuerNameHash);
    if(g_sPublicKeyCorrected != NULL)
        free(g_sPublicKeyCorrected);

    return TRUE;
}

/*************************************************************************
 * FUNCTION     : iCheckPEM
 * DESCRIPTION  : This function checks the unwrapped cert for PEM format
 * PARAMETER    : unsigned char **
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bCheckPEM(unsigned char **pCertificate)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    if(*pCertificate == NULL)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("iCheckPEM: "),
                DLT_STRING("pCertificate is NULL"));
        return FALSE;
    }
    /* If certificate, crl, etc. starts with "-----BEGIN " it is most likely PEM coded (rfc1421) */
    if(strncmp((char*)*pCertificate,"-----BEGIN ", 11) == 0){
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("iCheckPEM: "),
                DLT_STRING("SUCCESS in pCertificate comparison"));
        return TRUE;
    }
    else return FALSE;
}

/*************************************************************************
 * FUNCTION     : iReadFromWrappedFile
 * DESCRIPTION  : This function reads the wrapped certificate
 * PARAMETER    : FILE *, unsigned char **
 * RETURNVALUE  : int
*************************************************************************/
int iReadFromWrappedFile(FILE *pDeviceCertFilePointer, unsigned char **pWrappedCert)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    long l_FileSize = 0;
    int l_iRawCertLength = 0;
    gchar certPath[MAX_CONF_SIZE+1] = { '\0' };

    if(!bGetSPMStringSetting("deviceCertPath", certPath))
    {
       DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Failed to get cert path"));
       return FAILURE;
    }

    if((pDeviceCertFilePointer = fopen(certPath, "r")) == NULL)
    {
        //ERROR IN READING WRAPPED CERT FILE
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("vReadFromWrappedFile: "),
                DLT_STRING("Error in opening the file"));
        return FAILURE;
    }

    if ( fseek ( pDeviceCertFilePointer, 0 , SEEK_END ) != 0 ) {
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("vReadFromWrappedFile: "),
                DLT_STRING("Error in fseek"));
        fclose(pDeviceCertFilePointer);
        return FAILURE;
    } else {
        l_FileSize = ftell(pDeviceCertFilePointer);
        if ( l_FileSize < 0 ) {
             DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("vReadFromWrappedFile: "),
                          DLT_STRING("Error getting the filesize"));
             fclose(pDeviceCertFilePointer); 
             return FAILURE;
        } else {
            l_iRawCertLength = l_FileSize;
        }
        rewind ( pDeviceCertFilePointer );
    }

    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("vReadFromWrappedFile:RawCertLength : "),
            DLT_INT(l_iRawCertLength));

    *pWrappedCert = (char *)malloc(l_iRawCertLength);
    if(*pWrappedCert == NULL)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("vReadFromWrappedFile: "),
                DLT_STRING("Error in allocating mem for Cert"));
        fclose(pDeviceCertFilePointer);
        return FAILURE;
    }

    if( fread(*pWrappedCert, 1, l_iRawCertLength, pDeviceCertFilePointer) 
                                                      != l_iRawCertLength) 
    {
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("vReadFromWrappedFile: "),
                DLT_STRING("Error in reading the raw cert"));
    }
    else
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("iUnwrapCert:pWrappedCert : "),
                DLT_STRING(*pWrappedCert));
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("vReadFromWrappedFile: "),
                DLT_STRING("Raw cert read successfully"));
    }

    if(pDeviceCertFilePointer != NULL)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                DLT_STRING("Closing file pointer"));
        fclose(pDeviceCertFilePointer);
    }

    return l_iRawCertLength;
}

/*************************************************************************
 * FUNCTION     : bUnwrapCert
 * DESCRIPTION  : This function unwraps the wrapped certificate
 * PARAMETER    : void
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bUnwrapCert()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    sdc_session_t* l_pSession = NULL;
    sdc_wrap_unwrap_type_t* l_pType = NULL;
    sdc_error_t l_eError;
    FILE *l_pDeviceCertFilePointer = NULL;
    unsigned char * l_pWrappedCert = NULL;
    int l_iWrappedCertLength;
    int l_iUnwrappedCertLength;
    gboolean bResult = FALSE;

    //Read Wrapped file first
    l_iWrappedCertLength = iReadFromWrappedFile(l_pDeviceCertFilePointer, &l_pWrappedCert);

    if((!l_iWrappedCertLength) || (l_pWrappedCert == NULL))
    {
         DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
            DLT_STRING("(): Error reading the wrapped cert"));
         if(l_pWrappedCert)
             free(l_pWrappedCert); 
         return bResult;
    }
    l_eError = sdc_open_session(&l_pSession);

    if(SDC_OK == l_eError)
    {
        l_eError = sdc_unwrap_formatted_extract_type(l_pWrappedCert, l_iWrappedCertLength, &l_pType);
        if(SDC_OK == l_eError)
        {
            l_eError = sdc_unwrap_formatted_autoload_key(l_pSession, l_pWrappedCert, l_iWrappedCertLength);
            if(SDC_OK == l_eError)
            {
                l_eError = sdc_unwrap_formatted(l_pSession, l_pType, l_pWrappedCert, l_iWrappedCertLength, &g_pUnwrappedCert, &l_iUnwrappedCertLength);
                if(SDC_OK == l_eError)
                {
                    if(bCheckPEM(&g_pUnwrappedCert))
                    {
                        //SUCCESS
                        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                                DLT_STRING("SUCCESS in iCheckPEM"));
                        bResult = TRUE;
                    }
                    else
                    {
                        //FAILURE. NOT A PEM ENCODED CERTIFICATE
                        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                                DLT_STRING("Error in iCheckPEM"));
                    }
                }
                else
                {
                    //ERROR. sdc_unwrap_formatted returned error
                    DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                            DLT_STRING("Error in sdc_unwrap_formatted"));
                }
            }
            else
            {
                //ERROR. sdc_unwrap_formatted_autoload_key returned error
                DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                        DLT_STRING("Error in sdc_unwrap_formatted_autoload_key"));
            }
        }
        else
        {
            //ERROR. sdc_unwrap_formatted_extract_type returned error
            DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                    DLT_STRING("Error in sdc_unwrap_formatted_extract_type"));
        }
    }
    else
    {
        //ERROR. sdc_open_session returned error
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("Error in opening a SDC session"));
    }

    if (l_pType != NULL)
    {
       l_eError = sdc_wrap_unwrap_type_free(l_pType);
       if (SDC_OK != l_eError)
       {
          DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("iUnwrapCert: "),
                   DLT_STRING("Error in sdc_wrap_unwrap_type_free"));
          DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
             DLT_STRING("sdc_wrap_unwrap_type_free Error string: "),
                  DLT_STRING(sdc_get_error_string(l_eError)));
       }
       l_pType = NULL;
    }

    if (l_pSession != NULL)
    {
       l_eError = sdc_close_session(l_pSession);
       if (SDC_OK != l_eError)
       {
           DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                   DLT_STRING("Error in sdc_wrap_unwrap_type_free"));
           DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
               DLT_STRING("():sdc_close_session Error string: "),
                   DLT_STRING(sdc_get_error_string(l_eError)));
       }
       l_pSession = NULL;
    }

    if(l_pWrappedCert != NULL)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                DLT_STRING("():Freeing raw cert pointer"));
        free(l_pWrappedCert);
    }

    if(l_pDeviceCertFilePointer != NULL)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                DLT_STRING("Closing file pointer"));
        fclose(l_pDeviceCertFilePointer);
    }

   return bResult;
}

/*************************************************************************
 * FUNCTION     : bComputeHash
 * DESCRIPTION  : This function computes the hashes
 * PARAMETER    : unsigned char *, unsigned char *
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bComputeHash(unsigned char *arg_sPubKey, unsigned char *arg_sIssuerName)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    //const EVP_MD *l_pFprint_type = NULL;
    int l_iHashSize;
    gboolean l_bResult = FALSE;

    //Set the digest method & calculate the cert fingerprint
    //l_pFprint_type = EVP_sha256();

    //SHA-256 creates a 256bit hash
    if(!bCalculateMessageDigest(g_pUnwrappedCert, strlen(g_pUnwrappedCert), &g_certHash, &l_iHashSize))
    {
        //ERROR
        l_bResult = FALSE;
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("():Error in X509_digest"));
        return l_bResult;
    }
    else
    {
        l_bResult = TRUE;
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("Cert hash: "),
                DLT_RAW(g_certHash,32));
    }

    if(!bCalculateMessageDigest(arg_sPubKey, strlen(arg_sPubKey), &g_pubKeyHash, &l_iHashSize))
    {
        //ERROR
        l_bResult = FALSE;
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("():Error in X509_pubkey_digest"));
        return l_bResult;
    }
    else
    {
        l_bResult = TRUE;
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("PubKey hash: "),
                DLT_RAW(g_pubKeyHash,32));
    }

    if(!bCalculateMessageDigest(arg_sIssuerName, strlen(arg_sIssuerName), &g_issuerNameHash, &l_iHashSize))
    {
        //ERROR
        l_bResult = FALSE;
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("():Error in X509_NAME_digest"));
        return l_bResult;
    }
    else
    {
        l_bResult = TRUE;
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("IssuerName hash: "),
                DLT_RAW(g_issuerNameHash,32));
    }

    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
        DLT_STRING(" g_serialNumber: "), DLT_RAW(g_serialNumber,g_iSerialSize));

    return l_bResult;
}

/*************************************************************************
 * FUNCTION     : bExtractCertInfo
 * DESCRIPTION  : This function extracts certificates' info
 * PARAMETER    : void
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bExtractCertInfo()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    X509 *l_pCert = NULL;
    ASN1_INTEGER *l_pSerialNumber = NULL;
    unsigned char *l_pIssuerName = NULL;
    unsigned char *l_sPubKey = (char *)malloc(KEY_LENGTH * sizeof(char));
    BIO *l_pCertbio = NULL;
    BIO *l_pPubKeyBio = BIO_new (BIO_s_mem());
    gboolean l_bResult = TRUE;

    do
    {
        if(g_pUnwrappedCert != NULL)
        {
            //CERT IS ALREADY UNWRAPPED. DONT READ WRAPPED FILE AGAIN
            DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                    DLT_STRING("Cert: "), DLT_STRING(g_pUnwrappedCert));
        }
        else if(!bUnwrapCert())
        {
            //ERROR. Return error from here
            DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                    DLT_STRING("():Error in Unwrapping Cert"));
            l_bResult = FALSE;
            break;
        }

        //These functions call initialize openssl for correct work
        OpenSSL_add_all_algorithms();
        ERR_load_BIO_strings();
        ERR_load_crypto_strings();

        l_pCertbio = BIO_new(BIO_s_mem()); //----> THIS LINE IS TO READ CERT FROM A FILE-STREAM
        BIO_puts(l_pCertbio, g_pUnwrappedCert);//----> THIS LINE IS TO READ CERT FROM A FILE-STREAM

        if(l_pCertbio == NULL)
        {
            DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                    DLT_STRING("(): Error in READING THE CERT BYTE ARRAY"));
            l_bResult = FALSE;
            break;
        }

        //Load the certificate from file (PEM).
        if (! (l_pCert = PEM_read_bio_X509(l_pCertbio, NULL, 0, NULL))) {
            DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                    DLT_STRING("Error in PEM_read_bio_X509"));
            l_bResult = FALSE;
            break;
        }

        //Extract the certificate's serial number.
        l_pSerialNumber = X509_get_serialNumber(l_pCert);
        if(l_pSerialNumber == NULL)
        {
            DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                    DLT_STRING("l_pSerialNumber is NULL from X509_get_serialNumber"));
            l_bResult = FALSE;
            break;
        }
        else
        {
            DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                    DLT_STRING("(): l_pSerialNumber: "),
                    DLT_RAW(l_pSerialNumber->data,l_pSerialNumber->length));
            DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                    DLT_STRING("():Serial Length: "),
                    DLT_INT(l_pSerialNumber->length));
            memcpy(g_serialNumber, l_pSerialNumber->data, l_pSerialNumber->length);
            g_iSerialSize = l_pSerialNumber->length;
        }

        //Extract the certificate's public key data.
        if(!PEM_write_bio_PUBKEY(l_pPubKeyBio, X509_get_pubkey(l_pCert)))
        {
            DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                    DLT_STRING("():l_pPubKey is NULL from X509_get_pubkey"));
            l_bResult = FALSE;
            break;
        }

        if(!BIO_read(l_pPubKeyBio, l_sPubKey, KEY_LENGTH))
        {
            //ERROR IN READING FROM PUBLIC KEY BIO
            DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                    DLT_STRING("Cannot read PubKey from BIO"));
            l_bResult = FALSE;
            break;
        }
        else
        {
            vCorrectPublicKeyString(l_sPubKey);
        }

        //Extract the certificate's issuer name
        l_pIssuerName = X509_NAME_oneline(X509_get_issuer_name(l_pCert),NULL,0);
        if(l_pIssuerName == NULL)
        {
            DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                    DLT_STRING("():l_pIssuerName is NULL from X509_get_issuer_name"));
            l_bResult = FALSE;
            break;
        }

        if(g_sPublicKeyCorrected != NULL)
        {
            DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+bExtractCertInfo: "),
                    DLT_STRING(g_sPublicKeyCorrected));
        }
        else
        {
            DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+bExtractCertInfo: "),
                    DLT_STRING("g_sPublicKeyCorrected is NULL"));
            l_bResult = FALSE;
            break; 
        }
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+bExtractCertInfo: "),
                    DLT_STRING(l_pIssuerName));

        //Compute the hash here
        if(!bComputeHash(g_sPublicKeyCorrected,l_pIssuerName))
        {
            DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                    DLT_STRING("():Failed to compute cert hash"));
            l_bResult = FALSE;
            break;
        }
    }while(0);

    if(l_pCert != NULL)
        X509_free(l_pCert);

    free(l_sPubKey);

    if(l_pIssuerName != NULL)
        free(l_pIssuerName);

    if(l_pCertbio != NULL)
        BIO_free_all(l_pCertbio);

    BIO_free_all(l_pPubKeyBio);

    return l_bResult;
}

