#include <glib.h>
#include <gio/gio.h>
#include "dlt/dlt.h"
#include "SeamlessMidw_HMI_DBusHandler.h"
#include "seamlessMidw_DBusHandler.h"
#include "seamless_controller.h"
#include "SeamlessMidw_HMI_enums.h"
#include "TrustedDeviceList_Handler.h"
#include "SeamlessMidw_SPS_DBusHandler.h"
#include "Seamless_WBL_DBusHandler.h"

DLT_DECLARE_CONTEXT(SPM_HMI);

static SeamlessMiddlewareHMI* pSeamlessHMIProxyObj = NULL;
static gboolean               gPairingInProgress =   FALSE;

/***************************************************************************
* Function:     bExportSeamlessHMI
* Description:  This method exposes the seamless pairing HMI Interface
* Parameters:   void
* Return:       void
****************************************************************************/
gboolean bExportSeamlessHMI()
{ 
    DLT_REGISTER_CONTEXT(SPM_HMI,"SMHM","Seamless midw HMI Interface logging");
    DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

    if(poGetSeamlessHMIProxyObj() == NULL)
    {
        DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), DLT_STRING(
            "():seamless HMI Proxy is NULL, failed to expose interface"));
        return FALSE;
    }

    GError *error = NULL;
    GDBusConnection* connection = NULL;

    if((connection = poGetSPMDBusConn()) == NULL)
    {
        DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): poGetSPMDBusConn returned NULL"));
        return FALSE;
    }

    if (!g_dbus_interface_skeleton_export(
             G_DBUS_INTERFACE_SKELETON(poGetSeamlessHMIProxyObj()),
                                           connection,
                  "/com/bosch/SeamlessMiddleware/HMI",
                  &error) )
    {
        if (error)
        {
            DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
			    DLT_STRING("():Seamless HMI interface export error"),
 				DLT_STRING(error->message));

            g_clear_error(&error);
        }
        else DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
		    DLT_STRING("():Seamless HMI interface export error"));
        return FALSE;
    }
    else 
    {
        DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), DLT_STRING(
                 "():Seamless HMI interface skeleton exported successfully"));
    }

    if(!bLoadTrustedDeviceInfo())
    {
        DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
              DLT_STRING("Failed loading trusted device info"));
    }
   
    g_signal_connect(poGetSeamlessHMIProxyObj(), "handle-update-trusted-device",
                     G_CALLBACK (handle_update_trusted_device), NULL);
     
     DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
     return TRUE;
}

/**************************************************************************
* Function:    poGetSeamlessHMIProxyObj
* Description: Creates and returns SeamlessPairing proxy object
* Parameters:  void
* Return:      CertificateManager* 
***************************************************************************/
SeamlessMiddlewareHMI* poGetSeamlessHMIProxyObj()
{
    if(pSeamlessHMIProxyObj == NULL)
          pSeamlessHMIProxyObj = seamless_middleware_hmi_skeleton_new();
    return pSeamlessHMIProxyObj;
}

/*************************************************************************
* Function:    vUnExportSeamlessHMI
* Description: unexports the Seamless HMI Interface
* Parameters:  void
* Return:      void
**************************************************************************/
void vUnExportSeamlessHMI()
{
    DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING("+"),
                 DLT_STRING(__FUNCTION__));
    if(pSeamlessHMIProxyObj != NULL)
    {
         g_dbus_interface_skeleton_unexport(
		           G_DBUS_INTERFACE_SKELETON(pSeamlessHMIProxyObj));
         g_clear_object(&pSeamlessHMIProxyObj);
    }
    vDestroyHashTable();

    DLT_LOG(SPM_HMI,DLT_LOG_INFO, DLT_STRING("-"),
                      DLT_STRING(__FUNCTION__));
    DLT_UNREGISTER_CONTEXT(SPM_HMI);
}  
  
/**************************************************************************
* Function:    handle_update_trusted_device 
* Description: This method handles the UpdateTrustedDevice
               request from HMI 
* Parameters:  proxy object, invocation pointer, data, user data
* Return:      gboolean
****************************************************************************/
gboolean  handle_update_trusted_device( SeamlessMiddlewareHMI *object,
                                        GDBusMethodInvocation *invocation,
                                        guchar                 action,
                                        GVariant*              deviceInfo,
                                        gpointer               user_data
                                      ) 
{
    DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING("+"),
                DLT_STRING(__FUNCTION__));
    DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
        DLT_STRING("(): Seamless Action requested = "), DLT_UINT((guint) action));

    //guint16 status = uValidateSeamlessAction((SeamlessPairingAction) action);
    guint16 status = SP_REQUEST_REJECTED;
    gboolean pairInProgress = bIsPairingInProgress();

     switch(action)
     {
         case ADD_DEVICE:
         {
             if((pairInProgress) || (!bWBLAvailable()) || (!bSPSIfaceAvailable()))
             {
                  DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                      DLT_STRING("(): Add Device request rejected"));
                  status = pairInProgress ? SP_REQUEST_REJECTED : 
                                                 SP_REQUEST_INTERNAL_ERROR; 
             }
             else
             {               
                   seamless_middleware_hmi_complete_update_trusted_device (
                                                poGetSeamlessHMIProxyObj(),
                                                               invocation,
                                                     SP_REQUEST_ACCEPTED  );
                   vStartSeamlessPairing();
                   return TRUE; 
             }
         }
         break;

         case CANCEL_PAIR:
         {  
              if(pairInProgress)
              {
                  DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                             DLT_STRING("(): Proccessing cancel request"));
                  status = SP_REQUEST_ACCEPTED;
                  vCloseSeamlessSession(SP_CANCELLED, SP_ERR_NO_ERROR); 
              }
              else
                 status = SP_REQUEST_REJECTED;
         }
         break;

         case REMOVE_DEVICE:
         {
              if(pairInProgress)
              {
                  DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                     DLT_STRING("(): Pair in progress, delete request rejected"));
                  status = SP_REQUEST_REJECTED;
              }   
              else
              {
                  if(!bRemoveTrustedDevice(deviceInfo))
                  {
                      DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                          DLT_STRING("(): bRemoveTrustedDevice failed"));   
                      status = SP_REQUEST_INTERNAL_ERROR;
                  }
                  else 
                     status = SP_REQUEST_ACCEPTED;  
              }
         }   
         break; 
 
         case REMOVE_ALL:
         {
                 GVariant* trustedList = 
                       seamless_middleware_hmi_get_trusted_device_list(
                                           poGetSeamlessHMIProxyObj() );

                  if(!bRemoveAllTrustedDevices(trustedList))
                  {
                      DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                          DLT_STRING("(): bRemoveAllTrustedDevices failed"));
                      status = SP_REQUEST_INTERNAL_ERROR;
                  }
                  else
                     status = SP_REQUEST_ACCEPTED;
         } 
         break; 
     
         case UPDATE_DEVICE:
         {
                 if(!bUpdateTrustedDeviceListInfo(deviceInfo, UPDATE_DEVICE)) 
                 {
                     status = SP_REQUEST_INTERNAL_ERROR;
                     DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                        DLT_STRING("(): bUpdateTrustedDeviceListInfo failed"));
                 } 
                 else 
                   status = SP_REQUEST_ACCEPTED;
         }
         break;          

         default:
                 status = SP_REQUEST_REJECTED;
                 DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                               DLT_STRING("(): Invalid seamless action"));
    }
    seamless_middleware_hmi_complete_update_trusted_device(
                               poGetSeamlessHMIProxyObj(),
                                               invocation,
                                                   status );
    DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING("-"),
                  DLT_STRING(__FUNCTION__));
    return TRUE;
}

/***************************************************************************
* Function:    vUpdateTrustedDeviceList
* Description: This method updates the property 'TrustedDeviceList'
* Parameters:  devices list as GVariant
* Return:      void
****************************************************************************/
void vUpdateTrustedDeviceList(GVariant* deviceList)
{
    DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING("+"),
                DLT_STRING(__FUNCTION__));

    if(poGetSeamlessHMIProxyObj()== NULL)
    {
         DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): Seamless HMI proxy is NULL"));
         return;
    }

    seamless_middleware_hmi_set_trusted_device_list(
                                         poGetSeamlessHMIProxyObj(),
                                         deviceList 
                                                   );
    DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING("-"),
                  DLT_STRING(__FUNCTION__));
} 

/***************************************************************************
* Function:    vEmitSeamlessPairingStatus
* Description: This method emits the signal 'SeamlessPairingStatus'
* Parameters:  status, error code, deviceInfo
* Return:      void
****************************************************************************/
void vEmitSeamlessPairingStatus( guint16 status,
                                 guint16 errorCode,
                                 GVariant* deviceInfo )
{
    DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING("+"),
                DLT_STRING(__FUNCTION__));

    gPairingInProgress = FALSE; 

    if(poGetSeamlessHMIProxyObj()== NULL)
    {
        DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): Seamless HMI proxy is NULL"));
        return;
    }
    if(!deviceInfo)
    {
        DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
             DLT_STRING("(): NULL device info"));   
        return;
    }       
    seamless_middleware_hmi_emit_seamless_pairing_status( poGetSeamlessHMIProxyObj(),
                                                          deviceInfo,
                                                          status, 
                                                          errorCode 
                                                        );

    DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING("-"),
                DLT_STRING(__FUNCTION__));     
}
 
/***************************************************************************
* Function:    bEmitQRCodeUpdate
* Description: This method emits the signal 'QRCodeUpdate'
* Parameters:  qrCodeData
* Return:      gboolean
****************************************************************************/
gboolean bEmitQRCodeUpdate(char* qrFileName)
{
    DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING("+"),
                DLT_STRING(__FUNCTION__));

    gPairingInProgress = FALSE;

    if(poGetSeamlessHMIProxyObj() ==  NULL)
    {
        DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): Seamless HMI proxy is NULL"));
        return FALSE;
    }
    if(!qrFileName)
    {
        DLT_LOG(SPM_HMI, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
             DLT_STRING("(): NULL qrFileName"));
        return FALSE;
    }
    seamless_middleware_hmi_emit_qrcode_update( poGetSeamlessHMIProxyObj(),
                                                qrFileName                                
                                              );

    DLT_LOG(SPM_HMI, DLT_LOG_INFO, DLT_STRING("-"),
                DLT_STRING(__FUNCTION__));
    return TRUE;  
}

