/*!
*******************************************************************************
* @file             : FC_SeamlessMidw_BT_ClientHandler.cpp
*******************************************************************************
*  - PROJECT:       : Seamless Pairing IPCM
*  - SW-COMPONENT   : SP_IPCM
*  - DESCRIPTION    : Client Handler to Bluetooth Component
*  - COPYRIGHT      : &copy; 2018 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date      | Name          |  Version | Modification
* -----------|---------------|-------------------------------------------------
* 30.05.2018 | IPD5KOR      | 1.0.0    | methods for Bluetooth Client Handler
******************************************************************************/

/******************************************************************************/
/*                         INCLUDES                                           */
/******************************************************************************/
#include "FC_SeamlessMidw_BT_ClientHandler.h"
#include "dlt/dlt.h"

DLT_IMPORT_CONTEXT(SPM_Bluetooth);

/******************************************************************************/
/*			   DEFINES                                            */
/******************************************************************************/
/******************************************************************************/
/*			CCA MESSAGE MAP					      */
/******************************************************************************/
BEGIN_MSG_MAP(fc_seamless_midw_tcl_bt_clienthandler, ahl_tclBaseWork)

         ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_VEHICLEBTFRIENDLYNAME,
                  AMT_C_U8_CCAMSG_OPCODE_STATUS,
                  vHandle_VehicleBTFriendlyName_Status )

         ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_VEHICLEBTADDRESS,
                  AMT_C_U8_CCAMSG_OPCODE_STATUS,
                  vHandle_VehicleBTAddress_Status )

END_MSG_MAP()

/******************************************************************************/
/*      GLOBAL ITEMS                                                          */
/******************************************************************************/
fc_seamless_midw_tcl_bt_clienthandler* 
fc_seamless_midw_tcl_bt_clienthandler::m_poBluetoothClientHandler = NULL;

/******************************************************************************/
/*                                                                            */
/*          METHODS                                                           */
/*                                                                            */
/******************************************************************************/

/******************************************************************************
* Function:    fc_seamless_midw_tcl_bt_clienthandler
* Description: single parameter constructor
* Parameters:  main app pointer
* Return:      void
******************************************************************************/
fc_seamless_midw_tcl_bt_clienthandler::fc_seamless_midw_tcl_bt_clienthandler(
         seamlessMidwMainApp *poMainAppl)
    : ahl_tclBaseOneThreadClientHandler(
          (ahl_tclBaseOneThreadApp*)poMainAppl,
          CCA_C_U16_SRV_FB_BLUETOOTHSETTINGS, /* ID of used Service */
          MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION, /* MajorVersion of Service */
          MOST_BTSETFI_C_U16_SERVICE_MINORVERSION  /* MinorVersion of Service */
                                       )
{
    DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO,DLT_STRING("+"), 
	                       DLT_STRING( __FUNCTION__));    
    //Initialize private data members
    m_seamlessBTFriendlyName = "";
    m_seamlessBTAddress = "";
 
    //Register for BT settings updates
    vAddAutoRegisterForProperty( MOST_BTSETFI_C_U16_VEHICLEBTADDRESS);
    vAddAutoRegisterForProperty( MOST_BTSETFI_C_U16_VEHICLEBTFRIENDLYNAME);
	
    DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO,DLT_STRING("-"), 
	                       DLT_STRING( __FUNCTION__)); 
}

/******************************************************************************
* Function:    ~fc_gateway_tcl_bluetoothclienthandler
* Description: destructor
* Parameters:  void
* Return:      void
******************************************************************************/
fc_seamless_midw_tcl_bt_clienthandler::~fc_seamless_midw_tcl_bt_clienthandler()
{
    DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO,DLT_STRING("+"), 
	                       DLT_STRING( __FUNCTION__));

    vRemoveAutoRegisterForProperty(MOST_BTSETFI_C_U16_VEHICLEBTADDRESS);
    vRemoveAutoRegisterForProperty(MOST_BTSETFI_C_U16_VEHICLEBTFRIENDLYNAME);

    if (m_poBluetoothClientHandler != NULL)
    {
        delete m_poBluetoothClientHandler;
        m_poBluetoothClientHandler = NULL;
    }

	DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO,DLT_STRING("-"), 
	                        DLT_STRING( __FUNCTION__));
}
/******************************************************************************
* Function:    getInstance
* Description: singleton. Function to get bluetooth client-handler instance
* Parameters:  main app pointer
* Return:      fc_seamless_midw_tcl_bt_clienthandler instance pointer
******************************************************************************/

fc_seamless_midw_tcl_bt_clienthandler*
fc_seamless_midw_tcl_bt_clienthandler::getInstance(seamlessMidwMainApp
                                                   *poMainAppl)
{
    DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO,DLT_STRING("+"),
                           	DLT_STRING( __FUNCTION__));
                
    if (m_poBluetoothClientHandler == NULL)
    {
        m_poBluetoothClientHandler =
                new fc_seamless_midw_tcl_bt_clienthandler(poMainAppl);
    }

    DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO,DLT_STRING("-"), 
	                       DLT_STRING( __FUNCTION__));
    return m_poBluetoothClientHandler;
}

/******************************************************************************
* Function:    vOnServiceAvailable
* Description: CCA callback function
* Parameters:  void
* Return:      void
******************************************************************************/
tVoid fc_seamless_midw_tcl_bt_clienthandler::vOnServiceAvailable() 
{
    DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO,DLT_STRING("+-"), DLT_STRING( __FUNCTION__));
}

/******************************************************************************
* Function:    vOnServiceUnavailable
* Description: CCA callback function
* Parameters:  void
* Return:      void
******************************************************************************/
tVoid fc_seamless_midw_tcl_bt_clienthandler::vOnServiceUnavailable() 
{
    DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO,DLT_STRING("+-"), 
	                       DLT_STRING( __FUNCTION__));
}

/*******************************************************************************
 ** FUNCTION:   vHandle_VehicleBTFriendlyName_Status(..)
 ******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for VehicleBTFriendlyName property status
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 ******************************************************************************/
tVoid fc_seamless_midw_tcl_bt_clienthandler::
              vHandle_VehicleBTFriendlyName_Status(amt_tclServiceData* poMessage)
{
   DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO,DLT_STRING("+"), 
	                       DLT_STRING( __FUNCTION__));
    
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_btsetfi_tclMsgVehicleBTFriendlyNameStatus oVehicleBTFriendlyNameStatus;

   // Unmarshal the FI visitor message to the given FI data object.
   if (oVisitorMsg.s32GetData(oVehicleBTFriendlyNameStatus, 
                  MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      if (oVehicleBTFriendlyNameStatus.sVehicleBTFriendlyName.szValue)
      {
           DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
	  	          DLT_STRING("(): Vehicle BT Friendly Name: "),
   	   DLT_STRING(oVehicleBTFriendlyNameStatus.sVehicleBTFriendlyName.szValue));
           m_seamlessBTFriendlyName = std::string(
                       oVehicleBTFriendlyNameStatus.sVehicleBTFriendlyName.szValue);
      }
      else 
      {
           DLT_LOG(SPM_Bluetooth,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
  		                   DLT_STRING("(): NULL friendly name"));
      } 				   
   }
   else
	   DLT_LOG(SPM_Bluetooth,DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
		    DLT_STRING("(): Failed to fetch friendly name"));
			
   oVehicleBTFriendlyNameStatus.vDestroy();
   DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO,DLT_STRING("-"), 
	                       DLT_STRING( __FUNCTION__));
}  


/*******************************************************************************
 ** FUNCTION:   vHandle_VehicleBTAddress_Status(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for VehicleBTAddress property status
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 ******************************************************************************/
tVoid fc_seamless_midw_tcl_bt_clienthandler::
              vHandle_VehicleBTAddress_Status(amt_tclServiceData* poMessage)
{
   DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO,DLT_STRING("+"),
                               DLT_STRING( __FUNCTION__));
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_btsetfi_tclMsgVehicleBTAddressStatus vehicleBTAddressStatus;

   // Unmarshal the FI visitor message to the given FI data object.
   if (oVisitorMsg.s32GetData(vehicleBTAddressStatus, 
                  MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      if (vehicleBTAddressStatus.sVehicleBTAddress.szValue)
      {
           DLT_LOG(SPM_Bluetooth, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                            DLT_STRING("(): Vehicle BT Raw address is "), 
           DLT_STRING(vehicleBTAddressStatus.sVehicleBTAddress.szValue));
           m_seamlessBTAddress = std::string(
                        vehicleBTAddressStatus.sVehicleBTAddress.szValue);
           for(int i = 2; i < m_seamlessBTAddress.length(); i+= 3)
           {
                m_seamlessBTAddress.insert(i, 1 , ':');
           }  
      }
   }
   else
   {
         DLT_LOG(SPM_Bluetooth, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Failed fetching vehicleBTAddressStatus"));
   }

   vehicleBTAddressStatus.vDestroy();
   DLT_LOG(SPM_Bluetooth,DLT_LOG_INFO,DLT_STRING("-"),
                               DLT_STRING( __FUNCTION__));
}
 
/*****************************************************************************
 ** FUNCTION:   iFetchSeamlessBTName(..)
 ****************************************************************************/
/* DESCRIPTION:
 *   This function provides BT friendly name of the vehicle
 *
 *  PARAMETERS:
 *     IN: sBluetoothName     :  BT friendly name will be copied to this arg
 *     IN: uBluetoothNameSize : Size of arg sBlueoothName
 *  RETURNVALUE: int  : 0 For Failure,
                        1 For Success
*****************************************************************************/
int fc_seamless_midw_tcl_bt_clienthandler::iFetchSeamlessBTName(
                                   char*        sBluetoothName, 
                               unsigned int uBluetoothNameSize )
{
    DLT_LOG(SPM_Bluetooth, DLT_LOG_INFO, DLT_STRING("+"),
                               DLT_STRING( __FUNCTION__));

    unsigned int l_uBTFriendlyNameSize = m_seamlessBTFriendlyName.length();
    if(!l_uBTFriendlyNameSize)
    {
         DLT_LOG(SPM_Bluetooth, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): BT Name not set"));
         return 0;
    } 
    if((!sBluetoothName) ||
         (uBluetoothNameSize <=  l_uBTFriendlyNameSize))
    {
         DLT_LOG(SPM_Bluetooth, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): Null or bad input"));
          return 0;
    }
    strcpy(sBluetoothName, m_seamlessBTFriendlyName.c_str());
    DLT_LOG(SPM_Bluetooth, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
        DLT_STRING("():Seamless BT Name is "), DLT_STRING(sBluetoothName));
 
    DLT_LOG(SPM_Bluetooth, DLT_LOG_INFO, DLT_STRING("-"),
                               DLT_STRING( __FUNCTION__));
    return 1;
}

/****************************************************************************
 ** FUNCTION:   iFetchSeamlessBTAddress(..)
 ****************************************************************************/
/* DESCRIPTION:
 *   This function provides the BT Address of the vehicle
 *
 *  PARAMETERS:
 *     IN: sBlueoothAddress  : The BT Address will be copied to this arg
 *     IN: uBluetoothAddressSize : Size of arg uBluetoothAddressSize 
 *  RETURNVALUE: int  :  0 For Failure,
                         1 For Success        
*****************************************************************************/
int fc_seamless_midw_tcl_bt_clienthandler::iFetchSeamlessBTAddress(
                               char*        sBluetoothAddress,
                               unsigned int uBluetoothAddressSize )
{
    DLT_LOG(SPM_Bluetooth, DLT_LOG_INFO, DLT_STRING("+"),
                               DLT_STRING( __FUNCTION__));
    unsigned int l_uBTAddressSize = m_seamlessBTAddress.length();
    if(!l_uBTAddressSize)
    {
         DLT_LOG(SPM_Bluetooth, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): BT Address not set"));
         return 0;
    }
    if((!sBluetoothAddress) ||
         (uBluetoothAddressSize <= l_uBTAddressSize))
    {
         DLT_LOG(SPM_Bluetooth, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): Null or bad input"));
          return 0;
    }
    strcpy(sBluetoothAddress, m_seamlessBTAddress.c_str());

    DLT_LOG(SPM_Bluetooth, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
      DLT_STRING("():Seamless BT Address is "), DLT_STRING(sBluetoothAddress));


    DLT_LOG(SPM_Bluetooth, DLT_LOG_INFO, DLT_STRING("-"),
                               DLT_STRING( __FUNCTION__));
    return 1;
}

#ifdef __cplusplus
extern "C"
{
#endif
/***************************************************************************
FUNCTION:   iFetchSeamlessBTAddressWrapper(..)
****************************************************************************/
/* DESCRIPTION:
 *   Wrapper function to fetch the vehicle Bluetooth address
 *
 *  PARAMETERS:
 *     IN: sBlueoothAddress  : The BT Address will be copied to this arg
 *     IN: uBluetoothAddressSize : Size of arg uBluetoothAddressSize
 *  RETURNVALUE: int  :  0 For Failure,
                         1 For Success
****************************************************************************/
int iFetchSeamlessBTAddressWrapper( char*        sBluetoothAddress,
                                    unsigned int uBluetoothAddressSize )
{
    if(!fc_seamless_midw_tcl_bt_clienthandler::m_poBluetoothClientHandler)
    {
        DLT_LOG(SPM_Bluetooth, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): m_poBluetoothClientHandler is NULL"));
        return 0;
    } 
   
    return fc_seamless_midw_tcl_bt_clienthandler::
       m_poBluetoothClientHandler->iFetchSeamlessBTAddress( sBluetoothAddress, 
                                                            uBluetoothAddressSize);
}

/****************************************************************************
FUNCTION:   iFetchSeamlessBTNameWrapper(..)
*****************************************************************************/
/* DESCRIPTION:
 *   Wrapper function to fetch the vehicle Bluetooth friendly name
 *
 *  PARAMETERS:
 *     IN: sBlueoothName  : The BT Name will be copied to this arg
 *     IN: uBluetoothNameSize : Size of arg sBlueoothName
 *  RETURNVALUE: int  :  0 For Failure,
                         1 For Success
*****************************************************************************/
int iFetchSeamlessBTNameWrapper( char*        sBluetoothName,
                                 unsigned int uBluetoothNameSize )
{
    if(!fc_seamless_midw_tcl_bt_clienthandler::m_poBluetoothClientHandler)
    {
        DLT_LOG(SPM_Bluetooth, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): m_poBluetoothClientHandler is NULL"));
        return 0;
    }

   return fc_seamless_midw_tcl_bt_clienthandler::
       m_poBluetoothClientHandler->iFetchSeamlessBTName( sBluetoothName,
                                                         uBluetoothNameSize);
}

#ifdef __cplusplus
}
#endif
