#include "core/profileData.h"

#include "piviConfig.h"
#include "pivi/core/ServicePIVI.h"
#include "asf/core/Application.h"
#include "asf/core/ComponentContainer.h"
#include "asf/core/ComponentDescription.h"
#include "asf/dbus/DBusConnector.h"
#include "pivi/core/ProfileCfgComp.h"
#include "asfComp/app/core/ProfileMngr.h"
#include "pthread.h"

#define DP_S_IMPORT_INTERFACE_BASE
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_if.h"

#include <core/profileApp.h>
#include <pivi/core/ProfileCfgComp.h>
#include "pivi/core/profilePIVI_VCANClient.h"
#include <pivi/config/profilePIVIDefines.h>

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#include <pivi/core/profilePIVITrace.h>
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PROFILEMANAGER_APP_PIVI
#include "trcGenProj/Header/piviConfig.cpp.trc.h"
#endif

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_KDS_FI
#include "dp_kds_if.h" //include to read kds
#endif

using namespace std;



static const profilePIVIConf aProfilePIVIConfig[MAX_PROFILE_PIVI] =
{     /*name, 	imageId, IKeyValue, WelcomeGuidanceSetting, profileStatus*/
		{"ZEROTH_PROFILE", 	0, 		5, 		welcomeGuidanceOn,	usEmpty},
		{"Guest", 	0, 		5, 		welcomeGuidanceOn,	usGuest},
		{"", 		0, 		5, 		welcomeGuidanceOn,	usEmpty},
		{"", 		0, 		5, 		welcomeGuidanceOn,	usEmpty},
		{"", 		0, 		5, 		welcomeGuidanceOn,	usEmpty}
};

static const std::string aClientDatabase[] =
{
	"ASF:AppHmi_Homescreen",
	"ASF:AppHmi_System",
	"CCA:dispvidctrl",
	"ASF:AppHmi_Phone",
	"ASF:AppHmi_Sds"
};

extern "C"
{
	void* getConfig()
	{
		return new pivi::core::Piviconfig();
	}
}

namespace pivi{ namespace core{

Piviconfig * Piviconfig::m_PIVIConfig = NULL;

pthread_t g_CopyTimerThread1;

Piviconfig::Piviconfig():m_maxPIVIProfiles(MAX_PROFILE_PIVI)
{
	m_PIVIData = new profileDataPIVI();
	m_pVCANClient = new profilePIVI_VCANClient();
	m_PIVIData->Init(m_maxPIVIProfiles);
	m_PIVIConfig =this;
	m_DigitalAssistant = DIGITAL_ASSISTANT_OFF;
	m_lastDeletedProfile = INVALID_USER;
	m_createdUser = INVALID_USER;
	m_InitOnPowerMode = false;

	const int size = sizeof aClientDatabase / sizeof(std::string);
	for(int i = 0;i < size; i++)
	{
		m_expectedClients.push_back(profileMngr::Client(0,aClientDatabase[i]));
	}
}

void Piviconfig::startASFClients(::boost::shared_ptr< asf::dbus::DBusConnector > & DBus,::boost::shared_ptr< asf::core::ComponentContainer > & App)
{
	ETG_TRACE_USR4(("ProfilePIVI PIVICONFIG: Start ASF client"));

	::asf::core::Application* app = ::asf::core::Application::getApplication();

	::boost::shared_ptr< ::asf::core::ComponentFactory< ::pivi::core::ProfileCfgComp > > piviServiceFactory(new ::asf::core::ComponentFactory< ::pivi::core::ProfileCfgComp >);
		::asf::core::ComponentDescriptionSharedPtr piviServiceDescription(new ::asf::core::ComponentDescription(1, "piviService", true,  false, piviServiceFactory));
		App->addComponentDescription(piviServiceDescription);

			// Port 'profilePIVIPort'

			::asf::core::ProvidedPort& piviService_profilePIVIPort = piviServiceDescription->addProvidedPort (::asf::core::ProvidedPort ("bosch.cm.profilePIVI", "profilePIVIPort"));
			(void)piviService_profilePIVIPort; // prevent unused variable warning


			// Expression

			DBus->exportByName (piviService_profilePIVIPort, "/com/bosch/cm/profilePivi");
			
}
bool Piviconfig::InitOnPowerMode(uint8_t & startUser)
{
	uint8_t tempUserID = startUser;
	ETG_TRACE_USR4(("ProfilePIVI PIVICONFIG: InitOnPowerMode"));
	m_InitOnPowerMode = true;
	if(getDigitalAssistant() && getLoginKeylinkStatus())
	{
		startUser = startupUserIdentification();
		emitVCANSignalsOnStatrtUp(startUser);
	}
	else
	{
		startUser = PIVI_DEFAULT_PROFILE;
		emitDefRegistrationVCANSignals();
	}
	if(startUser == tempUserID) //user not changed
	{
		ETG_TRACE_USR4(("ProfilePIVI PIVICONFIG: InitOnPowerMode Loading same user profile"));
		updatePropertiesOnAccONTrigger();

	}
	else //user changed
	{
		ETG_TRACE_USR4(("ProfilePIVI PIVICONFIG: InitOnPowerMode Loading new user: %d ", startUser));
	}
	ETG_TRACE_USR4(("Piviconfig:: InitOnPowerMode startUser: %d",startUser));

	return true;
}
// ************************** Init profile data ********************************************************
bool Piviconfig::Init(uint8_t & startUser)
{
	configureDigitalAssistant();// read from KDS

	if (getDigitalAssistant())
	{
		if(!m_pVCANClient->writeInitialFlag(en_Initializing))							// write initializing status to CSM
		{
			ETG_TRACE_ERR(("Profile Piviconfig - Initializing status update failed"));
		}
	}
	if(getDigitalAssistant() && getLoginKeylinkStatus())
	{
		startUser = startupUserIdentification();
		emitVCANSignalsOnStatrtUp(startUser);												// emit required VCAN signals on startup user identification
	}
	else
	{
		startUser = PIVI_DEFAULT_PROFILE;
		emitDefRegistrationVCANSignals();
	}
	ETG_TRACE_USR4(("Piviconfig::startUser: %d",startUser));

	return true;
}
void Piviconfig::configureDigitalAssistant()
{
	m_DigitalAssistant = DIGITAL_ASSISTANT_ON; //hardcoded until configuration is available
	return;
/*
	tU8 digitalAssistant = 0x00;
	if (DP_S32_NO_ERR != DP_s32GetConfigItem("SystemConfiguration1", "DigitalAssistant", &m_DigitalAssistant, 1))
	{
		ETG_TRACE_FATAL(("KDS read Digital Assistant failed"));
	}
	else
	{
		ETG_TRACE_USR4(("ProfileHandler::readDigitalAssistantConfiguration: Config : 0x%x", m_DigitalAssistant));
	}

	*/
}


// ************************** load profile data ********************************************************

bool Piviconfig::load()
{
	ETG_TRACE_USR4(("ProfilePIVI Piviconfig: loading Datapools"));
	return m_PIVIData->load();
}

// ************************** reset profile ********************************************************

bool Piviconfig:: deleteProfileData(profileMngr::profileData & Data,unsigned char ID)
{
	ETG_TRACE_USR4(("PIVICONFIG-deleteProfileData userID:%d",ID));
	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
	   return false;
	}
	setLastdeletedProfile(ID);
	return doDefSet(Data, ID);
}
// ************************** isSetProfileNameAllowed ********************************************************

bool Piviconfig:: isSetProfileNameAllowed(profileMngr::profileData & Data,unsigned char userID,const std::string& profileName)
{
	ETG_TRACE_USR4(("isSetProfileNameAllowed userID:%d",userID));
	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
	   ETG_TRACE_ERR(("'Digital Assistant disabled.."));
	   return false;
	}
	if((userID < maxProfiles()) && (Data.getProfileStatus()[userID] != usGuest) )
	{
		return true;
	}
	return false;
}
// ************************** isSetProfileImageAllowed ********************************************************

bool Piviconfig:: isSetProfileImageAllowed(profileMngr::profileData & Data,unsigned char userID)
{
	ETG_TRACE_USR4(("isSetProfileImageAllowed userID:%d",userID));
	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
	   ETG_TRACE_ERR(("'Digital Assistant disabled.."));
	   return false;
	}
	if(userID < maxProfiles() && (Data.getProfileStatus()[userID] != usGuest))
		return true;
	else
		return false;
}
// ************************** isDeleteProfileAllowed ********************************************************

bool Piviconfig:: isDeleteProfileAllowed(profileMngr::profileData & Data,unsigned char userID)
{
	ETG_TRACE_USR4(("isDeleteProfileAllowed userID:%d",userID));
	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
	   ETG_TRACE_ERR(("'Digital Assistant disabled.."));
	   return false;
	}
	if(userID < maxProfiles() && (Data.getProfileStatus()[userID] == usNormal))
		return true;
	else
		return false;
}

bool Piviconfig::isCopyProfileAllowed(profileMngr::profileData & Data,unsigned char srcID,unsigned char destID)
{
	ETG_TRACE_USR4(("isCopyProfileAllowed  srcID:%d destID:%d",srcID,destID));
	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
	   ETG_TRACE_ERR(("'Digital Assistant disabled.."));
	   return false;
	}

	// Emit VCAN signals to be sent on start of copy profile
	emitVCANSignalsOnCopyProfile(srcID, destID);
	startCopyTimer();									// start 2 sec timer drive copy and 3 for ITS settings for copy process

	return true;
}

bool Piviconfig::doDefSet(profileMngr::profileData & Data)
{

	ETG_TRACE_USR4(("ProfilePIVI Piviconfig:  doDefSet default load"));
	const int mapSize = sizeof aProfilePIVIConfig / sizeof(profilePIVIConf);

	for (int i=0;i<mapSize;i++)
	{
		//filled the default values
		Data.setName(i,aProfilePIVIConfig[i]._profileName);
		Data.setImage(i,aProfilePIVIConfig[i]._profileImageID);
		Data.setStatus(i,aProfilePIVIConfig[i]._profileStatus);
		m_PIVIData->DefsetKey(i,aProfilePIVIConfig[i]._profileIKeyValue);
		m_PIVIData->setWelcomeGuidance(i,aProfilePIVIConfig[i]._profileWelcomeGuidanceSetting);
	}
	m_PIVIData->KeyLinkedProfileDefset();
	emitDefRegistrationVCANSignals(); //send default CAN signals on Restore factory setting

	return true;
}

bool Piviconfig::doDefSet(profileData & Data, unsigned char UserID)
{
	//filled the default values
	Data.setName(UserID,aProfilePIVIConfig[UserID]._profileName);
	Data.setImage(UserID,aProfilePIVIConfig[UserID]._profileImageID);
	Data.setStatus(UserID,aProfilePIVIConfig[UserID]._profileStatus);
	m_PIVIData->DefsetKey(UserID,aProfilePIVIConfig[UserID]._profileIKeyValue);
	m_PIVIData->setWelcomeGuidance(UserID,aProfilePIVIConfig[UserID]._profileWelcomeGuidanceSetting);
	m_PIVIData->SetDelinkUserID(UserID);
	ETG_TRACE_USR4(("ProfilePIVI Piviconfig:  doDefSet reset userid:%d", UserID));
	return true;
}

bool Piviconfig::doChangeName(profileData& Data, unsigned char ID,	std::string& Name)
{
	ETG_TRACE_USR4(("ProfilePIVI Piviconfig:  doChangeName uID:%d",ID));
	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
	   ETG_TRACE_ERR(("'Digital Assistant disabled.."));
	   return false;
	}
	Name = Data.getProfileNames()[ID];
	return true;
}

bool Piviconfig::doChangeImage(profileData& Data, unsigned char ID,	unsigned char Idx)
{
	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
	   ETG_TRACE_ERR(("'Digital Assistant disabled.."));
	   return false;
	}
	return false;
}


void Piviconfig::updatePropertiesOnAccONTrigger()
{
	ETG_TRACE_USR4(("PIVICONFIG-updatePropertiesOnAccONTrigger"));
	ServicePIVI::getInstance().updatePropertiesOnstartup();

	ServicePIVI::getInstance().sendStartupUpdate(true);
	m_InitOnPowerMode = false;
	ServicePIVI::getInstance().sendStartupUpdate(false);
}


void Piviconfig::changeToProfileCompetionStatus(actionState status,
		unsigned char oldUserID, unsigned char newUserID) {
		
	ETG_TRACE_USR4(("PIVICONFIG-changeToProfileCompetionStatus oldUid:%d newUid:%d",oldUserID,newUserID));
	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
	   ETG_TRACE_ERR(("'Digital Assistant disabled.."));
	   return;
	}

	if(status == asSuccess)
	{	//update DPs
		unsigned char key = 0;
		m_pVCANClient->readUserIdSignal(key);

		m_PIVIData->changeActiveProfile(newUserID,key);

		bool WGSettings = m_PIVIData->getWelcomeGuidance(newUserID);
		ServicePIVI::getInstance().sendWelcomeGuidanceActive(WGSettings);
		if(m_InitOnPowerMode)
		{
			updatePropertiesOnAccONTrigger();
		}

	//	emitVCANSignalsOnChangeProfile(newUserID);//emit VCAN signals ; emit from isProfileChangeAllowed
	}

}
void Piviconfig::copyProfileCompletionStatus(actionState status, profileData & Data, unsigned char srcID, unsigned char destID, uint8_t & changeUser)
{
	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
		ETG_TRACE_ERR(("'Digital Assistant disabled.."));
		return;
	}
	changeUser = srcID;//source profile remains as the active profile after copy operation.

}

void Piviconfig::deleteProfileCompletionStataus(actionState status, unsigned char userID,uint8_t & activeUser)
{
	ETG_TRACE_USR4(("PIVICONFIG-deleteProfileCompletionStataus uID:%d",userID));
	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
		ETG_TRACE_ERR(("'Digital Assistant disabled.."));
		return;
	}
	if(status == asSuccess)
	{
		if(activeUser == getLastdeletedProfile()) //if active user is deleted, make Guest as active user                                                                                                            // check whether the deleted user is active user
		{
			activeUser = PIVI_DEFAULT_PROFILE;
			m_pVCANClient->writeCurrentLoginUserInfo(INDEX_GUEST_PROFILE);
		}
		else
		{
			ETG_TRACE_USR4(("Active profile not deleted..! Change user not required."));
		}
		emitVCANSignalsOnDeleteProfile(userID);
	}
}

bool Piviconfig::isProfileChangeAllowed(unsigned char profileIDX, unsigned char activeProfile)
{
	ETG_TRACE_USR4(("PIVICONFIG-isProfileChangeAllowed requestedID:%d activeID:%d",profileIDX,activeProfile));
	bool isAllowed = false;
	bool loginUserChangePermitFlag = false;
	if(!m_pVCANClient->readLoginUserChangePermitFlag(loginUserChangePermitFlag))
	{
		ETG_TRACE_ERR(("LoginUserChangePermitFlag VCAN read error.."));
	}
	else
	{
		ETG_TRACE_USR4(("LoginUserChangePermitFlag: %d",loginUserChangePermitFlag));
	}
	if((true == loginUserChangePermitFlag) && (profileIDX != activeProfile))
	{
		isAllowed = true;
	}
	if(true == loginUserChangePermitFlag)
	{
		emitVCANSignalsOnChangeProfile(profileIDX);
	}
	ETG_TRACE_USR4(("ProfileChangeAllowed :%d",isAllowed));
	return isAllowed;
}

bool Piviconfig::isCreateProfileAllowed(profileData& Data,	const std::string& name, unsigned char& userID)
{
	bool bResult = FALSE;
	unsigned char keyID = KEY_NOT_USED;

	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
	   ETG_TRACE_ERR(("'Digital Assistant disabled.."));
	   return false;
	}

	if(m_pVCANClient->readUserIdSignal(keyID))
	{
		// Checking the current key is valid
		if(keyID != NO_KEYLESS_OPERATION)
		{
			std::vector<enUserStatus>  vProfileStatus = Data.getProfileStatus();
			for(int index=(INDEX_GUEST_PROFILE+1); index < m_maxPIVIProfiles; index++)
			{
				ETG_TRACE_USR4(("PIVICONFIG-isCreateProfileAllowed: profile status: %d", vProfileStatus.at(index)));
				if(vProfileStatus.at(index) == usEmpty)			// Identify first empty slot for user-key mapping
				{
					userID = index;
					bResult = TRUE;
					break;
				}
			}
		}
		else
		{
			ETG_TRACE_ERR(("PIVICONFIG-isCreateProfileAllowed: Keyless operation create profile not permitted"));
		}
	}
	else
	{
		ETG_TRACE_ERR(("PIVICONFIG-isCreateProfileAllowed: Key read error"));
	}

	ETG_TRACE_USR4(("PIVICONFIG-isCreateProfileAllowed: user id: %d   result: %d", userID, bResult));

	return bResult;
}

bool Piviconfig::doCreateProfile(profileData& Data, unsigned char ID,		const std::string& Name, unsigned char Idx)
{
	ETG_TRACE_USR4(("PIVICONFIG-doCreateProfile UserID:%d",ID));
	unsigned char keyID;
	bool bResult = FALSE;

	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
	   ETG_TRACE_ERR(("'Digital Assistant disabled.."));
	   return false;
	}

	bResult = m_pVCANClient->readUserIdSignal(keyID);
	if(bResult)
	{
		Data.setName(ID, Name);
		Data.setImage(ID, Idx);
		Data.setStatus(ID, usNormal);

		m_PIVIData->setKey(ID,keyID);
		m_PIVIData->setWelcomeGuidance(ID,aProfilePIVIConfig[ID]._profileWelcomeGuidanceSetting);
		m_PIVIData->KeyLinkedProfileSetProfile(keyID,ID);
		ETG_TRACE_USR4(("PIVICONFIG-doCreateProfile: new user profile created"));
		m_createdUser = ID;
		bResult = TRUE;
	}
	else
	{
		ETG_TRACE_ERR(("PIVICONFIG-doCreateProfile: read user key failed"));
	}

	return bResult;
}

bool Piviconfig::store()
{
	ETG_TRACE_USR4(("Piviconfig::store"));
	m_PIVIData->store();
	m_PIVIData->storeKeyLinkedProfile();
	return true;
}

void Piviconfig::createProfileCompletionStatus(actionState status,	unsigned char & newUserID)
{
	if(m_DigitalAssistant != DIGITAL_ASSISTANT_ON)
	{
	   ETG_TRACE_ERR(("'Digital Assistant disabled.."));
	   return;
	}
	if(status == asSuccess)
	{
		newUserID = m_createdUser;
		bool WGSettings = m_PIVIData->getWelcomeGuidance(newUserID);
		ServicePIVI::getInstance().sendWelcomeGuidanceActive(WGSettings);
		
		emitVCANSignalsOnCreateProfile(newUserID);
	}
}

unsigned char Piviconfig::maxProfiles()
{
	return m_maxPIVIProfiles;
}

bool Piviconfig::getLoginKeylinkStatus()
{
	bool loginKeyLinkStatus = false;
	if(!m_pVCANClient->readLoginKeyLinkStatus(loginKeyLinkStatus))		// Read VCAN login_key_link_status
	{
		ETG_TRACE_ERR(("Profile Piviconfig - Login key link status read error"));
	}
	return loginKeyLinkStatus;
}

unsigned char Piviconfig::startupUserIdentification()
{
	unsigned char keyID = KEY_NOT_USED;

	bool isKeyMapped = false;
	unsigned char userID = PIVI_DEFAULT_PROFILE;

	// Read VCAN key_id signal
	if(m_pVCANClient->readUserIdSignal(keyID))
	{
		// Check if key is valid
		if((keyID >= KEY1) && (keyID <= KEY4))
		{
			isKeyMapped = m_PIVIData->iskeymapped(keyID);

			if(isKeyMapped)											//Check if key mapped to any user and Key link status
			{
				// Read the userID mapped to current key
				userID = m_PIVIData->getUserIDMapped(keyID);
			}
			else
			{
				ETG_TRACE_ERR(("key not mapped, load GUEST profile"));
			}
		}
		else
		{
			ETG_TRACE_USR4(("Invalid key value received, load GUEST profile"));
		}
	}
	else
	{
		ETG_TRACE_ERR(("Profile Piviconfig - keyid read error"));
	}


	ETG_TRACE_USR4(("Profile Piviconfig keymapped , iskeymapped:%d , key:%d , userID:%d", isKeyMapped, keyID, userID));
	return userID;
}


/*************************** startCopyTimer  ***************************************/

void Piviconfig::startCopyTimer()
{
	ETG_TRACE_USR4(("Piviconfig : create Timer \n"));
	pthread_create(&g_CopyTimerThread1,0,profileCopyTimerHandler,this);
}

//**********************************************************************************

void * Piviconfig::profileCopyTimerHandler(void * pParam)
{
	ETG_TRACE_USR4(("profileCopyTimerHandler :timer proc \n"));
	if (pParam==0) return 0;
	Piviconfig * pPiviConfig= static_cast<Piviconfig*> (pParam);
	sleep(COPY_PF_DRIVE_WAIT_TIME_SECS);									// Sleep 2 secs for copy drive settings
	pPiviConfig->emitSignalOnCopyDriveSettingTimeout();
	//sleep(COPY_PF_ITS_WAIT_TIME_SECS - COPY_PF_DRIVE_WAIT_TIME_SECS);		// sleep(2+1) 3 secs for copy ITS settings
	pPiviConfig->emitSignalOnCopyITSSettingTimeout();

	return 0;
}

void Piviconfig::emitDefRegistrationVCANSignals()
{
	ETG_TRACE_USR4(("profileCopyTimerHandler : emitDefRegistrationVCANSignals"));
	m_pVCANClient->writeCurrentLoginUserInfo(INDEX_GUEST_PROFILE);
	m_pVCANClient->writeUserRegistrationInfo(INDEX_GUEST_PROFILE, en_Registered);
	for(int profile=INDEX_GUEST_PROFILE+1; profile < m_maxPIVIProfiles; profile++)
	{
		m_pVCANClient->writeUserRegistrationInfo(profile, en_NotRegistered);	
	}		
}

void Piviconfig::emitVCANSignalsOnStatrtUp(unsigned char userId)
{
	ETG_TRACE_USR4(("profileCopyTimerHandler : emitVCANSignalsOnStatrtUp"));
	bool res = false;
	std::map<unsigned char,bool> registerdUsers = m_PIVIData->getRegisteredUserList();

	// Update CSM for login user info
	if(!m_pVCANClient->writeCurrentLoginUserInfo(userId))										// Send current login user info to CSM
	{
		ETG_TRACE_FATAL(("PIVI_VCANClient CurrentLoginUserInfo csm write failed"));
	}
	res = m_pVCANClient->writeUserRegistrationInfo(INDEX_GUEST_PROFILE, en_Registered);			// By default, guest profile (user index) will be in registered state for all the keys
	if(!res)
	{
		ETG_TRACE_FATAL(("PIVI_VCANClient UserRegistrationInfo write failed"));
	}

	for(int profile=INDEX_GUEST_PROFILE+1; profile < m_maxPIVIProfiles; profile++)									// Update user Reg status for user1 to user3
	{
		if(registerdUsers[profile] == true)
		{																						// user registered with a valid key
			res = m_pVCANClient->writeUserRegistrationInfo(profile, en_Registered);
			if(!res)
			{
				ETG_TRACE_FATAL(("PIVI_VCANClient UserRegistrationInfo write failed"));
			}
		}
		else
		{
			res = m_pVCANClient->writeUserRegistrationInfo(profile, en_NotRegistered);			// user is not registered with a valid key
			if(!res)
			{
				ETG_TRACE_FATAL(("PIVI_VCANClient UserRegistrationInfo write failed"));
			}
		}
	}
}


void Piviconfig::emitVCANSignalsOnCreateProfile(unsigned char newUserID)
{
	bool res = m_pVCANClient->writeCurrentLoginUserInfo(newUserID);				//write current login user info to CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("PIVI_VCANClient CurrentLoginUserInfo write failed"));
	}

	res = m_pVCANClient->writeUserRegistrationInfo(newUserID, en_Registered);		//write userX Reg info to CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("PIVI_VCANClient UserRegistrationInfo write failed"));
	}
}

void Piviconfig::emitVCANSignalsOnChangeProfile(unsigned char newUserID)
{
	ETG_TRACE_USR4(("emitVCANSignalsOnChangeProfile"));
	bool res = false;
	res = m_pVCANClient->writeCurrentLoginUserInfo(newUserID);				// write CurrentLoginUserInfo status to CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("PIVI_VCANClient CurrentLoginUserInfo csm write failed"));
	}
	res = m_pVCANClient->toggleUserChangeSW(newUserID);					// write toggleUserChangeSW status for userX to CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("PIVI_VCANClient UserChangeSW csm write failed"));
	}
}

void Piviconfig::emitVCANSignalsOnDeleteProfile(unsigned char userID)
{
	bool res;
	res = m_pVCANClient->writeUserRegistrationInfo(userID, en_NotRegistered);			// make deleted user as 'not registered' in CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("PIVI_VCANClient UserRegistrationInfo write failed"));
	}
}

void Piviconfig::emitVCANSignalsOnCopyProfile(unsigned char srcID,unsigned char destID)
{
	bool res = false;
	res = m_pVCANClient->writeCSMCopySetting(en_CSMSignalCopyDriveSetting, en_Copy);		// write CopyDriveSetting status to CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("PIVI_VCANClient CopyDriveSetting csm write failed"));
	}
	res = m_pVCANClient->writeCSMCopySetting(en_CSMSignalCopyITSSetting, en_Copy);			// write CopyITSSetting status to CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("PIVI_VCANClient CopyITSSetting csm write failed"));
	}

	res = m_pVCANClient->writeCopyFromUser(srcID);											// write CopyFromUser status to CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("PIVI_VCANClient CopyFromUser csm write failed"));
	}
	res = m_pVCANClient->writeCopyToUser(destID);											// write CopyToUser status to CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("PIVI_VCANClient CopyToUser csm write failed"));
	}
}

void Piviconfig::emitSignalOnCopyDriveSettingTimeout()
{
	bool res;

	ETG_TRACE_USR4(("emitSignalOnCopyDriveSettingTimeout \n"));
	res = m_pVCANClient->writeCSMCopySetting(en_CSMSignalCopyDriveSetting, en_NotCopy);	// write CopyDriveSetting status to CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("Piviconfig CopyDriveSetting csm write failed"));
	}
	res = m_pVCANClient->writeCopyFromUser(NOT_COPY_USER);									// write CopyFromUser status to CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("Piviconfig CopyFromUser csm write failed"));
	}
	res = m_pVCANClient->writeCopyToUser(NOT_COPY_USER);									// write CopyToUser status to CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("Piviconfig CopyToUser csm write failed"));
	}
}

void Piviconfig::emitSignalOnCopyITSSettingTimeout()
{
	ETG_TRACE_USR4(("emitSignalOnCopyITSSettingTimeout \n"));
	bool res = m_pVCANClient->writeCSMCopySetting(en_CSMSignalCopyITSSetting, en_NotCopy);	// write CopyITSSetting status to CSM
	if(!res)
	{
		ETG_TRACE_FATAL(("Piviconfig CopyITSSetting csm write failed"));
	}
}

unsigned char Piviconfig::getLastdeletedProfile()
{
	return m_lastDeletedProfile;
}
void Piviconfig::setLastdeletedProfile(unsigned char userID)
{
	m_lastDeletedProfile = userID;
}

} 
}

