/*****************************************************************************
* FILE:         ServiceProvider.h
* PROJECT:      A-IVI project
* SW-COMPONENT: profilemanager
*----------------------------------------------------------------------------
* DESCRIPTION:  core logic for Online Profile Manager
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim
*****************************************************************************/

#ifndef _SERVICEPROVIDER_H_
#define _SERVICEPROVIDER_H_

#include <boost/shared_ptr.hpp>
#include <asf/core/Logger.h>
#include <asf/core/Proxy.h>

#include <string>
#include <list>

#include "IServiceProvider.h"
#include "IServiceAvailable.h"
#include "AdapterBase.h"
#include "ServiceProviderTypes.h"

//using namespace ::de::bosch::cm::ConfigManager::ConfigManagement;
//using namespace com::bosch::cm::dumm::DummService;

namespace de{ namespace bosch { namespace cm { namespace ConfigManager{namespace ConfigManagement { class ConfigManagementProxy ; }}}}}
namespace com{ namespace bosch { namespace cm { namespace dumm{namespace DummService { class DummServiceProxy ; }}}}}
namespace bosch { namespace cm { namespace profile { class ProfileProxy ; }}}
namespace tcu_usb_main_fi{ class Tcu_usb_main_fiProxy ; }

namespace profileMngr
{

class ConfigMgmtSvcAdapter;
class ProfileSvcAdapter;
class DummSvcAdapter;
class UsbTcuSvcAdapter;

class ServiceProvider
   : public ::profileMngr::IServiceProvider
   , public ::asf::core::ServiceAvailableIF
{
public:
   //definition of the port name for each provided proxy - must match the require statements in the cmc
   static const ::std::string CONFIGURATION_MGMT_SVC_PROXY_NAME;
   static const ::std::string PROFILE_SVC_PROXY_NAME;
   static const ::std::string DUMM_SVC_PROXY_NAME;
   static const ::std::string USBTCU_SVC_PROXY_NAME;

   ServiceProvider(IServiceAvailable& serviceAvailableReceiver, const std::list<AdapterType>& requiredAdapter);
   virtual ~ServiceProvider();

   /**
    * @brief Releases all proxies in use, marks all adapters as invalid.
    */
   void clear();

   /**
    * @brief Checks if all required services are up. If so the onAvailable() method is called of the IServiceAvailable interface
    * if it isn't already called.
    * This method is internally used and should be used by user on construction time to force an available check.
    */
   void allServicesAvailable();

   /** @return true in case onAvailable is already called otherwise false is returned.
    */
   bool areAllServicesAvailable() const
   {
      return( m_serviceAvailable );
   }

   /**
    * @brief Creates a new Adapter object if not exists, put it under control and returns it
    * The caller always gets an adapter if it was requested by setRequiredAdapters only
    * Otherwise an empty shared pointer is returned.
    *
    * @return a shared pointer instance of an RcFiAdapter.
    */
   ::boost::shared_ptr<ConfigMgmtSvcAdapter> getConfigMgmtSvcAdapter();
   ::boost::shared_ptr<ProfileSvcAdapter> getProfileSvcAdapter();
   ::boost::shared_ptr<DummSvcAdapter> getDummSvcAdapter();
   ::boost::shared_ptr<UsbTcuSvcAdapter> getUsbTcuSvcAdapter();

   // [Begin] ServiceAvailableIF implementation
   virtual void onAvailable(
      const ::boost::shared_ptr<asf::core::Proxy>&,
      const asf::core::ServiceStateChange& state);
   virtual void onUnavailable(
      const ::boost::shared_ptr<asf::core::Proxy>&,
      const asf::core::ServiceStateChange& state);
   // [End] ServiceAvailableIF implementation

protected:
   ServiceProvider();

private:
   //stores the current state reported to the listener
   bool m_serviceAvailable;
   IServiceAvailable& m_svcAvailableReceiver;
   ::std::list<AdapterType> m_adapterInUse;
   ::boost::shared_ptr< ::de::bosch::cm::ConfigManager::ConfigManagement::ConfigManagementProxy > m_configurationMgmtSvcProxy;
   ::boost::shared_ptr< ::bosch::cm::profile::ProfileProxy > m_profileSvcProxy;
   ::boost::shared_ptr< ::com::bosch::cm::dumm::DummService::DummServiceProxy > m_dummSvcProxy;
   ::boost::shared_ptr< ::tcu_usb_main_fi::Tcu_usb_main_fiProxy > m_usbTcuSvcProxy;

   ::boost::shared_ptr<ConfigMgmtSvcAdapter > m_configurationMgmtSvcAdapter;
   ::boost::shared_ptr<ProfileSvcAdapter > m_profileSvcAdapter;
   ::boost::shared_ptr<DummSvcAdapter > m_dummSvcAdapter;
   ::boost::shared_ptr<UsbTcuSvcAdapter > m_usbTcuSvcAdapter;


private:
   /**
    * @brief Required to be called once at startup time to indicate the required adapters.
    * Implementation will prepare the required proxies.
    * @param requiredAdapter a list container filled with values taken from AdapterType.
    */
   void getRequiredProxies(const ::std::list<AdapterType>& adapters);
   bool checkRequiredServicesAvailable() const;
   void checkProxyAvailable(
      const ::boost::shared_ptr< ::asf::core::Proxy>& proxy,
      int& nbrOfAvailableProxies) const;

   /** Bind the related proxy to its adapter. This method is called if a proxy is reported as available or
    * in case the service provider method allServicesAvailable is called. */
   void setProxyToAdapter(AdapterType type);

   /** Removes the related proxy from its adapter. Using the adapter can cause an exception now because proxy
    * is not longer available. This method is called if a proxy is reported as unavailable (disconnected). */
   void delProxyFromAdapter(AdapterType type);

   /** iterates over all required adapter and binds the adapter to its proxy if proxy is available.
    * All adapters are ready to be used when method returns.
    */
   void setAllProxiesToAdapters();

   const AdapterType getProxyType(const ::boost::shared_ptr< ::asf::core::Proxy>& proxy) const;

   ServiceProvider(const ServiceProvider& rsh);
   ServiceProvider& operator=( const ServiceProvider& other );

   DECLARE_CLASS_LOGGER();
};

} /* namespace profileMngr */

#endif  // _SERVICEPROVIDER_H_
