/*****************************************************************************
* FILE:         IServiceAvailable.h
* PROJECT:      A-IVI project
* SW-COMPONENT: profilemanager
*----------------------------------------------------------------------------
* DESCRIPTION:  core logic for Online Profile Manager
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim
*****************************************************************************/

#include <string>
#include <list>
#include <dlt/dlt.h>

#include "de/bosch/cm/ConfigManager/ConfigManagementProxy.h"
#include "bosch/cm/profileProxy.h"
#include "com/bosch/cm/dumm/DummServiceProxy.h"
#include "tcu_usb_main_fiProxy.h"

#include "ConfigMgmtSvcAdapter.h"
#include "ProfileSvcAdapter.h"
#include "DummSvcAdapter.h"
#include "UsbTcuSvcAdapter.h"

#include "ServiceProvider.h"
#include "ServiceProviderTypes.h"

DLT_IMPORT_CONTEXT(PROFILEDATA_COMPONENT);

namespace profileMngr
{

/** Define the port names used to create the proxies. Same names are used in the cmc files. */
const ::std::string ServiceProvider::CONFIGURATION_MGMT_SVC_PROXY_NAME = "ConfigManagementClientFiPort";
const ::std::string ServiceProvider::PROFILE_SVC_PROXY_NAME = "userProfilePort";
const ::std::string ServiceProvider::DUMM_SVC_PROXY_NAME = "fcdummservicePort";
const ::std::string ServiceProvider::USBTCU_SVC_PROXY_NAME = "tcuUsbMainFiPort";

ServiceProvider::ServiceProvider(
   IServiceAvailable& serviceAvailableReceiver,
   const std::list<AdapterType>& requiredAdapter)
   : m_serviceAvailable(false)
   , m_svcAvailableReceiver(serviceAvailableReceiver)
   , m_adapterInUse()
{
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("ServiceProvider constructor called"));
	getRequiredProxies( requiredAdapter );
}

ServiceProvider::~ServiceProvider()
{
	m_configurationMgmtSvcProxy.reset();
	m_profileSvcProxy.reset();
	m_dummSvcProxy.reset();
	m_usbTcuSvcProxy.reset();

	m_configurationMgmtSvcAdapter.reset();
	m_profileSvcAdapter.reset();
	m_dummSvcAdapter.reset();
	m_usbTcuSvcAdapter.reset();
}

void ServiceProvider::allServicesAvailable()
{
   if ( !m_serviceAvailable && checkRequiredServicesAvailable() )
   {
      setAllProxiesToAdapters();
      m_serviceAvailable = true;
      m_svcAvailableReceiver.onAvailable();
   }
}

void ServiceProvider::getRequiredProxies(const ::std::list<AdapterType>& adapters)
{
   for( ::std::list<AdapterType>::const_iterator adapterType = adapters.begin();
         adapterType != adapters.end();
         ++adapterType)
   {
	   AdapterType type = *adapterType;
	   m_adapterInUse.push_back(*adapterType);
	   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO,DLT_STRING("ServiceProvider: adapter"),DLT_INT8(type));

	   switch ( type )
	   {
	   case CONFIGURATIONMGMT_SVC:
		   if ( !m_configurationMgmtSvcProxy )
		   {
			   m_configurationMgmtSvcProxy =
					   ::de::bosch::cm::ConfigManager::ConfigManagement::ConfigManagementProxy::createProxy("ConfigManagementClientFiPort"/*CONFIGURATION_MGMT_SVC_PROXY_NAME*/, *this);
			   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("ServiceProvider::getRequiredProxies: CONFIGURATIONMGMT_SVC proxy created"));
			   //LOG_ASSERT(m_configurationMgmtSvcProxy);
		   }
		   break;
	   case PROFILE_SVC:
		   if( !m_profileSvcProxy )
		   {
			   m_profileSvcProxy =
					   ::bosch::cm::profile::ProfileProxy::createProxy("userProfilePort"/*PROFILE_SVC_PROXY_NAME*/, *this);
			   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("ServiceProvider::getRequiredProxies: PROFILE_SVC proxy created"));
			   //LOG_ASSERT(m_profileSvcProxy);
		   }
		   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("ServiceProvider::getRequiredProxies: PROFILE_SVC proxy OUTSIDE"));
		   break;
	   case DUMM_SVC:
		   if ( !m_dummSvcProxy )
		   {
			   m_dummSvcProxy =
					   ::com::bosch::cm::dumm::DummService::DummServiceProxy::createProxy("fcdummservicePort"/*DUMM_SVC_PROXY_NAME*/, *this);
			   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("ServiceProvider::getRequiredProxies: DUMM_SVC proxy created"));
			   //LOG_ASSERT(m_dummSvcProxy);
		   }
		   break;
	   case USBTCU_SVC:
		   if ( !m_usbTcuSvcProxy )
		   {
			   m_usbTcuSvcProxy =
					   ::tcu_usb_main_fi::Tcu_usb_main_fiProxy::createProxy("tcuUsbMainFiPort"/*USBTCU_SVC_PROXY_NAME*/, *this);
			   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("ServiceProvider::getRequiredProxies: USBTCU_SVC proxy created"));
			   //LOG_ASSERT(m_usbTcuSvcProxy);
		   }
		   break;
	   default:
		   LOG_ASSERT("Unsupported adapter required - please extend implementation.");
		   break;
	   }
   }
}

void ServiceProvider::clear()
{
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("clear is called to release all provider related resources"));
   // clear adapters
   if ( m_configurationMgmtSvcAdapter )
   {
      m_configurationMgmtSvcAdapter->clearProxy();
   }
   if( m_profileSvcAdapter )
   {
      m_profileSvcAdapter->clearProxy();
   }
   if ( m_dummSvcAdapter )
   {
      m_dummSvcAdapter->clearProxy();
   }
   if ( m_usbTcuSvcAdapter )
   {
      m_usbTcuSvcAdapter->clearProxy();
   }

   // clear proxies
   if ( m_configurationMgmtSvcProxy )
   {
      m_configurationMgmtSvcProxy.reset();
   }
   if ( m_profileSvcProxy )
   {
	   m_profileSvcProxy.reset();
   }
   if ( m_dummSvcProxy )
   {
      m_dummSvcProxy.reset();
   }
   if ( m_usbTcuSvcProxy )
   {
      m_usbTcuSvcProxy.reset();
   }
}

::boost::shared_ptr<ConfigMgmtSvcAdapter> ServiceProvider::getConfigMgmtSvcAdapter()
{
   if ( !m_configurationMgmtSvcAdapter )
   {
      if ( m_configurationMgmtSvcProxy )
      {
         //create and add a new adapter
         m_configurationMgmtSvcAdapter = ::boost::shared_ptr<ConfigMgmtSvcAdapter>(new ConfigMgmtSvcAdapter());

         if ( m_configurationMgmtSvcProxy->getServiceState() != ::asf::core::ServiceState__Disconnected )
         {
            m_configurationMgmtSvcAdapter->setProxy(m_configurationMgmtSvcProxy);
         }
      }
      else
      {
    	  DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Adapter requested that is not set as required before"));
          return( ::boost::shared_ptr<ConfigMgmtSvcAdapter>() );
      }
   }

   return( m_configurationMgmtSvcAdapter );
}

::boost::shared_ptr<ProfileSvcAdapter> ServiceProvider::getProfileSvcAdapter()
{
   if ( !m_profileSvcAdapter )
   {
      if ( m_profileSvcProxy )
      {
         //create and add a new adapter
         m_profileSvcAdapter = ::boost::shared_ptr<ProfileSvcAdapter>(new ProfileSvcAdapter());

         if ( m_profileSvcProxy->getServiceState() != ::asf::core::ServiceState__Disconnected )
         {
            m_profileSvcAdapter->setProxy(m_profileSvcProxy);
         }
      }
      else
      {
    	  DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Adapter requested that is not set as required before"));
    	  return( ::boost::shared_ptr<ProfileSvcAdapter>() );
      }
   }

   return( m_profileSvcAdapter );
}

::boost::shared_ptr<DummSvcAdapter> ServiceProvider::getDummSvcAdapter()
{
   if ( !m_dummSvcAdapter )
   {
      if ( m_dummSvcProxy )
      {
         //create and add a new adapter
         m_dummSvcAdapter = ::boost::shared_ptr<DummSvcAdapter>(new DummSvcAdapter());

         if ( m_dummSvcProxy->getServiceState() != ::asf::core::ServiceState__Disconnected )
         {
            m_dummSvcAdapter->setProxy(m_dummSvcProxy);
         }
      }
      else
      {
    	 DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Adapter requested that is not set as required before"));
         return( ::boost::shared_ptr<DummSvcAdapter>() );
      }
   }

   return( m_dummSvcAdapter );
}

::boost::shared_ptr<UsbTcuSvcAdapter> ServiceProvider::getUsbTcuSvcAdapter()
{
   if ( !m_usbTcuSvcAdapter )
   {
      if ( m_usbTcuSvcProxy )
      {
         //create and add a new adapter
         m_usbTcuSvcAdapter = ::boost::shared_ptr<UsbTcuSvcAdapter>(new UsbTcuSvcAdapter());

         if ( m_usbTcuSvcProxy->getServiceState() != ::asf::core::ServiceState__Disconnected )
         {
            m_usbTcuSvcAdapter->setProxy(m_usbTcuSvcProxy);
         }
      }
      else
      {
    	  DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Adapter requested that is not set as required before"));
          return( ::boost::shared_ptr<UsbTcuSvcAdapter>() );
      }
   }

   return( m_usbTcuSvcAdapter );
}

// ServiceAvailableIF implementation
void ServiceProvider::onAvailable(
		const ::boost::shared_ptr< ::asf::core::Proxy>& proxy,
		const ::asf::core::ServiceStateChange& state)
{
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("ServiceProvider::onAvailable"),DLT_INT(state.getCurrentState()));
	if ( state.getCurrentState() == ::asf::core::ServiceState__Available )
	{
		setProxyToAdapter(getProxyType(proxy));
		allServicesAvailable();
	}
	else
	{
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("ServiceProvider::onAvailable: State: of proxy is not expected"));
	}
}

void ServiceProvider::onUnavailable(
		const ::boost::shared_ptr< ::asf::core::Proxy>& proxy,
		const ::asf::core::ServiceStateChange& state)
{
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("ServiceProvider::onUnavailable"));
	if ( state.getCurrentState() == asf::core::ServiceState__Disconnected )
	{
		delProxyFromAdapter(getProxyType(proxy));
		//avoid calling onUnavailable if onAvailable is not called before
		if ( m_serviceAvailable )
		{
			m_svcAvailableReceiver.onUnavailable();
			m_serviceAvailable = false;
		}
	}
	else
	{
		m_svcAvailableReceiver.onPaused();
		//the overall concept for this scenario has to be discussed
	}
}

void ServiceProvider::setAllProxiesToAdapters()
{
   for( ::std::list<AdapterType>::const_iterator adapterType = m_adapterInUse.begin();
         adapterType != m_adapterInUse.end();
         ++adapterType)
   {
      setProxyToAdapter(*adapterType);
   }
}

//lint complains setProxyToAdapter/delProxyFromAdapter could be made constant, but in fact members were changed
//lint -e1762
void ServiceProvider::setProxyToAdapter(AdapterType type)
{
   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("ServiceProvider::setProxyToAdapter"),DLT_INT(type));
   switch (type)
   {
   case CONFIGURATIONMGMT_SVC:
      if ( m_configurationMgmtSvcAdapter && m_configurationMgmtSvcProxy )
      {
         m_configurationMgmtSvcAdapter->setProxy(m_configurationMgmtSvcProxy);
      }
      break;
   case PROFILE_SVC:
      if( m_profileSvcAdapter && m_profileSvcProxy )
      {
         m_profileSvcAdapter->setProxy(m_profileSvcProxy);
      }
      break;
   case DUMM_SVC:
      if ( m_dummSvcAdapter && m_dummSvcProxy )
      {
         m_dummSvcAdapter->setProxy(m_dummSvcProxy);
      }
      break;
   case USBTCU_SVC:
      if ( m_usbTcuSvcAdapter && m_usbTcuSvcProxy )
      {
         m_usbTcuSvcAdapter->setProxy(m_usbTcuSvcProxy);
      }
      break;
   default:
	   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Forgotten to add this adapter type:"),DLT_INT(type));
      break;
   }
}

void ServiceProvider::delProxyFromAdapter(AdapterType type)
{
   switch (type)
   {
   case CONFIGURATIONMGMT_SVC:
      if ( m_configurationMgmtSvcAdapter )
      {
         m_configurationMgmtSvcAdapter->clearProxy();
      }
      break;
   case PROFILE_SVC:
      if( m_profileSvcAdapter )
      {
    	  m_profileSvcAdapter->deRegisterForProperties();
    	  m_profileSvcAdapter->clearProxy();
      }
      break;
   case DUMM_SVC:
      if ( m_dummSvcAdapter )
      {
         m_dummSvcAdapter->clearProxy();
      }
      break;
   case USBTCU_SVC:
      if ( m_usbTcuSvcAdapter )
      {
         m_usbTcuSvcAdapter->clearProxy();
      }
      break;
   default:
	   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Forgotten to add this adapter type:"),DLT_INT(type));
      break;
   }
}
//lint +e1762

bool ServiceProvider::checkRequiredServicesAvailable() const
{
   int nbrOfRequiredProxies = m_adapterInUse.size();
   int nbrOfAvailableProxies = 0;

   //one call for each proxy. Null proxies are skipped (not counted): they weren't created as no required.
   checkProxyAvailable(m_configurationMgmtSvcProxy, nbrOfAvailableProxies);
   checkProxyAvailable(m_profileSvcProxy, nbrOfAvailableProxies);
   checkProxyAvailable(m_dummSvcProxy, nbrOfAvailableProxies);
   checkProxyAvailable(m_usbTcuSvcProxy, nbrOfAvailableProxies);
   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_DEBUG,DLT_INT(nbrOfAvailableProxies),DLT_STRING("of"),\
		   DLT_INT(nbrOfRequiredProxies),DLT_STRING("required proxies are available"));
   return( nbrOfRequiredProxies == nbrOfAvailableProxies );
}

void ServiceProvider::checkProxyAvailable(
   const ::boost::shared_ptr< ::asf::core::Proxy>& proxy,
   int& nbrOfAvailableProxies) const
{
   if ( proxy )
   {
      if ( proxy->isAvailable() )
      {
         nbrOfAvailableProxies++;
      }
   }
}

const AdapterType ServiceProvider::getProxyType(const ::boost::shared_ptr< ::asf::core::Proxy>& proxy) const
{
   if ( proxy == m_configurationMgmtSvcProxy )
   {
      return( CONFIGURATIONMGMT_SVC );
   }
   if ( proxy == m_profileSvcProxy )
   {
      return( PROFILE_SVC );
   }
   if ( proxy == m_dummSvcProxy )
   {
      return( DUMM_SVC );
   }
   else if ( proxy == m_usbTcuSvcProxy )
   {
      return( USBTCU_SVC );
   }
   return(UNKNOWN);
}


} /* namespace profileMngr */
