/******************************************************************************
 *	FILE:        : UnlinkProfileCommand.cpp
 *	PROJECT:     : A-IVI project
 *  SW-COMPONENT : ProfileManager
 *----------------------------------------------------------------------------
  *                This is a state class to realize the Various states of UnLink Profile
 * DESCRIPTION:    Command Execution. The following is the sequence of execution
 *                 1. Get the Status of the Profiles from ProfileBase Interface
 *                 2. Generate a new Correlation from FC_DUMM Interface
 *                 3. Send the LOCAL_UNLINK_REQUEST to FC_DUMM
 *----------------------------------------------------------------------------
 * COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim
 *****************************************************************************/

#include <stdlib.h>
#include <dlt/dlt.h>
#include "UnlinkProfileCommand.h"
#include "online/onlineData.h"
#include "online/jsonParser.h"
#include "online/jsonUtils.h"

DLT_IMPORT_CONTEXT(PROFILEDATA_COMPONENT);

using namespace com::bosch::cm::dumm::DummService;
namespace profileMngr
{

UnlinkProfileCommand::UnlinkProfileCommand():
   m_completionCallback(0),
   m_clientAct(0),
   m_userID(0),
   m_Data(0)
{
}

UnlinkProfileCommand::~UnlinkProfileCommand()
{
   //lint -e1506 reset should be virtual to be able to Mock so disable lint warning
   reset();
}

void UnlinkProfileCommand::init(uintptr_t act,
		uint8_t userID,
		boost::shared_ptr< DummSvcAdapter > dummSvcAdapter,
		boost::shared_ptr< ProfileSvcAdapter > profileSvcAdapter,
		onlineData& data,
		IUnlinkProfileCompletion& completionCallback)
{
   m_clientAct = act;
   m_userID = userID;
   m_Data = &data;
   m_completionCallback = &completionCallback;
   m_dummSvcAdapter = dummSvcAdapter;
   m_profileSvcAdapter = profileSvcAdapter;
}

void UnlinkProfileCommand::returnSuccess()
{
	m_Data->setaccountName(m_userID,"");
	m_Data->setkID(m_userID,"");
	m_Data->store();

	if(m_completionCallback)
	{
		m_completionCallback->unlinkProfileCompletionSuccess(m_clientAct,rsSuccess);
	}

	reset();
}

void UnlinkProfileCommand::returnError(resultState errorType)
{
   if(m_completionCallback)
   {
      m_completionCallback->unlinkProfileCompletionError(m_clientAct,errorType);
   }

   reset();
}

void UnlinkProfileCommand::reset()
{
   setBusy(false);
   m_completionCallback = 0;
   m_userID = 0;
   m_dummSvcAdapter.reset();
   m_profileSvcAdapter.reset();
   m_profileLocalUnlinkAckProperty.reset();
 }

void UnlinkProfileCommand::execute()
{
   if((!m_dummSvcAdapter) || (!m_profileSvcAdapter) || (!m_Data))
   {
      returnError();
      return;
   }

   // Check the Status of Connectivity_Activation and Remote_Vehicle_Settings
   if(!m_Data->isvNextCommAllowed())
   {
	   returnError(rsServiceDisabled);//return SERVICE_DISABLED
	   return;
   }

   setBusy(true);
   GetProfileTypesCommand& getProfileTypeCmdRef = GetProfileTypesCommand::getInst();
   if(!getProfileTypeCmdRef.isBusy())
   {
	   getProfileTypeCmdRef.init(m_profileSvcAdapter,*this);
	   getProfileTypeCmdRef.execute();
   }
   else
	   returnError();
}

// IGetProfileTypesCompletion - called from GetProfileTypesCommand
void UnlinkProfileCommand::getProfileTypesCompleted(bool bSuccess)
{
	//Step 1: Get new Correlation ID
	//If current active User is of type Linked/LinkedPin, only then proceed with the execution
	if(bSuccess && isUnlinkProfileAllowed(m_userID))
	{
		if(m_dummSvcAdapter->isValid())
		{
			m_dummSvcAdapter->sendGenCorrelationIdRequest(*this);
		}
		else
		{
			returnError();
		}
	}
	else
	{   // return ACCESS_DENIED
		returnError(rsDenied);
	}
}

void UnlinkProfileCommand::onGenCorrelationIdError(const ::boost::shared_ptr< DummServiceProxy >& proxy,
		                     const ::boost::shared_ptr< GenCorrelationIdError >& error)
{
	returnError();
}

void UnlinkProfileCommand::onGenCorrelationIdResponse(const ::boost::shared_ptr< DummServiceProxy >& proxy,
		                        const ::boost::shared_ptr< GenCorrelationIdResponse >& response)
{
	std::string correlationID = response->getCorrelationId();

	//Step 2: Send DataUpload Request to fc_dumm
	//TODO: Get the Json Data Header and Payload
	std::string uploadHeader,uploadData;
	std::string kID = m_Data->getkID(m_userID);
	formJsonUploadHeaderAndUploadData(correlationID,kID,uploadHeader,uploadData);
	if(m_dummSvcAdapter->isValid())
	{
		m_profileLocalUnlinkAckProperty.reset(new ProfileLocalUnlinkAckProperty(m_dummSvcAdapter, *this));
		m_profileLocalUnlinkAckProperty->doRegister();

		m_dummSvcAdapter->sendUploadDataRequest(*this,uploadHeader,uploadData);
	}
	else
	{
		returnError();
	}
}

void UnlinkProfileCommand::onUploadDataError(const ::boost::shared_ptr< DummServiceProxy >& proxy,
		               const ::boost::shared_ptr< UploadDataError >& error)
{
	returnError();
}
void UnlinkProfileCommand::onUploadDataResponse(const ::boost::shared_ptr< DummServiceProxy >& proxy,
		                  const ::boost::shared_ptr< UploadDataResponse >& response)
{
	//Check the Status in the Response Message and act accordingly
	const RequestInfo info = response->getRequestInfo();
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("LinkProfileCommand::onUploadDataResponse: ResponseStatus ->"),
			DLT_INT(info.getResponseStatus()),DLT_STRING("RequestState ->"),DLT_INT(info.getRequestState()));
	if((info.getResponseStatus() == ResponseStatus__OK) &&
			((info.getRequestState() == RequestState__QUEUED) || (info.getRequestState() == RequestState__IN_PROGRESS)))
	{
		//Wait for AckData from vNext.
		//TODO:Populate the Correlation ID in the Reference Database
		m_Data->addCorrelationID(m_userID,info.getCorrelationId());
	}
	else
	{
		returnError();
	}
}

void UnlinkProfileCommand::onProfileLocalUnlinkAckDataUpdate(const ::std::string& sHeader, const ::std::string& sData)
{
	// Parse the Message and extract the status the Status
	uint8_t status; uint8_t errorcode;
	jsonParser parser;
	if(parser.getUpLocalUnLinkStatus(sData,status,errorcode))
	{
		if(status == jsonUtils::OK)
			returnSuccess();
		else
			returnError(rsFailed);// return UNLINK_FAILED
	}
	else
		returnError();
}
void UnlinkProfileCommand::onProfileLocalUnlinkAckDataRegError(uint8_t error)
{
	//TODO: To check how the Data needs to be handled
}

void UnlinkProfileCommand::formJsonUploadHeaderAndUploadData(const std::string& correlationID, const std::string& kID, std::string& uploadHeader, std::string& uploadData)
{
	uint16 uploadFunctionID = UploadFunctionID__UserProfileLocalUnLink;
	uploadHeader = "";
    uploadData   = "";
    char value[10];
	std::string timestamp    = getTimeStamp();
	uploadHeader = "\n{\n\"UploadFunctionID\":";
	sprintf (value , "%d", uploadFunctionID );
	uploadHeader += value;
	uploadHeader += ",\n\"CorrelationId\":\"";
	uploadHeader += correlationID;
	uploadHeader += "\",\n\"UserId\":\"Renault\",\n\"Token\":\"";
	uploadHeader += kID;
	uploadHeader += "\"\n}";

	uploadData = "\n{\n\"TimeStamp\":\"";
	uploadData += timestamp;
	uploadData += "\",\n\"RemoteOrder/Sent/MapInstallRequest/MapChoice\":\"TBC\",\n\"RemoteOrder/Sent/MapInstallRequest/MapInfo\":\"TBC";
	uploadData += "\"\n}";
}

std::string UnlinkProfileCommand::getTimeStamp()
{
   time_t rawtime;
   struct tm* timeinfo;
   char buffer [50];
   time(&rawtime);
   timeinfo = localtime(&rawtime);
   strftime(buffer, 50, "%FT%T%z", timeinfo);
   std::string result(buffer);
   result.insert(22, ":");
   return result;
}

bool UnlinkProfileCommand::isUnlinkProfileAllowed(uint8_t user)
{
	bool ret = false;
	if(m_profileSvcAdapter->isValid())
	{
		if(user == m_profileSvcAdapter->getActiveProfile())
		{
			profileType type = GetProfileTypesCommand::getInst().getProfileTypes()[user];
			if((profileType__LINKED_PIN == type) || (profileType__LINKED == type))
			{
				ret = true;
			}
		}
	}
	return ret;
}



} // profileMngr

