/******************************************************************************
 *	FILE:        : LinkProfileCommand.cpp
 *	PROJECT:     : A-IVI project
 *  SW-COMPONENT : ProfileManager
 *----------------------------------------------------------------------------
 *                 This is a state class to realize the Various states of Link Profile
 * DESCRIPTION:    Command Execution. The following is the sequence of execution
 *                 1. Get the Status of the Profiles from ProfileBase Interface
 *                 2. Validate User Credentials with USB_TCU Interface
 *                 3. If Credential Validation is successful,get the ID Token from USB_TCU
 *                 4. Generate a new Correlation from FC_DUMM Interface
 *                 5. Send the LOCAL_LINK_REQUEST to FC_DUMM
 *----------------------------------------------------------------------------
 * COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim
 *****************************************************************************/

#include <stdlib.h>
#include <dlt/dlt.h>
#include "LinkProfileCommand.h"
#include "online/onlineData.h"
#include "online/jsonParser.h"
#include "online/jsonUtils.h"


DLT_IMPORT_CONTEXT(PROFILEDATA_COMPONENT);

using namespace com::bosch::cm::dumm::DummService;
using namespace bosch::cm::profile;
namespace profileMngr
{

LinkProfileCommand::LinkProfileCommand():
   m_completionCallback(0),
   m_clientAct(0),
   m_Data(0),
   m_UserID(0)
{
}

LinkProfileCommand::~LinkProfileCommand()
{
   //lint -e1506 reset should be virtual to be able to Mock so disable lint warning
   reset();
}

void LinkProfileCommand::init(uintptr_t act,
        const std::string& userAccountName,
        const std::string& pwd,
		boost::shared_ptr< DummSvcAdapter > dummSvcAdapter,
		boost::shared_ptr< UsbTcuSvcAdapter > tcuUsbSvcAdapter,
		boost::shared_ptr< ProfileSvcAdapter > profileSvcAdapter,
		onlineData& data,
		ILinkProfileCompletion& completionCallback)
{
   m_clientAct = act;
   m_completionCallback = &completionCallback;
   m_dummSvcAdapter = dummSvcAdapter;
   m_UsbTcuSvcAdapter = tcuUsbSvcAdapter;
   m_profileSvcAdapter = profileSvcAdapter;
   m_Data = &data;
   m_UserAccntName = userAccountName;
   m_accountPwd = pwd;
}

void LinkProfileCommand::returnSuccess()
{
	m_Data->setaccountName(m_UserID,m_UserAccntName);
	m_Data->setkID(m_UserID,m_KID);
	m_Data->store();
	// Remove the Corr ID from the reference Database
	m_Data->removecorrelationID(m_corrID);

	if(m_completionCallback)
	{
		m_completionCallback->linkProfileCompletionSuccess(m_clientAct,rsSuccess);
	}

	reset();
}

void LinkProfileCommand::returnError(resultState errorType)
{
    if(!m_corrID.empty() && (m_Data))
    	m_Data->removecorrelationID(m_corrID);

    //TODO: Check if mapping needs to be handled here or in CompImpl
	if(m_completionCallback)
	{
		m_completionCallback->linkProfileCompletionError(m_clientAct,errorType);
	}

    reset();
}

void LinkProfileCommand::reset()
{
   setBusy(false);
   m_completionCallback = 0;
   m_Data = 0;
   m_UserAccntName.clear();
   m_accountPwd.clear();
   m_KID.clear();
   m_corrID.clear();
   m_dummSvcAdapter.reset();
   m_UsbTcuSvcAdapter.reset();
   m_profileSvcAdapter.reset();
   m_profileLocalLinkAckProperty.reset();
}

void LinkProfileCommand::execute()
{
   if((!m_dummSvcAdapter) || (!m_UsbTcuSvcAdapter) || (!m_profileSvcAdapter) || (!m_Data))
   {
      returnError();
      return;
   }

   // Check the Status of Connectivity_Activation and Remote_Vehicle_Settings
   if(!m_Data->isvNextCommAllowed())
   {
	   returnError(rsServiceDisabled);//return SERVICE_DISABLED
	   return;
   }

   setBusy(true);
   //Step 1: Get the Profile Types
   GetProfileTypesCommand& getProfileTypeCmdRef = GetProfileTypesCommand::getInst();
   if(!getProfileTypeCmdRef.isBusy())
   {
      getProfileTypeCmdRef.init(m_profileSvcAdapter,*this);
      getProfileTypeCmdRef.execute();
   }
   else
	   returnError();
}

// IGetProfileTypesCompletion - called from GetProfileTypesCommand
void LinkProfileCommand::getProfileTypesCompleted(bool bSuccess)
{
	m_Data->setaccountName(m_UserID,m_UserAccntName);
	//TODO: extraction of KID is not possible yet with available USB_tcu interface
	m_Data->setkID(m_UserID,m_KID);
	m_Data->store();

	//Step 2: Validate Credentials
	//If current active User is of type Normal/ProtectedPin, only then proceed with the execution
	if(bSuccess && isLinkProfileAllowed())
	{
		if(m_UsbTcuSvcAdapter->isValid())
		{
			m_UsbTcuSvcAdapter->sendValidateUserCredentialsStart(*this,m_UserAccntName,m_accountPwd);
		}
		else
		{
			returnError();
		}
	}
	else
	{   // return ACCESS_DENIED
		returnError(rsDenied);
	}
}

void LinkProfileCommand::onValidateUserCredentialsError(const ::boost::shared_ptr< ::tcu_usb_main_fi::Tcu_usb_main_fiProxy >& proxy,
		                                    const ::boost::shared_ptr< ::tcu_usb_main_fi::ValidateUserCredentialsError >& error)
{
	returnError();
}
void LinkProfileCommand::onValidateUserCredentialsResult(const ::boost::shared_ptr< ::tcu_usb_main_fi::Tcu_usb_main_fiProxy >& proxy,
		                                     const ::boost::shared_ptr< ::tcu_usb_main_fi::ValidateUserCredentialsResult >& result)
{

	//Step 3:Decode the Access Token
	if(tcu_usb_main_fi_types::T_e8_UserAuthenticationResultType__USB_TCU_USER_AUTHENTICATION_RESULT_SUCCESS != result->getResult())
	{
		// return INVALID_CREDENTIALS
		returnError(rsInValidCredentials);
	}
	else
	{
		//Step 3:Decode the Access Token
		//const std::string& token = result->getToken();
		if(m_UsbTcuSvcAdapter->isValid())
		{
			m_UsbTcuSvcAdapter->sendDecodeIdTokenStart(*this, result->getToken());
		}
		else
		{
			returnError();
		}
	}

}


void LinkProfileCommand::onDecodeIdTokenError(const ::boost::shared_ptr< ::tcu_usb_main_fi::Tcu_usb_main_fiProxy >& proxy,
		const ::boost::shared_ptr< ::tcu_usb_main_fi::DecodeIdTokenError >& error)
{
	returnError();
}

void LinkProfileCommand::onDecodeIdTokenResult(const ::boost::shared_ptr< ::tcu_usb_main_fi::Tcu_usb_main_fiProxy >& proxy,
		const ::boost::shared_ptr< ::tcu_usb_main_fi::DecodeIdTokenResult >& result)
{
	const ::tcu_usb_main_fi_types::T_usb_tcu_UserAuthentication_DecodedToken_Info info = result->getResult();
	if(info.hasKID())
	{
		m_KID = info.getKID();
		bool bExists = false;
		//Check if the KID already exists
		std::vector<std::string> kIDs = m_Data->getkIDs();
		for(uint8_t ind = 0; ind < kIDs.size(); ind++)
			if(kIDs[ind] == m_KID)
				{bExists = true;break;}
		if(bExists)
			{returnError(rsUserAlreadyLinked);return;} // return USER_IS_ALREADY_LINKED

		//Step 4: proceeding with Requesting GenCorrelationID
		if(m_dummSvcAdapter->isValid())
		{
			m_dummSvcAdapter->sendGenCorrelationIdRequest(*this);
		}
		else
		{
			returnError();
		}
	}
	else
	{
		returnError();
	}
}



void LinkProfileCommand::onGenCorrelationIdError(const ::boost::shared_ptr< DummServiceProxy >& proxy,
		                     const ::boost::shared_ptr< GenCorrelationIdError >& error)
{
	returnError();
}

void LinkProfileCommand::onGenCorrelationIdResponse(const ::boost::shared_ptr< DummServiceProxy >& proxy,
		                        const ::boost::shared_ptr< GenCorrelationIdResponse >& response)
{
	m_corrID = response->getCorrelationId();

	//Step 5: Send DataUpload Request to fc_dumm
	// Get the Json Data Header and Payload
	std::string uploadHeader,uploadData;
	formJsonUploadHeaderAndUploadData(m_corrID,uploadHeader,uploadData);

	if(m_dummSvcAdapter->isValid())
	{
	   m_profileLocalLinkAckProperty.reset(new ProfileLocalLinkAckProperty(m_dummSvcAdapter,m_Data->getLocalLinkOrderTimeout(),*this));
       m_profileLocalLinkAckProperty->doRegister();

       m_dummSvcAdapter->sendUploadDataRequest(*this,uploadHeader,uploadData);
       m_profileLocalLinkAckProperty->startTimer();

       //Populate the Correlation ID in the Reference Database
       //m_Data->addCorrelationID(m_UserID,m_corrID);
 	}
	else
	{
		returnError();
	}
}

void LinkProfileCommand::onUploadDataError(const ::boost::shared_ptr< DummServiceProxy >& proxy,
		               const ::boost::shared_ptr< UploadDataError >& error)
{
	returnError();
}
void LinkProfileCommand::onUploadDataResponse(const ::boost::shared_ptr< DummServiceProxy >& proxy,
		                  const ::boost::shared_ptr< UploadDataResponse >& response)
{
	//Check the Status in the Response Message and act accordingly
	const RequestInfo info = response->getRequestInfo();
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("LinkProfileCommand::onUploadDataResponse: ResponseStatus ->"),
			DLT_INT(info.getResponseStatus()),DLT_STRING("RequestState ->"),DLT_INT(info.getRequestState()));
	if((m_corrID == info.getCorrelationId()) && (info.getResponseStatus() == ResponseStatus__OK) &&
			((info.getRequestState() == RequestState__QUEUED) || (info.getRequestState() == RequestState__IN_PROGRESS)))
	{
       //Wait for AckData from vNext.
       //Populate the Correlation ID in the Reference Database
	   m_Data->addCorrelationID(m_UserID,info.getCorrelationId());
	}
	else
	{
		returnError();
	}
}

void LinkProfileCommand::onProfileLocalLinkAckDataUpdate(const ::std::string& sHeader, const ::std::string& sData)
{
    // Parse the Message and extract the status the Status
	uint8_t status; uint8_t errorcode;
	std::string corrID;
	jsonParser parser;
	if((parser.getCorrelationID(sHeader,corrID)) && (corrID == m_corrID) && (parser.getUpLocalLinkStatus(sData,status,errorcode)))
	{
	   if(status == jsonUtils::OK)
     	   returnSuccess();
       else
     	   returnError(rsFailed);//TODO: return LINK_FAILED
 	}
	else
		returnError();
}

void LinkProfileCommand::onProfileLocalLinkAckDataRegError(uint8_t error)
{
	//TODO: To check how the Data needs to be handled
	returnError();
}

void LinkProfileCommand::onProfileLocalLinkAckDataTimeout()
{
	returnError(rsConnectionTimeout); //TODO:return CONNECTION_TIMEOUT
}

void LinkProfileCommand::formJsonUploadHeaderAndUploadData(const std::string& correlationID, std::string& uploadHeader, std::string& uploadData)
{
	uint16 uploadFunctionID = UploadFunctionID__UserProfileLocalLink;
	uploadHeader = "";
    uploadData   = "";
    char value[10];
	std::string timestamp    = getTimeStamp();
	uploadHeader = "\n{\n\"UploadFunctionID\":";
	sprintf (value , "%d", uploadFunctionID );
	uploadHeader += value;
	uploadHeader += ",\n\"CorrelationId\":\"";
	uploadHeader += correlationID;
	uploadHeader += "\",\n\"UserId\":\"Renault\",\n\"Token\":\"";
	uploadHeader += m_KID;
	uploadHeader += "\"\n}";

	uploadData = "\n{\n\"TimeStamp\":\"";
	uploadData += timestamp;
	uploadData += "\",\n\"RemoteOrder/Sent/MapInstallRequest/MapChoice\":\"TBC\",\n\"RemoteOrder/Sent/MapInstallRequest/MapInfo\":\"TBC";
	uploadData += "\"\n}";
}

std::string LinkProfileCommand::getTimeStamp()
{
   time_t rawtime;
   struct tm* timeinfo;
   char buffer [50];
   time(&rawtime);
   timeinfo = localtime(&rawtime);
   strftime(buffer, 50, "%FT%T%z", timeinfo);
   std::string result(buffer);
   result.insert(22, ":");
   return result;
}

bool LinkProfileCommand::isLinkProfileAllowed()
{
	bool ret = false;
	if(m_profileSvcAdapter->isValid())
	{
		m_UserID = m_profileSvcAdapter->getActiveProfile();
		profileType type = GetProfileTypesCommand::getInst().getProfileTypes()[m_UserID];
		uint32_t privacyMode = m_profileSvcAdapter->getPrivacyMode();
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("LinkProfileCommand::isLinkProfileAllowed: activeProfileType"),\
				DLT_INT(type),DLT_STRING("privacyMode:"),DLT_INT(privacyMode));
		if((!privacyMode) && ((profileType__NORMAL == type) || (profileType__PROTECTED == type)))
		{
			uint8_t noOfSecuredProfiles = 0;
			const std::vector< profileType >& profileTypes = GetProfileTypesCommand::getInst().getProfileTypes();
			for( ::std::vector<profileType>::const_iterator iter = profileTypes.begin();
					iter != profileTypes.end();
			         ++iter)
			   if((profileType__LINKED_PIN == (*iter)) || (profileType__LINKED == (*iter)))
				   ++noOfSecuredProfiles;

			if(noOfSecuredProfiles < m_Data->getMaxSecuredProfiles())
				ret = true;
		}

	}
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("LinkProfileCommand::isLinkProfileAllowed ?"),DLT_INT(ret));
	return ret;
}



} // profileMngr

