

/*****************************************************************************
* FILE:         profileApp.cpp
* PROJECT:      A-IVI project
* SW-COMPONENT: profilemanager
*----------------------------------------------------------------------------
* DESCRIPTION:  core logic for Profile Manager
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2016 Robert Bosch GmbH, Hildesheim
*****************************************************************************/

#include "core/profileApp.h"
#include "core/configInterface.h"


#define DP_S_IMPORT_INTERFACE_BASE
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_if.h"

#include "cca/CCAAppMain.h"

#include "asf/core/Application.h"
#include "app/core/ProfileMngr.h"
#include "asf/core/SignalHandler.h"

#include "core/jobs/changeUser.h"
#include "core/jobs/copyUser.h"
#include "core/jobs/createUser.h"
#include "core/jobs/deleteUser.h"
#include "core/jobs/loginUser.h"
#include "core/jobs/image.h"
#include "core/jobs/valetMode.h"

#include <semaphore.h>
#include <pthread.h>
#include <systemd/sd-daemon.h>

#include "core/profileTrace.h"
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PROFILEMANAGER_APP
#include "trcGenProj/Header/profileApp.cpp.trc.h"
#endif


#define OEM_TYPE_PIVI 22
const uint8_t useDPUser=255;


const char * LibFiles[]={"/opt/bosch/base/lib/libprofile-renault_so.so","/opt/bosch/base/lib/libprofile-pivi_so.so"};
bool g_bTestMode=false;// Profile manager test mode. ignores all client registrations

using namespace std;

uint32_t getTickCount()
{
    timespec tp;
    ::clock_gettime (CLOCK_MONOTONIC, &tp);

    return (tp.tv_sec * 1000 + tp.tv_nsec / 1000000);
}

namespace profileMngr {

ProfileApp *ProfileApp::m_pInstance=NULL;

pthread_t g_FailTimerThread;


using namespace std;

//*************************** ProfileApp() ***********************************

ProfileApp::ProfileApp(configIF & Cfg)
:m_dpSrvIf(* new dp_tclSrvIf())
,m_Config(Cfg)
,m_NumInterfaces(0)
,m_NewTimerID(1)
,m_pCurrentJob(0)
,m_VariantType(2)
,m_pLoginJob(0)
,m_bDPLocked(false)
,m_pOngoingProfileJob(0)
,m_ClientConfiguration(*this)
,m_bStartupLoginChecked(false)
{
	m_pInstance=this;
	sem_init(&m_LockSem,0,1);
	m_PinTimeStamp.assign(Cfg.maxProfiles(),0);
	m_pInterfaces=new commInterface*[num_interfaces];

	//get the maximum number of profiles
	 m_Data.Init(m_Config.maxProfiles());
	 updateCurrentUser(useDPUser);
	 if(!(m_Data.load() && m_Config.load())) //if any of the DP is invalid
	 {
		  m_Config.doDefSet(m_Data);//lastmode data invalid, Load the Default data from config
     }
	 uint32_t ongoingAction=m_Data.getOngoingProfileOperation();
	 if (ongoingAction!=0)
	 {
		 uint8_t dest=GET_ONGOING_DEST(ongoingAction);
		 m_Data.m_ProfileStatus[dest]=usBlocked;//prevent the user, to select the partially copied profile
		 if (GET_ONGOING_MODE(ongoingAction) == 0)
			 m_pOngoingProfileJob=new DeleteUserJob(0,dest,m_Clients.size(),*this);
		 else
		 {
			 setEarlyDPUser(dest);
			 updateCurrentUser(useDPUser);
			 m_Data.setOngoingProfileOperation(0,0,0);
		 }
		 ETG_TRACE_COMP(("unfinished copy operation to profile %i, status: %i",dest,GET_ONGOING_MODE(ongoingAction)));
	 }
	 updateUser(useDPUser);
 }
//*************************** ~ProfileApp ***************************************

ProfileApp::~ProfileApp()
{
if (m_pCurrentJob!=0)
	{delete m_pCurrentJob;m_pCurrentJob=0;}

   delete [] m_pInterfaces;
   m_pInterfaces = NULL;

   if (m_pOngoingProfileJob!=0)
   {delete m_pOngoingProfileJob;m_pOngoingProfileJob=0;}

   delete &m_dpSrvIf;
   ImageJob::detachLib();
}

bool ProfileApp::AddJob(Job* pJob)
{
	lock();
	if (m_pCurrentJob!=0)
	{unlock();delete pJob;ETG_TRACE_FATAL(("!!Current Job not finished!!"));return false;}

	m_pCurrentJob=pJob;
	do 	{} while (m_pCurrentJob->stateFunction()==true);

	if (m_pCurrentJob->isFinished()==false)
	{   // When a Login User Job is running, FailSafeTimer should not be started
		if((m_pCurrentJob->isTimerExpiredRequired()== true))
			startFailTimer(m_pCurrentJob);
	}
	else
	{
			delete m_pCurrentJob;m_pCurrentJob=0;
			m_pLoginJob=0;
	}
	unlock();
	return true;
}
void ProfileApp::JobDone()
{
	if (m_pCurrentJob!=0)
	{
		if (m_pCurrentJob->isFinished()==false)
			return;

		lock();
		delete m_pCurrentJob;
		m_pCurrentJob=0;
		m_pLoginJob=0;
		unlock();
	}
	if (m_pOngoingProfileJob!=0)
		if (AddJob(m_pOngoingProfileJob))
			m_pOngoingProfileJob=0;
}

std::string err("fatal");
const std::string & ProfileApp::getClientName(uint8_t Index)
{
	if (Index < m_Clients.size())
		return m_Clients[Index]->getName();
	return err;
}

void ProfileApp::sendPinRequest(unsigned char  Request)
{
	m_Data.m_pinRequest=Request;
	for (int p=0;p < m_NumInterfaces;++p)
			m_pInterfaces[p]->updatePinRequested();
}

void ProfileApp::sendDataUpdate(dataChangeStatus status)
{
	for (int p=0;p < m_NumInterfaces;++p)
		m_pInterfaces[p]->sendDataUpdate(status);
}
void ProfileApp::sendProfileID(uint8_t id)
{
	for (int p=0;p < m_NumInterfaces;++p)
		m_pInterfaces[p]->updateProfileID();
}
void ProfileApp::sendProfileStatus(profileState state)
{
	m_Data.m_AppState=state;
	for (int p=0;p < m_NumInterfaces;++p)
		m_pInterfaces[p]->updateProfileStatus();
}
void ProfileApp::sendCopyProfile(uint8_t src,uint8_t dest)
{
	for (int p=0;p < m_NumInterfaces;++p)
		m_pInterfaces[p]->sendCopyProfile(src,dest);
}
void ProfileApp::sendCreateProfile(uint8_t id)
{
	for (int p=0;p < m_NumInterfaces;++p)
		m_pInterfaces[p]->sendCreateProfile(id);
}

void ProfileApp::sendDeleteProfile(uint8_t id)
{
	for (int p=0;p < m_NumInterfaces;++p)
		m_pInterfaces[p]->sendDeleteProfile(id);
}

void ProfileApp::sendActiveProfileType()
{
	for (int p=0;p < m_NumInterfaces;++p)
		m_pInterfaces[p]->updateActiveProfileType();
}

//*******************************************Datapool interface *************************************************
bool ProfileApp::updateCurrentUser(uint8_t newUser)
{
    if(newUser==useDPUser)
    {
	if (m_dpSrvIf.s32GetEndUser(m_Data.m_CurrentProfile) != DP_S32_NO_ERR)
		{ETG_TRACE_FATAL(("Error reading current User!"));return false;}
    }
    else
        m_Data.m_CurrentProfile=newUser;
    ETG_TRACE_COMP(("updateCurrentUser: %i",m_Data.m_CurrentProfile));
	return true;
}
void ProfileApp::updateUser(uint8_t newUser)
{
	if (newUser!=useDPUser)
	{
		if(setEarlyDPUser(newUser)==false)
			return;
		m_Data.m_CurrentProfile=newUser;
	}

	 if (m_Data.getStatus() >= 0x80)// protected by pin
	 {
		 if (m_bStartupLoginChecked==false)
		 {
			 m_bStartupLoginChecked=true;
			 timespec time;
			 clock_gettime(CLOCK_REALTIME,&time);
			 traceL(trLoginTimestamp,4,time.tv_sec);
			 uint64_t storedTime=m_Data.getLogoutTime();
			 if (storedTime ==0 || time.tv_sec-storedTime <= m_Data.getAuthenticationValidTime())
				 return;
		 }
		 if(newUser == useDPUser)
		 {   /* If a Dummy value of 255 is sent, then read the CurrentUser from DP */
			 newUser = getCurrentUser();
		 }
		 ChangeUserJob *pChange=new ChangeUserJob(0,newUser,newUser,0,*this);
		 m_pLoginJob=new loginUser(pChange,*this);
		 AddJob(m_pLoginJob);
		 ETG_TRACE_COMP(("startup with PIN protected profile"));

	 }
}

bool ProfileApp::lockDP()
{
	if (m_bDPLocked==false)
	{
		if (m_dpSrvIf.s32Lock(DP_U32_LOCK_MODE_END_USER) == DP_S32_NO_ERR)
		{m_bDPLocked=true;return true;}
		else
			{ETG_TRACE_FATAL(("Datapool Lock Failed")); return false;}
	}
	return true;//If DP already locked, then simply return
}

bool ProfileApp::clearDPUser(uint8_t User)
{
	if (m_bDPLocked==true)
	{
		if (m_dpSrvIf.s32SetEndUserDefault(User) != DP_S32_NO_ERR)
		{ETG_TRACE_FATAL(("Datapool s32SetEndUserDefault Failed"));return false;}
		return true;
	}
	return false;
}

bool ProfileApp::copyDPUser(uint8_t srcID,uint8_t destID)
{
	if (m_bDPLocked==true)
	{
		if (m_dpSrvIf.s32CopyEndUser(srcID, destID)!=DP_S32_NO_ERR)
		{ETG_TRACE_FATAL(("Datapool s32CopyEndUser Failed"));return false;}
		return true;
	}
	return false;
}

bool ProfileApp::unlockDP()
{
	if (m_bDPLocked)
	{
		m_bDPLocked=false;
		if (m_dpSrvIf.s32Unlock()!= DP_S32_NO_ERR)
			return false;
	}
	return true;
}


bool ProfileApp::setDPUser(uint8_t destID)
{
	bool ret = true;
	if (m_bDPLocked==true)
	{
		if (m_dpSrvIf.s32SetEndUser(destID)!=DP_S32_NO_ERR)
		{ETG_TRACE_FATAL(("Datapool s32SetEndUser Failed"));ret = false;}
	}
	else
		ret=false;
	return ret;
}

void ProfileApp::loadUser(uint8_t newUser)
{
    updateCurrentUser(newUser);

    for (int p=0;p < m_NumInterfaces;++p)
    {
          m_pInterfaces[p]->updateProfileID();
		   m_pInterfaces[p]->updateProfileName();
		   m_pInterfaces[p]->updateProfileImage();
		   m_pInterfaces[p]->updatePrivacyMode();
		   m_pInterfaces[p]->updateActiveProfileType();
	}
}

bool ProfileApp::setEarlyDPUser(uint8_t newUser)
{
	bool bRet=lockDP();

	if(bRet==true)
		bRet=setDPUser(newUser);
	unlockDP();
	return bRet;
}



void ProfileApp::AddInterface(commInterface * pPs)
{
	if (pPs!=NULL && m_NumInterfaces < num_interfaces)
	{ m_pInterfaces[m_NumInterfaces++]=pPs;}

	// At start up, when all the interfaces are added, initialize the attributes
	if(m_NumInterfaces == num_interfaces)
	{
		for (int p=0;p < m_NumInterfaces;++p)
		{
			m_pInterfaces[p]->updateProfileID();
			m_pInterfaces[p]->updateProfileImage();
			m_pInterfaces[p]->updateProfileName();
			m_pInterfaces[p]->updateActiveProfileType();
			if(m_Data.getValetModeStatus()==true)
			{
				m_Data.m_AppState=psValetMode;
			}
			m_pInterfaces[p]->updateProfileStatus();
			m_pInterfaces[p]->sendDataUpdate(dsIdle);
			m_pInterfaces[p]->sendCopyProfile(0xFF,0xFF);
			m_pInterfaces[p]->sendCreateProfile(0xFF);
			m_pInterfaces[p]->sendDeleteProfile(0xFF);
			m_pInterfaces[p]->updatePrivacyMode();

		}
	}
}

//*************************** Synchron Commands (not queued) ***********************************
actionState ProfileApp::setProfileImage( unsigned char ImgID,unsigned char userID)
{
	ETG_TRACE_USR4(("-->setProfileImage userId:%d ImageId:%u",userID ,ImgID));

	if(m_Config.isSetProfileImageAllowed(m_Data,userID))
	{
		m_Data.setImage(userID,ImgID);
		m_Data.store();

		if(m_Data.currentProfile()==userID)
		{
			//send updates to interfaces
			for (int p=0;p < m_NumInterfaces;++p)
				m_pInterfaces[p]->updateProfileImage();
		}
		return asSuccess;
	}
	else
	{
		ETG_TRACE_COMP(("setProfileImage not allowed"));
		return asDenied;
	}

}

actionState ProfileApp::setProfileName(const std::string& profileName,unsigned char userID)
{
	ETG_TRACE_USR4(("-->setProfileName:userId:%d profileName:%s ",userID ,profileName.c_str()));

	std::string name=profileName;
	trimStr(name);

	if(m_Config.isSetProfileNameAllowed(m_Data,userID,name))
	{
		m_Data.setName(userID,name);
		m_Data.store();

		if(m_Data.currentProfile()==userID)
		{
			//send updates to interfaces
			for (int p=0;p < m_NumInterfaces;++p)
				m_pInterfaces[p]->updateProfileName();
		}
		return asSuccess;
	}
	else
	{
		ETG_TRACE_COMP(("setProfileName not allowed"));
		return asDenied;
	}
}



//*************************** Asynchron Commands (using m_pCurrentJob) *********************************
actionState ProfileApp::doCopyProfile(commInterface & requester,uintptr_t act,unsigned char srcID,unsigned char destID)
{
	ETG_TRACE_USR4(("-->doCopyProfile srcId:%d destID:%d",srcID ,destID));
	if (m_ClientConfiguration.systemIsUp()==false && g_bTestMode==false)
		return asInitializing;
	CopyUserJob* pAdd=new CopyUserJob(act,srcID,destID,m_Clients.size(),*this);
	pAdd->setRequester(&requester);

	if (AddJob(pAdd)==false)
		return asPending;

	return asSuccess;
}

//*************************** doDeleteProfile  *********************************


actionState ProfileApp::doDeleteProfile(commInterface & requester,uintptr_t act,unsigned char ID)
{
	ETG_TRACE_USR4(("-->doDeleteProfile UserID:%d",ID));

	if (m_ClientConfiguration.systemIsUp()==false && g_bTestMode==false)
		return asInitializing;
	DeleteUserJob *pAdd=new DeleteUserJob(act,ID,m_Clients.size(),*this);
	pAdd->setRequester(&requester);

	if (AddJob(pAdd)==false)
		return asPending;

	return asSuccess;
}
//*************************** setValetMode  *********************************

actionState ProfileApp::setValetMode(commInterface & requester,uintptr_t act,bool enable)
{
	ETG_TRACE_USR4(("-->set Valet Mode UserID:%d",enable));

	ValetModeJob *pAdd=new ValetModeJob(act,enable,m_Clients.size(),*this);
	pAdd->setRequester(&requester);

	if (AddJob(pAdd)==false)
		return asPending;
	return asSuccess;

}

//*************************** doSwitchToProfile  *********************************




actionState ProfileApp::doSwitchToProfile(commInterface * requester,uintptr_t act,unsigned char ID)
{
	ETG_TRACE_COMP(("-->doSwitchToProfile:UserId: %d, %i clients",ID,m_Clients.size() ));
	if (ID >= m_Config.maxProfiles())
	{ETG_TRACE_FATAL(("doSwitchToProfile:Invalid Profile ID" ));return asFailed;}


	if (m_Data.getProfileStatus()[ID]==usBlocked)
		{ETG_TRACE_FATAL(("doSwitchToProfile:Profile Locked" ));return asLocked;}


	if (m_Data.getValetModeStatus())
			if (configIF::getInst().isValetModeProfileChangeAllowed(m_Data,ID)==false)
				{ETG_TRACE_FATAL(("doSwitchToProfile:Valet Mode Profile Denied " ));return asDenied;}

	if (m_pLoginJob)
	{
		if (m_pLoginJob->isFinished()==false)
			{ETG_TRACE_COMP(("doSwitchToProfile:Login replaced" ));m_pLoginJob->finish();}
		JobDone();
	}


	ChangeUserJob *pAdd=new ChangeUserJob(act,ID,m_Data.m_CurrentProfile,m_Clients.size(),*this);
	pAdd->setRequester(requester);

	ETG_TRACE_COMP(("	doSwitchToProfile:type: %i", ETG_ENUM(TR_PROFILE_STATUS,m_Data.getProfileStatus()[ID] )));
	if (m_Data.getProfileStatus()[ID] >= 0x40 )// Profile secured by Pin / Remember me
	{
		m_pLoginJob=new (std::nothrow) loginUser(pAdd,*this);
		if (m_pLoginJob==0)
			return asFailed;
		if (AddJob(m_pLoginJob)==false)
		return asPending;
	}
	else
	{
		if (AddJob(pAdd)==false)
			return asPending;
	}

	if (!m_ClientConfiguration.systemIsUp())
		{m_Data.setStartupUserChange(true);	ETG_TRACE_USR4(("-->setStartupUserChange flag :%d ",m_Data.getStartupUserChange()));}


	return asSuccess;
}

actionState ProfileApp::doCreateProfile(commInterface & requester,uintptr_t act,const std::string& profileName, unsigned char imgID)
{
	ETG_TRACE_USR4(("-->doCreateProfile profileName: imgID:%d, %s",imgID,profileName.c_str()));

	if (m_ClientConfiguration.systemIsUp()==false && g_bTestMode==false)
		return asInitializing;
	std::string name=profileName;
	trimStr(name);
	CreateUserJob *pAdd=new CreateUserJob(act,name,imgID,m_Clients.size(),*this);
	pAdd->setRequester(&requester);

	if (AddJob(pAdd)==false)
		return asPending;

	return asSuccess;
}


//*************************** ImgImport functions ***************************************
actionState ProfileApp::doLoadImage(commInterface &requester,uintptr_t act,const std::string & FileName,unsigned short width,unsigned short height,unsigned short slot,bool bAssign )
{
	char * err=NULL;

	ImageJob *pImg=new ImageJob(act,FileName,width,height,slot,bAssign,*this);
	pImg->setRequester(&requester);

	if (AddJob(pImg)==false)
		return asPending;

	return asSuccess;
}


//*************************** deleteUserImage  functions ***************************************
actionState ProfileApp::doDeleteUserImage(unsigned short slotId)
{
	 ETG_TRACE_USR4(("-->doDeleteUserImage "));

     if (!ImageJob::deleteUserImage(slotId))
         return asFailed;

     return asSuccess;
}

bool ProfileApp::doDefSet(enDefSetMode Mode)
{
	//TODO:This check on the Flag will be enabled only after the LCM Interfaces are tested
	//if(m_bAreAllExpectedClientsRegistered)
	{
		switch (Mode)
		{
		case dsExecute:			//Reset the Values with  Default data using config interface
			m_Config.doDefSet(m_Data);
		    m_Data.doDefSet();
		    loadUser(useDPUser);
		    break;
		case dsFinalize: //store data to datapool. not possibel in Execute Phase
			m_Data.store();
			m_Config.store();
			sendDataUpdate(dsProfileReset);  //send update

			break;
		}
			return true;
	}
	return false;
}

//*************************** Timer Handling Routines  ***************************************
void ProfileApp::startFailTimer(Job * pJob)
{
	uint32_t TimerID=++m_NewTimerID;
	if (TimerID==0)
		TimerID=++m_NewTimerID;

	if(pJob !=NULL)
		 pJob->setTimer(TimerID);

	pthread_create(&g_FailTimerThread,0,timerproc,&m_NewTimerID);
}
void * ProfileApp::timerproc(void * pParam)
{
	uint32_t TimerID =  *((uint32_t*)pParam);
	ProfileApp & App=ProfileApp::getInst();
	sleep(configIF::getInst().FailureTimeout/1000);
	usleep(configIF::getInst().FailureTimeout*1000 % 1000000);

	App.lock();

	if (App.m_pCurrentJob!=NULL && App.m_pCurrentJob->getTimer()==TimerID && App.m_pCurrentJob->isFinished()==false )
	{

			ETG_TRACE_FATAL(("Job cancelled" ));
			App.m_pCurrentJob->finish();
			for (int k=0;k < App.m_pCurrentJob->numClients();++k)
			{
				ETG_TRACE_ERR(("Application status: %i (%s)",ETG_ENUM(TR_EN_CLSTATUSUPD,App.m_pCurrentJob->details()[k]),App.m_Clients[k]->getName().c_str() ));
			}

			delete App.m_pCurrentJob;
			App.m_pCurrentJob=0;
			App.m_pLoginJob=0;

	}
	App.unlock();

	return 0;
}

//**************************************client Handling Routines******************************************************************
uint32_t ProfileApp::onNewClient(std::string & Name)
{
	if (g_bTestMode)
	{ETG_TRACE_FATAL(("ProfileApp::onNewClient(%s) Testing mode, Ignoring registration",Name.c_str()));return 0;}
	lock();
	uint32_t retID= 0;
    if(m_ClientConfiguration.isClientConfigured(Name))
    {
		retID = m_Clients.size()+1;
		Client* pCl=new Client(retID,Name);
		ETG_TRACE_COMP(("-->%iProfileApp::onNewClient(%s)<--",retID,Name.c_str()));
		m_Clients.push_back(pCl);
		m_ClientConfiguration.onClientStarted(Name);
    }
    else
    {
       ETG_TRACE_FATAL(("ProfileApp::onNewClient(%s), Client Not Configured in DataBase yet",Name.c_str()));
       m_IgnoredClients.push_back(std::string(Name));
    }
    unlock();

    if(m_ClientConfiguration.systemIsUp())
    {
     	configIF::getInst().allClientsRegistered();
     	if (m_pLoginJob!=0)//send psLocked state, if a login is ongoing
     		sendProfileStatus(psLocked);
     	else if (m_Data.getValetModeStatus())
     		sendProfileStatus(psValetMode);
     	else
     		sendProfileStatus(psUnlocked);
    }

    if(m_ClientConfiguration.systemIsUp() && m_pOngoingProfileJob!=0)
    	if (AddJob(m_pOngoingProfileJob))
    		m_pOngoingProfileJob=0;

	return retID;
}

void ProfileApp::updateConfiguredProcessList(std::vector<std::string>& oProcesList)
{
	lock();
	m_ClientConfiguration.filterBySPM(oProcesList);
	unlock();
}
void ProfileApp::handleSysPwrModeTrigger()
{
	uint8_t startUser = getCurrentUser();
	configIF::getInst().InitOnPowerMode(startUser);
	if (startUser!= getCurrentUser())
	{
		doSwitchToProfile(0,0,startUser);
	}
}

void ProfileApp::onClientStatus(uint32_t ClientID,ClientStatusUpdate status)
{
	--ClientID;//client id starts at one, but need to map to array 0..num__clients
	if (ClientID >= m_Clients.size())
	{
	   if (!g_bTestMode) ETG_TRACE_FATAL(("Application used invalid ApplicationID %i",ClientID));
	   return;
	}
		else
		ETG_TRACE_COMP(("-->ProfileApp::onClientStatus: ClientID %d, Status %d",ClientID,ETG_ENUM(TR_EN_CLSTATUSUPD,status)));

	lock();
	if (m_pCurrentJob == 0)
	{ unlock();return;}

	m_pCurrentJob->updateStatus(status,ClientID);
	if (m_pCurrentJob->clientsOpen()==0)
		do  {} while(m_pCurrentJob->stateFunction()==true) ;
	unlock();
	JobDone();
}

void ProfileApp::setPrivacyMode(uint32_t Mode)
{
	traceN(trcSetPrivacyMode,4,Mode);
	m_Data.setPrivacyMode(Mode);
	for (int p=0;p < m_NumInterfaces;++p)
	{
		m_pInterfaces[p]->updatePrivacyMode();
	}
}
//***************************************** Protection handling-*****************************************
actionState ProfileApp::setProfilePin(uint8_t Index,std::string Pin)
{
	if (Index!= m_Data.currentProfile() || (m_pCurrentJob!=0 && m_pCurrentJob->isFinished()==false))
		{ETG_TRACE_FATAL(("setProfilePin: User not Active"));return asDenied;}

	loginUser inst(0,*this);
	actionState  res=inst.setProfilePin(Index,Pin);
	return res;
}

void ProfileApp::cancelLogin(uint8_t Index)
{

		ETG_TRACE_COMP(("Trigger Cancel "));
	if (m_pLoginJob!=0 && m_pLoginJob->isFinished()==false)
	{
		m_pLoginJob->Cancel();
		JobDone();
	}
	if ((m_Data.currentProfile()== Index ) && (m_Data.getProfileStatus()[Index]>= 0x80))
	{
		if (m_pCurrentJob==NULL)
		{

			uint8_t defaultUser;
			//get the guest user Id
			m_Config.getDefaultProfile(defaultUser);
			ChangeUserJob *pAdd=new ChangeUserJob(0,defaultUser/*newUser*/,m_Data.currentProfile() ,m_Clients.size(),*this);

			AddJob(pAdd);

		}
	}
}
actionState ProfileApp::checkPin(uint8_t Index,std::string Pin)
{

	if (Pin.empty())
	{
		cancelLogin(Index);
		return asCancelled;
	}
	unsigned char hash[CHECKSUM_DIGEST_LENGTH];
	loginUser::getSha(Pin,hash);
	const unsigned char * pPHash=m_Data.getPINHash()[Index];
	//Get the pinCount

	uint32_t pinCnt = m_Data.getPinCounter()[Index];
	uint32_t iCurrentTime=getTickCount(); //get currentTime

	if (pinCnt > m_Data.getMaxPINAttemptsBeforeDelay())
	{

        uint32_t elapsed_time= 0;
        ETG_TRACE_COMP(("checkPin: Failed Time stamp %d",m_Data.getMaxAttempsBeforeDelayTimeStamp()[Index]));
        if (m_Data.getMaxAttempsBeforeDelayTimeStamp()[Index]!=0)
		    elapsed_time=iCurrentTime-m_Data.getMaxAttempsBeforeDelayTimeStamp()[Index];

	     if(elapsed_time!=0 && elapsed_time < m_Data.getPINAuthenticationDelay())
		 {
	    	 return asDeniedWaiting;

	      }
	 }

	if (pPHash==0)
	{ETG_TRACE_FATAL(("checkPin: no hash stored"));	}//should not happen, check already done in loginbUser::staeFunction().
	else if (memcmp(pPHash,hash,CHECKSUM_DIGEST_LENGTH)==0 )
	{
		if (m_pLoginJob!=0 && m_pLoginJob->isFinished()==false)
		{
			ETG_TRACE_COMP(("checkPin: start user Change"));
			m_pLoginJob->loginDone();
			JobDone();
		}
		else
			ETG_TRACE_COMP(("checkPin: correct Pin"));
		m_PinTimeStamp[Index]=getTickCount();

		//reset the time stamp,pinCounter
		m_Data.setMaxAttempsBeforeDelayTimeStamp(0,Index);
		m_Data.setPinCounter(Index,0);
		m_Data.store();

		return asSuccess;
	}
	else
	{

		ETG_TRACE_COMP(("checkPin: wrong Pin [%i]",pinCnt));

		m_Data.increPinCounter(Index);
		m_Data.store();

		//get the updated pin counter
		pinCnt=m_Data.getPinCounter()[Index];

		if (pinCnt < m_Data.getMaxPINAttemptsTotal())
		{

			//send the pinRequest Updated
			unsigned char iPinRequest=m_Data.getMaxPINAttemptsTotal() - m_Data.getPinCounter()[Index];
			sendPinRequest(iPinRequest);

			if (pinCnt > m_Data.getMaxPINAttemptsBeforeDelay())
			{
				m_Data.setMaxAttempsBeforeDelayTimeStamp(getTickCount(),Index);  //get the Current Time in microsecond
				return asDeniedDelayed;
			}
		}
		else
		{
			m_Data.setStatus(Index,usBlocked);
			m_Data.store();//NCG3D-85144
			//send pin Request
			sendPinRequest(0);
			//reset the counter
			m_Data.setMaxAttempsBeforeDelayTimeStamp(0,Index);
			cancelLogin(Index);

			return asLocked;
		}
	}
	return asDenied;
}
actionState ProfileApp::setRememberMe(uint8_t Index)
{
	if (Index!= m_Data.currentProfile() ||(m_pCurrentJob!=0 && m_pCurrentJob->isFinished()==false))
	{ETG_TRACE_FATAL(("setRememberMe: User not Active"));return asDenied;}

	loginUser li(0,*this);
	return li.setRememberMe(Index);
}
actionState ProfileApp::unprotectProfile(uint8_t Index)
{
	if (Index!= m_Data.currentProfile() || (m_pCurrentJob!=0 && m_pCurrentJob->isFinished()==false))
	{ETG_TRACE_FATAL(("unprotectProfile: User not Active"));return asDenied;}

	loginUser li(0,*this);
	return li.unprotectProfile(Index);
}
bool ProfileApp::isUserUnlocked(uint8_t Index)
{return (m_PinTimeStamp[Index]!=0) && (getTickCount()-m_PinTimeStamp[Index] < m_Data.getPinVerificationValidity());}

actionState ProfileApp::setlinkProfileStatus(bool isLinked)
{
	actionState actResult = asDenied;
	enUserStatus currentState = m_Data.getStatus();
	enUserStatus newState = currentState ;
	if(isLinked)
	{
		switch(currentState)
		{
		case usNormal:
			//TODO: To trigger HMI to choose an Authentication Method. Now default Remember is chosen
			newState = usLinkedRemember;
			break;
		case usProtectedRemember:
			newState = usLinkedRemember;
			break;
		case usProtectedPin:
			newState = usLinkedPin;
			break;
		default:
			break;
		}
	}
	else
	{
		switch(currentState)
		{
		case usLinkedRemember:
			newState = usProtectedRemember;
			break;
		case usLinkedPin:
			newState = usProtectedPin;
			break;
		default:
			break;
		}
	}

	if(currentState != newState)
	{
		m_Data.setStatus(m_Data.currentProfile(),newState);
		m_Data.store();
		actResult = asSuccess;
		for (int p=0;p < m_NumInterfaces;++p)
			 m_pInterfaces[p]->updateActiveProfileType();
	}
	return actResult;
}

//************************************************Variant Handling ****************************************************************
void ProfileApp::updateVariantType()
{
    /* Read the KDS Value and Get the Variant Configuration;  4 Bit [0x00] not configured [0x01] Navi [0x02] DA */
	if (DP_S32_NO_ERR != DP_s32GetConfigItem("AIVIVariantCoding", "VariantSelection", &m_VariantType, 1))
	{   ETG_TRACE_FATAL(("Reading of AIVIVariantCoding:VariantSelection not successful"));}
}


bool ProfileApp::onTraceCmd(uint16_t Cmd,const unsigned char* pData)
{
	switch (Cmd)
	{
		case TR_PROFILEMANAGER_TST_ADD_CLIENT:
		{
			ETG_TRACE_FATAL(("case of TR_PROFILEMANAGER_CMD_ADD_START_CLIENT %s",pData+3));
			std::string name(reinterpret_cast<const char*>(pData+3));
			m_Data.addStartClient(name);
			m_Data.storeClients();
		}
			return true;
		case TR_PROFILEMANAGER_TST_REMOVE_CLIENT:
			return m_ClientConfiguration.onTraceCmd(Cmd,pData+3);
		case TR_PROFILEMANAGER_CMD_DO_DEFSET:
				ETG_TRACE_FATAL(("case of TR_PROFILEMANAGER_CMD_DO_DEFSET"));
				doDefSet(ProfileApp::dsExecute);
		return true;
		case TR_PROFILEMANAGER_CMD_GET_CLIENT_LIST:
				for(int k=0; k < m_Clients.size(); k++)
					ETG_TRACE_FATAL(("-->Registered Client ID: %d,Name: %s",m_Clients[k]->getID(),m_Clients[k]->getName().c_str()));

				for(int k=0; k < m_IgnoredClients.size(); k++)
					ETG_TRACE_FATAL(("-->Ignored Client: Name: %s",m_IgnoredClients[k].c_str()));
			return m_ClientConfiguration.onTraceCmd(Cmd,pData+3);

	}
	return false;
}
void ProfileApp::storeLogoffTime()
{
	timespec time;
	clock_gettime(CLOCK_REALTIME,&time);
	traceL(trLogoutTimeStamp,4,time.tv_sec);
	m_Data.storeLogoutTime(time.tv_sec);
}

// ************************** trimStr ********************************************************
void ProfileApp::trimStr(std::string& s)
{
   //Configured to Remove Leading and Trailing Spaces and Tabs
   size_t pos = s.find_first_not_of(" \t");
   s.erase(0, pos);

   pos = s.find_last_not_of(" \t");
   if (string::npos != pos)
      s.erase(pos+1);
}

} //::profileMngr

//**************************main**************************

using namespace profileMngr;
int main(int argc, char* argv[])
{
	int VariantID=0;

#if !defined (OSAL_GEN3_SIM) && !defined (OSAL_GEN4)
 	uint8 OEMType = 0xFF;
 	if (DP_S32_NO_ERR != DP_s32GetConfigItem("CMVariantCoding", "OEMType", &OEMType, 1))
 		ETG_TRACE_FATAL(("Error :Loading OEM type"));

 	ETG_TRACE_COMP(("OEMID %d", ETG_ENUM(TR_OEM_ID,OEMType) ));

 	//Check if it PIVI / Renault
 	switch (OEMType)
 	{
 		case OEM_TYPE_PIVI:VariantID=1; //PIVI library
 		break;
 	}
#endif
	for (int k=0;k < argc;++k)
	if (argv[k][0]=='-' && argv[k][2]==0)
		switch (argv[k][1])
		{
			case 'v':if (argc > k+1) VariantID=atoi(argv[k+1]);++k;break;//select variant for library to load
			case 't':g_bTestMode=true;break;
		}

	char const* err=configIF::loadConfig(LibFiles[VariantID]);

	if (err!=NULL)
	{
		printf("Error Loading config Library: %s\n",err);
		ETG_TRACE_FATAL(("Loading config Library failed: %s",err));
		sd_notify(0,"READY=1");
		_exit(excNoConfig);//_exit() dose not use at_exit callback. OSAL will not print a call stack and reset the target
	}

    DP_vCreateDatapool();
	ProfileApp profileApp(configIF::getInst());

    uint8_t startUser=profileApp.getCurrentUser();
    configIF::getInst().Init(startUser);//wait for the IKey signal for max 500 ms inside init call
    if (startUser!= profileApp.getCurrentUser())
    {
   		profileApp.updateUser(startUser);
    }

    CCAAppMain::initCCA();
    CCAAppMain * pApp=new (std::nothrow) CCAAppMain(profileApp);
    if (pApp != NULL)
    {
        if ( pApp->bInitInstance(0, CCA_C_U16_APP_FC_USERMANAGER) == FALSE )
        {
            ETG_TRACE_FATAL(("bInitInstance Failed" ));
        }
        else
        	ETG_TRACE_COMP(("CCA Server Started" ));
    }

    sd_notify(0,"READY=1");
	return	app::core::executeProfileMngr(argc,argv);
}


