/*!
*******************************************************************************
* @file             : SmartProxy.c
*******************************************************************************
*  - PROJECT:       : IPCM
*  - SW-COMPONENT   : IPCM
*  - DESCRIPTION    : Smart Proxy performs traffic classification
*  - COPYRIGHT      : &copy; 2017 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          | Version  | Modification
* ----------|---------------|----------|---------------------------------------
*22.09.2017 | rhk6kor(RBEI/ECO2)       | 0.0.1    | Initial version
*05.10.2017 | Deepa Jose(RBEI/ECO2)    | 0.0.2    | Registrations & Dispatcher process
*10.01.2018 | Ashsih Kumar (RBEI/ECO2) | 0.0.3    | Scheduler modification
******************************************************************************/
#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include "dlt/dlt.h"

#include "agentplugin.h"
#include "SmartProxy.h"
#include "SmartProxyInterface.h"
#include "MessageDispatcher.h"

//DLT CONTEXT DECLARATION
DLT_DECLARE_CONTEXT(SMART_PROXY);

//Registration objects with DBComMgr
interface_agents* iAgent_Client = NULL;
interface_agents* iAgent_Server = NULL;
static GHashTable* hash_signalinfo = NULL;

#define E2E_INTERFACE "com.bosch.AutomotiveProxy.E2ENavigation"
#define E2E_XML       "/var/opt/bosch/static/com.bosch.AutomotiveProxy.E2ENavigation.xml"
#define E2E_OBJ_PATH  "/com/bosch/AutomotiveProxy/E2ENavigation"

#define VD_INTERFACE "com.bosch.AutomotiveProxy.VehicleInfoManager"
#define VD_XML "/var/opt/bosch/static/com.bosch.AutomotiveProxy.VehicleInfoManager.xml"
#define VD_OBJ_PATH "/com/bosch/AutomotiveProxy/VehicleInfoManager"

#define AUTOMOTIVE_PROXY_DBUS "com.bosch.AutomotiveProxy"

static GDBusNodeInfo * nodeinfo;
static GHashTable *hServiceParams_hashtable= NULL;
static char* strIntrospectxml(char* arg_filename);
static gboolean bReadNodeInfo_onstartup(int clientIndex);
static void vStroreInterfaceKeys(char* interface_name,char* serviceName,int type);
static gboolean gKeyValue_construct(GDBusArgInfo **argument,char* service_name,int type);
static GVariant* KEYVALUE_sendparkLocation(char *service_name, GVariant* gparams);
static GVariant* KEYVALUE_updateVehicleInfo(GVariant* gparams);
static void vhashKeyDestroy(gpointer data);
static void vhashValueDestroy(gpointer data);

#define NUMBER_OF_CLIENTS 2

static StIpcmClientInfo g_ipcm_client_info[NUMBER_OF_CLIENTS] = {
         { AUTOMOTIVE_PROXY_DBUS, E2E_XML, E2E_OBJ_PATH, E2E_INTERFACE, NULL, &vE2EIfaceAvailable_cb },
         { AUTOMOTIVE_PROXY_DBUS, VD_XML, VD_OBJ_PATH, VD_INTERFACE, NULL, &vVDIfaceAvailable_cb }
         //Add Next client info here 
                                                                };  
                     
void vhashKeyDestroy(gpointer data)
{
    DLT_LOG(SMART_PROXY, DLT_LOG_INFO,DLT_STRING("SP- vhashKeyDestroy"));
	if(data != NULL)
	{
		g_free (data);
	}
}

void vhashValueDestroy(gpointer data)
{
    DLT_LOG(SMART_PROXY, DLT_LOG_INFO,DLT_STRING("SP- vhashValueDestroy"));
	if(data != NULL)
	{
		g_free (data);
	}
}


/******************************************************************************
* Function:       vCleanUpSmartProxy
* Description:    vCleanUpSmartProxy
* Parameters:     (void)
* Return:         void
*****************************************************************************/
void vCleanUpSmartProxy()
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
	,DLT_STRING("SP- vCleanUpSmartProxy"));
	//set the condition for dispatcher thread exit
}



/**
*******************************************************************************
** FUNCTION   : vParsexmlfile
*******************************************************************************
* \fn     vParsexmlfile
* \brief  Function to cleanup smartproxy
* \param  None.
* \retval void.
******************************************************************************/
tString vParsexmlfile(tString arg_filename)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
	,DLT_STRING("SP- vParsexmlfile"));
	if (g_file_test (arg_filename, G_FILE_TEST_EXISTS))
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
		,DLT_STRING("SP- E2E file exists"));

		gsize l_length;char * l_content;
		if (g_file_get_contents (arg_filename, &l_content, &l_length, NULL))
		{
			return l_content;
		}
	}
	return NULL;
}

/**
*******************************************************************************
** FUNCTION   : sendMethodResponse
*******************************************************************************
* \fn     sendMethodResponse
* \brief  forwards the response received to DBus Communication manager
* \param  Message*
* \retval tVoid.
******************************************************************************/
tVoid sendMethodResponse(Message *msg_reponse)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
	,DLT_STRING("SP- sendMethodResponse"));

	guint32 request_handle;
	GVariant* value = NULL;
	GError *gError = NULL;
	//    GenericMethodResponse(request_handle, value, &gError);
}

/**
*******************************************************************************
** FUNCTION   : vGenericMethodCall_request
*******************************************************************************
* \fn     vGenericMethodCall_request
* \brief  
* \param  Message*
* \retval tVoid.
******************************************************************************/

tVoid vGenericMethodCall_request (tCString sender, tCString method_name, 
GVariant *parameters
,const void* request_handle)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
	,DLT_STRING("SP- vGenericMethodCall_request"));
}

/**
*******************************************************************************
** INTERFACE TO RPC CLIENT   : vCleanUpSmartProxy
*******************************************************************************
* \fn     vCleanUpSmartProxy
* \brief  Function to cleanup smartproxy
* \param  None.
* \retval void.
******************************************************************************/
tString* sGetDBusInterfaceSignals()
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
	,DLT_STRING("SP- sGetDBusInterfaceSignals"));

	char* arg_filename = "/var/opt/bosch/static/com.bosch.AutomotiveProxy.E2ENavigation.xml";
	gchar * l_content = NULL;
	char** l_SignalNames = NULL;

	if(NULL != arg_filename)
	{
		if (g_file_test (arg_filename, G_FILE_TEST_EXISTS))
		{
			DLT_LOG(SMART_PROXY, DLT_LOG_INFO, DLT_STRING("IPCM- file exists"));

			gsize l_length;
			if (g_file_get_contents (arg_filename, &l_content, &l_length, NULL))
			{
				DLT_LOG(SMART_PROXY, DLT_LOG_INFO
				, DLT_STRING("\n IPCM- file content: ")
				, DLT_STRING(l_content));
			}
		}

		if(l_content != NULL)
		{

			GError *l_poerror = NULL;
			GDBusNodeInfo *nodeinfo;
			static GDBusInterfaceInfo *interface_info = NULL;

			guint32 i_arrlen = 2;
			l_SignalNames = (char**) malloc( i_arrlen * sizeof(char*));

			nodeinfo = g_dbus_node_info_new_for_xml (l_content, &l_poerror);
			gint n = 0;
			for (n = 0; nodeinfo->interfaces != NULL &&
			nodeinfo->interfaces[n] != NULL; n++)
			{
				interface_info = nodeinfo->interfaces[n];
				gint m = 0;
				for (m = 0; interface_info->signals != NULL
				&& interface_info->signals[m] != NULL; m++)
				{
					const GDBusSignalInfo *signal = interface_info->signals[m];
					DLT_LOG(SMART_PROXY, DLT_LOG_INFO
					, DLT_STRING("\n IPCM-SMPRXY: signalname: ")
					, DLT_STRING(signal->name));

					l_SignalNames[m] = (char*) malloc(( strlen(signal->name) +1) * sizeof(char));
					memset(l_SignalNames[m], 0, strlen(signal->name) + 1);
					strncpy(l_SignalNames[m], signal->name, strlen(signal->name));
				}
			}
		}
		else
		{
			DLT_LOG(SMART_PROXY, DLT_LOG_INFO
			, DLT_STRING("\n IPCM-SMPRXY: Error: file is empty "));
		}
	}
	else
	{
		DLT_LOG(SMART_PROXY, DLT_LOG_INFO
		, DLT_STRING("\n IPCM-SMPRXY: Error: file not found "));
	}
	return l_SignalNames;
}

/******************************************************************************
* Function:       vhandleMethodRequest_cb
* Description:    vhandleMethodRequest_cb
* Parameters:     (const gchar*,const gchar*,GVariant*,const void*)
* Return:         void
*****************************************************************************/
void vhandleMethodRequest_cb(const gchar *sender,const gchar *methodName,
GVariant *gparams,const void *request_handle)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
	,DLT_STRING("SP- vhandleMethodRequest_cb"));


	//Message
	Header *HeaderData = malloc(sizeof(Header));
	memset(HeaderData,0,sizeof(Header));
	HeaderData->messageType = MT_REQUEST;
        HeaderData->topicType = IPCM_DEFAULT_TOPIC;
 
	guint32 t_requesthandlerID = (guint32)request_handle;
	HeaderData->requestHandle = (guint64)t_requesthandlerID;
	HeaderData->name = strdup(methodName);
	HeaderData->sender = strdup("DEFAULT");
	HeaderData->intent = strdup("DEFAULT");

	Message *MessageData = malloc(sizeof(Message));
	memset(MessageData,0,sizeof(Message));
	MessageData->header = HeaderData;
	MessageData->payload = gparams;

	//Send to Message Dispathcer for processing
	bHandleMessageRequest(MessageData);
}



/******************************************************************************
* Function:       vhandleSignal_cb
* Description:    vhandleSignal_cb
* Parameters:     (const gchar*,GVariant*)
* Return:         void
*****************************************************************************/
void vhandleSignal_cb(const gchar* signalName,GVariant* gparams)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- vhandleSignal_cb"));
	DLT_LOG(SMART_PROXY, DLT_LOG_INFO,DLT_STRING("SP- Signal Name"),DLT_STRING(signalName));

	char* Spayload = g_variant_print(gparams,TRUE);
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- Signal Params"),DLT_STRING(Spayload)); 
	free(Spayload);
	
	//Message
	Header *HeaderData = malloc(sizeof(Header));
	memset(HeaderData,0,sizeof(Header));
	HeaderData->messageType = MT_SIGNAL;
	HeaderData->requestHandle = 0;
	HeaderData->name = strdup(signalName);
	HeaderData->sender = strdup("DEFAULT");
	HeaderData->intent = strdup("DEFAULT");

	Message *MessageData = malloc(sizeof(Message));
	memset(MessageData,0,sizeof(Message));

	GVariant* gConstructedVariant = NULL;
	
	//Payload payload_data;
	if(0 == strcmp(signalName,"SendParkingLocation"))
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- Signal = SendParkingLocation"));
		gConstructedVariant = KEYVALUE_sendparkLocation("SendParkingLocation",gparams);
                HeaderData->topicType = IPCM_DEFAULT_TOPIC; 
	}
        else if(!strcmp(signalName, "UpdateVehicleInfo"))
        {
              DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                          DLT_STRING("(): SP- Signal = UpdateVehicleInfo"));
              HeaderData->topicType= IPCM_TELEMETRY_TOPIC;
              gConstructedVariant = KEYVALUE_updateVehicleInfo( gparams );  
        }
	else
	{
		DLT_LOG(SMART_PROXY, DLT_LOG_INFO,DLT_STRING("SP- UnKnown Signal recevied"));
	}
        MessageData->header = HeaderData;
        if(NULL != gConstructedVariant)
        {
             MessageData->payload = gConstructedVariant;
	     //Send to Message Dispathcer for processing
   	     bHandleMessageRequest(MessageData);
        }
        else
        {
                free( HeaderData->sender );
                free( HeaderData->intent );
                free( HeaderData->name );
                free( HeaderData );
                free (MessageData );
        } 
}

/**
*******************************************************************************
** FUNCTION   : StoreInterfaceKeys
*******************************************************************************
* \fn     StoreInterfaceKeys
* \brief  function to get interface keys
* \param  interface_name, serviceName, type
* \retval gboolean.
******************************************************************************/
gboolean StoreInterfaceKeys(char* interface_name, char* serviceName,int type)
{
	DLT_LOG(SMART_PROXY, DLT_LOG_INFO,DLT_STRING("SP- StoreInterfaceKeys"));
	gboolean bReturn = false;
	if(interface_name)
	{
                int client_id = 0;
                for(client_id = 0; client_id < NUMBER_OF_CLIENTS; client_id++)
                {
                    if(!strcmp(interface_name, g_ipcm_client_info[client_id].iface_name))
                         break;
                }

                if((client_id >= NUMBER_OF_CLIENTS) || (NULL != g_ipcm_client_info[client_id].nodeInfo))
                {
                     DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING(__FUNCTION__), DLT_STRING(
                                                                 "Failed to get nodeInfo"));
                     return FALSE;
                }

		GDBusInterfaceInfo *interfaces = g_dbus_node_info_lookup_interface(
                                            g_ipcm_client_info[client_id].nodeInfo,
                                                               interface_name 
                                                                                  );
		if(interfaces)
		{
                        for(client_id = 0; client_id < NUMBER_OF_CLIENTS; client_id++)
                        {
                          if(!strcmp(interface_name, g_ipcm_client_info[client_id].iface_name))
                              break;
                        } 
                        
                        if((client_id >= NUMBER_OF_CLIENTS) || (NULL != g_ipcm_client_info[client_id].nodeInfo))
                        {
                           DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING(__FUNCTION__), DLT_STRING(
                                                                     "Failed to get nodeInfo"));      
                           return FALSE;
                        }
			DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- VALID INTERFACE FOUND"));
			switch(type)
			{
			case 0: //signal
				DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- SIGNAL LOOKUP"));
				GDBusSignalInfo *signal = g_dbus_interface_info_lookup_signal( interfaces,
                                                                                               serviceName );
				
				if(signal)
				{
					DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- SIGNAL INFO FOUND"));
					GDBusArgInfo **args = signal->args;
					bReturn = gKeyValue_construct(args,serviceName,type);
				}
				else
					DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- SIGNAL INFO NOT FOUND"));
				break;

			case 1: //Method
				DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- METHOD LOOKUP"));
				DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- XX RETURNING FALSE ALWAYS XX"));
				break;

			case 2:  //property
				DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- PROPERTY LOOKUP"));
				DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- XX RETURNING FALSE ALWAYS XX"));
				break;
			}
		}
		else
		{
			DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- INTERFACE IS INVALID"));
		}
	}
	else
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- NODE INFO IS NULL"));
	}
	return bReturn;
}
/**
*******************************************************************************
** FUNCTION   : KEYVALUE_sendparkLocation
*******************************************************************************
* \fn     KEYVALUE_sendparkLocation
* \brief  function to construct key value pair for parking location
* \param  interface_toLookfar,serviceName,type
* \retval gboolean.
******************************************************************************/
GVariant* KEYVALUE_sendparkLocation(char *service_name,GVariant* gparams)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- KEYVALUE_sendparkLocation"));
	GVariantBuilder *builder = NULL;
	GVariant *gVariantParam = NULL;
	
	gchar *VIN = "DEFAULT";
	gchar *Token = "DEFAULT";
	gchar *TimeStamp ="DEFAULT";
	gdouble Latitude = 0.0;
	gdouble Longitude = 0.0;
	gchar *Address = "DEFAULT";
	gboolean Accuracy = FALSE;
	gboolean PrivacyModeFlag = FALSE;
	
	g_variant_get(gparams,"(sssddsbb)",
					&VIN,
					&Token,
					&TimeStamp,
					&Latitude,
					&Longitude,
					&Address,
					&Accuracy,
					&PrivacyModeFlag);
	
	
	builder = g_variant_builder_new(G_VARIANT_TYPE("a{sv}"));
	g_variant_builder_add (builder,"{sv}","VIN",g_variant_new_string(VIN));
	g_variant_builder_add (builder,"{sv}","Token",g_variant_new_string(Token));
	g_variant_builder_add (builder,"{sv}","TimeStamp",g_variant_new_string(TimeStamp));
	g_variant_builder_add (builder,"{sv}","Latitude",g_variant_new_double(Latitude));
	g_variant_builder_add (builder,"{sv}","Longitude",g_variant_new_double(Longitude));
	g_variant_builder_add (builder,"{sv}","Address",g_variant_new_string(Address));
	g_variant_builder_add (builder,"{sv}","Accuracy",g_variant_new_boolean(Accuracy));
	g_variant_builder_add (builder,"{sv}","PrivacyModeFlag",g_variant_new_boolean(PrivacyModeFlag));
    //Builder Completed
	gVariantParam = g_variant_builder_end(builder);
	
	char* Spayload = g_variant_print(gVariantParam,TRUE);
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- KEYVALUE sendparkLocation"),DLT_STRING(Spayload)); 
	free(Spayload);
	
	//Freeing the String Variables
	g_free(VIN);
	g_free(Token);
	g_free(TimeStamp);
	g_free(Address);
	
	return gVariantParam;
	
	/*GVariantBuilder *builder;
	GVariant *gVariantParam =NULL;
	builder = g_variant_builder_new(G_VARIANT_TYPE("a{sv}"));*/

	/*gpointer value = g_hash_table_lookup(hServiceParams_hashtable,service_name);
	GList *list = value;
	if(list == NULL)
	{
		return NULL;
	}
	
	gsize iChildNumResponseInfo =0;
	iChildNumResponseInfo = g_variant_n_children(gparams);





	g_variant_get(gparams, "(sssddsbb)", &VIN, &Token, &TimeStamp
	, &Latitude, &Longitude, &Address,&Accuracy,&PrivacyModeFlag);

	printf("VIN :: %s\n",VIN);
	printf("Token :: %s\n",Token);
	printf("TimeStamp :: %s\n",TimeStamp);
	printf("Address :: %s\n",Address);
	printf("Latitude :: %lf\n",Latitude);
	printf("Longitude :: %lf\n",Longitude);
	printf("Accuracy :: %d\n",Accuracy);
	printf("PrivacyModeFlag :: %d\n",PrivacyModeFlag);

	GList *next = list->next;
	if(list)
	{
		printf("Key is::%s \n",list->data);
		g_variant_builder_add (builder, "{sv}", list->data,g_variant_new("s",VIN));
	}
	list = next;
	if(list)
	{
		next = list->next;
		printf("Key is::%s \n",list->data);
		g_variant_builder_add (builder, "{sv}", list->data,g_variant_new("s",Token));
	}
	list= next;
	if(list)
	{
		next = list->next;
		printf("Key is::%s \n",list->data);
		g_variant_builder_add (builder, "{sv}" , list->data,g_variant_new("s",TimeStamp));
	}
	list= next;
	if(list)
	{
		next = list->next;
		printf("Key is::%s \n",list->data);
		g_variant_builder_add (builder, "{sv}", list->data,g_variant_new("d",Latitude));
	}
	list= next;
	if(list)
	{
		next = list->next;
		printf("Key is::%s \n",list->data);
		g_variant_builder_add (builder, "{sv}" , list->data,g_variant_new("d",Longitude));
	}
	list= next;
	if(list)
	{
		next = list->next;
		printf("Key is::%s \n",list->data);
		g_variant_builder_add (builder, "{sv}", list->data,g_variant_new("s",Address));
	}
	list= next;
	if(list)
	{
		next = list->next;
		printf("Key is::%s \n",list->data);
		g_variant_builder_add (builder, "{sv}", list->data,g_variant_new("b",Accuracy));
	}
	list= next;
	if(list)
	{
		next = list->next;
		printf("Key is::%s \n",list->data);
		g_variant_builder_add (builder, "{sv}", list->data,g_variant_new("b",PrivacyModeFlag));
	}
	gVariantParam = g_variant_builder_end (builder);
	g_variant_builder_unref (builder);

	return gVariantParam;*/

}

/**
*******************************************************************************
** FUNCTION   : KEYVALUE_updateVehicleinfo
*******************************************************************************
* \fn     KEYVALUE_updateVehicleInfo
* \brief  function to construct key value pair 
          for vehicle info update
* \param  gparams
* \retval GVariant*
******************************************************************************/
GVariant* KEYVALUE_updateVehicleInfo (GVariant* gparams)
{
     DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("+"),
                           	 DLT_STRING(__FUNCTION__));
	 
     GVariantBuilder outer_builder;
     GVariantBuilder inner_builder;
     GVariant *l_constructedVariant = NULL;
     GVariant *vehicleInfo = NULL;
     GVariantIter iter;
     
     gchar*  propertyName = NULL;
     gchar*  valueType = NULL;
     gchar*  value = NULL;
     gchar*  zone = NULL;
     gdouble accuracy = 0;
     guchar  notificationType = 0;	 

     g_variant_get(gparams, "(@a(ssssdy))", &vehicleInfo);
     g_variant_iter_init (&iter, vehicleInfo);
     
     g_variant_builder_init (&outer_builder, G_VARIANT_TYPE ("aa{sv}"));
	 
     while (g_variant_iter_next (&iter,   "(ssssdy)",
                              	       &propertyName,
                                          &valueType,
				 	      &value,
					       &zone,
				           &accuracy,
				   &notificationType ))
     {	 
           g_variant_builder_init (&inner_builder, G_VARIANT_TYPE ("a{sv}"));  
	   if( (propertyName) && (valueType) && (value) && (zone) )
 	   {
	        g_variant_builder_add (&inner_builder,"{sv}", "property", 
	                            g_variant_new_string(propertyName));
			
	        g_variant_builder_add (&inner_builder, "{sv}", "datatype", 
 	                             g_variant_new_string(valueType));

                g_variant_builder_add (&inner_builder, "{sv}", "value",
                                         g_variant_new_string(value));
            
                g_variant_builder_add (&inner_builder, "{sv}", "zone",
                                       	g_variant_new_string(zone));
            
                g_variant_builder_add (&inner_builder, "{sv}", "accuracy",
                                        g_variant_new_double(accuracy));

                g_variant_builder_add (&inner_builder, "{sv}", "status",
                                     g_variant_new_byte(notificationType));

           }
           else
           {
               DLT_LOG(SMART_PROXY,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                        DLT_STRING("(): Null vehicle info params"));
                return NULL;
           }
  	   g_variant_builder_add(&outer_builder, "a{sv}", &inner_builder);
	   g_variant_builder_end(&inner_builder);
     }		

     l_constructedVariant = g_variant_builder_end(&outer_builder);
     return l_constructedVariant;
}	

/****************************************************************************
* Function:       vE2EIfaceAvailable_cb
* Description:    vE2EIfaceAvailable_cb
* Parameters:     (interface_agents*)
* Return:         void
*****************************************************************************/
void vE2EIfaceAvailable_cb(interface_agents* interfaceAgent)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- vDBusComMgrclient_cb"));
	if(interfaceAgent)
	{
		bSubscribesignal(interfaceAgent,"SendParkingLocation",&vhandleSignal_cb);
		gchar* interfaceName = E2E_INTERFACE ;

		gchar* signalName = "SendParkingLocation";
		
		gboolean bStoreKeys_InHash = StoreInterfaceKeys(interfaceName, signalName, 0);
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- key store in hash ="),DLT_BOOL(bStoreKeys_InHash));
		bSubscribesignal(interfaceAgent,"SendDestinationToPhone", &vhandleSignal_cb);
	}
	else
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
		,DLT_STRING("SP- interface Agent is NULL"));
	}
}

/****************************************************************************
* Function:       vVDIfaceAvailable_cb
* Description:    vVDIfaceAvailable_cb
* Parameters:     (interface_agents*)
* Return:         void
*****************************************************************************/
void vVDIfaceAvailable_cb(interface_agents* interfaceAgent)
{
     DLT_LOG(SMART_PROXY,DLT_LOG_INFO, DLT_STRING("+"), DLT_STRING(__FUNCTION__)); 
     if(interfaceAgent)
     {
                gchar* interfaceName = VD_INTERFACE;
                gchar* signalName = "UpdateVehicleInfo";

                gboolean bStoreKeys_InHash = StoreInterfaceKeys(interfaceName,signalName,0);
                DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- key store in hash ="),DLT_BOOL(bStoreKeys_InHash));
                bSubscribesignal(interfaceAgent, "UpdateVehicleInfo", &vhandleSignal_cb);
     }    
     DLT_LOG(SMART_PROXY,DLT_LOG_INFO, DLT_STRING("-"), DLT_STRING(__FUNCTION__)); 
}       

/******************************************************************************
* Function:       Register_DBusComMgr
* Description:    Register_DBusComMgr
* Parameters:     (void)
* Return:         void
*****************************************************************************/
gboolean Register_DBusComMgr(int clientIndex)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
	,DLT_STRING("SP- Register_DBusComMgr"));
	GError *gError = NULL;

	//Registration as Client with DBusComMgr
	tString Client_XML = vParsexmlfile(g_ipcm_client_info[clientIndex].xml_path);
	tString Client_BusName = g_strdup(g_ipcm_client_info[clientIndex].bus_name);
	tString Client_ObjPath = g_strdup(g_ipcm_client_info[clientIndex].obj_path);

	if(Client_XML == NULL)
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
		,DLT_STRING("SP- Client_XML is not found"));
		return FALSE;
	}
	iAgent_Client = RegisterClient( Client_XML, Client_BusName, Client_ObjPath,
                                          &gError, g_ipcm_client_info[clientIndex].fn_callback );

	if(iAgent_Client == NULL)
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
		,DLT_STRING("SP- DBusComMgr Client Registeration failed"));
		return FALSE;
	}

	//Registration as Server with DBusComMgr
	tString Server_BusName = g_strdup(SP_SERVER_BUSNAME);
	tString Server_ObjPath = g_strdup(SP_SERVER_OBJPATH);
	iAgent_Server = RegisterServer(Server_XML, Server_BusName, Server_ObjPath,
                                                &gError, &vhandleMethodRequest_cb);
	if(iAgent_Server == NULL)
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
		,DLT_STRING("SP- DBusComMgr Server Registeration failed"));
		return FALSE;
	}
	return TRUE;
}

/******************************************************************************
* Function:       bReadNodeInfo_onstartup
* Description:    bReadNodeInfo_onstartup
* Parameters:     xml
* Return:         void
*****************************************************************************/
gboolean bReadNodeInfo_onstartup(int clientIndex)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- bReadNodeInfo_onstartup"));
	gboolean bReturn = false;
	char* arg_xml = strIntrospectxml(g_ipcm_client_info[clientIndex].xml_path);
	GError *l_poerror = NULL;
	hServiceParams_hashtable = g_hash_table_new_full( g_str_hash,
                                                          g_str_equal,
                                                          vhashKeyDestroy,
                                                          vhashValueDestroy );
	if(arg_xml)
	{
		DLT_LOG(SMART_PROXY, DLT_LOG_INFO,DLT_STRING("SP- EXTRACTING NODE INFO"));
		g_ipcm_client_info[clientIndex].nodeInfo = 
                                     g_dbus_node_info_new_for_xml (arg_xml,&l_poerror);
\
		if(g_ipcm_client_info[clientIndex].nodeInfo)
		{
			DLT_LOG(SMART_PROXY, DLT_LOG_INFO,DLT_STRING("SP- NODEINFO NOT NULL"));
			bReturn = true;
		}

	}
	else
	{
		DLT_LOG(SMART_PROXY, DLT_LOG_INFO,DLT_STRING("SP- EXTRACTING NODE INFO"));
	}
	return bReturn;
}

/******************************************************************************
* Function:       strIntrospectxml
* Description:    strIntrospectxml
* Parameters:     xml path
* Return:         xml content
*****************************************************************************/
char* strIntrospectxml(char* arg_filename)
{
	char* l_content;
	if (g_file_test (arg_filename, G_FILE_TEST_EXISTS)) {
		gssize l_length;

		if (g_file_get_contents (arg_filename, &l_content, &l_length, NULL)) {
			return l_content;
		}
	}
	return NULL;
}

/******************************************************************************
* Function:       print_hashTable
* Description:    function to print hashTable
* Parameters:     service_name
* Return:         none
*****************************************************************************/
void  print_hashTable(char *service_name)
{
	DLT_LOG(SMART_PROXY, DLT_LOG_INFO,DLT_STRING("SP- print_hashTable"));

	gpointer value = g_hash_table_lookup(hServiceParams_hashtable,service_name);
	GList *list= value;
	while(list)
	{
		GList *next = list->next;
		printf("list val::%s\n",list->data);
		list = next;
	}
}

/******************************************************************************
* Function:       gKeyValue_construct
* Description:    function to construct key value
* Parameters:     service_name
* Return:         bool
*****************************************************************************/
gboolean gKeyValue_construct(GDBusArgInfo **argument,char* service_name,int type)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- gKeyValue_construct"));
	//Generalise the code without switch case by handling out args
	gboolean bInsertHash = false;
	if(argument)
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- valid args"));
		switch(type)
		{
		case 0 :
			DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- EXTRACTING SIGNAL PARAMS"));
			int i=0;
			GDBusArgInfo *Argstruct;
			GList *Param_list = NULL;
			while(NULL != argument[i])
			{
				printf("value is :%d\n",i);
				Argstruct = argument[i];
				printf(" name :: %s\n",Argstruct->name);
				Param_list = g_list_append(Param_list,Argstruct->name);
				i++;
			}
			printf("No of args are: %d\n",i);
			//g_list_foreach(Param_list, (GFunc)printf, NULL);
			if(hServiceParams_hashtable != NULL && service_name != NULL)
			{
				if(g_hash_table_lookup(hServiceParams_hashtable,service_name) == NULL)
				{
					if(Param_list != NULL)
					{
						bInsertHash = g_hash_table_insert(hServiceParams_hashtable,service_name,Param_list);
						DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- Value inserted in hash ="),DLT_BOOL(bInsertHash));
						//print_hashTable(service_name);
					}
					else
					{
						DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- SERVICE OR ARGUMENT IS NULL"));
					}
				}
				else
				{
					DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- HASH VALUE ALREADY PRESNET"));
				}
			}
			else
			{
				DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- HASH TABLE NOT CREATED VALUE OR SERVICE NAME NULL"));
			}
			break;
		case 1: //Method
				DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- EXTRACTING METHOD PARAMS"));
				DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- XX RETURNING FALSE ALWAYS XX"));
				break;			
		

		case 2:  //property
				DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- EXTRACTING PROPERTY PARAMS"));
				DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- XX RETURNING FALSE ALWAYS XX"));
				break;
		}
	}
	else
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- ARGS LIST IS NULL"));
	}
	return bInsertHash;
}

/******************************************************************************
* Function:       vhandleMethodResponse_cb
* Description:    vhandleMethodResponse_cb
* Parameters:     (char*,char*,guint64)
* Return:         gboolean
*****************************************************************************/
guint64 vhandleMethodResponse_cb(const char* identity,const char* result,guint64 request_handle) 
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
	,DLT_STRING("SP- vhandleMethodResponse_cb"));
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
	,DLT_STRING("SP- METHOD NAME "),DLT_STRING(identity));
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
	,DLT_STRING("SP- RESULT "),DLT_STRING(result));
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self())
	,DLT_STRING("SP- REQUEST HANDLE "),DLT_UINT64(request_handle));

	if(search(request_handle))
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- VALID RESPONSE"));
		GError *error = NULL;
		guint32 t_requestHandle = (guint32)request_handle;	//GINT_TO_POINTER
		GenericMethodResponse((const void*)t_requestHandle,g_variant_new("(s)",result),&error);
		return 1;
	}
	else
	{
		DLT_LOG(SMART_PROXY, DLT_LOG_INFO, DLT_STRING("SP- INVALID RESPONSE"));
		return 0;
	}
	return 0;
}



/******************************************************************************
* Function:       bInitSmartProxy
* Description:    Function to init smart proxy
* Parameters:     (void)
* Return:         gboolean
*****************************************************************************/
gboolean bInitSmartProxy()
{
	DLT_REGISTER_CONTEXT(SMART_PROXY,"SMPY","SMART PROXY");
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- bInitSmartProxy"));

	// hash_signalinfo = g_hash_table_new_full(g_str_hash,g_str_equal,g_free
	// ,g_free);
	//readnode info
	//hServiceParams_hashtable = g_hash_table_new_full(g_str_hash,g_str_equal,vhashKeyDestroy,vhashValueDestroy);
	int loop = 0;
        for (loop = 0; loop < NUMBER_OF_CLIENTS; loop++)
        {
     	     bool bNodeinfo = bReadNodeInfo_onstartup(loop);
	     DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING(g_ipcm_client_info[loop].xml_path),DLT_STRING(" SP- NODE INFO READ:"),DLT_BOOL(bNodeinfo));
        }

	//sGetInterfaceArguments();
	if(!bInitMessageDispatcher())
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- failed to Initialse MessageDispatcher"));
		return FALSE;
	}
	else
	{
		if(!bRegisterRPCClinet("RPCClient",&vhandleMethodResponse_cb))
		{
			DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- failed to Register with RPC Client"));
			return FALSE;
		}
		else
		{
                        for(loop = 0; loop < NUMBER_OF_CLIENTS; loop++)
                        {
  			    if(!Register_DBusComMgr(loop))
			    {
			 	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- failed to Register with DBusComMgr"));
				return FALSE;
			    }
                        }
		}
	}
	return TRUE;
}
