#include "JsonRPC2_0Validation.h"
#include <vector>
#include <algorithm>
#define JSONRPC_VERSION "2.0"
DLT_DECLARE_CONTEXT(JSON_VAL);


std::vector<std::string> JSONRPC_RequestKeys = {"jsonrpc", "method","params","id"};
std::vector<std::string> JSONRPC_RequestKeys_NoParams = {"jsonrpc", "method","id"};
std::vector<std::string> JSONRPC_NotificationKeys = {"jsonrpc", "method","params"};
std::vector<std::string> JSONRPC_NotificationKeys_NoParams = {"jsonrpc", "method",};
std::vector<std::string> JSONRPC_ResponseKeys = {"jsonrpc", "result","id"};
std::vector<std::string> JSONRPC_ErrorResponseKeys = {"jsonrpc", "error","id"};
JsonRPC20Message *JsonRPC20Message::m_pJsonRPC20Message = NULL;
enum jsonDatatypes
{
    NULLVALUE =0,
    INTVALUE,
    UINTVALUE,
    REALVALUE,
    STRINGVALUE,
    BOOLEANVALUE,
    ARRAYVALUE,
    OBJECTVALUE
};


/******************************************************************************
** FUNCTION   : JsonRPC20Message
*******************************************************************************
* \fn     JsonRPC20Message::JsonRPC20Message
* \brief  Constructor
* \param  None.
* \retval None.
******************************************************************************/
JsonRPC20Message::JsonRPC20Message()
{
	DLT_REGISTER_CONTEXT(JSON_VAL,"JSON","jason validation context for logging");
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    m_protocol ="jsonrpc";
    m_method = "method";
    m_params = "params";
    m_id = "id";
    m_result = "result";
    m_error = "error";
    m_methodString ="";
    m_RequestId = -1;
}

/******************************************************************************
** FUNCTION   : ~JsonRPC20Message()
*******************************************************************************
* \fn     JsonRPC20Message::~JsonRPC20Message()
* \brief  Destructor
* \param  None.
* \retval None.
******************************************************************************/
JsonRPC20Message::~JsonRPC20Message()
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
}

/******************************************************************************
** FUNCTION   : getInstance
*******************************************************************************
* \fn     JsonRPC20Message::getInstance
* \brief  function to get instance of this class
* \param  None.
* \retval None.
******************************************************************************/

JsonRPC20Message* JsonRPC20Message::getInstance()
{
	DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	if(NULL == m_pJsonRPC20Message)
	{
		m_pJsonRPC20Message = new JsonRPC20Message();
	}
	return m_pJsonRPC20Message;
}

/******************************************************************************
** FUNCTION   : vSetMethod
*******************************************************************************
* \fn     JsonRPC20Message::vSetMethod
* \brief  function to set method
* \param  std string.
* \retval None.
******************************************************************************/
void JsonRPC20Message::vSetMethod(std::string method)
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    m_methodString =method;
}

/******************************************************************************
** FUNCTION   : vSetParams
*******************************************************************************
* \fn     JsonRPC20Message::vSetParams
* \brief  function to set params
* \param  Json::Value.
* \retval None.
******************************************************************************/
void JsonRPC20Message::vSetParams(Json::Value params)
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    m_RequestParams = params;
}

/******************************************************************************
** FUNCTION   : vSetId
*******************************************************************************
* \fn     JsonRPC20Message::vSetId
* \brief  function to set id
* \param  int.
* \retval None.
******************************************************************************/
void JsonRPC20Message::vSetId(int id)
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    m_RequestId = id;
}

/******************************************************************************
** FUNCTION   : strGetMethod
*******************************************************************************
* \fn     JsonRPC20Message::strGetMethod
* \brief  function to get method
* \param  None.
* \retval std string.
******************************************************************************/
std::string JsonRPC20Message::strGetMethod()
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    return m_methodString;
}

/******************************************************************************
** FUNCTION   : jsonGetParams
*******************************************************************************
* \fn     JsonRPC20Message::jsonGetParams
* \brief  function to get params
* \param  None.
* \retval std string.
******************************************************************************/
Json::Value JsonRPC20Message::jsonGetParams()
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    return m_RequestParams;
}

/******************************************************************************
** FUNCTION   : iGetID
*******************************************************************************
* \fn     JsonRPC20Message::iGetID
* \brief  function to get Id
* \param  None.
* \retval int.
******************************************************************************/
int JsonRPC20Message::iGetID()
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    return m_RequestId;
}

/******************************************************************************
** FUNCTION   : isEqualVectors
*******************************************************************************
* \fn     JsonRPC20Message::isEqualVectors
* \brief  function to check vectors
* \param  vector.
* \retval bool.
******************************************************************************/
bool JsonRPC20Message::isEqualVectors(std::vector<std::string> DefinedV
                                      ,std::vector<std::string> ReceivedV)
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    std::sort(DefinedV.begin(), DefinedV.end());
    std::sort(ReceivedV.begin(), ReceivedV.end());

    for( std::vector<std::string>::iterator contactItr = DefinedV.begin();
         contactItr != DefinedV.end(); contactItr++ )
    {
        //DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Defined vector::value: "),DLT_STRING((*contactItr).c_str()));
    }
    for( std::vector<std::string>::iterator contactItr = ReceivedV.begin();
         contactItr != ReceivedV.end(); contactItr++ )
    {
        //DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("ReceivedV vector::value: "),DLT_STRING((*contactItr).c_str()));
    }
    if(DefinedV == ReceivedV)
    {
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Similar Vectors\n"));
        return true;
    }
    else{
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Different Vectors\n"));
        return false;
    }
}

/******************************************************************************
** FUNCTION   : eReturnJsonRPC_MessageType
*******************************************************************************
* \fn     JsonRPC20Message::eReturnJsonRPC_MessageType
* \brief  function to check vectors
* \param  Json::Value.
* \retval JsonRPC2_0Err.
******************************************************************************/
 JsonRPC2_0Err JsonRPC20Message::eReturnJsonRPC_MessageType(Json::Value jsonObject)
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    int size = jsonObject.size();
    DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_INT(jsonObject.size()));
    std::vector<std::string> Jsonkeys = jsonObject.getMemberNames();

    Json::Value::Members memberNames = jsonObject.getMemberNames();
    DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(jsonObject.toStyledString().c_str()));

    for(unsigned int i=0; i<memberNames.size(); ++i)
    {

        std::string memberName = memberNames[i];
        Json::Value value = jsonObject[memberName];
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(memberName.c_str()));
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(value.toStyledString().c_str()));
    }
	
    if(isEqualVectors(JSONRPC_RequestKeys,Jsonkeys)
            || isEqualVectors(JSONRPC_RequestKeys_NoParams,Jsonkeys) )
    {
        return RPCJSON20_REQUEST;
    }
    else if(isEqualVectors(JSONRPC_NotificationKeys,Jsonkeys)
            || isEqualVectors(JSONRPC_NotificationKeys_NoParams,Jsonkeys))
    {
        return RPCJSON20_NOTIFICATION;

    }else if(isEqualVectors(JSONRPC_RequestKeys,Jsonkeys))
    {
        return RPCJSON20_RESPONSE;

    }else if(isEqualVectors(JSONRPC_RequestKeys,Jsonkeys))
    {
        return RPCJSON20_RESPONSE;
    }
    else
    {
        return JSONRPC20_INVALID_REQUEST;
    }
}

/******************************************************************************
** FUNCTION   : validateJson_RPC20
*******************************************************************************
* \fn     JsonRPC20Message::validateJson_RPC20
* \brief  function to validate json RPC 2.0 messages
* \param  Json::Value.
* \retval JsonRPC2_0Err.
******************************************************************************/
JsonRPC2_0Err JsonRPC20Message::validateJson_RPC20(const char* jsonstring)
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    Json::Value Jsonvalue;
    JsonRPC2_0Err iErrorcode = NOERROR;
    if(bIsJson(jsonstring,Jsonvalue))
    {
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("It's a valid JSON"));
        //check for RPC 2.0 format
        if(!Jsonvalue.empty())
        {
            DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Verify for JSON RPC 2.0 format\n"));

            iErrorcode =bIsJson_RPC20(Jsonvalue);
            if(NOERROR != iErrorcode)
            {
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("It's not in JSON RPC 2.0 format"));
            }else
            {
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Valid JSON RPC 2.0 format"));
            }
        }
        else
        {
            DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Null json"));
            iErrorcode = JSONRPC20_INVALID_REQUEST;
        }
    }
    else
    {
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Not a json"));
        iErrorcode = JSONRPC20_INVALID_REQUEST;
    }
    return iErrorcode;
}

/******************************************************************************
** FUNCTION   : bIsJson
*******************************************************************************
* \fn     JsonRPC20Message::bIsJson
* \brief  function to check json
* \param  Json::Value.
* \retval bool.
******************************************************************************/
bool JsonRPC20Message::bIsJson(const char* jsonstring,Json::Value& value)
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    Json::Reader reader;
    bool parsedSuccess = reader.parse(jsonstring, value);
    if(!parsedSuccess)
    {
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("IN- Failed to parse: Not a JSON\n"));
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(reader.getFormattedErrorMessages().c_str()));
    }
    else{

        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Success:: It's valid JSON\n"));
    }
    return parsedSuccess;
}

/******************************************************************************
** FUNCTION   : bIsJson_RPC20
*******************************************************************************
* \fn     JsonRPC20Message::bIsJson_RPC20
* \brief  function to check json
* \param  Json::Value.
* \retval JsonRPC2_0Err.
******************************************************************************/
 JsonRPC2_0Err JsonRPC20Message::bIsJson_RPC20(Json::Value Jsonvalue)
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    JsonRPC2_0Err iErrorcode = NOERROR;

    if(JSONRPC20_INVALID_REQUEST == eReturnJsonRPC_MessageType(Jsonvalue))
    {
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("construct error RPC json"));
        //construct error RPC json
        iErrorcode =JSONRPC20_INVALID_REQUEST;
    }
    else{
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("validate the values"));
        Json::Value::Members memberNames = Jsonvalue.getMemberNames();
        for(unsigned int i=0; i<memberNames.size(); ++i)
        {

            std::string memberName = memberNames[i];
            Json::Value value = Jsonvalue[memberName];
            //  DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Key: %s\n",memberName.c_str()));
            // DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Value: %s\n",value.toStyledString().c_str()));

            if(memberName.compare(m_protocol) ==0)
            {
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Key is :: josnrpc\n"));
                Json::Value value = Jsonvalue[memberName];
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_INT(value.isString()));
                if(value.isString())
                {
                    std::string val = value.asString();
                    DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(val.c_str()));
                    if(val.compare(JSONRPC_VERSION)!=0)
                    {
                        iErrorcode = JSONRPC20_VERSION_MISTACH;
                    }
                }
                else{
                    iErrorcode = JSONRPC20_INVALID_PROTOCOL_TYPE;
                }
            }
            else if(memberName.compare(m_method) ==0)
            {
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Key is :: method\n"));
                Json::Value value = Jsonvalue[memberName];
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_INT(value.isString()));
                if(value.isString())
                {
                    std::string val  = value.asString();
                    DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(val.c_str()));
                    if(val.empty())
                    {
                        iErrorcode = JSONRPC20_METHOD_NOT_FOUND;
                    }
                }
                else
                {
                    iErrorcode =JSONRPC20_INVALID_METHOD_TYPE;
                }
            }
            else if(memberName.compare(m_id) ==0)
            {
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Key is :: id\n"));
                Json::Value value = Jsonvalue[memberName];
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_INT(value.type()));
                if(value.type()== Json::ValueType::intValue)
                {
                    DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("id is of int type"));
                }
                else if(value.type() == Json::ValueType::stringValue)
                {
                    std::string val = value.asString();
                    if(val.empty())
                    {
                        iErrorcode = JSONRPC20_INVALID_ID_TYPE;
                    }
                }
                else
                {
                    //with id= null;
                    iErrorcode = JSONRPC20_INVALID_ID_TYPE;
                }
            }
            else if(memberName.compare(m_params) ==0)
            {
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Key is :: params\n"));
                Json::Value value = Jsonvalue[memberName];
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_INT(value.type()));
                if(!(value.type()== Json::ValueType::arrayValue || value.type() == Json::ValueType::objectValue))
                {
                    iErrorcode = JSONRPC20_INVALID_PARAMS_TYPE;
                }
            }
            else if(memberName.compare(m_error) ==0)
            {
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Key is :: error\n"));
                Json::Value value = Jsonvalue[memberName];
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_INT(value.type()));
                if(value.type() != Json::ValueType::objectValue)
                {
                    iErrorcode = JSONRPC20_INVALID_ERROR_TYPE;
                }
            }
            else if(memberName.compare(m_result) ==0)
            {
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Key is :: result\n"));;
                Json::Value value = Jsonvalue[memberName];
                DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_INT(value.type()));
                //result can be any type;
                if(value.empty())
                {
                    iErrorcode = JSONRPC20_NOCONTENT;
                }
            }
            //check for iErrorcode
            if(NOERROR != iErrorcode)
                break;
        }
    }
    return iErrorcode;
}

/******************************************************************************
** FUNCTION   : strConstruct_ErrorResponse
*******************************************************************************
* \fn     JsonRPC20Message::strConstruct_ErrorResponse
* \brief  function to check json
* \param  iErrorCode.
* \retval JsonRPC20Message.
******************************************************************************/
const char* JsonRPC20Message::strConstruct_ErrorResponse(int iErrorCode,int id)
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    std::string jsonString = "";
    Json::FastWriter writer;
   if(NOERROR != iErrorCode)
    {
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("construct error response and send\n"));
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_INT(iErrorCode));
        Json::Value value, errorObject;
        errorObject["code"] = iErrorCode;
        errorObject["message"] = getStrFromErr(iErrorCode);
        value[m_protocol]=JSONRPC_VERSION;
        value[m_error] = errorObject;
        if(iErrorCode == JSONRPC20_INVALID_ID_TYPE)
        {
            DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("Send id as null otherwise as received\n"));
            value[m_id] = NULLVALUE;
        }
        else
        {
            DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("set received ID to error resposne\n"));
            value[m_id] = id;
        }
        jsonString = writer.write(value);
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(jsonString.c_str()));
    }
    else
    {
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("******NoError***********\n"));
    }
    return strdup(jsonString.c_str());
}

/******************************************************************************
** FUNCTION   : strConstruct_Response
*******************************************************************************
* \fn     JsonRPC20Message::strConstruct_Response
* \brief  function to check json
* \param  Json::Value.
* \retval JsonRPC20Message.
******************************************************************************/
const char* JsonRPC20Message::strConstruct_Response(Json::Value result,int id)
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    Json::Value value;
    Json::FastWriter writer;
    value[m_protocol]=JSONRPC_VERSION;
    value[m_result] = result;
    value[m_id] = id;
    std::string jsonStr = writer.write(value);
    return g_strdup(jsonStr.c_str());
}

/******************************************************************************
** FUNCTION   : strConstruct_JSONRPC20_request
*******************************************************************************
* \fn     JsonRPC20Message::strConstruct_JSONRPC20_request
* \brief  function to check json
* \param  None
* \retval const char*.
******************************************************************************/
const char* JsonRPC20Message::strConstruct_JSONRPC20_request()
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    Json::Value value;
    Json::FastWriter writer;
    value[m_protocol]=JSONRPC_VERSION;
    value[m_method] = strGetMethod();
    value[m_params] = jsonGetParams();
    value[m_id] = iGetID();
    std::string jsonStr = writer.write(value);
    DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(jsonStr.c_str()));

    return strdup(jsonStr.c_str());
}

/******************************************************************************
** FUNCTION   : vValidateResponse_constructed
*******************************************************************************
* \fn     JsonRPC20Message::vValidateResponse_constructed
* \brief  function to check json
* \param  None
* \retval None
******************************************************************************/
void JsonRPC20Message::vValidateResponse_constructed()
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
}

/******************************************************************************
** FUNCTION   : getStrFromErr
*******************************************************************************
* \fn     JsonRPC20Message::getStrFromErr
* \brief  function to check json
* \param  iErrorCode
* \retval char*
******************************************************************************/
char* JsonRPC20Message:: getStrFromErr(int iErrorCode)
{
    DLT_LOG(JSON_VAL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    char* ErrMessage =NULL;
    switch(iErrorCode)
    {

    case JSONRPC20_PARSE_ERROR:
        ErrMessage = (char*)"JSON Parse error";
		DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_INVALID_REQUEST:
        ErrMessage = (char*)"Invalid JSON object";
		DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;

    case JSONRPC20_METHOD_NOT_FOUND:
        ErrMessage = (char*)"Method not found";
		DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_INVALID_PARAMS:
		ErrMessage = (char*)"Invaild parameters";
		DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_INTERNAL_ERROR:
	    ErrMessage = (char*)"Internal JSON-RPC error";
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_SERVER_ERROR :
        ErrMessage = (char*)"Server Error";
		DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_TIMEOUT_ERROR:
		ErrMessage = (char*)"Method request timed out";
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));        
        break;
    case JSONRPC20_SERVICE_UNAVAILABLE:
        ErrMessage = (char*)"Service is unavailable";
		DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_SERVICE_ERROR:
        ErrMessage = (char*)"Service Error";
		DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_AUTHORIZATION_ERROR:
        ErrMessage = (char*)"Authorization error";
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_REQUEST_TOO_LONG:
        ErrMessage = (char*)"Request too long";
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_TOO_MANY_REQUESTS:
        ErrMessage = (char*)"Too many requests";
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_VERSION_MISTACH:
        ErrMessage = (char*)"Version mismatch";
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_STORAGE_ERROR:
        ErrMessage = (char*)"Storage error";
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_INVALID_TOKEN:
        ErrMessage =(char*)"Invalid token";
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    case JSONRPC20_NOCONTENT:
        ErrMessage =(char*)"No content";
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING(ErrMessage));
        break;
    default :
        //DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("%s\n",ErrMessage));
        DLT_LOG(JSON_VAL ,DLT_LOG_INFO,DLT_STRING("default"));
    }
    return ErrMessage;
}

