/*
Example code for using jsoncpp and gio to convert JSON<->GVariant

MIT License

Copyright (C) 2017 Jens Georg <mail@jensge.org>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/


#include "jsoncppgvariant.h"

GVariant *jsonToGVariant(Json::Value& value)
{
    switch (value.type()) {
    case Json::nullValue:
        return g_variant_new ("()", nullptr);
    case Json::intValue:
        if (value.isInt64()) {
            return g_variant_new_int64(value.asInt64());
        } else {
            return g_variant_new_int32(value.asInt());
        }
        break;
    case Json::uintValue:
        if(value.isUInt64()) {
            return g_variant_new_uint64(value.asUInt64());
        } else {
            return g_variant_new_uint32(value.asUInt());
        }
        break;
    case Json::realValue:
        return g_variant_new_double(value.asDouble());
    case Json::stringValue:
        // Safe to use, jsoncpp will return empty string if string is nullptr
        return g_variant_new_string(value.asString().c_str());
    case Json::booleanValue:
        return g_variant_new_boolean(value.asBool() ? TRUE : FALSE);
    case Json::arrayValue:
    {
        if (value.size() == 0) {
            return g_variant_new("as", nullptr);
        }

        std::list<GVariant*> values;
        std::list<const GVariantType*> types;
        for (auto &i: value) {
            auto variant = jsonToGVariant(i);
            values.push_back(variant);
            types.push_back(g_variant_get_type(variant));
        }
        //auto isTuple = std::find_if(types.begin(), types.end(), [&types] (auto const x) { return g_variant_type_equal (x, *(types.begin())) != TRUE; }) != types.end();
        auto isTuple = std::find_if(types.begin(), types.end(), [&types] (decltype(*types.begin()) x) { return g_variant_type_equal (x, *(types.begin())) != TRUE; }) != types.end();

        GVariantBuilder builder;
        g_variant_builder_init(&builder, isTuple ? G_VARIANT_TYPE_TUPLE : G_VARIANT_TYPE_ARRAY);
        //std::for_each(values.begin(), values.end(), [&builder] (auto x) { g_variant_builder_add_value(&builder, x); });
        std::for_each(values.begin(), values.end(), [&builder] (decltype(*values.begin()) x) { g_variant_builder_add_value(&builder, x); });
        return g_variant_builder_end(&builder);
    }
        break;
    case Json::objectValue:
    {
        if (value.size() == 0) {
            return g_variant_new("a{sv}", nullptr);
        }

        GVariantBuilder builder;
        g_variant_builder_init(&builder, G_VARIANT_TYPE_VARDICT);
        for (auto it = value.begin(); it != value.end(); ++it) {
            auto variant = jsonToGVariant(*it);
            if (variant != nullptr)
                g_variant_builder_add(&builder, "{sv}", it.key().asString().c_str(), variant);
        }
        return g_variant_builder_end(&builder);
    }
        break;
    default:
        break;
    }
}

Json::Value gvariantToJson(GVariant *variant)
{
    auto type = g_variant_get_type(variant);
    assert(g_variant_type_is_definite(g_variant_get_type(variant)) == TRUE);
    // Handle basic types
    if (g_variant_type_equal (type, G_VARIANT_TYPE_BOOLEAN)) {
        return Json::Value(g_variant_get_boolean(variant) == TRUE);
    } else if (g_variant_type_equal (type, G_VARIANT_TYPE_BYTE)) {
        return Json::Value(g_variant_get_byte(variant));
    } else if (g_variant_type_equal (type, G_VARIANT_TYPE_INT16)) {
        return Json::Value(g_variant_get_int16(variant));
    } else if (g_variant_type_equal(type, G_VARIANT_TYPE_UINT16)) {
        return Json::Value(g_variant_get_uint16(variant));
    } else if (g_variant_type_equal(type, G_VARIANT_TYPE_INT32)) {
        return Json::Value(g_variant_get_int32(variant));
    } else if (g_variant_type_equal(type, G_VARIANT_TYPE_UINT32)) {
        return Json::Value(g_variant_get_uint32(variant));
    } else if (g_variant_type_equal(type, G_VARIANT_TYPE_INT64)) {
        return Json::Value(Json::Value::Int64(g_variant_get_int64(variant)));
    } else if (g_variant_type_equal(type, G_VARIANT_TYPE_UINT64)) {
        return Json::Value(Json::Value::UInt64(g_variant_get_uint64(variant)));
    } else if (g_variant_type_equal(type, G_VARIANT_TYPE_HANDLE)) {
        return Json::Value(g_variant_get_handle(variant));
    } else if (g_variant_type_equal(type, G_VARIANT_TYPE_DOUBLE)) {
        return Json::Value(g_variant_get_double(variant));
    } else if (g_variant_type_equal(type, G_VARIANT_TYPE_STRING) ||
               g_variant_type_equal(type, G_VARIANT_TYPE_OBJECT_PATH) ||
               g_variant_type_equal(type, G_VARIANT_TYPE_SIGNATURE)) {
        return Json::Value(g_variant_get_string(variant, nullptr));
        // Let bytestring explicitly be handled as an array of bytes
    } else if (g_variant_type_equal(type, G_VARIANT_TYPE_VARIANT)) {
        auto internalVariant = g_variant_get_variant(variant);
        auto value = gvariantToJson(internalVariant);
        g_variant_unref(internalVariant);
        return value;
    } else if (g_variant_type_equal(type, G_VARIANT_TYPE_MAYBE)) {
        auto internalVariant = g_variant_get_maybe(variant);
        if (internalVariant == nullptr)
            return Json::Value();

        auto value = gvariantToJson(internalVariant);
        g_variant_unref(internalVariant);
    } else if (g_variant_type_is_subtype_of(type, G_VARIANT_TYPE_DICTIONARY)) {
        // Handle dictionary first, because a dictionary is also just an array
        Json::Value value{Json::objectValue};
        GVariantIter iter;
        GVariant *child;
        g_variant_iter_init(&iter, variant);
        while ((child = g_variant_iter_next_value(&iter)) != nullptr) {
            assert(g_variant_type_is_subtype_of(g_variant_get_type(child), G_VARIANT_TYPE_DICT_ENTRY));
            GVariantIter childIter;
            GVariant *dictEntry;

            g_variant_iter_init(&childIter, child);
            Json::Value dictEntryValues[2];
            int count = 0;
            while ((dictEntry = g_variant_iter_next_value(&childIter)) != nullptr) {
                Json::Value val;

                dictEntryValues[count] = gvariantToJson(dictEntry);
                count++;
                g_variant_unref(dictEntry);
            }
            value[dictEntryValues[0].asString()] = dictEntryValues[1];
            g_variant_unref(child);
        }

        return value;
    } else if (g_variant_type_is_subtype_of(type, G_VARIANT_TYPE_ARRAY) ||
               g_variant_type_is_subtype_of(type, G_VARIANT_TYPE_TUPLE)) {
        Json::Value value{Json::arrayValue};
        GVariantIter iter;
        GVariant *child;
        g_variant_iter_init(&iter, variant);
        while ((child = g_variant_iter_next_value(&iter)) != nullptr) {
            auto childValue = gvariantToJson(child);
            value.append(childValue);
            g_variant_unref(child);
        }
        return value;
    }
    else
    {
		return Json::Value();
    }
	
	return Json::Value();
}
