/*!
*******************************************************************************
* @file             : atl_if.h
*******************************************************************************
*  - PROJECT:       : IPCM
*  - SW-COMPONENT   : Abstract Transport Layer
*  - DESCRIPTION    : Provides abstraction over MQTT transport protocol
*  - COPYRIGHT      : &copy; 2017 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          | Version  | Modification
* ----------|---------------|----------|---------------------------------------
*28.09.2017 | Ashsih Kumar (RBEI/ECO2) | 0.0.1    | Initial version
******************************************************************************/
//HEADERS
#include "atl_if.h"
#include "atl.h"

//DLT CONTEXT DECLARATION
DLT_DECLARE_CONTEXT(ATL);

//Hash Map 
static GHashTable *HashMap;  

//Message ID counters
static tUInt Server_MessageID;
static tUInt Client_MessageID;

//Password
char* paswd_server = NULL;
char* paswd_client = NULL;

//LIB MOSQUITTO STRUCTURE 
struct mosquitto *structServer;
struct mosquitto *structClient;

//THREAD VARIABLE
pthread_t server_thread;
pthread_t client_thread;


/******************************************************************************
* Function:       iATL_Init
* Description:    To Initialise the ATL
* Parameters:     void
* Return:         gboolean
*****************************************************************************/
gboolean bFileCheck(const gchar *filename)
{
	gchar *contents;
	gsize len;
	GError *error = NULL;
	
	gboolean l_Check = g_file_get_contents(filename,&contents,&len,&error);
	if(!l_Check || len < MQTT_PASSWD_FILE_LEN)
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- File Content Error"),DLT_STRING(error->message));
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- File Content Length:"),DLT_INT(len));
		return FALSE;
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- File Content Verfied"));
	}
	return TRUE;
}


/******************************************************************************
* Function:       iATL_AppAvailableState
* Description:    iATL_AppAvailableState
* Parameters:     (tCString)
* Return:         gboolean
*****************************************************************************/
gboolean iATL_AppAvailableState(tCString sAppID)
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- iATL_AppAvailableState"));
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- AppID: = "),DLT_STRING(sAppID));
	if(g_hash_table_contains(HashMap,sAppID))
	{
		const char *sAppState =  (const char *)g_hash_table_lookup(HashMap,sAppID);
		if(!g_strcmp0(sAppState,"available"))
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- APP ID IS PRESENT"));
			return TRUE;
		}
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- APP ID PRESENT BUT state unavailable"));
		return FALSE;
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- APP ID NOT PRESENT IN HASH"));
		return FALSE;
	}
}


/******************************************************************************
* Function:       iATL_GetAppID
* Description:    iATL_GetAppID
* Parameters:     (tCString)
* Return:         tCString
*****************************************************************************/
tString iATL_GetAppID(tString messageTopic)
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- iATL_GetAppID"));
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- Message topics = "),
	DLT_STRING(messageTopic));
	
	//Topic Buffer creation
	tString sAppID;
	tString Topic_buffer = strdup(messageTopic);

	sAppID = strtok(Topic_buffer, "/");  
	sAppID = strtok(NULL, "/"); 
	sAppID = strtok(NULL, "/");

	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- Topics = "),
	DLT_STRING(messageTopic),
	DLT_STRING("AppID = "),
	DLT_STRING(sAppID));

	//Freeing Topic Buffer
	free(Topic_buffer);
	return strdup(sAppID);
}


/******************************************************************************
* Function:       iATL_RegisterConnState
* Description:    iATL_RegisterConnState
* Parameters:     (const char*,fpNotifyConnectionState ATLConnStateCB)
* Return:         ATL_ERR
*****************************************************************************/
ATL_ERR iATL_RegisterConnState(tCString peer,fpNotifyConnectionState ATLConnStateCB)
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RegisterATLConnState"));
	return ATL_ERR_FAIL;
}


/******************************************************************************
* Function:       Client_connect_cb
* Description:    Client_connect_cb
* Parameters:     (struct mosquitto *,void *,tInt)
* Return:         void
*****************************************************************************/
void Client_connect_cb(struct mosquitto *mosq, void *obj, tInt result)
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT connection callback"));
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT Conn status no = "),DLT_INT(result));	//aua8kor
	const char *Val = mosquitto_connack_string(result);
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT Conn status string = "),DLT_STRING(Val));
}


/******************************************************************************
* Function:       Client_message_cb
* Description:    Client_message_cb
* Parameters:     (struct mosquitto *,void *,const struct mosquitto_message *)
* Return:         void
*****************************************************************************/
void Client_message_cb(struct mosquitto *mosq,void *obj,
const struct mosquitto_message *message)
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT message callback"));

	//Messaage Data
	tString Client_ResponseTopic = message->topic;
	tInt Client_Payloadlen = message->payloadlen;
	tCString Client_Payload = (tCString) message->payload;

	//client message topic
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT message topics ="),
	DLT_STRING(Client_ResponseTopic));

	//client message payload length 
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT payload length="),
	DLT_INT(Client_Payloadlen));

	//client message payload data 
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT message payload ="),
	DLT_STRING(Client_Payload));

	//Fetch AppID
	tString sAppID = iATL_GetAppID(Client_ResponseTopic);
	
	if(iATL_AppAvailableState(sAppID))
	{
		//printf("\nASHISH APP ID = %s VALUE IS AVAILABLE\n\n",sAppID);
		//RPC client Callback
		Client_ResponseCB(sAppID,Client_Payload);
	}
	else
	{
		//printf("\nASHISH APP ID = %s VALUE IS UAVAILABLE\n",sAppID);
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- MESSGAE CANNOT BE SENT APPID NOT PRESENT"));
	}   
}


/******************************************************************************
* Function:       vpClientThreadRoutine
* Description:    vpClientThreadRoutine
* Parameters:     (void *)
* Return:         void
*****************************************************************************/
void *vpClientThreadRoutine(void *l_poClientThreadParams)
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT Thread Routine"));

	//Server mosquitto structure constructor 
	structClient = mosquitto_new(RPC_CLIENT,TRUE,NULL);

	if (structClient)
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT constructor Initialized"));
		
		//Username & Password Set for RPC Client
		if(paswd_client != NULL)
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT Passwd "),DLT_STRING(paswd_client));	
			int ClientRetVal =  mosquitto_username_pw_set(structClient,RPC_CLIENT,paswd_client);
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT Passwd Retun Value = "),DLT_INT(ClientRetVal));
			g_free(paswd_client); 
		}
		else
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT Passwd is NULL"));
			return NULL;
		}
		
		//Mosquitto MQTT callback functions
		mosquitto_connect_callback_set(structClient,Client_connect_cb);
		mosquitto_message_callback_set(structClient,Client_message_cb);
		
		//Connect to MQTT Broker
		tInt Client_conn = mosquitto_connect_async(structClient,
													MQTT_HOSTNAME,MQTT_PORT,10); 
		
		if(!Client_conn)
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT connected to MQTT broker"));
			
			//client Topic Subscribtion
			tInt Client_subs = mosquitto_subscribe(structClient,NULL,CLIENT_TOPIC_SUBSCRIBE_RESPONSE,QOS);   
			if (!Client_subs) 
			{
				DLT_LOG(ATL,DLT_LOG_INFO,
				DLT_STRING("ATL- RPC CLIENT Subscribed to topic = sp/ipcm/+/rpc/response"));
			}
			else
			{
				DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT Topic Subscription failed"),
				DLT_STRING("topic = sp/ipcm/+/rpc/response"));
			}
			//RPC client Connection mosquitto loop			
			Client_conn = mosquitto_loop_forever(structClient, -1, 1);
		}
		else 
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT connection failed with MQTT broker"));
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT connection Value = "),DLT_INT(Client_conn));
		}
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,
		DLT_STRING("ATL- RPC CLIENT constructor cannot be Initialized"));
	}
	return NULL;
}


/******************************************************************************
* Function:       iATL_Register_Client
* Description:    iATL_Register_Client
* Parameters:     void
* Return:         ATL_ERR
*****************************************************************************/
ATL_ERR iATL_Register_Client()
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT REGISTER"));

	tInt iClientThreadRet = pthread_create( &client_thread, 
	NULL,
	*vpClientThreadRoutine,
	NULL);
	
	if(iClientThreadRet)
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- pthread failed for RPC CLIENT"),
		DLT_INT(iClientThreadRet));
		return ATL_ERR_FAIL;
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- pthread created RPC CLIENT"),
		DLT_INT(iClientThreadRet));
		return ATL_ERR_SUCCESS;
	}   
}


/******************************************************************************
* Function:       Server_connect_cb
* Description:    Server_connect_cb
* Parameters:     (struct mosquitto *,void *,tInt)
* Return:         void
*****************************************************************************/
void Server_connect_cb(struct mosquitto *mosq, void *obj, tInt result)
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER connection callback"));
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER Conn status no = "),DLT_INT(result));	//aua8kor
	const char *Val = mosquitto_connack_string(result);
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER Conn status string = "),DLT_STRING(Val));
}


/******************************************************************************
* Function:       Server_message_cb
* Description:    Server_message_cb
* Parameters:     (struct mosquitto *,void *,const struct mosquitto_message *)
* Return:         void
*****************************************************************************/
void Server_message_cb(struct mosquitto *mosq,void *obj,
						const struct mosquitto_message *message)
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER message callback"));

	bool match = FALSE;
	mosquitto_topic_matches_sub(SERVER_TOPIC_SUBSCRIBE_APPSTATE,message->topic,&match);

	//Messaage Data
	tString Server_RequestTopic = message->topic;
	tInt Server_Payloadlen = message->payloadlen;
	tCString Server_Payload = (tCString) message->payload;

	//Server message topic
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER message topics = "),
	DLT_STRING(Server_RequestTopic));

	//Server message payload length 
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER payload length= "),
	DLT_INT(Server_Payloadlen));

	//Server message payload data 
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER message payload = "),
	DLT_STRING(Server_Payload));

	//APP State
	if (match)
	{
		//Payload Buffer creation
		tString payload_buffer = strdup(Server_Payload);
		tString sAppID = strtok(payload_buffer,".");     
		tString sAppState = strtok(NULL,".");         

		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- AppID = "),DLT_STRING(sAppID));
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- AppState = "),DLT_STRING(sAppState));
		
		gboolean l_bResult = g_hash_table_insert(HashMap,strdup(sAppID),strdup(sAppState));
		if(!l_bResult)
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- AppID cannot be Insterted"),DLT_STRING(sAppID));
            iATL_AppAvailableState(sAppID);
        }
		
        //Freeing Payload Buffer
        free(payload_buffer);
        payload_buffer = NULL;
	}
	else
	{
		//Fetch AppID
		tString sAppID = iATL_GetAppID(Server_RequestTopic);

		if(iATL_AppAvailableState(sAppID))
		{
			//RPC Server Callback
			Server_ResponseCB(sAppID,Server_Payload);
		}
		else
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- MESSGAE CANNOT BE SENT APPID NOT PRESENT"));
		}
	}
}


/******************************************************************************
* Function:       vpServerThreadRoutine
* Description:    vpServerThreadRoutine
* Parameters:     (void *)
* Return:         void
*****************************************************************************/
void *vpServerThreadRoutine(void *l_poServerThreadParams)
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER Thread Routine"));

	//Srever mosquitto structure constructor 
	structServer = mosquitto_new(RPC_SERVER,TRUE,NULL);  

	if (structServer)
	{
		DLT_LOG(ATL,DLT_LOG_INFO,
		DLT_STRING("ATL- RPC SERVER constructor Initialized"));
		
		//Username & Password Set for RPC Server
		if(paswd_server != NULL)
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER Passwd "),DLT_STRING(paswd_server));
			int ServerRetVal = mosquitto_username_pw_set(structServer,RPC_SERVER,paswd_server);
			//int ServerRetVal = mosquitto_username_pw_set(structServer,RPC_SERVER,"RzVpczA3Z2xPdwo=");	//aua8kor
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER Passwd Retun Value = "),DLT_INT(ServerRetVal));		
			g_free(paswd_server);   
		}
		else
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER Passwd is NULL"));
			return NULL;
		}
		
		//Mosquitto MQTT callback functions
		mosquitto_connect_callback_set(structServer,Server_connect_cb);
		mosquitto_message_callback_set(structServer,Server_message_cb);

		//Connect to MQTT Broker
		tInt Server_conn = mosquitto_connect_async(structServer,
													MQTT_HOSTNAME,MQTT_PORT,10); 
		
		if(!Server_conn)
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER connected to MQTT broker"));
			
			//Server Topic Subscribtion
			tInt Server_subs = mosquitto_subscribe(structServer,NULL,SERVER_TOPIC_SUBSCRIBE_REQUEST,QOS);   
			if (!Server_subs) 
			{
				DLT_LOG(ATL,DLT_LOG_INFO,
				DLT_STRING("ATL- RPC SERVER Subscribed to topic = ipcm/sp/+/rpc/request"));
			}
			else
			{
				DLT_LOG(ATL,DLT_LOG_INFO,
				DLT_STRING("ATL- RPC SERVER Topic Subscription failed"),
				DLT_STRING("topic = ipcm/sp/+/rpc/request"));
			}
			tInt AppState_subs = mosquitto_subscribe(structServer,NULL,SERVER_TOPIC_SUBSCRIBE_APPSTATE,QOS);   
			if (!AppState_subs) 
			{
				DLT_LOG(ATL,DLT_LOG_INFO,
				DLT_STRING("ATL- RPC SERVER Subscribed to topic = ipcm/sp/+/state"));
			}
			else
			{
				DLT_LOG(ATL,DLT_LOG_INFO,
				DLT_STRING("ATL- RPC SERVER Topic Subscription failed"),
				DLT_STRING("topic = ipcm/sp/+/state"));
			}
			//RPC Server Connection mosquitto loop
			Server_conn = mosquitto_loop_forever(structServer, -1, 1);
		}
		else 
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER connection failed with MQTT broker"));
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER connection Value = "),DLT_INT(Server_conn));
		}
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,
		DLT_STRING("ATL- RPC SERVER structure constructor cannot be Initialized"));
	}
	return NULL;
}


/******************************************************************************
* Function:       iATL_Register_Server
* Description:    iATL_Register_Server
* Parameters:     void
* Return:         ATL_ERR
*****************************************************************************/
ATL_ERR iATL_Register_Server()
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER REGISTER"));

	tInt iServerThreadRet = pthread_create( &server_thread, 
											NULL,
											*vpServerThreadRoutine,
											NULL);
	
	if(iServerThreadRet)
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- pthread failed for RPC SERVER"),
		DLT_INT(iServerThreadRet));
		return ATL_ERR_FAIL;
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- pthread creation RPC SERVER"),
		DLT_INT(iServerThreadRet));
		return ATL_ERR_SUCCESS;
	}
}


/******************************************************************************
* Function:       iATL_Register
* Description:    iATL_Register
* Parameters:     (const char*,fpHandleResponse)
* Return:         ATL_ERR
*****************************************************************************/
ATL_ERR iATL_Register(tCString peer,fpHandleResponse responseCB)
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- iATL_Register"));

	if(!strcmp(RPC_SERVER,peer))
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- PEER IS SERVER"));
		Server_ResponseCB = responseCB;
		return iATL_Register_Server();
	}
	else if(!strcmp(RPC_CLIENT,peer))
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- PEER IS CLIENT"));
		Client_ResponseCB = responseCB;
		return iATL_Register_Client();
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- wrong peer"));
		return ATL_ERR_FAIL;
	}
}


/******************************************************************************
* Function:       iATL_DeRegister
* Description     iATL_DeRegister
* Parameters      (tCString peer)
* Return          ATL_ERR
*****************************************************************************/
ATL_ERR iATL_DeRegister(tCString peer)
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- DeRegister"));   

	if(!strcmp(RPC_SERVER,peer))
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- PEER IS SERVER"));
		tInt ipthread_cancel = pthread_cancel(server_thread);
		server_thread = 0;
		Server_MessageID = 101;
		//Destroy the server Sructure 
		mosquitto_destroy(structServer);
		structServer = NULL;
		return ipthread_cancel;    // return to be made independent of pThread and should use enum
		//return ATL_ERR_SUCCESS;    // return to be made independent of pThread and should use enum
	}
	else if(!strcmp(RPC_CLIENT,peer))
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- PEER IS CLIENT"));
		tInt ipthread_cancel = pthread_cancel(client_thread);
		client_thread = 0;
		Client_MessageID = 201;
		//Destroy the client Sructure 
		mosquitto_destroy(structClient);
		structClient = NULL; 
		return ipthread_cancel;       // return to be made independent od pThread and should use enum
		//return ATL_ERR_SUCCESS;       // return to be made independent od pThread and should use enum
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- wrong peer"));
		return ATL_ERR_FAIL;
	}
}


/*****************************************************************************
* Function:       iATL_Publish_Client
* Description:    iATL_Publish_Client
* Parameters:     (const char*,const char*, eTopicType)
* Return:         ATL_ERR
*****************************************************************************/
ATL_ERR iATL_Publish_Client( tCString   appId,
                             tCString   payload,
                             eTopicType topicType )
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT publish"));
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT APPID"),
	DLT_STRING(appId));
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT Payload"),
	DLT_STRING(payload));
        DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
        DLT_STRING("(): Topic type is"), DLT_UINT(topicType));

	//aua8kor START new logic
	int retFail = 0;
	GList* Keys = NULL;
	Keys = g_hash_table_get_keys(HashMap);
	guint sizeHash = g_hash_table_size(HashMap);
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT HASH SIZE ="),
	DLT_INT(sizeHash));

        tInt Client_Payloadlen = strlen(payload);

        switch(topicType)
        {
           case IPCM_TELEMETRY_TOPIC:
           {
               DLT_LOG(ATL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
                                    DLT_STRING("(): Telemetry topic"));
            
               if(!mosquitto_publish(structClient,
                                &Client_MessageID,
                                CLIENT_TOPIC_PUBLISH_TELEMETRY,
                                Client_Payloadlen,
                                payload,
                                QOS,
                                FALSE))
               {
                   DLT_LOG(ATL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), DLT_STRING(
                                           "(): mosquitto_publish success"));
                   Client_MessageID++; 
                   return ATL_ERR_SUCCESS;  
               } 
               else
               {
                   DLT_LOG(ATL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), DLT_STRING(
                                       "():mosquitto_publish failed"));
                   return ATL_ERR_FAIL; 
               }
           }
           break;

           case IPCM_DEFAULT_TOPIC:
           {
 	       if(Keys != NULL && sizeHash > 0)		
	       {
	 	  GList* TempKeys = Keys;
		  DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- APPIDs is Available in hash"));
		
		  while(TempKeys != NULL)
		  {
			char* currentAppID = NULL;
			currentAppID = g_strdup((const char*)TempKeys->data);
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT publish AppID="),DLT_STRING(currentAppID));
			
			const char *sAppState =  (const char *)g_hash_table_lookup(HashMap,currentAppID);
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT AppID Sate="),DLT_STRING(sAppState));
			
			if(!g_strcmp0(sAppState,"available"))
			{
				//Topic Size
				tInt sClientTopic_Size = strlen(CLIENT_TOPIC_PUBLISH_REQUEST_P1) + 
				strlen(CLIENT_TOPIC_PUBLISH_REQUEST_P2) + strlen(currentAppID);
				DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT TOPIC SIZE ="),
				DLT_INT(sClientTopic_Size));
				sClientTopic_Size++;
				
				//Topic creation
				tString sClientTopic = malloc(sClientTopic_Size);	//coverity warnings
				memset(sClientTopic,0,sizeof(sClientTopic_Size));

				strcpy(sClientTopic,CLIENT_TOPIC_PUBLISH_REQUEST_P1);
				strcat(sClientTopic,currentAppID);
				strcat(sClientTopic,CLIENT_TOPIC_PUBLISH_REQUEST_P2);
				DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT publish TOPIC ="),
				DLT_STRING(sClientTopic));
				
				//Payload Size
				tInt Client_Payloadlen = strlen(payload);
				DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT PAYLOAD SIZE ="),
				DLT_INT(Client_Payloadlen));
				
				//Publish message to MQTT broker
				tInt PublishRet = mosquitto_publish(structClient,
				&Client_MessageID,
				sClientTopic,
				Client_Payloadlen,
				payload,
				QOS,
				FALSE);

				if(!PublishRet)
				{
					DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT message published"));
					Client_MessageID++;
					
					//Freeing Topic Buffer
					free(sClientTopic);
					free(currentAppID);
				}
				else
				{
					DLT_LOG(ATL,DLT_LOG_INFO,
					DLT_STRING("ATL- RPC CLIENT message publishing Failed"));
					retFail++;
				}  			
			}
			TempKeys=TempKeys->next;  
		  }
		  g_list_free(Keys);
		  if(retFail)
			return ATL_ERR_SUCCESS;
		  else
			return ATL_ERR_FAIL;
	    }
   	    else
	    {
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- KEY HASH IS NULL"));
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- APPIDs not Available in hash"));
		return ATL_ERR_FAIL;
	    }
         } //case IPCM_DEFAULT_TOPIC
         break;
         default: 
             DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                        DLT_STRING("(): Unknown topic type"));
             return ATL_ERR_FAIL;
     } // switch (topicType) 
}


/*****************************************************************************
* Function:       iATL_Publish_Server
* Description:    iATL_Publish_Server
* Parameters:     (const char*,const char*)
* Return:         ATL_ERR
*****************************************************************************/
ATL_ERR iATL_Publish_Server(tCString appId, tCString payload, eTopicType topicType)
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER publish"));
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER APPID"),DLT_STRING(appId));
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER Payload"),
	DLT_STRING(payload));

	if(!iATL_AppAvailableState(appId))
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- APP ID IS PRESENT"));
		return ATL_ERR_FAIL;
	}
	
	//Topic Size
	tInt sServerTopic_Size = strlen(SERVER_TOPIC_PUBLISH_RESPONSE_P1) + 
	strlen(SERVER_TOPIC_PUBLISH_RESPONSE_P2)+ strlen(appId);
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER TOPIC SIZE ="),
	DLT_INT(sServerTopic_Size));
	sServerTopic_Size++;

	//Topic creation
	tString sServerTopic = malloc(sServerTopic_Size);//coverity warnings
	memset(sServerTopic, 0, sizeof(sServerTopic_Size));

	strcpy(sServerTopic,SERVER_TOPIC_PUBLISH_RESPONSE_P1);
	strcat(sServerTopic,appId);
	strcat(sServerTopic,SERVER_TOPIC_PUBLISH_RESPONSE_P2);
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER publish TOPIC ="),
	DLT_STRING(sServerTopic));

	//Payload
	tInt Server_Payloadlen = strlen(payload);
	
	//Publish message to MQTT broker
	tInt PublishRet = mosquitto_publish(structServer,
	&Server_MessageID,
	sServerTopic,
	Server_Payloadlen,
	payload,
	QOS,
	FALSE);

	//Freeing Topic Buffer
	free(sServerTopic);

	if(!PublishRet)
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER message published"));
		Server_MessageID++;
		return ATL_ERR_SUCCESS;
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,
		DLT_STRING("ATL- RPC SERVER message publishing Failed"));
		return ATL_ERR_FAIL;
	}
}


/******************************************************************************
* Function:       iATL_SendMessage
* Description     iATL_SendMessage
* Parameters      (const char*,char*,const char, eTopicType*)
* Return          ATL_ERR
*****************************************************************************/
ATL_ERR iATL_SendMessage( tCString   peer,
                          tCString   appId,
                          tCString   payload, 
                          eTopicType topicType
                        )
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- ATL_SendMessage"));
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- Payload: "),DLT_STRING(payload));
	
	if(!strcmp(RPC_SERVER,peer))
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- PEER IS SERVER"));
		return iATL_Publish_Server(appId, payload, topicType);    
	}
	else if(!strcmp(RPC_CLIENT,peer))
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- PEER IS CLIENT"));
		return iATL_Publish_Client(appId, payload, topicType);   
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- wrong peer"));
		return ATL_ERR_FAIL;
	}

}


/******************************************************************************
* Function:       iATL_Init
* Description:    To Initialise the ATL
* Parameters:     void
* Return:         gboolean
*****************************************************************************/
gboolean iATL_Init()
{
	DLT_REGISTER_CONTEXT(ATL,"ATLL","ATL IPCM STATIC LIB");
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- iATL_Init"));
	
	//mosquitto structures
	structServer = NULL;
	structClient = NULL; 

	//pThread variables
	server_thread = 0;
	client_thread = 0;

	//Message ID variables
	Server_MessageID = 101;
	Client_MessageID = 201;

	//AppID List
	HashMap = g_hash_table_new_full(g_str_hash,g_str_equal,g_free,g_free);
	//g_hash_table_insert(HashMap,"DEFAULT_APPID","available");
	
	//TBImp
	//tInt mosquitto_lib_version(tInt* major,tInt* minor,tInt* revision)

	//mosquitto library Initialise
	mosquitto_lib_init();	

	//File Check
	if(!bFileCheck(MQTT_PASSWD_FILE_PATH))
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- MQTT Passwd file Check failed"));
		return FALSE;
	}
	
	//Fetching Password
	FILE* file = fopen(MQTT_PASSWD_FILE_PATH,"r");
	if(file != NULL)
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- MQTT Passwd file opened"));
		
		//Server password
		char tLine [128] = "";
		char temp[25] = "";
		char* str = NULL;
		
		fgets(tLine,sizeof tLine,file);
		str = strtok(tLine, ":");  
		str = strtok(NULL, ":");
		if(str)
		{
			strncpy(temp,str,strlen(str)-1);
		}
		else
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- Server password parsed NULL"));
		}
		paswd_server = strdup(temp);
		
		int rpcServer =0;
		if(paswd_server)
		{
			rpcServer = strlen(paswd_server);
		}
		else
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- Server password not populated"));
		}
		
		//Client password
		str = NULL;
		fgets(tLine,sizeof tLine,file);
		str = strtok(tLine, ":");  
		str = strtok(NULL, ":");
		if(str)
		{
			strncpy(temp,str,strlen(str) -1);
		}
		else
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- Client password parsed NULL"));
		}		
		paswd_client = strdup(temp);
		
		int rpcClient =0;
		if(paswd_client)
		{
			rpcClient = strlen(paswd_client);
		}
		else
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- Client password not populated"));
		}
			
		if(paswd_client == NULL || paswd_server == NULL)
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- Server or client password NULL"));
			int fRet = fclose(file);
			if(fRet < 0)
			{
				DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- File close error in LINE 816"));
				DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- fclose return value: "),DLT_INT(fRet)); 
				DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- fclose error string: "),DLT_STRING(strerror(errno)));
			}
			else
			{
				DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- File close as passwordnot fetched"));
			}
			return FALSE;			
		}		
		else
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- Server and client passwords NOT NULL"));
		}
		
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER Passwd "),DLT_STRING(paswd_server));
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER Passwd LEN "),DLT_INT(rpcServer)); 
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT Passwd "),DLT_STRING(paswd_client));
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT Passwd LEN "),DLT_INT(rpcClient)); 
		
		//File Close
		int fRet = fclose(file);
		if(fRet < 0)
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- File close error"));
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- fclose return value: "),DLT_INT(fRet)); 
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- fclose error string: "),DLT_STRING(strerror(errno))); 
			return FALSE;
		}
		else
		{
			DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- File close success"));
		}
		return TRUE;
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- MQTT Passwd file not opened"));
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING(MQTT_PASSWD_FILE_PATH));
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- File Error string no: "),DLT_STRING(strerror(errno))); 
		return FALSE;
	} 
}


/******************************************************************************
* Function:       iATL_Stop
* Description:    To Stop the ATL
* Parameters:     void
* Return:         ATL_ERR
*****************************************************************************/
ATL_ERR iATL_Stop()
{
	DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- iATL_Stop"));

	if(!iATL_DeRegister(RPC_SERVER))
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER DE-REGISTERED done"));
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC SERVER DE-REGISTERED failed"));
	}
	if(!iATL_DeRegister(RPC_CLIENT))
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT DE-REGISTERED done"));
	}
	else
	{
		DLT_LOG(ATL,DLT_LOG_INFO,DLT_STRING("ATL- RPC CLIENT DE-REGISTERED failed"));
	}

	//mosquitto library cleanup
	mosquitto_lib_cleanup();

	return ATL_ERR_SUCCESS;

	/*if( tInt iRet )
	return ATL_ERR_SUCCESS;
else
	return ATL_ERR_FAIL;*/
}
