/*
 * Copyright (C) 2013 Jörn Magens
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This Program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file LICENSE.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth 
 * Floor, Boston, MA  02110-1301  USA
 * https://www.gnu.org/licenses/lgpl-2.1.txt
 *
 * Author:
 * 	Jörn Magens <shuerhaaken@googlemail.com>
 * 	Pavel Vasin <rat4vier@gmail.com>
 */


#include "taginfo.h"
#include "taginfo_internal.h"



using namespace TagInfo;



// SpeexInfo

SpeexInfo::SpeexInfo(const String &filename) : Info(filename) {
    if(!file_name.isEmpty() && !create_file_ref())
        printf("Error creating file ref! %s\n", filename.toCString(false));
    if(taglib_file) {// && !taglib_file->isNull()) 
        xiphcomment = ((TagLib::Ogg::Speex::File *) taglib_file)->tag();
        if(!xiphcomment || xiphcomment->isEmpty()) {
            taglib_tag = ((TagLib::Ogg::Speex::File *) taglib_file)->tag();
            if(!taglib_tag) {
                printf("Cant get tag object from '%s'\n", file_name.toCString(false));
                valid = false;
            }
        }
    }
    else {
        printf("Cant get xiphcomment from '%s'\n", file_name.toCString(false));
        xiphcomment = NULL;
        valid = false;
    }
}


SpeexInfo::~SpeexInfo() {
}


bool SpeexInfo::create_file_ref() {
    if(file_name.isEmpty())
        return false;
    taglib_file = new TagLib::Ogg::Speex::File(file_name.toCString(false), true, TagLib::AudioProperties::Fast);
    if(taglib_file) {
        return true;
    }
    else {
        printf("TagLib::File creation failed for '%s'\n", file_name.toCString(false));
        return false;
    }
}


bool SpeexInfo::read(void) {
    if(Info::read()) {
        if(xiphcomment && !xiphcomment->isEmpty()) {
            if(!taglib_tag)
                read_virtual_tags((TagLib::Tag *)xiphcomment);
            
            if(xiphcomment->fieldListMap().contains("COMPOSER")) {
                composer = xiphcomment->fieldListMap()["COMPOSER"].front();
            }
            if(xiphcomment->fieldListMap().contains("DISCNUMBER")) {
                disk_number = atol(xiphcomment->fieldListMap()["DISCNUMBER"].front().toCString(false));
                if(disk_number < 1)
                    disk_number = 1;
            }
            if(xiphcomment->fieldListMap().contains("COMPILATION")) {
                is_compilation = xiphcomment->fieldListMap()["COMPILATION"].front() == String("1");
            }
            if(xiphcomment->fieldListMap().contains("ALBUMARTIST")) {
                album_artist = xiphcomment->fieldListMap()["ALBUMARTIST"].front();
            }
            else if(xiphcomment->fieldListMap().contains("ALBUM ARTIST")) {
                album_artist = xiphcomment->fieldListMap()["ALBUM ARTIST"].front();
            }
            // Rating
            if(xiphcomment->fieldListMap().contains("RATING")) {
                long rat = 0;
                rat = atol(xiphcomment->fieldListMap()["RATING"].front().toCString(true));
                if(rat) {
                    if(rat > 5) {
                        rating = popularity_to_rating(rat);
                    }
                    else {
                        rating = rat;
                    }
                }
            }
            if(xiphcomment->fieldListMap().contains("PLAY_COUNTER")) {
                long PlayCount = 0;
                PlayCount = atol(xiphcomment->fieldListMap()["PLAY_COUNTER"].front().toCString(true));
                playcount = PlayCount;
            }
            // Labels
            if(track_labels.size() == 0) {
                if(xiphcomment->fieldListMap().contains("TRACK_LABELS")) {
                    track_labels_string = xiphcomment->fieldListMap()["TRACK_LABELS"].front();
                    track_labels = split(track_labels_string, "|");
                }
            }
            if(artist_labels.size() == 0) {
                if(xiphcomment->fieldListMap().contains("ARTIST_LABELS")) {
                    artist_labels_string = xiphcomment->fieldListMap()["ARTIST_LABELS"].front();
                    artist_labels = split(artist_labels_string, "|");
                }
            }
            if(album_labels.size() == 0) {
                if(xiphcomment->fieldListMap().contains("ALBUM_LABELS")) {
                    album_labels_string = xiphcomment->fieldListMap()["ALBUM_LABELS"].front();
                    album_labels = split(album_labels_string, "|");
                }
            }
            if(xiphcomment->contains("COVERART")) // TODO
                has_image = true;
            return true;
        }
    }
    return false;
}


bool SpeexInfo::write(void) {
    if(xiphcomment) {
        if(changedflag) {
            if(changedflag & CHANGED_DATA_DISK_NUM)
                xiphcomment->addField("DISCNUMBER", format("%u", disk_number));
            
            if(changedflag & CHANGED_COMPOSER_TAG)
                xiphcomment->addField("COMPOSER", composer);
            
            if(changedflag & CHANGED_IS_COMPILATION_TAG) {
                if(is_compilation) {
                    xiphcomment->addField("COMPILATION", "1");
                }
                else {
                    xiphcomment->addField("COMPILATION", "0");
                }
            }
            
            if(changedflag & CHANGED_DATA_ALBUMARTIST)
                xiphcomment->addField("ALBUMARTIST", album_artist);
            
            if(changedflag & CHANGED_DATA_RATING) {
                xiphcomment->addField("RATING", format("%u", rating_to_popularity(rating)));
            }
            if(changedflag & CHANGED_DATA_PLAYCOUNT) {
                xiphcomment->addField("PLAY_COUNTER", format("%u", playcount));
            }
            // The Labels
            if(changedflag & CHANGED_TRACK_LABELS)
                check_xiph_label_frame(xiphcomment, "TRACK_LABELS", track_labels_string);
            if(changedflag & CHANGED_ARTIST_LABELS)
                check_xiph_label_frame(xiphcomment, "ARTIST_LABELS", artist_labels_string);
            if(changedflag & CHANGED_ALBUM_LABELS)
                check_xiph_label_frame(xiphcomment, "ALBUM_LABELS", album_labels_string);
            
            write_virtual_tags((TagLib::Tag *)xiphcomment);
        }
    }
    return Info::write();
}

bool SpeexInfo::can_handle_images(void) const {
    return true;
}

bool SpeexInfo::get_image(char*& data, int &data_length, ImageType &image_type) const {
    data = NULL;
    data_length = 0;
    return get_xiph_comment_cover_art(xiphcomment, data, data_length, image_type);
}

bool SpeexInfo::set_image(const char* data, int data_length, ImageType image_type) {
    return set_xiph_comment_cover_art(xiphcomment, data, data_length, image_type);
}

bool SpeexInfo::can_handle_lyrics(void) const {
    return true;
}

String SpeexInfo::get_lyrics(void) const {
    return get_xiph_comment_lyrics(xiphcomment);
}

bool SpeexInfo::set_lyrics(const String &lyrics) {
    return set_xiph_comment_lyrics(xiphcomment, lyrics);
}

