/*
 * Copyright (C) 2013 Jörn Magens
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This Program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file LICENSE.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth 
 * Floor, Boston, MA  02110-1301  USA
 * https://www.gnu.org/licenses/lgpl-2.1.txt
 *
 * Author:
 * 	Jörn Magens <shuerhaaken@googlemail.com>
 * 	Matias De lellis <mati86dl@gmail.com>
 * 	Pavel Vasin <rat4vier@gmail.com>
 */


#include "taginfo.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "taginfo_c.h"


using namespace TagInfo;
using namespace TagLib;


//*****************STRING MANAGEMENT*****************************************************
static List<char *> strings;
static bool stringManagementEnabled = false;

void taginfo_info_enable_string_management(BOOL management) {
    stringManagementEnabled = bool(management);
}

void taginfo_info_free_strings() {
    if(!stringManagementEnabled)
        return;

    for(List<char *>::Iterator it = strings.begin(); it != strings.end(); ++it)
        free(*it);
    strings.clear();
}

static char **string_list_to_c_array(const StringList &list, int *out_length) {
    if(list.isEmpty()) {
        //cout << "list not there"  << endl;
        return NULL;
    }
    uint len = list.size();
    char **array = (char**)malloc(len * sizeof(char*));
    int j = 0;
    for(StringList::ConstIterator it = list.begin(); it != list.end(); ++it) {
        array[j] = ::strdup((*it).toCString(false));
        if(stringManagementEnabled && array[j])
            strings.prepend(array[j]);
        j++;
    }
    *out_length = len;
    return array;
}

static StringList string_list_from_c_array(const char* const* array, int length) {
    StringList list;
    for(int j = 0; j < length; j++) {
        if(array[j]) {
            list.append(array[j]);
        }
    }
    return list;
}
//*****************/STRING MANAGEMENT****************************************************


//*****************FACTORY CREATION******************************************************
TagInfo_Info * taginfo_info_factory_make(const char *filename) {
    return reinterpret_cast<TagInfo_Info *>(Info::create_tag_info(filename));
}

TagInfo_Info * taginfo_info_factory_make_with_format(const char *filename,
                                                     TagInfo_MediaFileType media_file_type) {
    return reinterpret_cast<TagInfo_Info *>(Info::create_tag_info_with_format(filename, 
                                                                              (MediaFileType)media_file_type)
    );
}

TagInfo_Info *taginfo_info_factory_make_from_mime(const char *filename, const char *mime_type) {
    return reinterpret_cast<TagInfo_Info *>(Info::create_tag_info_from_mime(filename, 
                                                                            mime_type)
    );
}
//*****************/FACTORY CREATION*****************************************************


//*****************GENERAL MEMORY MANAGEMENT*********************************************
void taginfo_info_free(TagInfo_Info *info) {
    delete reinterpret_cast<Info *>(info);
}
//*****************/GENERAL MEMORY MANAGEMENT********************************************


//*****************TAG READ/WRITE API****************************************************
BOOL taginfo_info_read(TagInfo_Info *info) {
    Info *i = reinterpret_cast<Info *>(info);
    return i->read();
}
BOOL taginfo_info_write(TagInfo_Info *info) {
    Info *i = reinterpret_cast<Info *>(info);
    return i->write();
}

char * taginfo_info_get_artist(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    String artist = i->get_artist();
    char *s = ::strdup(artist.toCString(true));
    if(stringManagementEnabled && s)
        strings.prepend(s);
    return s;
}
void taginfo_info_set_artist(TagInfo_Info *info, const char *artist) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_artist(String(artist, String::UTF8));
}

char * taginfo_info_get_album(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    String album = i->get_album();
    char *s = ::strdup(album.toCString(true));
    if(stringManagementEnabled && s)
        strings.prepend(s);
    return s;
}
void taginfo_info_set_album(TagInfo_Info *info, const char *album) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_album(String(album, String::UTF8));
}

char * taginfo_info_get_title(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    String title = i->get_title();
    char *s = ::strdup(title.toCString(true));
    if(stringManagementEnabled && s)
        strings.prepend(s);
    return s;
}
void taginfo_info_set_title(TagInfo_Info *info, const char *title) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_title(String(title, String::UTF8));
}

char * taginfo_info_get_albumartist(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    String album_artist = i->get_album_artist();
    char *s = ::strdup(album_artist.toCString(true));
    if(stringManagementEnabled && s)
        strings.prepend(s);
    return s;
}
void taginfo_info_set_albumartist(TagInfo_Info *info, const char *albumartist) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_album_artist(String(albumartist, String::UTF8));
}

char * taginfo_info_get_comment(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    String comments = i->get_comments();
    char *s = ::strdup(comments.toCString(true));
    if(stringManagementEnabled && s)
        strings.prepend(s);
    return s;
}
void taginfo_info_set_comment(TagInfo_Info *info, const char *comment) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_comments(String(comment, String::UTF8));
}

char * taginfo_info_get_composer(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    String composer = i->get_composer();
    char *s = ::strdup(composer.toCString(true));
    if(stringManagementEnabled && s)
        strings.prepend(s);
    return s;
}
void taginfo_info_set_taginfo_info_set_composer(TagInfo_Info *info, const char *composer) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_composer(String(composer, String::UTF8));
}

char * taginfo_info_get_genre(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    String genre = i->get_genre();
    char *s = ::strdup(genre.toCString(true));
    if(stringManagementEnabled && s)
        strings.prepend(s);
    return s;
}
void taginfo_info_set_genre(TagInfo_Info *info, const char *genre) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_genre(String(genre, String::UTF8));
}

int taginfo_info_get_tracknumber(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    return i->get_tracknumber();
}
void taginfo_info_set_tracknumber(TagInfo_Info *info, int tracknumber) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_tracknumber(tracknumber);
}

int taginfo_info_get_year(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    return i->get_year();
}
void taginfo_info_set_year(TagInfo_Info *info, int year) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_year(year);
}

int taginfo_info_get_bitrate(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    return i->get_bitrate();
}

int taginfo_info_get_samplerate(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    return i->get_samplerate();
}

int taginfo_info_get_channels(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    return i->get_channels();
}

int taginfo_info_get_length(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    return i->get_length_seconds();
}

BOOL taginfo_info_get_has_image(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    //bool image_available = i->get_has_image();
    return i->get_has_image();
}

// deprecated - do not use
char * taginfo_info_get_disk_string(const TagInfo_Info *info) {
    return NULL;
}
// deprecated - do not use
void taginfo_info_set_disk_string(TagInfo_Info *info, const char *disk_string) {
}
int taginfo_info_get_disk_number(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    return i->get_disk_number();
}
void taginfo_info_set_disk_number(TagInfo_Info *info, int number) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_disk_number(number);
}

int taginfo_info_get_rating(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    return i->get_rating();
}
void taginfo_info_set_rating(TagInfo_Info *info, int rating) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_rating(rating);
}

int  taginfo_info_get_playcount(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    return i->get_playcount();
}
void taginfo_info_set_playcount(TagInfo_Info *info, int count) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_playcount(count);
}

BOOL taginfo_info_get_is_compilation(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    BOOL s = i->get_is_compilation();
    return s;
}
void taginfo_info_set_is_compilation(TagInfo_Info *info, BOOL is_compilation) {
    Info *i = reinterpret_cast<Info *>(info);
    i->set_is_compilation((bool)is_compilation);
}

char **taginfo_info_get_track_labels(const TagInfo_Info *info,
                                     int *labels_length) {
    const Info *i = reinterpret_cast<const Info *>(info);
    StringList list = i->get_track_labels_list();
    return string_list_to_c_array(list, labels_length);
}
void taginfo_info_set_track_labels(TagInfo_Info *info,
                                   const char* const* labels, int labels_length) {
    Info *i = reinterpret_cast<Info *>(info);
    StringList list = string_list_from_c_array(labels, labels_length);
    i->set_track_labels_list(list);
}

char **taginfo_info_get_album_labels(const TagInfo_Info *info,
                                     int *labels_length) {
    const Info *i = reinterpret_cast<const Info *>(info);
    StringList list = i->get_album_labels_list();
    return string_list_to_c_array(list, labels_length);
}
void taginfo_info_set_album_labels(TagInfo_Info *info,
                                   const char* const* labels, int labels_length) {
    Info *i = reinterpret_cast<Info *>(info);
    StringList list = string_list_from_c_array(labels, labels_length);
    i->set_album_labels_list(list);
}

char** taginfo_info_get_artist_labels(const TagInfo_Info *info,
                                      int * labels_length) {
    const Info *i = reinterpret_cast<const Info *>(info);
    StringList list = i->get_artist_labels_list();
    return string_list_to_c_array(list, labels_length);
}
void taginfo_info_set_artist_labels(TagInfo_Info *info,
                                    const char* const* labels, int labels_length) {
    Info *i = reinterpret_cast<Info *>(info);
    StringList list = string_list_from_c_array(labels, labels_length);
    i->set_artist_labels_list(list);
}

//****************/TAG READ/WRITE API****************************************************


//*****************IMAGES****************************************************************
BOOL taginfo_info_can_handle_image(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    return i->can_handle_images();
}

BOOL taginfo_info_get_image(const TagInfo_Info *info,
                            char** data, int* data_length, TagInfo_ImageType *image_type) {
    const Info *i = reinterpret_cast<const Info *>(info);
    ImageType *t  = reinterpret_cast<ImageType *>(image_type);
    bool v = i->get_image((*data), (*data_length), (*t));
    return v;
}

BOOL taginfo_info_set_image(TagInfo_Info *info,
                            const char* data, int data_length, TagInfo_ImageType image_type) {
    Info *i = reinterpret_cast<Info *>(info);
    bool v = i->set_image(data, data_length, (ImageType)image_type);
    return v;
}

void taginfo_free_image_data(char* data) {
    delete [] data;
}
//*****************/IMAGES***************************************************************


//*****************LYRICS****************************************************************
BOOL taginfo_info_can_handle_lyrics(const TagInfo_Info *info)  {
    const Info *i = reinterpret_cast<const Info *>(info);
    return i->can_handle_images();
}

char * taginfo_info_get_lyrics(const TagInfo_Info *info) {
    const Info *i = reinterpret_cast<const Info *>(info);
    String lyrics = i->get_lyrics();
    char *s = ::strdup(lyrics.toCString(true));
    if(stringManagementEnabled && s)
        strings.prepend(s);
    return s;
}

BOOL taginfo_info_set_lyrics(TagInfo_Info *info, const char * lyrics) {
    Info *i = reinterpret_cast<Info *>(info);
    return i->set_lyrics(String(lyrics, String::UTF8));
}
//*****************/LYRICS***************************************************************

