#include <string.h>
#include <unistd.h> //usleep()
#include <libudev.h> //role switch
#include <libusb-1.0/libusb.h>
#include <iostream>


typedef int tResult;
#define IAP2_CTL_ERROR -1
#define IAP2_OK 0
#define MICROCHIP_CMD 0x847C

/* recipient is the bridge device, not the hub */
#define MICROCHIP_VID   0x0424
#define USB84604_PID    0x2530

#define MICROCHIP_VID_STR   "0424"
#define USB84604_PID_STR    "2530"

/* PIO 16-23 Direction Register */
#define PIO23_DIR 0x0831
/* PIO 16-23 Output Register */
#define PIO23_OUT 0x0835

#define MICROCHIP_PATH  "1-1.3" //"1-1.3.3"
#define MITSUMIPORT2ES2_PATH  MICROCHIP_PATH //"1-1.3.3"
#define MITSUMIPORT2ES4_PATH  MICROCHIP_PATH //"1-1.3.3"
#define NO_RETRIAL (uint8_t)0

static tResult InitCarPlayMicroChipHub();
static tResult InitCarPlayMicroChipMitsumiPort2ES2();
static tResult InitCarPlayMicroChipMitsumiPort2ES4();

static int send_libusb_control_transfer(libusb_device_handle *dev_handle, uint8_t request_type, uint8_t bRequest, uint16_t wValue, uint16_t wIndex,
                                        unsigned char *data,uint16_t wLength,unsigned int timeout,uint8_t numOfRetrials=NO_RETRIAL);
int main(int argc, char* argv[])
{
    //-----------------------------
    // Check the number of parameters
    //----------------------------
    if (argc < 2) {
        
        // Tell the user how to run the program if no argument is called
        std::cout << "Usage: " << argv[0] << "1 or 2 or 3" << std::endl;
        //std::cerr << "0: AUTOMATIC: recognize HUB and set correct INIT" << std::endl;  tbd. later
        std::cout << "1: InitCarPlayMicroChipHub" << std::endl;
        std::cout << "2: InitCarPlayMicroChipMitsumiPort2ES2" << std::endl;
        std::cout << "3: InitCarPlayMicroChipMitsumiPort2ES4" << std::endl; 

        return 1;
    }
    // print argument
    std::cout << argv[0] << "argv[1]:"<< argv[1] << "!" << std::endl;


    if (strcmp(argv[1], "1") == 0) 
    {
        std::cout << "Calling InitCarPlayMicroChipHub" << std::endl;
        InitCarPlayMicroChipHub();
    }
    else if (strcmp(argv[1], "2") == 0) 
    {
        std::cout << "Calling InitCarPlayMicroChipMitsumiPort2ES2" << std::endl;
        InitCarPlayMicroChipMitsumiPort2ES2();
    }
    else if (strcmp(argv[1], "3") == 0) 
    {
        std::cout << "Calling InitCarPlayMicroChipMitsumiPort2ES4" << std::endl;
        InitCarPlayMicroChipMitsumiPort2ES4();
    }
    else
    {
        std::cerr << "Incorrect option" << std::endl; 
    }
     
    return 0;
}



tResult InitCarPlayMicroChipMitsumiPort2ES2()
{
    std::cout <<"Entering InitCarPlayMicroChipMitsumiPort2ES2" << std::endl;
    std::cout <<"InitCarPlayMicroChipMitsumiPort2ES2: uses InitCarPlayMicroChipHub()" << std::endl;
    tResult res=InitCarPlayMicroChipHub();
    std::cout <<"InitCarPlayMicroChipMitsumiPort2ES2: used InitCarPlayMicroChipHub(): res:"<<res << std::endl;
    std::cout <<"Leaving InitCarPlayMicroChipMitsumiPort2ES2" << std::endl;
    return res;
}

tResult InitCarPlayMicroChipMitsumiPort2ES4()
{
    std::cout <<"Entering InitCarPlayMicroChipMitsumiPort2ES4" << std::endl;
    tResult res = IAP2_CTL_ERROR; 

    bool bOk = true;
    unsigned char dat[2];
    uint16_t wLength;
    int ret = 0;
    
    //Reference see: https://hi-dms.de.bosch.com/docushare/dsweb/Get/Document-911362/CarPlay_Sequence_Ver0.93_Mitsumi_CMI-BD004-ES4_20161104.xlsx
    /*Init sequence*/
    /* 1. ISSUE DID check */
    /* 2. ISSUE Register Write Setup command for GPIO1 output setting*/
    /* 3. ISSUE USB to GPIO bridge command (PIO1) for enabling I2C bridging - PIO should be changed to HIGH for I2C communication */
    /* 3.1 Wait more than 5msec. For UCS2112 stable */
    /* 4. ISSUE Enable I2C Pass - through interface command*/
    /* 5. Add1 Issue "USB to GPIO bridge command" (PIO20) to switch state of CTL_SW signal Low level, this sequence connect logical Port1 to Physical port1 */
    /* 6. Add2 Issue Register Write Setup command for GPIO20 output setting*/
    /* 7. modify1 Issue USB to I2C bridge command for UCS Current limit setting (USB port1 and port2: Up to 1600mA)*/
    /* 8. modify2 Issue USB to I2C bridge command for UCS charger setting (Auto -> Latch) */
    /* 9. modify3 Issue USB to I2C bridge command for UCS charger setting (Port1:Power enable)*/
    /* 10. modify4 Issue USB to I2C bridge command for UCS charger setting (Port1 and 2:setting Pin_IGN,)
       /* 11. modify5 Issue USB to I2C bridge command for UCS charger setting (Port2:Power enab)*/
    /* Head Unit enumerates with Device1(if attached)*/
    /* Head Unit enumerates with Device2(if attached)*/

    //------------------------------------------------------------
    std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4:Init last step is 11." << std::endl;
    //------------------------------------------------------------
    
    int libusb_status = 0;
    libusb_context *usb_context = NULL;
    
    if(0 == (libusb_status = libusb_init(&usb_context)))
    {
        //switch hub
        libusb_device_handle* hndlToHub = libusb_open_device_with_vid_pid(usb_context, MICROCHIP_VID, USB84604_PID);
        if(hndlToHub != NULL) 
        {
            bOk = true;
            //------------------------------------------------------------
            std::cout <<"1/14 InitCarPlayMicroChipMitsumiPort2ES4:ISSUE DID check: successfull" << std::endl;
            //------------------------------------------------------------
        }
        else
        {
            bOk = false;
            std::cout <<"[ERROR]: InitCarPlayMicroChipMitsumiPort2ES4 not found" << std::endl;
        }
        
        if(bOk)
        {
            //-----------------------------------------------------------------------------------------
            std::cout <<"2/14 InitCarPlayMicroChipMitsumiPort2ES4:ISSUE Register Write Setup command for GPIO1 output setting" << std::endl;
            //-----------------------------------------------------------------------------------------
            dat[0] = 0x02; //data: (#GPIO1 is set Output)
            wLength = 0x01; //wLength (#1Byte Write )
            ret = send_libusb_control_transfer(hndlToHub,
              0x41,    // bmRequestType (#Host to device, vendor class, targeted to device )
              0x03,    // bRequest (#Register Write)
              0x0833,  // wValue (#GPIO 0-7 DIRECTION CONTROL REGISTER )
              0x0000,  // wIndex (#Reserved)
              dat,     // *data
              wLength, //wLength
              500);    // timeout
              std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4: ret:"<<ret << std::endl;
            if (ret != wLength) 
            {
                bOk = false;
                 std::cerr <<"[ERROR]: InitCarPlayMicroChipMitsumiPort2ES4:ISSUE Register Write Setup command for GPIO1 output setting" << std::endl;
            } 
        }
        if(bOk) 
        {
            //------------------------------------------------------------------------------------------------------------------------------------------------
            std::cout <<"3/14 InitCarPlayMicroChipMitsumiPort2ES4:Issue USB to GPIO bridge command (PIO1) for enabling I2C bridging - PIO should be changed to HIGH for I2C communication" << std::endl;
            //------------------------------------------------------------------------------------------------------------------------------------------
            /* GPIO1 is set High level */
            dat[0] = 0x02; //data (#GPIO1 is set High level)
            wLength = 0x01; //wLength (#1Byte Write )
            ret = send_libusb_control_transfer(hndlToHub,
              0x41,   // bmRequestType (#Host to device, vendor class, targeted to device )
              0x03,   // bRequest (#Register Write)
              0x0837, // wValue (#GPIO 0-7 OUTPUT REGISTER)
              0x0000, // wIndex (#Reserved )
              dat,    // *data
              wLength,// wLength 
              500);   // timeout
            std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4: ret:"<<ret;
            if (ret != wLength) 
            {
                bOk = false;
                std::cerr <<"[ERROR]: InitCarPlayMicroChipMitsumiPort2ES4:Issue USB to GPIO bridge command (PIO1)...." << std::endl;
            }
        }

        if(bOk)
        {
            //------------------------------------------------------------------------------------------------------------------------------------------------
            std::cout <<"4/14 InitCarPlayMicroChipMitsumiPort2ES4 Wait more than 5msec. For UCS2112 stable " << std::endl;
            //------------------------------------------------------------------------------------------------------------------------------------------
            usleep(5000);
        }

        if(bOk)
        {
            //---------------------------------------------------------------------------------
            std::cout <<"5/14 InitCarPlayMicroChipMitsumiPort2ES4:ISSUE Enable I2C Pass - through interface command" << std::endl;
            //---------------------------------------------------------------------------------
            wLength = 0x00; //wLength (# 0 byte data )
            ret = send_libusb_control_transfer(hndlToHub,
              0x41,   // bmRequestType (#Host to device, vendor class, targeted to device)
              0x70,   // bRequest (#CMD_I2C_ENTER_PASSTHRU)
              0x0000, // wValue
              0x0000, // wIndex
              NULL,   // *data
              0x00,   // wLength
              500);   // timeout
            std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4: ret:"<<ret;

            if (ret != wLength)
            {
                bOk = false;
                std::cerr <<"[ERROR]: InitCarPlayMicroChipMitsumiPort2ES4:ISSUE Enable I2C Pass - through interface command" << std::endl;
            } 
        }

        if(bOk)
        {
            //---------------------------------------------------------------------------------
            std::cout <<"6/14 InitCarPlayMicroChipMitsumiPort2ES4:Add1 Issue 'USB to GPIO bridge command' (PIO20) to switch state of CTL_SW signal Low level, this sequence connect logical Port1 to Physical port1" << std::endl;
            //---------------------------------------------------------------------------------
            dat[0] = 0x00; //Data (#GPIO20 is set Low level.  Connect Logical port 1 to Physical port 1)
            wLength = 0x01; //wLength (#1Byte Write )
            ret = send_libusb_control_transfer(hndlToHub,
            0x41,   // bmRequestType  (#Host to device, vendor class, targeted to device) 
            0x03,   // bRequest (#Register Write)
            0x0835, // wValue (#GPIO 17-20 OUTPUT REGISTER)
            0x0000, // wIndex (#Reserved)
            dat,   // *data
            wLength,// wLength 
            500);   // timeout
            std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4: ret:"<<ret << std::endl;

            if (ret != wLength)
            {
             bOk = false;
             std::cerr <<"[ERROR]: Add1 Issue 'USB to GPIO bridge command' (PIO20) to switch state" << std::endl;
            } 
        }

        if(bOk)
        {
             //---------------------------------------------------------------------------------------
             std::cout <<"7/14 MInitCarPlayMicroChipMitsumiPort2ES4:Add2 Issue Register Write Setup command for GPIO20 output setting" << std::endl;
             //---------------------------------------------------------------------------------------
             dat[0] = 0x10; //Data (#GPIO 20 is set Output mode)
             wLength = 0x01; //wLength (#1Byte Write )
             ret = send_libusb_control_transfer(hndlToHub,
               0x41,   // bmRequestType  (#Host to device, vendor class, targeted to device) 
               0x03,   // bRequest (#Register Write)
               0x0831, // wValue (#GPIO 17-20 DIRECTION CONTROL REGISTER)
               0x0000, // wIndex (#Reserved)
               dat,   // *data
               wLength,   // wLength 
               500);   // timeout
             std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4: ret:"<<ret;
        
             if (ret != wLength)
             {
                 bOk = false;
                 std::cerr <<"[ERROR]: MInitCarPlayMicroChipMitsumiPort2ES4: Add2 Issue Register Write Setup command for GPIO20 output setting" << std::endl;
             } 
         }

         if(bOk)
         {
             //---------------------------------------------------------------------------------------
             std::cout <<"8/14 InitCarPlayMicroChipMitsumiPort2ES4: modify1 Issue USB to I2C bridge command for UCS Current limit setting (USB port1 and port2: Up to 1600mA)" << std::endl;
             //---------------------------------------------------------------------------------------

             int iRetrialCount = 0;
             int iMaxRetrials  = 5;
             while(iRetrialCount < iMaxRetrials)
             {
                 //0x1424; //Data (#Setting current limit. Pyhgical Port2:1600mA , Port1:1600mA)
                 dat[0] = 0x14; //highByte 
                 dat[1] = 0x24;
                 wLength = 0x02; //wLength (#2Byte Write )

                 ret = send_libusb_control_transfer(hndlToHub,
                   0x41,   // bmRequestType  (#Vender specific command, Host-to-device data transfer) 
                   0x71,   // bRequest (#CMD_I2C_WRITE)
                   0x07AE, // wValue (#I2C Control flag 0x07, I2C address 0x57(1010_111b) + (r/w) b)
                   0x0000, // wIndex (#Reserved)
                   dat,   // *data
                   wLength,   // wLength 
                   500);   // timeout
                 std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4: ret:"<<ret << std::endl;

                 if(ret != wLength)
                 {
                     iRetrialCount++;
                     std::cout <<"[WARNING]: InitCarPlayMicroChipMitsumiPort2ES4: Could not set UCS current limit (trial ("<<iRetrialCount<<"/"<<iMaxRetrials<<")" << std::endl;
                     usleep(5000);
                 }
                 else
                 {
                     std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4: set UCS current limit - ok" << std::endl;
                     break;
                 }
             }
             if (ret != wLength)
             {
                 bOk = false;
                 std::cerr <<"[ERROR]: InitCarPlayMicroChipMitsumiPort2ES4: modify1 Issue USB to I2C bridge command for UCS Current limit setting (USB port1 and port2: Up to 1600mA)" << std::endl;
             } 
         }

         if(bOk)
         {
             //---------------------------------------------------------------------------------------
             std::cout <<"9/14 InitCarPlayMicroChipMitsumiPort2ES4: modify2 Issue USB to I2C bridge command for UCS charger setting (Auto -> Latch mode)" << std::endl;
             //---------------------------------------------------------------------------------------
             //0x1581; //Data (#Set Latch bit "1" (bit7))
             dat[0]  = 0x15;
             dat[1]  = 0x81;
             wLength = 0x02; //wLength (#2 Byte of Data(Register Address + 1 Byte of Data) )

             ret = send_libusb_control_transfer(hndlToHub,
               0x41,   // bmRequestType  (#Vender specific command, Host-to-device data transfer) 
               0x71,   // bRequest (#CMD_I2C_WRITE)
               0x07AE, // wValue (#I2C Control flag 0x07, I2C address 0x57(1010_111b) + (r/w) b)
               0x0000, // wIndex (#Reserved)
               dat,   // *data
               wLength,   // wLength (#2 Byte of Data(Register Address + 1 Byte of Data))
               500);   // timeout
             std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4: ret:"<<ret << std::endl;
        
             if (ret != wLength)
             {
                 bOk = false;
                std::cerr <<"[ERROR]: InitCarPlayMicroChipMitsumiPort2ES4:modify2 Issue USB to I2C bridge command for UCS charger setting (Auto -> Latch)" << std::endl;
             } 
         }

         
        if(bOk)
        {
         //---------------------------------------------------------------------------------------
         std::cout <<"10/14 InitCarPlayMicroChipMitsumiPort2ES4:modify3 Issue USB to I2C bridge command for UCS charger setting (Port1:Power enable)" << std::endl;
         //---------------------------------------------------------------------------------------
         //0x111F; //Data  (#ALERT_MASK:0,ALERT1_LINK:0,DSCHG1:0,PWR_EN1S:1,DISCHG_TIME:11,ATT1_TH1:11)
         dat[0] = 0x11; //highByte
         dat[1] = 0x1F;      //lowByte
         wLength = 0x02; //wLength (#2 Byte of Data(Register Address + 1 Byte of Data))

         ret = send_libusb_control_transfer(hndlToHub,
           0x41,   // bmRequestType  (#Vender specific command, Host-to-device data transfer) 
           0x71,   // bRequest (#CMD_I2C_WRITE)
           0x07AE, // wValue (#I2C Control flag 0x07, I2C address 0x57(1010_111b) + (r/w) b)
           0x0000, // wIndex (#Reserved)
           dat,   // *data
           wLength,// wLength 
           500);   // timeout
         std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4: ret:"<<ret << std::endl;

         if (ret != wLength)
         {
             bOk = false;
             std::cerr <<"[ERROR]: InitCarPlayMicroChipMitsumiPort2ES4: modify3 Issue USB to I2C bridge command for UCS charger setting (Port1:Power enable)" << std::endl;
         } 
        }

        if(bOk)
        {
          //---------------------------------------------------------------------------------------
           std::cout <<"11/14 InitCarPlayMicroChipMitsumiPort2ES4: modify4 Issue USB to I2C bridge command for UCS charger setting (Port1 and 2:setting Pin_IGN,))" << std::endl;
          //---------------------------------------------------------------------------------------
          //0x13E0; //Data (#PIN_IGN:1 , ATT_DIS:1, and DIS_TO:1,PWR_STATE<1:0>:00))
          dat[0] = 0x13; //highByte
          dat[1] = 0xE0;      //lowByte

          wLength = 0x02; //wLength (#2 Byte of Data(Register Address + 1 Byte of Data))

          ret = send_libusb_control_transfer(hndlToHub,
            0x41,   // bmRequestType  (#Host to device, vendor class, targeted to device) 
            0x71,   // bRequest (#CMD_I2C_WRITE)
            0x07AE, // wValue (#I2C Control flag 0x07, I2C address 0x57(1010_111b) + (r/w) b)
            0x0000, // wIndex (#Reserved)
            dat,   // *data
            wLength,// wLength 
            500);   // timeout
          std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4: ret:"<<ret << std::endl;

          if (ret != wLength)
          {
              bOk = false;
               std::cerr <<"[ERROR]: InitCarPlayMicroChipMitsumiPort2ES4: modify4 Issue USB to I2C bridge command for UCS charger setting (Port1 and 2:setting Pin_IGN,)" << std::endl;
          } 
        }

        if(bOk)
        {
          //---------------------------------------------------------------------------------------
          std::cout <<"12/14 MicroChipMitsumiPort2ES4: modify5 Issue USB to I2C bridge command for UCS charger setting (Port2:Power enab)" << std::endl;
          //---------------------------------------------------------------------------------------
          //0x1213; //Data (#ALERT_MASK:0,ALERT2_LINK:0,DSCHG2:0,PWR_EN2S:1,ATT_TH2:11))
          dat[0] = 0x12; //highByte
          dat[1] = 0x13;      //lowByte

          wLength = 0x02; //wLength (#2 Byte of Data(Register Address + 1 Byte of Data))

          ret = send_libusb_control_transfer(hndlToHub,
            0x41,   // bmRequestType  (#Vender specific command, Host-to-device data transfer) 
            0x71,   // bRequest (#CMD_I2C_WRITE)
            0x07AE, // wValue (#I2C Control flag 0x07, I2C address 0x57(1010_111b) + (r/w) b)
            0x0000, // wIndex (#Reserved)
            dat,   // *data
            wLength,// wLength (#2 Byte of Data(Register Address + 1 Byte of Data))
            500);   // timeout
          std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4: ret:"<<ret << std::endl;

          if (ret != wLength)
          {
              bOk = false;
              std::cerr <<"[ERROR]: InitCarPlayMicroChipMitsumiPort2ES4: modify5 Issue USB to I2C bridge command for UCS charger setting (Port2:Power enab)" << std::endl;
          } 
          else
          {
              /*====SUCCESS========*/
              res = IAP2_OK;
              /*==================*/
              std::cout <<"13/14 [RESULT:OK]: InitCarPlayMicroChipMitsumiPort2ES4: Initial_sequence Mitsumi HUB ok" << std::endl;
          }
        }

        //close handle if open
        if(hndlToHub != NULL)
        {
            std::cout <<"14/14 InitCarPlayMicroChipMitsumiPort2ES4:libusb_close" << std::endl;
            libusb_close(hndlToHub);
        }
    }
    else
    {
         std::cout <<"InitCarPlayMicroChipMitsumiPort2ES4: libusb context not initialized" << std::endl;
    }    
    

    std::cout <<"Leaving InitCarPlayMicroChipMitsumiPort2ES4" << std::endl;
    return res;

}
        

tResult InitCarPlayMicroChipHub()
{
    std::cout <<"Entering InitCarPlayMicroChipHub" << std::endl;
    tResult res = IAP2_CTL_ERROR;

    int libusb_status = 0;
    libusb_context *usb_context = NULL;
    
    if(0 == (libusb_status = libusb_init(&usb_context)))
    {
        //switch hub
        libusb_device_handle* hndlToHub = libusb_open_device_with_vid_pid(usb_context, MICROCHIP_VID, USB84604_PID);
        if(hndlToHub != NULL) {

            unsigned char dat[2];
            int ret = 0;

            /* set GPIO1 high for I2C SDA/SCL pull-up resistors */

            std::cout <<"InitCarPlayMicroChipHub: Issue Register Write Setup command for GPIO1 output setting" << std::endl;
            /* GPIO1 is set Output */
            dat[0] = 0x02;
            ret = libusb_control_transfer(hndlToHub,
              /* Host to device, vendor class, targeted to device */
              0x41,   // bmRequestType
              /* Register Write */
              0x03,   // bRequest
              /* GPIO 0-7 DIRECTION CONTROL REGISTER */
              0x0833, // wValue
              /* Reserved */
              0x0000, // wIndex
              dat,    // *data
              /* 1Byte Write */
              0x01,   // wLength
              500);   // timeout
            std::cout <<"InitCarPlayMicroChipHub: ret:"<<ret << std::endl;
            if (ret != 1) {
                std::cerr <<"[ERROR]: InitCarPlayMicroChipHub: Could not set GPIO1 to output" << std::endl;
            } else {

                std::cout <<"InitCarPlayMicroChipHub: :Issue USB to GPIO bridge command (PIO1) for enabling I2C bridging" << std::endl;
                /* GPIO1 is set High level */
                dat[0] = 0x02;
                ret = libusb_control_transfer(hndlToHub,
                  /* Host to device, vendor class, targeted to device */
                  0x41,   // bmRequestType
                  /* Register Write */
                  0x03,   // bRequest
                  /* GPIO 0-7 OUTPUT REGISTER */
                  0x0837, // wValue
                  /* Reserved */
                  0x0000, // wIndex
                  dat,    // *data
                  /* 1Byte Write */
                  0x01,   // wLength
                  500);   // timeout
                std::cout <<"InitCarPlayMicroChipHub: ret:"<<ret;
                if (ret != 1) {
                    std::cerr <<"[ERROR]: InitCarPlayMicroChipHub: Could not set GPIO1 to high" << std::endl;
                } else {

                    std::cout <<"InitCarPlayMicroChipHub: Issue Enable I2C pass-through interface command" << std::endl;
                    ret = libusb_control_transfer(hndlToHub,
                      /* Host to device, vendor class, targeted to device */
                      0x41,   // bmRequestType
                      /* CMD_I2C_ENTER_PASSTHRU */
                      0x70,   // bRequest
                      0x0000, // wValue
                      0x0000, // wIndex
                      NULL,   // *data
                      0x00,   // wLength
                      500);   // timeout
                    std::cout <<"InitCarPlayMicroChipHub: ret:"<<ret;

                    if (ret != 0) {
                        std::cerr <<"[ERROR]:InitCarPlayMicroChipHub:  Could not set I2C to pass-through" << std::endl;
                    } else {

                        int retry = 0;
                        int done = 0;
                        while (retry++ < 3 && done == 0) {
                           std::cout <<"InitCarPlayMicroChipHub: Issue USB to I2C bridge command for UCS setting (Port1: current limit setting)" << std::endl;
                            /* Setting current limit 1680mA */
                            dat[0] = 0x19;
                            dat[1] = 0x04;
                            ret = libusb_control_transfer(hndlToHub,
                              /* Host to device, vendor class, targeted to device */
                              0x41,   // bmRequestType
                              /* CMD_I2C_WRITE */
                              0x71,   // bRequest
                              /* I2C Control flag 0x07, I2C address 0x66(0110_0110b). For port1 control */
                              0x0766, // wValue
                              0x0000, // wIndex
                              /* 2 Byte of Data (Register Address + 1 Byte of Data) */
                              dat,    // *data
                              0x02,   // wLength
                              500);   // timeout
                            std::cout <<"InitCarPlayMicroChipHub: ret:"<<ret;
                            if (ret == 2) {
                                done = 1;
                            } else {
                                std::cerr <<"InitCarPlayMicroChipHub: Warning: Port1: Could not send I2C bridge command, retry ("<<retry<<")" << std::endl;
                            }
                        }
                        if (retry > 4) {
                          std::cerr <<"[ERROR]: InitCarPlayMicroChipHub: Port1: Could not send I2C bridge command" << std::endl;
                        } else {

                            std::cout <<"InitCarPlayMicroChipHub: Issue USB to I2C bridge command for UCS setting (Port2: current limit setting)" << std::endl;
                            /* Setting current limit 1680mA */
                            dat[0] = 0x19;
                            dat[1] = 0x04;
                            ret = libusb_control_transfer(hndlToHub,
                              /* Host to device, vendor class, targeted to device */
                              0x41,   // bmRequestType
                              /* CMD_I2C_WRITE */
                              0x71,   // bRequest
                              /* I2C Control flag 0x07, I2C address 0x64(0110_0100b). For port2 control */
                              0x0764, // wValue
                              0x0000, // wIndex
                              /* 2 Byte of Data (Register Address + 1 Byte of Data) */
                              dat,    // *data
                              0x02,   // wLength
                              500);   // timeout
                            std::cout <<"InitCarPlayMicroChipHub: ret:"<<ret << std::endl;
                            if (ret != 2) {
                                std::cerr <<"[ERROR]: InitCarPlayMicroChipHub: Port2: Could not send I2C bridge command" << std::endl;
                            } else {

                                std::cout <<"InitCarPlayMicroChipHub: Issue USB to I2C bridge command for UCS charger setting (Port1: Only changing Auto recovery -> Latch)" << std::endl;
                                /* Set Latch bit "1" (bit0) */
                                dat[0] = 0x17;
                                dat[1] = 0x3D;
                                ret = libusb_control_transfer(hndlToHub,
                                  /* Host to device, vendor class, targeted to device */
                                  0x41,   // bmRequestType
                                  /* CMD_I2C_WRITE */
                                  0x71,   // bRequest
                                  /* I2C Control flag 0x07, I2C address 0x66(0110_0110b). For port1 control */
                                  0x0766, // wValue
                                  0x0000, // wIndex
                                  /* 2 Byte of Data (Register Address + 1 Byte of Data) */
                                  dat,    // *data
                                  0x02,   // wLength
                                  500);   // timeout
                                std::cout <<"InitCarPlayMicroChipHub: ret:"<<ret << std::endl;
                                if (ret != 2) {
                                    std::cerr <<"[ERROR]: InitCarPlayMicroChipHub: Could not set charger setting" << std::endl;
                                } else {

                                    std::cout <<"Issue USB to I2C bridge command for UCS charger setting (Port2: Only changing Auto recovery -> Latch)" << std::endl;
                                    /* Set Latch bit "1" (bit0) */
                                    dat[0] = 0x17;
                                    dat[1] = 0x3D;
                                    ret = libusb_control_transfer(hndlToHub,
                                      /* Host to device, vendor class, targeted to device */
                                      0x41,   // bmRequestType
                                      /* CMD_I2C_WRITE */
                                      0x71,   // bRequest
                                      /* I2C Control flag 0x07, I2C address 0x64(0110_0100b). For port2 control */
                                      0x0764, // wValue
                                      0x0000, // wIndex
                                      /* 2 Byte of Data (Register Address + 1 Byte of Data) */
                                      dat,    // *data
                                      0x02,   // wLength
                                      500);   // timeout
                                    std::cout <<"InitCarPlayMicroChipHub: ret:"<<ret << std::endl;
                                    if (ret != 2) {
                                        std::cerr <<"[ERROR]: InitCarPlayMicroChipHub: Could not set charger setting";
                                    } else {

                                        std::cout <<"InitCarPlayMicroChipHub: Issue USB to I2C bridge command for UCS charger setting (Port1: setting Pin_IGN, Power enable, CDP)" << std::endl;
                                        /* PIN_IGN set "1" , PWR_EN set "1", and Set CDP mode */
                                        dat[0] = 0x17;
                                        dat[1] = 0xBF;
                                        ret = libusb_control_transfer(hndlToHub,
                                          /* Host to device, vendor class, targeted to device */
                                          0x41,   // bmRequestType
                                          /* CMD_I2C_WRITE */
                                          0x71,   // bRequest
                                          /* I2C Control flag 0x07, I2C address 0x66(0110_0110b). For port1 control */
                                          0x0766, // wValue
                                          0x0000, // wIndex
                                          /* 2 Byte of Data (Register Address + 1 Byte of Data) */
                                          dat,    // *data
                                          0x02,   // wLength
                                          500);   // timeout
                                        std::cout <<"InitCarPlayMicroChipHub: ret:"<<ret << std::endl;
                                        if (ret != 2) {
                                            std::cerr <<"[ERROR]: InitCarPlayMicroChipHub: Could not set PIO_20" << std::endl;
                                        } else {

                                            std::cout <<"InitCarPlayMicroChipHub: Issue USB to I2C bridge command for UCS charger setting (Port2: setting Pin_IGN, Power enable, CDP)" << std::endl;
                                            /* PIN_IGN set "1" , PWR_EN set "1", and Set CDP mode */
                                            dat[0] = 0x17;
                                            dat[1] = 0xBF;
                                            ret = libusb_control_transfer(hndlToHub,
                                              /* Host to device, vendor class, targeted to device */
                                              0x41,   // bmRequestType
                                              /* CMD_I2C_WRITE */
                                              0x71,   // bRequest
                                              /* I2C Control flag 0x07, I2C address 0x64(0110_0100b). For port2 control */
                                              0x0764, // wValue
                                              0x0000, // wIndex
                                              /* 2 Byte of Data (Register Address + 1 Byte of Data) */
                                              dat,    // *data
                                              0x02,   // wLength
                                              500);   // timeout
                                            std::cout <<"InitCarPlayMicroChipHub: ret:"<<ret << std::endl;
                                            if (ret != 2) {
                                                std::cerr <<"[ERROR]: InitCarPlayMicroChipHub: Could not set PIO_20" << std::endl;
                                            } else {
                                                /*====SUCCESS========*/
                                                res = IAP2_OK;
                                                /*==================*/
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            std::cout <<"InitCarPlayMicroChipHub:libusb_close" << std::endl;
            libusb_close(hndlToHub);
        } else {
            std::cout <<"InitCarPlayMicroChipHub: MCH not found" << std::endl;
        }
    }
    else
    {
         std::cout <<"InitCarPlayMicroChipHub: libusb context not initialized" << std::endl;
    }    

    
    std::cout <<"InitCarPlayMicroChipHub: res:"<<res<<" (IAP2_OK:"<<IAP2_OK<<",IAP2_CTL_ERROR="<<IAP2_CTL_ERROR<<")" << std::endl;
    std::cout <<"Leaving InitCarPlayMicroChipHub" << std::endl;
    return res;
}


int send_libusb_control_transfer(libusb_device_handle *dev_handle,uint8_t request_type, uint8_t bRequest, uint16_t wValue, uint16_t wIndex, unsigned char *data, uint16_t wLength, unsigned int timeout, uint8_t numOfRetrials)
{
    std::cout <<"Entering send_libusb_control_transfer" << std::endl;
    int     iRet  = 0;
    uint8_t retry = 0;
    int     done  = 0;

    while ((retry <= numOfRetrials) && (done == 0)) 
    {
        iRet = libusb_control_transfer(dev_handle,request_type, bRequest, wValue, wIndex, data, wLength, timeout );
        if (iRet == wLength) 
        {
            done = 1;
        } 
        else 
        {
           std::cerr <<"[Warning]: do retry"<<retry << std::endl;
        }
        retry++;
        
    }
    std::cout <<"Leaving send_libusb_control_transfer" << std::endl;
    return iRet;
}

