/************************************************************************
* FILE:         vdmmgr_trace.cpp
* PROJECT:      ARION Prototyp
* SW-COMPONENT: Virtual Device MediaManager
*----------------------------------------------------------------------
*
* DESCRIPTION: VD MediaManager
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author             | Modification
* 28.09.05  | CM-DI/ESA2 Fiebing | initial version
*
*************************************************************************/


#define _CLASS  VDMMGR_TR_TRACE

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
#include "Config.h"
#include <vector>

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_MEDIAMANAGER_TRACE
#include "trcGenProj/Header/vdmmgr_trace.cpp.trc.h"
#endif

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"                       // use AIL template with MessageMaps
#define AHL_S_IMPORT_INTERFACE_NOTIFICTABLE
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define VDMMGR_S_IMPORT_INTERFACE_MSG
#include "vdmmgr_if.h"                    // For VD MMgr interface

/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/
#include "vdmmgr_trace.h"                 // trace stuff
#include "vdmmgr_main.h"                  // for class vdmmgr_tclApp -> use function for TTFis commands
#include "vdmmgr_timer.h"                 // 'automatic reinsert CD' timer
#include "vdmmgr_service.h"               // to use service functions for TTFis commands
#include "vdmmgr_cdctrlif.h"              // to use CD control functions for TTFis commands
#include "vdmmgr_errorif.h"               // own thread for error notifications from OSAL (could come during OSAL function runs)

///*******************************************************************************
//| defines and macros (scope: module-local)
//|-----------------------------------------------------------------------------*/

#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#define VDMMGR_TR_CD_EJECT_SIM                        0x01
#define VDMMGR_TR_CD_NOTIFI_DEFECT_SIM                0x03
#define VDMMGR_TR_CD_NOTIFI_MEDIA_CHANGE_SIM          0x05
#define VDMMGR_TR_CD_NOTIFI_TOTAL_FAILURE_SIM         0x09
#define VDMMGR_TR_CD_NOTIFI_MEDIA_STATE_SIM           0x0B
#define VDMMGR_TR_CD_NOTIFI_DEVICE_STATE_SIM          0x0D
#define VDMMGR_TR_CD_GET_TYPE_AND_STATUS              0x0F
#define VDMMGR_TR_CD_DIAG_DRIVE_TEST                  0x11
#define VDMMGR_TR_CD_DIAG_SELECT_TRACK_TYPE           0x12
#define VDMMGR_TR_PRINT_MEDIAMANAGERSTATES            0x3F
#define VDMMGR_TR_CD_INSERT_SIM                       0x3E
#define VDMMGR_TR_DIAGFIRMWARE_VERSION                0x4A
#define VDMMGR_TR_DIAGDRIVE_VERSION                   0x4B
#define VDMMGR_TR_DIAGLOAD_STATE                      0x4C
#endif // #if defined VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE

///*******************************************************************************
//| typedefs (scope: module-local)
//|-----------------------------------------------------------------------------*/
//
///*******************************************************************************
//| variable definition (scope: global)
//|-----------------------------------------------------------------------------*/
//
///*******************************************************************************
//| variable definition (scope: module-local)
//|-----------------------------------------------------------------------------*/
//
//
///*******************************************************************************
//| function prototype (scope: module-local)
//|-----------------------------------------------------------------------------*/
//
///*******************************************************************************
//| function implementation (scope: module-local)
//|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: global)
|-----------------------------------------------------------------------------*/
vdmmgr_tclTrace::vdmmgr_tclTrace( tVoid )
{
    // init variable to avoid lint Warning 1401
    poMain = NULL;
    // Standardkonstruktor, do nothing
}

/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Constructor.
 *
 *  Creates a new vdmmgr_tclTrace object.
 *
 * @param   poMainService
 *
 * @date    2005-02-25
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
vdmmgr_tclTrace::vdmmgr_tclTrace( vdmmgr_tclApp*  poVdMmgrMainApp )
{
    poMain = poVdMmgrMainApp;
}


/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Destructor.
 *
 *  Destroys a vdmmgr_tclTrace object.
 *
 * @date    2005-02-25
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
vdmmgr_tclTrace::~vdmmgr_tclTrace()
{
    // init variable to avoid LINT warning -e1540
    poMain = NULL;
}


/******************************************************FunctionHeaderBegin******
 * CLASS       : vdmmgr_tclTrace
 * FUNCTION    : vdmmgr_Trace_vRx
 * CREATED     : 2005-07-27
 * AUTHOR      :
 * DESCRIPTION :   -
 * SYNTAX      : tVoid vdmmgr_Trace_vRx( tU8* pu8Data )
 * ARGUMENTS   :
 *               tU8*  pu8Data
 * RETURN VALUE:
 *               tVoid -
 * NOTES       :   -
 *******************************************************FunctionHeaderEnd******/
tVoid vdmmgr_tclTrace::vdmmgr_Trace_vRx( tPCU8 pu8Data ) const
{
    tU8   u8MsgCode   = pu8Data[1];
    tU8   u8Data1     = pu8Data[2];
    tU8   u8Data2     = pu8Data[3];
    tU8   u8Data3     = pu8Data[4];

    if( pu8Data != NULL )
    {
        switch( u8MsgCode )
        {
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        case VDMMGR_TR_CD_EJECT_SIM:
            ETG_TRACE_USR1(( "vdmmgr_Trace_vRx( ): Simulate incoming CCA message for method start CD eject" ));
            // Call function for simulate start of CD eject method.
            if( poMain && poMain->_vdmmgr_poCCAService )
            {
                // Store CD type before eject -> fast CD recognition in case of 'auto insert'.
                poMain->_vdmmgr_poCCAService->vSetCdTypeBeforeEject( poMain->_vdmmgr_poCCAService->enGetActCdType() );
                poMain->_vdmmgr_poCCAService->vNewStateInsert( MMGR_INSERT_AFTERON,  FALSE ); // Set InsertState to AfterOn
                poMain->_vdmmgr_poCCAService->vNewCdType     ( MMGR_EJECTING,        FALSE ); // Set CD type to 'Eject'.
                poMain->_vdmmgr_poCCAService->vNewCdState    ( MMGR_MEDIA_NOT_READY, TRUE );  // Set CD state to 'media not ready'.
            }
            if( poMain && poMain->_vdmmgr_poCdctrlIf )
            {
                (tVoid)poMain->_vdmmgr_poCdctrlIf->bEjectInThread( );
            }
            break;
        case VDMMGR_TR_CD_INSERT_SIM:
            if (poMain->_vdmmgr_poCdctrlIf) {
                ETG_TRACE_USR1(( "vdmmgr_Trace_vRx( ): Simulate incoming CCA message for method start CD insert" ));
                if(poMain->_vdmmgr_poCdctrlIf->e8GetLoaderState( ) == MMGR_MEDIA_IN_SLOT) {
                    if(poMain->_vdmmgr_poCdctrlIf->bCloseDoor())
                    {
                        ETG_TRACE_USR4(( "vdmmgr_Trace_vRx( ): vHandleInsert( ): CloseDoor success"));
                    }
                    else
                    {
                        ETG_TRACE_USR4(( "vdmmgr_Trace_vRx( ): vHandleInsert( ): CloseDoor failed"));
                    }
                }
                else {
                    ETG_TRACE_USR1(( "vdmmgr_Trace_vRx( ): vHandleInsert( ): Loader state is not MMGR_MEDIA_IN_SLOT"));
                }
            }
            else {
                ETG_TRACE_USR1(( "vdmmgr_Trace_vRx( ): vHandleInsert( ): Error : Reference failed"));
            }
            break;
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        case VDMMGR_TR_CD_NOTIFI_DEFECT_SIM:               // case, no break
        case VDMMGR_TR_CD_NOTIFI_MEDIA_CHANGE_SIM:         // case, no break
        case VDMMGR_TR_CD_NOTIFI_TOTAL_FAILURE_SIM:        // case, no break
        case VDMMGR_TR_CD_NOTIFI_MEDIA_STATE_SIM:          // case, no break
        case VDMMGR_TR_CD_NOTIFI_DEVICE_STATE_SIM:         // case, no break
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            vSimulateEventAfterOsalNotification( u8MsgCode, u8Data1 , u8Data2, u8Data3 );
            break;
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        case VDMMGR_TR_CD_GET_TYPE_AND_STATUS:
            if( poMain && poMain->_vdmmgr_poCCAService && poMain->_vdmmgr_poCdctrlIf )
            {
                T_e8DriveTempSensorStatus  DriveTempSensorStatus;
                ETG_TRACE_FATAL(( "vdmmgr_Trace_vRx( ): GetCDTyeAndStatus: CD Type: %d, CD State: %d, Insert State: %d, CD Device State: %d, CD Drive temp: %d, CD Drive temp sensor status: %d",
                                  ETG_ENUM(CD_TYPE, poMain->_vdmmgr_poCCAService->enGetActCdType()),                ETG_ENUM(MEDIA_STATE_CODE, poMain->_vdmmgr_poCCAService->enGetActCDState()),
                                  ETG_ENUM(CD_INSERT_STATE, poMain->_vdmmgr_poCCAService->enGetActCDInsertState()), ETG_ENUM(CD_DVD_DEVICE_STATE, poMain->_vdmmgr_poCCAService->enGetActCDDeviceState()),
                                  poMain->_vdmmgr_poCdctrlIf->s16GetTemperature( &DriveTempSensorStatus ), ETG_ENUM(DRIVE_SENSOR_STATE, DriveTempSensorStatus)   ));
            }
            break;
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        case VDMMGR_TR_CD_DIAG_DRIVE_TEST:
            if( poMain && poMain->_vdmmgr_poCCAService && poMain->_vdmmgr_poCdctrlIf )
            {
                T_DiagMethodControlStartDriveTest      DiagStartDriveTest;
                ETG_TRACE_FATAL(( "vdmmgr_Trace_vRx( ): VDMMGR_SimulateCdDriveTest: Start" ));
                // For simulation via TTFis
                poMain->_vdmmgr_poCCAService->_bSimulatedCdDriveTest = TRUE;
                DiagStartDriveTest.e8Duration                            = MMGR_SINGLE_RUN;                  // only single run
                DiagStartDriveTest.StartDriveTestParam.e16Drive          = MMGR_CD;                          // only for CD drive
                DiagStartDriveTest.StartDriveTestParam.e16DriveTestCmd   = (T_e16_DiagDrvTestCmd)u8Data2;    // could be: MMGR_START, MMGR_STOP, MMGR_CANCEL
                DiagStartDriveTest.StartDriveTestParam.e16DriveTestParam = (T_e16_DiagDrvTestParam)u8Data1;  // could be: MMGR_SHORT, MMGR_INTENSIVE, MMGR_SHORT_TEF_CD
                poMain->_vdmmgr_poCdctrlIf->vDiagDriveTest( &DiagStartDriveTest );
                ETG_TRACE_FATAL(( "vdmmgr_Trace_vRx( ): VDMMGR_SimulateCdDriveTest: End" ));
            }
            break;
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        case VDMMGR_TR_CD_DIAG_SELECT_TRACK_TYPE:
            if( poMain && poMain->_vdmmgr_poCCAService && poMain->_vdmmgr_poCdctrlIf )
            {
                T_e8_TrackType          e8TrackType;
                // For simulation via TTFis
                poMain->_vdmmgr_poCCAService->_bSimulatedSelectTrackType = TRUE;
                ETG_TRACE_FATAL(( "vdmmgr_Trace_vRx( ): VDMMGR_SimulateCdSelectTrackType: Start" ));
                e8TrackType = (T_e8_TrackType)u8Data1;
                poMain->_vdmmgr_poCdctrlIf->vDiagSelectTrackType( e8TrackType );
                ETG_TRACE_FATAL(( "vdmmgr_Trace_vRx( ): VDMMGR_SimulateCdSelectTrackType: End" ));
            }
            break;
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        case VDMMGR_TR_DIAGFIRMWARE_VERSION:
        {
            OSAL_trDriveVersion  rDriveVersion;
            tU8                  u8Cnt;
            if (poMain && poMain->_vdmmgr_poCdctrlIf)
            {
                if( poMain->_vdmmgr_poCdctrlIf->bGetDriveVersion( &rDriveVersion ) )
                {
                    // Copy drive version
                    for( u8Cnt = 0; u8Cnt< MMGR_MAX_FIRMWAREVERSIONLENGHT; u8Cnt++ )
                    {
                        ETG_TRACE_FATAL(( "vdmmgr_Trace_vRx( ): DiagFirmware version : FirmwareRevision[%d]: %d",u8Cnt, rDriveVersion.au8FirmwareRevision[u8Cnt]));
                    }
                }
                else
                {
                    ETG_TRACE_FATAL(( "vdmmgr_Trace_vRx( ): DiagFirmware version : Error"));
                }
            }
            break;
        }
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        case VDMMGR_TR_DIAGDRIVE_VERSION:
        {
            OSAL_trDriveVersion  rDriveVersion;
            tU8                  u8Cnt;
            if(poMain && poMain->_vdmmgr_poCdctrlIf)
            {
                if( poMain->_vdmmgr_poCdctrlIf->bGetDriveVersion( &rDriveVersion ) )
                {
                    // Copy drive version
                    for( u8Cnt = 0; u8Cnt< MMGR_MAX_DRIVEVERSIONLENGTH; u8Cnt++ )
                    {
                        ETG_TRACE_FATAL(( "vdmmgr_Trace_vRx( ): DiagFirmware version : FirmwareRevision[%d]: %d",u8Cnt,rDriveVersion.au8ModelNumber[u8Cnt]));
                    }
                }
                else
                {
                    ETG_TRACE_FATAL(( "vdmmgr_Trace_vRx( ): DiagDrive version : Error"));
                }
            }
        }
        break; // sim4hi: fix for Coverity CID-15881
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        case VDMMGR_TR_DIAGLOAD_STATE:
        {
            if(poMain && poMain->_vdmmgr_poCdctrlIf)
            {
                T_e8_DiagLoadStatus  e8Ret;
                e8Ret = poMain->_vdmmgr_poCdctrlIf->e8GetLoaderState( );
                ETG_TRACE_FATAL(( "vdmmgr_Trace_vRx( ): LoaderState: %d", ETG_ENUM(LOADER_STATE, e8Ret) ));
            }
        }
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE

#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        case VDMMGR_TR_PRINT_MEDIAMANAGERSTATES:
        {
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            if( poMain && poMain->_vdmmgr_poCdctrlIf )
            {
                poMain->_vdmmgr_poCdctrlIf->vPrintCDState();
            }
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        }
        break; // sim4hi: swapped lines to avoid possible Coverity issue
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        default:    // do nothing
            // Unknown simulation command -> trace
            ETG_TRACE_ERR(( "vdmmgr_Trace_vRx( ): Error: unknown message code: %d", u8MsgCode ));
            break;
        }
    }
    else
    {
        ETG_TRACE_ERR(( "vdmmgr_Trace_vRx( ): Error: -> invalid value for pointer pu8Data" ));
    }

}


/******************************************************FunctionHeaderBegin******
 * CLASS       : vdmmgr_tclTrace
 * FUNCTION    : vSimulateEventAfterOsalNotification( tU8 u8MsgCode, tU8 u8Data1, tU8 u8Data2, u8Data3=0 )
 * CREATED     : 2006-02-15
 * AUTHOR      :
 * DESCRIPTION : Receive incoming trace commands and simulation notification
 *               from dev/cdctrl0 and dev/cdctrl1
 * ARGUMENTS   :
 *               tU8 u8MsgCode, tU8 u8Data1, tU8 u8Data2
 * RETURN VALUE:
 *               tVoid -
 * NOTES       :   -
 *******************************************************FunctionHeaderEnd******/
tVoid vdmmgr_tclTrace::vSimulateEventAfterOsalNotification( tU8 u8MsgCode, tU8 u8Data1, tU8 u8Data2, tU8 u8Data3/*=0*/ ) const
{
    tU16              u16NotificationData;       /* For data of a notification  */
    tS32              s32Success  = OSAL_ERROR;
    OSAL_tEventHandle hCbkEvent;
    OSAL_tEventMask   oMask       = 0;

    FOR_FUTURE_USE( u8Data3 );

    u16NotificationData      = MAKEU16( u8Data2, u8Data1 );

    switch( u8MsgCode )
    {
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
    case VDMMGR_TR_CD_NOTIFI_DEFECT_SIM:
        ETG_TRACE_USR1(( "vSimulateEventAfterOsalNotification( ): EvSimulateEventAfterOsalNotification( ); CD Defect: %d", ETG_ENUM(OSAL_NOTIFI_DEFECT, u16NotificationData) ));
        if( poMain && poMain->_vdmmgr_poErrorIf )
        {
            poMain->_vdmmgr_poErrorIf->vSetDefectCdForSimulation( u16NotificationData );
        }
        oMask = VDMMGR_ERRORIF_EVENT_MASK_DEFECT_CD;
        break;
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
    case VDMMGR_TR_CD_NOTIFI_MEDIA_CHANGE_SIM:
        ETG_TRACE_USR1(( "vSimulateEventAfterOsalNotification( ): CD Media Change: %d", ETG_ENUM(OSAL_NOTIFI_MEDIA_CHANGE, u16NotificationData) ));
        if( poMain && poMain->_vdmmgr_poCdctrlIf )
        {
            poMain->_vdmmgr_poCdctrlIf->vSetMediaChangeForSimulation( u16NotificationData );
        }
        oMask = VDMMGR_EVENT_MASK_MEDIA_CHANGE;
        break;
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
    case VDMMGR_TR_CD_NOTIFI_TOTAL_FAILURE_SIM:
        ETG_TRACE_USR1(( "vSimulateEventAfterOsalNotification( ): CD Total Failure: %d", ETG_ENUM(OSAL_NOTIFI_TOTAL_FAILURE, u16NotificationData) ));
        if( poMain && poMain->_vdmmgr_poCdctrlIf )
        {
            poMain->_vdmmgr_poCdctrlIf->vSetTotalFailureForSimulation( u16NotificationData );
        }
        oMask = VDMMGR_EVENT_MASK_TOTAL_FAILURE;
        break;
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
    case VDMMGR_TR_CD_NOTIFI_MEDIA_STATE_SIM:
        ETG_TRACE_USR1(( "vSimulateEventAfterOsalNotification( ): CD Media State: %d", ETG_ENUM(OSAL_NOTIFI_MEDIA_STATE, u16NotificationData) ));
        if( poMain && poMain->_vdmmgr_poCdctrlIf )
        {
            poMain->_vdmmgr_poCdctrlIf->vSetMediaStateForSimulation( u16NotificationData );
        }
        oMask = VDMMGR_EVENT_MASK_MEDIA_STATE;
        break;
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
    case VDMMGR_TR_CD_NOTIFI_DEVICE_STATE_SIM:
        ETG_TRACE_USR1(( "vSimulateEventAfterOsalNotification( ): CD Device State: %d", ETG_ENUM(OSAL_NOTIFI_DEVICE_STATE, u16NotificationData) ));
        if( poMain && poMain->_vdmmgr_poCdctrlIf )
        {
            poMain->_vdmmgr_poCdctrlIf->vSetDeviceStateForSimulation( u16NotificationData );
        }
        oMask = VDMMGR_EVENT_MASK_CD_DEVICE_STATE;
        break;
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
    default:    // do nothing
        break;
    }

    if( oMask != 0)
    {
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        if(   ( u8MsgCode == VDMMGR_TR_CD_NOTIFI_MEDIA_CHANGE_SIM )
              || ( u8MsgCode == VDMMGR_TR_CD_NOTIFI_TOTAL_FAILURE_SIM )
              || ( u8MsgCode == VDMMGR_TR_CD_NOTIFI_MEDIA_STATE_SIM )
              || ( u8MsgCode == VDMMGR_TR_CD_NOTIFI_DEVICE_STATE_SIM )
              )
        {
            s32Success = OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent );
        }
        else if( u8MsgCode == VDMMGR_TR_CD_NOTIFI_DEFECT_SIM )
        {
            if( poMain && poMain->_vdmmgr_poErrorIf )
            {
                poMain->_vdmmgr_poErrorIf->vSendDefectEvent( VDMMGR_ERRORIF_EVENT_MASK_DEFECT_CD, u16NotificationData );
            }
        }
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
        if( s32Success == OSAL_OK )
        {
            s32Success = OSAL_s32EventPost( hCbkEvent, oMask, OSAL_EN_EVENTMASK_OR );  /*lint !e644 PQM_authorized_86 var is initialized */
            TRACE_On_OSAL_Error( s32Success, _CLASS )
                    s32Success = OSAL_s32EventClose( hCbkEvent );
            TRACE_On_OSAL_Error( s32Success, _CLASS )
        }
    }
}

