/************************************************************************
* FILE:         vdmmgr_main.cpp
* PROJECT:      ARION Prototyp
* SW-COMPONENT: Virtual Device MerdiaManager
*----------------------------------------------------------------------
*
* DESCRIPTION: VD MediaManager
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author             | Modification
* 29.08.05  | CM-DI/ESA2 Fiebing | initial version
*
*************************************************************************/

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
#include "Config.h"
#include <vector>

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_MEDIAMANAGER_TIMER
#include "trcGenProj/Header/vdmmgr_timer.cpp.trc.h"
#endif

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"         // use AIL template with MessageMaps
#define AHL_S_IMPORT_INTERFACE_NOTIFICTABLE
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define VDMMGR_S_IMPORT_INTERFACE_MSG
#include "vdmmgr_if.h"                       // For VD MMgr interface

#include "vdmmgr_main.h"                  // For the pointer to main app
#include "vdmmgr_timer.h"
#include "vdmmgr_service.h"
#include "vdmmgr_cdctrlif.h"
#include "vdmmgr_trace.h"


/******************************************************************************
FUNCTION:      vdmmgr_tclTimer( )
DESCRIPTION:   constructor
PARAMETER:     none
RETURNVALUE:   none
*******************************************************************************/
vdmmgr_tclTimer::vdmmgr_tclTimer( )
{
    // initialise members -----------------------------------------
   _phTimerHandle    = OSAL_C_INVALID_HANDLE;
   _pfOnTimer        = NULL;

   _u32Timeout       = 0;

   _u32TimerInterval = 0;
}


/******************************************************************************
FUNCTION:      Create( tU32 u32TimerMSec, tU32 u32TimerIntervalMSec, 
                                OSAL_tpfCallback pfOnTimer, vdmmgr_tclservice* poMmgrService )
DESCRIPTION:   creates a timer
PARAMETER:     
RETURNVALUE:   none
*******************************************************************************/
tBool vdmmgr_tclTimer::Create( tU32 u32TimerMSec, tU32 u32TimerIntervalMSec,
                               vdmmgr_tclservice* poMmgrService, tU32 u32Arg )
{
   tBool   bRet = TRUE;

   ETG_TRACE_USR1(( "Create( ): Create timer object for timer type: %d", ETG_ENUM(TIMER_TYPE, u32Arg) ));

    // initialise members -----------------------------------------
   _u32Timeout       = u32TimerMSec;
   _u32TimerInterval = u32TimerIntervalMSec;
   _phTimerHandle    = OSAL_C_INVALID_HANDLE;

   switch( u32Arg )
   {
      case VDMMGR_TIMERTYPE_AUTOMATIC_REINSERT:
         // create timer and check return value
         if( OSAL_s32TimerCreate( vOnTimeoutCdAutomaticReinsert, (tPVoid)poMmgrService , &_phTimerHandle ) != OSAL_OK )
         {
            TRACE_OSAL_ERROR( VDMMGR_TR_TIMER );
            _phTimerHandle = OSAL_C_INVALID_HANDLE;
            bRet = FALSE;
         }
         break;
      case VDMMGR_TIMERTYPE_ENSURE_CD_NO_MEDIA:
         // create timer and check return value
         if( OSAL_s32TimerCreate( vOnTimeoutEnsureCdNoMedia, (tPVoid)poMmgrService , &_phTimerHandle ) != OSAL_OK )
         {
            TRACE_OSAL_ERROR( VDMMGR_TR_TIMER );
            _phTimerHandle = OSAL_C_INVALID_HANDLE;
            bRet = FALSE;
         }
         break;
      case VDMMGR_TIMERTYPE_CD_GET_TEMPERATURE:
         // create timer and check return value
         if( OSAL_s32TimerCreate( vOnCdTemperatureTimerTimeout, (tPVoid)poMmgrService , &_phTimerHandle ) != OSAL_OK )
         {
            TRACE_OSAL_ERROR( VDMMGR_TR_TIMER );
            _phTimerHandle = OSAL_C_INVALID_HANDLE;
            bRet = FALSE;
         }
         break;
      default:    // Do nothing
         break;
   }

   return bRet;
}


/************************************************************************
FUNCTION:      ~vdmmgr_tclTimer()
DESCRIPTION:   destuctor: stops and deletes the timer
PARAMETER:     none
RETURNVALUE:   none
************************************************************************/
vdmmgr_tclTimer::~vdmmgr_tclTimer()
{
   tS32     s32RetVal;

   if( _phTimerHandle != OSAL_C_INVALID_HANDLE )
   {
      //stop the timer
      s32RetVal = OSAL_s32TimerSetTime( _phTimerHandle, 0, 0 );
      if( s32RetVal == OSAL_OK )
      {
         // delete the timer
         if( OSAL_s32TimerDelete( _phTimerHandle ) != OSAL_OK )
         {
            TRACE_OSAL_ERROR( VDMMGR_TR_TIMER );
         }
      }
      else
      {
         TRACE_OSAL_ERROR( VDMMGR_TR_TIMER );
      }
   }
   return;
}


/******************************************************************************
FUNCTION:      bStart( );
DESCRIPTION:   starts the timer 
PARAMETER:     void
RETURNVALUE:   void
*******************************************************************************/
tBool vdmmgr_tclTimer::bStart( )
{
   tS32  s32RetVal;

   if( _phTimerHandle != OSAL_C_INVALID_HANDLE )
   {
      ETG_TRACE_USR1(( "bStart( ): Start timer" ));
      s32RetVal = OSAL_s32TimerSetTime( _phTimerHandle, _u32Timeout, _u32TimerInterval );
      if( s32RetVal != OSAL_OK )
      {
         ETG_TRACE_ERR(( "bStart( ): Error: Start timer; Returns not OK" ));
      }
   }
   else
   {
      ETG_TRACE_ERR(( "bStart( ): Error: Start timer; No timer handle" ));
      s32RetVal = FALSE;
   }
   return (tBool)s32RetVal;
}


/************************************************************************
FUNCTION:      bReset( )
DESCRIPTION:   restarts the timer 
PARAMETER:     void
RETURNVALUE:   tBool
************************************************************************/
tBool vdmmgr_tclTimer::bReset( )
{
   tS32     s32RetVal;

   if( _phTimerHandle != OSAL_C_INVALID_HANDLE )
   {
      ETG_TRACE_USR1(( "bReset( ): Reset timer" ));
      s32RetVal = OSAL_s32TimerSetTime( _phTimerHandle, _u32Timeout, _u32TimerInterval );
      if(s32RetVal != OSAL_OK )
      {
         ETG_TRACE_ERR(( "bReset( ): Error: Reset timer; Returns not OK" ));
         TRACE_OSAL_ERROR( VDMMGR_TR_TIMER );
      }
   }
   else
   {
      ETG_TRACE_ERR(( "bReset( ): Error: Reset timer; No timer handle" ));
      s32RetVal = FALSE;
   }
   return (tBool)s32RetVal;
}


/************************************************************************
FUNCTION:      bStop( )
DESCRIPTION:   stops the timer to supervise the connection
PARAMETER:     void
RETURNVALUE:   tBool
************************************************************************/
tBool vdmmgr_tclTimer::bStop( )
{
   tS32     s32RetVal;

   if( _phTimerHandle != OSAL_C_INVALID_HANDLE )
   {
      ETG_TRACE_USR1(( "bStop( ): Stop timer" ));
      //stop the timer
      s32RetVal = OSAL_s32TimerSetTime( _phTimerHandle, 0, 0 );
      if( s32RetVal != OSAL_OK )
      {
         TRACE_OSAL_ERROR( VDMMGR_TR_TIMER );
         ETG_TRACE_ERR(( "bStop( ): Error: Stop timer; Returns not OK" ));
      }
   }
   else
   {
      ETG_TRACE_ERR(( "bStop( ): Error: Stop timer; No timer handle" ));
      s32RetVal = FALSE;
   }
   return ( s32RetVal == OSAL_OK );
}


/******************************************************************************
FUNCTION:      vDelete( );
DESCRIPTION:   Stops and deletes the timer 
PARAMETER:     void
RETURNVALUE:   void
*******************************************************************************/
tVoid vdmmgr_tclTimer::vDelete( )
{
   tS32     s32RetVal;

   if( _phTimerHandle != OSAL_C_INVALID_HANDLE )
   {
      //stop the timer
      s32RetVal = OSAL_s32TimerSetTime( _phTimerHandle, 0, 0 );
      if( s32RetVal == OSAL_OK )
      {
         // delete the timer
         if( OSAL_s32TimerDelete( _phTimerHandle ) != OSAL_OK )
         {
            TRACE_OSAL_ERROR( VDMMGR_TR_TIMER );
         }
      }
      else
      {
         TRACE_OSAL_ERROR( VDMMGR_TR_TIMER );
      }
   }
   return;
}


///************************************************************************
//FUNCTION:      vOnTimeoutCdAutomaticReinsert(tPVoid arg)
//DESCRIPTION:   callback for timer
//PARAMETER:     arg
//RETURNVALUE:   void
//************************************************************************/
tVoid vdmmgr_tclTimer::vOnTimeoutCdAutomaticReinsert( tPVoid arg )
{
   tS32                 s32Success;
   OSAL_tEventHandle    hCbkEvent;
   vdmmgr_tclservice*   poMmgrService = (vdmmgr_tclservice*)arg;

   if( poMmgrService )
   {
      ETG_TRACE_USR1(( "vOnTimeoutCdAutomaticReinsert( ): Start" ));
      if( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
      {
         s32Success = OSAL_s32EventPost( hCbkEvent, VDMMGR_EVENT_MASK_CMD_AUTOREINSERTCD,
                                         OSAL_EN_EVENTMASK_OR );
         if( s32Success == OSAL_ERROR )
         {
            ETG_TRACE_ERR(( "vOnTimeoutCdAutomaticReinsert( ): Error sending event VDMMGR_EVENT_MASK_CMD_AUTOREINSERTCD" ));
            TRACE_OSAL_ERROR( VDMMGR_TR_TIMER );
         }
         
         s32Success = OSAL_s32EventClose( hCbkEvent );
         if(s32Success == OSAL_ERROR)
         { 
            ETG_TRACE_ERR(( "vOnTimeoutCdAutomaticReinsert( ): Error closing event handle" ));
         }
      }
      else
      {
         ETG_TRACE_ERR(( "vOnTimeoutCdAutomaticReinsert( ): Error opening event handle" ));
         TRACE_OSAL_ERROR( VDMMGR_TR_TIMER );
      }
      ETG_TRACE_USR1(( "vOnTimeoutCdAutomaticReinsert( ): End" ));
   }
}


///************************************************************************
//FUNCTION:      vOnTimeoutEnsureCdNoMedia(tPVoid arg)
//DESCRIPTION:   callback for timer
//PARAMETER:     arg
//RETURNVALUE:   void
//************************************************************************/
tVoid vdmmgr_tclTimer::vOnTimeoutEnsureCdNoMedia( tPVoid arg )
{
   vdmmgr_tclservice*      poMmgrService = (vdmmgr_tclservice*)arg;

   if( poMmgrService )
   {
      poMmgrService->vNewCdType( MMGR_IN_SLOT );
      poMmgrService->vNewCdType( MMGR_NO_MEDIA );
      ETG_TRACE_ERR(( "vOnTimeoutEnsureCdNoMedia( ): Call of timeout function (ensure CD IN_SLOT and NO_MEDIA) due to missing info from Tanashin CD drive" ));
   }
}


tVoid vdmmgr_tclTimer::vOnCdTemperatureTimerTimeout( tPVoid pvArg )
{
   OSAL_tEventHandle hCbkEvent;

   FOR_FUTURE_USE( pvArg );

   if( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
   {
      (tVoid)OSAL_s32EventPost( hCbkEvent, VDMMGR_EVENT_MASK_CD_TEMP_TIMEOUT, OSAL_EN_EVENTMASK_OR );
      (tVoid)OSAL_s32EventClose( hCbkEvent );
   }
}

