/************************************************************************
* FILE:         vdmmgr_servicecd.cpp
* PROJECT:      MIB2_ENTRY
* SW-COMPONENT: Virtual Device Media Manager
*----------------------------------------------------------------------
*
* DESCRIPTION: VD MediaManager
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2008 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date       | Author             | Modification
* 18.02.2008 | CM-DI/PJ36 Fiebing | initial version
*
*************************************************************************/


#define _CLASS  VDMMGR_TR_SERVICE_CD

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
#include "Config.h"
#include <vector>

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_MEDIAMANAGER_SERVICE_CD
#include "trcGenProj/Header/vdmmgr_servicecd.cpp.trc.h"
#endif

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"                                // use AIL template with MessageMaps
#define AHL_S_IMPORT_INTERFACE_NOTIFICTABLE
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define VDMMGR_S_IMPORT_INTERFACE_MSG
#include "vdmmgr_if.h"                             // For VD MMgr interface

#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_TYPES  // or #define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_STDVISITORS
#include "cfc_fi_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#define DP_DATAPOOL_ID           //CCA_C_U16_APP_MMGR -> commented out due to problems
                                 // in  .vws_GEN/ai_projects/generated/components/datapool/fi/dp_tclSrvIf.h
                                 // and .vws_GEN/ai_projects/generated/components/datapool/fi/dp_tclConfigCm.h
#include "dp_generic_if.h"

/* ************************************************************************** */
/* include the public interface                                               */
/* ************************************************************************** */
#include "vdmmgr_main.h"
#include "vdmmgr_timer.h"        // 'automatic reinsert CD' timer
#include "vdmmgr_service.h"
#include "vdmmgr_trace.h"
#include "vdmmgr_clienthandlerspm.h"
#include "vdmmgr_cdctrlif.h"

   
/******************************************************************************
*  FUNCTION:    vHandleCDInfo( amt_tclServiceData* poMessage )
*
*  DESCRIPTION: react on messages with FID = MPLAY_MMGRFI_C_U16_CDINFO
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleCDInfo( amt_tclServiceData* poMessage )
{
   tU8         u8OpCode;

   u8OpCode = poMessage->u8GetOpCode();

   TRACE_CCA_InMessage( poMessage, _CLASS );

   // --- the opcode tells what we should do ---
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:   // --- upreg: register client ---
      {
         if( !bUpreg( poMessage ) )
         {
            // --- upreg failed: send an error message ---
            vSendError( poMessage, CCA_C_U16_ERROR_UPREG_FAILURE, _CLASS, __LINE__ );
         }
         else
         {
            // --- upreg was successful: every upreg message must be confirmed with a status message.
            // so send the actual state of the server ( concerning the given FID ) to the client ---
            vSendAnswerCDInfo( poMessage, _e8ActCDType, _e8ActCDState, _e8ActCDInsertState, _e8ActCDDeviceState );
         }
         break;
      }
      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG: // --- upreg: unregister client ---
      {
         if( !bRelUpreg(poMessage) )
         {
            // --- relupreg failed: send an error message ---
            vSendError( poMessage, CCA_C_U16_ERROR_RELUPREG_FAILURE, _CLASS, __LINE__ );
         }
         else
         {
            // --- the relupreg was successful so you can't talk to this client anymore.
            // therefore there are normally no actions to be done in this case ---
         }
         break;
      }
      case AMT_C_U8_CCAMSG_OPCODE_GET: // --- the client asks for some data ---
      {
         // send the requested data to the client
         vSendAnswerCDInfo( poMessage, _e8ActCDType, _e8ActCDState, _e8ActCDInsertState, _e8ActCDDeviceState );
         break;
      }
      default:
      {
         // unknown opcode
         TRACE_Warning( "unhandled OpCode", _CLASS );
         vSendError( poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__ );
         break;
      }
   }

   (tVoid)poMessage->bDelete();
}


/******************************************************************************
*  FUNCTION:    vHandleEject
*
*  DESCRIPTION: react on messages with FID = MMGR_C_U16_FKTID_CDEJECT
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleEject( amt_tclServiceData* poMessage )
{
   TRACE_CCA_InMessage(poMessage, _CLASS);

   // --- the opcode tells what we should do ---
   tU8 u8OpCode = poMessage->u8GetOpCode();
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART: // --- the client wants the server to do something ---
      {
         // every METHODSTART message must be confirmed with a METHODRESULT message.
         // so send the actual state of the server ( concerning the given FID ) to the client
         // First check CD device state. It's not possible to eject a media
         // if the device state is not 'MMGR_DEVICE_READY'.
         if( _e8ActCDDeviceState == MMGR_DEVICE_READY )
         {
            if(   ( _e8ActCDType != MMGR_NO_MEDIA )
               && ( _e8ActCDType != MMGR_EJECTING )
               && ( _e8ActCDType != MMGR_IN_SLOT )
              )
            {
               // Only send MMGR_ACCEPTED if a CD is inside
               vSendAnswerCDEject( poMessage, MMGR_ACCEPTED );
            }
            //sending a MMGR_NOTACCEPTED method result when the disc is ejecting or not in the slot might be OK.
            //But have to be discussed with in the team.
            else if(MMGR_EJECTING == _e8ActCDType || MMGR_NO_MEDIA == _e8ActCDType )
            {
            	vSendAnswerCDEject( poMessage, MMGR_NOTACCEPTED );
            }
            // Check if 'OSAL_C_S32_IOCTRL_CDCTRL_EJECTMEDIA' is already active
            if(    poMain && poMain->_vdmmgr_poCdctrlIf
               &&  poMain->_vdmmgr_poCCAClienthandlerspm
               && !poMain->_vdmmgr_poCdctrlIf->bIsEjectActive() )
            {
               // Check if Media isn't already in slot
               if( !poMain->_vdmmgr_poCCAClienthandlerspm->bIsMediaInSlot() )
               {
                  // Store CD type before eject -> fast CD recognition in case of 'auto insert'.
                  _e8CDTypeBeforeEject = _e8ActCDType;
                  ETG_TRACE_USR4(( "vHandleEject( ): Set _e8CDTypeBeforeEject to: %d, before start of CD eject", ETG_ENUM(CD_TYPE, _e8CDTypeBeforeEject) ));
                  vNewStateInsert( MMGR_INSERT_AFTERON,  FALSE );       // Set InsertState to AfterOn
                  vNewCdType     ( MMGR_EJECTING,        FALSE );       // Set CD type to 'Eject'.
                  vNewCdState    ( MMGR_MEDIA_NOT_READY, TRUE );        // Set CD state to 'media not ready'.
               }
               // Start eject.
               (tVoid)poMain->_vdmmgr_poCdctrlIf->bEjectInThread( );
            }
         }
         else
         {
            // Device is not ready
            vSendAnswerCDEject( poMessage, MMGR_DEVICENOTREADY );
         }
         break;
      }

      default:
      {
         // unknown opcode
         TRACE_Warning("unhandled OpCode", _CLASS);
         vSendError(poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}

/******************************************************************************
*  FUNCTION:    vHandleInsert
*
*  DESCRIPTION: react on messages with FID = MMGR_C_U16_FKTID_CDINSERT
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleInsert( amt_tclServiceData* poMessage )
{
   TRACE_CCA_InMessage(poMessage, _CLASS);
   // --- the opcode tells what we should do ---
   tU8 u8OpCode = poMessage->u8GetOpCode();
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART: // --- the client wants the server to do something ---
      {
         // Get loaderState
         T_e8_DiagLoadStatus e8LoaderState  = poMain->_vdmmgr_poCdctrlIf->e8GetLoaderState( );
         if(e8LoaderState == MMGR_MEDIA_IN_SLOT)
         {
            vSendAnswerCDInsert( poMessage, MMGR_ACCEPTED );
            if(poMain->_vdmmgr_poCdctrlIf->bCloseDoor())
            {
              ETG_TRACE_USR4(( "vHandleInsert( ): CloseDoor success"));
            }
            else
            {
              vSendAnswerCDInsert( poMessage, MMGR_DEVICENOTREADY );
            }
         }
         else
         {
            // Device is not ready
            vSendAnswerCDInsert( poMessage, MMGR_DEVICENOTREADY );
         }
         break;
      }
      default:
      {
         // unknown opcode
         TRACE_Warning("unhandled OpCode", _CLASS);
         vSendError(poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}

/******************************************************************************
*  FUNCTION:    vHandleCDTemp( amt_tclServiceData* poMessage )
*
*  DESCRIPTION: react on messages with FID = MMGR_C_U16_FKTID_CDTEMP
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleCDTemp( amt_tclServiceData* poMessage )
{
   tU8                        u8OpCode;
   tS16                       s16CDTemp;
   T_e8DriveTempSensorStatus  DriveTempSensorStatus = DRIVE_MOUNTED_OK;

   u8OpCode = poMessage->u8GetOpCode();

   TRACE_CCA_InMessage(poMessage, _CLASS);

   // The opcode tells what we should do
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:   // Upreg: register client
      {
         if( !bUpreg( poMessage ) )
         {
            // Upreg failed: send an error message
            vSendError( poMessage, CCA_C_U16_ERROR_UPREG_FAILURE, _CLASS, __LINE__ );
         }
         else
         {
            // Upreg was successful: every upreg message must be confirmed with a status message.
            // so send the actual state of the server ( concerning the given FID ) to the client
            if( bIsCDDriveMounted() )                                            // Check if a CD drive is mounted
            {                                                                    // Get CD drive temperature
               // Get temperature and send it. Then start timer in CD Control thread
               // Do this in CD Control thread because the drive function is serial (-> blocking!) 
               // in GEN2 platform 
               // 1. Store necessary message information for a later send of the status
               vCopyMsgDataToStoreForAnswer( poMessage, &_MessageDataForGetFirstCdTemp );
               // Send event to get temperature in CD control thread
               OSAL_tEventHandle hCbkEvent;
               if( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
               {
                  (tVoid)OSAL_s32EventPost( hCbkEvent, VDMMGR_EVENT_MASK_CD_GET_FIRST_TEMP, OSAL_EN_EVENTMASK_OR );
                  (tVoid)OSAL_s32EventClose( hCbkEvent );
               }
            }
            else
            {
               s16CDTemp = 0;
               DriveTempSensorStatus = DRIVE_NOT_MOUNTED;
               vSendAnswerCDTemp( poMessage, s16CDTemp, DriveTempSensorStatus );    // Send the requested data to the client
            }
         }
         break;
      }
      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG: // upreg: unregister client
      {
         if( !bRelUpreg(poMessage) )
         {
            // relupreg failed: send an error message
            vSendError( poMessage, CCA_C_U16_ERROR_RELUPREG_FAILURE, _CLASS, __LINE__ );
         }
         else
         {
            // The RelUpreg was successful so you can't talk to this client anymore.
            // Therefore there are normally no actions to be done in this case
            // Stop timeout in CD control if this is the last client
            ahl_tNotification* pNot;                                             // Pointer to Notification table
            pNot= poNotTable->poGetNotificationList( MPLAY_MMGRFI_C_U16_CDTEMP ); // Get Notification list
            if(   ( pNot == OSAL_NULL )                                          // No other client
               && ( poMain && poMain->_vdmmgr_poCdctrlIf )                       // Check pointer
              )
            {
               poMain->_vdmmgr_poCdctrlIf->vStopTemperatureTimeout();            // Stop timer
            }
         }
         break;
      }
      case AMT_C_U8_CCAMSG_OPCODE_GET:                                           // The client asks for some data
      {
         if( poMain && poMain->_vdmmgr_poCdctrlIf )
         {
            if( bIsCDDriveMounted() )                                            // Check if a CD drive is mounted
            {                                                                    // Get CD drive temperature
               s16CDTemp = poMain->_vdmmgr_poCdctrlIf->s16GetTemperature( &DriveTempSensorStatus );
            }
            else
            {
               s16CDTemp = 0;
               DriveTempSensorStatus = DRIVE_NOT_MOUNTED;
            }
            vSendAnswerCDTemp( poMessage, s16CDTemp, DriveTempSensorStatus );    // Send the requested data to the client
         }
         break;
      }
      default:
      {
         // unknown opcode
         TRACE_Warning( "unhandled OpCode", _CLASS );
         vSendError( poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__ );
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}


/*************************************************************************
*
* FUNCTION:    tVoid vSendStatusCDInfo( T_e8_CdType e8ActCDType,
*                                       T_e8_CdState e8ActCDState,
*                                       T_e8_InsertState e8ActCDInsertState,
*                                       T_e8DeviceState  e8DeviceState )
* 
* DESCRIPTION: sends an status - message
*
* PARAMETER:   
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid vdmmgr_tclservice::vSendStatusCDInfo( T_e8_CdType e8ActCDType, T_e8_CdState e8ActCDState, T_e8_InsertState e8ActCDInsertState, T_e8DeviceState  e8DeviceState )
{
   ETG_TRACE_USR4(( "vSendStatusCDInfo: e8ActCDType=%d, e8ActCDState=%d, e8ActCDInsertState=%d, e8DeviceState=%d", ETG_ENUM(CD_TYPE, e8ActCDType), ETG_ENUM(MEDIA_STATE_CODE, e8ActCDState), ETG_ENUM(CD_INSERT_STATE, e8ActCDInsertState), ETG_ENUM(CD_DVD_DEVICE_STATE, e8DeviceState) ));
   ETG_TRACE_USR4(( "szcUniqueDeviceID: %s",_szcUniqueDeviceID));
   (tVoid)ahl_bEnterCritical( hNotTableSem );

   if( _bSendStatusOK )    // Test if service is available
   {
      // --- run through the table to find all registered clients ---
      for( ahl_tNotification* pNot= poNotTable->poGetNotificationList( MPLAY_MMGRFI_C_U16_CDINFO );
           pNot != OSAL_NULL; pNot=pNot->pNext )
      {
         // Prepare method result data
         mplay_mmgrfi_tclMsgCDInfoStatus   oResultData;
         // Copy data
         oResultData.CDType.enType        = (mplay_fi_tcl_e8_CdType::tenType)e8ActCDType;
         oResultData.MediaState.enType    = (mplay_fi_tcl_e8_MediaState::tenType)e8ActCDState;
         oResultData.InsertState.enType   = (mplay_fi_tcl_e8_InsertState::tenType)e8ActCDInsertState;
         oResultData.e8DeviceState.enType = (mplay_fi_tcl_e8DeviceState::tenType)e8DeviceState;
         oResultData.szcUniqueID.bSet((tCString)_szcUniqueDeviceID,mplay_fi_tclString::FI_EN_UTF8);

         tU8                     u8OpCode;
         tU8                     u8Cnt;
         // Init result array
         for( u8Cnt = 0; u8Cnt< MMGR_MAX_DRIVEVERSIONLENGTH; u8Cnt++)
         {
            _diagDriveVersion.au8DriveVersion[u8Cnt] = VDMMGR_DIAG_INIT;
         }
         if( poMain && poMain->_vdmmgr_poCdctrlIf )
         {
            poMain->_vdmmgr_poCdctrlIf->vDiagGetDriveVersion( &_diagDriveVersion );
         }

         OSAL_trDriveVersion  rDriveVersion;
         for( u8Cnt = 0; u8Cnt< MMGR_MAX_DRIVEVERSIONLENGTH; u8Cnt++ )
         {
             rDriveVersion.au8ModelNumber[u8Cnt] = _diagDriveVersion.au8DriveVersion[u8Cnt];
         }

         oResultData.DriveVersion.bSet((tCString)rDriveVersion.au8ModelNumber,mplay_fi_tclString::FI_EN_UTF8);
         ETG_TRACE_USR4(( "vSendStatusCDInfo( ): Get drive versions HW: %s", oResultData.DriveVersion.szGet(mplay_fi_tclString::FI_EN_UTF8)));

         // construct result message
         fi_tclVisitorMessage oResultMsg( oResultData );
         vSendStatusVisitorMessage( pNot, MPLAY_MMGRFI_C_U16_CDINFO, &oResultMsg );
      }
   }
   (tVoid)ahl_bReleaseCritical( hNotTableSem );
}


/******************************************************************************
*
* FUNCTION:    vSendStatusCDTemp( tS16 s16CDTemp, T_e8DriveTempSensorStatus DriveTempSensorStatus )
* 
* DESCRIPTION: sends an status - message
*
* PARAMETER:   Function - ID and data to send
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendStatusCDTemp( tS16 s16CDTemp, T_e8DriveTempSensorStatus DriveTempSensorStatus )
{
   (tVoid)ahl_bEnterCritical( hNotTableSem );

   if( _bSendStatusOK )                               // Test if service is available
   {
      // Run through the table to find all registered clients
      for( ahl_tNotification* pNot= poNotTable->poGetNotificationList( MPLAY_MMGRFI_C_U16_CDTEMP );
           pNot != OSAL_NULL; pNot=pNot->pNext )
      {
         // Prepare method result data
         mplay_mmgrfi_tclMsgCDTempStatus  oResultData;
         // Copy data
         oResultData.CDTemp        = s16CDTemp;       // Temperature in degree Centigrade in 1K steps;
         // Status of drive temperature sensor
         oResultData.Status.enType = (mplay_fi_tcl_e8DriveTempSensorStatus::tenType)DriveTempSensorStatus;
         // construct result message
         fi_tclVisitorMessage oResultMsg( oResultData );
         vSendStatusVisitorMessage( pNot, MPLAY_MMGRFI_C_U16_CDTEMP, &oResultMsg );
      }
   }
   (tVoid)ahl_bReleaseCritical( hNotTableSem );
}


/******************************************************************************
*
* FUNCTION:    vSendAnswerCDInfo( const amt_tclServiceData* poMessage,
*                                 T_e8_CdType      e8ActCDType,
*                                 T_e8_CdState     e8ActCDState,
*                                 T_e8_InsertState e8ActCDInsertState,
*                                 T_e8DeviceState  e8DeviceState )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerCDInfo( const amt_tclServiceData* poMessage, T_e8_CdType e8ActCDType, T_e8_CdState e8ActCDState, 
                                            T_e8_InsertState e8ActCDInsertState, T_e8DeviceState  e8DeviceState )
{
   // Prepare method result data
   mplay_mmgrfi_tclMsgCDInfoStatus   oResultData;

   ETG_TRACE_USR4(( "vSendAnswerCDInfo: e8ActCDType=%d, e8ActCDState=%d, e8ActCDInsertState=%d, e8DeviceState=%d ", 
      e8ActCDType, e8ActCDState, e8ActCDInsertState, e8DeviceState ));

   // Copy data
   oResultData.CDType.enType        = (mplay_fi_tcl_e8_CdType::tenType)e8ActCDType;
   oResultData.MediaState.enType    = (mplay_fi_tcl_e8_MediaState::tenType)e8ActCDState;
   oResultData.InsertState.enType   = (mplay_fi_tcl_e8_InsertState::tenType)e8ActCDInsertState;
   oResultData.e8DeviceState.enType = (mplay_fi_tcl_e8DeviceState::tenType)e8DeviceState;
   oResultData.szcUniqueID.bSet((tCString)_szcUniqueDeviceID,mplay_fi_tclString::FI_EN_UTF8);

   tU8                     u8OpCode;
   tU8                     u8Cnt;
   OSAL_trDriveVersion  rDriveVersion;

   for( u8Cnt = 0; u8Cnt< MMGR_MAX_DRIVEVERSIONLENGTH; u8Cnt++ )
   {
      rDriveVersion.au8ModelNumber[u8Cnt] = _diagDriveVersion.au8DriveVersion[u8Cnt];
   }

   oResultData.DriveVersion.bSet((tCString)rDriveVersion.au8ModelNumber,mplay_fi_tclString::FI_EN_UTF8);
   ETG_TRACE_USR4(( "vSendAnswerCDInfo( ): Get drive versions HW: %s", oResultData.DriveVersion.szGet(mplay_fi_tclString::FI_EN_UTF8) ));

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   // Send it to the client
   vSendStatusAnswerVisitorMessage( poMessage, &oResultMsg );
}


/******************************************************************************
*
* FUNCTION:    vSendAnswerCDEject( const amt_tclServiceData* poMessage,
*                                  T_e8_MMgr_MethodResult enMsgResult )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerCDEject( const amt_tclServiceData* poMessage, 
                                             T_e8_MMgr_MethodResult enMsgResult )
{
   // Prepare method result data
   mplay_mmgrfi_tclMsgCDEjectMethodResult  oResultData;
   // Copy data
   oResultData.ServiceStatus.enType    = (mplay_fi_tcl_e8_MMgr_MethodResult::tenType)enMsgResult;

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   // Send it to the client
   vSendMethodResultVisitorMessage( poMessage, &oResultMsg );
}

/******************************************************************************
*
* FUNCTION:    vSendAnswerCDInsert( const amt_tclServiceData* poMessage,
*                                  T_e8_MMgr_MethodResult enMsgResult )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerCDInsert( const amt_tclServiceData* poMessage,
                                             T_e8_MMgr_MethodResult enMsgResult )
{
   // Prepare method result data
   mplay_mmgrfi_tclMsgCDEjectMethodResult  oResultData;
   // Copy data
   oResultData.ServiceStatus.enType    = (mplay_fi_tcl_e8_MMgr_MethodResult::tenType)enMsgResult;

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   // Send it to the client
   vSendMethodResultVisitorMessage( poMessage, &oResultMsg );
}

/******************************************************************************
*
* FUNCTION:    vSendAnswerCTemp( const amt_tclServiceData* poMessage, tS16 s16CDTemp,
*                                T_e8DriveTempSensorStatus DriveTempSensorStatus,
*                                const trMessageDataForMethodResult* poMsgData )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerCDTemp( const amt_tclServiceData* poMessage, tS16 s16CDTemp,
                                            T_e8DriveTempSensorStatus DriveTempSensorStatus,
                                            const trMessageDataForMethodResult* poMsgData/*=NULL*/ )
{
   // Prepare method result data
   mplay_mmgrfi_tclMsgCDTempStatus  oResultData;
   // Copy data
   oResultData.CDTemp        = s16CDTemp;
   oResultData.Status.enType = (mplay_fi_tcl_e8DriveTempSensorStatus::tenType)DriveTempSensorStatus;

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   if( poMessage )         // Check if there is a original message
   {
      // Send it to the client
      vSendStatusAnswerVisitorMessage( poMessage, &oResultMsg );
   }
   else if( poMsgData )    // ... or a a copy of the used parameter
   {
      // Send it to the client
      vSendStatusAnswerVisitorMessage( NULL, &oResultMsg, poMsgData );
   }
}


tVoid vdmmgr_tclservice::vNewCdType( T_e8_CdType enNewCDType, tBool bSendImmediately/*=TRUE*/ )
{
   ETG_TRACE_USR3(( "vNewCdType(): New CDType=%d; Actual CDType=%d, bSendImmediately=%d",
                    ETG_ENUM(CD_TYPE, enNewCDType), ETG_ENUM(CD_TYPE, _e8ActCDType), ETG_ENUM(BOOL, bSendImmediately) ));

   //Check for a change in CDType
   if( _e8ActCDType !=  enNewCDType )
   {
      _e8ActCDType = enNewCDType;
      // Give new CD type to clients.
      if( bSendImmediately )  // Check if it shoud be sent immediately (default) or wait for an additional change(s)
      {
         vSendStatusCDInfo( _e8ActCDType, _e8ActCDState, _e8ActCDInsertState, _e8ActCDDeviceState );
      }
   }
}


tVoid vdmmgr_tclservice::vNewCdState( T_e8_CdState e8NewCDState, tBool bSendImmediately/*=TRUE*/ )
{
   ETG_TRACE_USR3(( "vNewCdState(): New CDState=%d; Actual CDState=%d, bSendImmediately=%d",
                    ETG_ENUM(MEDIA_STATE_CODE, e8NewCDState), ETG_ENUM(MEDIA_STATE_CODE, _e8ActCDState), ETG_ENUM(BOOL, bSendImmediately) ));
   if( _e8ActCDState != e8NewCDState )
   {
      _e8ActCDState = e8NewCDState;
      // Give new CD state to clients.
      if( bSendImmediately )  // Check if it shoud be sent immediately (default) or wait for an additional change(s)
      {
         vSendStatusCDInfo( _e8ActCDType, _e8ActCDState, _e8ActCDInsertState, _e8ActCDDeviceState );
      }
   }
}


tVoid vdmmgr_tclservice::vNewCdDeviceState( T_e8DeviceState e8NewCdDeviceState, tBool bSendImmediately/*=TRUE*/  )
{
   ETG_TRACE_USR2(( "vNewCdDeviceState(): New CdDeviceState%d; Actual CdDeviceState=%d, bSendImmediately=%d",
                    ETG_ENUM(CD_DVD_DEVICE_STATE, e8NewCdDeviceState), ETG_ENUM(CD_DVD_DEVICE_STATE, _e8ActCDDeviceState), ETG_ENUM(BOOL, bSendImmediately) ));
   if( _e8ActCDDeviceState != e8NewCdDeviceState )
   {
      // Copy new CD device state
      _e8ActCDDeviceState = e8NewCdDeviceState;
      if( bSendImmediately )  // Check if it should be sent immediately (default) or wait for an additional change(s)
      {
         vSendStatusCDInfo( _e8ActCDType, _e8ActCDState, _e8ActCDInsertState, _e8ActCDDeviceState );
      }
   }
}


tVoid vdmmgr_tclservice::vNewStateInsert( T_e8_InsertState e8NewCDInsertState, tBool bSendImmediately/*=TRUE*/  )
{
   ETG_TRACE_USR3(( "vNewStateInsert(): New StateInsert=%d; Actual StateInsert=%d, bSendImmediately=%d", ETG_ENUM(CD_INSERT_STATE, e8NewCDInsertState), ETG_ENUM(CD_INSERT_STATE, _e8ActCDInsertState), ETG_ENUM(BOOL, bSendImmediately) ));
   // Send new status only for a different value.
   if( _e8ActCDInsertState != e8NewCDInsertState )
   {
      _e8ActCDInsertState = e8NewCDInsertState;
      // Give new CD insert state to clients.
      if( bSendImmediately )  // Check if it shoud be sent immediately (default) or wait for an additional change(s)
      {
         vSendStatusCDInfo( _e8ActCDType, _e8ActCDState, _e8ActCDInsertState, _e8ActCDDeviceState );
      }
   }
}


tVoid vdmmgr_tclservice::vSendCdDriveErrorInformation( mplay_fi_tcl_e16_DriveErrorValue::tenType e16ErrorValue )
{
   mplay_fi_tcl_DriveErrorInfo    clDriveErrorInfo;

   ETG_TRACE_USR3(( "vSendCdDriveErrorInformation(): e16ErrorValue=%d", ETG_ENUM(DRIVE_ERROR_INFO, e16ErrorValue) ));

   clDriveErrorInfo.e16Drive.enType       = mplay_fi_tcl_e16_DiagDrv::FI_EN_MMGR_CD;
   clDriveErrorInfo.e16ErrorValue.enType  = e16ErrorValue;

   vSendStatusDriveErrorInformation( &clDriveErrorInfo );
}


tVoid vdmmgr_tclservice::vCloseDoorAutoInsert( tVoid )
{
   T_e8_DiagLoadStatus   e8LoadStatus;

   ETG_TRACE_USR4(( "vCloseDoorAutoInsert(): Enter" ));

   // Init e8LoadStatus -> lint
   e8LoadStatus = MMGR_NO_MEDIA_IN_DRIVE;

   if( poMain && poMain->_vdmmgr_poCdctrlIf )
   {
      // First get loader state
      e8LoadStatus = poMain->_vdmmgr_poCdctrlIf->e8GetLoaderState( );
      ETG_TRACE_USR4(( "vCloseDoorAutoInsert(): LoaderState=%d", ETG_ENUM(LOADER_STATE, e8LoadStatus) ));
      // Check if CD is in slot (static or ejecting)
      if(   (   ( e8LoadStatus == MMGR_EJECT_IN_PROGRESS )
             && ( _bEjectStartet )
            )
         || ( e8LoadStatus == MMGR_MEDIA_IN_SLOT )
        )
      {
         // Set insert state
         vNewStateInsert( MMGR_INSERT_AUTOMATIC );
         // Write back the media value stored before ejecting the CD. For fast CD 
         // recognition before shutdown. (Only if it's CD ROM or CD-DA)
         if( _e8CDTypeBeforeEject != MMGR_INCORRECT )
         {
            ETG_TRACE_USR4(( "vCloseDoorAutoInsert(): Set back the CD media type recognized before eject was started: _e8CDTypeBeforeEject=%d", ETG_ENUM(CD_TYPE, _e8CDTypeBeforeEject) ));
            vNewCdType( _e8CDTypeBeforeEject );
         }
         // Now draw in the CD
         if( poMain->_vdmmgr_poCdctrlIf->bCloseDoor( ) != TRUE )
         {
            ETG_TRACE_USR4(( "vCloseDoorAutoInsert(): _vdmmgr_poCdctrlIf->bCloseDoor( ) returned with FALSE" ));
            // Check loader state again after failed reinsert
            // Check if CD is no longer in slot (-> MEDIA_EJECTED)
            if( poMain->_vdmmgr_poCdctrlIf->e8GetLoaderState( ) == MMGR_NO_MEDIA_IN_DRIVE )
            {
               ETG_TRACE_USR4(( "vCloseDoorAutoInsert(): No media in drive. Maybe the media was removed after autoinsert was started. Set InsertState to AFTER_ON and Type to NO_MEDIA " ));
               // Maybe the CD was removed after the automatic reinsert was started.
               vNewStateInsert( MMGR_INSERT_AFTERON, FALSE );  // No Media inside -> no automatic reinsert possible
               vNewCdType     ( MMGR_NO_MEDIA,       TRUE );   // Change back the media type
            }
         }
         else
         {
            ETG_TRACE_USR4(( "vCloseDoorAutoInsert(): _vdmmgr_poCdctrlIf->bCloseDoor( ) returned successful" ));
         }
         // Check if the AppStateChange was delayed due to CD insert
         if( _bAppStateChangeWaitForInsert && poMain )
         {
            poMain->vAppStateChanged( poMain->_u32AppState, 0 );  // Inform system that shutdown is OK now. No more delay necessary
            _bAppStateChangeWaitForInsert = FALSE;                // reset marker
         }
      }
   }
}


tVoid vdmmgr_tclservice::vGetLastModeData( tVoid )
{
   OSAL_tIODescriptor      fdLmMmgr;

   ETG_TRACE_USR1(( "vGetLastModeData(): Start" ));
   // Open MMgr-LLM file in FFS
   fdLmMmgr = OSAL_IOOpen( VDMMGR_LMM_FULL_PATHNAME, OSAL_EN_READWRITE );

   // Does file already exist?
   if( fdLmMmgr != OSAL_ERROR )
   {
      // Read persistent data into struct. bReadLmmData has to fill in consistent data
      vReadLastModeData( fdLmMmgr );
      // Close file
      if( OSAL_s32IOClose( fdLmMmgr ) == OSAL_ERROR )
      {
         // Error: not able to close OSAL handle
         ETG_TRACE_ERR(( "vGetLastModeData(): Error: OSAL_s32IOClose() returns with error" ));
      }
   }
   else        // File doesn't exist -> create a new one
   {
      // create file
      vCreateLastModeFile( );
   }
   ETG_TRACE_USR1(( "vGetLastModeData(): End: CD-type: %d(CD_TYPE,8)", ETG_ENUM(CD_TYPE, _e8ActCDType) ));
}


tVoid vdmmgr_tclservice::vReadLastModeData( OSAL_tIODescriptor fdLmMmgr )
{
   tS32                    s32FileSize;
   tPS8                    pas8Buffer    = NULL;
   vdmmgr_trLastModeData   rLastModeData = {0};

   ETG_TRACE_USR1(( "vReadLastModeData( ): Start" ));

   // Check pointer to struct and file descriptor
   if( fdLmMmgr != OSAL_ERROR )
   {
      // get file-size
      s32FileSize = OSALUTIL_s32FGetSize( fdLmMmgr );
      if(    ( s32FileSize != OSAL_ERROR ) 
          && ( s32FileSize >  0 )  
        )
      {
         // create temporary buffer
         pas8Buffer = new tS8[ sizeof(vdmmgr_trLastModeData) ];
         if ( pas8Buffer != NULL )
         {
            // Check for same size. If the size is different memory overwriting is possible
            if( s32FileSize == (tS32)sizeof(vdmmgr_trLastModeData) )
            {
               // Set file pointer
               if ( OSALUTIL_s32FSeek( fdLmMmgr, 0, SEEK_SET ) == OSAL_OK )
               {
                  // Read last mode data
                  if( OSAL_s32IORead( fdLmMmgr, pas8Buffer, (tU32)s32FileSize ) == s32FileSize )
                  {
                     // copy struct from buffer
                     // last mode data header
                     (tVoid)OSAL_pvMemoryCopy( (tPVoid)&rLastModeData, pas8Buffer, sizeof(vdmmgr_trLastModeData) );
                     // check last mode data
                     if( rLastModeData.u16LmmVersion == VDMMGR_LM_MODULE_VERSION )
                     {
                        // Copy CD type to act CD type
                        //lint -save -e788 Info 788;enum constant 'x' not used within defaulted switch
                        switch( _e8ActCDType )
                        {
                           case MMGR_NO_MEDIA:  // NO_MEDIA could be the Init value -> set stored LMM CD Type
                              // There wasn't an OSAL value, so NO_MEDIA is Init value
                              vNewCdType( rLastModeData.e8CDType );
                              break;
                           case MMGR_DATA:      // Inserted CD is already recognized from driver -> Check LMM value; This should never happen
                              vNewCdType( rLastModeData.e8CDType );
                              break;
                           default:  // do nothing for: MMGR_INSERTION, MMGR_INCORRECT, MMGR_INIT, MMGR_AUDIO, MMGR_DATA, MMGR_EJECTING, MMGR_IN_SLOT
                              break;
                        }
                        //lint -restore
                     }
                     else   // Store new version last mode data
                     {
                        ETG_TRACE_USR1(( "vReadLastModeData( ): Store new version last mode data" ));
                        vWriteLastModeData( fdLmMmgr );
                     }
                  }
                  else // Error: OSAL_s32IORead( ) didn't returns with different file size OSALUTIL_s32FGetSize()
                  {
                     ETG_TRACE_ERR(( "vReadLastModeData( ): Error: OSAL_s32IORead( ) didn't returns with different file size than OSALUTIL_s32FGetSize()" ));
                  }
               }
               else   // Error: OSALUTIL_s32FSeek( ) returns with error
               {
                  ETG_TRACE_ERR(( "vReadLastModeData( ): Error: OSALUTIL_s32FSeek( ) returns with error" ));
               }
            }
            else  // different size of LMM file and LMM struct -> maybe new
            {     // LMM version your corrupted LMM file
               ETG_TRACE_USR1(( "vReadLastModeData( ): Store new version last mode data" ));
               vWriteLastModeData( fdLmMmgr );
            }
            delete [] pas8Buffer;
         }
         else   // Error: Fail to create read buffer
         {
            ETG_TRACE_ERR(( "vReadLastModeData( ): Error: Fail to create read buffer" ));
         }
      }
      else   // Error: OSALUTIL_s32FGetSize( ) returns with error
      {
         ETG_TRACE_ERR(( "vReadLastModeData( ): Error: OSALUTIL_s32FGetSize( ) returns with error" ));
      }
   }
   else   // Error: invalid parameter
   {
      ETG_TRACE_ERR(( "vReadLastModeData( ): Error: invalid value: %d for file descriptor", fdLmMmgr ));
   }
}


tVoid vdmmgr_tclservice::vCreateLastModeFile( tVoid ) const
{
   OSAL_tIODescriptor      fdLmMmgr;

   // create new MMgr-LMM file
   ETG_TRACE_USR1(( "vCreateLastModeFile( ): Start" ));
   fdLmMmgr = OSAL_IOCreate( VDMMGR_LMM_FULL_PATHNAME, OSAL_EN_READWRITE );

   // create successful?
   if( fdLmMmgr != OSAL_ERROR )
   {
      // store data in flash
      vWriteLastModeData( fdLmMmgr );
      // Close file
      if( OSAL_s32IOClose( fdLmMmgr ) == OSAL_ERROR )
      {
         // Error: not able to close OSAL handle
         ETG_TRACE_ERR(( "vCreateLastModeFile( ): Error: not able to close OSAL handle" ));
      }
   }
   else  // Error: not able to create file
   {
      ETG_TRACE_ERR(( "vCreateLastModeFile( ): Error: not able to create file" ));
   }
}


tVoid vdmmgr_tclservice::vSetLastModeData( tVoid )
{
   OSAL_tIODescriptor      fdLmMmgr;

   // Set last mode data
   ETG_TRACE_USR1(( "vSetLastModeData( ): Start" ));
   // Open MMGR_LMM file in FFS
   fdLmMmgr = OSAL_IOOpen( VDMMGR_LMM_FULL_PATHNAME, OSAL_EN_READWRITE );
   // Does file exist?
   if( fdLmMmgr != OSAL_ERROR )
   {
      vWriteLastModeData( fdLmMmgr );
      // Close file
      if( OSAL_s32IOClose( fdLmMmgr ) == OSAL_ERROR )
      {
         // Error: not able to close OSAL handle
         ETG_TRACE_ERR(( "vSetLastModeData( ): Error: not able to close OSAL handle" ));
      }
   }
   else  // Error: file doesn't exist -> create a new one
   {
      ETG_TRACE_ERR(( "vSetLastModeData( ): Error: file doesn't exist -> create a new one" ));
      // create file
      vCreateLastModeFile( );
   }
   ETG_TRACE_USR1(( "vSetLastModeData( ): End: CD-type: %d", ETG_ENUM(CD_TYPE, _e8ActCDType) ));
}


tVoid vdmmgr_tclservice::vWriteLastModeData( OSAL_tIODescriptor fdLmMmgr ) const
{
   tS32                    s32BytesWritten;
   vdmmgr_trLastModeData   rLastModeData        = {0};
   tPS8                    pas8Buffer           = NULL;
   tU32                    u32NumOfBytesToWrite = sizeof(vdmmgr_trLastModeData);

   ETG_TRACE_USR1(( "vWriteLastModeData( ): Start" ));
   if( fdLmMmgr != 0)
   {
      // create temporary buffer
      pas8Buffer = new tS8[ u32NumOfBytesToWrite ];
      // check buffer
      if( pas8Buffer != NULL )
      {
         // Set file pointer; check if header exits
         if( OSAL_OK == OSALUTIL_s32FSeek( fdLmMmgr, 0, SEEK_SET ) )
         {
            if( (tU32)OSAL_s32IORead( fdLmMmgr, (tPS8)&rLastModeData, sizeof( vdmmgr_trLastModeData ) )
                == sizeof( vdmmgr_trLastModeData )
              )
            {
               // check last mode data
               if( rLastModeData.u16LmmVersion  == VDMMGR_LM_MODULE_VERSION )
               {
                  // increase write counter
                  ++rLastModeData.u16WriteCount;
               }
            }
         }
         // copy data into struct
         rLastModeData.u16LmmVersion   = VDMMGR_LM_MODULE_VERSION;   // LM module version
         rLastModeData.e8CDType        = _e8ActCDType;               // CD type
         // store data in buffer
         (tVoid)OSAL_pvMemoryCopy( pas8Buffer, (tPVoid)&rLastModeData, sizeof(vdmmgr_trLastModeData) );
         // write data to file
         if( OSALUTIL_s32FSeek( fdLmMmgr, 0, SEEK_SET ) == OSAL_OK )
         {
            s32BytesWritten = OSAL_s32IOWrite( fdLmMmgr, pas8Buffer, sizeof(vdmmgr_trLastModeData) );
            if( s32BytesWritten == OSAL_ERROR )
            {
               // Error: OSAL_s32IOWrite( ) returns with error
               ETG_TRACE_ERR(( "vWriteLastModeData( ): Error: OSAL_s32IOWrite( ) returns with error" ));
            }
         }
         else   // Error: OSALUTIL_s32FSeek( ) returns with error
         {
            ETG_TRACE_ERR(( "vWriteLastModeData( ): Error: OSALUTIL_s32FSeek( ) returns with error" ));
         }
         ETG_TRACE_USR1(( "vWriteLastModeData( ): End: CD-type: %d", ETG_ENUM(CD_TYPE, rLastModeData.e8CDType) ));
         delete [] pas8Buffer;
      }
      else   // Error: Fail to create read buffer
      {
         ETG_TRACE_ERR(( "vWriteLastModeData( ): Error: Fail to create read buffer" ));
      }
   }
   else   // Error: invalid parameter for file descriptor
   {
      ETG_TRACE_ERR(( "vWriteLastModeData( ): Error: invalid value: %d for file descriptor", fdLmMmgr ));
   }
}


tVoid vdmmgr_tclservice::vReadKdsCd( tVoid )
{
   // Check if a CD was inside at startup
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CHECK_CD_DRIVE_MOUNTED
   tU8   u8CdMounted = TRUE;
#ifdef USE_KDS_DEPENDENCY
   dp_tclKdsCMVariantCoding   oHWConfigurationString;
   if( ( oHWConfigurationString.u8GetCD( u8CdMounted ) )  ==  DP_U8_ELEM_STATUS_VALID )
   {
      ETG_TRACE_USR4(( " vReadKdsCd: data pool for u8GetCD() returns with CD drive mounted = %d", ETG_ENUM(BOOL, u8CdMounted) ));
      _e8CdDrivevMounted = ( u8CdMounted ) ? TRUE : FALSE;
   }
   else
   {
      ETG_TRACE_FATAL(( "vReadKdsCd: ERROR: Read data pool for u8GetCD() returns not with: DP_U8_ELEM_STATUS_VALID"));
   }
#endif //USE_WORKAROUND
   // no #else necessary because _e8CdDrivevMounted is set to TRUE in constructor
#endif // VARIANT_S_FTR_ENABLE_MMGR_CHECK_CD_DRIVE_MOUNTED
}


tVoid vdmmgr_tclservice::vSetInsertStateAfterLastModeData( tVoid )
{
   // Check if a CD was inside at startup
   if( _e8ActCDType == MMGR_NO_MEDIA )
   {
      _e8ActCDInsertState = MMGR_INSERT_AFTERON;
   }
   else
   {
      _e8ActCDInsertState = MMGR_INSERT_BEFOREON;
   }
}


T_e8_CdType vdmmgr_tclservice::enGetActCdType( tVoid ) const
{
   return _e8ActCDType;
}


T_e8_CdState vdmmgr_tclservice::enGetActCDState( tVoid ) const
{
   return _e8ActCDState;
}


T_e8_InsertState vdmmgr_tclservice::enGetActCDInsertState( tVoid ) const
{
   return _e8ActCDInsertState;
}


T_e8DeviceState vdmmgr_tclservice::enGetActCDDeviceState( tVoid ) const
{
   return _e8ActCDDeviceState;
}


tBool vdmmgr_tclservice::bIsCDDriveMounted( tVoid ) const
{
   return _e8CdDrivevMounted;
}


tVoid vdmmgr_tclservice::vSetCdTypeBeforeEject( T_e8_CdType e8NewCDTypeBeforeEject )
{
   _e8CDTypeBeforeEject = e8NewCDTypeBeforeEject;
}

tVoid vdmmgr_tclservice::vSetUniqueID(const tChar* szcMediaID)
{

  if (szcMediaID == NULL) {
    memset(&_szcUniqueDeviceID[0], 0, sizeof(_szcUniqueDeviceID));
    ETG_TRACE_USR4(("vSetUniqueID( ) :  !!!!!!!!!!!!! szcMediaID of Optical disk is NULL !!!!!!!!!!!!!"));
  }
  else {
    for(int index = 0; index <= (UNIQUE_ID_LENGTH - 2); index++) {
      _szcUniqueDeviceID[index] = szcMediaID[index];

    }
    _szcUniqueDeviceID[UNIQUE_ID_LENGTH-1] = '\0'; //adds NULL character at end
    ETG_TRACE_USR4(("vSetUniqueID( ) : Got the  _szcUniqueDeviceID of Optical disk %s",_szcUniqueDeviceID));
  }
}
