/************************************************************************
* FILE:         vdmmgr_clienthandler.cpp
* PROJECT:      MIB2_ENTRY
* SW-COMPONENT: Virtual Device Media Manager
*----------------------------------------------------------------------
*
* DESCRIPTION: VD MediaManager
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2006 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author             | Modification
* 20.04.06  | CM-DI/ESA2 Fiebing | initial version
*
*************************************************************************/

#define _CLASS  VDMMGR_TR_CLIENTDIAGLOG

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
#include "Config.h"
#include <map>
#include <vector>

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_MEDIAMANAGER_CLIENTDIAGLOG
#include "trcGenProj/Header/vdmmgr_clienthandlerdiaglog.cpp.trc.h"
#endif

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"         // use AIL template with MessageMaps
#define AHL_S_IMPORT_INTERFACE_NOTIFICTABLE
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define VDMMGR_S_IMPORT_INTERFACE_MSG
#include "vdmmgr_if.h"                       // For VD MMgr interface

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"

#include "vdmmgr_main.h"
#include "vdmmgr_ITC_Container.h"
#include "vdmmgr_ITC_Map.h"
#include "vdmmgr_clienthandlerdiaglog.h"
#include "vdmmgr_timer.h"                    // 'automatic reinsert CD' timer
#include "vdmmgr_service.h"                  // (CCA) send functions
#include "vdmmgr_trace.h"

// +++ MESSAGE MAP: enter the function IDs (FID) and the corresponding functions here.
//     the function will be called when a message with the corresponding FID arrives +++
BEGIN_MSG_MAP(vdmmgr_tclclienthandlerdiaglog, ahl_tclBaseWork)
ON_MESSAGE( MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT,     vHandleSaveTestResult )
ON_MESSAGE( MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT, vHandleSendNextTestResult )
END_MSG_MAP()


/*************************************************************************
*
* FUNCTION: vdmmgr_tclclienthandlerdiaglog::vdmmgr_tclclienthandlerdiaglog(vdmmgr_tclApp* poMainApp)
* 
* DESCRIPTION: constructor, creates object vdmmgr_tclclienthandlerdiaglog - object
*
* PARAMETER: vdmmgr_tclApp* poMainApp: main - object of this application 
*
* RETURNVALUE: none
*
*************************************************************************/
vdmmgr_tclclienthandlerdiaglog::vdmmgr_tclclienthandlerdiaglog( vdmmgr_tclApp* poVdMMgrMainApp )
{
   tU16  u16Itc;
   // --- this constructor calls ( implicit ) the constructor of the upper class ( framework )
   //     so it registers the combinations of FID and message handler with the framework ---
   poMain            = poVdMMgrMainApp;
   _u16RegID         = AMT_C_U16_REGID_INVALID;
   _bFidReg          = FALSE;
   _bRegAsyncStarted = FALSE;

   // Init ITC container for CD ITCs
   for( u16Itc = ITC_OPTICALDISC_COMUNICATION_TIMEOUT; u16Itc <= ITC_OPTICALDISC_LOAD_ERROR; ++ u16Itc )
   {
      vdmmgr_tclITC_Container* ptclContainer = new vdmmgr_tclITC_Container( u16Itc );//lint !e1524 Warning 1524;new in constructor for class 'vdmmgr_tclclienthandlerdiaglog' which has no explicit destructor
      (tVoid)_ItcMap.bAddITCContainer( ptclContainer );                              // to be deleted in ITCcontainer destructor
   }
}


vdmmgr_tclclienthandlerdiaglog::vdmmgr_tclclienthandlerdiaglog( )
{
}//lint !e1744 member 'x possibly not initialized by private constructor


/******************************************************************************
*
* FUNCTION:tVoid vdmmgr_tclclienthandlerdiaglog::vOnUnknownMessage(amt_tclBaseMessage* poMessage)
* 
* DESCRIPTION: handle unknown message
*
* PARAMETER:  unknown message
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclclienthandlerdiaglog::vOnUnknownMessage(amt_tclBaseMessage* poMessage)
{
   TRACE_On_Invalid_Pointer(poMessage, "poMessage", _CLASS);
   if (poMessage == NULL)
   {
      return;
   }

   TRACE_Warning("Unknown Message", _CLASS);

   tU8 u8Type = poMessage->u8GetType();
   if (u8Type != CCA_C_U8_TYPE_SVCDATA)
   {
      // incompatible to amt_tclServiceData; should not happen
      TRACE_Value(4, u8Type, "incompatible Message Type", _CLASS); 
      return;
   }

   amt_tclServiceData* poMsg = dynamic_cast<amt_tclServiceData*>( poMessage );
   if( poMsg )
   {
      TRACE_CCA_InMessage(poMsg, _CLASS);

      TRACE_On_Invalid_Pointer( poMain, "poMain", _CLASS );
      if( poMain && poMain->_vdmmgr_poCCAService )
      {
         poMain->_vdmmgr_poCCAService->vSendError(poMsg, CCA_C_U16_ERROR_UNKNOWN_FCT_ID, _CLASS, __LINE__);
      }
   }

   (tVoid)poMessage->bDelete();
}


/******************************************************************************
*
* FUNCTION: tVoid vdmmgr_tclclienthandlerdiaglog::vOnNewAppState(tU32 u32OldAppState, tU32 u32AppState)
* 
* DESCRIPTION: handles state change messages form the DiagLog
*
* PARAMETER: old state, new state
*
* RETURNVALUE: void
*
********************************************************************************/
tVoid vdmmgr_tclclienthandlerdiaglog::vOnNewAppState(tU32 u32OldAppState, tU32 u32AppState)
{
   TRACE_SPM_Transition(u32OldAppState, u32AppState, _CLASS);

   if( u32OldAppState == u32AppState )
   {
      return; //switching to the same state
   }

   switch(u32AppState)
   {
      case AMT_C_U32_STATE_NORMAL:
      case AMT_C_U32_STATE_DIAGNOSIS:
      {
         (tVoid)bRegisterForService();
      }
      break;

      case AMT_C_U32_STATE_PAUSE:
      case AMT_C_U32_STATE_OFF:
      break;

      default:
      {
         TRACE_Warning("unhandled power state", _CLASS);
      }
      break;
   }
}




tBool vdmmgr_tclclienthandlerdiaglog::bStoreITCValueInDiagLog( const midw_fi_tcl_TestResult* poDialogTestResult )
{
   tBool                   bRet;

   midw_fi_tcl_TestResult  oTestResult;


   if( _u16RegID != AMT_C_U16_REGID_INVALID )
   {
      bRet = TRUE;
      // Prepare method start data
      midw_diaglogfi_tclMsgSaveTestResultMethodStart oMsgStartData;
      // fill this oTestResult with ITC from data
      oTestResult.TroubleCode   = poDialogTestResult->TroubleCode;
      //fill this oTestResult with result type from data
      oTestResult.Result.enType = poDialogTestResult->Result.enType;
      //add the oTestResult into TestResultList
      oMsgStartData.TestResultList.TestResultList.push_back( oTestResult );

      //put the oMsgStartData into MethodStart message
      //lint {64}   "PQM_authorized_multi_29"
      //lint {1025} "PQM_authorized_multi_30"
      //lint {1703} "PQM_authorized_multi_31"
      fi_tclVisitorMessage    oDTCStartMsg( oMsgStartData );
      //construct start message
      oDTCStartMsg.vInitServiceData(
                        CCA_C_U16_APP_MMGR,                  // AppID of this application; source
                        CCA_C_U16_APP_DIAGLOG,               // AppID of the Server; target
                        AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,   // StreamType
                        0,                                   // StreamCounter
                        _u16RegID,                           // RegId
                        0,                                   // command counter
                        CCA_C_U16_SRV_DIAGLOG,               // SID of the service
                        MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT, // FKTID of the method
                        AMT_C_U8_CCAMSG_OPCODE_METHODSTART   // MethodStart
                                   );
      TRACE_CCA_OutMessage(&oDTCStartMsg, _CLASS, __LINE__);
      if( poMain && poMain->enPostMessage( &oDTCStartMsg ) != AIL_EN_N_NO_ERROR )
      {
         // +++ can't send status +++
         ETG_TRACE_ERR(( "bStoreITCValueInDiagLog( ): Error CCA message couldn't be posted" ));
         if( !oDTCStartMsg.bDelete() )
         {
            // Not able to delete unsent message.
            ETG_TRACE_ERR(( "bStoreITCValueInDiagLog( ): Error CCA message which couldn't be posted couldn't be deleted also" ));
         }
      }
   }
   else   // if( _u16RegID != AMT_C_U16_REGID_INVALID )
   {
      ETG_TRACE_USR1(( "bStoreITCValueInDiagLog( ): bStoreITCValue(): No valid '_u16RegID'" ));
      bRet = FALSE;
   }
   return bRet;
}


#ifdef USE_REPLACE_E8_TESTRESULT
tVoid vdmmgr_tclclienthandlerdiaglog::vSetITC( tU16 u16ITC, tU8 oe8TestResult )
#else
tVoid vdmmgr_tclclienthandlerdiaglog::vSetITC( tU16 u16ITC, midw_fi_tcl_TestResult::tenType oe8TestResult )
#endif
{
   midw_fi_tcl_TestResult  oDialogTestResult;
   tU16                    u16ErrorStatus;

   ETG_TRACE_USR1(( "vSetITC( ): ITC: %d, TestResult: %d", ETG_ENUM(ITC, u16ITC), ETG_ENUM(DIAGLOG_STATUS , oe8TestResult) ));
   oDialogTestResult.TroubleCode     = u16ITC;
#ifdef USE_REPLACE_E8_TESTRESULT
   oDialogTestResult.Result.enType   = (midw_fi_tcl_e8_TestResult::tenType)oe8TestResult;
#else
   oDialogTestResult.Result.enType   = oe8TestResult;
#endif



   (tVoid)bStoreITCValueInDiagLog( &oDialogTestResult );


   // Store value in ITC map to have the actual data if diagnosis asks for...
   // 1. Evaluate the result
   /*lint -save -e788 Info 788;enum constant 'x' not used within defaulted switch */
   switch( oe8TestResult )
   {
#ifdef USE_REPLACE_E8_TESTRESULT
      case midw_fi_tcl_e8_TestResult::FI_EN_PASSED:
      case midw_fi_tcl_e8_TestResult::FI_EN_PASSEDDIAGNOSTIC:
         u16ErrorStatus = (tU16)MMGR_TestPassed;
         break;
      case midw_fi_tcl_e8_TestResult::FI_EN_FAILED:
      case midw_fi_tcl_e8_TestResult::FI_EN_FAILEDDIAGNOSTIC:
         u16ErrorStatus = (tU16)MMGR_TestFailed;
         break;
      case midw_fi_tcl_e8_TestResult::FI_EN_NORESULT:
      case midw_fi_tcl_e8_TestResult::FI_EN_NORESULTDIAGNOSTIC :
#else
      case midw_fi_tcl_TestResult::FI_EN_PASSED:
      case midw_fi_tcl_TestResult::FI_EN_PASSEDDIAGNOSTIC:
         u16ErrorStatus = (tU16)MMGR_TestPassed;
         break;
      case midw_fi_tcl_TestResult::FI_EN_FAILED:
      case midw_fi_tcl_TestResult::FI_EN_FAILEDDIAGNOSTIC:
         u16ErrorStatus = (tU16)MMGR_TestFailed;
         break;
      case midw_fi_tcl_TestResult::FI_EN_NORESULT:
      case midw_fi_tcl_TestResult::FI_EN_NORESULTDIAGNOSTIC :
#endif
      default:
         u16ErrorStatus = (tU16)MMGR_RoutineNeverRan;
         break;
   }

   /*lint -restore*/
   // 2. Now store the error status
   (tVoid)_ItcMap.bSetErrorStatusForITC( u16ITC, u16ErrorStatus );
}

//-----------------------------------------------------------------------------
// handle functions
//-----------------------------------------------------------------------------


/*************************************************************************
*  FUNCTION:    vHandleSaveTestResult() const
*
*  DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*************************************************************************/
tVoid vdmmgr_tclclienthandlerdiaglog::vHandleSaveTestResult( amt_tclServiceData* poMessage ) 
{
   TRACE_CCA_InMessage(poMessage, _CLASS);

   tU8 u8OpCode = poMessage->u8GetOpCode( );
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT: 
      {
         // +++ requested action finished on server, handle result +++
         break;
      }
      default:
      {
         // +++ unknown opcode +++
         break;
      }
   }
   (tVoid)poMessage->bDelete();
} //lint !e1762  DevStudios says:'not const, please'


/*************************************************************************
*  FUNCTION:    vHandleSendNextTestResult()
*
*  DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*************************************************************************/
tVoid vdmmgr_tclclienthandlerdiaglog::vHandleSendNextTestResult( amt_tclServiceData* poMessage )
{
   TRACE_CCA_InMessage(poMessage, _CLASS);

   tU8 u8OpCode = poMessage->u8GetOpCode( );
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_STATUS: 
      {
         // Get data
         fi_tclVisitorMessage                            oMsg( poMessage );
         midw_diaglogfi_tclMsgSendNextTestResultStatus   oData;
         (tVoid)oMsg.s32GetData( oData );
         break;
      }
      default:
      {
         // unknown opcode +++
         break;
      }
   }
   (tVoid)poMessage->bDelete();
} //lint !e1762  DevStudios says:'not const, please'


//-----------------------------------------------------------------------------
// help functions
//-----------------------------------------------------------------------------


/******************************************************************************
*  FUNCTION:    vOnAsyncRegisterConf ( tU16 u16RegisterId, tU16 u16ServerAppId,
*                                      tU16 u16ServiceId, tU16 u16TargetSubId)
*
*  DESCRIPTION: Register for a service.
*
*  PARAMETER:   u16RegisterId
*               u16ServiceId
*
*  RETURNVALUE: TRUE:   NO_ERROR
*               FALSE:  ERROR
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclclienthandlerdiaglog::vOnAsyncRegisterConf( tU16 u16RegisterId, tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId)
{
   FOR_FUTURE_USE( u16ServerAppId );
   FOR_FUTURE_USE( u16TargetSubId );

   if( u16ServiceId == CCA_C_U16_SRV_DIAGLOG )
   {
      if( AMT_C_U16_REGID_INVALID != u16RegisterId )
      {
         _u16RegID = u16RegisterId;
      }
      else
      {
         // registration failed
         ETG_TRACE_USR1(( "vOnAsyncRegisterConf(): registration failed" ));
         // Assert if registration is failed => assert to show that something is wrong
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
}


/*************************************************************************
*  FUNCTION:    tBool vdmmgr_tclclienthandlerdiaglog::bRegisterForService();
*
*  DESCRIPTION: register for a service
*
*  PARAMETER:   void
*
*  RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*  History:
*  InitialVersion
* 
*************************************************************************/
tBool vdmmgr_tclclienthandlerdiaglog::bRegisterForService( )
{
   tBool bRetVal = FALSE;

   // --- registration invalid?   --- 
   if( _u16RegID == AMT_C_U16_REGID_INVALID && _bRegAsyncStarted == FALSE )
   {
      // --- we register for the service so the server can't have any FID - registration ---
      _bFidReg = FALSE;
      if( poMain )
      {
         // --- try to register ---
         _u16RegID = poMain->u16RegisterService( CCA_C_U16_SRV_DIAGLOG, VDMMGR_C_U16_SERVICE_DIAGLOG_MAJOR_VERSION, 
                                                 VDMMGR_C_U16_SERVICE_DIAGLOG_MINOR_VERSION, 0, CCA_C_U16_APP_DIAGLOG );
         _bRegAsyncStarted = TRUE;
         ETG_TRACE_USR1(( "bRegisterForService( ): called u16RegisterService()" ));
      }
   }
   else
   {
      // +++ it could be an error to call this function, if we are already registered. +++
      ETG_TRACE_USR1(( "bRegisterForService( ): -> _u16RegID == AMT_C_U16_REGID_INVALID" ));
   }

   return bRetVal;
}


/******************************************************************************
*  FUNCTION:    tBool vdmmgr_tclclienthandlerdiaglog::bUnregisterForService();
*
*  DESCRIPTION: register for a service
*
*  PARAMETER:   void
*
*  RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tBool vdmmgr_tclclienthandlerdiaglog::bUnregisterForService( )
{
   tBool bRetVal;

   bRetVal = TRUE;

   // +++   check if your application is ready to unregister   +++
   // ---    registration valid?   ---
   if( _u16RegID != AMT_C_U16_REGID_INVALID )
   {
      if( poMain )
      {
         // --- unregister for Service ---
         poMain->vUnregisterService( (tU16) CCA_C_U16_SRV_DIAGLOG );
         // Make register-ID invalid:
         _u16RegID = AMT_C_U16_REGID_INVALID;
         _bRegAsyncStarted = FALSE;
         ETG_TRACE_USR1(( "bUnregisterForService( ): unregistered diaglog service" ));
      }
   }
   else
   {
      // +++     it could be an error to call this function, if we are already unregistered +++
      ETG_TRACE_USR1(( "bUnregisterForService( ): unregistered diaglog service already done" ));
   }
   return bRetVal;
}


/******************************************************************************
*  FUNCTION:    tBool vdmmgr_tclclienthandlerdiaglog::bRegisterForFID( tU16 u16FID )
*
*  DESCRIPTION: register for a function of a service
*               !!! before this, you must register for the service !!!  
*
*  PARAMETER:   u16FID: FID of the function 
*
*  RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tBool vdmmgr_tclclienthandlerdiaglog::bRegisterForFID( tU16 u16FID )
{
   tBool bRetVal = TRUE;

    // --- registration invalid? ---
   if ( _u16RegID == AMT_C_U16_REGID_INVALID )
   {
      bRetVal= FALSE;
      // +++ registration for the service failed or not done +++
      ETG_TRACE_USR1(( "bRegisterForFID( ): service is not registered" ));
   }
   else
   {
      // +++ create message with upreg - request: +++
      gm_tclEmptyMessage oUpRegMessage( CCA_C_U16_APP_MMGR, CCA_C_U16_APP_DIAGLOG,
                                        _u16RegID, 0, CCA_C_U16_SRV_DIAGLOG,
                                        u16FID, AMT_C_U8_CCAMSG_OPCODE_UPREG );

      TRACE_CCA_OutMessage(&oUpRegMessage, _CLASS, __LINE__);
      if( poMain && poMain->enPostMessage( &oUpRegMessage ) != AIL_EN_N_NO_ERROR )
      {
         bRetVal= FALSE;
         // +++ message send error +++
         ETG_TRACE_ERR(( "bRegisterForFID( ): Error: Failed to register for function ID: %d", ETG_ENUM(MIDW_DIAGLOGFI, u16FID) ));
      }
   }
   return bRetVal;
}


/******************************************************************************
*  FUNCTION:    bUnregisterForFID(tU16 u16FID)
*
*  DESCRIPTION: unregister for a function of a service
*
*  PARAMETER:   u16FID: FID of the function
*
*  RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tBool vdmmgr_tclclienthandlerdiaglog::bUnregisterForFID( tU16 u16FID )
{
   tBool bRetVal= TRUE;

   // --- registration invalid? ---
   if( _u16RegID == AMT_C_U16_REGID_INVALID )
   {
      bRetVal= FALSE;
      // +++ registration for the service failed or not done or already unregistered! +++
      ETG_TRACE_USR1(( "bUnregisterForFID( ): Registration failed or unregistration already done" ));
   }
   // +++ create message with relupreg - request: +++
   gm_tclEmptyMessage oRelupRegMessage( CCA_C_U16_APP_MMGR, CCA_C_U16_APP_DIAGLOG,
                                        _u16RegID, 0, CCA_C_U16_SRV_DIAGLOG,
                                        u16FID, AMT_C_U8_CCAMSG_OPCODE_RELUPREG );

   TRACE_CCA_OutMessage(&oRelupRegMessage, _CLASS, __LINE__);
   if( poMain && poMain->enPostMessage( &oRelupRegMessage ) != AIL_EN_N_NO_ERROR )
   {
      bRetVal= FALSE;
      // +++ message send error +++
      ETG_TRACE_USR1(( "bUnregisterForFID( ): Error: Failed to unregister for function ID: %d", ETG_ENUM(MIDW_DIAGLOGFI, u16FID) ));
   }
   return bRetVal;
}


/******************************************************************************
*
* FUNCTION:    vOnServiceState
* 
* DESCRIPTION: handle a state change of the service we use
*
* PARAMETER: 
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclclienthandlerdiaglog::vOnServiceState( tU16 u16ServiceId, tU16 u16ServerId,
                                                   tU16 u16RegisterId, tU8  u8ServiceState,
                                                   tU16 u16SubId )
{
   tBool bRetVal;

   FOR_FUTURE_USE( u16RegisterId );
   FOR_FUTURE_USE( u16ServerId );
   FOR_FUTURE_USE( u16SubId );

   ETG_TRACE_USR1(( "vOnServiceState( ): Get service state: %d ", ETG_ENUM(AMT_SERVICE_STATE, u8ServiceState) ));

   if( u16ServiceId != CCA_C_U16_SRV_DIAGLOG )
   {
      // +++ this is not the service we use +++
      ETG_TRACE_USR1(( "vOnServiceState(): u16ServiceId != CCA_C_U16_SRV_DIAGLOG -> ID: %u(8,2,M)", ETG_ENUM(ail_u16ServiceId, u16ServiceId) ));
   }
   // --- analyze the state of the server ---
   switch( u8ServiceState )
   {
      case AMT_C_U8_SVCSTATE_AVAILABLE:
      {
         // the service is now available. --- register only once ---
         if( !_bFidReg )
         {
            // register for all FIDs you need
            bRetVal = bRegisterForFID( MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT );   // Error memory cleared

            if( bRetVal )
            {
               _bFidReg = TRUE;
            }
         }
         break;
      }
      case AMT_C_U8_SVCSTATE_NOT_AVAILABLE:
      {
         // +++ the service is not available anymore. +++
         break;
      }
      case AMT_C_U8_SVCSTATE_REG_INVALID:
      {
         // --- the server has lost our registration.  ---. +++ the service is not available. +++
         // --- register again. ---
         (tVoid)bRegisterForService( );
         break;
      }
      default:
      {
         // +++ unknown service state +++
         break;
      }
   }//switch
}
