/*-----------------------------------------------------------------------------*
 * PortListManager.h                                                         *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : G3G                                                           *
 * COPYRIGHT   : (c) 2013-2020 Robert Bosch GmbH, Hildesheim                   *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file PortListManager.h 
 *
 * \brief stores HW-Signals like PPON, OC, Undervoltage etc.  (please refer to struct trPortStates see common/enums.h)
 *
 * \version 26.03.2014, Christian K�chling (Bosch), version 1.0
 *          Initial version 
 
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

#ifndef PORTLISTMANAGER_BASE_H
#define PORTLISTMANAGER_BASE_H

/*-----------------------------------------------------------------------------*
 *defines
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * unsing namespace                                                               *
 *-----------------------------------------------------------------------------*/
using namespace statetbl; 


/*-----------------------------------------------------------------------------*
 * class declaration                                                                *
 *-----------------------------------------------------------------------------*/
class PortListManager
{
    public:
        
        /*!
        * \brief singleton pattern to get pointer of object to call functions
        *
        * \param [in] f_pCDevice device to be copied
        */
        static PortListManager * GetInstance();
        
        /*!
        * \brief singleton pattern to delete object
        *
        * \param [in] f_pCDevice device to be copied
        */
        static void  DestroyInstance();


        /*!
        * \brief Assignment Operator
        *
        * Will not be implemented. Avoids Lint Prio 3 warning: Info 1732: new
        * in constructor for class 'PortListManager' which has no assignment
        * operator.
        *
        * \note This is a technique to disable the assignment operator for this
        *       class. So if an attempt for the assignment is made compiler
        *       complains.
        */
        PortListManager& operator=(const PortListManager &oListManager);


        /*!
        * \brief 'setter' to fill content to PortListManager
        *
        * \param [in]  f_rArrayHWSignals holds according to trPortStates all necessary values
        */
        tVoid vSetHardwareSignals(IN const trPortStates (&f_rArrayHWSignals)[ARRAYSIZEFORUSB]);



        /*!
        * \brief 'getterr' to fill content to  f_rArrayHWSignals according to current setting in PortListManager
        *
        * \param [out]  f_rArrayHWSignals holds according to trPortStates all necessary values
        */
        tVoid vGetHardwareSignals(OUT trPortStates (&f_rArrayHWSignals)[ARRAYSIZEFORUSB]);

        /*!
        * \brief 'getterr' to find out if f_rArrayHWSignals are different from already stored values
        *
        * \param [out]  f_rArrayHWSignals holds according to trPortStates all necessary values
        */

        tBool GetIsDifferentHardwareSignals(IN const trPortStates (&f_rArrayHWSignals)[ARRAYSIZEFORUSB]);
        
        /*!
        * \brief size of array which stores HW signals is linked to number of available USB Connectors
        *
        */
        tUInt  uiGetSizeOfArrayHardwareSignals(tVoid) const;

        /*!
        * \brief get element of array
        *
        * \param [in]    f_uiIndex index of element - is align with number of usb connector - if system has two connectors then 1 is first 2 is second connector
        * \param [out] f_rPortState used to return element
        *
        * return TRUE: Element is valid, FALSE: Element invalid (reason: element is invalid (see vClear) or F_uiIndex has been out of bound)
        */
        tBool bGetElement (IN tUInt f_uiIndex, OUT trPortStates &f_rPortState) const;


        /*!
        * \brief print currently stored values
        *
        */
        tVoid vShowHardwareSignals() const;

 
        tVoid vClear(IN trPortStates &f_rPortState) const;

        /*!
                    * \brief: copy current port states
                    *
                    */
        tVoid GetCopyArrayPortStates(OUT trPortStates (&f_rArrayPortStates)[ARRAYSIZEFORUSB]);


        /*!
        * \brief 'setter' to fill content to PortListManager
        *
        * \param [in]  f_rArrayHWSignals holds according to trPortStates all necessary values
        * This is mainly used by PRM Manager to set the port power state
        */
        tVoid vSetPortPowerState(IN const trPortStates &f_rPortStates);
               


        
    protected:

        //-----------variables-------------------------------

        trPortStates m_rArrayPortStates[ARRAYSIZEFORUSB];
        
        //-----------functions-------------------------------

        /*!
             * \brief clear content of array. u8USBPortNo of trPortStates used to mark it valid or invalid
             *
             */
        //lint -sem(PortListManager::vClearArray,initializer)
        tVoid vClearArray();


        /*!
             * \brief after clear has been called in the constructor vInit is used to mark those entries eUSB1..eUSBx valid
             *           the service will then send it in with 'DevManagerNotifyConnectionStatus' after a clients has been registered
             *
             */
        //lint -sem(PortListManager::vInit,initializer)
        tVoid vInit();


    private:
        //singleton helper
        static PortListManager *m_pPortListManager;
        static DVMLOCK          m_singelton; 

        //-----------functions-------------------------------
        /*!
             * \brief constructor of object - is private for singleton class 
             *
             */
        PortListManager();

        //-----------functions-------------------------------
        /*!
             * \brief destructor of object - is private for singleton class 
             *
             */
        virtual ~PortListManager();


};

#endif // PORTLISTMANAGER_H

////////////////////////////////////////////////////////////////////////////////
// <EOF>

