/*-----------------------------------------------------------------------------*
 * VoltageManager.h                                                                *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen3                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file VoltageManager.h
 *
 * \brief This file holds the declaration for VoltageManager
 *
 * VoltageManager receives over current signals from USB HW. The prm signals are
 * used to trigger the mechanism to react on male function of a connector. E.g.
 * the VD_DeviceManager has to notify the clients about the MALFUNCTION and
 * OVER-CURRENT
 *
 * \version Initial Version
 * \version 05.08.2011, Koechling, Christian (Bosch), Interface VD_DeviceManager
 *          changed
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

#ifndef __VOLTAGE_MANAGER_H__
#define __VOLTAGE_MANAGER_H__

/*-----------------------------------------------------------------------------*
 * includes                                                                  *
 *-----------------------------------------------------------------------------*/
#include "IVoltageManager.h" //see tenSystemVoltage
#include "IStateTableHWMalfunction.h" //new interface

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * Structs                                                                     *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/


/*!
 * \class VoltageManager
 */
class VoltageManager: public IVoltageManager, public TFThread
{
 //=======
    public:
 //=======

       typedef enum
       {
          eThread_SystemVolt = 0,
          eThread_Last = 1
       }tenThreadFunction;


        /*! 
         * \brief constructor
         *
         * \todo think about using directly singleton call instead of handing over pointer
         */
        VoltageManager(IStateTableHWMalfunction& oIStateTableHWMalfunction);

        /*!
         * \brief private destructor for singleton
         */
        ~VoltageManager();
    
        /*!
         * \brief Initializes the object
         *
         * \return [int] initialization status OSAL_OK or OSAL_ERROR
         */
        int Initialize();


         /*! 
        * \brief read system voltage 
        *        
        * \param [out] true: is Critical voltage, false: is Operating voltage
        *
        */
        virtual bool IsCriticalVoltage();
 
        /*! 
         * \brief Get the actual state of the system voltage - this function is public to give TraceCmdManager access to it for testing
         *
         * \param [OUT] f_eSystemVoltage
   
         */
        virtual void vGetSystemVoltage(OUT tenSystemVoltage &f_eSystemVoltage);


        /*! 
         * \brief Get the actual state of board voltage - this function is public to give TraceCmdManager access to it for testing
         *
         * \param [OUT] f_u32BoardVoltage_ms - voltage in millivolt:
             
         */
        virtual void vGetBoardVoltage(OUT tU32 &f_u32BoardVoltage_mv);


        /*! 
          * \brief inform state table about voltage - this function is public to give TraceCmdManager access to it for testing
          *
          * \param [IN] f_eU60_drop SIG_TRUE: critical voltage SIG_FALSE: Normal voltage i.e. operating voltage
          *
          *  
          */
        virtual tVoid vSendVoltageWarning(IN tenSignalValue f_eU60_drop) const;


        /**
        * This function derives from TFThread and starts a user function in a thread from the thread factory's thread pool.
        * After leaving while loop used thread is stopped and released for other users
        *
        * @return void
        */
        virtual void Do(int functionID, void *ptr);
        
//=========
    protected:
//=========

        
        /*! 
         * \brief function to trace content of variable with respect to be below values
         *
         * \param [IN] map f_u32SystemVoltage to tenSystemVoltage
         */
        void vMapVoltage(OUT VoltageManager::tenSystemVoltage &eVolt, IN tU32 f_u32SystemVoltage) const;

        /*! 
         * \brief function to trace content of variable with respect to be below values
         *
         * \param [IN,OUT] map tenSystemVoltage to eCriticalVoltage which is then either SIG_UNDEF, SIG_TRUE or SIG_FALSE
         */
        void vMapVoltage(OUT tenSignalValue &eCriticalVoltage,IN VoltageManager::tenSystemVoltage eVolt);

        
        /*! 
          * \brief 
          *
          * \param [OUT] f_u32BoardVoltage_ms - voltage in millivolt:
              
          */
         void vEnableSystemVoltageNotification(IN tBool bEnable);

        /*! 
          * \brief routine to receive voltage notifications wrapps vWaitLoopForSystemVoltageChanged
          *
          * \param [IN] f_hEventHandle handle of event
          *
          *  
          */
        void vReceiveSystemVoltageNotification();

        /*! 
          * \brief Loop waiting for event notifieing a voltage change
          *
          * \param [IN] f_hEventHandle handle of event
          * \param [IN ] f_u32ClientID
          *
          *  
          */
        void vWaitLoopForSystemVoltageChanged(IN const OSAL_tEventHandle &f_hEventHandle, IN tU32 f_u32ClientID);

        /*! 
         * \brief function to trace content of variable 
         *
         * \param [IN] eCriticalVoltage
         */
        void vShowCriticalVoltage(tenSignalValue &eCriticalVoltage);


        /*! 
         * \brief function to stop loop of this thread
         *
         * \param [IN] eCriticalVoltage
         */
        tBool bStopThread();


//========
    private:
//========

         tenSignalValue m_eCriticalVoltage;

         OSAL_tIODescriptor m_hIOVoltDescriptor;


        tBool m_bStopThread;

        IStateTableHWMalfunction& m_oIStateTableHWMalfunction;
      
};

#endif //__PRM_MANAGER_H__

////////////////////////////////////////////////////////////////////////////////
// <EOF>

