/*-----------------------------------------------------------------------------*
 * UDevManager.h                                                               *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file UDevManager.h
 *
 * \brief This file holds the declaration for UDevManager.
 *
 * UDevManager is responsible for processing events related to device
 * connection and disconnection. The events are captured from the platform
 * and processed and then notified to the other clients registerred with the
 * device manager service
 *
 * The notifications are receive from UDEV and INOTIFY.
 *
 * Udev events are captured using udev_monitor
 *
 * Inotify is being used as an project specific event notification mechanism
 * for storage device events. The platform udev scripts have been modified to
 * generate a file mirroring the device node for the deivice partition,
 * containing the device mountpoint. On device connection the file is created
 * and on disconnection the file is removed. These events are catured and
 * processed.
 *
 * \version Initial Version
 * \version 05.08.2011, Koechling, Christian (Bosch), Interface VD_DeviceManager changed
 * \version 01.11.2011, Negi, Sunder (Montavista), New function added, redundant
 *          functions removed
 * \version 09.11.2011, Negi, Sunder (Montavista), Added new functions
 * \version 17.11.2011, Koechling, Christian (Bosch),
 *          -# compilabilty enable switch off log mechanism used by MV - see LogTrace.h
 *          -# cleanup: 1st step to shift includes to Common.h
 * \version 21.11.2011, Koechling, Christian (Bosch), cleanup includes and
 *          intermediate defines which marked bugfixes
 * \version 27.11.2011, Negi, Sunder (Montavista), Added functions for dbus access
 * \version 29.11.2011, Negi, Sunder (Montavista), Added vd_devicemanager_tclApp
 *          as member variable and modified functions who take
 *          vd_devicemanager_tclApp* as parameter
 * \version 05.12.2011, Negi, Sunder (Montavista), Added function to read
 *          productname from usb.ids file
 * \version 06.12.2011, Negi, Sunder (Montavista), cleanup
 * \version 06.12.2011, Koechling, Christian (Bosch), cleanup
 * \version 10.12.2011, Negi, Sunder (Montavista),
 *          -# Added: member m_pBaseUSBDev
 *          -# Cleanup: removed calls to getproductid
 * \version 12.12.2011, Koechling, Christian (Bosch), cleanup singleton, further
 *          care for simulating EOL-values
 * \version 15.12.2011, Negi, Sunder (Montavista), cleanup
 * \version 16.01.2012, Negi, Sunder (Montavista), Added: Functions for forcefull
 *          mount/unmount
 * \version 17.01.2012, Negi, Sunder (Montavista), Added: Code for hub details
 * \version 18.01.2012, Negi, Sunder (Montavista), add helper function to detect
 *          if HUB is connected
 * \version 24.01.2012, Negi, Sunder (Montavista), Disabled  storeudevevt
 * \version 26.01.2012, Negi, Sunder (Montavista), Enabled LOOKUP, added macros
 *          to disable udev trigger and delay disk info
 * \version 16.02.2012, Negi, Sunder (Montavista),
 *          -# BugFix: BUGFIX_LEXAR_DRIVE : Lexar drive not getting notified
 *          due to error while performing an MTP check on it
 *          -# Enhancement: USE_INOTIFY: Preparation for newer udev scripts
 *          -# Cleanup: Removed some class members, renamed functions
 * \version 22.02.2012, Negi, Sunder (Montavista)   , Cleanup: Few macros cleaned up
 * \version 23.02.2012, Koechling, Christian (Bosch), add TTFIS command VD_DVM_GET_CONNETED_HUBS
 * \version 07.03.2012, Koechling, Christian (Bosch), merged MV-code Feb27 -
 *          contains issue for startup synchronisation
 * \version 10.03.2012, Koechling, Christian (Bosch), cleanup history function for analysis
 * \version 02.06.2012, Koechling, Christian (Bosch),
 *          -# added mechanisms to use central device list manager,
 *          -# improved critical voltage behaviour and
 *          -# state machine to send combined messages for output of the first
 *          state machine
 * \version 19.07.2012, Koechling, Christian (Bosch), add mechanism to log
 *          histories for analysis to USB-stick
 * \version 26.07.2012, Negi, Sunder (MontaVista), Fixed crash due to mtp check
 * \version 27.07.2012, Negi, Sunder (MontaVista), Cleaned up file hierarchy
 * \version 02.08.2012, Negi, Sunder (MontaVista),
 *          -# Refactored debug information logging
 *          -# Added History Manager, refer define HISTORY_MANAGER
 * \version 03.08.2012, Koechling, Christian (Bosch), Added doxygen comments
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 10.08.2012, Negi, Sunder (MontaVista), renamed singleton functions
 * \version 11.08.2012, Negi, Sunder (MontaVista), general function moved to new
 *          location, see CLEANUP_UDEVMANAGER (utils.h)
 * \version 12.08.2012, Negi, Sunder (MontaVista), cleanup HISTORY_MANAGER
 * \version 13.08.2012, Negi, Sunder (MontaVista), cleanup logging to usb (WRITEHISTORYTOUSBSTICK)
 * \version 14.08.2012, Negi, Sunder (MontaVista), updates for single device store
 * \version 24.08.2012, Negi, Sunder (MontaVista), cleanup CLEANUP_UDEVMANAGER,
 *          HANDLE_GPT
 * \version 27.08.2012, Negi, Sunder (MontaVista), cleanup USE_CHARGE_APPLE
 * \version 27.08.2012, Koechling, Christian (Bosch), trigger FillStorage fct. by PDT-diag
 *
 *-----------------------------------------------------------------
 *                                   development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch) 
 *         -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

#ifndef __CUDEVMANAGER_H__
#define __CUDEVMANAGER_H__

extern "C"
{
#include <libudev.h>
#include <libmtp.h>
}

#include "IAutomounterAdapter.h"
#include "IUdevManagerInternalInput.h"
#include "IStorageInfoProvider.h"

// File containing vendoris/productis of all the USB devices
#define USB_IDS_FILE "/usr/share/usb.ids"

#define WATCH              INOTIFY_DIR
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
#define WATCH_UNSUPPORTED  INOTIFY_UNSUP_DIR
#endif //USE_EXTERNAL_UNSUPPORTED_DEVICES

/*-----------------------------------------------------------------------------*
 * Structs                                                                     *
 *-----------------------------------------------------------------------------*/
/*!
 * \struct hub_details
 *
 * \brief Stores details about a hub
 *
 * \todo inherit from CDevice
 */


enum
{
    enPipeReader   = 0,
    enPipeWriter 
 = 1
};

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace std;
using namespace udevmgr;
using namespace statetbl;

/*-----------------------------------------------------------------------------*
 * Forward declarations                                                        *
 *-----------------------------------------------------------------------------*/
class StateTable;
class CUSBDiagnosis;
class Info_rDmEolConfSettings;
class vd_devicemanager_tclApp;
class IDeviceRecognizer; //see INTERMEDIATEDEFINE_REFACTORUDEV
class StorageInfoProvider;


/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/
/*!
 * \class  CUDevManager
 */
class CUDevManager : public IUdevManagerInternalInput, public TFThread
{
    private:

        /*! 
         * \brief Constructor
         *
         * This Initializes all member variables to NULL.
         *
         *
         */

    CUDevManager(IDeviceRecognizer *pIDeviceRecognizer = NULL, IAutomounterAdapter *pIAutomounterAdapter = NULL); //see INTERMEDIATEDEFINE_REFACTORUDEV

        /*! 
         * \brief Destructor
         *
         * This frees/unrefs member variables. The _unref functions from libudev
         * generally check for a Not NULL value before trying to free.
         */
        ~CUDevManager();

        // static member variable declared for singleton class
        static CUDevManager  *m_pUDevManager;
        static DVMLOCK        m_singelton;

        StateTable           *m_pStateTable;

        IDeviceRecognizer    *m_pIDeviceRecognizer; //see INTERMEDIATEDEFINE_REFACTORUDEV

        struct udev         *m_pUdev;         ///< Udev library context
        struct udev_device  *m_pDev;          ///< object for udev object
        struct udev_device  *m_pParentHUBDev; ///< parent hub udev object
        struct udev_device  *m_pBaseUSBDev;   ///< parent usb udev object
        struct udev_monitor *m_pUdevMonitor;  ///< udev event source
        int                  m_iUDevMonitorFD;///< udev event source desriptor


        int m_iPipeFD[2];          ///< file descriptor  //RoadMap 14002 CDROMMounter (MixedModeDiag)

        //@todo partially unsued code here
        int m_iInotifyFD;                   ///< inotify file  descriptor
        int m_iInotifyWatchFD;              ///< inotify watch descriptor
        int m_iInotifyWDMscDev;             ///< inotify watch descriptor 
        int m_iInotifyWDUnsupDev;           ///< inotify watch descriptor
        int m_iInotifyWDAppleDev;           ///< inotify watch descriptor
        int m_InotifyWDMtpDev;              ///< inotify watch descriptor
        int m_iAutomounterLibFD;
        IAutomounterAdapter *m_pIAutomounterAdapter;

        //prevent that DTY_DVD_DRIVE is notified if disc is inserted. If Disc is inserted at startup there is not extra Drive notification but Drive including Disc
        int m_iNumOfDVDDriveNotification;

        //Check for call to GetAllConnectedDevices
        bool m_bIsActive;

        //to prevent race condition while access of list/vector
        DVMLOCK m_tableLock;
        DVMLOCK m_UdevLock;

        vector<CDevice> m_vDisableDevices;
        vector<CGlobalEnumerations::HubDetails_t *> m_vHubs; ///< list of hubs
        //Will store the mass storage device structure to calculate the storage info of it
        vector<CDevice> m_vDevicesWithOutStorageInfo;

        //TODO(kmv5cob):Check and Remove these Parenthub related Member variables and its associated function ParentHubSwapping, since they are unused
        int m_iUsb1HubPort;
        int m_iUsb2HubPort;

        // Action received from udev
        GENSTRING m_cAction;

        //Serial Id Of Parent Hub 1/2
        GENSTRING m_cParentUsb1Hub, m_cParentUsb2Hub;

        //Port of Parent Hub 1/2
        int m_cParentHub1Port, m_cParentHub2Port;

        //Serial Id Of Parent Hub 1/2
        GENSTRING m_cParentHub1Node, m_cParentHub2Node;

        //Parent HUB Serial ID
        GENSTRING m_cParentHubSerial;

        StorageInfoProvider *m_pStorageInfoProvider;

        //Is DVD drive in Mass storage mode
        tBool m_bIsDVDDriveMassStorageMode;

        Lock m_UpdateDVDDriveInfoLock;
		
        tBool GetIsDVDDriveMassStorageMode();
        tVoid SetIsDVDDriveMassStorageMode(tBool f_bSetValue);
        //used for testing to simulate udev without interference from reals driver connection
        tBool m_bConnectUDEV;

#ifdef CHECK_LAST_DEVICE
        vector <GENSTRING> m_vSeenDevices;
#endif // CHECK_LAST_DEVICE

#ifdef INCOMPLETE_DEV
        vector <CDevice> m_vIncompleteDevices;

#endif // INCOMPLETE_DEV

        /* Private methods */

        /*!
         * \brief This method iterates through the udev list entry and fills out
         *        the required properties as members of m_pCurDevice
         *
         * \param [in,out] f_pCurDevice pointer to current device
         *
         * \return DEVICEMANAGER_OK in case of success
         */
        int FillCurrDevice(CDevice* f_pCurDevice);

    /*!
        * \brief This method iterates through the udev list entry for internal SDCards and fills out
        *         the required properties as members of m_pCurDevice
        *
        * \param [in,out] f_pCurDevice pointer to current device
        *
        * \return DEVICEMANAGER_OK in case of success
        */
        int FillCurrDeviceSDInternal(CDevice* f_pCurDevice, GENSTRING f_cDevNode);


        /*!
            * \brief This method iterates through the udev list entry for internal CDRom and fills out
            *         the required properties as members of m_pCurDevice
            *
            * \param [in,out] f_pCurDevice pointer to current device
            *
            * \return DEVICEMANAGER_OK in case of success
            *
            */
        int FillCurrDeviceCDROM(CDevice* f_pCurDevice,GENSTRING f_cDevNode); //Roadmap 13010_CDROM


        int FillCurrDeviceCDDA(CDevice* f_pCurDevice, GENSTRING f_cDevNode);    //Roadmap 1615 Internal DVD Drive - option

        int FillCurrDeviceDVDDrive(CDevice* f_pCurDevice, GENSTRING f_cDevNode);//Roadmap 1615 Internal DVD Drive

        //Function to detect the USB Block device
        int DetectBlockDevice(CDevice* f_pCurDevice,CGlobalEnumerations::NOTIFICATION_Type enNotiType, GENSTRING f_cDevNode);


        /*!
         * \brief Detect that connected device is USB Block device.
         *
         * This function returns true if the connected usb device is having
         * "vfat" ,"ntfs" or "hfsplus" file formats.
         *
         * \param [in,out] f_pCurDevice pointer to current device
         *
         * \returns DEVICEMANAGER_ERROR_NODEVICE in case of device is not block device
         *          DEVICEMANAGER_ERROR_NOTVALID in case the block device has is not valid block device
         *          DEVICEMANAGER_DEVICE_DETECTED in case of device detected
         */
        int DetectBlockDevice(CDevice* f_pCurDevice);

  
        /*!
         * \brief Detect that connected device is HUB devices.
         *
         * \param [in,out] f_pCurDevice pointer to current device
         *
      * \return DEVICEMANAGER_ERROR_NODEVICE in case of no device detetcted
         *         DEVICEMANAGER_DEVICE_DETECTED in case of device detected
         *         DEVICEMANAGER_ERROR_INVALID_PARAM in case f_pCurDevice is NULL
         */
        int DetectHubDevices( const CDevice* f_pCurDevice);

        /*!
         * \brief Checks whther the hub was already found.
         *
         * \param [in] f_cDeviceNode the hub device name
         *
         * \return [CGlobalEnumerations::HubDetails_t*] if f_cDeviceNode is found in the vector
         *          NULL otherwise
         */
        CGlobalEnumerations::HubDetails_t* IsHubAlreadyConnected( GENSTRING f_cDeviceNode);

        bool IsHubAlreadyConnected(GENSTRING &sysName, CGlobalEnumerations::HUB_NUM_t &eHubNo);
        /*!
         * \brief Gets details about a hub
         *
         * \param [in] f_eHubNum the hub number
         *
         * \return [HubDetails_t*] if f_eHubNum is found in the vector
         *          NULL otherwise
         */
        CGlobalEnumerations::HubDetails_t* GetHub(CGlobalEnumerations::HUB_NUM_t f_eHubNum);



        /*!
         * \brief Check for device entry in map, if present otherwise insert it in map
         *
         * \param [in] f_pCurDevice pointer to current device
         *
         * \returns DEVICEMANAGER_ERROR_NODEVICE in case of no device detetcted
         *          DEVICEMANAGER_DEVICE_DETECTED in case of device detected
         *          DEVICEMANAGER_ERROR_INVALID_PARAM in case f_pCurDevice is NULL
         */

        int PushDeviceData(/*const*/ CDevice* f_pCurDevice);

        /*!
         * \brief Get the Serial Id of the connected Hub
         *
         * \param [in] f_pCurDevice pointer to current device
         *
         * \return DEVICEMANAGER_OK in case of success
         *         DEVICEMANAGER_ERROR_INVALID_PARAM in case f_pCurDevice is NULL
         */
        int GetParentSerialId(CDevice* f_pCurDevice);

        /*!
         * \brief This function gives device disconnection notification
         *
         * \param [in] f_cDeviceNode device removed
         * \param [in] f_eAction     disconnection type
         *
         * \returns DEVICEMANAGER_DEVICE_DETECTED in case of success
         */
        int DisconnectNotification( GENSTRING f_cDeviceNode,
                CGlobalEnumerations::REMOVED_DEVICE_TYPE_Type f_eAction);

        /*!
         * \brief This function gives hub disconnection notification
         *
         * \param [in] f_cDeviceNode hub removed
         * \param [in] f_eAction     disconnection type
         *
         * \returns DEVICEMANAGER_DEVICE_DETECTED in case of success
         */
        int DisconnectNotificationHUB( GENSTRING f_cDeviceNode,
                CGlobalEnumerations::REMOVED_DEVICE_TYPE_Type f_eAction);


        /*!
         * \brief This function does the parent hub details swapping if required
         *        for calibration parameter
         *
         * \param [in] f_pCurDevice the device
         *
         * \returns DEVICEMANAGER_DEVICE_DETECTED in case of success
         */
        void ParentHUBSwapping( const CDevice* f_pCurDevice);

             /*!
          * \brief This method is to add UUID of partition to serial number.
          *          In this manner different partitions of the same device are notified to 
          *          Mediaplayer which puts a parition to its database as if it would have received
          *          a different device
          *
          * \param [in,out] f_pCurDevice Check this device for valid partitions.
          *
          * \returns -
          */
        
        tVoid AddPartitionUUIDToSerialNumer(CDevice* f_pCurDevice) const;

        /*!
          * \brief This method is to Fill the Storage Info of Mass storage devices in a temporary thread.
          *   This method first check whether the thread to calculate the storage info is free or not.
          *   If it is free and if the devices found in m_vDevicesWithOutStorageInfo vector, it will pick
          *   it to calculate the storage info
          *
          * \returns -
          */
       tVoid  vFillStorageInfoMSC();


        /*!
         * \brief This method finds the first valid partition.
         *
         * \param [in,out] f_pCurDevice Check this device for valid partitions.
         *
         * \returns DEVICEMANAGER_OK if valid partition is found,
         *          DEVICEMANAGER_ERROR_NODEVICE if no valid partition is found,
         *          DEVICEMANAGER_ERROR_INVALID_PARAM if f_pCurDevice is NULL
         */
        int CheckForValidPartition(CDevice* f_pCurDevice);

#ifdef USE_TOTAL_USED_FREE_SIZE

        /*!
         * \brief gain total, free and used size
         *
         */
        tVoid FillStorageInfoSizeTUF(INOUT CDevice* f_pCurDevice);

#ifdef USE_REPLACE_DF_WITH_STATFS
        /*!
              * \brief checks size of  USB-Stick,SD-Card internal and SD-Card External total size, used size and free size used by FillStorageInfoSizeTUF (statfs is used)
              *
              *  \param [out]  total size 
               *  \param [out] used size
              *  \param [out]  free  size
              *
              *  \param [in] mount point e.g. 'dev/media/mystorage'
              *
              * \return [tVoid]
              */
           tVoid GetSizeMSC_TUF(OUT GENSTRING& f_totalSize,OUT GENSTRING& f_usedSize,OUT GENSTRING &f_freeSize, IN GENSTRING f_mountPoint);
#else
        
        /*!
         * \brief checks size of  USB-Stick,SD-Card internal and SD-Card External total size, used size and free size used by FillStorageInfoSizeTUF (df is used)
         *
         *  \param [out]  total size 
          *  \param [out] used size
         *  \param [out]  free  size
         *
         *  \param [in] mount point e.g. 'dev/media/mystorage'
         *
         * \return [tVoid]
         */

        tVoid GetSizeMSC(OUT GENSTRING& f_totalSize,OUT GENSTRING& f_usedSize,OUT GENSTRING &f_freeSize, IN GENSTRING f_mountPoint); //Roadmap 13001 ExtUserInformation, 13023 ExtConfiguration
#endif //USE_REPLACE_DF_WITH_STATFS

   

#endif //USE_TOTAL_USED_FREE_SIZE
        /*!
         * \brief Modifies device name as per GM requirements
         *
         * \param [in]  f_pCurDevice the device
         */
        tVoid adjustDeviceName(CDevice* f_pCurDevice) const;

        // Process the mounted device
        /*!
         * \brief Processes the inotify event for newly mounted device
         *
         * \param [in] f_cPartitionNode the device sdX1, sdX2 etc
         *
         * \return DEVICEMANAGER_DEVICE_DETECTED on success,
         *         otherwise on failure
         */
        int HandleMount (const char *f_cPartitionNode, GENSTRING &mountPoint);
    #ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
        int HandleMountUnsupported( const char *f_cPartitionNode, GENSTRING &mountPoint, CGlobalEnumerations::NOTIFICATION_Type enWNotiDevTyp );
    #endif // USE_EXTERNAL_UNSUPPORTED_DEVICES


        /*!
         * \brief Processes the inotify event for unmounted device
         *
         * Uses AddDevice() to add the device
         *
         * \param [in] f_cPartitionNode the device sdX1, sdX2 etc
         *
         * \see AddDevice()
         */
        void HandleUnMount (const char *f_cPartitionNode);

        /*!
         * \brief Processes the inotify event for unmounted device
         *
         * Uses AddDevice() to add the device
         *
         * \param [in] cDevNode the device sdX1, sdX2 etc
         *
         * \see AddDevice()
         */
        void HandleUnMount ( GENSTRING cDevNode);

    /*!
     * \brief push device to DeviceListManager
     *
     * \param [in] f_iDevUsable: DEVICE_OK==f_iDevUsable then it is pushed to list otherwhise not 
     * \param [in] f_pCurDevice  holds device properties
     *
     */
    void vCheckPushToList(int f_iDevUsable,const CDevice* f_pCurDevice);


    /*!
     * \brief Processes notification of devices from a pipe //Roadmap 13035 Overtemperature
     *
     *  
     *
     * \param [in] cDevNode the device sdX1, sdX2 etc
     *
     * \see AddDevice()
     */

    tVoid vCASENotificationPipe(IN trPipeParmsSendMountDev &f_oPipeParams);

    tVoid vCASENotificationUdev();

    tVoid vCASENotificationAutomount();

    tVoid vCASENotificationAutomounterLib();

 
  


    public:

        /*!
         * \brief Get CUDevManager object
         *
         * Provides the singleton CUDevManager object. First invocation of
         * this function creates the object
         *
         * \return [CUDevManager*] singleton object
         */
        static CUDevManager *GetInstance();

        /*!
         * \brief Destroys the singleton object
         */
        static void  DestroyInstance();

        /*!
         * \brief Assignment Operator
         *
         * Will not be implemented. Avoids Lint Prio 3 warning: Info 1732: new
         * in constructor for class 'CUDevManager' which has no assignment
         * operator.
         *
         * \note This is a technique to disable the assignment operator for this
         *       class. So if an attempt for the assignment is made compiler
         *       complains.
         */
        CUDevManager& operator=(const CUDevManager &oClientHandler);

#ifdef INTERMEDIATEDEFINE_REFACTORUDEV
        /*!
         * \brief setter used as long UdevManager is a singleton and not a class. Later this Interface could be injected using the contructor
         */
        void setIDeviceRecognizer(IDeviceRecognizer *pIDeviceRecognizer);
#endif


        /*!
         * \brief Get the pointer to a AutomounterAdapter class
         *
         * \param [in] pIAutomounterAdapter address to interface of AutomounterAdapter
         *
         * \return None
         */
        void SetIAutomounterAdapter(IAutomounterAdapter *pIAutomounterAdapter);

        /*!
         * \brief Get the pointer to a StorageInfoProvider class
         *
         * \param [in] pIStorageInfoProvider address to interface of StorageInfoProvider
         *
         * \return None
         */
        tVoid SetIStorageInfoProvider(IStorageInfoProvider *pIStorageInfoProvider);



        /*!
         * \brief Creates a new udev context
         *
         * \returns DEVICEMANAGER_OK in case of success
         *          DEVICEMANAGER_ERROR_MEM_ALLOC in case of failure
         */
        int InitializeUDevMonitor();

        /*!
         * \brief Creates udev monitor object
         *
         * \returns DEVICEMANAGER_OK in case of success
         *          DEVICEMANAGER_ERROR_MEM_ALLOC in case of failure in creating
         *          DEVICEMANAGER_ERROR_UNKNOWN in case problem in adding subsystem filter
         *          and in case of failure to enable receiving
         */
        int CreateUdevMonitor();

#ifdef USE_LOOKUP
        /*!
         * \brief Lookup/Enumerate already connected devices.
         *
         * The function looks for a few subsytems for devices. It traverses the
         * list and processes them.
         *
         * \returns DEVICEMANAGER_ERROR_MEM_ALLOC in case of failure in creating new CDevice
         *          DEVICEMANAGER_ERROR_UNKNOWN in case of any other API failure
         *          DEVICEMANAGER_OK in case of success
         */
        int LookupConnectedDevices();
#endif

        /*!
         * \brief prepares basic functionality for inotify mechanism
         *
         * Creates the folder /tmp/.automount
         */
        int InitializeInotifyMonitor() const;

        /*!
         * \brief Creates inotify watch over /tmp/.automount folder
         *
         * \return DEVICEMANAGER_OK in case of success
         *         DEVICEMANAGER_ERROR_MEM_ALLOC in case of failure in creating
         *         EVICEMANAGER_ERROR_UNKNOWN in case problem in adding
         *         subsystem filter and in case of failure to enable receiving
         */
        int CreateInotifyMonitor();


        /*!
           * \brief Creates watch on a pipe 
           *
           * \return DEVICEMANAGER_OK in case of success
           *         DEVICEMANAGER_ERROR_MEM_ALLOC in case of failure in creating
           *         EVICEMANAGER_ERROR_UNKNOWN in case problem in adding
           *         subsystem filter and in case of failure to enable receiving
           */

        int CreatePipeMonitor();

        /*!
         * \brief check for existing mount-points
         *
         * Checks the folder /tmp/.automount for the mounted devices. And
         * processes the mounted devices which have not been already notified
         * for connection.
         */
        tVoid LookupExistingMounts( CGlobalEnumerations::NOTIFICATION_Type enWDevTyp=CGlobalEnumerations::enNotifyAll, tCString fileName=NULL );

        /*!
         * \brief check for connected devices that are not mounted
         *
         * Checks the folder DIR_AUTOMOUNTER_MOUNTEDDEVICE_DB for the connected but not mounted devices. And
         * the devices not mounted may be due to corrupted file system or not supported file system or not having a existing file system
         *
         */
        tVoid LookupNotMountedDevices();

        /*!
         * \brief check for existing mount-points
         *
         * Checks the folder /tmp/.automount for the removed devices. And
         * processes the removed devices which have not been already notified
         * for disconnection.
         */
        tVoid LookupRemovedMounts();

        /*!
         * \brief: Processes a newly connected device - this function is used in combination with 
         *
         * \param [in] eNotiType  the service that informed about the event
         * \param [in] f_cDevNode device that is connected
         *
         * \return DEVICEMANAGER_DEVICE_DETECTED on successfull addition of the device
         */
        int AddDevice( CGlobalEnumerations::NOTIFICATION_Type eNotiType, GENSTRING f_cDevNode, const CDevice &l_oCurAvailDevInfo );
   
        //int AddDeviceToUnsupportedAtAllDevObserver(CGlobalEnumerations::NOTIFICATION_Type eNotiType, GENSTRING f_cDevNode, const CDevice &l_oCurAvailDevInfo ) //Roadmap 13001_ExtendedUserInformation
        //int DeleteDeviceFromUnsupportedAtAllDevObserver(CGlobalEnumerations::NOTIFICATION_Type eNotiType, GENSTRING f_cDevNode) //Roadmap 13001_ExtendedUserInformation
        //int TriggerSendUnsupportedAtAllDevice(CGlobalEnumerations::NOTIFICATION_Type eNotiType,GENSTRING f_cDevNode) //Roadmap 13001_ExtendedUserInformation
        // Adds the newly found device
        int AddDevice();

       
        int AddDeviceDVDRelated( GENSTRING f_cDevNode);

        /*!
         * \brief remove a device
         *
         * \param [in] f_cDevNode the device that is disconnected
         *
         * \return DEVICEMANAGER_ERROR_INVALID_PARAM, if f_cDevNode is EMPTY
         *         DEVICEMANAGER_ERROR_UNKNOWN, if unknown error
         *         DEVICEMANAGER_OK. if device removed successfully
         */
        int RemoveDevice( GENSTRING f_cDevNode);

        /*!
         * \brief Monitors udev events using monitor socket created in CreateMonitor.
         *
         * \return DEVICEMANAGER_ERROR_MEM_ALLOC in case of memory error
         *         DEVICEMANAGER_OK in case of success
         */
        int StartMonitorLooseBlocking();

        /*!
         * \brief Get the list of all connected devices.
         *
         * \param [out] f_vConnectedDevices fill this vector with list of
         *             connected devices
         */
        void GetAllConnectedDevices(vector<CDevice *> &f_vConnectedDevices) const;


        /*!
         * \brief Matches the serial number with the devices which are there in the vector.
         *
         * \param [in]  f_cSerialID serial number of the device to look for.
         * \param [out] f_pCDevice  device that has the provided serial id
         *
         * \return DEVICEMANAGER_ERROR_MEM_ALLOC in case of failure in creating new CDevice
         *          DEVICEMANAGER_ERROR_INVALID_PARAM in case f_cDeviceSerial is empty
         *          DEVICEMANAGER_DEVICE_DETECTED in case of success
         *          DEVICEMANAGER_ERROR_NODEVICE in case of no device detected
         */
        int LookupConnectedDevice(const GENSTRING f_cSerialID,  CDevice& f_oCDevice);

        /*!
         * \brief Displays all the Calibration parameters
         */
        void ShowCalibration() const;

#ifdef USE_FOURTH_USB_CONNECTOR
        /*!
         * \brief Get all the Dignosis values
         *
         * \param [in]  diag type i.e. if production line diagnosis is used to call this fct or other
         * \param [out] f_objUSBDiagnosisHUB1 container for diagnosis values for usb1
         * \param [out] f_objUSBDiagnosisHUB2 container for diagnosis values for usb2
         * \param [out] f_objUSBDiagnosisHUB3 container for diagnosis values for usb3
         * \param [out] f_objUSBDiagnosisHUB4 container for diagnosis values for usb4
         */
        void GetDiagnosisValues(CGlobalEnumerations::Diag_Type_e eDiagType,
                                                    CUSBDiagnosis &f_objUSBDiagnosisHUB1,
                                                    CUSBDiagnosis &f_objUSBDiagnosisHUB2,
                                                    CUSBDiagnosis &f_objUSBDiagnosisHUB3,
                                                    CUSBDiagnosis &f_objUSBDiagnosisHUB4);
#else
        /*!
        * \brief Get all the Dignosis values
        *
        * \param [in]  diag type i.e. if production line diagnosis is used to call this fct or other
        * \param [out] f_objUSBDiagnosisHUB1 container for diagnosis values for usb1
        * \param [out] f_objUSBDiagnosisHUB2 container for diagnosis values for usb2
        * \param [out] f_objUSBDiagnosisHUB3 container for diagnosis values for usb3
        */
        void GetDiagnosisValues(CGlobalEnumerations::Diag_Type_e eDiagType,
            CUSBDiagnosis &f_objUSBDiagnosisHUB1,
            CUSBDiagnosis &f_objUSBDiagnosisHUB2,
            CUSBDiagnosis &f_objUSBDiagnosisHUB3);
#endif
        tVoid connectUDEV(tBool bConnect);

        void  TraceConnectedHubs(tVoid);

        tBool lockNotification(tInt iTimout);
        tVoid lockNotification(tVoid);
        tVoid unlockNotification(tVoid);

#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
        void vPrintINotifyData         ( const char* strINotifyKeyWord[], char* strINotifyLine[], tUInt maxElem ) const;
        void vExtractContentMountDB    ( CDevice &f_oCurAvailDevInfo, char* strMountDBLine[],tUInt mountDB_line, tInt u16LenOfEntry ) const;
        void vExtractContentUnsupported( CDevice &f_oCurAvailDevInfo, char* strUnsupportedLine[], tUInt lineNo, tInt u16LenOfEntry ) const;
#endif //USE_EXTERNAL_UNSUPPORTED_DEVICES

#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
        void  vMapFileSystemType(INOUT CDevice *f_pCurAvailDevInfo) const;
#endif

               /*!
         * \brief see IUdevManagerInterfaceInput
         */
        virtual void vSendMountDevice(tBool bMount,GENSTRING devicename, GENSTRING mountPoint) const; //Roadmap 14002 MountCDROM (DiagMixeMode)


        /*!
         * \brief Sent the update about storage info of devices
         *
         * \param [in]  f_CurDevice details of the device with proper Storage data
         *
         */
        tVoid vUpdateStorageInfo(IN const CDevice f_CurDevice);
 
         /*!
         * \brief Sent the update about DVD Drive info
         *
         * \param [in]  f_DVDDriveInfo DVD drive info having Drivemode and Disctype
         *
         */
        void  vDVDDriveInfoUpdated(IN trDVDDriveInfo f_DVDDriveInfo);
        
		// For THREAD Function
        typedef enum
        {
           eThread_GetUdevManager = 0,
           eThread_Last       = 1
        }tenThreadFunction;
	
        /**
         * \brief Thread function -
         *
         * Temporary thread to respawn the CDROM detection flow (Fix for NCG3D-79558)
         */
        virtual tVoid vThreadFunction();
	
        /**
         * This function derives from TFThread and starts a user function in a thread from the thread factory's thread pool.
         * After leaving the function used thread is stopped and released for other users
         *
         * @return void
         */
        virtual void Do(int functionID, void *ptr);
};

#endif // __CUDEVMANAGER_H__

////////////////////////////////////////////////////////////////////////////////
// <EOF>
