/*-----------------------------------------------------------------------------*
 * StorageInfoProvider.cpp                                                                *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM Gen3                                                   *
 * COPYRIGHT   : (c) 2013-2020 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file StorageInfoProvider.cpp
 *
 * new StorageInfoProvider used to fetch the storage info to devices using a temporary thread


 * \version Initial Version
 * \ Version 24.08.2016, Rajeev Narayanan Sambhu (RBEI/EC03) (Bosch)
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2020
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"


#include "StorageInfoProvider.h"
#include "UDevManager.h"
#include "utils.h"
#include "DeviceManager_ErrorCodes.h"
#include "DeviceCard.h"
#include "device/DeviceListManager.h"





/*-----------------------------------------------------------------------------*
* ETG Tracing                                                                 *
*-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_UDEVMANAGER
#include "trcGenProj/Header/StorageInfoProvider.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
* name spaces                                                                 *
*-----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------*
* static variables of class                                                   *
*-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
StorageInfoProvider::StorageInfoProvider()
{
    m_pCUDevManager = NULL;
    m_IsThreadBusy  = FALSE;
}
/*-----------------------------------------------------------------------------*
 * Destructor                                                                 *
 *-----------------------------------------------------------------------------*/
StorageInfoProvider::~StorageInfoProvider()
{
    m_pCUDevManager = NULL;

}




/*-----------------------------------------------------------------------------*
int StorageInfoProvider::Initialize(tVoid)
 *-----------------------------------------------------------------------------*/
int StorageInfoProvider::Initialize(tVoid)
{
    tS32 s32Result = OSAL_OK;

    m_pCUDevManager = CUDevManager::GetInstance();

    return s32Result;
}

tVoid StorageInfoProvider::vGetSizeMSC(IN const CDevice &f_CurDevice )
{
    ETG_TRACE_USR4(("Begin: vGetSizeMSC "));
    if( TRUE == DeviceListManager::GetInstance()->isMassStorageDevice(f_CurDevice.m_eDeviceType) && 1 == f_CurDevice.m_iIsMounted)
    {

        CDevice *l_pCurDevice = new  CDevice();
        l_pCurDevice->FillDevice(f_CurDevice);

        tInt iThreadIndex = ThreadFactoryDVM::GetThreadFactory()->Do(IN this, (int)StorageInfoProvider::eThread_GetStorageInfo, (void*)l_pCurDevice);
        ETG_TRACE_COMP(("StorageInfoProvider: iThreadIndex:%d",iThreadIndex));
    }
    ETG_TRACE_USR4(("End: vGetSizeMSC "));

}

void StorageInfoProvider::vThreadFunction(IN CDevice *f_pCurDevice)
{
    ETG_TRACE_USR4(("Begin: vThreadFunction"));



    CDevice l_CurDevice = *f_pCurDevice;
    delete f_pCurDevice;


    int iRes;
    StorageInfo_t f_oInfo;

    //check if size is as expected:
    if(sizeof(unsigned long long) != sizeof(fsblkcnt_t))
    {
        ETG_TRACE_FATAL(("[WARNING]: unexpected difference in size of type fsblkcnt_t of struct statfs"));
        ETG_TRACE_FATAL(("[WARNING]: sizeof(unsigned long long)=%d",sizeof(unsigned long long)));
        ETG_TRACE_FATAL(("[WARNING]: sizeof(fsblkcnt_t)=%d",sizeof(fsblkcnt_t)));
    }


    char *p_strMountPoint = strdup(l_CurDevice.m_cMountPoint.toStdString().c_str());
    if(p_strMountPoint)
    {
        ETG_TRACE_USR4 (("vThreadFunction: Mount point strMountPoint: %s", p_strMountPoint));
        iRes = FillStorageInfo((char*)p_strMountPoint,f_oInfo);
        free(p_strMountPoint);
        ETG_TRACE_USR4(("vThreadFunction:sizeof(unsigned long long) = %d",sizeof(unsigned long long))); //on GM it is 4  hence 4*8 = 32bit Maxvalue is 2^(32-1)-1 = 2147483647 i.e. printed as a string 10 +1 for end of lstring '\0'


        char strTotalSize_KB[80]; //11 would be enough for 64bit max num would be 9223372036854775807 i.e. length 19+1 includign end of string '\0'
        char strUsedSpace_KB[80];
        char strFreeSpace_KB[80];

        if(iRes == DEVICEMANAGER_OK)
        {

            //improve check here
            if(sizeof(unsigned long long) > sizeof(unsigned long))
            {
                ETG_TRACE_FATAL(("[WARNING]: sizeof(unsigned long long)=%d",sizeof(unsigned long long)));
                ETG_TRACE_FATAL(("[WARNING]: sizeof(unsigned long int)=%d",sizeof(unsigned long int)));
            }
            //handover size in KB to string
            snprintf(strTotalSize_KB, sizeof(strTotalSize_KB),"%lu",(unsigned long)((f_oInfo.total_size)/1024)); //had trouble pritning unsigned long long since 4 bytes are used currently I will cast it. Which is not very nice
            snprintf(strUsedSpace_KB, sizeof(strUsedSpace_KB) , "%lu",(unsigned long)((f_oInfo.used_space)/1024));
            snprintf(strFreeSpace_KB, sizeof(strFreeSpace_KB) , "%lu",(unsigned long)((f_oInfo.free_space)/1024));


            l_CurDevice.m_u64TotalSize_KB = (tU64)atoll(strTotalSize_KB);
            l_CurDevice.m_u64UsedSize_KB  = (tU64)atoll(strUsedSpace_KB);
            l_CurDevice.m_u64FreeSize_KB  = (tU64)atoll(strFreeSpace_KB);
            vUpdateStorageInfo(l_CurDevice);

        }
    }
    else
    {
        ETG_TRACE_USR4 (("vThreadFunction: Mount point is NULL"));
    }
    ETG_TRACE_USR4(("End  : vThreadFunction"));
}


tVoid StorageInfoProvider::vUpdateStorageInfo(OUT CDevice f_CurDevice)
{
    ETG_TRACE_USR4(("Begin : vUpdateStorageInfo "));


    if(m_pCUDevManager)
    {
        m_pCUDevManager->vUpdateStorageInfo(f_CurDevice);
    }

    ETG_TRACE_USR4(("End   : vUpdateStorageInfo"));

}



/*-----------------------------------------------------------------------------*
tVoid StorageInfoProvider::Do(...)
 *-----------------------------------------------------------------------------*/
tVoid StorageInfoProvider::Do(int functionID, void *ptr)
{
    ETG_TRACE_USR4(("Begin:Do "));
    tenThreadFunction eFunctionID = (tenThreadFunction)functionID;
    switch(eFunctionID)
    {
        case StorageInfoProvider::eThread_GetStorageInfo: //execution tree for thread eThread_SystemVolt
        {
            ThreadFactoryDVM::GetThreadFactory()->SetName("VD_DVM:eThread_GetStorageInfo");
            m_StrongInfoLock.lock();
            m_IsThreadBusy = TRUE;
            CDevice *l_CurrDevice = (CDevice*)ptr;
            if(l_CurrDevice)
            {
                vThreadFunction(l_CurrDevice);
            }
            m_IsThreadBusy  = FALSE;
            m_StrongInfoLock.unlock();
            break;
        }
        default:
            break;
    }
    ETG_TRACE_USR4(("End  :Do"));
}

/*-----------------------------------------------------------------------------*
tBool IsFreeSlotsAvailable()
 *-----------------------------------------------------------------------------*/
tBool StorageInfoProvider::IsFreeSlotsAvailable()
{
    return !m_IsThreadBusy;
}


////////////////////////////////////////////////////////////////////////////////
// <EOF>



