
/*-----------------------------------------------------------------------------*
 *DeviceRecognizer.cpp                                                              *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen3                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file DeviceRecognizer.cpp
 *
 * \brief This file holds the implementation of DeviceRecognizer. It hides specific
 *-----------------------------------------------------------------
 *                                    development for Gen3:
 *-----------------------------------------------------------------
 *\version 17.12.2014, Christian Koechling (Bosch)
 *
 *
.*
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */


#include <stdio.h>
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL  //statetable uses ahl thread otherwhise this
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"
#include "Enums.h"
#include "Device.h"
#include "DeviceManager_ErrorCodes.h"
#include "config/ConfigurationManager.h"
#include "debug/HistoryManager.h"
#include "utils.h"
#include "usbutils.h"


#include "IDeviceRecognizer.h"
#include "DeviceRecognizer.h"

#include <blkid/blkid.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/*-----------------------------------------------------------------------------*
 * namespace                                                                     *
 *-----------------------------------------------------------------------------*/
//using namespace prmmgr;
//using namespace statetbl;

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_UDEVMANAGER
#include "trcGenProj/Header/DeviceRecognizer.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * defines                                                                 *
 *------------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * constants                                                                 *
 *------------------------------------------------------------------------------*/
const uint16_t AppleVendorID = 0x05ac;



/*-----------------------------------------------------------------------------*
 * function definitions                                                                *
 *------------------------------------------------------------------------------*/
DeviceRecognizer::DeviceRecognizer()
{
}


DeviceRecognizer::~DeviceRecognizer()
{
}

int DeviceRecognizer::FillCurrDevice(INOUT CDevice& f_oCurDevice, IN GENSTRING f_cDevNode,IN struct udev_device *f_pBaseUSBDev,IN struct udev *f_pUdev)
{
    return -1; //don't use yet under contruction
}

tBool DeviceRecognizer::AdjustMTPDeviceNotification(OUT CDevice* f_pCurDevice,IN LIBMTP_raw_device_t *f_pMTPRawdevices) const
{
    LIBMTP_mtpdevice_t *pDevice = NULL;
    tU8 u8SuccessCount = 0;
    tBool bRet = TRUE;
    char *strModelname,*strFriendlyname,*strSerialnumber,*strDeviceversion;

    if(f_pMTPRawdevices != NULL)
    {
        pDevice = LIBMTP_Open_Raw_Device_Uncached(f_pMTPRawdevices);

        if (pDevice != NULL)///@todo check if NULL is returned in error case
        {
            ETG_TRACE_USR4(("AdjustMTPDeviceNotification: pDevice      =%p",pDevice));
            ETG_TRACE_USR4(("AdjustMTPDeviceNotification: pDevice->next=%p",pDevice->next));
            strModelname        = LIBMTP_Get_Modelname(pDevice);
            strFriendlyname     = LIBMTP_Get_Friendlyname(pDevice);           //<--?
            strSerialnumber     = LIBMTP_Get_Serialnumber(pDevice);           //<--compare to mountpoint
            strDeviceversion    = LIBMTP_Get_Deviceversion(pDevice);          //<--compare to mountPoint

            if(NULL != strModelname)     {ETG_TRACE_USR4(("AdjustMTPDeviceNotification:LIBMTP_Get_Modelname     : %s",strModelname));}
            if(NULL != strFriendlyname)  {ETG_TRACE_USR4(("AdjustMTPDeviceNotification:LIBMTP_Get_Friendlyname  : %s",strFriendlyname));}
            if(NULL != strSerialnumber)  {ETG_TRACE_USR4(("AdjustMTPDeviceNotification:LIBMTP_Get_Serialnumber  : %s",strSerialnumber));}
            if(NULL != strDeviceversion) {ETG_TRACE_USR4(("AdjustMTPDeviceNotification:LIBMTP_Get_Deviceversion : %s",strDeviceversion));}

            //RawDevNum added to Accessory as additional information formerly it was added to serial number
#ifdef WORKAROUND_MTP_NODEVICENAME

            if(strModelname != NULL)
            {
#ifdef USE_MYSPIN_CHANGE
                f_pCurDevice->m_cAccessoryName = f_pCurDevice->m_cSerialID;
#else
                f_pCurDevice->m_cAccessoryName = strModelname;
#endif
            }

#else  //WORKAROUND_MTP_NODEVICENAME
            f_pCurDevice->m_cAccessoryName = "#";
            f_pCurDevice->m_cAccessoryName.append(GENSTRING_NUMBER(f_pCurDevice->m_iDevNUM));
#endif //WORKAROUND_MTP_NODEVICENAME


            //--------------
            //SERIAL NUMBER
            //--------------
            if(strSerialnumber != NULL)
            {
                f_pCurDevice->m_cShortSerial = strSerialnumber;
                f_pCurDevice->m_cSerialID    = strSerialnumber;
                u8SuccessCount++;
            }
            else
            {
                ETG_TRACE_FATAL(("[ERROR] AdjustMTPDeviceNotification: strSerialnumber not adjusted"));
            }
            //--------------
            //DEVICE VERSION
            //--------------
            if(strDeviceversion != NULL)
            {
                f_pCurDevice->m_cDeviceVersion = strDeviceversion;
                u8SuccessCount++;
            }
            else
            {
                ETG_TRACE_FATAL(("[ERROR] AdjustMTPDeviceNotification: strDeviceversion NULL f_pCurDevice->m_cDeviceVersion not adjusted"));
            }
            //--------------
            //DEVICE NAME
            //--------------
            if(strFriendlyname != NULL)
            {
                f_pCurDevice->m_cDeviceName = strFriendlyname;
                u8SuccessCount++;
            }
            else
            {
                ETG_TRACE_FATAL(("[WARNING] AdjustMTPDeviceNotification: strFriendlyname NULL try strModelname"));
                if(strModelname != NULL)
                {
                    f_pCurDevice->m_cDeviceName = strModelname;
                    u8SuccessCount++;
                }
                else
                {
                    ETG_TRACE_FATAL(("[ERROR] AdjustMTPDeviceNotification: strModelname NULL  f_pCurDevice->m_cDeviceName not adjusted"));
                }

            }

            if(u8SuccessCount==3) /*3 required elements above received*/
            {
                f_pCurDevice->m_cMountPoint = f_pCurDevice->m_cDeviceName + "/" + f_pCurDevice->m_cDeviceVersion + "/" + f_pCurDevice->m_cShortSerial + "/";
            }
            else
            {
                ETG_TRACE_FATAL(("[ERROR] AdjustMTPDeviceNotification: u8SuccessCount=%d (!=3)f_pCurDevice->m_cMountPoint not adjusted",u8SuccessCount));
                bRet = FALSE;
            }

            //free memory again
            if(NULL != strModelname)     {free(strModelname);}
            if(NULL != strFriendlyname)  {free(strFriendlyname);}
            if(NULL != strSerialnumber)  {free(strSerialnumber);}
            if(NULL != strDeviceversion) {free(strDeviceversion);}

            //free device
            LIBMTP_Release_Device(pDevice);
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR] AdjustMTPDeviceNotification: could not open raw device (gen3)"));
            bRet = FALSE;
        }
    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR] AdjustMTPDeviceNotification: f_pMTPRawdevices == NULL"));
        bRet = FALSE;
    }
    return bRet;
}


/*-----------------------------------------------------------------------------*
 * int DetectMTPDevice()                                                       *
 *-----------------------------------------------------------------------------*/
int DeviceRecognizer::DetectMTPDevice(INOUT CDevice* f_pCurDevice)
{
    ETG_TRACE_USR4 (("Begin : DetectMTPDevice"));
    //variable to hold the return value
    int l_iIsMTPDeviceFound = DEVICEMANAGER_ERROR_NODEVICE;

    if ((DVM_OFF == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_EXTERNAL_MTP)))
    {
        ETG_TRACE_ERR (("DetectMTPDevice : **** MTP Support is disabled ****"));
    }
    else
    {
        int     l_iDeviceIterator = 0;  // Iterator for total number of devices
        int     l_idevnum         = 0;  // Device Number
        int     l_iRawDevNum      = 0;  // Raw Device Number
        GENSTRING l_cSerialDevNum   = ""; // Short Device Serial

        if(NULL != f_pCurDevice)
        {
            if(NULL != m_pMTPRawdevices)
            {
                free(m_pMTPRawdevices);
                m_pMTPRawdevices = NULL;
                m_LIBMTP_Error = LIBMTP_ERROR_NONE;
                m_iNumrawDevices = 0;
            }
            try
            {
                m_LIBMTP_Error = LIBMTP_Detect_Raw_Devices(&m_pMTPRawdevices, &m_iNumrawDevices);
            }
            catch(...)
            {   //lint !e1775 Info 1775: prio2: catch block does not catch any declared exception -for default exception that's ok
                ETG_TRACE_ERR (("DetectMTPDevice: Exception in LIBMTP_Detect_Raw_Devices"));

                return l_iIsMTPDeviceFound;
            }

            if(m_LIBMTP_Error == LIBMTP_ERROR_NONE && m_iNumrawDevices > 0 )
            {
                while(l_iDeviceIterator < m_iNumrawDevices)
                {
                    l_idevnum    = m_pMTPRawdevices[l_iDeviceIterator].devnum;
                    l_iRawDevNum = f_pCurDevice->m_iDevNUM;

                    ETG_TRACE_USR3 (("DetectMTPDevice: Device NUM     = %d", l_idevnum));
                    ETG_TRACE_USR3 (("DetectMTPDevice: Raw Device NUM = %d", l_iRawDevNum));

                    if (l_idevnum == l_iRawDevNum)
                    {
                        //set the device type as MTP
                        f_pCurDevice->m_eDeviceType = CGlobalEnumerations::DTY_MTP;
                        f_pCurDevice->m_iIsMounted = 0;
                        //assign serial number to the detected MTP device
                        f_pCurDevice->m_cSerialID =  f_pCurDevice->m_cShortSerial;
                        l_cSerialDevNum=f_pCurDevice->m_cShortSerial;
#ifndef BUGFIX_MTP_SERIALNUMBER
                        l_cSerialDevNum.append("#");
                        l_cSerialDevNum.append(GENSTRING_NUMBER(f_pCurDevice->m_iDevNUM));
#endif

                        ETG_TRACE_USR3 (("DetectMTPDevice: l_cSerialDevNum        = %s",
                                         l_cSerialDevNum.toStdString().c_str()));

                        f_pCurDevice->m_cShortSerial = l_cSerialDevNum;
                        f_pCurDevice->m_cSerialID =  f_pCurDevice->m_cShortSerial;

                        //l_iDeviceIterator+1 is added to avoid confusion while analysing traces
                        ETG_TRACE_USR1 (("DetectMTPDevice: MTP Device Found =  %d", l_iDeviceIterator+1));

                        //set the return value to true indicating that MTP device is found
                        l_iIsMTPDeviceFound = DEVICEMANAGER_DEVICE_DETECTED;



                        int iRetrial = 0;
                        while(iRetrial < 2)
                        {
                            if(TRUE == AdjustMTPDeviceNotification(OUT f_pCurDevice, IN &m_pMTPRawdevices[l_iDeviceIterator]))
                            {
                                l_iIsMTPDeviceFound = DEVICEMANAGER_DEVICE_DETECTED;
                                if(iRetrial >0)
                                {
                                    ETG_TRACE_FATAL(("DetectMTPDevice: DTY_MTP  device mountpoint obtained (after a retrial) %s",f_pCurDevice->m_cMountPoint.toStdString().c_str()));
                                }
                                ETG_TRACE_COMP(("DetectMTPDevice: DTY_MTP  device mountpoint obtained %s",f_pCurDevice->m_cMountPoint.toStdString().c_str()));
                                break; //found leave while loop
                            }
                            else
                            {
                                iRetrial++;
                                l_iIsMTPDeviceFound = DEVICEMANAGER_ERROR_NODEVICE;
                                //TODO use a different macro for the timeout
                                usleep((useconds_t)(USE_DELAY_RETRIAL_SEARCH_MTP_MOUNTPOINT*1000));
                                ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                                ETG_TRACE_FATAL(("Retrial after %d ms - looking for DTY_MTP mountpoint",USE_DELAY_RETRIAL_SEARCH_SOUNDDEV));
                                ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!"));

                            }
                        }


                        ETG_TRACE_USR4 (("End : DetectMTPDevice"));

                        return l_iIsMTPDeviceFound;
                    }
                    else
                    {
                        ETG_TRACE_USR1 (("DetectMTPDevice: Does not support MTP = %d", l_iDeviceIterator));
                    }

                    l_iDeviceIterator++;
                }
            }
            else
            {
                ETG_TRACE_USR1 (("DetectMTPDevice: No MTP Devices found"));
            }
        }
    }


    ETG_TRACE_USR4 (("End : DetectMTPDevice"));
    return l_iIsMTPDeviceFound;
}

/*-----------------------------------------------------------------------------*
 * int DetectMACDevices(CDevice* f_pCurDevice)                                 *
 *-----------------------------------------------------------------------------*/
int DeviceRecognizer::DetectMACDevices(INOUT CDevice* f_pCurDevice) const
{
    ETG_TRACE_USR4(("Begin: DetectMACDevices"));

    //variable to hold the return value
    int l_iAppleDeviceFound = DEVICEMANAGER_ERROR_NODEVICE;

    //check pointer
    if( NULL != f_pCurDevice )
    {
        GENSTRING l_cLowerSerialID = f_pCurDevice->m_cSerialID.toLower();

        //detect Apple device bz vendor and product id
        if ((f_pCurDevice->m_iVendorID == AppleVendorID)
                && (0x12 == (f_pCurDevice->m_iProductID >> 8)))
        {
            ETG_TRACE_USR1(("DetectMACDevices:Serial ID of device : %s",
                            l_cLowerSerialID.toStdString().c_str()));

            //---------------------
            //care for IPOD
            //---------------------
            if(true == l_cLowerSerialID.contains("ipod"))
            {
                ETG_TRACE_COMP(("DetectMACDevices: IPOD found"));

                f_pCurDevice->m_eDeviceType = CGlobalEnumerations::DTY_IPOD;
                l_iAppleDeviceFound = DEVICEMANAGER_DEVICE_DETECTED;
            }
            //---------------------
            //care for IPAD
            //---------------------
            else if(true == l_cLowerSerialID.contains("ipad"))
            {
                if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_EXTERNAL_IPAD))
                {
                    ETG_TRACE_COMP(("DetectMACDevices: IPAD found"));
                    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eConf_ENABLE_NOTIFY_IPAD_AS_IPOD))
                    {
                        f_pCurDevice->m_eDeviceType = CGlobalEnumerations::DTY_IPOD;
                    }
                    else
                    {
                        f_pCurDevice->m_eDeviceType = CGlobalEnumerations::DTY_IPAD;
                    }
                    l_iAppleDeviceFound = DEVICEMANAGER_DEVICE_DETECTED;
                }
                else
                {
                    l_iAppleDeviceFound = DEVICEMANAGER_ERROR_UNKNOWN;
                    ETG_TRACE_FATAL(("[ok but check] DetectMACDevices: IPAD unsupperted"));
                }
            }
            //---------------------
            //care for IPHONE
            //---------------------
            else if(true == l_cLowerSerialID.contains("iphone"))
            {
                ETG_TRACE_COMP(("DetectMACDevices:IPHONE found"));

                f_pCurDevice->m_eDeviceType = CGlobalEnumerations::DTY_IPHONE;
                l_iAppleDeviceFound = DEVICEMANAGER_DEVICE_DETECTED;
            }
            else
            {
                l_iAppleDeviceFound = DEVICEMANAGER_ERROR_UNKNOWN;

                ETG_TRACE_FATAL(("[ok but check] DetectMACDevices: Unsupported Apple device"));
            }

            /*result ok*/
            if( (DEVICEMANAGER_DEVICE_DETECTED == l_iAppleDeviceFound) ||
                    (DEVICEMANAGER_ERROR_UNKNOWN   == l_iAppleDeviceFound))  //unknown but chargable - see config.h (USE_CHARGE_APPLE & eCONF_ENABLE_EXTERNAL_IPAD)
            {
                ETG_TRACE_USR4(("DetectMACDevices:f_pCurDevice->m_cDevnode:      %s",
                                f_pCurDevice->m_cDevnode.toStdString().c_str()));
                ETG_TRACE_USR4(("DetectMACDevices:f_pCurDevice->m_cShortSerial:  %s",
                                f_pCurDevice->m_cSerialID.toStdString().c_str()));
                ETG_TRACE_USR4(("DetectMACDevices:f_pCurDevice->m_eDeviceType:   %d", ETG_ENUM(DVM_DEVICE_TYPE,f_pCurDevice->m_eDeviceType)));
                ETG_TRACE_USR4(("DetectMACDevices:f_pCurDevice->m_cDeviceVersion:%s",
                                f_pCurDevice->m_cDeviceVersion.toStdString().c_str()));


                if (DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_CHARGE_APPLEDEVS))
                {
                    ETG_TRACE_USR4(("DetectMACDevices: eCONF_ENABLE_CHARGE_APPLEDEVS: DVM_ON"));
                            ChargeAppleDevice(f_pCurDevice->m_iVendorID, f_pCurDevice->m_iProductID);
                }
            } // if(DEVICEMANAGER_DEVICE_DETECTED)
        }
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
        else if (   (f_pCurDevice->m_iVendorID == AppleVendorID)
                    && ((f_pCurDevice->m_iProductID >> 8) == 0x13) )
        {
            ETG_TRACE_USR1(("DetectMACDevices: Serial ID of device : %s", l_cLowerSerialID.toStdString().c_str()));
            if( l_cLowerSerialID.contains("ipod") == true )
            {
                ETG_TRACE_COMP(( "DetectMACDevices: not supported IPOD found. Change m_eDeviceType to DTY_NOT_SUPPORTED" ));
                f_pCurDevice->m_eDeviceType = CGlobalEnumerations::DTY_NOT_SUPPORTED;
                l_iAppleDeviceFound = DEVICEMANAGER_DEVICE_DETECTED;
            }
            else
            {
                ETG_TRACE_USR1(("DetectMACDevices: No not supported iPod device detected"));
            } //if (f_pCurDevice->m_cVendorID == AppleVendorID)
        }
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
        else
        {
            ETG_TRACE_USR1(("DetectMACDevices: Not Apple device"));
        } //if (f_pCurDevice->m_cVendorID == AppleVendorID)
    }
    else
    {
        ETG_TRACE_ERR(("DetectMacDevice: [ERROR] NULL Pointer f_pCurDevice"));
        l_iAppleDeviceFound = DEVICEMANAGER_ERROR_INVALID_PARAM;
    } // if( NULL != f_pCurDevice )

    ETG_TRACE_USR4 (("End  : CUDevManager::DetectMAcDevices"));

    return l_iAppleDeviceFound;
}

int DeviceRecognizer::DetectDevice(IN CGlobalEnumerations::DETECTION_Type enDetectionType, INOUT CDevice* f_pCurDevice)
{
    int l_iIsDeviceFound = CGlobalEnumerations::enBlockDevice;
    ETG_TRACE_USR4(("Begin: DetectDevice enDetectionType:%d",ETG_CENUM(CGlobalEnumerations::DETECTION_Type,enDetectionType)));

    switch(enDetectionType)
    {
        case CGlobalEnumerations::enMTPDevice:
            l_iIsDeviceFound = DetectMTPDevice(INOUT f_pCurDevice);
            break;
        case CGlobalEnumerations::enAppleDev:
            l_iIsDeviceFound = DetectMACDevices(INOUT f_pCurDevice);
            break;
        case CGlobalEnumerations::enBlockDevice:
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("End  : DetectDevice"));
    return l_iIsDeviceFound;
}

////////////////////////////////////////////////////////////////////////////////
// <EOF>

