//@todo: This cold has access to text interface of automounter and shared lib interface
// Recommendation is to put different behaviour into its own class. Decorator pattern is used to give either or bheaviour to the
// VD_DVM. 

/*-----------------------------------------------------------------------------*
 * AutomounterAdaptSharedLibIF.cpp                                                                  *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM Gen3                                                   *
 * COPYRIGHT   : (c) 2013-2020 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file AutomounterAdaptSharedLibIF.cpp
 *
 * helper class to connect VD_Devicemanager to new automounter mechanism of Adit
 

 * \version Initial Version
 * \version 03.12.2013, Koechling, Christian (Bosch)
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2020
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include <stdexcept>
#include <string>
#include <unistd.h>
#include <iostream>
#include <fstream>
#include <unistd.h>

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"

#include "AutomounterAdaptSharedLibIF.h"
#include "config/ConfigurationManager.h"

#include <signal.h>
#include <sys/epoll.h>
#include <errno.h>
#include <poll.h>
//#include <sys/inotify.h>
//#include <unistd.h>
#include "Utils.h"

/*-----------------------------------------------------------------------------*
* ETG Tracing                                                                 *
*-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"
     
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_UDEVMANAGER
#include "trcGenProj/Header/AutomounterAdaptSharedLibIF.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST



//see /opt/tooling/imx-staging/DI_BINARY_REPOSITORY_LINUX_xxx/usr/include
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
extern "C" //shared lib is a c-shared lib
{
    #include "automounter_api.h"
    #include "automounter_api_ctrl.h"
    #include "automounter_api_events.h"
    #include "automounter_api_info.h"
    #include "automounter_types.h"
}
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
* name spaces                                                                 *
*-----------------------------------------------------------------------------*/
//using namespace automounteradapter;


/*-----------------------------------------------------------------------------*
* static variables of class                                                                 *
*-----------------------------------------------------------------------------*/
#ifdef TEST_AUTOMOUNTER_REGISTER_ALL_CALLBACKFUNCTIONS
static automounter_api_callbacks_t  rFunction = 
{
    .on_establish_connection_success = &AutomounterAdaptSharedLibIF::establish_connection_success,
    .on_establish_connection_failure = &AutomounterAdaptSharedLibIF::establish_connection_failure,
    .on_connection_lost              = &AutomounterAdaptSharedLibIF::connection_lost,
    .on_device_detected              = &AutomounterAdaptSharedLibIF::device_detected,
    .on_device_nomedia               = &AutomounterAdaptSharedLibIF::device_nomedia,
    .on_device_automounting          = &AutomounterAdaptSharedLibIF::device_automounting,
    .on_device_automounted           = &AutomounterAdaptSharedLibIF::device_automounted,
    .on_device_unmounting            = &AutomounterAdaptSharedLibIF::device_unmounting,
    .on_device_unmounted             = &AutomounterAdaptSharedLibIF::device_unmounted,
    .on_device_invalid               = &AutomounterAdaptSharedLibIF::device_invalid,
    .on_partition_detected           = &AutomounterAdaptSharedLibIF::partition_detected,
    .on_partition_unsupported        = &AutomounterAdaptSharedLibIF::partition_unsupported,
    .on_partition_mounting           = &AutomounterAdaptSharedLibIF::partition_mounting,
    .on_partition_mounted            = &AutomounterAdaptSharedLibIF::partition_mounted,
    .on_partition_mount_err          = &AutomounterAdaptSharedLibIF::partition_mount_err,
    .on_partition_remounting         = &AutomounterAdaptSharedLibIF::partition_remounting,
    .on_partition_unmounting         = &AutomounterAdaptSharedLibIF::partition_unmounting,
    .on_partition_unmounted          = &AutomounterAdaptSharedLibIF::partition_unmounted,
    .on_partition_invalid            = &AutomounterAdaptSharedLibIF::partition_invalid,
    .on_update_device_info           = &AutomounterAdaptSharedLibIF::update_device_info,
    .on_update_partition_info        = &AutomounterAdaptSharedLibIF::update_partition_info,
    .on_snapshot_complete            = &AutomounterAdaptSharedLibIF::snapshot_complete,
};
#else  //for performance reason less callbacks are registered
static automounter_api_callbacks_t  rFunction = 
{
    .on_establish_connection_success = &AutomounterAdaptSharedLibIF::establish_connection_success,
    .on_establish_connection_failure = &AutomounterAdaptSharedLibIF::establish_connection_failure,
    .on_connection_lost              = &AutomounterAdaptSharedLibIF::connection_lost,
    .on_device_detected              = &AutomounterAdaptSharedLibIF::device_detected,
    .on_device_nomedia               = NULL,
    .on_device_automounting          = NULL,
    .on_device_automounted           = NULL,
    .on_device_unmounting            = NULL,
    .on_device_unmounted             = &AutomounterAdaptSharedLibIF::device_unmounted,
    .on_device_invalid               = &AutomounterAdaptSharedLibIF::device_invalid,
    .on_partition_detected           = NULL,
    .on_partition_unsupported        = &AutomounterAdaptSharedLibIF::partition_unsupported,
    .on_partition_mounting           = NULL,
    .on_partition_mounted            = &AutomounterAdaptSharedLibIF::partition_mounted,
    .on_partition_mount_err          = &AutomounterAdaptSharedLibIF::partition_mount_err,
    .on_partition_remounting         = NULL,
    .on_partition_unmounting         = NULL,
    .on_partition_unmounted          = &AutomounterAdaptSharedLibIF::partition_unmounted,
    .on_partition_invalid            = &AutomounterAdaptSharedLibIF::partition_invalid,
    .on_update_device_info           = &AutomounterAdaptSharedLibIF::update_device_info,
    .on_update_partition_info        = &AutomounterAdaptSharedLibIF::update_partition_info,
    .on_snapshot_complete            = &AutomounterAdaptSharedLibIF::snapshot_complete,
};

#endif



/*-----------------------------------------------------------------------------*
* class                                                                 *
*-----------------------------------------------------------------------------*/



/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
AutomounterAdaptSharedLibIF::AutomounterAdaptSharedLibIF()
{
}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                 *
 *-----------------------------------------------------------------------------*/
AutomounterAdaptSharedLibIF::~AutomounterAdaptSharedLibIF()
{
}


/*-----------------------------------------------------------------------------*
 * printElemOfStruct(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::printElemOfStruct(const char* strItemName,const char* strItem )
{
    char Buffer[80];
    //sleep(1);
    memset(Buffer,'\0',80);
    if(strItemName)
    {
        if(strItem)
        {
           snprintf(Buffer,79,"%s: %s",strItemName,strItem); 
        }
        else
        {
           snprintf(Buffer,79,"%s: -",strItemName); 
        }
        ETG_TRACE_USR4(("%s",Buffer));
    }
    
}

/*-----------------------------------------------------------------------------*
 * printiElemOfStruct(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::printiElemOfStruct(const char* strItemName,int iValue )
{
    char Buffer[80];
    memset(Buffer,'\0',80);
    snprintf(Buffer,79,"%s: %d",strItemName,iValue); 
}

/*-----------------------------------------------------------------------------*
 * printbElemOfStruct(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::printbElemOfStruct(const char* strItemName,bool bValue )
{
    char Buffer[80];
    memset(Buffer,'\0',80);
    if(bValue)
    {
        snprintf(Buffer,79,"%s: true",strItemName); 
    }
    else
    {
        snprintf(Buffer,79,"%s: false",strItemName); 
    }
}

/*-----------------------------------------------------------------------------*
 * print_device_info(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::print_device_info(const device_info_t *device_info)
{
    int lenOfString;
    char Buffer[80];
    memset(Buffer,'\0',80);

    snprintf(Buffer,79,"State-Device   : %s",automounter_api_get_device_state_string(device_info->state));
    ETG_TRACE_USR4(("%s",Buffer));

/* see //see "%_SWBUILDROOT%/generated/staging/$(COMPILERENV)/usr/include/automounter/automounter_types.h"/>
typedef struct __attribute__((aligned(8),packed)) device_info_t
{
	const char *interface_id;
	device_state_t state;
	const char *identifier;
	int detected_partition_cnt;
	const char *device_type;
	const char *device_handler_id;
	size_t data_structure_real_size;
	long extension_bitmask;
} device_info_t;

use automounterctl monitor to see what happens

*/

    ETG_TRACE_USR4(("-------------------------------"));
    ETG_TRACE_USR4(("DeviceInfo   :"));

    //----------
    //interface_id
    //----------
    if(device_info->interface_id)
    {
        lenOfString = strlen_r(device_info->interface_id);
        ETG_TRACE_USR4(("    device_info->interface_id-len      : %d" ,lenOfString));
        ETG_TRACE_USR4(("    device_info->interface_id          : %s" ,device_info->interface_id));
    }
    else
    {
        ETG_TRACE_USR4(("    device_info->interface_id          : - " ));
    }

    //----------
    //identifier
    //----------
    if(device_info->identifier)
    {
        lenOfString = strlen_r(device_info->identifier);
        ETG_TRACE_USR4(("    device_info->identifier-len        : %d" ,lenOfString));
        ETG_TRACE_USR4(("    device_info->identifier            : %s" ,device_info->identifier));
    }
    else
    {
        ETG_TRACE_USR4(("    device_info->identifier            : - " ));
    }

    //-------------------
    //detected_partition_cnt
    //-------------------
    ETG_TRACE_USR4(("    device_info->detected_partition_cnt    : %d" ,device_info->detected_partition_cnt));

    //-------------------
    //device_handler_id
    //-------------------
    if(device_info->device_handler_id)
    {
        lenOfString = strlen_r(device_info->device_handler_id);
        ETG_TRACE_USR4(("    device_info->device_handler_id-len : %d" ,lenOfString));
        ETG_TRACE_USR4(("    device_info->device_handler_id     : %s" ,device_info->device_handler_id));
    }
    else
    {
        ETG_TRACE_USR4(("    device_info->device_handler_id     : - "));
    }

    //-------------------
    //device_type
    //-------------------
    if(device_info->device_type) // sim4hi: fix for Coverity CID-19984
    {
        lenOfString = strlen_r(device_info->device_type);
        ETG_TRACE_USR4(("    device_info->device_type-len       : %d" ,lenOfString));
        ETG_TRACE_USR4(("    device_info->device_type           : %s" ,device_info->device_type));
    }
    else
    {
        ETG_TRACE_USR4(("    device_info->device_type           : - "));
    }

    ETG_TRACE_USR4(("-------------------------------"));


}


/*-----------------------------------------------------------------------------*
 * print_part_info(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::print_part_info(const partition_info_t *part_info)
{
    char Buffer[80];
 

    memset(Buffer,'\0',80);
    snprintf(Buffer,79,"State-Partition: %s",automounter_api_get_partition_state_string(part_info->state));
    ETG_TRACE_USR4(("%s",Buffer));



/* see //see "%_SWBUILDROOT%/generated/staging/$(COMPILERENV)/usr/include/automounter/automounter_types.h"/>
typedef struct __attribute__((aligned(8),packed)) partition_info_t
{
    const char *interface_id;
    partition_state_t state;
    partition_unsupported_reason_t unsupported_reason;
    error_code_t error;
    bool mounted_writable;
    int partition_no;
    const char *identifier;
    const char *mount_src;
    const char *mount_fs;
    const char *mount_point;
    size_t data_structure_real_size;
    long extension_bitmask;
}

*/

    ETG_TRACE_USR4(("-------------------------------"));
    ETG_TRACE_USR4(("PartitionInfo:"));
    if(part_info->interface_id)
    {
        ETG_TRACE_USR4(("     part_info->interface_id     :%s",part_info->interface_id));
    }
    else
    {
        ETG_TRACE_USR4(("     part_info->interface_id     :-"));
    }

    if(part_info->identifier)
    {
        ETG_TRACE_USR4(("    part_info->identifier        :%s",part_info->identifier));
    }
    else
    {
        ETG_TRACE_USR4(("    part_info->identifier        :-"));
    }

    ETG_TRACE_USR4(("    part_info->mounted_writable :0x%d",part_info->mounted_writable));
    ETG_TRACE_USR4(("    part_info->partition_no      :%d ",part_info->partition_no));

    if(part_info->mount_src)
    {
        ETG_TRACE_USR4(("    part_info->mount_src         :%s",part_info->mount_src));
    }
    else
    {
        ETG_TRACE_USR4(("    part_info->mount_src         :-"));
    }

    if(part_info->mount_fs)
    {
        ETG_TRACE_USR4(("    part_info->mount_fs          :%s",part_info->mount_fs));
    }
    else
    {
        ETG_TRACE_USR4(("    part_info->mount_fs          :-"));
    }

    if(part_info->mount_point)
    {
        ETG_TRACE_USR4(("    part_info->mount_point        :%s",part_info->mount_point));
    }
    else
    {
        ETG_TRACE_USR4(("    part_info->mount_point        :-"));
    }

    if (part_info->state==PARTITION_UNSUPPORTED)
    {
        const char* strUnsupportedReason = automounter_api_get_partition_unsupported_reason_string(part_info->unsupported_reason);
        if(strUnsupportedReason)
        {
            ETG_TRACE_USR4(("    Unsupported reason:       %s",strUnsupportedReason));
        }

    }
    if (part_info->error != RESULT_OK)
    {
         ETG_TRACE_USR4(("    NOK-RESULT:   Error Code   %d",(int)part_info->error));
    }

    ETG_TRACE_USR4 (("-------------------------------"));

}


/*-----------------------------------------------------------------------------*
 * establish_connection_success(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::establish_connection_success()
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("establish_connection_success"));
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * establish_connection_failure(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::establish_connection_failure()
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("establish_connection_failure"));
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}


/*-----------------------------------------------------------------------------*
 * connection_lost(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::connection_lost()
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("connection_lost"));
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * device_detected(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::device_detected(const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("device_detected"));
    print_device_info(device_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}


/*-----------------------------------------------------------------------------*
 * device_nomedia(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::device_nomedia(const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("device_nomedia"));
    print_device_info(device_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * device_automounting(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::device_automounting(const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("device_automounting"));
    print_device_info(device_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * device_automounted(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::device_automounted(const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("device_automounted"));
    print_device_info(device_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * device_unmounting(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::device_unmounting(const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("device_unmounting"));
    print_device_info(device_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}


/*-----------------------------------------------------------------------------*
 * device_unmounted(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::device_unmounted(const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("device_unmounted"));
    print_device_info(device_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * device_invalid(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::device_invalid(const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("device_invalid"));
    print_device_info(device_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}


/*-----------------------------------------------------------------------------*
 * partition_detected(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::partition_detected(const partition_info_t *partition_info, const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("partition_detected"));
    print_device_info(device_info);
    print_part_info(partition_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * partition_unsupported(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::partition_unsupported(const partition_info_t *partition_info, const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("partition_unsupported"));
    print_device_info(device_info);
    print_part_info(partition_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * partition_mounting(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::partition_mounting(const partition_info_t *partition_info, const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("partition_mounting"));
    print_device_info(device_info);
    print_part_info(partition_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * partition_mounted(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::partition_mounted(const partition_info_t *partition_info, const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("partition_mounted"));
    print_device_info(device_info);
    print_part_info(partition_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * partition_mount_err(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::partition_mount_err(const partition_info_t *partition_info, const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("partition_mount_err"));
    print_device_info(device_info);
    print_part_info(partition_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * partition_remounting(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::partition_remounting(const partition_info_t *partition_info, const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("partition_remounting"));
    print_device_info(device_info);
    print_part_info(partition_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * partition_unmounting(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::partition_unmounting(const partition_info_t *partition_info, const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("partition_unmounting"));
    print_device_info(device_info);
    print_part_info(partition_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * partition_unmounted(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::partition_unmounted(const partition_info_t *partition_info, const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("partition_unmounted"));
    print_device_info(device_info);
    print_part_info(partition_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * partition_invalid(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::partition_invalid(const partition_info_t *partition_info, const device_info_t *device_info)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("partition_invalid"));
    print_device_info(device_info);
    print_part_info(partition_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}


/*-----------------------------------------------------------------------------*
 * update_device_info(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::update_device_info(const device_info_t *device_info, int request_id)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("update_device_info: %d",request_id));
    print_device_info(device_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * update_partition_info(...)                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::update_partition_info(const partition_info_t *partition_info,const device_info_t *device_info, int request_id)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("update_partition_info: request_id: %d",request_id));
    print_device_info(device_info);
    print_part_info(partition_info);
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}

/*-----------------------------------------------------------------------------*
 * snapshot_complete                                                                 *
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::snapshot_complete(int request_id)
{
    ETG_TRACE_COMP((">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
    ETG_TRACE_COMP(("snapshot_complete: request_id: %d",request_id));
    ETG_TRACE_COMP(("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<"));
}



/*-----------------------------------------------------------------------------*
FillDevice(...)
 *-----------------------------------------------------------------------------*/
//tbd.: still have to learn  which of these functions are necessary to be used later if automounterShared lib will be coupled to UdevManager
IAutomounterAdapter::tenResIAutomounterAdapt_Type AutomounterAdaptSharedLibIF::FillDevice(OUT CDevice& f_device)
{
    ETG_TRACE_USR4(("Begin: FillDevice()"));
    
    IAutomounterAdapter::tenResIAutomounterAdapt_Type enResult =  IAutomounterAdapter::enRES_ERROR_GENERAL;

    f_device.m_cMountPoint = "tbd not yet implmented";
    ETG_TRACE_USR4(("End  : FillDevice()"));

    return enResult; //since it is not yet fully implemented
}

/*-----------------------------------------------------------------------------*
FillDevice(...)
 *-----------------------------------------------------------------------------*/
IAutomounterAdapter::tenResIAutomounterAdapt_Type AutomounterAdaptSharedLibIF::FillDevice(OUT CDevice& device, IN  std::string strPathfilename)
{
    IAutomounterAdapter::tenResIAutomounterAdapt_Type enResult =  IAutomounterAdapter::enRES_ERROR_GENERAL;

    ETG_TRACE_COMP(("FillDevice: enResult:%d", ETG_CENUM(IAutomounterAdapter::tenResIAutomounterAdapt_Type, enResult)))
    return enResult;
}

/*-----------------------------------------------------------------------------*
GetExtractBlockDevName(...)
 *-----------------------------------------------------------------------------*/
IAutomounterAdapter::tenResIAutomounterAdapt_Type AutomounterAdaptSharedLibIF::GetExtractBlockDevName(OUT std::string &strDevName, IN std::string strAutomounterPartitionFileName) const
{
    IAutomounterAdapter::tenResIAutomounterAdapt_Type enResult =  IAutomounterAdapter::enRES_ERROR_GENERAL;
   
    
   return enResult;
}




/*-----------------------------------------------------------------------------*
Initialize(...) 
 *-----------------------------------------------------------------------------*/
int AutomounterAdaptSharedLibIF::Initialize(tVoid) 
{
    ETG_TRACE_COMP(("Begin: AutomounterAdaptSharedLibIF: Initialize"));

    tS32 s32Result = OSAL_OK;

#ifdef CPP_COMPATIBLE_VERSION //see automounter_types.h if 
    ETG_TRACE_USR4(("[OK]: AutomounterAdaptSharedLibIF: Automounter shared lib is allowed to be used: CPP_COMPATIBLE_VERSION ")); //previous versions  cause a reset do to different memory alignement of parameter structure for c and cpp compiled code

    //@todo should be called once only add error mechanism here
#ifdef USE_AUTOMOUNTER_SHAREDLIB_TFTTHREAD_TEST
    ETG_TRACE_USR4(("{TEST TEST] AutomounterAdaptSharedLibIF: start thread AutomounterAdaptSharedLibIF"));
#else //USE_AUTOMOUNTER_SHAREDLIB_TFTTHREAD_TEST
    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_AUTOMOUNTER_SHAREDLIBIF)) 
#endif //USE_AUTOMOUNTER_SHAREDLIB_TFTTHREAD_TEST
    {
        tInt iThreadIndex = ThreadFactoryDVM::GetThreadFactory()->Do(IN this, (int)AutomounterAdaptSharedLibIF::eThread_AutomounterAdapter, NULL); 
        ETG_TRACE_COMP(("AutomounterAdaptSharedLibIF: iThreadIndex:%d",iThreadIndex));
    }

#else //CPP_COMPATIBLE_VERSION
    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_AUTOMOUNTER_SHAREDLIBIF)) 
    {
        //
        ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
        ETG_TRACE_FATAL(("[ERROR] AutomounterAdaptSharedLibIF: Can't use automounter shared lib wrong version: CPP_COMPATIBLE_VERSION not defined in automounter_types.h"));
        ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
    }

#endif //CPP_COMPATIBLE_VERSION
    ETG_TRACE_COMP(("End : AutomounterAdaptSharedLibIF: Initialize"));
    return s32Result;
}


/*-----------------------------------------------------------------------------*
void AutomounterAdaptSharedLibIF::Do(...)
 *-----------------------------------------------------------------------------*/
void AutomounterAdaptSharedLibIF::Do(int functionID, void *ptr)
{
    ETG_TRACE_USR4(("Begin:AutomounterAdaptSharedLibIF::Do"));
    (void) ptr;
    tenThreadFunction eFunctionID = (tenThreadFunction)functionID;
    switch(eFunctionID)
    {
        case AutomounterAdaptSharedLibIF::eThread_AutomounterAdapter: //execution tree for thread eThread_SystemVolt 
            ThreadFactoryDVM::GetThreadFactory()->SetName("VD_DVM:eThread_AutomounterAdapter");
            vThreadFunction();
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("End  :AutomounterAdaptSharedLibIF::Do"));
}


//thread is started if Initialized is called correctly
void AutomounterAdaptSharedLibIF::vThreadFunction()
{
    ETG_TRACE_USR4(("Begin: AutomounterAdaptSharedLibIF::vThreadFunction "));

    int ReqId = -1;
    tBool bTerminate = FALSE;
    
    const char* pVD_DVM = "vd_devicemanager";
    error_code_t error;
    tS32 fd;
    struct pollfd am_pollfd_struc;
    const char* pcVersion;
    
    //---------------------------------------------------------------------------
    //connect to automounter
    //---------------------------------------------------------------------------
    error = automounter_api_init(pVD_DVM, LOGGER_LEVEL_DEBUG,(bool)true);
    
    pcVersion = automounter_api_get_version_string();
    if(pcVersion)
    {
        ETG_TRACE_COMP(("============================================================================================"));
        ETG_TRACE_COMP(("AutomounterAdaptSharedLibIF::vThreadFunction: automounter_api_get_version_string():%s",pcVersion));
        ETG_TRACE_COMP(("============================================================================================"));
    }
    
    
    if(RESULT_OK != error)
    {
        bTerminate = TRUE;
        ETG_TRACE_FATAL(("[ERROR] AutomounterAdaptSharedLibIF::vThreadFunction:automounter_api_init: %d",error));
    }
    else
    {
      ETG_TRACE_COMP(("[OK] AutomounterAdaptSharedLibIF::vThreadFunction:automounter_api_init"));
      automounter_api_register_callbacks(&rFunction);
      ETG_TRACE_COMP(("AutomounterAdaptSharedLibIF::vThreadFunction:automounter_api_register_callbacks()"));
    
      error = automounter_api_try_connect();
    
      ETG_TRACE_COMP(("vThreadFunction: automounter_api_try_connect() returns %d",ETG_CENUM(error_code_t,error) ));
      //vTraceError((int)error);
    
      if ((error == RESULT_OK)|| (error == RESULT_DAEMON_NOT_RUNNING))
      {
          ETG_TRACE_COMP(("[OK ]AutomounterAdaptSharedLibIF::vThreadFunction:automounter_api_try_connect()"));
          if((fd = automounter_api_get_pollfd()) !=-1)
          {
              ETG_TRACE_USR4(("[OK ]AutomounterAdaptSharedLibIF::vThreadFunction:automounter_api_get_pollfd()"));
              am_pollfd_struc.fd=fd;
              am_pollfd_struc.events=POLLIN;
              bTerminate = FALSE;
          }
          else
          {
              ETG_TRACE_FATAL(("[ERROR]AutomounterAdaptSharedLibIF::vThreadFunction:automounter_api_get_pollfd()"));
              bTerminate = TRUE;
              NORMAL_M_ASSERT_ALWAYS();
          }
      }
      else
      {
         ETG_TRACE_FATAL(("[ERROR ]AutomounterAdaptSharedLibIF::vThreadFunction:automounter_api_try_connect()"));
      }
    
    }
    
    //---------------------------------------------------------------------------------
    // wait for events of automounter
    //---------------------------------------------------------------------------------
    while(bTerminate == FALSE)
    {
      ETG_TRACE_FATAL(("AutomounterAdaptSharedLibIF::vThreadFunction: Wait for autmounter event "));
      if (poll(&am_pollfd_struc,1,-1) > 0) //blocking here
      {
          ETG_TRACE_COMP((">========================================================================"));
          ETG_TRACE_COMP(("AutomounterAdaptSharedLibIF::vThreadFunction: Received Autmounter event "));
          ETG_TRACE_COMP(("------------------------------------------------------------------------"));
          if(am_pollfd_struc.revents == EPOLLIN)
          {
             //---------------------------------------------------------------------------------
              //this triggers registered callbacks to be called
             //---------------------------------------------------------------------------------
             automounter_api_dispatch_event();
          }
          else
          {
              ETG_TRACE_FATAL(("AutomounterAdaptSharedLibIF::vThreadFunction:automounter strange epoll_wait return"));
          }
          ETG_TRACE_COMP(("========================================================================<"));
       }
       else
       {
          if(errno != EINTR)
          {
              ETG_TRACE_FATAL(("AutomounterAdaptSharedLibIF::vThreadFunction:automounter epoll_wait failed"));
          }
       }
    }
    
    //---------------------------------------------------------------------------------
    // autmounter disconnect
    //---------------------------------------------------------------------------------
    automounter_api_disconnect();
    automounter_api_deinit();

    ETG_TRACE_USR4(("End  : AutomounterAdaptSharedLibIF::vThreadFunction"));
}



////////////////////////////////////////////////////////////////////////////////
// <EOF>



