/*-----------------------------------------------------------------------------*
 * DiaglibHandlerGen2.cpp                                   *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file DiaglibHandlerGen2.cpp
 *
 * \brief Implementation of a Dignosis Diaglib Data ID Listener
 *
 * \version 12.11.2010, Ramkumar Muniraj (Bosch), version 1.0
 * \version 12.05.2011, Ramkumar Muniraj (Bosch), version 1.1
 * \version 05.08.2011, Koechling, Christian (Bosch), Interface VD_DeviceManager
 *          changed, ETG-traces improved, cleanup diagnosis
 * \version 19.08.2011, Ramkumar Muniraj (Bosch), bugfixing
 * \version 17.11.2011, Koechling, Christian (Bosch), cleanup: 1st step to shift
 *          includes to Common.h
 * \version 21.11.2011, Koechling, Christian (Bosch), cleanup includes and
 *          intermediate defines which marked bugfixes
 * \version 28.11.2011, Koechling, Christian (Bosch),
 *          cleanup - add headers
 * \version 06.01.2012, Koechling, Christian (Bosch), cleanup FATAL traces,
 *          cleanup workaround MISSING_IN_FILE_dia_ldSpec
 * \version 19.01.2012, Koechling, Christian (Bosch), diag over current is
 *          deduced from OC-signal
 * \version 05.03.2012, Koechling, Christian (Bosch)| cleanup: function call
 *          adjusted - see bGetDiagnosis
 * \version 27.07.2012, Negi, Sunder (MontaVista), Cleaned up file hierarchy
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 27.08.2012, Koechling, Christian (Bosch), trigger FillStorage fct. by PDT-diag
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

#include "Config.h"

#define INCLUDE_VD_DVM_AILAHL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#define DIAGLIB_INCLUDE_GENERIC
#include <diaglib_if.h>
using namespace diaglib;

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "Diagnosis.h"
#include "StateTable.h"
#include "WorkThread.h"
#include "interface/DeviceManagerInterface.h"
#include "diag/DiagLibHandlerGen2.h"
#include "diag/IDiagLibHandler.h"
#include "diag/DiagLibHandlerGen3.h"
#include "client/VD_DeviceManager_clienthandler_Diaglog.h"
#include "service/VD_DeviceManager_service_DeviceManager.h"
#include "VD_DeviceManager_main.h"

#include "config/ConfigurationManager.h"

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_CLIENTHANDLER_DIAGLOG
#include "trcGenProj/Header/DiagLibHandlerGen2.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * Defines                                                                     *
 *-----------------------------------------------------------------------------*/
#define EMPTYUSB 0

/*!
 * see PID $C747 HMI Module Input / Output
 */
enum
{
    eMaskBit2_SD   = 0,
    eMaskBit0_USB1 = (tInt)eUSB1,
    eMaskBit1_USB2 = (tInt)eUSB2,
    eMaskBit2_USB3 = (tInt)eUSB3
#ifdef USE_FOURTH_USB_CONNECTOR
    ,eMaskBit3_USB4 = (tInt)eUSB4
#endif //USE_FOURTH_USB_CONNECTOR
};

#define MASK_BIT0_USB1 0x01;   ///< USB1
#define MASK_BIT1_USB2 0x02;   ///< USB2
#define MASK_BIT2_USB3 0x05;   ///< USB3
#define MASK_BIT2_USB4 0x06;   ///< USB4
#define MASK_BIT2_SD   0x04;   ///< SD-card



//USE_THIRD_USB_CONNECTOR  (marker to find this comment see config.h)
//design: better let the code look generic and put special part here at the top
// PSA uses in addition to USB1 and USB2 also USB3 to test this and existing dependency and to ensure that functionality of GM continues to work
// GM has been used to test PSA feature.
// see config.h If USE_TEST_GMWITH3PORTSFORPSA is used then all devices connected to USB2 will be treated as devices connected to USB3
// since for below diaglib dependency even for PSA won't be give that's why for USB3 cases these will be mapped again back to USB2 to keep code compiling
//the corresponding implmentation further below in the code is treated
//discussed with Lars Plischek at on the phone: (20/04/2015) in doubt he recommended to contact andreas Evers if USB3 dependent ID's become necessary
#if !defined(DATAID_USB_VBUS_USB_3)
#define IS_UNDEFINED_DATAID_USB_VBUS_USB_3
#define DATAID_USB_VBUS_USB_3_READ_RESULT_PARAM1 DATAID_USB_VBUS_USB_2_READ_RESULT_PARAM1
#warning "define not found in FC_Diagnosis/Framework/systemAdapter/dia_IdSpec.h "
#warning "DATAID_USB_VBUS_USB_3_READ_RESULT_PARAM1 mapped to DATAID_USB_VBUS_USB_2_READ_RESULT_PARAM1"
#endif

#if !defined(DATAID_USB_VERSION_USB_3)
#define IS_UNDEFINED_DATAID_USB_VERSION_USB_3
#define DATAID_USB_VERSION_USB_3_READ_RESULT_PARAM1 DATAID_USB_VERSION_USB_2_READ_RESULT_PARAM1
#define DATAID_USB_VERSION_USB_3_READ_RESULT_PARAM2 DATAID_USB_VERSION_USB_2_READ_RESULT_PARAM2
#define DATAID_USB_VERSION_USB_3_READ_RESULT_PARAM3 DATAID_USB_VERSION_USB_2_READ_RESULT_PARAM3
#warning "define not found in FC_Diagnosis/Framework/systemAdapter/dia_IdSpec.h "
#warning "DATAID_USB_VERSION_USB_3_READ_RESULT_PARAM1 mapped to DATAID_USB_VERSION_USB_2_READ_RESULT_PARAM1"
#warning "DATAID_USB_VERSION_USB_3_READ_RESULT_PARAM2 mapped to DATAID_USB_VERSION_USB_2_READ_RESULT_PARAM2"
#warning "DATAID_USB_VERSION_USB_3_READ_RESULT_PARAM3 mapped to DATAID_USB_VERSION_USB_2_READ_RESULT_PARAM3"
#endif

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace statetbl;

/*-----------------------------------------------------------------------------*
 * tU32 u32ProcessUSBVersionRequest(tenDevManagerUSBHost usb,                  *
 *                                  tContext MsgContext, tBool bTestMode)      *                                            *
 *-----------------------------------------------------------------------------*/
#ifdef TTFISCMD_PRODUCTIONLINETEST
tU32 DiaglibHandlerGen2::u32ProcessUSBVersionRequest(
        tenDevManagerUSBHost usb, tContext MsgContext, tBool bTestMode)
#else
tU32 DiaglibHandlerGen2::u32ProcessUSBVersionRequest(
        tenDevManagerUSBHost usb, tContext MsgContext)
#endif
{
    ETG_TRACE_USR4(("Begin: DiaglibHandlerGen2::u32ProcessUSBVersionRequest"));

#ifdef TTFISCMD_PRODUCTIONLINETEST
    if(TRUE==bTestMode)
    {
        ETG_TRACE_FATAL(("++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"));
        ETG_TRACE_FATAL(("              USB%d DIAGNOSIS REQUEST  (u32ProcessUSBVersionRequest)    ",usb));
        ETG_TRACE_FATAL(("------------------------------------------------------------------------"));
    }
    else
#endif
    {
        ETG_TRACE_USR1(("++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"));
        ETG_TRACE_USR1(("              USB%d DIAGNOSIS REQUEST  (u32ProcessUSBVersionRequest)    ",usb));
        ETG_TRACE_USR1(("------------------------------------------------------------------------"));
    }

    tU32 u32Return                =  U32_DIAGLIB_RETURN_NOT_OK;
    tenDiagDataResult oDataResult =  EN_DIAGDATA_NOT_OK;

    tclParameterVector oTmpVec;  // create object for response
    tBool bResult = FALSE;

    //Class variable to hold HUB1,HUB2,HUB3,HUB4 Diagnosis data
#ifdef USE_FOURTH_USB_CONNECTOR
    CUSBDiagnosis  oUSBDiagnosisHUB1,oUSBDiagnosisHUB2,oUSBDiagnosisHUB3,oUSBDiagnosisHUB4;
#else
    CUSBDiagnosis  oUSBDiagnosisHUB1,oUSBDiagnosisHUB2,oUSBDiagnosisHUB3;
#endif//USE_FOURTH_USB_CONNECTOR
    CUSBDiagnosis* poUSBDiagnosisHUB[ARRAYSIZEFORUSB];

    poUSBDiagnosisHUB[0] = NULL;
    poUSBDiagnosisHUB[eUSB1] = &oUSBDiagnosisHUB1;
    poUSBDiagnosisHUB[eUSB2] = &oUSBDiagnosisHUB2;
    poUSBDiagnosisHUB[eUSB3] = &oUSBDiagnosisHUB3;
#ifdef USE_FOURTH_USB_CONNECTOR
    poUSBDiagnosisHUB[eUSB4] = &oUSBDiagnosisHUB4;

    DeviceManagerInterface::GetInstance()->bGetDiagnosis(
                                             CGlobalEnumerations::DIAG_PDT_ONDEMAND,
                                             oUSBDiagnosisHUB1, oUSBDiagnosisHUB2,
                                             oUSBDiagnosisHUB3, oUSBDiagnosisHUB4);
#else
    DeviceManagerInterface::GetInstance()->bGetDiagnosis(
                                            CGlobalEnumerations::DIAG_PDT_ONDEMAND,
                                            oUSBDiagnosisHUB1, oUSBDiagnosisHUB2,
                                            oUSBDiagnosisHUB3);
#endif
    ETG_TRACE_USR2(("USB%d:u32ProcessUSBVersionRequest:oUSBDiagnosisHUB.m_ulSize=%ul",
            usb, poUSBDiagnosisHUB[usb]->m_ulSize));

    if( (EMPTYUSB!=poUSBDiagnosisHUB[usb]->m_ulSize))
    {
       bResult=TRUE;
    }

    if (TRUE == bResult)
    {
        oDataResult = EN_DIAGDATA_OK; // data valid
        trParameter paramaUsbVendor, paramaUsbVersion, paramaUsbSize;

        switch(usb)
        {
            case eUSB1:
                paramaUsbVendor.enType       = static_cast<tenParameterType>(DATAID_USB_VERSION_USB_1_READ_RESULT_PARAM1);

                paramaUsbVendor.u16Value     = poUSBDiagnosisHUB[usb]->m_cVendor.Hex2UShort();
                paramaUsbVersion.enType      = static_cast<tenParameterType>(DATAID_USB_VERSION_USB_1_READ_RESULT_PARAM2);
                paramaUsbVersion.u16Value    = (tU16)poUSBDiagnosisHUB[usb]->m_cDeviceVersion.toInt();
                paramaUsbSize.enType         = static_cast<tenParameterType>(DATAID_USB_VERSION_USB_1_READ_RESULT_PARAM3);
                paramaUsbSize.u32Value       = (tU32)poUSBDiagnosisHUB[usb]->m_ulSize;
                break;

            case eUSB2:
                paramaUsbVendor.enType    = static_cast<tenParameterType>(DATAID_USB_VERSION_USB_2_READ_RESULT_PARAM1);
                paramaUsbVendor.u16Value     = poUSBDiagnosisHUB[usb]->m_cVendor.Hex2UShort();
                paramaUsbVersion.enType   = static_cast<tenParameterType>(DATAID_USB_VERSION_USB_2_READ_RESULT_PARAM2);

                paramaUsbVersion.u16Value    = (tU16)poUSBDiagnosisHUB[usb]->m_cDeviceVersion.toInt();
                paramaUsbSize.enType      = static_cast<tenParameterType>(DATAID_USB_VERSION_USB_2_READ_RESULT_PARAM3);
                paramaUsbSize.u32Value       = (tU32)poUSBDiagnosisHUB[usb]->m_ulSize;
                break;
            case eUSB3:
                paramaUsbVendor.enType    = static_cast<tenParameterType>(DATAID_USB_VERSION_USB_3_READ_RESULT_PARAM1);
                paramaUsbVendor.u16Value     = poUSBDiagnosisHUB[usb]->m_cVendor.Hex2UShort();
                paramaUsbVersion.enType   = static_cast<tenParameterType>(DATAID_USB_VERSION_USB_3_READ_RESULT_PARAM2);

                paramaUsbVersion.u16Value    = (tU16)poUSBDiagnosisHUB[usb]->m_cDeviceVersion.toInt();
                paramaUsbSize.enType      = static_cast<tenParameterType>(DATAID_USB_VERSION_USB_3_READ_RESULT_PARAM3);
                paramaUsbSize.u32Value       = (tU32)poUSBDiagnosisHUB[usb]->m_ulSize;
                break;

            default:
                ETG_TRACE_FATAL(("[ERROR]:  u32ProcessUSBVersionRequest usb:%d", (tInt)usb));
                ETG_TRACE_ERRMEM(("[ERROR]: u32ProcessUSBVersionRequest usb:%d", (tInt)usb));
                DVM_NORMAL_M_ASSERT_ALWAYS();
        }

#ifdef USE_DIAGSOLUTION_SIZEONLY
        paramaUsbVendor.u16Value =0;
        paramaUsbVersion.u16Value=0;
        ETG_TRACE_USR1(("[ok] DIAGNOSIS:06/01/2012 - Vendor & Version: result of discussion: poduction line diagnosis doe not use it (GM-project)"));
#endif

#ifdef TTFISCMD_PRODUCTIONLINETEST
        if(TRUE == bTestMode)
        {
            ETG_TRACE_FATAL(("                                                          "));
            ETG_TRACE_FATAL(("~~~~~~~~~~~~~~~~~~~DIAG-RESULT: OK~~~~~~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("[USB%1d]: paramaUsbVendor.enType            %d                     ",
                               usb,ETG_ENUM(TENPARAMETERTYPE,paramaUsbVendor.enType)));
            ETG_TRACE_FATAL(("[USB%1d]: paramaUsbVendor.u16Value          %d                     ",
                               usb,paramaUsbVendor.u16Value));
            ETG_TRACE_FATAL(("[USB%1d]: paramaUsbVersion.enType           %d                     ",
                               usb,ETG_ENUM(TENPARAMETERTYPE,paramaUsbVersion.enType)));
            ETG_TRACE_FATAL(("[USB%1d]: paramaUsbVersion.u16Value         %d                     ",
                               usb,paramaUsbVersion.u16Value));
            ETG_TRACE_FATAL(("[USB%1d]: paramaUsbSize.enType              %d                     ",
                               usb,ETG_ENUM(TENPARAMETERTYPE,paramaUsbSize.enType)));
            ETG_TRACE_FATAL(("[USB%1d]: paramaUsbSize.u32Value            %u                     ",
                               usb,paramaUsbSize.u32Value));
            ETG_TRACE_FATAL(("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("                                                         "));

        }
        else
#endif
        {
            ETG_TRACE_FATAL(("                                                          "));
            ETG_TRACE_FATAL(("~~~~~~~~~~~~~~~~~~~DIAG-RESULT: OK~~~~~~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("[USB%1d]: paramaUsbVendor.enType            %d                     ",
                               usb,ETG_ENUM(TENPARAMETERTYPE,paramaUsbVendor.enType)));
            ETG_TRACE_FATAL(("[USB%1d]: paramaUsbVendor.u16Value          %d                     ",
                               usb,paramaUsbVendor.u16Value));
            ETG_TRACE_FATAL(("[USB%1d]: paramaUsbVersion.enType           %d                     ",
                               usb,ETG_ENUM(TENPARAMETERTYPE,paramaUsbVersion.enType)));
            ETG_TRACE_FATAL(("[USB%1d]: paramaUsbVersion.u16Value         %d                     ",
                               usb,paramaUsbVersion.u16Value));
            ETG_TRACE_FATAL(("[USB%1d]: paramaUsbSize.enType              %d                     ",
                               usb,ETG_ENUM(TENPARAMETERTYPE,paramaUsbSize.enType)));
            ETG_TRACE_FATAL(("[USB%1d]: paramaUsbSize.u32Value            %d                     ",
                               usb,paramaUsbSize.u32Value));
            ETG_TRACE_FATAL(("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"));
            ETG_TRACE_FATAL(("                                                         "));
        }

        oTmpVec.push_back(paramaUsbVendor);
        oTmpVec.push_back(paramaUsbVersion);
        oTmpVec.push_back(paramaUsbSize);
    }  // if (TRUE == bResult)
    else
    {
        ETG_TRACE_ERR(("+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"));
        ETG_TRACE_ERR(("!!! Failed to retrieve USB%d information version with error: %d",usb, bResult));
        ETG_TRACE_ERR(("+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"));

        oDataResult = EN_DIAGDATA_NOT_OK;
        trParameter paramaError;
        paramaError.enType         = static_cast<tenParameterType>(EN_PARAMETER_TYPE_U8ERROR);
        paramaError.u8ErrorValue   = (tenInternalError)EN_ERROR_FATAL_INTERNAL;
        oTmpVec.push_back(paramaError);
    }  // End of if-else; if (TRUE == bResult)

    if(TRUE == bTestMode)
    {
        ETG_TRACE_FATAL(("[ok] u32ProcessUSBVersionRequest: TEST MODE - don't send anything to DIAG"));
    }
    else
   // Send message back to diagnosis
    {
        if ((OSAL_NULL != m_poDiaglibService) && (OSAL_NULL != m_poDataIDIf))
        {
           if (TRUE == m_poDataIDIf->bSendDiagnosisDataResult(oDataResult,
                                                              oTmpVec, MsgContext))
           {
               u32Return = U32_DIAGLIB_RETURN_OK;
               ETG_TRACE_USR4(("USB%d:bSendDiagnosisDataResult successfull",usb));
           }  // if (TRUE == m_poDataIDIf->bSendDiagnosisDataResult(oDataResult
           else
           {
              u32Return = U32_DIAGLIB_RETURN_NOT_OK;

              ETG_TRACE_ERR(("USB%d:!!! bSendDiagnosisDataResult failed",usb));
           }  // End of if-else; if (TRUE == m_poDataIDIf->bSendDiagnosisDataResul..
        }  // if ((OSAL_NULL != m_poDiaglibService) && (OSAL_NULL != m_poDataIDIf))
        else
        {
           u32Return = U32_DIAGLIB_RETURN_NOT_OK;

           ETG_TRACE_USR4(("USB%d:!!! m_poDiaglibService or m_poDataIDIf  == OSAL_NULL",usb));
        }  // End of if-else; if ((OSAL_NULL != m_poDiaglibService) && (OSAL_NULL !..
    }
    ETG_TRACE_USR4(("End  : USB%d:u32ProcessUSBVersionRequest",usb));
    return u32Return;
}  // tU32 DiaglibHandlerGen2::u32ProcessUSBVersionRequest(tConte..

/*-----------------------------------------------------------------------------*
 * tU32 u32ProcessUSBVbusRequest(tenDevManagerUSBHost usb, tContext MsgContext,*
 *                               tBool bTestMode)                              *
 *-----------------------------------------------------------------------------*/
tU32 DiaglibHandlerGen2::u32ProcessUSBVbusRequest(
        tenDevManagerUSBHost usb, tContext MsgContext,tBool bTestMode)
{
       ETG_TRACE_USR4(("Begin: DiaglibHandlerGen2::u32ProcessUSBVbusRequest"));
    if(TRUE == bTestMode)
    {

        ETG_TRACE_FATAL(("++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"));
        ETG_TRACE_FATAL(("                  USB%d DIAGNOSIS REQUEST (u32ProcessUSBVbusRequest)    ",usb));
        ETG_TRACE_FATAL(("------------------------------------------------------------------------"));
    }
    else
    {
        ETG_TRACE_USR4(("Begin: DiaglibHandlerGen2::u32ProcessUSBVbusRequest"));
        ETG_TRACE_USR1(("++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"));
        ETG_TRACE_USR1(("                  USB%d DIAGNOSIS REQUEST (u32ProcessUSBVbusRequest)    ",usb));
        ETG_TRACE_USR1(("------------------------------------------------------------------------"));
    }

   tBool bStatus   =  0 ;
   tU32 u32Return                =  U32_DIAGLIB_RETURN_NOT_OK;
   tenDiagDataResult oDataResult =  EN_DIAGDATA_NOT_OK;

   tclParameterVector oTmpVec;  // create object for response
#ifdef USE_FOURTH_USB_CONNECTOR
   trParameter paramaUsb1OverCurrentStatusPin, paramaUsb2OverCurrentStatusPin, paramaUsb3OverCurrentStatusPin, paramaUsb4OverCurrentStatusPin;
#else
   trParameter paramaUsb1OverCurrentStatusPin, paramaUsb2OverCurrentStatusPin, paramaUsb3OverCurrentStatusPin;

#endif//USE_FOURTH_USB_CONNECTOR
   tBool bResult = FALSE;
   trParameter paramaError;

    //Class variable to hold HUB1,HUB2,HUB3,HUB4 Diagnosis data
#ifdef USE_FOURTH_USB_CONNECTOR
   CUSBDiagnosis oUSBDiagnosisHUB1, oUSBDiagnosisHUB2, oUSBDiagnosisHUB3, oUSBDiagnosisHUB4;
#else
   CUSBDiagnosis oUSBDiagnosisHUB1, oUSBDiagnosisHUB2, oUSBDiagnosisHUB3;
#endif//USE_FOURTH_USB_CONNECTOR
   CUSBDiagnosis *poUSBDiagnosisHUB[ARRAYSIZEFORUSB];
   poUSBDiagnosisHUB[0] = NULL;
   poUSBDiagnosisHUB[eUSB1] = &oUSBDiagnosisHUB1;
   poUSBDiagnosisHUB[eUSB2] = &oUSBDiagnosisHUB2;
#ifdef USE_FOURTH_USB_CONNECTOR
   poUSBDiagnosisHUB[eUSB3] = &oUSBDiagnosisHUB4;
   DeviceManagerInterface::GetInstance()->bGetDiagnosis(CGlobalEnumerations::DIAG_PDT_ONDEMAND,
                                                            oUSBDiagnosisHUB1,
                                                            oUSBDiagnosisHUB2,
                                                            oUSBDiagnosisHUB3,
                                                            oUSBDiagnosisHUB4);
#else
   poUSBDiagnosisHUB[eUSB3] = &oUSBDiagnosisHUB3;
   DeviceManagerInterface::GetInstance()->bGetDiagnosis(CGlobalEnumerations::DIAG_PDT_ONDEMAND,
                                                            oUSBDiagnosisHUB1,
                                                            oUSBDiagnosisHUB2,
                                                            oUSBDiagnosisHUB3);
#endif//USE_FOURTH_USB_CONNECTOR




   if(TRUE == bTestMode)
   {
       ETG_TRACE_FATAL(("poUSBDiagnosisHUB[USB%d]->m_enOC = %d",usb,
                        ETG_ENUM(TENSIGNALVALUE,poUSBDiagnosisHUB[usb]->m_enOC)));
       ETG_TRACE_FATAL(("poUSBDiagnosisHUB[USB%d]->m_enPPON = %d",usb,
                        ETG_ENUM(TENSIGNALVALUE,poUSBDiagnosisHUB[usb]->m_enPPON)));
       //Note: if OC1 (Over current signal) happens then PPON1 is used to switch on VBUS2
       //      i.e. PPON lasts longer due to timout in prm hnce we take that signal
   }
   else
   {
       ETG_TRACE_USR2(("poUSBDiagnosisHUB[USB%d]->m_enOC = %d",usb,
                        ETG_ENUM(TENSIGNALVALUE,poUSBDiagnosisHUB[usb]->m_enOC)));
       ETG_TRACE_USR2(("poUSBDiagnosisHUB[USB%d]->m_enPPON = %d",usb,
                        ETG_ENUM(TENSIGNALVALUE,poUSBDiagnosisHUB[usb]->m_enPPON)));
       //Note: if OC1 (Over current signal) happens then PPON1 is used to switch on VBUS2
       //      i.e. PPON lasts longer due to timout in prm hnce we take that signal
   }




    switch(poUSBDiagnosisHUB[usb]->m_enOC)
    {
        //------------------------------------------------------------
        case SIG_FALSE:
        //------------------------------------------------------------
             bStatus = TRUE;
             bResult = TRUE;
             break;
        //------------------------------------------------------------
        case SIG_TRUE:
        //------------------------------------------------------------
            bStatus = FALSE;
            bResult = TRUE;
            break;
        //------------------------------------------------------------
        case SIG_UNDEF:
        //------------------------------------------------------------
            bResult = FALSE;
            break;
        case SIG_UNUSED:
        case SIG_ANY:
             break;
        default:
            ETG_TRACE_FATAL((" [ERROR]: u32ProcessUSBVbusRequest usb:%d", (tInt)usb));
            ETG_TRACE_ERRMEM(("[ERROR]: u32ProcessUSBVbusRequest usb:%d", (tInt)usb));
            DVM_NORMAL_M_ASSERT_ALWAYS();
    }

    if (TRUE == bResult)
    {
         switch(usb)
         {
             case eUSB1:
                 oDataResult = EN_DIAGDATA_OK; // data valid
                 paramaUsb1OverCurrentStatusPin.enType    = static_cast<tenParameterType>(DATAID_USB_VBUS_USB_1_READ_RESULT_PARAM1);
                 paramaUsb1OverCurrentStatusPin.bValue    = bStatus ;
                 oTmpVec.push_back(paramaUsb1OverCurrentStatusPin);
                 break;

             case eUSB2:
                 oDataResult = EN_DIAGDATA_OK; // data valid
                 paramaUsb2OverCurrentStatusPin.enType    = static_cast<tenParameterType>(DATAID_USB_VBUS_USB_2_READ_RESULT_PARAM1);
                 paramaUsb2OverCurrentStatusPin.bValue   = bStatus ;
                 oTmpVec.push_back(paramaUsb2OverCurrentStatusPin);
                 break;
             case eUSB3:
                 oDataResult = EN_DIAGDATA_OK; // data valid
                 paramaUsb3OverCurrentStatusPin.enType    = static_cast<tenParameterType>(DATAID_USB_VBUS_USB_3_READ_RESULT_PARAM1);
                 paramaUsb3OverCurrentStatusPin.bValue   = bStatus ;
                 oTmpVec.push_back(paramaUsb3OverCurrentStatusPin);
                 break;

             default:
                ETG_TRACE_FATAL((" [ERROR]: u32ProcessUSBVbusRequest usb:%d", (tInt)usb));
                ETG_TRACE_ERRMEM(("[ERROR]: u32ProcessUSBVbusRequest usb:%d", (tInt)usb));
                DVM_NORMAL_M_ASSERT_ALWAYS();
                 break;
         }

         ETG_TRACE_USR4(("USB%d:oDataResult = EN_DIAGDATA_OK      ",usb));
         ETG_TRACE_USR4(("USB%d:UsbOverCurrentStatus = %d",usb, bStatus));
     }
     else
     {
         oDataResult = EN_DIAGDATA_NOT_OK;
         paramaError.enType         = static_cast<tenParameterType>(EN_PARAMETER_TYPE_U8ERROR);
         paramaError.u8ErrorValue   = (tenInternalError)EN_ERROR_FATAL_INTERNAL;
         oTmpVec.push_back(paramaError);

         ETG_TRACE_USR4(("USB%d:oDataResult = EN_DIAGDATA_NOT_OK    ",usb));
      }

    if(TRUE == bTestMode)
    {
        ETG_TRACE_FATAL(("[ok] u32ProcessUSBVbusRequest: TEST MODE - don't send anything to DIAG"));
    }
    else
    {
        // Send message back to diagnosis
        if ((OSAL_NULL != m_poDiaglibService) && (OSAL_NULL != m_poDataIDIf))
        {
            if (TRUE == m_poDataIDIf->bSendDiagnosisDataResult(oDataResult,
                            oTmpVec, MsgContext))
            {
               u32Return = U32_DIAGLIB_RETURN_OK;

               ETG_TRACE_USR4(("USB%d:bSendDiagnosisDataResult successfull",usb));
            }  // if (TRUE == m_poDataIDIf->bSendDiagnosisDataResult(oDataResult
            else
            {
               u32Return = U32_DIAGLIB_RETURN_NOT_OK;

               ETG_TRACE_ERR(("USB%d:!!! bSendDiagnosisDataResult failed",usb));
            }  // End of if-else; if (TRUE == m_poDataIDIf->bSendDiagnosisDataResul..
        }  // if ((OSAL_NULL != m_poDiaglibService) && (OSAL_NULL != m_poDataIDIf))
        else
        {
           u32Return = U32_DIAGLIB_RETURN_NOT_OK;


           ETG_TRACE_USR4(("USB%d:!!! m_poDiaglibService or m_poDataIDIf  == OSAL_NULL",usb));
        }  // End of if-else; if ((OSAL_NULL != m_poDiaglibService) && (OSAL_NULL !..
    }

    ETG_TRACE_USR4(("End  : DiaglibHandlerGen2::u32ProcessUSBVbusRequest"));

    return u32Return;
}  // tU32 DiaglibHandlerGen2::u32ProcessUSBVbusRequest(tC..

/*-----------------------------------------------------------------------------*
 * Parameterized constructor                                                   *
 *-----------------------------------------------------------------------------*/
DiaglibHandlerGen2::DiaglibHandlerGen2(
                    vd_devicemanager_tclApp* const cpoMainAppl,
                    tclServiceDiaglib* const cpoDiaglibService):
                    m_poDiaglibService(cpoDiaglibService), m_poDataIDIf(OSAL_NULL),
                    m_poVdDmApp(cpoMainAppl)
{
   ETG_TRACE_USR4(("---> DiaglibHandlerGen2 entered"));


   if(OSAL_NULL != m_poDiaglibService)
   {
      m_poDataIDIf   =  m_poDiaglibService->poGetDiagData();

      if (OSAL_NULL != m_poDataIDIf)
      {
         m_poDataIDIf->vRegisterListener(DATAID_USB_VERSION_USB_1, this);
         m_poDataIDIf->vRegisterListener(DATAID_USB_VERSION_USB_2, this);
         m_poDataIDIf->vRegisterListener(DATAID_USB_VBUS_USB_1   , this);
         m_poDataIDIf->vRegisterListener(DATAID_USB_VBUS_USB_2   , this);
         m_poDataIDIf->vRegisterListener(DATAID_USB_MEDIA_STATUS , this);

         ETG_TRACE_USR4(("--- Registration of  USB_VERSION_USB_1, \
         USB_VERSION_USB_2,USB_VBUS_USB_1 and USB_VBUS_USB_2,DATAID_USB_MEDIA_STATUS with diaglib is done"));
      }
      else
      {
         ETG_TRACE_ERR(("!!! Registration with Diaglibfailed for USB_VERSION_USB_1, \
          USB_VERSION_USB_2,USB_VBUS_USB_1 and USB_VBUS_USB_2,DATAID_USB_MEDIA_STATUS : \
                   m_poDataIDIf==OSAL_NULL"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("!!! Registration with Diaglibfailed for USB_VERSION_USB_1, \
          USB_VERSION_USB_2,USB_VBUS_USB_1 and USB_VBUS_USB_2,DATAID_USB_MEDIA_STATUS :  \
                       m_poDiaglibService==OSAL_NULL"));
   }

   ETG_TRACE_USR4(("<--- DiaglibHandlerGen2 exit"));
}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
DiaglibHandlerGen2::~DiaglibHandlerGen2()
{
   m_poDiaglibService   =  OSAL_NULL;
   m_poDataIDIf         =  OSAL_NULL;
   m_poVdDmApp          =  OSAL_NULL;
}

/*-----------------------------------------------------------------------------*
 * tU32 vOnDiagnosisDataRead(tU32 u32DiagDataId, tContext MsgContext)          *
 *-----------------------------------------------------------------------------*/
tU32 DiaglibHandlerGen2::vOnDiagnosisDataRead(
        tU32 u32DiagDataId, tContext MsgContext)
{
    ETG_TRACE_USR4(("--> DiaglibHandlerGen2::vOnDiagnosisDataRead"));

    tU32 u32Return    =  U32_DIAGLIB_RETURN_NOT_OK;

    // Check for expected DiagData ID
    switch(u32DiagDataId)
    {
       case DATAID_USB_VERSION_USB_1:
           u32Return = u32ProcessUSBVersionRequest(eUSB1,MsgContext);
           break;

       case DATAID_USB_VERSION_USB_2:
           u32Return = u32ProcessUSBVersionRequest(eUSB2,MsgContext);
           break;
#ifndef IS_UNDEFINED_DATAID_USB_VERSION_USB_3
       case DATAID_USB_VERSION_USB_3:
           u32Return = u32ProcessUSBVersionRequest(eUSB3,MsgContext);
           break;
#endif//IS_UNDEFINED_DATAID_USB_VERSION_USB_3

       case DATAID_USB_VBUS_USB_1:
           u32Return = u32ProcessUSBVbusRequest(eUSB1,MsgContext);
           break;

       case DATAID_USB_VBUS_USB_2:
           u32Return = u32ProcessUSBVbusRequest(eUSB2,MsgContext);
           break;
#ifndef IS_UNDEFINED_DATAID_USB_VBUS_USB_3
       case DATAID_USB_VBUS_USB_3:
            u32Return = u32ProcessUSBVbusRequest(eUSB3,MsgContext);
            break;
#endif //IS_UNDEFINED_DATAID_USB_VBUS_USB_3
       case
           DATAID_USB_MEDIA_STATUS:
           u32Return = u32ProcessHMIModuleInput(MsgContext); //1.    PID $C747 HMI Module Input / Output
           break;
       default:
           ETG_TRACE_ERR(("!!! DiaglibHandlerGen2:: \
              vOnDiagnosisDataRead : unknow DiagData ID = %d ", u32DiagDataId ));
    }

   ETG_TRACE_USR4(("<-- DiaglibHandlerGen2::\
                            vOnDiagnosisDataRead(%x)",u32Return));

   return u32Return;
}

/*-----------------------------------------------------------------------------*
 * tU32 vOnDiagnosisDataWrite(tU32 u32DiagDataId,                              *
 *              const tclParameterVector& rfcoParamVector, tContext MsgContext)*
 *-----------------------------------------------------------------------------*/
tU32 DiaglibHandlerGen2::vOnDiagnosisDataWrite(tU32 /*u32DiagDataId*/
                 ,const tclParameterVector& /*rfcoParamVector*/
         ,tContext /*MsgContext*/)
{
   ETG_TRACE_ERR(("!!! DiaglibHandlerGen2::vOnDiagDataWrite \
                     => ERROR : not supported"));

   return U32_DIAGLIB_RETURN_NOT_OK;
}

/*-----------------------------------------------------------------------------*
 * tU32 u32ProcessHMIModuleInput(tContext MsgContext)                          *
 *-----------------------------------------------------------------------------*/
tU32 DiaglibHandlerGen2::u32ProcessHMIModuleInput(tContext MsgContext)
{

    ETG_TRACE_USR4(("Begin: u32ProcessHMIModuleInput"));
    ETG_TRACE_COMP(("++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"));
    ETG_TRACE_COMP(("     HMI DIAGNOSIS REQUEST (PID $C747 HMI Module Input / Output)        "));
    ETG_TRACE_COMP(("------------------------------------------------------------------------"));

    //variables
    tU8  u8Result                = 0;
    tU32 u32Return                = (tU32)U32_DIAGLIB_RETURN_NOT_OK;
    tenDiagDataResult oDataResult;

    //trParameter        paramaError;
    trParameter        paramaUSBMediaStatus;
    tclParameterVector oTmpVec;  // create object for response
#ifdef USE_FOURTH_USB_CONNECTOR
    //class variable to hold HUB1,HUB2,HUB3,HUB4 Diagnosis data
    CUSBDiagnosis oUSBDiagnosisHUB1,
        oUSBDiagnosisHUB2, oUSBDiagnosisHUB3, oUSBDiagnosisHUB4;
#else
    //class variable to hold HUB1,HUB2,HUB3 Diagnosis data
    CUSBDiagnosis oUSBDiagnosisHUB1,
        oUSBDiagnosisHUB2, oUSBDiagnosisHUB3;
#endif//USE_FOURTH_USB_CONNECTOR
    tU32 u32BitMaskArray[ARRAYSIZEFORUSB];
    u32BitMaskArray[eMaskBit2_SD]                 = MASK_BIT2_SD;
    u32BitMaskArray[eMaskBit0_USB1 /*==eUSB1*/] = MASK_BIT0_USB1;
    u32BitMaskArray[eMaskBit1_USB2 /*==eUSB2*/] = MASK_BIT1_USB2;
    u32BitMaskArray[eMaskBit2_USB3 /*==eUSB3*/] = MASK_BIT2_USB3;
#ifdef USE_FOURTH_USB_CONNECTOR
    u32BitMaskArray[eMaskBit3_USB4 /*==eUSB4*/] = MASK_BIT2_USB4;
#endif//USE_FOURTH_USB_CONNECTOR
    CUSBDiagnosis *poUSBDiagnosisHUB[ARRAYSIZEFORUSB];
    poUSBDiagnosisHUB[0]     = NULL;
    poUSBDiagnosisHUB[eUSB1] = &oUSBDiagnosisHUB1;
    poUSBDiagnosisHUB[eUSB2] = &oUSBDiagnosisHUB2;
    poUSBDiagnosisHUB[eUSB3] = &oUSBDiagnosisHUB3;
#ifdef USE_FOURTH_USB_CONNECTOR
    poUSBDiagnosisHUB[eUSB4] = &oUSBDiagnosisHUB4;
#endif//USE_FOURTH_USB_CONNECTOR
    //get current device info
    oUSBDiagnosisHUB1.clear();
    oUSBDiagnosisHUB2.clear();
    oUSBDiagnosisHUB3.clear();
#ifdef USE_FOURTH_USB_CONNECTOR
    oUSBDiagnosisHUB4.clear();
    DeviceManagerInterface::GetInstance()->bGetDiagnosis(CGlobalEnumerations::DIAG_PDT_ONDEMAND,
                                                         oUSBDiagnosisHUB1,
                                                         oUSBDiagnosisHUB2,
                                                         oUSBDiagnosisHUB3,
                                                         oUSBDiagnosisHUB4);
#else
    DeviceManagerInterface::GetInstance()->bGetDiagnosis(CGlobalEnumerations::DIAG_PDT_ONDEMAND,
                                                         oUSBDiagnosisHUB1,
                                                         oUSBDiagnosisHUB2,
                                                         oUSBDiagnosisHUB3);
#endif//USE_FOURTH_USB_CONNECTOR

    //---------------------------------------------------------------------------------------
    //get result based on recognition of UdevMgr of this component
    //---------------------------------------------------------------------------------------
    for(tInt i=(tInt)eUSB1;i<(tInt)ARRAYSIZEFORUSB;i++)
    {
       //these values could be used to identify if a device is conected to e.g. USB1 or USB2
       //and what type is connected
       ETG_TRACE_COMP(("poUSBDiagnosisHUB[USB%1d]->m_cSerialID = %s",i,
                        poUSBDiagnosisHUB[i]->m_cSerialID.toStdString().c_str()));
       ETG_TRACE_COMP(("poUSBDiagnosisHUB[USB%1d]->m_cMountPoint = %s",i,
                        poUSBDiagnosisHUB[i]->m_cMountPoint.toStdString().c_str()));
       ETG_TRACE_COMP(("poUSBDiagnosisHUB[USB%1d]->m_eDeviceType = %d",i, ETG_ENUM(DVM_DEVICE_TYPE,poUSBDiagnosisHUB[i]->m_eDeviceType)));
       ETG_TRACE_COMP(("poUSBDiagnosisHUB[USB%1d]->m_ulSize = %u",     i,poUSBDiagnosisHUB[i]->m_ulSize));

       tBool bSetResult = FALSE;
       switch(poUSBDiagnosisHUB[i]->m_eDeviceType)
       {
           case CGlobalEnumerations::DTY_USB:
           case CGlobalEnumerations::DTY_SD:
           case CGlobalEnumerations::DTY_IPOD:
           case CGlobalEnumerations::DTY_IPHONE:
           case CGlobalEnumerations::DTY_MTP:
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES //check if this switch is necessary here
           case CGlobalEnumerations::DTY_NOT_SUPPORTED:
#endif
           case CGlobalEnumerations::DTY_SD_INTERNAL:
           case CGlobalEnumerations::DTY_CDROM:
               bSetResult = TRUE;
               break;
           case CGlobalEnumerations::DTY_IPAD:
               if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_EXTERNAL_IPAD))
               {
                   bSetResult = TRUE;
               }
               break;
           default:
               break;
       }

       if(bSetResult == TRUE)
       {
            u8Result = (tU8)(u8Result  | u32BitMaskArray[i]);
           ETG_TRACE_COMP(("u32ProcessHMIModuleInput:USB%i : %d connected",i, ETG_ENUM(DVM_DEVICE_TYPE,poUSBDiagnosisHUB[i]->m_eDeviceType)));

           if(CGlobalEnumerations::DTY_SD  == poUSBDiagnosisHUB[i]->m_eDeviceType)
           {
               ETG_TRACE_COMP(("u32ProcessHMIModuleInput:USB%i :SD card found",i));
               u8Result = (tU8)(u8Result  | u32BitMaskArray[eMaskBit2_SD]);//cast is ok possible values of u32BitMaskArray are in range
           }
           //@todo further possibilities to recognized more precise:
           //DTY_SD,DTY_USB                : check for existence of mount point, size and serialID
           //DTY_IPOD,DTY_IPHONE,(DTY_IPAD): check for existence of mount point, and serialID
       }

    }
    //-------------------------------------------------------------------------
    //result ok - prepare handover to Diagnosis
    //-------------------------------------------------------------------------
    ETG_TRACE_USR4(("u32ProcessHMIModuleInput: oDataResult = EN_DIAGDATA_OK    "));
    ETG_TRACE_COMP(("u32ProcessHMIModuleInput: u8Result = 0x%x",u8Result));

    oDataResult = EN_DIAGDATA_OK; // data valid
    paramaUSBMediaStatus.enType    = static_cast<tenParameterType>(DATAID_USB_MEDIA_STATUS_READ_RESULT_u32MediaStatusBitField );
    paramaUSBMediaStatus.u32Value  = (tU32)u8Result;
    oTmpVec.push_back(paramaUSBMediaStatus);

    //@todo cleanup this can be put to its own function
    //-------------------------------------------------------------------
    // Send message back to diagnosis
    //-------------------------------------------------------------------
    if ((OSAL_NULL != m_poDiaglibService) && (OSAL_NULL != m_poDataIDIf))
    {
        if (TRUE == m_poDataIDIf->bSendDiagnosisDataResult(oDataResult,
                        oTmpVec, MsgContext))
        {
           ETG_TRACE_USR4(("u32ProcessHMIModuleInput:bSendDiagnosisDataResult successfull"));
           u32Return = U32_DIAGLIB_RETURN_OK;
        }  // if (TRUE == m_poDataIDIf->bSendDiagnosisDataResult(oDataResult
        else
        {
           ETG_TRACE_ERR(("u32ProcessHMIModuleInput:!!! bSendDiagnosisDataResult failed"));
           u32Return = U32_DIAGLIB_RETURN_NOT_OK;
        }  // End of if-else; if (TRUE == m_poDataIDIf->bSendDiagnosisDataResul..
    }  // if ((OSAL_NULL != m_poDiaglibService) && (OSAL_NULL != m_poDataIDIf))
    else
    {
       u32Return = U32_DIAGLIB_RETURN_NOT_OK;
       ETG_TRACE_USR4(("u32ProcessHMIModuleInput:!!! m_poDiaglibService or m_poDataIDIf  == OSAL_NULL"));
    }  // End of if-else; if ((OSAL_NULL != m_poDiaglibService) && (OSAL_NULL !..

    ETG_TRACE_USR4(("End  : u32ProcessHMIModuleInput"));

    return u32Return;
}  // tU32 DiaglibHandlerGen2::u32ProcessUSBVbusRequest(tC..

/*-----------------------------------------------------------------------------*
 * tVoid vOnDiagnosisDataWriteTEST()                                           *
 *-----------------------------------------------------------------------------*/
tVoid DiaglibHandlerGen2::vOnDiagnosisDataWriteTEST(tVoid)
{
   ETG_TRACE_FATAL(("-------------------------/vOnDiagnosisDataWriteTEST-------------------------"));

   u32ProcessUSBVersionRequest(eUSB1,0,/*bTestMode=*/TRUE);
   u32ProcessUSBVersionRequest(eUSB2,0,/*bTestMode=*/TRUE);
   u32ProcessUSBVersionRequest(eUSB3, 0,/*bTestMode=*/TRUE);
#ifdef USE_FOURTH_USB_CONNECTOR
   u32ProcessUSBVersionRequest(eUSB4,0,/*bTestMode=*/TRUE);
#endif//USE_FOURTH_USB_CONNECTOR
   u32ProcessUSBVbusRequest(eUSB1,0,   /*bTestMode=*/TRUE);
   u32ProcessUSBVbusRequest(eUSB2,0,   /*bTestMode=*/TRUE);
   u32ProcessUSBVbusRequest(eUSB3, 0,   /*bTestMode=*/TRUE);
#ifdef USE_FOURTH_USB_CONNECTOR
   u32ProcessUSBVbusRequest(eUSB4,0,   /*bTestMode=*/TRUE);
#endif//USE_FOURTH_USB_CONNECTOR
   //u32ProcessHMIModuleInput(MsgContext)

   ETG_TRACE_FATAL(("-------------------------vOnDiagnosisDataWriteTEST/-------------------------"));
}

////////////////////////////////////////////////////////////////////////////////
// <EOF>
