/*-----------------------------------------------------------------------------*
 * DeviceListManager.h                                                         *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*
*-----------------------------------------------------------------
*                                    development for Gen3:
*-----------------------------------------------------------------
*\version 23.10.2013, Christian Koechling (Bosch)
*          -# start tp replace QStrings by replacing QString by define GENSTRING
*
* \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
*/


/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file DeviceListManager.h
 *
 * \brief This file holds declaraion for class DeviceListManager
 *
 * \version 25.08.2011,Koechling, Christian (Bosch), Initial Version
 * \version 27.07.2012, Negi, Sunder (MontaVista), Cleaned up file hierarchy
 * \version 02.08.2012, Negi, Sunder (MontaVista),
 *          -# Refactored debug information logging
 *          -# Added History Manager, refer define HISTORY_MANAGER
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 10.08.2012, Negi, Sunder (MontaVista), renamed singleton functions
 * \version 13.08.2012, Negi, Sunder (MontaVista), cleanup logging to usb (WRITEHISTORYTOUSBSTICK)
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

#ifndef __DEVICELIST_MANAGER_H__
#define __DEVICELIST_MANAGER_H__

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace statetbl;
using namespace devlistmgr;

/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/
/*!
 * \class DeviceListManager
 *
 * \brief functions as device store
 */
class DeviceListManager
{
    protected:

        // static member variable declared for singleton class
        static DeviceListManager     *m_pDeviceListManager;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate solution
        static DVMLOCK  m_singelton;
#endif

        vector<CDevice>             m_vConnectedDevices_UdevMgr;
        vector<structNotifyClients> m_vConnectedDevices_StateTable;
        vector<CDevice>             m_vConnectedDevices_Service;
        CDeviceCard                 m_rDevicesCard[ARRAYSIZEFORCARD];

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate solution
        static  DVMLOCK m_lock;
#endif


        static tBool bPlatformEventsLocked[];

        tVoid _deleteList(DEVICELIST_Type eListType, tenDevManagerUSBHost eUsb);






                /*!
         * \brief
         *
         * \param [in] eDevListType
         * \param [in] cMount
         *
         * \return [tenDevManagerUSBHost]
         */
        tenDevManagerUSBHost mapMountPoint(DEVICELIST_Type eDevListType ,const char* cMount);

#ifdef REFACTOR_STATETABLE_OUTPUT

        /*helper originally placed in StateTable coul be replaced if DeviceListManager would be refactored to wor with an single list only*/
        tBool bFillDevice(OUT CDevice* pCurDevice_Dest, IN const structNotifyClients* pCurDevice_Src) const;

        tBool bFillDevice( OUT structNotifyClients* pCurDevice_Dest, IN const CDevice* pCurDevice_Src) const;
#endif


#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
        public:
#endif

        /*!
         * \brief constructor
         */
        DeviceListManager(tVoid);

        /*!
         * \brief constructor
         */
        ~DeviceListManager(tVoid);


    public:

        /*!
         * \brief Get Device list manager object
         *
         * Provides the singleton Device list manager object. First invocation of
         * this function creates the object
         *
         * \return [DeviceListManager*] singleton object
         */
        static DeviceListManager * GetInstance();

        /*!
         * \brief Initializes the object
         *
         * \return [int] initialization status OSAL_OK or OSAL_ERROR
         */
        int Initialize() const;

        /*!
         * \brief Destroys the singleton object
         */
        static void  DestroyInstance();

     /*!
      * \brief check if a device is at a certain USBConnector e.g. at eUsb1 or eUSB2
      *
      *  \param [in] eUsb: eUSB1 or eUSB2 ...
      *  \param [in] onlyPlayableDevice:if true, Do not count non-playable devices like DTY_DVD_DRIVE and DTY_HUB, else consider all devices
      *
      * \return: true: USB_DEV_CONNECTED,USB_DEV_UNAVAIL_BAT_LOWVOLT,USB_DEV_UNAVAIL_HW_MALFUNC,USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT,USB_DEV_UNAVAIL_CDDRIVE_TEMPERATURE
      *               false: USB_DEV_UNDEFINED,USB_DEV_REMOVED_BY_USR


      */
     bool IsDeviceAtConnector(tenDevManagerUSBHost eUsb,DEVICELIST_Type eDevListType,const bool onlyPlayableDevice = false);


     /*!
      * \brief trace lists
      */
     void ShowList(DEVICELIST_Type eDevListType);
        /*!
         * \brief Assignment Operator
         *
         * Will not be implemented. Avoids Lint Prio 3 warning: Info 1732: new
         * in constructor for class 'DeviceListManager' which has no assignment
         * operator.
         *
         * \note This is a technique to disable the assignment operator for this
         *       class. So if an attempt for the assignment is made compiler
         *       complains.
         */
        DeviceListManager& operator=(const DeviceListManager &oListManager);



        /*!
         * \brief find number of devices connected at usb in list eDevListType
         *
         * \param [in]  eDevListType
         * \param [in]  usb
         * \param [out] iNumOfConnectedDevices
         * \param [out] iAnyApplyFor
         */
        tVoid checkNumOfDevicesAtConnector(DEVICELIST_Type eDevListType,
                tenDevManagerUSBHost usb,
                tInt &iNumOfConnectedDevices,
                tInt &iAnyApplyFor);


#ifdef REFACTOR_STATETABLE_OUTPUT
        /*!
             * \brief update list is e.g. used is StateTable has calculated result of connected devices
             *
             * \param [out] eDevListTypeTarget - this list should be updated according to the content of eDevListTypeSource
             * \param [in]   eDevListTypeSource    all elements of this list should appear in eDevListTypeTarget in the same way
             *
             */
        tVoid UpdateList(OUT  DEVICELIST_Type eDevListTypeTarget,IN  DEVICELIST_Type eDevListTypeSource);

        /*!
             * \brief compare content of two lists this is thought to prevent unneccessary events to clients repeating already send messages
             *          ignores if elements are stored in differen order in that vector
             *
             * \param [in] eDevListTypeOne:    list to compare with eDevListTypeTwo
             * \param [in] eDevListTypeTwo: - list to compare with eDevListTypeOne
             *
             */
        tBool bGetIsListEqual(IN  DEVICELIST_Type eDevListTypeOne,IN  DEVICELIST_Type eDevListTypeTwo);



        tVoid vEraseFromListRemovedDevices(DEVICELIST_Type eDevListType);

        tVoid CheckSetStateAtConnectorsDueToApplyAllDevices();

        tVoid SetStateAtConnectorsDueToHWSignalsAndVoltage(IN voltageFailure_Result_t (&f_result)[ARRAYSIZEFORUSB]);


        tVoid GetCopyDeviceListOfService(vector<CDevice>  &f_ListOfSendDevicestServices);




#endif


        /*!
         * \brief Add a device to list
         *
         * \param [in] eDevListType the list type
         * \param [in] eUsb         the usb connecter
         * \param [in] oNewCurDevice new device to add
         *
         * \return [tInt]
         */
        tInt ListAddDevice(DEVICELIST_Type eDevListType,
                tenDevManagerUSBHost eUsb,
                structNotifyClients oNewCurDevice);

        /*!
         * \brief Add a device to list
         *
         * \param [in] eDevListType the list type
         * \param [in] device new device to add
         */
        tVoid ListAddDevice(DEVICELIST_Type eDevListType,
                const CDevice  &device);

        /*!
         * \brief
         *
         * \param [in] eDevListType the list type
         * \param [in] l_oDevice device to remove
         */
        tVoid vListAddRemoveDevice(DEVICELIST_Type eDevListType,CDevice l_oDevice);

        /*!
         * \brief
         *
         * \param [in] eDevListType the list type
         * \param [in] eUsb the usb connector
         *
         * \return [tInt]
         */
        tInt ListChkDeleteDevice(DEVICELIST_Type eDevListType,
                tenDevManagerUSBHost eUsb);

        /*!
         * \brief List all devices at connector eUsb in list eDevListType
         *
         * \param [in] eDevListType the list type
         * \param [in] eUsb the usb connector
         */
        tVoid TraceAllDevice(DEVICELIST_Type eDevListType,
                tenDevManagerUSBHost eUsb);

        /*!
         * \brief List all devices at connector eUsb in list eDevListType that
         *        are in apply connect device
         *
         * \param [in] eDevListType the list type
         * \param [in] eUsb the usb connector
         * \param [in] eApply
         * \param [in] f_oCurDevice
         */
        tVoid ListApplyConnectDevice(DEVICELIST_Type eDevListType,
                tenDevManagerUSBHost eUsb,
                DEVICE_CONNECTSTATUS_Type eApply,
                structNotifyClients &f_oCurDevice);

        /*!
         * \brief Get list of devices
         *
         * \param [in]  eDevListType the list type
         * \param [out] f_vConnectedDevices list of devices
         */
        void getAllConnectedDevices(DEVICELIST_Type eDevListType,
                vector<structNotifyClients> &f_vConnectedDevices);

        /*!
         * \brief Get list of devices
         *
         * \param [in]  eDevListType the list type
         * \param [out] f_vConnectedDevices list of devices
         */
        void getAllConnectedDevices(DEVICELIST_Type eDevListType,
                vector<CDevice *> &f_vConnectedDevices);

        /*!
         * \brief Get list size
         *
         * \param [in] eDevListType the list type
         *
         * \return [unsigned int] list size
         */
        unsigned int getSizeOfList(DEVICELIST_Type eDevListType) const;

        /*!
         * \brief Get element from list at particular index
         *
         * \param [in]  iIndex the index
         * \param [out] device the device
         *
         * \return [tBool] true or false
         */
        tBool getElement(tUInt iIndex, structNotifyClients &device);

        /*!
         * \brief Get element from list at particular index
         *
         * \param [in]  eDevListType the list type
         * \param [in]  iIndex the index
         * \param [out] device the device
         *
         * \return [tBool] true or false
         */
        tBool getElement(DEVICELIST_Type eDevListType,tUInt iIndex, CDevice &device);

        /*!
         * \brief Update element from list at particular index
         *
         * \param [in]  eDevListType the list type
         * \param [in]  iIndex the index
         * \param [out] device the updated device
         *
         * \return [tBool] true or false
         */
        tBool setElement(DEVICELIST_Type eDevListType,tUInt iIndex, const CDevice &device);

        /*!
         * \brief Delete element from list at particular index
         *
         * \param [in]  eDevListType the list type
         * \param [in]  iIndex the index
         *
         * \return [tBool] true or false
         */
        tBool deleteElement(DEVICELIST_Type eDevListType,tUInt iIndex);

        /*!
         * \brief Delete list
         *
         * \param [in] eListType the list type
         */
        tVoid deleteList(DEVICELIST_Type eListType);

        /*!
         * \brief Delete list
         *
         * \param [in] eUsb the usb connector
         */
        tVoid deleteLists(tenDevManagerUSBHost eUsb);

private:
        /*!
         * \brief Lock lists
         */
        static tVoid lockList();

        /*!
         * \brief Unlock lists
         */
        static tVoid unlockList();

public:
        /*!
         * \brief Fill device objects
         *
         * \param [out] pNotifyClients_Dest the destination device object
         * \param [out] pNotifyClients_Src the source device object
         *
         * \return [tBool] true or false
         */
        tBool bFillstructNotifyClients(structNotifyClients *pNotifyClients_Dest,
                const structNotifyClients *pNotifyClients_Src) const;

        /*!
         * \brief Is device already connected
         *
         * \param [in] f_cDeviceNode the device node
         *
         * \return [tBool] true or false
         */
        tBool IsAlreadyConnected(GENSTRING f_cDeviceNode);

        /**
         * \brief
         *
         * \param [in] eDevListType the list type
         * \param [in] l_pCurDevice the device
         */
        void Set_eConnectStatus(DEVICELIST_Type eDevListType,
                structNotifyClients *l_pCurDevice);//Roadmap13035_Overtemp - rename function to SetConnectStatus


        /**
         * \brief
         *
         * \param [in] eDevListType the list type
         * \param [in] l_pCurDevice the device
         */

     void SetConnectStatus(DEVICELIST_Type eDevListType,
                CDevice &l_oCurDevice,DEVICE_CONNECTSTATUS_Type eConnectStatus);//Roadmap13035_Overtemp - rename function to SetConnectStatus

        /*!
         * \brief lock platform events at usb connector
         *
         * \param [in] eUsb the usb connector
         */
        void lockPlatformEvents(tenDevManagerUSBHost eUsb) const;

        /*!
         * \brief unlock platform events at usb connector
         *
         * \param [in] eUsb the usb connector
         */
        void unlockPlatformEvents(tenDevManagerUSBHost eUsb) const;

        /*!
         * \brief get locked platform events at usb connector
         *
         * \param [in] eUsb the usb connector
         */
        tBool getPlatformEventsLocked(tenDevManagerUSBHost eUsb) const;

        tVoid TraceAllDevice2Logfile(FILE *l_pFileLog,DEVICELIST_Type eDevListType,tenDevManagerUSBHost eUsb);

        tVoid setCardInfo( const CDeviceCard& l_oDeviceCard );

        tVoid getCardInfo( CDeviceCard* poDeviceCard, tenSlot eSlot ) const;

        /*!
         * \brief Check whether the provided devices is present in the list of devices
         *
         * \param [in] f_iVendorID the vendor id
         * \param [in] f_iProductID the product id
         * \param [in] f_eDeviceType the device type to search for
         * return [tBool] true or false
         */
        tBool isDeviceConnected(IN uint16_t f_iVendorID,IN uint16_t f_iProductID,IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType) const;

#ifdef MSD_BIGGEST_PARTITON_FEATURE
        /*!
         * \brief Checks whether all partitions are retried to mount
         * return [tBool] true or false
         */
        tBool isAllPartitionsRetried();

        /*!
         * \brief Updates the u8TotalValidMountedPartitions of devices
         * \param [in] f_cSerialID the Serial ID to be updated for in list of devices
         * \param [in] u8TotalValidMountedPartitions the count to be updated for in list of devices
         */
        tVoid UpdatePartitionsCount(GenString f_cSerialID, tU8 u8TotalValidMountedPartitions);

        /*!
         * \brief Checks whether platform notified all the partitions of Mass storage device
         *
         * return [tBool] true or false
         */
        tBool PlatformHasAllPartitionsofDeviceNotified();

        /*!
         * \brief Finds the biggest partition of mass storage device and change the connect status to Connected
         * \param [in] f_cSerialID the Serial ID to be searched for in list of devices
         *
         *
         */
        tVoid FindBiggestPartitionfDeviceAndUpdateList(GenString f_cSerialID);
        /*!
         * \brief Sorts the devices with the same serial ID (stored in the cAccessaryname2 field of structure) in the list based on the size of partitions in the list.
         * \param [in] f_cSerialID the Serial ID to be searched for in list of devices
         *
         *
         */
        tVoid SortPartitionsAndUpdateList(GenString f_cSerialID);

#endif//MSD_BIGGEST_PARTITON_FEATURE




        /*!
         * \brief Update the timestamp of malfunction in state table device list
         *
         * \param [in] eUsb the usb connector
         * \param [in] s32MalfunctionElapsedTime_sec the usb connector
         */
        tVoid UpdateMalfunctionInfoWithTimeStamp(IN tenDevManagerUSBHost eUsb,IN tS32 s32MalfunctionStartTime);

        /*!
         * \brief Update the timestamp of malfunction in state table device list
         *
         * \param [in] f_eDeviceType the device type either Optical disc or SD internal
         * \param [in] s32MalfunctionElapsedTime_sec the usb connector
         */
        tVoid UpdateMalfunctionInfoWithTimeStamp(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType,IN tS32 s32MalfunctionElapsedTime_sec);

        /*!
         * \brief Update the Connection state of state table device list to Malfunction
         *
         * \param [in] eUsb the usb connector
         * \param [in]  f_eConnectStatus the Connect status to be updated
         *
         */
        tVoid SetStateAtConnectorsDueToMalfunction(IN tenDevManagerUSBHost eusb, IN DEVICE_CONNECTSTATUS_E f_eConnectStatus);

        /*!
         * \brief Update the Connection state of state table device list to Malfunction
         *
         * \param [in] f_eDeviceType the device type either Optical disc or SD internal
         * \param [in]  f_eConnectStatus the Connect status to be updated
         *
         */
        tVoid SetStateAtConnectorsDueToMalfunction(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType, IN DEVICE_CONNECTSTATUS_E f_eConnectStatus);

        /*!
         * \brief Check whether the removal is caused by malfunction
         *
         * \param [in] eUsb the usb connector
         *
         */
        tBool bIsRemovedCausedByMalfunction(IN tenDevManagerUSBHost eUsb);

        /*!
         * \brief Gets the duration of malfunction
         *
         * \param [in]  eUsb the usb connector
         * \param [out] s32MalfunctionElapsedTime_sec Duration of Malfunction
         */
        tBool GetMalfunctionTimeStampForDevices(IN tenDevManagerUSBHost eusb,OUT tS32 &s32MalfunctionElapsedTime_sec);

        /*!
         * \brief Gets the duration of malfunction
         *
         * \param [in]  f_eDeviceType the device type either Optical disc or SD internal
         * \param [out] s32MalfunctionElapsedTime_sec Duration of Malfunction
         */
        tBool GetMalfunctionTimeStampForDevices(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType,OUT tS32 &s32MalfunctionElapsedTime_sec);

        /*!
         * \brief Gets the Device type of Optical disc by searching through the device list, it can be CDROM or CDDA
         *
         * \param [in]  f_eDeviceType the device type either CDROM or CDDA
         */
        tVoid GetOpticalDiscDeviceType(OUT CGlobalEnumerations::DEVICE_TYPE_Type &f_eDeviceType);

        /*!
         * \brief Store the Time stamp of malfunction
         *
         * \param [in]  eUsb the usb connector
         * [out] u64MalfunctionTimeStamp Timestamp of Malfunction start
         */
        tVoid vStoreMalfunctionTimeStamp(IN tenDevManagerUSBHost eusb, IN tU64 u64MalfunctionTimeStamp);


          /*!
           * \brief Store the Time stamp if there is high Risk for Malfunction
           *
           * \param [in]  eUsb the usb connector
           * [out] u64MalfunctionTimeStamp Timestamp of Malfunction start
           */
        tVoid vStoreHighRiskForMalfunctionTimeStamp(IN tU64 f_u64HighRiskForMalfunctionTimeStamp);

        /*
         * \brief Store the Time stamp of Connection
         *
         * \param [in] oCurDevice  device to be updated in the list
         */
        tVoid vStoreConnectTimeStamp(const structNotifyClients& oCurDevice);

        /*!
         * \brief Store the Time stamp of malfunction
         *
         * \param [in] oCurDevice  device to be updated in the list
         */
        tVoid vStoreRemoveTimeStamp(const structNotifyClients& oCurDevice);

#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
        /*!
         * \brief Reset the Time stamp of malfunction for the USB connector supplied
         *
         * \param [in] eusb  the usb connector
         */
        tVoid ResetMalfunctionTimeStamp(IN tenDevManagerUSBHost eusb );
#endif//TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION

        /*!
         * \brief function to update connect state of CDROM or CDDA devices. This function is called when there
         * is over temperature scenario present in the optical drive
         * This function will iterate through the list of devices and for CDROM or CDDA Devices update the connect status
         *  USB_DEV_INTERNAL_APPLY_CONNECT  => USB_DEV_UNAVAIL_CDDRIVE_TEMPERATURE
         *
         */
        tVoid SetStateAtConnectorsDueToOverTemperature();


        /*!
         * \brief Checks the device type is a mass storage device or not
         *
         * \param [in] f_eDeviceType device type to be checked
         */
         tBool isMassStorageDevice(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType);



#ifdef USE_HIGHRISK4USB_HWMALFUNCTION
        tVoid CheckSetStateTemporaryNotAvailableAllDevices(tU64 u64TestValue = 0);
        tVoid CheckSetStateConnectAfterTemporaryNotAvailableAllDevices();
        tVoid CheckSetStatePermanentlyNotAvailableAllDevices();

        tBool bCheckIncrementMalfunctionEclapsedTimeAllDevices(tU16 u16Increment_sec);


#endif

        /*!
         * \brief function to check whether the device is connected or not on a perticular port
         * \param [in] eDevListType  : List type statetable, Udevmanager,Service
         * \param [in] eUsb          : the USB connector
         * \param [in] f_eDeviceType : device type
         */
        tBool isDeviceTypeConnected(IN DEVICELIST_Type eDevListType,IN tenDevManagerUSBHost eUsb,IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType) const;

        /*!
         * \brief function to check whether the device is a part of Multipartition device
         * \param [in] eDevListType  : List type statetable, Udevmanager,Service
         * \param [in] f_AccessoryName2 : The AccessoryName2 to check for Multipartition
         */
        tBool isDeviceMultiPartitioned(IN DEVICELIST_Type eDevListType, IN GENSTRING f_AccessoryName2) const;


        /*!
         * \brief function to check whether the device type is still connected after the specified time and update the device type
         * \param [in] f_eDeviceType : device type
         * \param [in] u64RetryElapsedTime_sec the usb connector
         */
        tBool UpdateDeviceTypeIfRetryCompleted(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType, IN tU64 u64RetryElapsedTime_sec);

        /*!
         * \brief function to stop the retrial for for device type, this update will come from SPI client handler
         * \param [in] fSerialID : serial id of the device
         * \param [out] If the retry for  any of the devices is done, then return true.
         */
        tBool vStopRetryForDevice(IN GENSTRING fSerialID);
};

#endif //__DEVICELIST_MANAGER_H__

////////////////////////////////////////////////////////////////////////////////
// EOF
