/*-----------------------------------------------------------------------------*
 * Device.cpp                                                                  *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file Device.cpp
 *
 * Container to hold device information i.e. properties of the connected devices
 * and it is used for notification to clients.
 *
 * \version Initial Version
 * \version 05.08.2011, Koechling, Christian (Bosch), Interface VD_DeviceManager changed
 * \version 01.11.2011, Koechling, Christian (Bosch),
 *          -# add 'ProductID to charge Apple devices'
 *          -# add AccessoryName for IPOD and FC_MediaPlayer
 *          -# Statemachine cares for >1 devices connected to HUB
 *          -# cleanup
 * \version 01.11.2011, Negi, Sunder (Montavista), Traces modified
 * \version 17.11.2011, Koechling, Christian (Bosch),
 *          -# cleanup: 1st step to shift includes to Common.h
 *          -# merge of latest MV bugfixes from Sunder 9.11.2011
 * \version 28.11.2011, Koechling, Christian (Bosch), cleanup - add headers
 * \version 06.12.2011, Koechling, Christian (Bosch), enable ETGTrace, add send counter
 * \version 16.11.2011, Negi, Sunder (Montavista), force mount added
 * \version 14.03.2012, Koechling, Christian (Bosch), cleanup define USE_BUGFIX_HUB
 * \version Negi, Sunder (Montavista), Modify as per device type
 * \version Negi, Sunder (Montavista), Update traces
 * \version 27.07.2012, Negi, Sunder (MontaVista), Cleaned up file hierarchy
 * \version 02.08.2012, Negi, Sunder (MontaVista),
 *          -# Refactored debug information logging
 *          -# Added History Manager, refer define HISTORY_MANAGER
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 *          fixed traces again
 * \version 09.08.2012, Negi, Sunder (MontaVista), usb and port no are made integer
 * \version 14.08.2012, Negi, Sunder (MontaVista), added member to check if
 *          this device is being used for storing the traces
 * \version 24.08.2012, Negi, Sunder (MontaVista), cleanup HANDLE_GPT
 * \version 28,08,2012, Negi, Sunder (MontaVista), modified class members
 *
 *-----------------------------------------------------------------
 *                                    development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch) 
 *          -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include "Enums.h"
#include "DeviceCard.h"

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_STATE_TABLE  /*Reason: triggered by stateTable which is e.g. calling a Show()*/
#include "trcGenProj/Header/DeviceCard.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
CDeviceCard::CDeviceCard()
{
    m_eSlot              = eCard0;
    m_eConnected         = SIG_UNDEF;
    m_bOverCurrent       = FALSE;
    m_bSlotEnable        = TRUE;
    m_u16MediaState      = 0;
    m_u16MediaType       = 0;
    m_bHW_WriteProtected = FALSE;
    m_bSW_WriteProtected = FALSE;
    m_bMounted           = FALSE;
    m_u8ManufactureId    = 0x00;
    m_u32SerialNumber    = 0x00000000;
    m_u8SDCardSpecVersion= 0x00;
    m_u64CardSize        = 0x0000000000000000;
    for( tInt i=0; i<SizeCIDRegister; ++i )
    {
        m_u8CIDRegister[i] = 0;
    }

}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
CDeviceCard::~CDeviceCard()
{
    // Do nothing
}

/*-----------------------------------------------------------------------------*
 * void vSetErrorValues()                                                      *
 *-----------------------------------------------------------------------------*/
void CDeviceCard::vSetErrorValues()
{
    m_eSlot              = eCard0;
    m_eConnected         = SIG_UNDEF;
    m_strUUIDDeviceName  = "";
    m_strMountPoint      = "";
    m_bOverCurrent       = FALSE;
    m_bSlotEnable        = TRUE;
    m_u16MediaState      = OSAL_C_U16_MEDIA_NOT_READY;
    m_u16MediaType       = OSAL_C_U16_MEDIA_EJECTED;
    m_bHW_WriteProtected = FALSE;
    m_bSW_WriteProtected = FALSE;
    m_bMounted           = FALSE;
    m_u8ManufactureId    = 0xFF;
    m_u32SerialNumber    = 0xFFFFFFFF;
    m_u8SDCardSpecVersion= 0xFF;
    m_u64CardSize        = OSAL_C_U64_MAX;
    for( tInt i=0; i<SizeCIDRegister; ++i )
    {
        m_u8CIDRegister[i] = 0;
    }
}

/*-----------------------------------------------------------------------------*
 * void CDeviceCard::FillDevice(...)
 *-----------------------------------------------------------------------------*/
void CDeviceCard::FillDevice( const CDeviceCard& f_CDeviceCard )
{
    m_eSlot              = f_CDeviceCard.m_eSlot;
    m_eConnected         = f_CDeviceCard.m_eConnected;
    m_strUUIDDeviceName  = f_CDeviceCard.m_strUUIDDeviceName;
    m_strMountPoint      = f_CDeviceCard.m_strMountPoint;
    m_bOverCurrent       = f_CDeviceCard.m_bOverCurrent;
    m_bSlotEnable        = f_CDeviceCard.m_bSlotEnable;
    m_u16MediaState      = f_CDeviceCard.m_u16MediaState;
    m_u16MediaType       = f_CDeviceCard.m_u16MediaType;
    m_bHW_WriteProtected = f_CDeviceCard.m_bHW_WriteProtected;
    m_bSW_WriteProtected = f_CDeviceCard.m_bSW_WriteProtected;
    m_bMounted           = f_CDeviceCard.m_bMounted;
    m_u8ManufactureId    = f_CDeviceCard.m_u8ManufactureId;
    m_u32SerialNumber    = f_CDeviceCard.m_u32SerialNumber;
    m_u8SDCardSpecVersion= f_CDeviceCard.m_u8SDCardSpecVersion;
    m_u64CardSize        = f_CDeviceCard.m_u64CardSize;
    for( tInt i=0; i<SizeCIDRegister; ++i )
    {
        m_u8CIDRegister[i] = f_CDeviceCard.m_u8CIDRegister[i];
    }
}

/*-----------------------------------------------------------------------------*
 * void Show()                                                                 *
 *-----------------------------------------------------------------------------*/
void CDeviceCard::Show()
{
    ETG_TRACE_USR4 (("/---------------CDEVICECARD STRUCTURE------------------"));

    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_eSlot               = %d",     ETG_ENUM(DVM_CARD_SLOT, m_eSlot) ));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_eConnected          = %d",     ETG_ENUM(TENSIGNALVALUE, m_eConnected) ));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_strUUIDDeviceName   = %s",     m_strUUIDDeviceName.toStdString().c_str()));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_strMountPoint       = %s",     m_strMountPoint.toStdString().c_str()));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_bOverCurrent        = %d",     ETG_ENUM(BOOL, m_bOverCurrent) ));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_bSlotEnable         = %d",     ETG_ENUM(BOOL, m_bSlotEnable) ));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_u16MediaState       = %d",     ETG_ENUM(DVM_PRM_TABLE_STATUS_MEDIA_STATE, m_u16MediaState) ));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_u16MediaType        = %d",     ETG_ENUM(DVM_PRM_TABLE_NOTIF, m_u16MediaType) ));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_bHW_WriteProtected  = %d",     ETG_ENUM(BOOL, m_bHW_WriteProtected) ));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_bSW_WriteProtected  = %d",     ETG_ENUM(BOOL, m_bSW_WriteProtected) ));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_bMounted            = %d",     ETG_ENUM(BOOL, m_bMounted) ));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_u8ManufactureId     = 0x%x",    m_u8ManufactureId ));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_u32SerialNumber     = 0x%x",    m_u32SerialNumber ));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_u8SDCardSpecVersion = 0x%x",    m_u8SDCardSpecVersion ));
    ETG_TRACE_USR4 (("CDeviceCard::Show(): m_u64CardSize         = %d (KB)", m_u64CardSize/1024 ));
    for( tInt i=0; i<SizeCIDRegister; ++i )
    {
        ETG_TRACE_USR4 (("CDeviceCard::Show(): m_u8CIDRegister[%d]   = %d (KB)", i,m_u8CIDRegister[i] ));
    }
//    ETG_TRACE_USR4 (("Send Time (ms)    = %d  ", m_rSendTime_ms));
    ETG_TRACE_USR4 (("---------------CDEVICECARD STRUCTURE-------------------/"));




   

    return;
}


////////////////////////////////////////////////////////////////////////////////
// <EOF>
