/*-----------------------------------------------------------------------------*
 * TraceCmdManager.h                                                           *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file TraceCmdManager.h
 *
 * \brief Implements the TTFiS command based debug infortmation retrieval
 * 
 * \version 17.11.2011, Koechling, Christian (Bosch), cleanup: 1st step to shift
 *          includes to Common.h
 * \version 24.11.2011, Koechling, Christian (Bosch), cleanup: singelton:semaphore
 *          and private contructor
 * \version 06.12.2011, Koechling, Christian (Bosch), enable setting EOL values
 * \version 12.12.2011, Koechling, Christian (Bosch), cleanup singleton, further
 *          care for simulating EOL-values
 * \version 09.01.2012, Koechling, Christian (Bosch), cleanup conctructor TraceCmdMGR
 * \version 23.01.2012, Koechling, Christian (Bosch), added further test command
 *          for CalcHashID
 * \version 16.02.2012, Negi, Sunder (Montavista), cleanup -- lint
 * \version 23.02.2012, Koechling, Christian (Bosch), add TTFIS command
 *          VD_DVM_GET_CONNETED_HUBS
 * \version 10.03.2012, Koechling, Christian (Bosch),
 *          -# added commands to get history of messages send to diagnosis
 *          -# to switch off cyclic diag permanently
 * \version 17.03.2012, Koechling, Christian (Bosch), add command to check what
 *          could be found in /tmp/.automount for automatic tests
 * \version 22.03.2012, Koechling, Christian (Bosch), cleanup function names 'HISTORY'
 * \version 27.07.2012, Negi, Sunder (MontaVista), Cleaned up file hierarchy
 * \version 02.08.2012, Negi, Sunder (MontaVista),
 *          -# Refactored debug information logging
 *          -# Added History Manager, refer define HISTORY_MANAGER
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 *          fixed traces again
 * \version 10.08.2012, Negi, Sunder (MontaVista), cleanup constructors, renamed
 *          singleton functions
 * \version 13.08.2012, Negi, Sunder (MontaVista),
 *          -# cleanup logging to usb (WRITEHISTORYTOUSBSTICK)
 *          -# changed access specifiers of few functions to publilc
 *
 * \copyright  Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

#ifndef __TRACE_CMD_MANAGER_H__
#define __TRACE_CMD_MANAGER_H__

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
    
#include "StateTable.h"


#ifdef __cplusplus
extern "C" {
#endif


/*-----------------------------------------------------------------------------*
 * Defines                                                                     *
 *-----------------------------------------------------------------------------*/
#define DevManagerTRACE_MAX 239
#define DevManagerTRACE_MAX_BYTES 10
#define DevManagerTTFIScmd_ValHeaderFlag_1st  0xAF
#define DevManagerTTFIScmd_ValHeaderFlag_2nd  0xFE

/*-----------------------------------------------------------------------------*
 * forward declaration                                                                     *
 *-----------------------------------------------------------------------------*/
class IVoltageManager;
class IPrmManager;
class IPrmManagerCard;

/*-----------------------------------------------------------------------------*
 * Enums                                                                       *
 *-----------------------------------------------------------------------------*/
/*!
 * \enum tenDevManagerTTFISDataIndex
 *
 * \brief special cmd header
 */
typedef enum
{
   index_header1   = 0x00,
   index_header2   = 0x01,
   index_cmd       = 0x02,
   index_signal    = 0x03,
   index_signalVal = 0x04
}tenDevManagerTTFISDataIndex;

/*!
 * \enum tenDevManagerTTFIScmd
 *
 * \brief commands
 */
typedef enum
{
    TTFIScmd_Help                                   = 0x01,
    TTFIScmd_PrintConfiguration                     = 0x02,
    TTFIScmd_SimPrmSignal                           = 0x03,
    TTFIScmd_SimPrmSignal_U60_drop                  = 0x04,  //obsolete
    TTFIScmd_SimPrmSignal_U50_USB_Reset             = 0x05,  //obsolete
    TTFIScmd_SimPrmSignal_OC1                       = 0x06,  //obsolete
    TTFIScmd_SimPrmSignal_OC2                       = 0x07,  //obsolete 
    TTFIScmd_SimPrmSignal_PPON1                     = 0x08,  //obsolete
    TTFIScmd_SimPrmSignal_PPON2                     = 0x09,  //obsolete
    TTFIScmd_SimPrmSignal_mountedUSB1_prm           = 0x0A,  //obsolete
    TTFIScmd_SimPrmSignal_mountedUSB2_prm           = 0x0B,  //obsolete
    TTFIScmd_SimPrmSignal_mountedUSB1_udev          = 0x0C,  //obsolete
    TTFIScmd_SimPrmSignal_mountedUSB2_udev          = 0x0D,  //obsolete
    TTFIScmd_SimMSG2Client                          = 0x0E,
    TTFIScmd_GetSignals                             = 0x0F,
    TTFIScmd_GetLastMsg2clients                     = 0x10,
    TTFIScmd_SetTimerInterval_sec                   = 0x11,
    TTFIScmd_SetTimerInterval_msec                  = 0x12,
    TTFIScmd_GetConnectedDevices                    = 0x13,
    TTFIScmd_InterfaceConnect                       = 0x14,
    TTFIScmd_CyclicDiagnosis                        = 0x15,
    TTFIScmd_ShowConfiguration                      = 0x16,
    TTFIScmd_Sendtesttime_sec                       = 0x17,
    TTFIScmd_HistoryUdevEvts                        = 0x18,
    TTFIScmd_HistoryUdevEvts_Clear                  = 0x19,
    TTFIScmd_HistoryLastMsgsSend2Clients            = 0x1A,
    TTFIScmd_HistoryLastMsgsSend2Clients_Clear      = 0x1B,
    TTFIScmd_SetServiceAvailable                    = 0x1C,
    TTFIScmd_SimEOLValues                           = 0x1D,
    TTFIScmd_TriggerReadEOLValuesFromFile           = 0x1E,
    TTFIScmd_SimulateOverCurrentPlatformDTY_USB     = 0x1F,
    TTFIScmd_SimulateUndervoltagePlatformDTY_USB    = 0x20,
    TTFIScmd_GetConnectedHubs                       = 0x21,
    TTFIScmd_CyclicDiagnosis_Permanent              = 0x22,
    TTFIScmd_HistoryLastMsgsSend2Diag               = 0x23,
    TTFIScmd_HistoryLastMsgsSend2Diag_Clear         = 0x24,
    TTFIScmd_HistoryCallbacksPrm                    = 0x25,
    TTFIScmd_HistoryCallbacksPrm_Clear              = 0x26,
    TTFIScmd_HistoryCallbacksVolt                   = 0x27,
    TTFIScmd_HistoryCallbacksVolt_Clear              = 0x28,
    TTFIScmd_CheckAutomountDir                      = 0x29,
    TTFIScmd_ServiceUnavailable_Permanent           = 0x30,
    TTFIScmd_BlockMonitor_Permanent                 = 0x31
   ,TTFIScmd_HistoryStateTable                      = 0x32
#ifdef TTFISCMD_PRODUCTIONLINETEST
   ,TTFIScmd_ProductionLineTest                     = 0x33
#endif
#ifdef USE_HUB_CONNECTED_TO_HUB
   ,TTFIScmd_EnableHubConnectedToHub_Permanent      = 0x34
#endif
   ,TTFIScmd_SimulateCDDriveOverTemperature         = 0x35    //Roadmap_13035_CDDriveOvertemperature
   ,TTFIScmd_SimulateMixeModeCDRom_Mount            = 0x36,   //Roadmap_14002_MountCDRom (MixeMode)
    TTFIScmd_SimulateMixeModeCDRom_UnMount          = 0x37,   //Roadmap_14002_MountCDRom (MixeMode)
    TTFIScmd_TriggerPrmConfiguration                = 0x38,   //Roadmap_14003_Diagnosis2 (Control USB HW-Connectors)
    TTFIScmd_PRMMANAGER_vGetSDCardState             = 0x39,
    TTFiscmd_PRMMANAGER_bGetUSBHardwareConnector    = 0x40,   //Roadmap_14003_Diagnosis2 (Control USB HW-Connectors)
    TTFiscmd_PRMMANAGER_bSetUSBHardwareConnectorON  = 0x41,   //Roadmap_14003_Diagnosis2 (Control USB HW-Connectors)
    TTFiscmd_PRMMANAGER_bSetUSBHardwareConnectorOFF = 0x42,   //Roadmap_14003_Diagnosis2 (Control USB HW-Connectors)
    TTFiscmd_VoltageManager_vGetSystemVoltage       = 0x43,   //Roadmap undervoltage 
    TTFiscmd_Voltagemanager_vGetBoardVoltage        = 0x44,   //Roadmap undervoltage
    TTFiscmd_Voltagemanager_vSendVoltageWarning     = 0x45,   //Roadmap undervoltage - in gen3 drv_volt notifies critical voltage. This TTFIS test command triggers the same function of Statetable
    TTFiscmd_LinuxCmd_DbgAnalysis                   = 0x46,   //for testing analyse.exc calls it and could be used by testers to get more infor about platofrm in case of error
    //TTFiscmd_LinuxCmd_Reboot                        = 0x47,   //used for semiautomatic test to observe device detection at startup - see ttfis scripts in VD_DeviceManager/_test
    TTFIScmd_InternalHUB_Test                       = 0x48,   //PSA and RNAIVI use an internal HUB this TTFIS command is to trigger commands in usbutil.cpp to test communication
    TTFIScmd_BTABoxisAlive_Test                     = 0x49,   //PSA and RNAIVI use an BTA Telematics Box this TTFIS command is to Simuate the FC-ATP input.
    TTFIScmd_BTABoxDeviceSpeed_Test                 = 0x4A,   //PSA and RNAIVI use internal HUB this TTFIS command is used to simulate the connection speed and connecion status
    TTFIScmd_CDDAConnection                         = 0x4B,   //RNAIVI: simulate insert, eject connect_playable, remove_playable (for an inserted CDDA) to MediaPlayer. sr,sg added with ttfis parameter
    TTFIScmd_CDDAConnection_srsgWithUdev            = 0x4C,   //RNAIVI: simulate insert, eject connect_playable, remove_playable (for an inserted CDDA) to MediaPlayer  sr,sg gained from udev
    TTFIScmd_EjectOpticalDisc                       = 0x4D,   //RNAIVI: command to eject  an optical Disc
    TTFIScmd_OpticalDiscSlotStatus                  = 0x4E,   //RNAIVI: ttfis command to simulate input from VD_MediaManager i.e. slot status - this should be forwarded to Mediaplayer whihc has to forwar it to HMI
    TTFIScmd_GetConfigurationValue                  = 0x4F,   //PSA and RNAIVI: ttfis command to simulate the GetConfiguration input from outside world eg: Engineering Mode
    TTFIScmd_SetConfigurationValue                  = 0x50,   //PSA and RNAIVI: ttfis command to simulate the SetConfiguration input from outside world eg: Engineering Mode
    TTFIScmd_SimulateMalfunctionUSBConnector        = 0x51,   //RNAIVI: ttfis command to simulate the Malfunction of USB connectors
    TTFIScmd_HistoryHardwareMalfunction             = 0x52,   //Backtrace to see behaviour of HW malfunction (mechanism: 'termporary not available')
    TTFIScmd_HistoryHardwareMalfunction_Clear       = 0x53,    //clear Backtrace to see behaviour of HW malfunction (mechanism: 'termporary not available')
    TTFIScmd_ExternalHub_DetailsAdd                 = 0x54,    //Add ExternalHub to datapool
    TTFIScmd_ExternalHub_DetailsRemove              = 0x55,    //Remove ExternalHub from datapool
    TTFIScmd_ExternalHub_Print                      = 0x56,    //Print ExternalHub from datapool
    TTFIScmd_ExternalHub_Clear                      = 0x57,    //Clear all ExternalHub from datapool
    TTFIScmd_TelematicsBox_DetailsAdd               = 0x58,    //Add TelematicsBox to datapool
    TTFIScmd_TelematicsBox_DetailsRemove            = 0x59,    //Remove TelematicsBox from datapool
    TTFIScmd_TelematicsBox_Print                    = 0x60,    //Print TelematicsBox from datapool
    TTFIScmd_TelematicsBox_Clear                    = 0x61,    //Clear all TelematicsBox from datapool
    TTFIScmd_ShowGenericKDS_DeviceManager           = 0x62,    //show what project has written to address defined for generic KDS VD_DeviceManager
    TTFIScmd_ShowGenericKDS_Mediaplayer             = 0x63,    //show what project has written to address defined for generic KDS Mediaplayer
    TTFIScmd_ShowRNAIVIKDS_FEA_DvdMusicboxMediaplayer = 0x64,   //show what project has written to address defined for HW DVD used
    TTFIScmd_RootDaemon_Execute_Mkdir               = 0x65,   //Execute Mkdir root command
    TTFIScmd_RootDaemon_Execute_Cmd1                = 0x66,   //Execute root command
    TTFIScmd_RootDaemon_Execute_Cmd2                = 0x67   //Execute root command
}tenDevManagerTTFIScmd;


/*!
 * \enum tenDevManagerTTFIScmd_InterfaceConnectParm1
 *
 * \brief parameter one
 */
typedef enum
{
   TTFIScmd_InterfaceConnect_PRM        = 0x00, //not 2 has beebn formerly SPM
   TTFIScmd_InterfaceConnect_UDEV       = 0x02,
   TTFIScmd_InterfaceConnect_CCAclients = 0x03
}tenDevManagerTTFIScmd_InterfaceConnectParm1;

/*!
 * \enum tenDevManagerTTFIScmd_InterfaceConnectParm2
 *
 * \brief parameter two
 */
typedef enum
{
   TTFIScmd_InterfaceConnect_Disconnect = 0x00,
   TTFIScmd_InterfaceConnect_Connect    = 0x01
}tenDevManagerTTFIScmd_InterfaceConnectParm2;

/*!
 * \enum tenDevManagerTTFIScmd_Enable
 *
 * \brief Enable/disable
 */
typedef enum
{
   TTFIScmd_Disable   = 0x00,
   TTFIScmd_Enable    = 0x01
}tenDevManagerTTFIScmd_Enable;

/*!
 * \enum tenDevManagerTTFISdevice
 */
typedef enum
{
   TTFISdevice_sda   = 0x00,
   TTFISdevice_sda1  = 0x01,
   TTFISdevice_sdb   = 0x02,
   TTFISdevice_sdb1  = 0x03,
}tenDevManagerTTFISdevice;

/*-----------------------------------------------------------------------------*
 * Forward declarations                                                        *
 *-----------------------------------------------------------------------------*/
class PrmManager;
class StateTable;
class CUDevManager;


/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace statetbl;

/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/
/*!
 * \class TraceCmdManager
 *
 * This class is responsible for handling trace commands send from outside to
 * VD_devicemanager. This mechanism is used for test/debug purpose and to gather
 * VD_Devicemanager internal state.
 */
class TraceCmdManager
{
    private:

        /*! 
         * \brief Parameterized constructor
         *
         * \param [in] poMainApp main application
         */
        TraceCmdManager();

       /*! 
         * \brief Destructor
        */
        ~TraceCmdManager();

        // static member variables declared for singleton class
        static DVMLOCK          m_singelton;
        static TraceCmdManager *m_pTraceCmdManager;

        voltageFailure_HWSignalsAll_t m_signal;

        StateTable                         *m_pStateTableApp;

        OSAL_tIODescriptor m_TraceHandle;

    public:

        //variable used for test only
        static OSAL_tMSecond  m_Testdelay_ms;

        /*! 
         * \brief Get Trace command manager object
         *
         * Provides the singleton Trace command manager object. First invocation of
         * this function creates the object
         *
         * \return [TraceCmdManager*] singleton object
         */
        static TraceCmdManager * GetInstance();

        /*! 
         * \brief Destroys the singleton object
         */
        static void  DestroyInstance();

        /*!
         * \brief Assignment Operator
         *
         * Will not be implemented. Avoids Lint Prio 3 warning: Info 1732: new
         * in constructor for class 'TraceCmdManager' which has no assignment
         * operator.
         *
         * \note This is a technique to disable the assignment operator for this
         *       class. So if an attempt for the assignment is made compiler
         *       complains.
         */
         TraceCmdManager& operator=(const TraceCmdManager &oClientHandler);

         /*! 
         * \brief Initializes the object
         *
         * \return [int] initialization status OSAL_OK or OSAL_ERROR
         */
        int Initialize() const;

        /*!
         * \brief Sets the file to dump debug information to
         *
         * \param [in] poDumpFile the file pointer
         */
        void setDumpFile(FILE* poDumpFile);

        /*! 
         * \brief TTFiS command 'VD_DVM_SHOW_CONFIGURATION'
         *
         * For post analysis this command can be used to see version information
         * and to get informed about main defines set in Config.h further more
         * it shows if TTFIS commands have been used to switch of permantly
         * e.g. diagnosis etc.
         */
        tVoid TTFIScmdShowConfiguration(tVoid);

        /*! 
         * \brief TTFiS command 'VD_DVM_PRINT_CONFIGURATION'
         *
         * This command prints the current Configuration values.
         */
        tVoid TTFIScmdPrintConfiguration(tVoid);

        /*! 
         * \brief TTFiS command 'VD_DVM_GET_CONNECTED_DEVICES'
         *
         * For post analysis this command can be used to see which devices have
         * been notified to VD_DeviceManager. It helps to see if
         * VD_DEviceManager has announced all devices correctly to its clients.
         * If yes then root cause of a bug should be searched in upper layer
         * objects e.g. like FC_Mediaplayer
         */
        tVoid TTFIScmdCheckAutomountDir(tVoid);

        /*! 
         * \brief TTFiS command 'VD_DVM_GET_CONNECTED_DEVICES'
         *
         * For post analysis this command can be used to see what devices are
         * connected from the perspective of VD_DeviceManager. It helps to see
         * if there is an internal bug in the VD_DeviceManager
         *
         * \param [in] pU8Data
         */
        tVoid TTFIScmdGetConnectedDevices(tPCU8 pU8Data);

//========================================================================================================
//  Interface injection functions  - see IPrmManagerCard
//---------------------------------------------------------------------------------------------------------------------
         /*! 
           * \brief command to inject PrmManager interface
           *
           * injected interface is e.g. used to test commands of prm e.g. get SDCard info normaly triggered by diagnosis
           *
           */
        tVoid SetPrmManagerIF(IN IPrmManager *f_pIPrmManager);

        /*! 
           * \brief command to inject PrmManagerCard interface
           *
           * injected interface is e.g. used to test commands of prm e.g. get SDCard info normaly triggered by diagnosis
           *
           */
        tVoid SetPrmManagerCardIF(IN IPrmManagerCard *f_pIPrmManager);

        /*! 
          * \brief command to inject VoltageManager interface
          *
          * injected interface is e.g. used to test commands like getting actual voltage
          *
          */
        tVoid SetVoltageManagerIF(IN IVoltageManager *f_pIVoltageManager);

        /*! 
          * \brief command to inject IStateTableMount interface
          *
          * injected interface e.g. to simulate slotstate
          *
          */
        tVoid SetStateTableMountIF(IStateTableMount *f_pIStateTableMount);
//========================================================================================================

    protected:

        IPrmManager      *m_pIPrmManager;
        IPrmManagerCard  *m_pIPrmManagerCard;
        IVoltageManager  *m_pIVoltageManager;
        IStateTableMount *m_pIStateTableMount;

        /*! 
         * \brief opens a trace channel
         */
        tVoid TraceOpen(void);

        /*! 
         * \brief closes the trace channel
         */
        tVoid TraceClose(void);

        //static tVoid vHandleTraceCmd(tU8 const* const cpu8Buffer);
        static tVoid vChannelMgr (const tUChar* puchData);

        /*! 
         *  \brief vChannelMgr calls this function 
         *
         * \param [in] pU8Data array containing TTFIS cmd bytes
         */
        tVoid TTFIScmdHandler(tPCU8 pU8Data);

        /*! 
         * \brief Calls specific TTFiS command
         *
         * TTFIScmdHandler calls this function and it finds out which command
         * has been called and therefore calls the corresponding subfunction
         *
         * \param [in] pU8Data array containing TTFIS cmd byte
         */
        tVoid TTFIScmdHandler_cmd(tPCU8 pU8Data);

        /*! 
         * \brief TTFiS command 'VD_DVM_SHOW_HELP_MANUAL'
         *
         * It gives an overview about available ttfis commands
         */
        tVoid TTFIScmdShowHelpManual(tVoid) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_SIMULATE_PRM_SIGNAL_*'
        *
         * This offfers for test purpose to set certain prm related signals to
         * test the state machine
         *
         * \param [in] enSignal
        * \param [in] pU8Data contains which signals should be set how
        */
        tVoid TTFIScmdHandler_signal(tenDevManagerSignals enSignal, tPCU8 pU8Data);

        /*! 
         * \brief TTFiS command 'VD_DVM_SIMULATE_PRM_SIGNAL'
         *
         * This offers for test purpose to set all prm related signals at once
         * to test the state machine
        *
        * \param [in] pU8Data contains which signals should be set how
        */
        tVoid TTFIScmdHandler_signals(tPCU8 pU8Data);

        /*! 
         * \brief TTFiS command 'VD_DVM_GET_SIGNALS'
         *
         * Shows the current notified values of prm concerning HW signals like
         * OC and PPON
        */
        tVoid TTFIScmdHandler_getSignals(tVoid);

        /*! 
         * \brief TTFiS command 'VD_DVM_GET_LASTMSG_SENDTOCLIENTS'
         *
         * This command offers for test purpose to get the last message send to
         * a the clients. This function has been implmented early in the
         * development phase and we recommend to use
         * 'VD_DVM_HISTORY_MSGS_SEND2CLIENTS' instead.
         */
        tVoid TTFIScmdGetLastMsg2clients(tVoid) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_SET_TIMER_INTERVAL_sec' or
         *        'VD_DVM_SET_TIMER_INTERVAL_ms'
         *
         * This command offers for test purpose to change evaluation timer
         * interval of the state machine. Example if udev sends a mount or prm
         * over current states then the state table is triggers an execution
         * timer. After will have been expired the result of the state table
         * will be executed. By increasing the interval it is more easy to
         * simulate differnt inputs before the state machine is triggered to
         * deliver its result based on such inputs.
        *
        * \param [in] millisec interval time in milliseconds
        */
        tVoid TTFIScmdSetTimerInterval_ms(OSAL_tMSecond millisec);

        /*! 
         * \brief TTFiS command 'VD_DVM_SIMULATE_MSG2CLIENTS'
        *      
         * This command offers for test purpose to send any outputs offered by
         * the FI to the clients. In this manner, behaviour of clients could be
         * implemented before the state table has been fully implemented.
         *
         * \param [in] pU8Data contains details which input should be
         *             disconnected/connected from
        */
        tVoid TTFIScmdHandler_SimSendMsgtoClients(tPCU8 pU8Data);
        tVoid TTFIScmdSimUSBDevice(tPCU8 pU8Data) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_INTERFACE_CONNECT'
         *      
         * This command offers for test purpose to disconnect the state machine
         * from its different inputs one by one. In this manner it is possible
         * to disconnect from udev, prm or spm.
         *          *
         * \param [in] pU8Data contains details which input should be
         *             disconnected/connected from
         */
        tVoid TTFIScmdInterfaceConnect(tPCU8 pU8Data) const;

        tVoid TTFIScmdSendtesttime_ms(OSAL_tMSecond millisec) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_SETSERVICE'
         *      
         * This command 'VD_DVM_SETSERVICE' offers for test purpose to set the
         * service of VD_DeviceManager available or unavailable. This command
         * can be used to check the behaviour of clients e.g. if the service is
         * made available late after startup etc. Furthermore embedded in the
         * overall system it is possible to run tests made for VD_DeviceManager.
         * Resulting outputs of VD_DeviceManager are suppressed to keep the
         * overall system in its current state during such a test
         *      
         * \param [in] pU8Data contains details if service should be made
         *             available or unavailable
         */
        tVoid TTFIScmdSetServiceAvailable(tPCU8 pU8Data) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_SIMULATE_EOLVALUES_*'
         *
         * These set of commands change the EOL values forr test purpose. The
         * changes are not permanent and offer testing behaviour on changes of
         * EOL values without restarting the system after a change.
         *
         * \param [in] pU8Data contains details about EOL parameter one wants to manipulate
         */
        tVoid TTFIScmdSimEOLValues(tPCU8 pU8Data) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_TRIGGER_READ_EOLVALUES_FROMFILE'
         *
         * This command triggers reading EOL values from the system eol file.
         */
        tVoid  TTFIScmdTriggerReadEOLValuesFromFile(tVoid) const;


		 /*! 
         * \brief TTFiS command 'VD_DVM_TRIGGER_READ_KDSVALUES_FROMFILE'
         *
         * This command triggers reading EOL values from the system eol file.
         */
        tVoid  TTFIScmdTriggerReadKDSValuesFromFile(tVoid) const; //Roadmap 13001_ExtendedUserInformation

        /*! 
         * \brief TTFiS command 'VD_DVM_GET_CONNECTED_HUBS'
         *
         * This command is to find out if any HUBS are connected
         */
        tVoid TTFIScmdGetConnectedHubs(tVoid) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_CYCLIC_DIAG'
         *
         * This command is used for test and debug purpose: switch of cyclic
         * diagnosis (DTC-diagnosis) to analyse more easy device detection traces.
         *
         * \param [in] pU8Data contains parameter to enable/disable cyclic diagnosis
         */
        tVoid TTFIScmdCyclicDiagnosis(tPCU8 pU8Data) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_CYCLIC_DIAG_PERMANENT'
         *
         * This command is used for test and debug purpose: switch of cyclic
         * diagnosis (DTC-diagnosis) to analyse more easy device detection traces.
         * In addition to function 'TTFIScmdCyclicDiagnosis'this will be done
         * permanetly i.e. even after restart of the system cyclic diagnosis
         * will keep switched off.
         *
         * \param [in] pU8Data contains parameter to enable/disable cyclic diagnosis
         */
        tVoid TTFIScmdCyclicDiagnosis_Permanent(tPCU8 pU8Data) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_HISTORY_UDEV_EVTS_CLEAR'
         *
         * This command is used to clear the ringbugger of events received
         * from udev.
         */
        tVoid TTFIScmdHistoryUdevEvts_Clear(tVoid) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_HISTORY_CALLBACK_PRM_CLEAR'
         *
         * This command is used to clear the ringbugger of callbacks received
         * from PRM.
         */
        tVoid TTFIScmdHistoryCallbacksPrm_Clear(tVoid) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_HISTORY_CALLBACKS_VOLT_CLEAR'
         *
         * This command is used to clear the ringbugger of notifications
         * received from drv volt.
         */
        tVoid TTFIScmdHistoryCallbacksVolt_Clear(tVoid) const;
        
        /*! 
         * \brief TTFiS command 'VD_DVM_HISTORY_MSGS_SEND2DIAGNOSIS_ClEAR'
         *
         * This command is used to clear the ringbugger of messages sent to
         * diagnosis.
         */
        tVoid TTFIScmdHistoryLastMsgsSend2Diag_Clear(tVoid) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_HISTORY_MSGS_SEND2CLIENTS_CLEAR'
         *
         * This command is used to clear the ringbugger of messages sent to
         * clients.
         */
        tVoid TTFIScmdHistoryLastMsgsSend2Clients_Clear(tVoid) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_HISTORY_MSGS_SEND2DIAGNOSIS'
         *
         * For post analysis this command can be used to see what has been send
         * to diagnosis (DTC). Messages are available in a ring buffer of size
         * HELPER_ANAYLSIS_SEND2DIAGNOSIS see 'Config.h'
         */
        tVoid TTFIScmdHistoryLastMsgsSend2Diag(tVoid) const;
        
        /*! 
         * \brief TTFiS command 'VD_DVM_HISTORY_MSGS_SEND2CLIENTS'
         *
         * For post analysis this command can be used to see what has been send
         * to clients (e.g. FC_Mediaplayer). Messages are available in a ring
         * buffer of size HELPER_ANAYLSIS_SEND2CLIENTS see 'Config.h'
         */
        tVoid TTFIScmdHistoryLastMsgsSend2Clients(tVoid) const;
              
        /*! 
         * \brief TTFiS command 'VD_DVM_HISTORY_UDEV_EVTS'
         *
         * For post analysis ttfis command 'VD_DVM_HISTORY_UDEV_EVTS' can be
         * used to see what has been by udev to VD_DeviceManager (e.g. Apple
         * devices and MTP devices are notified this way). Messages are available
         * in a ring buffer of size HELPER_ANALYSIS_UDEVEVENTS see 'Config.h'.
         *        It helps to see if underlying platform objects cause an error.
         */
        tVoid TTFIScmdHistoryUdevEvts(tVoid) const;
                                   
        /*! 
         * \brief TTFiS command 'VD_DVM_HISTORY_CALLBACK_PRM'
         *
         * For post analysis this command can be used to see what callbacks have
         * been send by PRM
         */
        tVoid TTFIScmdHistoryCallbacksPrm(tVoid) const;
        
        /*! 
         * \brief TTFiS command 'VD_DVM_HISTORY_CALLBACKS_VOLT'
         *
         * For post analysis this command can be used to see what notifications
         * have been send by drv_volt
         */
        tVoid TTFIScmdHistoryCallbacksVolt(tVoid) const;

        /*! 
         * \brief TTFiS command 'VD_DVM_PERMANENT_SERVICEAVAILABLE_STARTUP'
         *
         * This offers for test purpose to set the service of VD_DeviceManager
         * available or unavailable at startup. Comand 'VD_DVM_SETSERVICE' can
         * be used to make the service of VD_DeviceManager available later to
         * test behaviour of clients.
         *
         * \param [in] pU8Data contains details if service should be made available
         *             or unavailable
         */
        tVoid TTFIScmdServiceUnavailable_Permanent(tPCU8 pU8Data) const; //for testing make service unavailable at startup

        ///@todo check if this is ununsed code
        tVoid TTFIScmdBlockMonitor_Permanent(tPCU8 pU8Data) const;

        /*! 
         * \brief TTFiS command 'TTFIScmdSimulateOverCurrentPlatformDTY_USB'
         *
         * This offers for test with connected devices to notify a overcurrent  transition to clients
         *
        */
        tVoid  TTFIScmdSimulateOverCurrentPlatformDTY_USB(tVoid) const;   

     /* \brief TTFiS command 'TTFIScmdSimulateUndervoltagePlatformDTY_USB'
         *
         * This offers for test with connected devices to notify a undervoltage transition to clients i.e. it simulates platform behaviour for VD_DVM
         *
        */
        tVoid  TTFIScmdSimulateUndervoltagePlatformDTY_USB(tVoid) const;

        /*! 
         *  \brief helper function to check if a parameter says disable/enable
         *
         * \param [in] u8Value value to be checked
         * 
         * \return [tBool] TRUE or FALSE
         */
        tBool mapConnectParm(tU8 u8Value) const;

        /*! 
         * \brief helper function to store a file to mark permanent states e.g.
         * used by 'VD_DVM_CYCLIC_DIAG_PERMANENT'
         *
         * \param [in] pU8Data   contains information if mechanism should be
         *             established permanently or not. If not the corresponding
         *             file will be deleted
         * \param [in] pFileName name of permanent file
         * \param [in] pFileText test which should be written to that file:
         *             normally it should contain a explanation
         * 
         * \return [tBool] TRUE or FALSE
         */
        tVoid permanentMechanismFileBased(tPCU8 pU8Data,tPCChar pFileName,tPCChar pFileText) const;
        
        tVoid TTFIScmdHistoryStateTable(tVoid) const;

#ifdef TTFISCMD_PRODUCTIONLINETEST

        /*! 
         * \brief TTFiS command 'VD_DVM_PRODUCTIONLINE_TEST'
         *
         * The so called PDT (production diagnosis test) is from outside using
         * diagnosis interface. To ease checking if implementation changes did
         * not 'destroy' functionality (regression test) this command is available
         */
        tVoid TTFIScmdProductionLineTest(tVoid) const;
#endif


#ifdef USE_HUB_CONNECTED_TO_HUB
         /*! 
             * \brief TTFiS command 'TTFIScmdEnableHubConnectedToHubPermanent'
             *
             * This command is used for development: e.g. in VAGMIBE the IPOD-USBCable
             * contains certification chip and acts as a HUB. To connect PC for trace and a device
             * this means to connect another HUB at this. HUB at HUB is normally not supported.
             * TTFIS VD_DVM_PERMANENT_HUBCONNECTEDHUB enables this. Such a command was used
             * to prevent that this feature is enabled by the customer.
             *
             * \param [in] pU8Data contains parameter to enable/disables this feature permanently
             */

         tVoid TTFIScmdEnableHubConnectedToHubPermanent(tPCU8 pU8Data) const;
#endif

         /*! 
             * \brief TTFiS command 'TTFIScmdSimulateCDDriveOverTemperature - VD_DVM_SIMULATE_CDDRIVE_OVERTEMPERATURE TRUE/FALSE'
             *
             * This command is used for development: to simulate that FC_HeatControl sends CDDriveOvertemperature (TRUE/FALSE)
             *
             * \param [in] pU8Data contains parameter to enable/disables this feature
             */

         tVoid TTFIScmdSimulateCDDriveOverTemperature(tPCU8 pU8Data) const;    //Roadmap_13035_CDDriveOvertemperature


        /*! 
             * \brief TTFiS command 'VD_DVM_SIMULATE_MIXMODE_CDROM_MOUNT    EMPTY/SR0/SR1'
             *
             * This command is used for development: to simulate that FC_Diagnosis want to Mount MixeMode CD (used in production line)
             *
             * \param [in] pU8Data contains parameter i.e. a number representing devicename of internal cdrom e.g. emtpy, sr0 or sr1
             */

        tVoid TTFIScmd_SimulateMixeModeCDRomMount(tPCU8 pU8Data) const;     //Roadmap_14002_MountCDRom (MixeMode)

       /*! 
             * \brief TTFiS command 'VD_DVM_SIMULATE_MIXMODE_CDROM_UNMOUNT    EMPTY/SR0/SR1'
             *
             * This command is used for development: to simulate that FC_Diagnosis want to Mount MixeMode CD (used in production line)
             *
             * \param [in] pU8Data contains parameter i.e. a number representing devicename of internal cdrom e.g. emtpy, sr0 or sr1
             */
        tVoid TTFIScmd_SimulateMixeModeCDRomUnMount(tPCU8 pU8Data) const; //Roadmap_14002_MountCDRom (MixeMode)


        /*! 
                 * \brief TTFiS command 'VD_DVM_SET_CONFIGURATION_TRIGGER_PRM'
                 *
                 * This command is used for development: after VD_DVM_SET_CONFIGURATION_VALUE_eCONF_ENABLE_HW_USBx DISABLE/ENABLE has been used
                 * VD_DVM_SET_CONFIGURATION_TRIGGER_PRM is used to trigger executes new demanded configuraton to enable/disable USBConnectors (USB Power)
                 *
                 * \param [in] pU8Data no extra paameter give by command
                 */
         tVoid TTFIScmdTriggerPrmConfiguration(tVoid) const; //Roadmap_14002_MountCDRom (MixeMode)

         /*! 
                  * \brief TTFiS command 'VD_DVM_TEST_PRMMANAGER_vGetSDCardState'
                  *
                  * This command is used for development for projects which have an internal SDCard reader in the target. It triggers getting corresponding platform information 
                  * about the card if connected. Normally this information is used for diagnosis features
                  *
                  * \param [in] pU8Data no extra parameter give by command
                  */
         tVoid TTFIScmd_PRMMANAGERvGetSDCardState(tVoid) const;
         
         /*! 
                   * \brief TTFiS command 'TTFiscmd_PRMMANAGERbGetUSBHardwareConnector'
                   *
                   * This command is used for development for projects which offer to switch onoff power of an usb connector. This command triggers to check the current setting
                   *
                   * \param [in] pU8Data : Connector i.e. USB1 or USB2
                   */
         tVoid TTFiscmd_PRMMANAGERbGetUSBHardwareConnector(tPCU8 pU8Data) const;

         /*! 
                   * \brief TTFiS command 'TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorON'
                   *
                   * This command is used for development for projects which offer to switch onoff power of an usb connector. This command triggers to switch on a certain connector 
                   *
                   * \param [in] pU8Data: Connector i.e. USB1 or USB2
                   */
         tVoid TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorON(tPCU8 pU8Data) const;

         /*! 
                   * \brief TTFiS command 'TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorOFF'
                   *
                   * This command is used for development for projects which offer to switch onoff power of an usb connector. This command triggers to switch off a certain
                   *
                   * \param [in] pU8Data: Connector i.e. USB1 or USB2
                   */
         tVoid TTFiscmd_PRMMANAGERbSetUSBHardwareConnectorOFF(tPCU8 pU8Data) const;

         /*! 
                   * \brief TTFiS command 'TTFiscmd_VoltageManagervGetSystemVoltage'
                   *
                   * This command is used for development to check if undrlying platform command works ok. System voltage means e.g. critical voltage low voltage, high critical etc.
                   *
                   */
                   
         tVoid TTFiscmd_VoltageManagervGetSystemVoltage(void) const;
         
         /*! 
                   * \brief TTFiS command 'TTFiscmd_VoltagemanagervGetBoardVoltage'
                   *
                   * This command is used for development to check if undrlying platform command works ok. Board voltage means e.g. 6500millivolt/12000millivolt
                   *
                   */
         tVoid TTFiscmd_VoltagemanagervGetBoardVoltage(void) const;

         /*! 
                   * \brief TTFiS command 'TTFiscmd_VoltagemanagervSendVoltageWarning'
                   *
                   * This command is used for development to check how VD_DeviceManager would react if ciritcal voltage or operation voltage is send
                   *
                   */
         tVoid TTFiscmd_VoltagemanagervSendVoltageWarning(tPCU8 pU8Data);


         
         /*! 
                   * \brief TTFiS command 'TTFiscmd_VoltagemanagerCmdDbgAnalysis'
                   *
                   * This command is used by script analyse.exc in order to find out in case of problems with device detection if root cause is e.g. caused by platform
                   *
                   */
         tVoid TTFiscmd_LinuxCmdDbgAnalysis(void) const;

         /*! 
		   * \brief TTFiS command 'VD_DVM_HUBINTERNAL_TEST'
		   *
		   * for targets with an internal HUB this is used to test corresponding function in usbutils
		   *
		   */
         tVoid TTFIScmd_InternalHUBTest(void) const;
         /*!
		   * \brief TTFiS command 'VD_DVM_SIMULATE_BTABOX_ISALIVE'
		   *
		   * for targets with an BTA Box(PSA for example) this is used to test the ITC information of BTA is send properly.
		   * if TRUE means the communication with the upper layer communication to BTA is alive
		   * Otherwise(FALSE) the BTA communication is not alive.
		   */
         tVoid TTFIScmd_BTABoxisAliveTest(tPCU8 pU8Data ) const;
         /*!
		   * \brief TTFiS command 'VD_DVM_SIMULATE_BTABOX_DEVICESPEED'
		   *
		   * for targets with an BTA Box(PSA for example) this is used to test the ITC information of BTA is send properly.
		   * It has 3 params. USB port number, connection status and device speed
		   * Connection status TRUE means device connected and device speed TRUE means highspeed device expected
		   *
		   */

         tVoid TTFIScmd_BTABoxDeviceSpeedTest(tPCU8 pU8Data) const;

         /*!
		   * \brief TTFiS command 'VD_DVM_SIMULATE_CDDA_PLATFORMNOTIFICATION'
		   *
		   * command is used to test how VD_DeviceManager reacts if platform would notify a CDDA. Sr and Sg values given by ttfis parameter
		   *
		   */

         tVoid TTFIScmdCDDAConnection(tPCU8 pU8Data) const;


         /*!
		   * \brief TTFiS command 'VD_DVM_SIMULATE_CDDA_PLATFORMNOTIFICATION'
		   *
		   * command is used to test how VD_DeviceManager reacts if platform would notify a CDDA. Sr and Sg values gained from udev
		   *
		   */
         tVoid TTFIScmdCDDAConnectionSrSgWithUdev(tPCU8 pU8Data) const;


         /*!
		   * \brief TTFiS command 'VD_DVM_EJECT_OPTICALDISC'
		   *
		   * command to trigger VD_DVM to send eject command to VD_MMGR
		   *
		   */
         tVoid TTFIScmdEjectOpticalDisc(tPCU8 pU8Data) const;

          /*!
            * \brief TTFiS simulate slot status opticl disc to VD_DeviceManager. 
            */
          tVoid TTFIScmdOpticalDiscSlotStatus(tPCU8 pU8Data) const;

          /*!
            * \brief TTFiS simulate the Get Configuration Value from outside world eg: Engineering Mode to VD_DeviceManager.
            */
          tVoid TTFIScmdGetConfigurationValue(tPCU8 pU8Data) const;

          /*!
            * \brief TTFiS simulate the Set Configuration Value from outside world eg: Engineering Mode to VD_DeviceManager.
            */
          tVoid TTFIScmdSetConfigurationValue(tPCU8 pU8Data) const;

          /*!
            * \brief TTFiS simulate the Malfunction of USB connector.
            */
          tVoid TTFIScmdSimulateMalfunctionUSBConnector(tPCU8 pU8Data) const;

         /*!
            * \brief TTFiS command to get a backtrace of mechanism HWMalfunction ('temporary not available')
            */
         tVoid  TTFIScmdHistoryHardwareMalfunction() const;

         
         /*!
            * \brief TTFiS command to get a backtrace of mechanism HWMalfunction ('temporary not available')
            */
         tVoid  TTFIScmdHistoryHardwareMalfunction_Clear() const;

         /*!
            * \brief TTFiS command to simulate addition/update of external hub
            */
         tVoid  TTFIScmdExternalHubDetailsAdd(tPCU8 pU8Data) const;

         /*!
            * \brief TTFiS command to simulate removal of external hub
            */
         tVoid  TTFIScmdExternalHubDetailsRemove(tPCU8 pU8Data) const;

         /*!
            * \brief TTFiS command to PRINT external hub
            */
         tVoid  TTFIScmdExternalHubPrint() const;

         /*!
            * \brief TTFiS command to Clear all external hub
            */
         tVoid  TTFIScmdExternalHubDPClear() const;

         /*!
            * \brief TTFiS command to simulate addition/update of telematics box device
            */
         tVoid  TTFIScmdTelematicsBoxDetailsAdd(tPCU8 pU8Data) const;

         /*!
            * \brief TTFiS command to simulate removal of telematics box device
            */
         tVoid  TTFIScmdTelematicsBoxDetailsRemove(tPCU8 pU8Data) const;

         /*!
            * \brief TTFiS command to PRINT external hub
            */
         tVoid  TTFIScmdTelematicsBoxPrint() const;

         /*!
            * \brief TTFiS command to Clear all external hub
            */
         tVoid  TTFIScmdTelematicsBoxDPClear() const;


		/*!
		 * \brief TTFiS command to show generic kds devicemanager value and print this - just reading for debug purpose 
		 */
			 
		tVoid  TTFIScmdShowGenericKDSDeviceManager() const;


			/*!
		 * \brief TTFiS command to  show generic kds value and print this - just reading for debug purpose does not 
		     Note: Care for changing related address if change is done in Mediaplayer
		     Currently we expect this to be stable
		 */
			 
		tVoid  TTFIScmdShowGenericKDSMediaplayer() const;


		/*!
		 * \brief TTFiS command to  show kds value relevant for Music Box feature or project RNAIVI and print this - just reading for debug purpose does not 
		     Note: Care for changing related address if change is done in Mediaplayer
		     Currently we expect this to be stable
		 */
			 
		tVoid  TTFIScmdShowRNAIVIKDS_FEA_DvdMusicboxMediaplayer() const;

		/*!
         * \brief TTFiS command to  executre rootdaemon root command Mkdir
         */

        tVoid  TTFIScmdRootDaemonExecute_Mkdir() const;

        /*!
         * \brief TTFiS command to  executre rootdaemon root command Cmd1
         */

        tVoid  TTFIScmdRootDaemonExecute_Cmd1() const;

        /*!
         * \brief TTFiS command to  executre rootdaemon root command Cmd2
         */

        tVoid  TTFIScmdRootDaemonExecute_Cmd2() const;

         FILE *m_poDumpFile;  ///< file to dump traces to
};

#ifdef __cplusplus
}
#endif

#endif //__TRACE_CMD_MANAGER_H__

////////////////////////////////////////////////////////////////////////////////
// <EOF>
