/*-----------------------------------------------------------------------------*
 * ServiceHistory.cpp                                                          *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file ServiceHistory.cpp
 *
 * \brief This file holds the implementation for CServiceHistory
 *
 * \version 02.08.2012, Negi, Sunder (MontaVista), initial version
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 13.08.2012, Negi, Sunder (MontaVista), cleanup logging to usb (WRITEHISTORYTOUSBSTICK)
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "debug/ObjectHistory.h"
#include "debug/ServiceHistory.h"
#include "config/ConfigurationManager.h" //for eclapsed time

//#include <stdio.h>antilint: repeatedly included but does not have a standard include guard


/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_HISTORYMANAGER
#include "trcGenProj/Header/ServiceHistory.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
CServiceHistory::CServiceHistory()
    :CObjectHistory() 
{
    CServiceHistory::ClearHistory ();

}//lint !e1566  
//Warning 1566: prio3: member 'CServiceHistory::m_currIndex'       might have been initialized by a separate function but no '-sem(CServiceHistory::ClearHistory,initializer)' was seen
//Warning 1566: prio3: member 'CServiceHistory::m_iSendNo' might have been initialized by a separate function but no '-sem(CServiceHistory::ClearHistory,initializer)' was seen
//see function ClearHistory

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
CServiceHistory::~CServiceHistory()
{
    // Do nothing
}

/*-----------------------------------------------------------------------------*
 * void ShowLastMsgSend2Clients ()                                             *
 *-----------------------------------------------------------------------------*/
void CServiceHistory::ShowLastMsgSend2Clients()
{
    ETG_TRACE_USR3 (("ShowLastMsgSend2Clients: Begin"));

    if (m_currIndex)
    {
        CDevice& l_oDevice = m_aoLastNotifiedDevices[m_currIndex-1];

        ETG_TRACE_FATAL (("[ok] ***************/ShowLastMsgSend2Clients(eUpdateClients)**********"));

        ETG_TRACE_FATAL (("[ok] Type Of Notify: %i", ETG_ENUM(NOTIFICATION_Type, l_oDevice.m_eTypeOfNotify) )); //e.g. third notification since startup

        ETG_TRACE_FATAL (("[ok] Serial ID     : %s",
                l_oDevice.m_cSerialID.toStdString().c_str()));

        ETG_TRACE_FATAL (("[ok] Send Time (ms): %d", l_oDevice.m_rSendTime_ms));

        ETG_TRACE_FATAL (("[ok] USB Port      : %s",
                l_oDevice.m_cUSBPort.toStdString().c_str()));
        ETG_TRACE_FATAL (("[ok] Device Type   : %d", ETG_ENUM(DVM_DEVICE_TYPE, l_oDevice.m_eDeviceType)));
        ETG_TRACE_FATAL (("[ok] Device Version: %s",
                l_oDevice.m_cDeviceVersion.toStdString().c_str()));
        ETG_TRACE_FATAL (("[ok] Device Name   : %s",
                l_oDevice.m_cDeviceName.toStdString().c_str()));
        ETG_TRACE_FATAL (("[ok] Acessory Name : %s",
                l_oDevice.m_cAccessoryName.toStdString().c_str()));
        ETG_TRACE_FATAL (("[ok] Mount Point   : %s",
                l_oDevice.m_cMountPoint.toStdString().c_str()));
        ETG_TRACE_FATAL (("[ok] Connect Status: %d",
                ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE, l_oDevice.m_eConnectStatus)));
        ETG_TRACE_FATAL (("[ok] ***************ShowLastMsgSend2Clients\**************************"));
        ETG_TRACE_FATAL (("[ok] Manufacturer  : %s", l_oDevice.m_cManufacturer.toStdString().c_str()));
        ETG_TRACE_FATAL (("[ok] iVendorID     : 0x%04x", l_oDevice.m_iVendorID));
        ETG_TRACE_FATAL (("[ok] iProductID    : 0x%04x", l_oDevice.m_iProductID));
        ETG_TRACE_FATAL (("[ok] u64Size [KB]  : %u", (tU32)(l_oDevice.m_u64Size/1024) ));
    }
    else
    {
        ETG_TRACE_FATAL (("[ok] ***************/ShowLastMsgSend2Clients(eUpdateClients)**********"));
        ETG_TRACE_FATAL (("[ok]        No message send to clients yet                           "));
        ETG_TRACE_FATAL (("[ok] ***************ShowLastMsgSend2Clients\**************************"));
    }

    ETG_TRACE_USR3 (("ShowLastMsgSend2Clients: End"));
}

/*-----------------------------------------------------------------------------*
 * void AddToHistory (CDevice f_oDevice)                                       *
 *-----------------------------------------------------------------------------*/
void CServiceHistory::AddToHistory (CDevice& f_oDevice)
{
    m_LockHistory.lock();

    f_oDevice.m_iSendNo = ++m_iSendNo;

    m_aoLastNotifiedDevices[m_currIndex].FillDevice (f_oDevice);

    m_currIndex++;

    if (m_currIndex >= HELPER_ANAYLSIS_SEND2CLIENTS)
    {
        m_currIndex = 0;
    }

    m_LockHistory.unlock();
}

/*-----------------------------------------------------------------------------*
 * void TraceHistory ()                                                        *
 *-----------------------------------------------------------------------------*/
void CServiceHistory::TraceHistory()
{
    ETG_TRACE_USR3 (("TraceHistory: Begin"));
    ETG_TRACE_FATAL (("[ok] ===============================->Service==============================="));

    tU64 u64StartTimeAbsoluteDeviceManager = ConfigurationManager::GetInstance()->GetStartTime_ms();
    ETG_TRACE_U64_FATAL((char*)"u64StartTimeAbsoluteDeviceManager (ms) : ",u64StartTimeAbsoluteDeviceManager);
    ETG_TRACE_U64_FATAL((char*)"u64StartTimeAbsoluteDeviceManager (sec): ",u64StartTimeAbsoluteDeviceManager/1000);

    // dump to file, if available
    if (NULL != m_poDumpFile)
    {
        fprintf (m_poDumpFile,
                "\n[ok] ===============================->Service===============================");
    }

    tBool bUsed = FALSE;

    for (int itr = 0;itr < HELPER_ANAYLSIS_SEND2CLIENTS;itr++)
    {
        if (-1 != m_aoLastNotifiedDevices[itr].m_iSendNo)
        {
            bUsed = TRUE;
            ETG_TRACE_FATAL (("[ok] ***************/TraceStoredLastMsgs2Clients [%2d]*******",itr));
            ETG_TRACE_FATAL (("[ok] [%2d]: Send No       : %d",
                    itr, m_aoLastNotifiedDevices[itr].m_iSendNo)); //e.g. third notification since startup
            ETG_TRACE_FATAL (("[ok] [%2d]: Send Time (ms): %d ms",
                    itr, m_aoLastNotifiedDevices[itr].m_rSendTime_ms));
            ETG_TRACE_FATAL (("[ok] [%2d]: Type Of Notify: %i", itr, ETG_ENUM(NOTIFICATION_Type, m_aoLastNotifiedDevices[itr].m_eTypeOfNotify) ));

            ETG_TRACE_FATAL (("[ok] [%2d]: Serial ID     : %s",
                    itr, m_aoLastNotifiedDevices[itr].m_cSerialID.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: USB Port      : %s",
                    itr, m_aoLastNotifiedDevices[itr].m_cUSBPort.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Device Type   : %d", itr, ETG_ENUM(DVM_DEVICE_TYPE, m_aoLastNotifiedDevices[itr].m_eDeviceType)));
            ETG_TRACE_FATAL (("[ok] [%2d]: Device Version: %s",
                    itr, m_aoLastNotifiedDevices[itr].m_cDeviceVersion.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Device Name   : %s",
                    itr, m_aoLastNotifiedDevices[itr].m_cDeviceName.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Acessory Name : %s",
                    itr, m_aoLastNotifiedDevices[itr].m_cAccessoryName.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Mount Point   : %s",
                    itr, m_aoLastNotifiedDevices[itr].m_cMountPoint.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: Connect Status: %d",
                    itr, ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,
                            m_aoLastNotifiedDevices[itr].m_eConnectStatus)));
            ETG_TRACE_FATAL (("[ok] [%2d]: Manufacturer  : %s", itr, m_aoLastNotifiedDevices[itr].m_cManufacturer.toStdString().c_str()));
            ETG_TRACE_FATAL (("[ok] [%2d]: iVendorID     : 0x%04x", itr, m_aoLastNotifiedDevices[itr].m_iVendorID));
            ETG_TRACE_FATAL (("[ok] [%2d]: iProductID    : 0x%04x", itr, m_aoLastNotifiedDevices[itr].m_iProductID));
            ETG_TRACE_FATAL (("[ok] [%2d]: u64Size [KB]  : %u", itr, (tU32)(m_aoLastNotifiedDevices[itr].m_u64Size/1024) ));

            ETG_TRACE_FATAL (("[ok] ***************TraceStoredLastMsgs2Clients [%2d]************************",itr));
            ETG_TRACE_FATAL (("[ok]                                                                         "));

            // dump to file, if available
            if (NULL != m_poDumpFile)
            {
                fprintf (m_poDumpFile, "\n[ok] ***************/TraceStoredLastMsgs2Clients [%2d]*******",
                        itr);
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Send No       : %d",
                        itr, m_aoLastNotifiedDevices[itr].m_iSendNo); //e.g. third notification since startup
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Send Time (ms): %u ms",
                        itr, (tUInt)m_aoLastNotifiedDevices[itr].m_rSendTime_ms);
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Type Of Notify: %i", itr, ETG_ENUM(NOTIFICATION_Type, m_aoLastNotifiedDevices[itr].m_eTypeOfNotify) );

                fprintf (m_poDumpFile, "\n[ok] [%2d]: Serial ID     : %s",
                      itr, m_aoLastNotifiedDevices[itr].m_cSerialID.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: USB Port      : %s",
                        itr, m_aoLastNotifiedDevices[itr].m_cUSBPort.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Device Type   : %d", itr, ETG_ENUM(DVM_DEVICE_TYPE, m_aoLastNotifiedDevices[itr].m_eDeviceType));
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Device Version: %s",
                        itr, m_aoLastNotifiedDevices[itr].m_cDeviceVersion.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Device Name   : %s",
                        itr, m_aoLastNotifiedDevices[itr].m_cDeviceName.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Acessory Name : %s",
                        itr, m_aoLastNotifiedDevices[itr].m_cAccessoryName.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Mount Point   : %s",
                        itr, m_aoLastNotifiedDevices[itr].m_cMountPoint.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Connect Status: %d",
                        itr, ETG_ENUM(DEVICE_CONNECTSTATUS_TYPE,
                                m_aoLastNotifiedDevices[itr].m_eConnectStatus));
                fprintf (m_poDumpFile, "\n[ok] [%2d]: Manufacturer  : %s", itr, m_aoLastNotifiedDevices[itr].m_cManufacturer.toStdString().c_str());
                fprintf (m_poDumpFile, "\n[ok]  [%2d]: iVendorID    : 0x%04x", itr, m_aoLastNotifiedDevices[itr].m_iVendorID);
                fprintf (m_poDumpFile, "\n[ok]  [%2d]: iProductID   : 0x%04x", itr, m_aoLastNotifiedDevices[itr].m_iProductID);
                fprintf (m_poDumpFile, "\n[ok] [%2d]: u64Size [KB]  : %u", itr, (tUInt)(m_aoLastNotifiedDevices[itr].m_u64Size/1024) );
                fprintf (m_poDumpFile, "\n[ok] ***************TraceStoredLastMsgs2Clients [%2d]************************",
                        itr);
                fprintf (m_poDumpFile, "\n[ok]                                                                         ");
            }
        }
    }

    if (FALSE == bUsed)
    {
        ETG_TRACE_FATAL (("[ok] ***************/TraceStoredLastMsgs2Clients ***********************"));
        ETG_TRACE_FATAL (("[ok]        No message send to clients yet                              "));
        ETG_TRACE_FATAL (("[ok] ***************TraceStoredLastMsgs2Clients ************************"));

        // dump to file, if available
        if (NULL != m_poDumpFile)
        {
            fprintf (m_poDumpFile, "\n[ok] ***************/TraceStoredLastMsgs2Clients ***********************");
            fprintf (m_poDumpFile, "\n[ok]        No message send to clients yet                              ");
            fprintf (m_poDumpFile, "\n[ok] ***************TraceStoredLastMsgs2Clients ************************");
        }
    }

    ETG_TRACE_FATAL (("[ok] ================================Service<-=============================="));

    // dump to file, if available
    if (NULL != m_poDumpFile)
    {
        fprintf (m_poDumpFile,
                "\n[ok] ================================Service<-==============================");
    }

    ETG_TRACE_USR3 (("TraceHistory: End"));
}

/*-----------------------------------------------------------------------------*
 * void ClearHistory ()                                                        *
 *-----------------------------------------------------------------------------*/
void CServiceHistory::ClearHistory()
{
    for (int itr = 0; itr < HELPER_ANAYLSIS_SEND2CLIENTS; itr++)
    {
    //list for debugging

        m_aoLastNotifiedDevices[itr].m_rSendTime_ms   =  0;
        m_aoLastNotifiedDevices[itr].m_eTypeOfNotify  = CGlobalEnumerations::enNotifyAll;
        m_aoLastNotifiedDevices[itr].m_iSendNo        = -1;
        m_aoLastNotifiedDevices[itr].m_cSerialID      = "";
        m_aoLastNotifiedDevices[itr].m_cUSBPort       = "";
        m_aoLastNotifiedDevices[itr].m_eDeviceType    = CGlobalEnumerations::DTY_UNKNOWN;
        m_aoLastNotifiedDevices[itr].m_cDeviceVersion = "";
        m_aoLastNotifiedDevices[itr].m_cDeviceName    = "";
        m_aoLastNotifiedDevices[itr].m_cAccessoryName = "";
        m_aoLastNotifiedDevices[itr].m_eConnectStatus = USB_DEV_UNDEFINED;
        m_aoLastNotifiedDevices[itr].m_cManufacturer  = "";
        m_aoLastNotifiedDevices[itr].m_iVendorID      = 0;
        m_aoLastNotifiedDevices[itr].m_iProductID     = 0;
        m_aoLastNotifiedDevices[itr].m_u64Size        = 0;
    }

    m_iSendNo   = 0;
    m_currIndex = 0;
}   //lint !e1565 Warning 1565: member 'CObjectHistory::m_poDumpFile' (VD_DeviceManager/inc/debug/ObjectHistory.h) not assigned by initializer function

////////////////////////////////////////////////////////////////////////////////
// <EOF>
