/*-----------------------------------------------------------------------------*
 * HistoryManager.h                                                            *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file HistoryManager.H
 *
 * \brief This file holds the declaration for CHistoryManager
 *
 * \version 02.08.2012, Negi, Sunder (MontaVista), initial version
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 13.08.2012, Negi, Sunder (MontaVista), changes to dump traces to file
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

#ifndef HISTORYMANAGER_H_
#define HISTORYMANAGER_H_

/*-----------------------------------------------------------------------------*
 * namespace - Enums.h has to be included first                                                                    *
 *-----------------------------------------------------------------------------*/
using namespace dvmDiaglog;

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "debug/ObjectHistory.h"
#include "debug/DeviceProcessorHistory.h"
#include "debug/ServiceHistory.h"
#include "debug/InterfaceHistory.h"
#include "debug/PrmHistory.h"
#include "debug/VoltClientHistory.h"
#include "debug/DiagClientHistory.h"
#include "debug/HWMalfunctionHistory.h"
#include "debug/HistoryManager.h"

/*-----------------------------------------------------------------------------*
 * Defines                                                                     *
 *-----------------------------------------------------------------------------*/
#define HISTORYMANAGER         (CHistoryManager::GetInstance())
#define DEVICEPROCESSORHISTORY (CHistoryManager::GetInstance()->getDevHistory())
#define PRMHISTORY             (CHistoryManager::GetInstance()->getPrmHistory())
#define SERVICEHISTORY         (CHistoryManager::GetInstance()->getServiceHistory()) //depricated replaced by INTERFACEHISTORY
#define INTERFACEHISTORY       (CHistoryManager::GetInstance()->getInterfaceHistory())
#define VOLTCLIENTHISTORY      (CHistoryManager::GetInstance()->getVoltClientHistory())
#define DIAGCLIENTHISTORY      (CHistoryManager::GetInstance()->getDiagClientHistory())
#define HWMALFUNCTIONHISTORY   (CHistoryManager::GetInstance()->getHwMalfunctionHistory()) //mechanism temporary not available


/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/
/*!
 * \class CHistoryManager
 *
 * \brief class to manage the debug information for all the objects
 */
class CHistoryManager
{

    public: // -> lint: Warning 1536: prio1: Exposing low access member 'CHistoryManager::m_oxxxHistory'

        
        CDeviceProcessorHistory m_oDevHistory;
        CServiceHistory         m_oServiceHistory;
        InterfaceHistory        m_oInterfaceHistory;
        CPrmHistory             m_oPrmHistory;
        VoltClientHistory       m_oVoltClientHistory;
        CDiagClientHistory      m_oDiagClientHistory;
        HWMalfunctionHistory    m_oHWMalfunctionHistory;
    private:

        static CHistoryManager *m_poHistoryManager;
        static DVMLOCK          m_singelton;

    
        FILE *m_poDumpFile;
        char *m_cLogFile;
        char *m_cPlatformLogfile;

        /*!
         * \brief private constructor
         */
        CHistoryManager();

        /*!
         * \brief private destructor
         */
        ~CHistoryManager();

    public:

        /*!
         * \brief Get History manager object
         *
         * Provides the singleton History manager object. First invocation of
         * this function creates the object
         *
         * \return [CHistoryManager*] singleton object
         */
        static CHistoryManager *GetInstance();

        /*!
         * \brief Destroys the singleton object
         */
        static void DestroyInstance();

        /*!
         * \brief getter function for DeviceProcessor history member
         *
         * \return [CDeviceProcessorHistory&] the DeviceProcessor history object
         */
        CDeviceProcessorHistory& getDevHistory ();

        /*!
         * \brief getter function for Service history member (deprecated)
         *
         * \return [CServiceHistory&] the Service history object
         */
        CServiceHistory& getServiceHistory ();

    /*!
         * \brief getter function for Interface history member (will replace getServiceHistory)
         *
         * \return [CServiceHistory&] the Service history object
         */
        InterfaceHistory& getInterfaceHistory ();

        /*!
         * \brief getter function for Prm history member
         *
         * \return [CPrmHistory&] the Prm history object
         */
        CPrmHistory& getPrmHistory ();

        /*!
         * \brief getter function for voltage client handler history member
         *
         * \return [VoltClientHistory&] the Volt client handler history object
         */
        VoltClientHistory& getVoltClientHistory();

        /*!
         * \brief getter function for Diagnosis client handler history member
         *
         * \return [CDiagClientHistory&] the Diagnosis client handler object
         */
        CDiagClientHistory& getDiagClientHistory();

        /*!
         * \brief getter function for HW Malfunction  client handler history member
         *
         * \return [HWMalfunctionHistory&] the HWMalfunction client handler object
         */

        HWMalfunctionHistory& getHwMalfunctionHistory();


        /*!
         * \brief Sets the file to dump debug information to
         *
         * \param [in] poDumpFile the file pointer
         */
        void setDumpFile(FILE* poDumpFile);

        /*!
         * \brief Clears the dump file
         */
        void ClearDumpFile();

        /*!
         * \brief Stores platform traces to usb
         *
         * Uses the member m_cPlatfromDumpFile as the file to dump information to
         *
         * \return [int] error status
         */
        int DumpPlatformHistoryToUSBStick() const;

        /*!
         * \brief Stores traces to usb
         *
         * Uses the member m_cDumpFile the file to dump information to
         *
         * \return [int] error status
         */
        int DumpHistoryToUSBStick();

        /*!
         * \brief Initialize dumping to usb
         *
         * \param [in] f_cUSBMountPoint the usb mountpoint
         *
         * \return [int] error status
         */
        int InitDumpHistoryToUSBStick(const char *f_cUSBMountPoint);

        /*!
         * \brief Stop dumping to usb
         */
        void StopDumpHistoryToUSBStick();
};

#endif // HISTORYMANAGER_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
