/*-----------------------------------------------------------------------------*
 * StateTable.h                                                                *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file StateTable.h
 *
 * \brief This file holds the declaration for StateTable which uses inputs of UdevManger and prm to evaluate a result to be send to the
 *        clients.
 *
 *-----------------------------------------------------------------
 *                                   development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch)
 *         -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

#ifndef __STATE_TABLE_H__
#define __STATE_TABLE_H__

//default values
#define  OBSERVATION_INTERVAL "TIMER_INTERVAL_STATETABLE"
#define  STATETABLE_TIME_INTERVAL_DEFAULT 501 /*ms*/
#define  NUMOFCONNECTORS (eUSB2+1)
#define  MAXMSGSINQ 25
#define  STATETABLEQ "PrmMgrQ"
#define  STATETABLEQUDEV "PrmMgrQU"

/*-----------------------------------------------------------------------------*
 * includes                                                         *
 *-----------------------------------------------------------------------------*/
#include "IStateTableMount.h"
#include "IStateTableVoltage.h"


#include "IStateTableHWMalfunction.h"
#include "IStateTableHWSignals.h"




//Note. Possible todo:  if we replace this by forward declaration a lot of dependencies can be seen e.g. diaglib. diaglog etc.
//since in the meantime pure interfaces have been added it owuld be possible to decrease number of includes for those entities
#include "Timer.h"
#include "TimerDVM.h"
#include "MessageQueue.h"
#include "MessageQueueDVM.h"

/*-----------------------------------------------------------------------------*
 * Forward declarations                                                        *
 *-----------------------------------------------------------------------------*/
class IPrmManagerCard;
class CUDevManager;

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace statetbl;

/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/
/*!
 * \class StateTable
 */
class StateTable : public IStateTableVoltage,
                       public IStateTableMount,
                       public IStateTableHWMalfunction,
                       public IStateTableHWSignals,
                       public TFThread
{
    public:


    typedef enum
    {
       eThread_StateTable = 0,
       eThread_Last       = 1
    }tenThreadFunction;


        /*!
         * \brief Get StateTable object
         *
         * Provides the singleton StateTable object. First invocation of
         * this function creates the object
         *
         * \return [StateTable*] singleton object
         */

        static StateTable* GetInstance();


        /*!
         * \brief Initializes the object
         *
         * \return [int] initialization status OSAL_OK or OSAL_ERROR
         */
        int Initialize();

        /*!
         * \brief Destroys the singleton object
         */
        static void  DestroyInstance();

        /*!
         * \brief Assignment Operator
         *
         * Will not be implemented. Avoids Lint Prio 3 warning: Info 1732: new
         * in constructor for class 'StateTable' which has no assignment
         * operator.
         *
         * \note This is a technique to disable the assignment operator for this
         *       class. So if an attempt for the assignment is made compiler
         *       complains.
         */
        StateTable& operator=(const StateTable &oClientHandler);


        /**
        * This function derives from TFThread and starts a user function in a thread from the thread factory's thread pool.
        * After leaving while loop used thread is stopped and released for other users
        *
        * @return void
        */
        virtual void Do(int functionID, void *ptr);


        /*!
         * \brief send message to clients
         *
         * This function uses _send2Clients
         *
         * \param [in] usb    send message to clients connected to connector usb
         * \param [in] result result of state table to be send to clients at
         *             connector usb
         * \param [in] bTest  TRUE: triggered by TTFIS test command,
         *             FALSE: normal execution mode
         *
         * \see _send2Clients
         */
        tVoid send2Clients(tenDevManagerUSBHost usb,
                const voltageFailure_Result_t& result,tBool bTest=FALSE);

        /*!
         * \brief send message to clients
         *
         * This function is used as subfunction by send2Clients
         *
         * \param [in] usb    send message to clients connected to connector usb
         * \param [in] result result of state table to be send to clients at
         *             connector usb
         * \param [in] bTest  TRUE: triggered by TTFIS test command,
         *             FALSE: normal execution mode
         */
        tVoid _send2Clients(tenDevManagerUSBHost usb,
                const voltageFailure_Result_t& result,tBool bTest=FALSE) const;

        /*!
         * \brief send test message to clients
         *
         * Triggered by TTFIS command: e.g. VD_DVM_SIMULATE_MSG2CLIENTS. The
         * function uses function send2Clients
         *
         * \param [in] usb               send message to clients connected to
         *             connector usb
         * \param [in] eConnectionStatus message according to the FI
         *
         * \see _send2Clients
         */
        tVoid send2ClientsTest(tenDevManagerUSBHost usb,
                DEVICE_CONNECTSTATUS_Type  eConnectionStatus);


        /*!
         * \brief see IStateTableHWSignals
         */
        virtual tVoid getLastMsgSendtoClients(tenDevManagerUSBHost usb,
                voltageFailure_Result_t *pMessage) const;


        /*!
         * \brief Simulates signals
         *
         * In order to simulate different HW-signals (OC,PPON - i.e.
         * over-current signalling) this functions is triggered by the
         * TraceCmdManager i.e. by a TTFIS command
         *
         * \param [in] signal contains the signal i.e. all information which
         *              would be normally send by prm
         *
         * \see TraceCmdManager
         */
        tVoid sendSignalsTest( const voltageFailure_HWSignalsAll_t& signal);

        /*!
         * This function derives from the available inputs and the existing
         * state tables m_voltageFailureStateTable_USB1&2 the corresponding result
         *
         * \param [in]  usb     connector i.e. usb1 or usb2 ...
         * \param [out] pResult value will be filled for caller with latest result
         */
        tVoid getResult(tenDevManagerUSBHost usb,voltageFailure_Result_t *pResult);

        tVoid CalcStateTableResult(IN tenDevManagerUSBHost usb,IN voltageFailure_HWSignalsAll_t &oCurrSignal,OUT voltageFailure_Result_t *pResult);

        /*!
         *  \brief Sets the interval time
         *
         *  A ttfis cmd can be used to change the timer interval. I.e. after
         *  expiration of an interval the result for clients is calculated for
         *  purpose of test it is possible to adjust this to observe it in a
         *  better way.
         *
         * \param [in] u32IntervalTimer time in milliseconds
         */
        tVoid setIntervalTime(tU32 u32IntervalTimer);

        /*!
         *  \brief update IntervalTimeFromConfiguration
         *
         *  A ttfis cmd can be used to change the timer  time in the configuration manager.
         *  To use the new time this command has to be called
         *
         */
        tVoid updateIntervalTimeFromConfiguration();


        /*!
         * \brief check to start the trigger timer. I.e. each time if a signal
         *        is changed this timer is started
         *
         * \param [in] isDeviceConnection true(trigger timer for connection)/false(trigger timer for Disconnection)
         */
        tVoid vChkTriggerTimer(const tBool isDeviceConnection);


        /*!
         * \brief
         *
         * Notified by FC_HeatControl to service of VD_DVM to inform about over temperature at cd drive
         *
         * \param [in] bOverTemperature   true/false
         */
        tVoid vSetOverTemperatureCDDrive(IN tBool bOverTemperature); //Roadmap 13035 Over temperature

        /*!
         * \brief Notified by Client handler of VD_Mediamanager about a CDDA device
         *
         * Used by Client handler of VD_Mediamanager to inform  about the cdda device.
         *
         * Clienthandler writes into a queue.
         * Function StartMonitorLooseBlocking listens to that queue -
         *
         * \param [in] eMsgType   will be always enCDDANotified
         * \param [in] DEVICE_CONNECTSTATUS_Type  connection status of CDDA.
         * \param [in] pCurDevice the Device information
        */

        tVoid vsetCDDANotified(IN tenMsgTyp eMsgType,IN DEVICE_CONNECTSTATUS_Type  enConnectStatus, IN const CDevice* pCurDevice);


        /*!
        * \brief Notified by Client handler of VD_Mediamanager about a CDDA device Slot state
        *
        * Used by Client handler of VD_Mediamanager about the cdda device.
        *
        * Clienthandler writes into a queue.
        * Function StartMonitorLooseBlocking listens to that queue -
        *
        * \param [in] eMsgType   will be always enCDDANotified
        * \param [in]  eOpticalDiscSlotState  Slot state of Optical Disc.
        */

        tVoid vSetOpticalDiscSlotStateNotified(IN tenMsgTyp eMsgType, IN tDMOpticalDiscSlotState eOpticalDiscSlotState);


        /*!
          * \brief Used by prm to set HW signals i.e. these are written into a
          *        queue and this queue is read by StartMonitorFunction of this
          *        component - see see 'case enHWSignals' function used only by setHWSignals
          *
          *  \param [in] pPortState
          */

        tVoid _setHWSignals(const UsbPortState *pPortState);

        /*!
         * \brief If the execution timer expires then this function is called
         *        and writes expiration event to queue. Thread function
         *        StartMonitorLooseBlocking listens to that queue - see
         *        'case enTimerExp'
         * \param  [in] this pointer necessary if function is triggered by osal timer
         */
        static tVoid setTimerExpired(StateTable *f_pStateTable);

        /*!
         *  \brief To prevent sending results in the lookup phase where the
         *         VD_DeviceManager has not yet set its service available
         *         this function is used to disable execution time first.
         *         Afterwards it will enabled. The execution timer is used to
         *         trigger the state machine to calculate a result to send it to
         *         the clients i.e. this function is used to suppress the
         *         mechanism first.
         *
         *  \param [in] bEnable TRUE means enable execution timer, FALSE means
         *              disable it
         */
        tVoid enableExecutionTimer(tBool bEnable);

        /*!
         * \brief Used for shutdown msgQ based thread - see destructor. In order
         *        to finish execution of thread function StartMonitorLooseBlocking
         *        this function is used to send the corresponding event to a
         *        queue. function 'StartMonitorLooseBlocking' does listen to
         *        that queue - see 'case enShutdown'
         */
        tVoid vShutdown(tVoid);

        /*!
         * \brief Copy between device structures
         *
         * \param [in] pNotifyClients_Src source of information
         * \param [in] pCurDevice_Dest    destination of information
         *
         * \return [tBool]
         *
         * \todo this function will be redundant if the number of device lsits can be shriked to a single one
         */
        tBool bFillCDevice( CDevice* pCurDevice_Dest, const structNotifyClients *pNotifyClients_Src) const;


        /*!
         * \brief get list of all currently connected devices
         *
         * \param [out] f_vConnectedDevices vector which will be filled
         */
        void GetAllConnectedDevices(vector<structNotifyClients> &f_vConnectedDevices);

       /*!
        * \brief if vDiagMountCDRom is called to trigger mount of MixedMode CDROM (not mounted by automounter)
        *
        * \parm 'strDeviceName holds name of device e.g. /dev/sr0
        */
         IN tVoid vDiagMountCDRom(IN tBool bMount, IN const std::string &strDeviceName /*e.g. /dev/sr0*/); //Roadmap 14002 MountCDROM (diagMixeMode)

       /*!
         * \brief setTestHookNoErase()
         *
         * \parm 'strDeviceName holds name of device e.g. /dev/sr0
         */
     void SetTestHookNoErase(IN bool bNoErase);

         /*!
         * \brief Notified by Client handler of VD_Mediamanager about a CDDA device CD Info
         *
         * Used by Client handler of VD_Mediamanager about the cdda device.
         *
         * Clienthandler writes into a queue.
         * Function StartMonitorLooseBlocking listens to that queue -
         *
         * \param [in] eMsgType   will be always enCDDANotified
         * \param [in]  trOpticalDiskCDInfo   CDInfo Structure - Device State and Drive Version of Optical Disc.
         */

         tVoid vSetOpticalDiscCDInfoNotified(IN tenMsgTyp eMsgType, IN trOpticalDiskCDInfo f_OpticalDiskCDInfo);


         /*!
         * \brief Notified by PRMManager about a Power state of USB port
         *
         *
         *
         * Clienthandler writes into a queue.
         * Function StartMonitorLooseBlocking listens to that queue -
         *
         * \param [in] eMsgType   will be always enPortState
         * \param [in]  f_rPortState   PorState Structure - Port number, powerON state, OC and UV
         */
         IN tVoid vSetPortPowerStateNotified(IN tenMsgTyp eMsgType,IN const trPortStates &f_rPortState);

         /*!
         * Retrial mechanism for multipartition mount update
         */
         tVoid vRetryForAllPartitionMounted();

         /*!
         * Retrial mechanism for connected  device type
         * * \parm [in] f_eDeviceType: the device type such as CDROM, Interanl SD card
         */
         tVoid vRetryForDeviceTypeConnected(IN  CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType);
//========================================================================================================
//  Interface injection functions  - see IPrmManagerCard
//---------------------------------------------------------------------------------------------------------------------

         IN tVoid SetPrmManagerCardIF(IPrmManagerCard *f_pIPrmManagerCard);


//========================================================================================================
//  Interface functions  - see IStateTableHWSignals, IStateTableMount, IStateTableVoltage
//========================================================================================================

            /*!
             * \brief see IStateTableHWSignals
             */
            virtual tVoid setHWSignals(UsbPortState *pPortState);



            /*!
             * \brief see IStateTableHWSignals
             */
            virtual tVoid SetCardSignals(IN tenSignalValue f_enConnected, IN tChar (&f_strDeviceName)[STRING_LENGTH],tU16 f_u16MediaType);


            /*!
                 * \brief see IStateTableHWSignals
                 */
            virtual tVoid setCardOverCurrentStatus( const CardNotInfoOcStatus* pInfo );

             /*!
              * \brief see IStateTableHWSignals
              */
             virtual tVoid getSignals(tenDevManagerUSBHost usb,voltageFailure_HWSignalsAll_t *pSignal);



             /*!
              * \brief see IStateTableVoltage
              */
            virtual tVoid setVoltage(tenDevManagerUSBHost eUsb,tenSignalValue eU60_drop);

            /*!
             * \brief see IStateTableMount
             */
            virtual tVoid setMount(tenMsgTyp eMsgType,tenDevManagerUSBHost eUsb,
                    tenSignalValue eMount, const CDevice* pCurDevice);

            /*!
             * \brief see IStateTableMount
             */
            virtual tenDevManagerUSBHost GetConnector( const CDevice *pCurDevice) const;



            /*!
             * \brief see IStateTableHWMalfunction
             */
            virtual tVoid SetMalfunctionUSBConnector(IN tBool bIsMalfunction, IN tenDevManagerUSBHost eUsbConnectorNum,IN tenTypeOFMalfunction eTypeOFMalfunction = eIsMalfunction);

            /*!
             * \brief see IStateTableHWMalfunction
             */
            virtual tVoid SetMalfunctionOpticalDisc(IN tBool bIsMalfunction);

            /*!
             * \brief see IStateTableHWMalfunction
             */
            virtual tVoid SetMalfunctionSDCardInternal(IN tBool bIsMalfunction);


            /*!
             * \brief see vStopRetryForDevice
             * This calls the devicelistmanager vStopRetryForDevice. If it returns true that means that devicemanager can detect the device as MTP
             * \parm fSerialID
             *
             */
            virtual tVoid vStopRetryForDevice(IN GENSTRING fSerialID);




//=====================================protected==============================================================
    protected:

    /*!
        * \brief trace posrt states send by prm to error memory
        */
    tVoid TraceErrorMem(UsbPortState *pPortState);


   //============================================================================
   //protected functions : subcases used in StateTable::vThreadFunction()
   //============================================================================

     /*!
      * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
      *
      * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
      * \parm u64TestValue: this value is used within unit tests to inject e.g. TimeStamp which is notmally taken from system command
      */
     tVoid vCASEenMountUdev(IN const util_StructMsg &oMsg,tU64 u64TestValue=0);

     /*!
      * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
      *
      * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
       * \parm u64TestValue: this value is used within unit tests to inject e.g. TimeStamp which is normally taken from system command
      */
     tVoid vCASEenCDDANotified(IN const util_StructMsg &oMsg,tU64 u64TestValue=0);

     /*!
      * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
      *
      * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
      */
     tVoid vCASEenOpticalDiscSloteStateNotified(IN const util_StructMsg &oMsg);

     /*!
      * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
      *
      * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
      */
     tVoid vCASEenPortPowerInfoNotified(IN const util_StructMsg &oMsg);

     /*!
      * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
      *
      * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
      */
     tVoid vCASEenOpticalDiscCDInfoNotified(IN const util_StructMsg &oMsg);

     /*!
      * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
      * \parm u64TestValue: this value is used within unit tests to inject e.g. TimeStamp which is notmally taken from system command
      */
     tVoid vCASEenTimerExp(tU64 u64TestValue=0);

      /*!
       * \brief case used in evaluating message queue in thread StartMonitorLooseblocking - old version
       *
       */
     tVoid vCASEenTimerExpOld(tVoid);


     /*!
      * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
      *
      * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
      */
     tVoid vCASEenHWSignals(IN const util_StructMsg &oMsg);


     /*!
      * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
      *
      * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
      */
     tVoid vCASEenCardSignals(IN const util_StructMsg &oMsg) const;

     /*!
      * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
      *
      * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
      */
     tVoid  vCASEenCardOcStatus(IN const util_StructMsg &oMsg) const;

     /*!
      * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
      *
      * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
      */
     tVoid vCASEenVoltWarning(IN const util_StructMsg &oMsg);

     /*!
      * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
      *
      * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
      */
    tVoid vCASEenTestSend2Client(IN const util_StructMsg &oMsg);

    /*!
     * \brief case function used in thread functionStartMonitorLoosBlocking
     *
     *used to notify over temperature for connected CDRom
     */
    tVoid vCASEenCDDriveOverTemp(IN const util_StructMsg &oMsg) const; //Roadmap 130035


    /*!
     * \brief case function used in thread functionStartMonitorLoosBlocking
     *
     *used to notify over temperature for connected CDRom
     */
    tVoid vCASEenDiagMountCDRom(IN const util_StructMsg &oMsg);

    /*!
      * \brief case function used in thread functionStartMonitorLoosBlocking
      *
      */
    tVoid  vCASEenDevRecAfterClv();


     /*!
      * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
      *
      */
    tVoid vCASEenTriggerTimer(IN const util_StructMsg &oMsg);


    /*!
     * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
     *  This function will store the time stamp of Malfunction occurred in the devices connected to USB connector
     * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
     */
    tVoid vCASEenMalfunctionUsbConnector(IN const util_StructMsg &oMsg, tU64 u64TestValue=0); //0 means is not used


    /*!
     * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
     * This function will store the time stamp of Malfunction in the opticaldisc
     * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
     */
    tVoid vCASEenMalfunctionOpticalDisc(IN const util_StructMsg &oMsg);

    /*!
     * \brief case used in evaluating message queue in thread StartMonitorLooseblocking
     * This function will store the time stamp of Malfunction in the SD Card Internal
     * \parm oMsg: which is unmarshalled in this function and contains necessary input parameters
     */
    tVoid vCASEenMalfunctionSDCardInternal(IN const util_StructMsg &oMsg);

    /*!
     * \brief This function will Check whether the USB connectors has permanent malfunction
     * \parm [in] eusb: Usb Connectors
     */
    tBool isADevicePermanentMalfunction( IN tenDevManagerUSBHost eusb);

    /*!
     * \brief This function will Check whether the Device types such as SD internal or CDROM has malfunction
     * \parm [in] f_eDeviceType: the device type such as CDROM, Interanl SD card
     */
    tBool isADevicePermanentMalfunction( IN  CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType);

    /*!
     * \brief This function will send malfunction permanent to clients
     *
     */
    tVoid vSendMalfunctionPermanent(IN tenDevManagerUSBHost eusb);

    /*!
     * \brief This function will send malfunction permanent to clients
     * \parm f_eDeviceType: the device type such as CDROM, Interanl SD card
     */
    tVoid vSendMalfunctionPermanent(OUT CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType);

    /*!
     * \brief This function will set device type to either CDROM or CDDA
     * \parm  [out] f_eDeviceType: contains the device type to be filled
     *
     */
    tVoid vSetOpticalDiscDeviceType(OUT CGlobalEnumerations::DEVICE_TYPE_Type &f_eDeviceType);

    /*!
     * \brief This function will set device type to either SD Internal
     * \parm  [out] f_eDeviceType: contains the device type to be filled
     *
     */
    tVoid vSetSDInternalDeviceType(OUT CGlobalEnumerations::DEVICE_TYPE_Type &f_eDeviceType);


    /*!
     * \brief This function will Check whether the Optical Disc or SD internal has permanent malfunction
     *
     */
    tBool bIsMalfunctionOccuredAtDevices(IN const CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType) const;

#ifndef USE_HIGHRISK4USB_HWMALFUNCTION

    /*!
     * \brief This function will set device type to either SD Internal
     * \parm [in] eusb: Usb Connectors
     * \parm  [out] f_eConnectStatus: contains the Connection Status to be updated
     *
     */
    tVoid vDecideConnectStatusUsbConnector( IN tenDevManagerUSBHost eusb,OUT DEVICE_CONNECTSTATUS_E &f_eConnectStatus);



#endif

    /*!
         * \brief This function will check the malfunction Status and decide the connect status
         * \parm  [out] f_eDeviceType: contains the device type to be filled
         * \parm  [out] f_eConnectStatus: contains the Connection Status to be updated
         *
         */
        tVoid vDecideConnectStatusDevices(IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType,OUT DEVICE_CONNECTSTATUS_E &f_eConnectStatus);


    //============================================================================
    //protected functions of interface which are mapped public unit tests to prevent usage of threads in unit test
    //============================================================================



    /*!
     * \brief see IStateTableMount
     */
    virtual tBool setMountFill(IN tenMsgTyp eMsgType,IN tenDevManagerUSBHost eUsb,IN tenSignalValue eMount, IN const CDevice* pCurDevice,OUT util_StructMsg &oMsg);

    /*!
     * \brief see IStateTableMount
     */
    virtual tBool setCDDANotifiedFill(IN tenMsgTyp eMsgType, IN tenSignalValue eMount, IN const CDevice* pCurDevice,OUT util_StructMsg &oMsg);

    /*!
    * \brief see IStateTableMount
    */
    tBool setOpticalDiscSlotStateNotifiedFill(IN tenMsgTyp eMsgType,IN tDMOpticalDiscSlotState eOpticalDiscSlotState,OUT util_StructMsg &oMsg);



    /*!
    * \brief see IStateTableMount
    */
    tBool SetPortPowerStateNotifiedFill(IN tenMsgTyp eMsgType,IN trPortStates  rPortState,OUT util_StructMsg &oMsg);




    /*!
     * \brief see IStateTableVoltage
     */
    virtual tBool setVoltageFill(IN tenDevManagerUSBHost eUsb,IN tenSignalValue eU60_drop, OUT util_StructMsg &oMsg);


    /*!
     * \brief see IStateTableVoltage
     */
    virtual tBool setHWSignalsFill(IN const UsbPortState *pPortState,OUT util_StructMsg &oMsg);

    /*!
     * \brief see IStateTableHWSignals
     */
    virtual tBool SetCardSignalsFill(IN tenSignalValue f_enConnected, IN tChar (&f_strDeviceName)[STRING_LENGTH], IN tU16 f_u16MediaType,OUT util_StructMsg &oMsg);

    /*!
     * \brief see IStateTableHWSignals
     */
    virtual tBool setCardOverCurrentStatusFill( IN const CardNotInfoOcStatus* pInfo, OUT util_StructMsg &oMsg);

    /*!
    * \brief see IStateTableMount
    */
    tBool setOpticalDiscCDInfoNotifiedFill(IN tenMsgTyp eMsgType,IN trOpticalDiskCDInfo f_OpticalDiskCDInfo,OUT util_StructMsg &oMsg);



   //============================================================================
   //protected functions : other
   //============================================================================

   /*!
    * \brief set current concerning a certain USB connector - this function
    *        is used by queue listener StartMonitorLooseBlocking see
    *        case enVoltWarning, case enHWSignals, case enMountUdev,
    *
    * \param [in] signal signal to be set
    */
tVoid setSignals( const voltageFailure_HWSignalsAll_t&  signal);


    /*!
    * \brief Fill structure structNotifyClients with information available
    *        in CDevice
    *
    * \param [out] pNotifyClients_Dest destination structNotifyClients object
    * \param [in]  pCurDevice_Src source Device object
    *
    * \return [tBool]
    */
    tBool bFillstructNotifyClients(structNotifyClients *pNotifyClients_Dest,
          const CDevice *pCurDevice_Src) ;


    /*!
    * \brief see IStateTableHWMalfunction
    */
    virtual tBool SetFillMalfunctionUSBConnector(IN tBool bIsMalfunction,
          IN tenDevManagerUSBHost UsbConnectorNum, IN tenTypeOFMalfunction eTypeOFMalfunction, OUT util_StructMsg &oMsg);

#ifndef USE_HIGHRISK4USB_HWMALFUNCTION
    /*!
    * \brief see IStateTableHWMalfunction
    */
    virtual tVoid StoreMalfunctionUSBConnectorWithTimeStamp(IN tBool bIsMalfunction,
          IN tenDevManagerUSBHost UsbConnectorNum);

    /*!
       * \brief see IStateTableHWMalfunction
       */

    virtual tVoid StoreHighRiskForMalfunctionUSBConnectorWithTimeStamp(IN tBool bIsMalfunction); //intermediate adjust StoreMalfunctionUSBConnectorWithTimeStamp
#endif //USE_HIGHRISK4USB_HWMALFUNCTION

    /*!
    * \brief see IStateTableHWMalfunction
    */
    virtual tBool bIsRemovedCausedByMalfunction(IN tenDevManagerUSBHost UsbConnectorNum);

    /*!
    * \brief see IStateTableHWMalfunction
    */
    virtual tBool SetFillMalfunctionSDCardInternal (IN tBool bIsMalfunction, OUT util_StructMsg &oMsg);


    /*!
    * \brief see IStateTableHWMalfunction
    */
    virtual tBool SetFillMalfunctionOpticalDisc(IN tBool bIsMalfunction, OUT util_StructMsg &oMsg);



    /*!
     * \brief see IStateTableHWMalfunction
     *
     */
    virtual tVoid  vMonitoringForMalfunctionPermanent();


//====================================================================================================

    protected:
        // static member variable declared for singleton class
        static StateTable       *m_pStateTableManager;
        static DVMLOCK           m_singelton;

        //objects used by this class
        CUDevManager            *m_poManagerUdev;

        DVMLOCK m_lock;

        tBool                            m_bIsActive;
        tBool                            m_bCommonTimerStarted;//Timer for disconnection,malfunction and other events
        tBool                            m_bConnectTimerStarted;//NCG3D-146730: Timer for connection events
        tBool                            m_bEnableExecutionTimer;
        tInt                             m_iIntervalSignalCounter;//received during interval
        tU32                             m_u32CommonIntervalTimer_ms;//Timer value used when a disconnect/malfuncion/other events are received
        tU32                             m_u32ConnectIntervalTimer_ms;//NCG3D-146730: Timer value used when a connect event is received

#ifdef MONITOR_PERMANENT_HW_FAILURE
        tU32                             m_u32CountPermanentOvercurrent[ARRAYSIZEFORUSB];
#endif
        util_StructQInfo                 m_msgQ;

        DVMLOCK m_table;
        DVMLOCK m_thread;

        /*@todo check if this might be extended by element resultCDevice*/
        voltageFailure_Result_t                        m_result[ARRAYSIZEFORUSB];        //holds result of state table

        CDevice                                        m_TestDevice[ARRAYSIZEFORUSB];

        voltageFailure_HWSignalsAll_t    m_signal[ARRAYSIZEFORUSB];

        voltageFailure_Result_t          m_resultLastSendtoClient[ARRAYSIZEFORUSB];

        static voltageFailure_IFOutput_t m_voltageFailureStateTable_USB1[];
        static voltageFailure_IFOutput_t m_voltageFailureStateTable_USB2[];
        static voltageFailure_IFOutput_t m_voltageFailureStateTable_USB3[];
#ifdef USE_FOURTH_USB_CONNECTOR
        static voltageFailure_IFOutput_t m_voltageFailureStateTable_USB4[];
#endif


        static StateTableElemInformClients_t  m_stateTableInformClients[]; //state table to adjust messages for clients i.e. to send several messages because of a certain state
        static MsgListElemInformClients_t     m_msgListInformClients[];    //holds the messages which should be send for a certain state

        tBool m_bOverTemperature;  //Roadmap 13035 Overtemperature
        tBool m_bMountedCDRomDiag; //Roadmap 14002 MontCDRom (MixedModeDiag)

        IPrmManagerCard *m_pIPrmManagerCard;

        bool m_bTestHookNoErase;

        TimerDVM m_TimerDvm1UsbRecognitionAfterClvEvent;
        TimerDVM m_TimerDvm2SignalAccumulationTimer;

#ifndef USE_HIGHRISK4USB_HWMALFUNCTION
        //to store the info about malfunction
        tBool m_bHasMalfunctionUsbConnector;
        structNotifyMalfunction m_oNotifyMalfunction[eUSBAll];//Store the USB connectors which has malfunction
#endif

        tBool m_bHasMalfunctionOpticalDisc;
        tBool m_bHasMalfunctionSDInternal;

        MessageQueueDVM m_MessageQueueDVM;


        /*!
         * \brief if prm mount signals are not used statemachine sends fakes a
         *        prm mount and sends it to itself
         *
         * \param [in] m_eDeviceType e.g. DTY_USB, DTY_SD etc.
         *
         * \return [tBool]
         *
         * \todo cleanup possible: it would be better to create a new state
         *       table where prm mount depedency is set to SIG_ANY
         */
        tBool bChkSendMyselfPrmMount(
                CGlobalEnumerations::DEVICE_TYPE_Type m_eDeviceType) const;


        /*!
         * \brief send a message using a queue
         *
         * \param [in] pQInfo holds all necessary information to do this job
         * \param [in] pMsg   will contain the received message
         */
        tVoid util_vSendMsg2Q(const util_StructQInfo* pQInfo,const util_StructMsg* pMsg) ;

        /*!
         * \brief listen to a queue for new messages
         *
         * \param [in] pQInfo     holds all necessary information to do this job
         * \param [in] pMsg       will contain the received message
         * \param [in] timeout_ms after timout_ms without receiving a message
         *             function will return (used to check for Permanent HW function)
         *
         * \return [tS32]
         */
        tS32 util_s32Wait4Q(const util_StructQInfo* pQInfo,util_StructMsg* pMsg,unsigned int timeout_ms) ;


        /*!
         * \brief prepare HW signals to be handed ofer to PortListManager
         *
         * \param [in] array of Hardware-Signals e.g. PPON (voltage supply of USBConnector), OC (Fault line) etc.
          * \param [OUT] DiffersFromPreviousStored -returns if values have changed
         *
         */
        tVoid vMapAndStoreHWSignals(const IN voltageFailure_HWSignalsAll_t  (&f_rSignals)[ARRAYSIZEFORUSB],OUT tBool &bDiffersFromPreviousStored) const ;

#ifdef REFACTOR_STATETABLE_OUTPUT
        /*!
          * \brief sets output states of devices in taking undervoltage, (un)mounts (Apply Connect, Apply Remove), HW singnals in to account
          *
          * \param [in] f_result: array with result of state calculation according to m_voltageFailureStateTable_USB1 & 2

          *
          */
       tVoid vMapDevicePropertiesStore(IN voltageFailure_Result_t (&f_result)[ARRAYSIZEFORUSB]);
#endif


        /*!
         * \brief Thread function - it invokes StartMonitorLooseBlocking
         *
         * Thread to listen to message queue filled by prm, udev.
         * clientHandler_spm and timerExp
         */
        virtual tVoid vThreadFunction();

        /*!
         * \brief central function
         *
         * Listens to message queue filled by prm, udev. clientHandler_spm and
         * timerExp etc. Triggers state machine to give a result and this will
         * be send to clients
         */
        tVoid StartMonitorLooseBlocking();



        /*!
         * \brief case function used in thread functionStartMonitorLoosBlocking
         *
         * in production line a MixeMode-CD is used which is normally not mounted
         * since it is trated as Audio-CD
         * To reach content of the CDROM-part of the CD CDiagnosis sends this command
         */
        tVoid vSubCASEenDiagMountCDRom(IN const GENSTRING &strDeviceName);   //Roadmap 14002 MountCDROM (DiagMixeMode)

        /*!
         * \brief case function used in thread functionStartMonitorLoosBlocking
         *
         * in production line a MixeMode-CD is used which is normally not mounted
         * since it is treated as Audio-CD
         * To reach content of the CDROM-part of the CD CDiagnosis sends mount
         * Afterwards FC_Diagnosis sends a command to unmount it which triggers
         * this case.
         */
        tVoid vSubCASEenDiagUnMountCDRom(IN const GENSTRING &strDeviceName); //Roadmap 14002 MountCDROM (DiagMixeMode)



            /*!
             * \brief store current OverTemperatureValue (CDDrive)
             *        is used if no CDROM is connected but overtermperature for the device has already been notified
             *        In case a CDROM is inserted then StateTable takes care to take such overtemperature into account
             *        for notification towards the service

             * \param [in] bOverTemperature: TRUE/FALSE
             */

        tVoid vStoreOverTemperature(tBool bOverTemperature);


        /*!
         * \brief Create some test devices for diagnosis
         */
        tVoid fillTestDevices(tVoid);



        /*!
         * \brief Trigger timer to be started.
         *
         * It is started in case enMountUdev, case enHWSignals,
         * case enVoltWarning: If the timer expires then in StartMonitorLooseBlocking a
         * result of the stateMachine will evaluated
         *
         * \param [in] isDeviceConnection TRUE if the trigger is for a device connection
         *                                FALSE otherwise
         */
        tVoid vTriggerTimer(const tBool isDeviceConnection = false);


        /*!
         * \brief This function acts as a filter to suppress certain messages to be send.
         *
         * Config.h and maybe registry will be used to decide which state is
         * usable to send messages to a client,
         *
         * \param [in] usb           related to connector usb
         * \param [in] connectstatus based on config.h (and planned on registry)
         * \param [in] bTest         TRUE: dont filter send everythink,
         *             FALSE in no test mode it depends on configuration what will be send
         *
         * \return [tBool]
         */
        tBool checkSendBasedOnConfig(tenDevManagerUSBHost usb,
                DEVICE_CONNECTSTATUS_Type connectstatus, tBool bTest) const;

    #ifdef USE_SENT_ALL_CHANGES
        // Private static functions
        static bool bOnTimeoutUsbRecognitionAfterClvEvent(timer_t timerID , void* instance ,const void *userData); //used my linux timer
        static void vOnTimeoutUsbRecognitionAfterClvEvent( tPVoid arg ); //used by OSAL version but also called from bOnTimeoutUsbRecognitionAfterClvEvent
        tBool bStartTimer( );
        tBool bStopTimer( );
    #endif //#if USE_SENT_ALL_CHANGES


    static bool bOnTimeoutSignalAccumulationTimer(timer_t timerID , void* instance ,const void *userData);//used my linux timer

    static void vOnTimeoutSignalAccumulationTimer(tPVoid arg );//used by OSAL version but also called from vOnTimeoutSignalAccumulationTimer


        template< class T > void vDeleteNoThrow( T*& rfpArg ) throw();




#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
    public: /*to make protected functions testable with google test a testable class is derived this is possible only if class has public contricto & destructor*/
#endif

    /*!
     * \brief Private constructor for singleton
     *
     */
    StateTable();

    /*!
     * \brief private destructor for singleton
     */
    ~StateTable();

};

#endif //__STATE_TABLE_H__

////////////////////////////////////////////////////////////////////////////////
// <EOF>
