/*-----------------------------------------------------------------------------*
 * IStateTableMount.h                                                      *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM G3G                                                   *
 * COPYRIGHT   : (c) 2014-2016  Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file IStateTableMount.h
 *
 * \brief Interface used by UdevManager to inform StateTable
 *
 * \version 25.09.2014, Koechling Christian (CM-AI/ECD1), version 1.0
 
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */


#ifndef INTERFCACE_STATETABLE_MOUNT_H
#define INTERFCACE_STATETABLE_MOUNT_H  

/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/
/*!
 * \class IStateTableMount
 */

class IStateTableMount
{
    public:

        /*!
         * \brief Destructor
         */
        virtual ~IStateTableMount(){};

        /*!
         * \brief pCurDevice contains connector num as int value. It is returned as tenDevManagerUSBHost
         *
         * \param [in] pCurDevice the device object
         *
         * \return [tenDevManagerUSBHost] usb at which connector the device with
         *         (such mount point) is connected to
         */
        virtual tenDevManagerUSBHost GetConnector( const CDevice *pCurDevice) const = 0;


        /*!
        * \brief Notified by UDevManager about a new mount
        *
        * Used by UDevManager to tell if mount has been done. It writes into a
        * queue. Function StartMonitorLooseBlocking listens to that queue -
        * see enMountUdev
        *
        * \param [in] eMsgType   will be always enMountUdev
        * \param [in] eUsb       mount at connector eUsb e.g. usb1 or usb2 or ...
        * \param [in] eMount     SIG_TRUE mean mount, SIG_FALSE means FALSE
        * \param [in] pCurDevice the Device information
        */
        virtual tVoid setMount(IN tenMsgTyp eMsgType,IN tenDevManagerUSBHost eUsb,IN tenSignalValue eMount, IN const CDevice* pCurDevice) = 0;


        /*!
        * \brief Notified by Client handler of VD_Mediamanager about a CDDA device
        *
        * Used by Client handler of VD_Mediamanager about the cdda device.
        *
        * Clienthandler writes into a queue.
        * Function StartMonitorLooseBlocking listens to that queue -
        *
        * \param [in] eMsgType   will be always enCDDANotified
        * \param [in] DEVICE_CONNECTSTATUS_Type  connection status of CDDA.
        * \param [in] pCurDevice the Device information
        */

        virtual tVoid vsetCDDANotified(IN tenMsgTyp eMsgType, IN DEVICE_CONNECTSTATUS_Type  enConnectStatus, IN const CDevice* pCurDevice) = 0;

        /*!
        * \brief Notified by Client handler of VD_Mediamanager about a CDDA device Slot state
        *
        * Used by Client handler of VD_Mediamanager about the cdda device.
        *
        * Clienthandler writes into a queue.
        * Function StartMonitorLooseBlocking listens to that queue -
        *
        * \param [in] eMsgType   will be always enCDDANotified
        * \param [in]  eOpticalDiscSlotState  Slot state of OpticalDisc.
        */

         virtual tVoid vSetOpticalDiscSlotStateNotified(IN tenMsgTyp eMsgType, IN tDMOpticalDiscSlotState eOpticalDiscSlotState) = 0;

    protected:

        /*!
        * \brief helper function used by public setCDDANotified function. function is mentioned here to emphasis binding to setCDDANotified-function.
        * In unit tests no threads are used. if setCDDANotified is called then this normally fills a message queue. A Thread is waiting for that message and calls corresponding vCase-function
        * Hence in unit tests below function is used which fills msg-content. The unit test does call directly the corresponding vCase-function. I.e. it does what the thread does but
        * same function are envolved to be as close as possible at what happens with the thread
        */
        virtual tBool setCDDANotifiedFill(IN tenMsgTyp eMsgType,IN tenSignalValue eMount, IN const CDevice* pCurDevice,OUT util_StructMsg &oMsg) = 0;


        /*!
        * \brief helper function used by public setOpticalDiscSlotStateNotifiedFill function. function is mentioned here to emphasis binding to setCDDANotified-function.
        * In unit tests no threads are used. if setCDDANotified is called then this normally fills a message queue. A Thread is waiting for that message and calls corresponding vCase-function
        * Hence in unit tests below function is used which fills msg-content. The unit test does call directly the corresponding vCase-function. I.e. it does what the thread does but
        * same function are envolved to be as close as possible at what happens with the thread
        */
        virtual tBool setOpticalDiscSlotStateNotifiedFill(IN tenMsgTyp eMsgType,IN tDMOpticalDiscSlotState   eOpticalDiscSlotState,OUT util_StructMsg &oMsg) = 0;


        /*!
              * \brief helper function used by public setvoltage function. function is mentioned here to emphasise binding to setMount-function.
              *          In unit tests no threads are used. if setvolate is called then this normally fills a messag queue. A Thread is waiting for that message and calls correspoding vCase-function
              *          Hence in unit tests below function is used which fills msg-content. The unit test does call directly the correspodning vCase-function. I.e. it does what the thread does but
              *          same function are envolved to be as close as possible at what happens with the thread
              */
        virtual tBool setMountFill(IN tenMsgTyp eMsgType,IN tenDevManagerUSBHost eUsb,IN tenSignalValue eMount, IN const CDevice* pCurDevice,OUT util_StructMsg &oMsg) = 0;
		
		

};
// class IStateTableMount

#endif // INTERFCACE_STATETABLE_MOUNT_H

////////////////////////////////////////////////////////////////////////////////
// <EOF>

