/*-----------------------------------------------------------------------------*
 * ReaderKDS.cpp                                                    *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : G3G                                                   *
 * COPYRIGHT   : (c) 2012 - 2020 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*! 
 * \file ReaderKDS.cpp
 *
 * \brief Implementation to open, read and close the EOL file.
 *
 * \version 05.03.2014, Koechling Christian (CM-AI/ECD1), initial version
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2020
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"


#include "config/IConfigurationPersistent.h"
#include "config/ReaderKDS.h"


/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"
     
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_CONFIGMANAGER
  #include "trcGenProj/Header/ReaderKDS.cpp.trc.h"
  #endif
  #include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST


/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
ReaderKDS::ReaderKDS()
{
    ETG_TRACE_USR4(("Begin:ReaderKDS"));

    m_oFileDescrKDS = OSAL_IOOpen(OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READONLY);

    if ( OSAL_ERROR != m_oFileDescrKDS )
    {
        ETG_TRACE_USR2(("ReaderKDS: KDS opened successful "));
    }
    else
    {
        ETG_TRACE_FATAL(("ReaderKDS: KDS could not be opened "));
    }

    ETG_TRACE_USR4(("End  :ReaderKDS"));

}// ReaderKDS::ReaderKDS()


/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
ReaderKDS::~ReaderKDS()
{
    ETG_TRACE_USR4(("Begin:~ReaderKDS"));

    OSAL_s32IOClose(m_oFileDescrKDS);

    ETG_TRACE_USR4(("End  :~ReaderKDS"));
} // ReaderKDS::~ReaderKDS()  


/*-----------------------------------------------------------------------------*
 * tBool bRead(tCU8 cu8Table, tCU16 cu16Offset, tU8& rfu8Value)                *
 *-----------------------------------------------------------------------------*/
tBool ReaderKDS::bReadKDS(IN tU16 u16EntryID, IN tU16 u16EntryLength, OUT  tU8 (&f_u8Array)[KDS_MAX_ENTRY_LENGTH])
{
    ETG_TRACE_USR4(("Begin:bReadKDS"));

    tBool      bRetVal = FALSE;
    tU32       u32LastErrorCode;
    tsKDSEntry rEntryData;

    rEntryData.u16Entry       = u16EntryID;
    rEntryData.u16EntryLength = u16EntryLength;
    rEntryData.u16EntryFlags  = M_KDS_ENTRY_FLAG_NONE;
    memset(rEntryData.au8EntryData,0, sizeof(rEntryData.au8EntryData)); //clear return value
    
    //nok
    if(OSAL_ERROR == m_oFileDescrKDS)
    {
       ETG_TRACE_FATAL(("[ERROR] bReadKDS:KDS has not been opened before!"));
    }
    //ok
    else
    { 
        //nok
        if(u16EntryLength > KDS_MAX_ENTRY_LENGTH)
        {
            DVM_NORMAL_M_ASSERT_ALWAYS();
            ETG_TRACE_FATAL(("[ERROR] bReadKDS:u16EntryLength > KDS_MAX_ENTRY_LENGTH (%d > %d)",u16EntryLength,KDS_MAX_ENTRY_LENGTH));
        }
        //ok
        else
        {
            //nok
            if(OSAL_s32IORead(m_oFileDescrKDS,(tS8 *)&rEntryData,(tS32)sizeof(rEntryData)) == OSAL_ERROR)
            {
                u32LastErrorCode =  OSAL_u32ErrorCode();
                ETG_TRACE_FATAL(("[ERROR] bReadKDS:error read u32LastErrorCode:0x%x ",u32LastErrorCode));
            }
            //ok-Result
            else
            {
                bRetVal = TRUE;
                memset(&(f_u8Array[0]), 0, u16EntryLength);
                memcpy(&(f_u8Array[0]) /*dest*/, &(rEntryData.au8EntryData[0]) /*source*/, rEntryData.u16EntryLength /*num of bytes to copy*/);

                ETG_TRACE_USR2(("bReadKDS: rEntryData.u16EntryLength: %d bytes read",rEntryData.u16EntryLength));
            }
        }
    }

    ETG_TRACE_USR4(("End  :bReadKDS"));
    return bRetVal;
} // tBool ReaderKDS::bRead(tCU16 cu16Offset, tU8& rfu8Value)



int ReaderKDS::GetTypeOfReader()
{
    return (int)eReader_KDS;
}
  


////////////////////////////////////////////////////////////////////////////////
// <EOF>
