/*-----------------------------------------------------------------------------*
 * utils.h                                                                     *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2010 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file utils.h
 *
 * \brief This file holds declaration of common utility functions
 *
 * \version 11.08.2012, Negi, Sunder (MontaVista), initial version
 * \version 13.08.2012, Negi, Sunder (MontaVista), changes for logging to usb
 *
 *-----------------------------------------------------------------
 *                                   development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch)
 *         -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

#ifndef UTILS_H_
#define UTILS_H_

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"
#include "Enums.h"

/*-----------------------------------------------------------------------------*
 * Defines                                                                     *
 *-----------------------------------------------------------------------------*/
#define TMP_BLOCKSIZE (tU32)1024
#define TMP_LOCALVARIABLE (char*)"RESULT_SIZES"


// File containing vendoris/productis of all the USB devices
#define USB_IDS_FILE "/usr/share/usb.ids"

/*-----------------------------------------------------------------------------*
 * Structs                                                                     *
 *-----------------------------------------------------------------------------*/
/*!
 * \struct hub_details
 *
 * \brief Stores details about a hub
 *
 * \todo inherit from CDevice
 */

/*
see $_SWBUILDROOT/ai_projects/generated/staging/gen3armmake/usr/include/bits
for struct statfs in statfs.h it uses type fsblkcnt_t
               struct statfs {
               ...
               fsblkcnt_t f_blocks;  //Total data blocks in filesystem
               fsblkcnt_t f_bfree;    //Free blocks in filesystem
               fsblkcnt_t f_bavail;  //Free blocks available to
               ...
                };

fsblkcnt_t can be found in types.h in the same directory

//Type to count file system blocks.
__STD_TYPE __FSBLKCNT_T_TYPE __fsblkcnt_t;

in typesizes.h
#define    __FSBLKCNT_T_TYPE    __ULONGWORD_TYPE

back in types.h
#define __ULONGWORD_TYPE    unsigned long int
hence it is unsigned long int

//code will be kept to long long since this woud cover both

*/
typedef struct StorageInfo
{
        unsigned long long free_space = 0; ///< free space on device
        unsigned long long used_space = 0; ///< used space on device
        unsigned long long total_size = 0; ///< total size of device
} StorageInfo_t;





/*-----------------------------------------------------------------------------*
 * Function declarations                                                       *
 *-----------------------------------------------------------------------------*/

/*!
 * \brief Reads the product name from usb.ids file
 *
 * \param [in]  f_sVendorID  vendor id of the device
 * \param [in]  f_sProductID product id of the device
 *
 * \return [const char *] the product name if device product,vendor id are
 *         found in the file, NULL otherwise
 */
const char *ReadProductNameFromUSBIDS (const char *f_sVendorID,
        const char *f_sProductID);

/*!
 * \brief Tests for SD card
 *
 * This method finds the parent of the device and checks whether it is an
 * SD Card reader. Device type is modified and SD card flag is set on success.
 *
 * \param [in] f_poDev Check this device is an SD Card or not.
 *
 * \returns  true in case device is SD card, false otherwise
 *
 */
bool IsSDCard(struct udev_device *f_poDev);


/*!
 * \brief Check what type of device is connected
 *
 * \param [in]  f_poDev      the udev device object
 * \param [out] f_eDeviceType the device type found
 *
 * \return DEVICEMANAGER_OK on success
 */
int CheckDeviceType (struct udev_device *f_poDev,
        CGlobalEnumerations::DEVICE_TYPE_E& f_eDeviceType);




/*!
 * \brief Finds the device size details
 *
 * \param [in]  f_pcMountPoint the device mountpiint
 * \param [out] f_oInfo        the device size details
 *
 * \return DEVICEMANAGER_OK on success,
 *         DEVICEMANAGER_ERROR_INVALID_PARAM when mountpoint is invalid,
 *         DEVICEMANAGER_ERROR_UNKNOWN if an error occurred.
 */
int FillStorageInfo(const char *f_pcMountPoint, StorageInfo_t& f_oInfo);


/*!
 * \brief Finds the USB device (only total) size
 *
 * \param [in]  f_pUDev the udev context
 *
 * \return OSAL_C_U64_MAX if udev_device_get_sysattr_value( f_pDev, "size" ) fails,
 *         the total size in byte,
 */

/*-----------------------------------------------------------------------------*
 * tU64 FillStorageInfoSize( struct udev_device *f_pUdev )                            *
 *-----------------------------------------------------------------------------*/
tU64 FillStorageInfoSize( struct udev_device *f_pdev );


/*!
 * \brief Check whether the udev_device is an storage device.
 *
 * All the interfaces for the usb device are traversed and checked for
 * 'usb-storage' support.
 *
 * \param [in] f_pUDev the udev context
 * \param [in] f_pDev the udev_device to check for
 *
 * \return true if its a storage device, false otherwise
 */
bool IsStorageDevice(struct udev *f_pUDev, struct udev_device *f_pDev);

/*!
 * \brief Finds the partition table type for the disk
 *
 * \param [in] f_pcBlockDevNode the disk node /dev/sdX
 *
 * \return [CGlobalEnumerations::PT_Type_e] partition table type
 */
CGlobalEnumerations::PT_Type_e FindPartitionTableType (
        const char *f_pcBlockDevNode);

/*!
 * \brief Check if partition is GPT partition type
 *
 * \param [in] f_pcBlockDevNode the disk node /dev/sdX
 *
 * \return bool If PT_GPT is found in one of the partitions
 */
bool IsGPTPartitionTypeFound(const char *f_pcBlockDevNode);

/*!
 * \brief: Checks if the device is mounted ot not
 *
 * This method reads the file (typically /proc/mounts) to determine if
 * if the current device is mounted or not. The members m_iIsMounted and
 * m_cMountPoint are set accordingly
 *
 * \param [in]  f_cMtabFile Parse the device against this file. This
 *              would be either /etc/mtab or /proc/mounts
 * \param [in]  f_pcDeviceNode the device node
 * \param [out] f_cMountPoint the mount-point
 *
 * \returns true if mounted, false otherwise
 */

int CheckIfMounted(const char *f_cMtabFile,CDevice *f_pCurDevice);

/**
 * \brief Lists the connected usb devices
 *
 * \return error status
 */
int ListConnectedDevices();

//int GetPartitionNumber (GENSTRING f_cPartitionNode, int &iPartitionNumber);  //RoadMap 13001_ExtendedUserInformation

/*!
 * \brief Get details about the partitions of a connected device
 *
 * \param [in]  f_cPartitionNode the device partition node to test
 * \param [out] f_cFSType the filesystem of f_cPartitionNode
 * \param [out] f_cUUID the UUID of f_cPartitionNode
 * \param [out] f_cLabel the LABEL of f_cPartitionNode
 *
 * \return DEVICEMANAGER_ERROR_INVALID_PARAM if invalid param,
 *         DEVICEMANAGER_ERROR_UNKNOWN if libudev api fails,
 *         DEVICEMANAGER_OK is success,
 *         DEVICEMANAGER_ERROR_NOTVALID if this is a disk with partition table
 */
int GetPartitionDetails (GENSTRING f_cPartitionNode, GENSTRING& f_cFSType,
        GENSTRING& f_cUUID, GENSTRING& f_cLabel);

/*!
 * \brief Get Number of valid partitions
 *
 * \param [in]  f_BlockDevnode the device block  node to test eg: /dev/sda
 * \param [out] f_u8TotalValidPartitons the number of valid partitions of block device
 *
 * \return
 *         DEVICEMANAGER_ERROR_UNKNOWN if api fails,
 *         DEVICEMANAGER_OK is success,
 *
 */
int FindTotalValidPartitions(const char* f_BlockDevnode,int &f_u8TotalValidPartitons);

/*!
 * \brief Generate udev_device object for the mount point
 *
 * \param [in] f_poUdev the udev context
 * \param [in] f_cMountPoint the device mount-point
 *
 * \return [struct udev_device*] valid pointer on success, NULL on error
 */
struct udev_device *GenerateUDevDevice(udev *f_poUdev,
        const char *f_cMountPoint);

/*!
 * \brief Finds the parent USB device for a device
 *
 * \param [in] f_poDev the udev device
 *
 * \return [struct udev_device*] valid pointer on success, NULL on error
 */
struct udev_device *FindBaseUSBDevice(struct udev_device *f_poDev);

/*!
 * \brief Get details about apple hardware name
 *
 * Generates the apple alsa name and hid dev mountpoint.
 *
 * \param [in]  f_poDev the udev device
 * \param [out] f_pcAlsaHwName the alsa hardware
 * \param [out] f_pcHidDevice the hid device
 *
 * \return DEVICEMANAGER_OK on success
 */
int GetDeviceFileName(struct udev_device *f_poDev, GENSTRING &f_pcAlsaHwName,
        GENSTRING &f_pcHidDevice);


/*!
 * \brief Checks if a certain keyfile exists
 *
 * \param [in]  type of keyfile
 * \param [in ] f_pcMountPoint usb device mount-point
 *
 * \return true, if it exists
 */

bool CheckKeyFileExists(IN CGlobalEnumerations::KeyFile_Type_e eKeyType,IN const char* f_pcMountPoint);


/*!
 * \brief Checks if a certain keyfile exists
 *
 * \param [in]  type of keyfile
 * \param [in ] f_pcMountPoint usb device mount-point
 *
 * \return true, if it exists
 */

bool CheckKeyDirExists(IN CGlobalEnumerations::KeyFile_Type_e eKeyType,IN const char* f_pcMountPoint);


/*!
 * \brief Checks whether logging to usb device is required
 *
 * \param [in] f_pcMountPoint usb device mount-point
 *
 * \return true, if logging to usb is required, false otherwise
 */
bool CheckForLogKeyFile( const char* f_pcMountPoint);



/*!
 * \brief Checks if a key file could be found on a connected usb-stick
 *
 * \param [in] f_pcMountPoint usb device mount-point
 *
 * \return [bool] true, if logging to usb is required, false otherwise
 */
bool CheckKeyFile(IN CGlobalEnumerations::KeyFile_Type_e eKeyType,IN const char* f_pcMountPoint);



#ifdef USE_VALIDATE_AND_CONVERT2REGION_LANGUAGE


/*!
 * \brief adjust meta data in dependency of region e.g. China and GBK
 *
 * \param [in] u8RegionType: e.g. China, North america, Europe ...
 * \param [in] f_cMetaData: data to be adjusted
 *
 * \return void
 */
GENSTRING ValidateAndConvertToRegionLanguage (GENSTRING f_cFSType, const GENSTRING &f_cMetaData, tU8 marketingRegion);

#endif //#ifdef USE_VALIDATE_AND_CONVERT2REGION_LANGUAGE


/*!
 * \brief checks if USB-Stick,SD-Card at HUB or internal SD-card and Nullpointer check
 *
 * \param [in] pName e.g. sda1,sdb1 or mmcblk...
 *
 * \return [bool] true if valid false if not valid type
 */
bool IsValidMscType( tCString pDevName, tU8 u8Offset );



/*!
 * \brief mediaplayer has to know if drive is e.g. sr0,sg0 or sr1, sg1 or sr0,sg1 etc.
 *
 * \param [OUT] strSRx and strSGx hold the corresponding string mentioned in description above
 *
 * \return [bool] TRUE:Found, FALSE: Not found
 */

tBool bFindScsiCDDrive(OUT std::string &strSRx, OUT std::string &strSGx);


/*!
 * \brief Form serialId from Vendorid , productid and sysname/unique number
 *
 * \param [in] vendorID of the usb device
 * \param [in] productID of the usb device
 * \param [out] serialId a unique string formed from vendorid,productid and sysname/unique number
 *
 * \return [void]
 */
void FormSerialIdFromVendorIdProductIdSysname(IN const uint16_t &vendorID, IN const uint16_t &productID, IN GENSTRING &sysName,OUT GENSTRING &serialId);


/*!
 * \brief make serial id unique
 *
 * \param [inout] Append sysname/unique number to serial id to make it unique
 *
 * \return [void]
 */
void makeSerialIdUnique(IN GENSTRING &sysName, INOUT GENSTRING &serialId);

#endif // UTILS_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
