/*-----------------------------------------------------------------------------*
 * utils.cpp                                                                   *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2010 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file utils.cpp
 *
 * \brief This file holds implementation of common utility functions
 *
 * \version 11.08.2012, Negi, Sunder (MontaVista), initial version
 * \version 14.08.2012, Negi, Sunder (MontaVista), changes for logging to usb
 *
 *-----------------------------------------------------------------
 *                                    development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch)
 *          -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "DeviceManager_ErrorCodes.h"
#include "common/utils.h"

extern "C"
{
#include <mntent.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/statfs.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ftw.h>
#include <unistd.h> //for GCC 4.7.3 
#include <dirent.h>

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include <blkid/blkid.h>
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

#include <libudev.h>

#include <libusb-1.0/libusb.h>
}

#include <iostream>
#include <fstream>

#include "config/ConfigurationManager.h"
#include "AutomounterAdaptTextIF.h"

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_UDEVMANAGER
#include "trcGenProj/Header/utils.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

//#include <gst/gst.h>


using namespace std;

int err;

/*-----------------------------------------------------------------------------*
 * defines                                                                *
 *-----------------------------------------------------------------------------*/
#define MAXSIZESTRING 1024

#ifdef USE_VALIDATE_AND_CONVERT2REGION_LANGUAGE
#define IS_EQUAL   ( 0)
#endif

/*-----------------------------------------------------------------------------*
 * local typedefs                                                                *
 *-----------------------------------------------------------------------------*/
typedef struct CD_sDevName_tag
{
    //tBool bDevFound;
    // tBool bDefaultDevNameUsed;
    //tBool bDevCouldBeOpened;
#define CD_DEV_NAME_LEN            32
    char szDevName[CD_DEV_NAME_LEN];

    tBool bSRFound;
    tBool bSGFound;
    tBool bDefaultSRNameUsed;
    tBool bDefaultSGNameUsed;
    tBool bSRCouldBeOpened;
    tBool bSGCouldBeOpened;
    char szDevSRName[CD_DEV_NAME_LEN];
    char szDevSGName[CD_DEV_NAME_LEN];

} CD_tsDevName;


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST //TODO(KMV5COB):Remove this and add mocks for udev auotmounter and blkid to avoid compilation issues.
/*-----------------------------------------------------------------------------*
 * local functions                                                                *
 *-----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------*
 * const char *ReadProductNameFromUSBIDS (const char *f_sVendorID,             *
 *              const char *f_sProductID)                                      *
 *-----------------------------------------------------------------------------*/
const char *ReadProductNameFromUSBIDS (const char *f_sVendorID,
                                       const char *f_sProductID)
{
    ETG_TRACE_USR3 (("ReadProductNameFromUSBIDS: Begin"));
    char l_sVendorID[1024];
    char l_sProductID[1024];


    const char *l_cProductName = NULL;

    if((NULL != f_sVendorID) && (NULL != f_sProductID))
    {
        //care for case if f_sVendorID or f_sProductID would be not null terminated
        strncpy(l_sVendorID, f_sVendorID,  sizeof(l_sVendorID)-1 ); //1024-1=1023 is last index
        strncpy(l_sProductID,f_sProductID, sizeof(l_sProductID)-1);
        l_sVendorID[sizeof(l_sProductID)-1]  = '\0';
        l_sProductID[sizeof(l_sProductID)-1] = '\0';

        if ((0 != strlen (l_sVendorID) ) && (0 != strlen (l_sProductID)) )
        {
            FILE *l_oUSBIDS_file;

            if (NULL == (l_oUSBIDS_file = fopen (USB_IDS_FILE, "r")))
            {
                err = errno;
                ETG_TRACE_ERR (("ReadProductNameFromUSBIDS: [ERROR] 'fopen' failed [%s]",
                                strerror (err)));
            }
            else
            {
                char *l_cLine  = NULL;
                char *l_cIndex = NULL;
                size_t l_iLen;
                bool l_bVendorIDFound = false;

                // find vendorid first
                while (!feof(l_oUSBIDS_file))
                {
                    getline (&l_cLine, &l_iLen, l_oUSBIDS_file);

                    if (false == l_bVendorIDFound)
                    {
                        if (NULL != (l_cIndex = strstr (l_cLine, l_sVendorID)))
                        {
                            ETG_TRACE_USR4 (("ReadProductNameFromUSBIDS: Vendor ID found"));

                            l_bVendorIDFound = true;
                        }
                        else
                        {
                            // Continue checking for VendorID
                        } // if (vendor id found)
                    }
                    else
                    {
                        if (NULL != (l_cIndex = strstr (l_cLine, l_sProductID)))
                        {
                            ETG_TRACE_USR4 (("ReadProductNameFromUSBIDS: Product ID found"));
                            //Fix for the reset issue. If the product id is found in the usbids file
                            //and product name is not there. strstr will give NULL.
                            //So this null check will prevent null to be passed to strdup function
                            char *l_ctempProductName = strstr (l_cIndex, " ");
                            if(NULL != l_ctempProductName)
                            {
                                l_cProductName = strdup(l_ctempProductName);
                                ETG_TRACE_USR1 (("ReadProductNameFromUSBIDS: Product name: %s",
                                                 l_cProductName));
                            }
                            else
                            {
                                ETG_TRACE_USR1 (("ReadProductNameFromUSBIDS: Product name Not available"));

                            }

                            break;
                        }
                        else
                        {
                            // Continue checking for ProductID
                        } // if (product id found)
                    }// if (l_bVendorIDFound)

                    // Free memory
                    free (l_cLine);
                    l_cLine = NULL;
                } // while (USB_IDS_FILE does not end)

                // Close the file
                fclose (l_oUSBIDS_file);
            } // if (USB_IDS_FILE failed to open)
        }
        else
        {
            ETG_TRACE_ERR (("ReadProductNameFromUSBIDS: [ERROR] Invalid parameters strlen"));
        } // if (f_pCurDevice is not NULL)
    }
    else
    {
        ETG_TRACE_ERR (("ReadProductNameFromUSBIDS: [ERROR] Invalid parameters - NULL"));
    }

    ETG_TRACE_USR3 (("ReadProductNameFromUSBIDS: End"));

    return l_cProductName;
}

/*-----------------------------------------------------------------------------*
 * bool IsSDCard(struct udev_device *f_poDev)                                  *
 *-----------------------------------------------------------------------------*/
bool IsSDCard(struct udev_device *f_poDev)
{
    ETG_TRACE_USR3 (("IsSDCard: Begin"));

    /* Assuming that if the device is a card, then the parent
     * should be a Card Reader
     */
    bool l_bIsSDCard = false;

    if (NULL != f_poDev)
    {
        char product[MAXSIZESTRING];

        if (NULL != udev_device_get_sysattr_value(f_poDev,"product"))
        {
            ETG_TRACE_COMP(("IsSDCard: Product value found"));

            strncpy (product, udev_device_get_sysattr_value(f_poDev,"product"), sizeof(product));
            product[tInt(sizeof(product)-1)]= '\0';

            ETG_TRACE_USR4 (("IsSDCard: Parent device node Product = %s", product));

            if ((strstr (product, "Reader")) || (strstr (product, "reader")))
            {
                ETG_TRACE_USR4(("IsSDCard: Parent is a Card Reader"));

                l_bIsSDCard = true;
            }
            else
            {
                ETG_TRACE_USR4(("IsSDCard: Parent is Not a Card Reader"));
            }
        }
        else
        {
            ETG_TRACE_USR4(("IsSDCard: Product value not found"));

            ETG_TRACE_USR2 (("IsSDCard: Checking usb.ids file for USB product name"));

            /* For some card readers the product name is found from usb.ids file
             * This is another way to check out whether a usb_device is an SD Card reader.
             *
             * TODO: Find a more certain way to get SD card property for a device
             */
            const char *l_sVendorID  = udev_device_get_sysattr_value(f_poDev, "idVendor");
            const char *l_sProductID = udev_device_get_sysattr_value(f_poDev, "idProduct");

            const char *f_cProductName = ReadProductNameFromUSBIDS (l_sVendorID, l_sProductID);

            if (NULL != f_cProductName)
            {
                ETG_TRACE_USR1 (("IsSDCard: f_cProductName = %s", f_cProductName));

                if (('\0' != *(f_cProductName+0))
                        && GENSTRING(f_cProductName).contains(GENSTRING("reader"), CASE_INSENSITIVE))
                {
                    ETG_TRACE_COMP (("IsSDCard: Product name does contain 'reader'"));

                    l_bIsSDCard = true;
                }
                else
                {
                    ETG_TRACE_COMP (("IsSDCard: Product name does not contain 'reader'"));
                }
                free((void*)f_cProductName); // sim4hi: fix for Coverity CID-17035
            }
            else
            {
                ETG_TRACE_USR1 (("IsSDCard: Product not found in usb.ids file"));
            }
        }
    }
    else
    {
        ETG_TRACE_ERR (("IsSDCard: [ERROR] f_poDev is NULL"));
    } // if (f_pCurDevice is NULL)

    ETG_TRACE_USR3 (("IsSDCard: End"));

    return l_bIsSDCard;
}



/*-----------------------------------------------------------------------------*
 * CGlobalEnumerations::DEVICE_TYPE_E CheckDeviceType(                         *
 *              struct udev_device *f_poDev,                                   *
 *              CGlobalEnumerations::DEVICE_TYPE_E& f_eDeviceType)             *
 *-----------------------------------------------------------------------------*/
int CheckDeviceType (struct udev_device *f_poDev,
                     CGlobalEnumerations::DEVICE_TYPE_E& f_eDeviceType)
{
    ETG_TRACE_USR3 (("CheckDeviceType: Begin"));

    int l_iError = DEVICEMANAGER_OK;

    f_eDeviceType = CGlobalEnumerations::DTY_UNKNOWN;

    if (NULL != f_poDev)
    {
        // If device belongs to 'block' subsystem then its a storage device
        if (!strcmp ("block", udev_device_get_subsystem(f_poDev)))
        {
            ETG_TRACE_USR1 (("CheckDeviceType: Device is an Block Device"));
            f_eDeviceType = CGlobalEnumerations::DTY_SD;
        }
        else
        {
            // Check for other device types
        }
    }
    else
    {
        ETG_TRACE_FATAL (("CheckDeviceType: [ERROR] f_pCurDevice is NULL"));
        l_iError = DEVICEMANAGER_ERROR_NULL_POINTER;
    }

    ETG_TRACE_USR3 (("CheckDeviceType: End"));

    return l_iError;
}


/*-----------------------------------------------------------------------------*
 * int FillStorageInfo(const char *f_pcMountPoint, StorageInfo_t& f_oInfo)     *
 *-----------------------------------------------------------------------------*/
int FillStorageInfo(IN const char *f_pcMountPoint, OUT StorageInfo_t& f_oInfo)
{
    ETG_TRACE_USR3 (("FillStorageInfo: Begin"));

    int l_iError = DEVICEMANAGER_OK;

    f_oInfo.used_space = 0;
    f_oInfo.free_space = 0;
    f_oInfo.total_size = 0;

    if (f_pcMountPoint && (0 != strlen(f_pcMountPoint) ))
    {
        //Structure statfs() gives information about a mounted file system.
        struct statfs l_fsb;

        ETG_TRACE_USR4 (("FillStorageInfo: Mount point: %s", f_pcMountPoint));

        if (0 == statfs(f_pcMountPoint, &l_fsb))
        {
            // cast to unsigned is OK because: f_bsize is defined as __SWORD_TYPE (-> int #if __WORDSIZE == 32)
            // Assumption: 'optimal transfer block size' is positiv (-> Fields that are undefined for a particular file system are set to 0)
            unsigned long long l_llbsize = (unsigned int)l_fsb.f_bsize;
            unsigned long long l_llbfree = l_fsb.f_bfree;
            unsigned long long l_llblocks = l_fsb.f_blocks;

            f_oInfo.free_space = (l_llbfree * l_llbsize);
            f_oInfo.total_size = (l_llblocks * l_llbsize);
            f_oInfo.used_space = (f_oInfo.total_size - f_oInfo.free_space);

            ETG_TRACE_USR4 (("FillStorageInfo: Total blocks = %u", l_llblocks));
            ETG_TRACE_USR4 (("FillStorageInfo: Block Size   = %u", l_llbsize));
            ETG_TRACE_USR4 (("FillStorageInfo: Free blocks  = %u", l_llbfree));
            ETG_TRACE_USR4 (("FillStorageInfo: Free space   = %u (bytes)", (tU32)f_oInfo.free_space));  //koe2hi cast because if ulonglong>tU32
            ETG_TRACE_USR4 (("FillStorageInfo: Used space   = %u (bytes)", (tU32)f_oInfo.used_space));  //we might get a crash with ETG
            ETG_TRACE_USR4 (("FillStorageInfo: Total space  = %u (bytes)", (tU32)f_oInfo.total_size));
        }
        else
        {
            err = errno;

            ETG_TRACE_ERR (("FillStorageInfo: [ERROR] 'statfs' failed [%s]", strerror (err)));
            l_iError = DEVICEMANAGER_ERROR_UNKNOWN;
        }
    }
    else
    {
        ETG_TRACE_ERR (("FillStorageInfo: [ERROR] Invalid mount point"));
        l_iError = DEVICEMANAGER_ERROR_INVALID_PARAM;
    }

    ETG_TRACE_USR3 (("FillStorageInfo: End"));

    return l_iError;
}

/*-----------------------------------------------------------------------------*
 * tU64 FillStorageInfoSize( struct udev_device *f_pdev )                            *
 *-----------------------------------------------------------------------------*/
tU64 FillStorageInfoSize( struct udev_device* f_pdev )
{
    tU64                size = OSAL_C_U64_MAX;  // default: error value
    static const int    kSectorSize = 512;

    if( f_pdev != NULL )
    {
        const char* size_attr = udev_device_get_sysattr_value( f_pdev, "size" );
        if( size_attr )
        {
            size = (tU64)atoll( size_attr );    // udev_device_get_sysattr_value(..."size") is supposed to return only a positiv size for SD Card devices
            size = size * kSectorSize;
        }
    }
    return size;
}

/*-----------------------------------------------------------------------------*
 * bool IsStorageDevice(struct udev *f_pUDev, struct udev_device *f_pDev)      *
 *-----------------------------------------------------------------------------*/
bool IsStorageDevice(struct udev *f_pUdev, struct udev_device *f_pDev)
{
    ETG_TRACE_USR3 (("IsStorageDevice: Begin"));

    bool l_bIsSupportsStorage = false;

    if (NULL != f_pDev)
    {
        unsigned int bNumInterfaces      = 0;
        unsigned int bConfigurationValue = 0;
        char str[64];
        memset( str, '\0', sizeof(str) );

        ETG_TRACE_USR4 (("IsStorageDevice: Syspath = %s",
                         udev_device_get_syspath (f_pDev)));

        if (NULL != udev_device_get_sysattr_value(f_pDev,"bNumInterfaces"))
        {
            strcpy (str, udev_device_get_sysattr_value(f_pDev,
                                                       "bNumInterfaces"));
            sscanf (str, "%u", &bNumInterfaces);
        }

        if (NULL != udev_device_get_sysattr_value(f_pDev,"bConfigurationValue"))
        {
            strcpy (str, udev_device_get_sysattr_value(f_pDev,
                                                       "bConfigurationValue"));
            sscanf (str, "%u", &bConfigurationValue);
        }

        ETG_TRACE_USR4 (("IsStorageDevice: bNumInterfaces      = %d",
                         bNumInterfaces));
        ETG_TRACE_USR4 (("IsStorageDevice: bConfigurationValue = %d",
                         bConfigurationValue));

        char interface[MAXSIZESTRING];
        struct udev_device * l_dev = NULL;

        for( unsigned int i = 0; i < bNumInterfaces; i++)
        {
            // generate interface path
            snprintf (interface,MAXSIZESTRING, "%s/%s:%u.%u", udev_device_get_syspath (f_pDev),
                      udev_device_get_sysname (f_pDev), bConfigurationValue, i);

            l_dev = udev_device_new_from_syspath (f_pUdev, interface);

            if (NULL != l_dev)
            {
                ETG_TRACE_USR4 (("IsStorageDevice: Interface = %s", interface));

                const char *driver = udev_device_get_driver(l_dev); //returns string or null

                if(NULL != driver)
                {
                    if (0 != strlen (driver)) //ok: called with '\0' terminated string always according to doc of udev_device_get_driver
                    {
                        ETG_TRACE_USR3 (("IsStorageDevice: Driver    = %s", driver));

                        // NOTE: the better way could be to check for device class and subclass values
                        if (0 == strcasecmp (driver, "usb-storage"))
                        {
                            l_bIsSupportsStorage = true;
                        }
                    }
                    else
                    {
                        ETG_TRACE_USR4(("[WARNING]strlen(driver) == 0"));
                    }
                }
                else
                {
                    ETG_TRACE_USR4(("[WARNING]driver == NULL")); //check to use fatal trace here
                }

                // Cleanup
                udev_device_unref (l_dev);
                l_dev = NULL;
            } // if (NULL != l_dev)
            else
            {
                err = errno;

                ETG_TRACE_ERR (("IsStorageDevice: [ERROR] 'udev_device_new_from_syspath' failed [%s]",
                                strerror (err)));
            }
        } // for (all interfaces)
    }
    else
    {
        ETG_TRACE_ERR (("IsStorageDevice: [ERROR] f_pDev is NULL"));
    }

    ETG_TRACE_USR3 (("IsStorageDevice: End"));

    return l_bIsSupportsStorage;
}

/*-----------------------------------------------------------------------------*
 * CGlobalEnumerations::PT_Type_e FindPartitionTableType(                      *
 *              const char *f_pcBlockDevNode);                                 *
 *-----------------------------------------------------------------------------*/
CGlobalEnumerations::PT_Type_e FindPartitionTableType (const char *f_pcBlockDevNode)
{
    ETG_TRACE_USR3 (("FindPartitionTableType: Begin"));

    int l_iErrorStatus = DEVICEMANAGER_OK;

    CGlobalEnumerations::PT_Type_e pttype = CGlobalEnumerations::PT_UNKNOWN;

    if ((NULL != f_pcBlockDevNode)
            && ('\0' != *(f_pcBlockDevNode+0)))
    {
        blkid_probe pr;
        const char *l_cData = NULL;
        const char *l_cName = NULL;
        int nvals = 0, n;
        size_t len;
        int fd;

        pr = blkid_new_probe();

        ETG_TRACE_USR4 (("FindPartitionTableType: Device = %s", f_pcBlockDevNode));

        // Check whether device is valid or not
        fd = open (f_pcBlockDevNode, O_RDONLY);

        if ((fd < 0) || (pr == NULL))
        {
            err = errno;

            ETG_TRACE_ERR (("FindPartitionTableType: [ERROR] Opening device or 'blkid_probe' failed [%s]",
                            strerror (err)));

            l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
#ifdef BUGFIX_CLOSE
            if(fd >= 0)
            {
                close(fd);
            }
            if(NULL != pr)
            {
                blkid_free_probe(pr);
            }
#endif
        }
        else
        {
            if (0 > blkid_probe_set_device (pr, fd, (long long)0, (long long)0))
            {
                err = errno;

                ETG_TRACE_ERR (("FindPartitionTableType: [ERROR] 'blkid_probe_set_device' failed [%s]",
                                strerror (err)));

                l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
            }
            else
            {
                if (-1 == blkid_probe_enable_partitions (pr, 1))
                {
                    err = errno;

                    ETG_TRACE_ERR (("FindPartitionTableType: [ERROR] 'blkid_probe_enable_partitions' failed [%s]",
                                    strerror (err)));

                    l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                }
                else
                {
                    if (-1 == blkid_probe_set_partitions_flags(pr, (1 << 2) /*BLKID_PARTS_ENTRY_DETAILS*/))
                    {
                        err = errno;

                        ETG_TRACE_ERR (("FindPartitionTableType: [ERROR] 'blkid_probe_set_partitions_flags' failed [%s]",
                                        strerror (err)));

                        l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                    }
                    else
                    {
                        if (-1 == blkid_probe_enable_superblocks(pr, 1))
                        {
                            ETG_TRACE_ERR (("FindPartitionTableType: [ERROR] 'blkid_probe_enable_superblocks' failed [%s]",
                                            strerror (err)));;

                            l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                        }
                        else
                        {
                            if (0 > blkid_do_safeprobe(pr))
                            {
                                err = errno;

                                ETG_TRACE_ERR (("FindPartitionTableType: [ERROR] 'blkid_do_safeprobe' failed [%s]",
                                                strerror (err)));

                                l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                            }
                            else
                            {
                                char strTraceOut[MAXSIZESTRING];

                                // Display the values
                                nvals = blkid_probe_numof_values(pr);

                                for (n = 0; n < nvals; n++)
                                {
                                    if (blkid_probe_get_value(pr, n, &l_cName, (const char **)&l_cData, &len))
                                        continue;
                                    snprintf( strTraceOut, MAXSIZESTRING,"%s=\"%s\"", l_cName, (const char*)l_cData);

                                    ETG_TRACE_USR4 (("GetPartitionDetails: DATA: %s", strTraceOut));

                                    if (0 == strcmp(l_cName, "PTTYPE"))
                                    {
                                        // device has a partition table
                                        ETG_TRACE_COMP(("FindPartitionTableType: Device has a partition table"));

                                        GENSTRING ptable = GENSTRING((const char*)l_cData);

                                        ETG_TRACE_USR1 (("FindPartitionTableType: '%s' partition table", (const char *)l_cData));

                                        if (GENSTRING("msdos") == ptable || GENSTRING("dos") == ptable) {
                                            pttype = CGlobalEnumerations::PT_MSDOS;
                                        } else if (GENSTRING("gpt") == ptable) {
                                            pttype = CGlobalEnumerations::PT_GPT;
                                        } else if (GENSTRING("guid") == ptable) {
                                            pttype = CGlobalEnumerations::PT_GUID;
                                        } else {
                                            ETG_TRACE_USR1 (("FindPartitionTableType: UNKNOWN partition table"));
                                        }
                                    }
                                } // for (all values for the partition)
                            } // if (blkid_do_safeprobe succeed)
                        } // if (blkid_probe_enable_superblocks succeed)
                    } // if (blkid_probe_set_partitions_flags succeed)
                } // if (blkid_probe_enable_partitions succeed)
            } // if (blkid_probe_set_device succeed)

            // Free the file descriptor
            close(fd);

            // Free the blkid probe
            blkid_free_probe(pr);
        } // if (open() succeeds)
    }
    else
    {
        ETG_TRACE_ERR (("FindPartitionTableType: [ERROR] f_cBlockDevNode is Empty"));

        l_iErrorStatus = DEVICEMANAGER_ERROR_INVALID_PARAM;
    }

    ETG_TRACE_COMP(("FindPartitionTableType l_iErrorStatus=0x%x",l_iErrorStatus));
    ETG_TRACE_USR3 (("FindPartitionTableType: End"));

    return pttype;
}


/*-----------------------------------------------------------------------------*
 * bool IsGPTPartitionTypeFound(                      *
 *              const char *f_pcBlockDevNode);                                 *
 *-----------------------------------------------------------------------------*/
bool IsGPTPartitionTypeFound (const char *f_pcBlockDevNode)
{
    ETG_TRACE_USR3 (("IsGPTPartitionTypeFound: Begin"));

    int l_iErrorStatus = DEVICEMANAGER_OK;
    bool retval = false;
    CGlobalEnumerations::PT_Type_e pttype = CGlobalEnumerations::PT_UNKNOWN;

    if ((NULL != f_pcBlockDevNode)
            && ('\0' != *(f_pcBlockDevNode+0)))
    {
        blkid_probe pr;
        const char *l_cData = NULL;
        const char *l_cName = NULL;
        int nvals = 0, n;
        size_t len;
        int fd;

        pr = blkid_new_probe();

        ETG_TRACE_USR4 (("IsGPTPartitionTypeFound: Device = %s", f_pcBlockDevNode));

        // Check whether device is valid or not
        fd = open (f_pcBlockDevNode, O_RDONLY);

        if ((fd < 0) || (pr == NULL))
        {
            err = errno;

            ETG_TRACE_ERR (("IsGPTPartitionTypeFound: [ERROR] Opening device or 'blkid_probe' failed [%s]",
                            strerror (err)));

            l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
#ifdef BUGFIX_CLOSE
            if(fd >= 0)
            {
                close(fd);
            }
            if(NULL != pr)
            {
                blkid_free_probe(pr);
            }
#endif
        }
        else
        {
            if (0 > blkid_probe_set_device (pr, fd, (long long)0, (long long)0))
            {
                err = errno;

                ETG_TRACE_ERR (("IsGPTPartitionTypeFound: [ERROR] 'blkid_probe_set_device' failed [%s]",
                                strerror (err)));

                l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
            }
            else
            {
                if (-1 == blkid_probe_enable_partitions (pr, 1))
                {
                    err = errno;

                    ETG_TRACE_ERR (("IsGPTPartitionTypeFound: [ERROR] 'blkid_probe_enable_partitions' failed [%s]",
                                    strerror (err)));

                    l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                }
                else
                {
                    if (-1 == blkid_probe_set_partitions_flags(pr, (1 << 2) /*BLKID_PARTS_ENTRY_DETAILS*/))
                    {
                        err = errno;

                        ETG_TRACE_ERR (("IsGPTPartitionTypeFound: [ERROR] 'blkid_probe_set_partitions_flags' failed [%s]",
                                        strerror (err)));

                        l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                    }
                    else
                    {
                        if (-1 == blkid_probe_enable_superblocks(pr, 1))
                        {
                            ETG_TRACE_ERR (("IsGPTPartitionTypeFound: [ERROR] 'blkid_probe_enable_superblocks' failed [%s]",
                                            strerror (err)));;

                            l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                        }
                        else
                        {
                            if (0 > blkid_do_safeprobe(pr))
                            {
                                err = errno;

                                ETG_TRACE_ERR (("IsGPTPartitionTypeFound: [ERROR] 'blkid_do_safeprobe' failed [%s]",
                                                strerror (err)));

                                l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                            }
                            else
                            {
                                char strTraceOut[MAXSIZESTRING];

                                // Display the values
                                nvals = blkid_probe_numof_values(pr);

                                for (n = 0; n < nvals; n++)
                                {
                                    if (blkid_probe_get_value(pr, n, &l_cName, (const char **)&l_cData, &len))
                                        continue;
                                    snprintf( strTraceOut, MAXSIZESTRING,"%s=\"%s\"", l_cName, (const char*)l_cData);

                                    ETG_TRACE_USR4 (("IsGPTPartitionTypeFound: DATA: %s", strTraceOut));

                                    if (0 == strcmp(l_cName, "PTTYPE"))
                                    {
                                        // device has a partition table
                                        ETG_TRACE_COMP(("IsGPTPartitionTypeFound: Device has a partition table"));

                                        GENSTRING ptable = GENSTRING((const char*)l_cData);

                                        ETG_TRACE_USR1 (("IsGPTPartitionTypeFound: '%s' partition table", (const char *)l_cData));

                                        if (GENSTRING("msdos") == ptable || GENSTRING("dos") == ptable) {
                                            pttype = CGlobalEnumerations::PT_MSDOS;
                                        } else if (GENSTRING("gpt") == ptable) {
                                            pttype = CGlobalEnumerations::PT_GPT;
                                        } else if (GENSTRING("guid") == ptable) {
                                            pttype = CGlobalEnumerations::PT_GUID;
                                        } else {
                                            ETG_TRACE_USR1 (("IsGPTPartitionTypeFound: UNKNOWN partition table"));
                                        }

                                        if(CGlobalEnumerations::PT_GPT == pttype)
                                        {
                                            retval = true;
                                        }
                                    }
                                } // for (all values for the partition)
                            } // if (blkid_do_safeprobe succeed)
                        } // if (blkid_probe_enable_superblocks succeed)
                    } // if (blkid_probe_set_partitions_flags succeed)
                } // if (blkid_probe_enable_partitions succeed)
            } // if (blkid_probe_set_device succeed)

            // Free the file descriptor
            close(fd);

            // Free the blkid probe
            blkid_free_probe(pr);
        } // if (open() succeeds)
    }
    else
    {
        ETG_TRACE_ERR (("IsGPTPartitionTypeFound: [ERROR] f_cBlockDevNode is Empty"));

        l_iErrorStatus = DEVICEMANAGER_ERROR_INVALID_PARAM;
    }

    ETG_TRACE_COMP(("IsGPTPartitionTypeFound l_iErrorStatus=0x%x",l_iErrorStatus));
    ETG_TRACE_USR3 (("IsGPTPartitionTypeFound: End"));

    return retval;
}

/*-----------------------------------------------------------------------------*
 * bool CheckIfMounted(const char *f_cMtabFile, const char *f_pcDeviceNode,    *
 *              GENSTRING& f_cMountPoint)                                        *
 *-----------------------------------------------------------------------------*/
int CheckIfMounted(const char *f_cMtabFile,CDevice *f_pCurDevice)
{    
    ETG_TRACE_USR4(("Begin: CUDevManager::CheckIfMounted"));
    int l_iReturnValue = DEVICEMANAGER_OK;
    
    if (NULL == f_cMtabFile || NULL == f_pCurDevice)
    {
        ETG_TRACE_ERR(("CheckIfMounted: Invalid arguments"));
        l_iReturnValue = DEVICEMANAGER_ERROR_UNKNOWN;
    }
    else
    {
        if (-1 == f_pCurDevice->m_iValidPartitionNo)
        {
            ETG_TRACE_ERR(("CheckIfMounted: Invalid partition number = %d", f_pCurDevice->m_iValidPartitionNo));
            l_iReturnValue = DEVICEMANAGER_ERROR_UNKNOWN;
        }
        else
        {
            ETG_TRACE_USR2(("CheckIfMounted: Valid partition number = %d", f_pCurDevice->m_iValidPartitionNo));

            FILE            *file;
            struct mntent   *entry;

            file = setmntent(f_cMtabFile, "r");
            if (NULL == file)
            {
                ETG_TRACE_ERR(("CheckIfMounted: Error opening %s", f_cMtabFile));
                l_iReturnValue = DEVICEMANAGER_ERROR_UNKNOWN;
            }
            else
            {
                ETG_TRACE_USR3(("CheckIfMounted: File opened successfully : %s", f_cMtabFile));
                GENSTRING l_cValidPartitionNode = "";  //Device Node
                l_cValidPartitionNode = f_pCurDevice->m_cBlockDevnode;
                ETG_TRACE_USR3 (("CheckIfMounted: l_cValidPartitionNode = %s", l_cValidPartitionNode.toStdString().c_str()));
                l_iReturnValue = DEVICEMANAGER_ERROR_NOTVALID; //to prepare a retrial

                ETG_TRACE_USR4(("CheckIfMounted:%s (l_cValidPartitionNode)",l_cValidPartitionNode.toStdString().c_str()));
                for(entry = getmntent(file); entry; entry = getmntent(file))
                {
                    if(l_cValidPartitionNode == entry->mnt_fsname)      //lint !e1702 Info 1702: operator 'operator==' is both an ordinary function '...' and a member function '...'
                    {
                        ETG_TRACE_USR3 (("CheckIfMounted: l_cMountPoint         = %s", entry->mnt_dir));
                        ETG_TRACE_USR3 (("CheckIfMounted: l_cValidPartitionNode = %s", entry->mnt_fsname));
                        f_pCurDevice->m_cMountPoint = entry->mnt_dir;
                        f_pCurDevice->m_iIsMounted  = 1;
                        l_iReturnValue = DEVICEMANAGER_OK;
                        break;
                    }
                }
                endmntent(file);
            }
        } // if (-1 == f_pCurDevice->m_iValidPartitionNo)
    }
    if (DEVICEMANAGER_OK == l_iReturnValue)
    {
        ETG_TRACE_USR4(("CheckIfMounted: Device mountpoint found"));
    }
    ETG_TRACE_USR4(("End  : CheckIfMounted"));
    
    return l_iReturnValue;
}


/*-----------------------------------------------------------------------------*
 * int ListConnectedDevices()                                                  *
 * ----------------------------------------------------------------------------*/
static void print_devs( libusb_device* const(*devs) )
{  
    libusb_device *dev;
    int i = 0;

    while ((dev = devs[i++]) != NULL) {
        struct libusb_device_descriptor desc;
        int r = libusb_get_device_descriptor(dev, &desc);
        if (r < 0) {
            ETG_TRACE_ERR (("ListConnectedDevices: [ERROR] failed to get device descriptor"));
            return;
        }

        char str[MAXSIZESTRING];
        snprintf (str, MAXSIZESTRING, "%04x:%04x (bus %d, device %d)",
                  desc.idVendor, desc.idProduct,
                  libusb_get_bus_number(dev), libusb_get_device_address(dev));

        ETG_TRACE_USR1 (("ListConnectedDevices: Seen Device: %s", str));
    }
}

int ListConnectedDevices()
{
    ETG_TRACE_USR3 (("ListConnectedDevices: Begin"));

    libusb_device **devs;
    int r;
    ssize_t cnt;

    r = libusb_init(NULL);
    if (r < 0)
        return r;

    cnt = libusb_get_device_list(NULL, &devs);
    if (cnt < 0)
        return (int) cnt;

    print_devs(devs);
    libusb_free_device_list(devs, 1);

    libusb_exit(NULL);

    ETG_TRACE_USR3 (("ListConnectedDevices: End"));

    return 0;
}

int FindTotalValidPartitions(const char* f_BlockDevnode,int &f_u8TotalValidPartitons)
{

    ETG_TRACE_USR3 (("FindTotalValidPartitions: Begin"));
    int l_iErrorStatus = DEVICEMANAGER_OK;
    f_u8TotalValidPartitons = 0;
    //To check if the partition is EFI
    bool isEFI =false;
    // Store label
    const char *l_plabel;


    blkid_probe l_probe = blkid_new_probe_from_filename(f_BlockDevnode);
    if (!l_probe)
    {
        ETG_TRACE_USR3 (("FindTotalValidPartitions: [ERROR] Opening device %s failed",f_BlockDevnode));
        l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
    }
    else
    {

        // Get number of partitions
        blkid_partlist l_partlist;
        int nparts = 0; // sim4hi: fix for Coverity CID-128652
        int i = 0;

        l_partlist = blkid_probe_get_partitions(l_probe);
        if(l_partlist)
        {
            nparts = blkid_partlist_numof_partitions(l_partlist);
        }
        else
        {
            ETG_TRACE_ERR (("FindTotalValidPartitions: l_partlist is NULL"));
        }            

        if (nparts <= 0)
        {
            ETG_TRACE_USR3 (("FindTotalValidPartitions: Number partition is zero"));
            l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
        }
        const int l_isize = 20;

        ETG_TRACE_USR3 (("FindTotalValidPartitions: Number of partitions:%d", nparts));
        blkid_free_probe(l_probe);


        // Store file system type
        const char *l_pfilesstemtype;

        for (i = 0; i < nparts; i++)
        {
            char l_cDevName[l_isize];

            snprintf(l_cDevName,l_isize,"%s%d", f_BlockDevnode, (i+1));

            l_probe = blkid_new_probe_from_filename(l_cDevName);
            if(l_probe)
            {
                if (blkid_do_probe(l_probe) == 0)
                {
                    if (blkid_probe_lookup_value(l_probe, "LABEL", &l_plabel, NULL) == 0)
                    {
                        if (l_plabel)
                        {
                            ETG_TRACE_USR3 (("FindTotalValidPartitions: LABEL=%s", l_plabel));

                            if(GENSTRING("EFI") == GENSTRING(l_plabel))
                            {
                                isEFI = true;
                            }
                        }
                        else
                        {
                            ETG_TRACE_ERR (("FindTotalValidPartitions: l_plabel is NULL"));
                        }
                    }
                    else
                    {
                        ETG_TRACE_ERR (("FindTotalValidPartitions: blkid_probe_lookup_value() LABEL returned error"));
                    }     
                    
                    
                    if (blkid_probe_lookup_value(l_probe, "TYPE", &l_pfilesstemtype, NULL) == 0)
                    {
                        if(l_pfilesstemtype)
                        {
                            ETG_TRACE_USR3 (("FindTotalValidPartitions: TYPE=%s", l_pfilesstemtype));
                            AutomounterAdaptTextIF l_AutomounterAdaptTextIF;
                            if(TRUE == l_AutomounterAdaptTextIF.GetDeviceMountState(f_BlockDevnode,l_cDevName))
                            {
                                if(DEVICEMANAGER_OK == ConfigurationManager::GetInstance()->IsSupportedFS(l_pfilesstemtype))
                                {
                                    ETG_TRACE_USR3 (("FindTotalValidPartitions: Valid Partition found.  "));
                                    f_u8TotalValidPartitons++;
                                }
                                else
                                {
                                        ETG_TRACE_USR3 (("FindTotalValidPartitions: Invalid partitions found"));
                                }
                            }
                            else
                            {
                                ETG_TRACE_USR3 (("FindTotalValidPartitions: Unsupported Partition found.  "));
                            }
                        }
                        else
                        {
                            ETG_TRACE_ERR (("FindTotalValidPartitions: l_pfilesstemtype is NULL"));
                        }
                    }
                    else
                    {
                        ETG_TRACE_ERR (("FindTotalValidPartitions: blkid_probe_lookup_value() TYPE returned error"));
                    }

                    blkid_free_probe(l_probe);  
                }
                else
                {
                     ETG_TRACE_ERR (("FindTotalValidPartitions: blkid_do_probe() returned error"));
                }               

            }
        }
        ETG_TRACE_USR3 (("FindTotalValidPartitions: Total number of valid Partitions : %d ",f_u8TotalValidPartitons));
    }

    // Case when the disk is mac-formatted
    // The partition is shown as 'vfat' with label 'EFI' but
    // its not valid for processing, so it needs to be skipped
    if ((f_u8TotalValidPartitons > 1)
            && (IsGPTPartitionTypeFound(f_BlockDevnode)) // Mac formatted
            && isEFI)          // Label is "EFI"
    {
        // Skip the partition by decrementing the count
        ETG_TRACE_USR1 (("FindTotalValidPartitions: 'EFI' partition found, Decrementing count.."));
        f_u8TotalValidPartitons--;
    }
    ETG_TRACE_USR3 (("FindTotalValidPartitions: FINAL Total number of valid Partitions : %d ",f_u8TotalValidPartitons));
    ETG_TRACE_USR3 (("FindTotalValidPartitions: End"));
    return l_iErrorStatus;

}

/*-----------------------------------------------------------------------------*
 * int GetPartitionDetails (GENSTRING f_cPartitionNode, GENSTRING& f_cFSType,      *
 *          GENSTRING& f_cUUID, GENSTRING& f_cLabel)                               *
 *-----------------------------------------------------------------------------*/
int GetPartitionDetails (GENSTRING f_cPartitionNode, GENSTRING& f_cFSType,
                         GENSTRING& f_cUUID, GENSTRING& f_cLabel)
{
    ETG_TRACE_USR3 (("GetPartitionDetails: Begin"));

    int l_iErrorStatus = DEVICEMANAGER_OK;

    if (!f_cPartitionNode.isEmpty())
    {
        blkid_probe pr;
        const char *l_cData = NULL;
        const char *l_cName = NULL;
        int nvals = 0, n;
        size_t len;
        int fd;

        pr = blkid_new_probe();

        ETG_TRACE_COMP(("GetPartitionDetails: Device = %s", f_cPartitionNode.toStdString().c_str()));
        // Check whether device is valid or not
        fd = open(f_cPartitionNode.toStdString().c_str(), O_RDONLY);

        if ((fd < 0) || (pr == NULL))
        {
            ETG_TRACE_ERR (("GetPartitionDetails: [ERROR] Opening device %s or 'blkid probe' failed",
                            f_cPartitionNode.toStdString().c_str()));

            l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
#ifdef BUGFIX_CLOSE
            if(fd >= 0)
            {
                close(fd);
            }
            if(NULL != pr)
            {
                blkid_free_probe(pr);
            }
#endif
        }
        else
        {
            if (0 > blkid_probe_set_device(pr, fd, (long long)0, (long long)0))
            {
                ETG_TRACE_ERR (("GetPartitionDetails: [ERROR] 'blkid_probe_set_device' failed"));

                l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
            }
            else
            {
                if (-1 == blkid_probe_enable_partitions(pr, 1))
                {
                    ETG_TRACE_ERR (("GetPartitionDetails: [ERROR] 'blkid_probe_enable_partitions' failed [%s]",
                                    strerror (err)));

                    l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                }
                else
                {
                    if (-1 == blkid_probe_set_partitions_flags(pr, (1 << 2) /*BLKID_PARTS_ENTRY_DETAILS*/))
                    {
                        ETG_TRACE_ERR (("GetPartitionDetails: [ERROR] 'blkid_probe_set_partitions_flags' failed [%s]",
                                        strerror (err)));

                        l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                    }
                    else
                    {
                        if (-1 == blkid_probe_enable_superblocks(pr, 1))
                        {
                            err = errno;

                            ETG_TRACE_ERR (("GetPartitionDetails: [ERROR] 'blkid_probe_enable_superblocks' failed [%s]",
                                            strerror (err)));

                            l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                        }
                        else
                        {
                            if (0 > blkid_do_safeprobe(pr))
                            {
                                ETG_TRACE_ERR (("GetPartitionDetails: [ERROR] 'blkid_do_safeprobe' failed [%s]",
                                                strerror (err)));

                                l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
                            }
                            else
                            {
                                // Set to EMPTY
                                f_cUUID   = "";
                                f_cFSType = "";
                                f_cLabel  = "";
                                char strTraceOut[MAXSIZESTRING];

                                // Display the values
                                nvals = blkid_probe_numof_values(pr);

                                for (n = 0; n < nvals; n++)
                                {
                                    if (blkid_probe_get_value(pr, n, &l_cName, (const char **)&l_cData, &len))
                                        continue;
                                    snprintf( strTraceOut,MAXSIZESTRING, "%s=\"%s\"", (const char *)l_cName, (const char*)l_cData);

                                    ETG_TRACE_USR4 (("GetPartitionDetails: DATA: %s", strTraceOut));

                                    if (0 == strcmp(l_cName, "TYPE"))
                                    {
                                        f_cFSType = GENSTRING(l_cData);
                                    }
                                    else if (0 == strcmp(l_cName, "UUID"))
                                    {
                                        f_cUUID = GENSTRING(l_cData);
                                    }
                                    else if (0 == strcmp(l_cName, "LABEL"))
                                    {
                                        f_cLabel = GENSTRING(l_cData);
                                        ETG_TRACE_COMP(("f_cLabel: %s ",f_cLabel.toStdString().c_str()));
                                    }
                                    else if (0 == strcmp(l_cName, "PTTYPE"))
                                    {
                                        // device has a partition table
                                        ETG_TRACE_USR2 (("GetPartitionDetails: Device has a partition table"));

                                        l_iErrorStatus = DEVICEMANAGER_ERROR_NOTVALID;
                                    }
                                    else
                                    {
                                        // Do nothing
                                    }
                                } // for (all values for the partition)
                            } // if (blkid_do_safeprobe succeed)
                        } // if (blkid_probe_enable_superblocks succeed)
                    } // if (blkid_probe_set_partitions_flags succeed)
                } // if (blkid_probe_enable_partitions succeed)
            } // if (blkid_probe_set_device succeed)

            // Free the file descriptor
            close(fd);

            // Free the blkid probe
            blkid_free_probe(pr);
        } // if (open() succeeds)
    }
    else
    {
        ETG_TRACE_ERR (("GetPartitionDetails: [ERROR] f_cPartitionNode is Empty"));

        l_iErrorStatus = DEVICEMANAGER_ERROR_INVALID_PARAM;
    }

    if ((DEVICEMANAGER_ERROR_NOTVALID != l_iErrorStatus)
            && (f_cFSType.isEmpty() /*|| f_cUUID.isEmpty()*/))
    {
        ETG_TRACE_ERR (("GetPartitionDetails: [ERROR] Empty values"));

        l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
    }
#ifdef BUGFIX_HARDISK
    if ((DEVICEMANAGER_ERROR_NOTVALID == l_iErrorStatus)  && !f_cFSType.isEmpty())
    {
        ETG_TRACE_ERR (("GetPartitionDetails: Disk has filesystem type. Can be mounted"));

        l_iErrorStatus = DEVICEMANAGER_OK;
    }
#endif

    ETG_TRACE_USR3 (("GetPartitionDetails: End"));

    return l_iErrorStatus;
}

/*-----------------------------------------------------------------------------*
 * struct udev_device *GenerateUDevDevice(udev *f_poUdev,                      *
 *              const char *f_cMountPoint)                                     *
 *-----------------------------------------------------------------------------*/
struct udev_device *GenerateUDevDevice(udev *f_poUdev, const char *f_cMountPoint)
{
    if( f_cMountPoint )
    {
        ETG_TRACE_USR3(( "GenerateUdevDevice: Begin; (for MountPoint: %s)", f_cMountPoint ));
    }
    else
    {
        ETG_TRACE_USR3 (("GenerateUdevDevice: Begin"));
    }

    int l_iError = DEVICEMANAGER_OK;
    struct udev_device *l_poDev = NULL;

    if (NULL != f_cMountPoint && NULL != f_poUdev)
    {
        // Find the actual device number using the mountpoint
        struct stat l_stat;
#ifdef BUGFIX_CLOSE
        int result = stat (f_cMountPoint, &l_stat);
#else
        int fd = stat (f_cMountPoint, &l_stat);
#endif

#ifdef BUGFIX_CLOSE
        if (-1 != result)
#else
        if (-1 != fd)
#endif
        {
            // Generate the block udev_device object from the device number
            l_poDev = udev_device_new_from_devnum (f_poUdev, 'b', l_stat.st_dev);

            if (NULL != l_poDev)
            {
                ETG_TRACE_USR1 (("GenerateUdevDevice: udev_device generated successfully"));
            }
            else
            {
                err = errno;

                ETG_TRACE_ERR (("GenerateUdevDevice: [ERROR] 'udev_device_new_from_devnum' failed [%s]",
                                strerror (err)));

                l_iError = DEVICEMANAGER_ERROR_UNKNOWN;
            }
#ifdef BUGFIX_CLOSE
#else
            close (fd);
#endif
        }
        else
        {
            err = errno;

            ETG_TRACE_ERR (("GenerateUdevDevice: [ERROR] 'stat' failed [%s]",
                            strerror (err)));

            l_iError = DEVICEMANAGER_ERROR_FILE_OPEN;
        }
    }
    else
    {
        ETG_TRACE_ERR (("GenerateUdevDevice: [ERROR] f_cMountPoint is NULL"));

                l_iError = DEVICEMANAGER_ERROR_NULL_POINTER;
    }

    ETG_TRACE_COMP (("GenerateUdevDevice: l_iError = 0x%x",l_iError));
    ETG_TRACE_USR3 (("GenerateUdevDevice: End"));

    return l_poDev;
}

/*-----------------------------------------------------------------------------*
 * struct udev_device *FindBaseUSBDevice(struct udev_device *f_poDev)          *
 *-----------------------------------------------------------------------------*/
struct udev_device *FindBaseUSBDevice(struct udev_device *f_poDev)
{
    ETG_TRACE_USR3 (("FindBaseUSBDevice: Begin"));

    int l_iError = DEVICEMANAGER_OK;
    struct udev_device *l_poBaseDev = NULL;

    if (NULL != f_poDev)
    {
        // Get the parent 'usb_device' for this device
        if (GENSTRING("usb_device") != GENSTRING(udev_device_get_devtype (f_poDev)))
        {
            ETG_TRACE_USR1 (("FindBaseUSBDevice: Generating base USB device for %s",
                             udev_device_get_devnode (f_poDev)));

            // Get the parent device name
            l_poBaseDev = udev_device_get_parent_with_subsystem_devtype (f_poDev,
                                                                         "usb", "usb_device");
        }
        else
        {
            l_poBaseDev = f_poDev;
        }
    }
    else
    {
        ETG_TRACE_ERR (("FindBaseUSBDevice: [ERROR] f_poDev is NULL"));

        l_iError = DEVICEMANAGER_ERROR_INVALID_PARAM;
    }

    ETG_TRACE_COMP (("FindBaseUSBDevice: l_iError = 0x%x",l_iError));
    ETG_TRACE_USR3 (("FindBaseUSBDevice: End"));

    return l_poBaseDev;
}

/*-----------------------------------------------------------------------------*
 * int getappledeviceinfo(const char *fpath, const struct stat *sb, int tflag, *
 *          struct FTW *ftwbuf)                                                *
 *-----------------------------------------------------------------------------*/
char g_cHwPath[MAXSIZESTRING];
char g_cHiddevPath[MAXSIZESTRING];

int getappledeviceinfo(const char *fpath, const struct stat *sb, int tflag, struct FTW *ftwbuf)
{
    //usleep ((useconds_t)(USE_APPLEDEVICEINFO_DELAY*1000));

    //ETG_TRACE_USR4 (("CallBack getappledeviceinfo: %s", fpath));
    (void)sb;
    (void)tflag;
    (void)ftwbuf;

    if (strstr (fpath, "pcmC"))
    {
        g_cHwPath[0] = '\0';
        strncpy(g_cHwPath, fpath, MAXSIZESTRING-1); //-1 to care for calling with maximum size argument of 1024 byte in fpath in order to get end of string in
        return FTW_STOP;
    }

    return 0;           /* To tell nftw() to continue */
}

/*-----------------------------------------------------------------------------*
 * int GetDeviceFileName(udev_device *f_poDev, GENSTRING &f_pcAlsaHwName,        *
 *              GENSTRING &f_pcHidDevice)                                        *
 *-----------------------------------------------------------------------------*/
int GetDeviceFileName(struct udev_device *f_poDev, GENSTRING &f_pcAlsaHwName,
                      GENSTRING &f_pcHidDevice)
{
    ETG_TRACE_USR3 (("GetDeviceFileName : Inside"));

    int   flags        = FTW_PHYS|FTW_ACTIONRETVAL;
    char *l_pcTemp     = NULL;
    const char *l_cDevicePath = udev_device_get_syspath (f_poDev);
    int l_iErrorStatus = DEVICEMANAGER_OK;

    ETG_TRACE_USR4 (("GetDeviceFileName: Traversing device path %s",
                     l_cDevicePath));

    g_cHwPath[0] = '\0';
    g_cHiddevPath[0] = '\0';
    /// Traverse the syspath
    if (nftw(l_cDevicePath, getappledeviceinfo, 20, flags) == -1)//lint !e1055 !e746 Symbol undeclared although it is declared in ftw.h
    {
        err = errno;

        ETG_TRACE_ERR (("GetDeviceFileName: [ERROR] 'nftw' failed [%s]",
                        strerror (err)));

        l_iErrorStatus = DEVICEMANAGER_ERROR_UNKNOWN;
    }
    else
    {
        ETG_TRACE_USR4 (("GetDeviceFileName: hiddev full Path         = %s", g_cHiddevPath));
        ETG_TRACE_USR4 (("GetDeviceFileName: Sound Hardware full Path = %s", g_cHwPath));

        // Parse the hiddev name
        if (strlen (g_cHiddevPath)) //Nullpointer check not necessary see definition of array g_cHiddevPath[]
        {
            char l_sHidDevicePath[MAXSIZESTRING];

            ETG_TRACE_USR4 (("GetDeviceFileName: Parsing for hid device name"));

            l_pcTemp = strrchr (g_cHiddevPath, '/');
            snprintf (l_sHidDevicePath,MAXSIZESTRING, "/dev/usb%s", l_pcTemp);
            f_pcHidDevice = l_sHidDevicePath;

            ETG_TRACE_USR1 (("GetDeviceFileName: Sending Mount Point f_pcHidDevice = %s",
                             f_pcHidDevice.toStdString().c_str() ));
        }
        else
        {
            ETG_TRACE_USR1 (("GetDeviceFileName: Setting f_pcHidDevice to EMPTY"));
            f_pcHidDevice = "";
            ETG_TRACE_USR1 (("For GEN3 this is not error case - SERIAL ID of device will be filled later"));
        } //if (g_cHiddevPath is valid)

        // Parse the sound hw name
        if (strlen(g_cHwPath)) //Nullpointer check not necessary see definition of array g_cHwPath[] in code
        {
            char *l_cCardNumber   = NULL;
            char *l_cDeviceNumber = NULL;
            char l_sAlsaDeviceName[MAXSIZESTRING];
            l_sAlsaDeviceName[0] = '\0';

            ETG_TRACE_USR4 (("GetDeviceFileName: Parsing for sound hardware device name"));

            l_pcTemp = strrchr(g_cHwPath, '/');
            ETG_TRACE_USR4 (("GetDeviceFileName: Device found is %s", l_pcTemp));

            if( l_pcTemp )
            {
                l_cCardNumber   = strchr(l_pcTemp, 'C');
                l_cDeviceNumber = strchr(l_pcTemp, 'D');
            }
            else
            {
                ETG_TRACE_FATAL(("GetDeviceFileName: l_pcTemp == NULL"));
            }
            if( l_cCardNumber && l_cDeviceNumber )
            {
                snprintf(l_sAlsaDeviceName,MAXSIZESTRING, "hw:%c,%c",
                         *(l_cCardNumber + 1), *(l_cDeviceNumber+1));
            }
            else
            {
                ETG_TRACE_FATAL(("GetDeviceFileName: l_cCardNumber or  l_cDeviceNumber == NULL"));
            }
            f_pcAlsaHwName = l_sAlsaDeviceName;

            ETG_TRACE_USR1 (("GetDeviceFileName: Sending Sound Hardware Device f_pcAlsaHwName = %s",
                             f_pcAlsaHwName.toStdString().c_str()));
        }
        else
        {
            ETG_TRACE_USR1 (("GetDeviceFileName: Setting f_pcAlsaHwName to EMPTY"));

            f_pcAlsaHwName = "";

            l_iErrorStatus = DEVICEMANAGER_ERROR_NOTVALID;
        } //if (g_cHwPath is valid)
    } // if (nftw failed)

    ETG_TRACE_USR3 (("GetDeviceFileName: End"));

    return l_iErrorStatus;
}


/*-----------------------------------------------------------------------------*
 * bool CheckForLogKeyFile(const char* f_pcMountPoint)                         *
 *-----------------------------------------------------------------------------*/
bool CheckForLogKeyFile(const char* f_pcMountPoint)
{
    ETG_TRACE_USR3 (("CheckForLogKeyFile: Begin"));

    bool l_bKeyFileExists = false;
    FILE *l_pFileLogging;
    char key_filename[1024];
    char key2log[WRITEHISTORYTOUSBSTICK_KEYLEN+1];
    key2log[WRITEHISTORYTOUSBSTICK_KEYLEN] = '\0';

    snprintf (key_filename, sizeof(key_filename), "%s/%s", f_pcMountPoint,
              WRITEHISTORYTOUSBSTICK_KEYFILENAME);

    ETG_TRACE_USR4 (("CheckForLogKeyFile: Checking for file %s",
                     f_pcMountPoint));

    l_pFileLogging = fopen (key_filename,"r");

    if (NULL != l_pFileLogging)
    {
        ETG_TRACE_USR4 (("CheckForLogKeyFile: key file found"));

        // read the string equal to key length
        if (WRITEHISTORYTOUSBSTICK_KEYLEN ==
                fread(key2log, 1, WRITEHISTORYTOUSBSTICK_KEYLEN, l_pFileLogging))
        {
            if (0 == strncmp(key2log, WRITEHISTORYTOUSBSTICK_KEY,
                             WRITEHISTORYTOUSBSTICK_KEYLEN-1))
            {
                ETG_TRACE_FATAL (("[ok] !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                ETG_TRACE_FATAL (("[ok] !!!!     Logging history to usb requested       !!!!"));
                ETG_TRACE_FATAL (("[ok] !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));

                l_bKeyFileExists = true;
            }
            else
            {
                ETG_TRACE_ERR (("CheckForLogKeyFile: KEY found is wrong"));
            }
        }
        else
        {
            ETG_TRACE_ERR (("CheckForLogKeyFile: KEY not found"));
        }

        fclose (l_pFileLogging);
    }
    else
    {
        ETG_TRACE_USR1 (("CheckForLogKeyFile: key file not found in usb"));
    }

    ETG_TRACE_USR3 (("CheckForLogKeyFile: End"));

    return (l_bKeyFileExists);
}

bool CheckKeyFileExists(CGlobalEnumerations::KeyFile_Type_e eKeyType,const char* f_pcMountPoint)
{
    ETG_TRACE_USR3 (("CheckKeyFileExists: Begin"));
    bool bRet = false;
    FILE *l_pFile;
    char key_filename[1024];

    GENSTRING strKeyFile;
    GENSTRING strKeyPathFileName;

    switch(eKeyType)
    {
    case CGlobalEnumerations::KEY_LOGDGBINFO:
        strKeyFile = WRITEHISTORYTOUSBSTICK_KEYFILENAME;
        break;
    case CGlobalEnumerations::Key_LSIM_SDCARD_FAKE:
        strKeyFile = KEYFILE_FAKE_SDCARD;
        break;
    default:
        break;
    }

    snprintf (key_filename, sizeof(key_filename), "%s/%s", f_pcMountPoint,
              strKeyFile.toStdString().c_str());

    l_pFile = fopen (key_filename,"r");
    if (NULL != l_pFile)
    {
        bRet = true;
        fclose(l_pFile);
    }
    
    ETG_TRACE_USR3 (("CheckKeyFileExists: End (bRet=0x%x)",bRet));
    return bRet;
}

bool CheckKeyDirExists(IN CGlobalEnumerations::KeyFile_Type_e eKeyType,IN const char* f_pcMountPoint)
{
    ETG_TRACE_USR3 (("CheckKeyDirExists: Begin"));
    bool bRet = false;
    DIR *pDir;
    char key_dirname[1024];

    GENSTRING strKeyDir;
    GENSTRING strKeyPathDirName;

    switch(eKeyType)
    {
    case CGlobalEnumerations::KEY_LOGDGBINFO:
        strKeyDir = WRITEHISTORYTOUSBSTICK_KEYFILENAME;
        break;
    case CGlobalEnumerations::Key_LSIM_SDCARD_FAKE:
        strKeyDir = KEYFILE_FAKE_SDCARD;
        break;
    default:
        break;
    }

    snprintf (key_dirname, sizeof(key_dirname), "%s/%s", f_pcMountPoint,
              strKeyDir.toStdString().c_str());

    pDir = opendir (key_dirname);
    if (pDir != NULL)
    {
        bRet = true;
        (void) closedir (pDir);
    }

    ETG_TRACE_USR3 (("CheckKeyDirExists: End (bRet=0x%x)",bRet));
    return bRet;
}

/*-----------------------------------------------------------------------------*
 * bool CheckKeyFile(CGlobalEnumerations::KeyFile_Type_e eKeyType,             *
 *              const char* f_pcMountPoint)                                    *
 *-----------------------------------------------------------------------------*/
bool CheckKeyFile(CGlobalEnumerations::KeyFile_Type_e eKeyType,
                  const char* f_pcMountPoint)
{
    ETG_TRACE_USR3 (("CheckKeyFile: Begin"));

    ETG_TRACE_USR1(("CheckKeyFile: eKeyType=%d f_pcMountPoint=%s",eKeyType,f_pcMountPoint));

    bool l_bKeyFileExists = false;
    FILE *l_pFileLogging;
    char key_filename[1024];
    tU32 u32Len = 0;
    
    //@todo-->check if this could be done in ConfigurationMgr to make this part of the code more generic
    tU32 keyLen[tInt(CGlobalEnumerations::KEY_LAST_VAL)];
    GENSTRING keyText[tInt(CGlobalEnumerations::KEY_LAST_VAL)];
    GENSTRING keyFileName[tInt(CGlobalEnumerations::KEY_LAST_VAL)];

    keyText[CGlobalEnumerations::KEY_LOGDGBINFO ]      = WRITEHISTORYTOUSBSTICK_KEY;
    keyLen [CGlobalEnumerations::KEY_LOGDGBINFO ]      = WRITEHISTORYTOUSBSTICK_KEYLEN;
    keyFileName [CGlobalEnumerations::KEY_LOGDGBINFO ] = WRITEHISTORYTOUSBSTICK_KEYFILENAME;
    //<--

    tU32 LenOfKey = keyLen[eKeyType];
    char key2log[LenOfKey+1];
    key2log[LenOfKey] = '\0';

    snprintf (key_filename, sizeof(key_filename), "%s/%s", f_pcMountPoint,
              keyFileName[eKeyType].toStdString().c_str());

    ETG_TRACE_USR4 (("CheckKeyFile: Checking for file %s",
                     key_filename));

    l_pFileLogging = fopen (key_filename,"r");

    if (NULL != l_pFileLogging)
    {
        ETG_TRACE_FATAL (("[ok] =>=================================================================="));
        ETG_TRACE_FATAL (("[ok] CheckKeyFile: key file found"));

        // read the string equal to key length
        u32Len = (tU32)fread(key2log, 1, LenOfKey, l_pFileLogging);

        ETG_TRACE_FATAL (("[ok] CheckKeyFile: u32Len  =%d",u32Len));
        ETG_TRACE_FATAL (("[ok] CheckKeyFile: LenOfKey=%d",LenOfKey));

        if (LenOfKey == u32Len)
        {
            ETG_TRACE_USR4(("key:%s",key2log));
            if (0 == strncmp(key2log, keyText[eKeyType].toStdString().c_str(),LenOfKey-1))
            {
                ETG_TRACE_FATAL (("[ok] !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                ETG_TRACE_FATAL (("[ok] !!!!     Logging information to USB-stick       !!!!"));
                ETG_TRACE_FATAL (("[ok] !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));

                l_bKeyFileExists = true;
            }
            else
            {
                ETG_TRACE_ERR (("CheckKeyFile: KEY found is wrong"));
            }
        }
        else
        {
            ETG_TRACE_ERR (("CheckKeyFile: KEY not found"));
        }

        ETG_TRACE_FATAL (("[ok] ==================================================================<="));
        fclose (l_pFileLogging);
    }
    else
    {
        ETG_TRACE_USR1 (("CheckKeyFile: key file not found in usb"));
    }

    ETG_TRACE_USR3 (("CheckKeyFile: End"));

    return (l_bKeyFileExists);
}


/*-----------------------------------------------------------------------------*
 * bool CheckKeyFile(CGlobalEnumerations::KeyFile_Type_e eKeyType,             *
 *              const char* f_pcMountPoint)                                    *
 *-----------------------------------------------------------------------------*/
bool IsValidMscType( tCString pDevName, tU8 u8Offset )
{
    ETG_TRACE_USR4(( "IsValidMscType: Begin" ));

    bool    bRet = false;
    tCString pName = NULL;

    ETG_TRACE_USR4(("IsValidMscType: u8Offset: %d pDevName:%s",u8Offset,pDevName));

    if(pDevName)
    {
        if( strlen(pDevName) > u8Offset)
        {
            pName = pDevName + u8Offset;
        }
        else
        {
            pName = NULL;
            ETG_TRACE_USR4(("IsValidMscType:strlen(pDevName):%d  AUTOMOUNTER_NAMEOFFSET:%d ",strlen(pDevName),(int)u8Offset));
        }
    }
    // e.g. /dev/sr0                     (CDROM)                                                           pname:partition_dev_sr0
    // e.g. /dev/sda /dev/sda1    (USB-Stick or SDCard at usb connected HUB)       pname:partition_dev_sda1
    // e.g. /dev/mmcblk0p1         (internal SDCard)                                               pname:partition_dev_mmcblk0p1
    if(( pName ) && ('.' != *pName ) )// check for valid pointer to file name, name[0] = '.'; don't process hidden files
    {
        if( ('s' == *pName) && ('d' == *(pName+1)) )
        {
            ETG_TRACE_USR4(("IsValidMscType:external usb-stick or external sdcard"));
            bRet = true;
        }
        else if( ('m' == *pName) && ('m' == *(pName+1)) && ('c' == *(pName+2)) )  // ->  process 'mmc'; internal SD Card)
        {
            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_INTERNAL_SDCARD))
            {
                ETG_TRACE_USR4(("IsValidMscType:internal sdcard"));
                bRet = true;
            }
        }
        else if( ('s' == *pName) && ('r' == *(pName+1))  )   // ->  process 'cdrom'; CD ROM)
        {
            if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_INTERNAL_CDROM))
            {
                ETG_TRACE_USR4(("IsValidMscType:internal cdrom"));
                bRet = true;
            }
        }
        else
        {
            ETG_TRACE_USR4(("IsValidMscType:Unknown Type"));
        }

    }
    ETG_TRACE_USR4(( "IsValidMscType: End; Valid: %d ", ETG_ENUM( BOOL, bRet ) ));

    return bRet;
}


#ifdef USE_VALIDATE_AND_CONVERT2REGION_LANGUAGE

/* bool CMediaIndexerStore::ValidateUTF8 (GENSTRING f_cMetaDataToValidate )
*
* Brief: Validate a string received from the Taglib/Gstreamer is utf8 encoded or not.
*
* Params:  f_cMetaDataToValidate  string need to be validated.
*
* Return: True if string is utf8 encoded else False.
*/
bool  ValidateUTF8 (const GENSTRING &f_cMetaDataToValidate)
{
    bool l_bValidationStatus;
    const char * l_cTemp = (const char *)f_cMetaDataToValidate.TOASCII;
    l_bValidationStatus = (bool)g_utf8_validate( l_cTemp , -1, NULL);

    return l_bValidationStatus;
}



/* GENSTRING ValidateAndConvertToRegionLanguage (GENSTRING f_cMetaData)
 *
 * Brief: function is thought to be used with !!deviceName!!
 * Validate string as utf-8, if not utf8 encoded then assuming a GBK string and convert
 * from GBK to utf8. This function need to called only when market region is CHINA_MARKETING_REGION.
 *
 *
 * Params:  f_cMetaData  string need to be validated.
 *
 * Return:  retuns the utf-8 encoded string.
 */
GENSTRING ValidateAndConvertToRegionLanguage (GENSTRING f_cFSType, const GENSTRING &f_cMetaData, tU8 marketingRegion)
{
    // ETG_TRACE_USR4(("Inside: MarketingRegionSupport::ValidateAndConvertToRegionLanguage"));
    int i;

    tBool bConvert7BitAscii = FALSE;
    GENSTRING l_cTempMetaData = f_cMetaData ;

    ETG_TRACE_USR1(("ValidateAndConvertToRegionLanguage:marketingRegion: %d",ETG_ENUM(TEN_DEV_MANAGER_EOL_REGION,marketingRegion)));
    //------------------------------------------------
    //UTF8:windows ntfs and apple hfs+ use uft8
    //------------------------------------------------
    if (ValidateUTF8(l_cTempMetaData))
    {
        ETG_TRACE_COMP(("ValidateAndConvertToRegionLanguage: is UTF8"));

        //---------apple hfsplus---------------
        if(f_cFSType.compare("hfsplus") == IS_EQUAL)
        {
            ETG_TRACE_COMP(("ValidateAndConvertToRegionLanguage: ->hfsplus"));

            l_cTempMetaData = GENSTRING_FROMUTF8(f_cMetaData.TOASCII);
            //Linux and (most?) other Unix-like operating systems use the so called normalization form C (NFC)
            // for its UTF-8 encoding by default
            //Darwin, the base of the Macintosh OS enforces normalization form D (NFD)
            l_cTempMetaData = l_cTempMetaData.normalized(NORMALIZATIONFORMC);
            l_cTempMetaData = GENSTRING(l_cTempMetaData.TOUTF8);
        }
        //---------windows ntfs---------------
        else if(f_cFSType.compare("ntfs") == IS_EQUAL)
        {
            ETG_TRACE_COMP(("ValidateAndConvertToRegionLanguage: ->ntfs"))
                    l_cTempMetaData = f_cMetaData;
        }
        //here we assume FAT and others and replace all values above 128 by a questionmark '?'
        //reason we don't know which country specific code page is used
        //----------other----------------------
        else
        {
            ETG_TRACE_COMP(("ValidateAndConvertToRegionLanguage: ->unknown utf8 partition treat it normalized C"));
                    l_cTempMetaData = f_cMetaData;
        }
    }
    //------------------------------------------------
    //FAT or GBK or others i.e. not UTF8
    //------------------------------------------------
    else
    {
        //here we could make it marketing dependent
        ETG_TRACE_COMP(("ValidateAndConvertToRegionLanguage: is vfat, gbk or others"));

        switch(marketingRegion)
        {
        case 0x08://GIS 344 - China
        {
            ETG_TRACE_USR4(("China region:String is NOT UTF8 encoded-> convert from GBK to UTF8"));
            //glib character conversion
            gchar* l_cConvertRet = g_convert(l_cTempMetaData.TOASCII, -1, "UTF-8", "GBK", NULL, NULL, NULL);
            if (!l_cConvertRet)
            {
                ETG_TRACE_USR4(("ERROR: String is not GBK encoded as expected"));
                l_cTempMetaData = f_cMetaData;
                bConvert7BitAscii = TRUE;
            }
            else
            {
                // GBK conversion successful we now have UTF8
                l_cTempMetaData = l_cConvertRet;
                g_free(l_cConvertRet);
                bConvert7BitAscii = FALSE;
            }
            break;
        }
        case 0x00: //USA             (EF7)
        case 0x01: //Canada          (Z49)
        case 0x02: //Mexico          (CV3)
        case 0x03: //Central America (CZ1) BELIZ, COSTA RICA, EL SALVADOR, GUATAMALA, HONDURAS, NICARAGUA, PANAMA
        case 0x04: //Caribian        (CW4) ANTIGUA, ANTILLES, BAHAMAS, BARBADOS, BERMUDA, CAY, HAITI, DMINCN, JAMAICA, NASSAU, S.M.,TRINIDAD
        case 0x05: //Europe          (C4E)
        case 0x06: //Russia          (CZ3)
        case 0x07: //United Kingdom  (CU9)
            //case 0x08: //China           (CZ2)
        case 0x09: //Japan           (CV5)
        case 0x0A: //Korea           (CW1)
        case 0x0B: //Taiwan          (CU6) //GIS 344 - Taiwan:   same behavior as in china w.r.t dest entry(Ref - GMNGA-51949)
        case 0x0C: //Philippines     (D2D)
        case 0x0D: //GUAM            (CW6)
        case 0x0E: //Australia
        case 0x0F: //Gulf Areas      (CW2)  BAHRAIN, KUWAIT, OMAN, QATAR, SAUDI ARABIA, UAE
        case 0x10: //Kuwait          (CU7)
        case 0x11: //Saudi Arabia    (CU8)
        case 0x12: //Dubai           (CR4)
        case 0x13: //Israel          (CV4)
        case 0x14: //Argentina       (CX1)
        case 0x15: //Chile           (CV6)
        case 0x16: //Venezuela       (CW5)
        case 0x17: //Columbia        (CX3)
        case 0x18: //Bolivia         (CX3)
        case 0x19: //Peru            (CL5)
        case 0x1A: //Ecuador         (CX3)
        case 0x1B: //Africa          (CW9)
        case 0x1C: //Egypt           (CR8)
            bConvert7BitAscii = TRUE;
            break;
        default:
            bConvert7BitAscii = TRUE;
            break;
        }

        
        //-------------------------------------------------------------------------------------------------------------------------
        //if not UTF8 and not China GBK then convert to International Ascii i.e. letters below 128 i.e. 7Bit Ascii packed in bytes
        //-------------------------------------------------------------------------------------------------------------------------
        if(bConvert7BitAscii == TRUE)
        {
            ETG_TRACE_COMP(("ValidateAndConvertToRegionLanguage: is not marketingRegion China "));
            ETG_TRACE_COMP(("ValidateAndConvertToRegionLanguage: replace characters >127 (country dependent) using a space"));
            l_cTempMetaData = f_cMetaData;

            i = 0;
            while(i<f_cMetaData.TOASCII_SIZE )
            {
                if(f_cMetaData.TOASCII_AT(i) > 127) //lint !e713:Loss of precision (arg. no. 1)
                {
                    l_cTempMetaData.replace(i,1,' ');
                }
                i++;
            }
            //since it is ascii values between 0...127 it equals UTF-8
        }

    }
    
    //ETG_TRACE_USR4(("Leaving: MarketingRegionSupport::ValidateAndConvertToRegionLanguage"));
    return l_cTempMetaData ;
}
#endif //USE_VALIDATE_AND_CONVERT2REGION_LANGUAGE


//=========================================================================================

/******************************************************************************
 *FUNCTION      :bGetFirstCDSys
 *DESCRIPTION   :Searches for DevName of first CD drive
 *PARAMETER     :tDevCDData *pShMem: Pointer to shared memory
 *               CD_tsDevName * psDevOut: Pointer to structure to be initialized
 *RETURNVALUE   :TRUE: found
 *HISTORY:      :Created by srt2hi 2014
 *****************************************************************************/
static tBool bGetFirstCDSys(struct udev *udev, const char *pcszMascaPath,const char* pcszSysName, char *pszDevNameOut)
{
    ETG_TRACE_USR4(("Begin: bGetFirstCDSys"));
    tBool bFound = FALSE;
    struct udev_enumerate *enumerate;
    struct udev_list_entry *listEntry;
    struct udev_device *dev;

    if(!pszDevNameOut || !pcszSysName || !pcszMascaPath || !udev)
        return FALSE;

    enumerate = udev_enumerate_new(udev);
    udev_enumerate_add_match_sysname(enumerate, pcszSysName);
    udev_enumerate_scan_devices(enumerate);
    listEntry = udev_enumerate_get_list_entry(enumerate);
    udev_list_entry_foreach(listEntry, listEntry)
    {
        const char *path;
        path = udev_list_entry_get_name(listEntry);
        if(strstr(path, pcszMascaPath))
        {
            bFound = TRUE;
            dev = udev_device_new_from_syspath(udev, path);
            strcpy(pszDevNameOut, udev_device_get_devnode(dev));
            udev_device_unref(dev);
        } //strstr
    }
    udev_enumerate_unref(enumerate);
    ETG_TRACE_USR4(("End: bGetFirstCDSys :bFound:0x%x",bFound));
    return bFound;
}



/******************************************************************************
 *FUNCTION      :u32GetFirstCD
 *DESCRIPTION   :Searches for DevName of first CD drive
 *PARAMETER     :tDevCDData *pShMem: Pointer to shared memory
 *              CD_tsDevName * psDevOut: Pointer to structure to be initialized
 *RETURNVALUE   :OSAL_E_NOERROR: cd drive found and could be opened
 *               OSAL_E_DOESNOTEXIST: no CDd drive found
 *HISTORY:      :Created by srt2hi 2013
 *****************************************************************************/
static tBool bGetFirstCD(CD_tsDevName *psDevOut)
{
    ETG_TRACE_USR4(("Begin: bGetFirstCD"));
    tBool bRet     = TRUE;
    tBool bSRFound = FALSE;
    tBool bSGFound = FALSE;
    int   fd;
    struct udev *udev;

    if(!psDevOut)
        return FALSE;

    memset(psDevOut, 0, sizeof(CD_tsDevName));

    udev = udev_new();

    if(udev)
    {
        struct udev_enumerate *enumerate;
        struct udev_list_entry *listEntry;
        struct udev_device *dev;
        enumerate = udev_enumerate_new(udev);

        udev_enumerate_add_match_subsystem(enumerate, "masca");
        udev_enumerate_scan_devices(enumerate);
        listEntry = udev_enumerate_get_list_entry(enumerate);
        udev_list_entry_foreach(listEntry, listEntry)
        {
            const char *path;
            path = udev_list_entry_get_name(listEntry);
            dev = udev_device_new_from_syspath(udev, path);
            bSGFound = bGetFirstCDSys(udev, path, "sg*", psDevOut->szDevSGName);
            bSRFound = bGetFirstCDSys(udev, path, "sr*", psDevOut->szDevSRName);
            udev_device_unref(dev);
        }
        udev_enumerate_unref(enumerate);
    }
    else //if(udev)
    {
        //error get udev listEntry
        ETG_TRACE_FATAL(("bGetFirstCD: NoAccess"));
        bRet = FALSE;
    } //else //if(udev)

    psDevOut->bSGFound = bSGFound;
    psDevOut->bSRFound = bSRFound;

    psDevOut->bDefaultSGNameUsed = !bSGFound;
    psDevOut->bDefaultSRNameUsed = !bSRFound;

    //no listEntry found, use default name e.g. "/dev/sr0"
    if(!bSGFound)
        strncpy(psDevOut->szDevSGName, "/dev/sr0",sizeof(psDevOut->szDevSGName) - 1);

    if(!bSRFound)
        strncpy(psDevOut->szDevSRName, "/dev/sr0",sizeof(psDevOut->szDevSRName) - 1);

    fd = open(psDevOut->szDevSRName, O_RDWR | O_NONBLOCK);
    if(-1 != fd)
    {
        psDevOut->bSRCouldBeOpened = TRUE;
        (void)close(fd);
    }
    else
    {
        psDevOut->bSRCouldBeOpened = FALSE;

        ETG_TRACE_FATAL(("Error open listEntry psDevOut->szDevSRName [%s]", psDevOut->szDevSRName));
        ETG_TRACE_FATAL(("Error open listEntry strerror(errno)       [%s]", strerror(errno)));

    }

    fd = open(psDevOut->szDevSGName, O_RDWR | O_NONBLOCK);
    if(-1 != fd)
    {
        psDevOut->bSGCouldBeOpened = TRUE;
        (void)close(fd);
    }
    else
    {
        psDevOut->bSGCouldBeOpened = FALSE;
        ETG_TRACE_FATAL(("Error open listEntry psDevOut->szDevSGName [%s]", psDevOut->szDevSGName));
        ETG_TRACE_FATAL(("Error open listEntry strerror(errno)       [%s]", strerror(errno)));
    }

    if(bRet)
    {
        if(psDevOut->bSRCouldBeOpened && psDevOut->bSGCouldBeOpened)
        {
            bRet = TRUE;
        }
        else
        {
            bRet = FALSE;
        }
    }

    ETG_TRACE_USR4(("End  : bGetFirstCD (bRet:%d)",bRet));
    return bRet;
}




tBool bFindScsiCDDrive(OUT std::string &strSRx, OUT std::string &strSGx)
{
    ETG_TRACE_USR4(("Begin: CD_SCSI_IF_u32FindCDDrive"));
    tBool  bRet;

    CD_tsDevName  rCDDrive;

    bRet = bGetFirstCD(&rCDDrive);

    ETG_TRACE_USR4(("CD_SCSI_IF_u32FindCDDrive: bRet [0x%08X]",          bRet));
    ETG_TRACE_USR4(("CD_SCSI_IF_u32FindCDDrive: SRFound [%d]",          (int)rCDDrive.bSRFound));
    ETG_TRACE_USR4(("CD_SCSI_IF_u32FindCDDrive: SGFound [%d]",          (int)rCDDrive.bSGFound));
    ETG_TRACE_USR4(("CD_SCSI_IF_u32FindCDDrive: DefaultSRUsed [%d]",    (int)rCDDrive.bDefaultSRNameUsed));
    ETG_TRACE_USR4(("CD_SCSI_IF_u32FindCDDrive: DefaultSGUsed [%d]",    (int)rCDDrive.bDefaultSGNameUsed));
    ETG_TRACE_USR4(("CD_SCSI_IF_u32FindCDDrive: SRCouldBeOpened [%d]",  (int)rCDDrive.bSRCouldBeOpened));
    ETG_TRACE_USR4(("CD_SCSI_IF_u32FindCDDrive: SGCouldBeOpened [%d]",  (int)rCDDrive.bSGCouldBeOpened));
    ETG_TRACE_USR4(("CD_SCSI_IF_u32FindCDDrive:  SRName [%s]",           rCDDrive.szDevSRName));
    ETG_TRACE_USR4(("CD_SCSI_IF_u32FindCDDrive:  SGName [%s]",           rCDDrive.szDevSGName));
    

    if(!bRet)
    {
        ETG_TRACE_FATAL(("No usable CD-Drive found"));
    } //if(OSAL_E_NOERROR != u32Result)
    else
    {
        strSRx = rCDDrive.szDevSRName;
        strSGx = rCDDrive.szDevSGName;

    }


    ETG_TRACE_USR4(("End  : CD_SCSI_IF_u32FindCDDrive"));
    return bRet;
}
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-------------------------------------------------------------------------------------------------------------*
 * FormSerialIdFromVendorIdProductId(const uint16_t& vendorID, const uint16_t& productID, GENSTRING &serialId)
 *
 *  *----------------------------------------------------------------------------------------------------------*/
void FormSerialIdFromVendorIdProductIdSysname(IN const uint16_t &vendorID, IN const uint16_t &productID, IN GENSTRING &sysName,OUT GENSTRING &serialId)
{
    ETG_TRACE_USR3 (("Begin : FormSerialIdFromVendorIdProductId"));

    if(vendorID)
    {
        serialId = std::to_string(vendorID) + '_';
    }
    if(productID)
    {
        serialId += std::to_string(productID) + '_';
    }

    /*
     *This is needed to uniquely idenfity devices when 2 or more devices
     *with same vendorid and productid are connected at the same time.
     */
    makeSerialIdUnique(IN sysName, INOUT serialId);

    serialId.trimmed();

    ETG_TRACE_USR3 (("END : FormSerialIdFromVendorIdProductId: serialId = %s", serialId.toStdString().c_str()));
}

/*------------------------------------------------------------*
 * makeSerialIdUnique(INOUT GENSTRING &serialId)              *
 *------------------------------------------------------------*/
void makeSerialIdUnique(IN GENSTRING &sysName, INOUT GENSTRING &serialId)
{
    ETG_TRACE_USR3 (("Begin : makeSerialIdUnique serialId = %s",serialId.toStdString().c_str()));

    if(!sysName.isEmpty())
    {
        serialId += sysName;
    }
    else
    {
        /*
         *If sysname is empty, then go for generating a unique number
         *Limitation :
         * 1.Device connected with serialnumber formed as vidA_pidB_uniquenumberX
         * 2.When a malfunction starts, Device removal comes for the connected device leading to connection status as HW_MALFUNCTION
         * 3.When malfunction ends, device connection comes again for the same device.
         *  Here a new serial id (vidA_pidB_randomnumberX+1) will be formed and DVM thinks this as new device,
         *  leading to stale entry in devicemanager lists with connection status as HW_MALFUNCTION_PERMANENT.
         *  Note:Will live with this LIMITATION, since this is only for unsupported devices to show popup in HMI and the stale entry
         *  will be cleaned upon IGN OFF/ON.
         */
        static uint32_t serialNumber = 1;
        serialId += std::to_string(serialNumber);
        serialNumber++;
    }
    ETG_TRACE_USR3 (("END : makeSerialIdUnique: serialId = %s", serialId.toStdString().c_str()));
}
//=========================================================================================
////////////////////////////////////////////////////////////////////////////////
// <EOF>
