/*-----------------------------------------------------------------------------*
 * usbutils.cpp                                                                *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2010 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file usbutils.cpp
 *
 * \brief This file holds implementation oof libusb wrapper functions
 *
 * \version 27.08.2012, Negi, Sunder (MontaVista), initial version
 *
 *-----------------------------------------------------------------
 *                                    development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch)
 *          -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 *
 * \copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"
#include "Enums.h"
#include "Device.h"
#include "DeviceCard.h"
#include "DeviceManager_ErrorCodes.h"
#include "usbutils.h"
#include "config/ConfigurationManager.h"

#include <bitset>
#include <unistd.h>
extern "C"
{

#include <errno.h>
#include <libusb-1.0/libusb.h>
#include <libmtp.h>
}


/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_UDEVMANAGER
#include "trcGenProj/Header/usbutils.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST
#include "ETGTrace.h"

#define CHARGE_APPLE_TIMOUT 1000 /*ms*/

static int err = 0;

/*! the libusb context */
static struct libusb_context *ctx = NULL;

const uint16_t AppleVendorID = 0x05ac;

//WORKAROUND DELAY
const tUInt uiRetrialDelay_ms  = 200/*ms*/;
const tUInt uiNMaxNumOfRetrials = 5;

//missing in include of libusb
#define _LIBUSB_CLASS_AUDIOVIDEODEVICES 0x10
#define _LIBUSB_CLASS_DILLBOARDDEVICE   0x11
#define _LIBUSB_CLASS_MISCELLANEUS      0xEF

/*-----------------------------------------------------------------------------*
 * int LibUSBInit()                                                            *
 *-----------------------------------------------------------------------------*/
int LibUSBInit()
{
    ETG_TRACE_USR3 (("LibUSBInit: Begin"));

    err = 0;

    if (NULL == ctx)
    {
        err = libusb_init (&ctx);

        if (0 != err)
        {
            ETG_TRACE_FATAL (("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
            ETG_TRACE_FATAL (("!!!!        Error initializing libUSB        !!!!"));
            ETG_TRACE_FATAL (("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
#ifdef USE_LIBUSBMTP_ERRMEM
            ETG_TRACE_ERRMEM (("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
            ETG_TRACE_ERRMEM (("!!!!        Error initializing libUSB        !!!!"));
            ETG_TRACE_ERRMEM (("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
#endif //USE_LIBUSBMTP_ERRMEM
            ctx = NULL;

        }
        else
        {
            ETG_TRACE_SYS (("LibUSBInit: libUSB initialization success"));
        }
    }
    else
    {
        ETG_TRACE_SYS (("LibUSBInit: already initialized"));
    }

    ETG_TRACE_USR3 (("LibUSBInit: End"));

    return (err);
}

/*-----------------------------------------------------------------------------*
 * int LibUSBExit()                                                            *
 *-----------------------------------------------------------------------------*/
int LibUSBExit()
{
    ETG_TRACE_USR3 (("LibUSBExit: Begin"));

    if (NULL != ctx)
    {
        libusb_exit (ctx);
        ETG_TRACE_SYS (("LibUSBExit: libUSB de-initialization success"));
    }
    else
    {
        ETG_TRACE_FATAL (("LibUSBExit: not yet initialized"));
    }

    ETG_TRACE_USR3 (("LibUSBExit: End"));

    return (err);
}

/*-----------------------------------------------------------------------------*
 * libusb_device *LibUSBGetDevice(const char *f_cUsbDevPath)                   *
 *-----------------------------------------------------------------------------*/
libusb_device *LibUSBGetDevice(const char *f_cUsbDevPath)
{
    ETG_TRACE_USR3 (("LibUSBGetDevice: Begin"));

    ETG_TRACE_USR1 (("LibUSBGetDevice: f_cUsbDevPath: %s",f_cUsbDevPath));

    libusb_device *l_poDev = NULL;

    tBool l_bLoop;
    tUInt l_uiCountRetrials;

    if (NULL == ctx)
    {
        ETG_TRACE_ERR (("LibUSBGetDevice: [ERROR] libusb not initialized"));
    }
    else
    {
        if (NULL == f_cUsbDevPath || '\0' == f_cUsbDevPath[0])
        {
            ETG_TRACE_ERR (("LibUSBGetDevice: [ERROR] invalid usb device path"));
        }
        else
        {
            libusb_device **l_ppDevList;
            ssize_t l_iNumDevs, itr;
            char l_cDevPath[128];

            l_uiCountRetrials = 0;
            l_bLoop           = TRUE;
            while(l_bLoop)
            {
                l_iNumDevs = libusb_get_device_list(ctx, &l_ppDevList);
                if(l_iNumDevs < 0 /*=LIBUSB_SUCCESS*/ )
                {
                    ETG_TRACE_ERR(("[ERROR] LibUSBGetDevice: l_iNumDevs=libusb_get_device_list(...) l_iNumDevs: %d (%s)",(int)l_iNumDevs,libusb_strerror((libusb_error)l_iNumDevs)));  //according to documentation libusb_strerror never returns 0
#ifdef USE_LIBUSBMTP_ERRMEM
                    ETG_TRACE_ERRMEM(("[ERROR] LLibUSBGetDevice: l_iNumDevs=libusb_get_device_list(...) l_iNumDevs: %d (%s)",(int)l_iNumDevs,libusb_strerror((libusb_error)l_iNumDevs)));
#endif //USE_LIBUSBMTP_ERRMEM
                }
                ETG_TRACE_USR4(("LibUSBGetDevice: l_iNumDevs: %d",l_iNumDevs));

                for (itr = 0; itr < l_iNumDevs; itr++)
                {
                    ETG_TRACE_USR4(("LibUSBGetDevice: itr: %d",itr));
                    uint8_t l_iBusNum = libusb_get_bus_number (l_ppDevList[itr]);
                    uint8_t l_iDevNum = libusb_get_device_address (l_ppDevList[itr]);

                    snprintf (l_cDevPath, sizeof(l_cDevPath), "%s/%03u/%03u",
                              "/dev/bus/usb", l_iBusNum, l_iDevNum);

                    ETG_TRACE_USR4(("LibUSBGetDevice: l_iBusNum: %d",l_iBusNum));
                    ETG_TRACE_USR4(("LibUSBGetDevice: l_iDevNum: %d",l_iDevNum));
                    ETG_TRACE_USR4(("LibUSBGetDevice: l_cDevPath %s",l_cDevPath));
                    ETG_TRACE_USR4(("LibUSBGetDevice:-------------------------"));
                    if (!strcmp (l_cDevPath, f_cUsbDevPath))
                    {
                        ETG_TRACE_USR4(("LibUSBGetDevice: Device found: %s",l_cDevPath));
                        l_poDev = l_ppDevList[itr];
                        l_bLoop = FALSE;
                        break;
                    }
                }
                ++l_uiCountRetrials;
                //retrial
                if((l_uiCountRetrials <= uiNMaxNumOfRetrials) && (l_bLoop==TRUE))
                {
                    ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                    ETG_TRACE_FATAL(("LibUSBGetDevice: l_uiCountRetrials:%d  uiNMaxNumOfRetrials: %d",l_uiCountRetrials,uiNMaxNumOfRetrials));
                    ETG_TRACE_FATAL(("LibUSBGetDevice: Workaround RaceCond Udev&libusb: "));
                    ETG_TRACE_FATAL(("LibUSBGetDevice: Wait %d uiRetrialDelay_ms ms ",uiRetrialDelay_ms));
#ifdef USE_LIBUSBMTP_ERRMEM
#ifdef USE_LIBUSBRETRIAL_ERRMEM
                    ETG_TRACE_ERRMEM(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
                    ETG_TRACE_ERRMEM(("LibUSBGetDevice: l_uiCountRetrials:%d  uiNMaxNumOfRetrials: %d",l_uiCountRetrials,uiNMaxNumOfRetrials));
                    ETG_TRACE_ERRMEM(("LibUSBGetDevice: Workaround RaceCond Udev&libusb: "));
                    ETG_TRACE_ERRMEM(("LibUSBGetDevice: Wait %d uiRetrialDelay_ms ms ",uiRetrialDelay_ms));
#endif //USE_LIBUSBRETRIAL_ERRMEM
#endif //USE_LIBUSBMTP_ERRMEM
                    usleep((useconds_t)(uiRetrialDelay_ms*1000));
                    ETG_TRACE_FATAL(("LibUSBGetDevice: Retrial now"));
                    ETG_TRACE_FATAL(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
#ifdef USE_LIBUSBMTP_ERRMEM
#ifdef USE_LIBUSBRETRIAL_ERRMEM
                    ETG_TRACE_ERRMEM(("LibUSBGetDevice: Retrial now"));
                    ETG_TRACE_ERRMEM(("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"));
#endif //USE_LIBUSBRETRIAL_ERRMEM
#endif //USE_LIBUSBMTP_ERRMEM
                }
                //finished trying
                else
                {
                    if(l_bLoop == TRUE)
                    {
                        ETG_TRACE_FATAL(("LibUSBGetDevice: Device not found in libusb - finish retrying"));
                    }
                    l_bLoop = FALSE;
                }

            }
            /*
            if(l_iNumDevs >=0)
            {
                libusb_free_device_list (l_ppDevList, 1); //can't be used here chech wher eit can be used
            }
*/
        }
    }

    ETG_TRACE_USR3 (("LibUSBGetDevice: End"));

    return (l_poDev);
}

/*-----------------------------------------------------------------------------*
 * int LibUSBGetString(libusb_device_handle *f_poDev, uint8_t f_iId,           *
 *              unsigned char *f_cBuffer, size_t f_iBufSize)                   *
 *-----------------------------------------------------------------------------*/
int LibUSBGetString(libusb_device_handle *f_poDev, uint8_t f_iId,      /*due to a reset foundin Suzuki projects excessive error case implementation*/
                    unsigned char *f_cBuffer, size_t f_iBufSize)
{
    int ret = 0;
    unsigned int iIndex = 0;  //is the index to add an end of line '\0' i.e. one byte behind payload
    int l_iBufSize = (int)(f_iBufSize - 1); //to keep last byte for '\0'

    ETG_TRACE_USR4(("LibUSBGetString: f_iId:%d, f_iBufSize: %d",f_iId,f_iBufSize));
    //routine to copy
    if (f_poDev && f_iId)
    {
        ret = libusb_get_string_descriptor_ascii(f_poDev, f_iId,
                                                  (unsigned char *)f_cBuffer, (int)l_iBufSize);     // cast to int because 4. parameter length is int

        ETG_TRACE_USR4(("LibUSBGetString: ret:%d, f_iBufSize: %d",ret,f_iBufSize));
        //ok
        if(ret > 0 /*=LIBUSB_SUCCESS*/)
        {
            //ok
            if(ret <= l_iBufSize)
            {
                iIndex = (unsigned int)ret;
                ETG_TRACE_USR4(("LibUSBGetString: ret < f_iBufSize-1"));
            }
            ///nok
            else
            {
                iIndex = (unsigned int)l_iBufSize;
                ETG_TRACE_COMP(("[ERROR] LibUSBGetString ret <0"));
            }
        }
        //nok
        else if(ret == 0)
        {
            ETG_TRACE_COMP(("[WARNING] LibUSBGetString: libusb_get_string_descriptor_ascii returned ret==0"));
        }
        //nok if <0
        else
        {
            ETG_TRACE_COMP(("[ERROR]: LibUSBGetString: ret=%d",/*ETG_CENUM(libusb_error,*/ret/*)*/));
#ifdef USE_LIBUSBMTP_ERRMEM
            ETG_TRACE_FATAL(("[ERROR]: LibUSBGetString: ret=libusb_get_string_descriptor_ascii(...) ret=%d (%s)",/*ETG_CENUM(libusb_error,*/(int)ret/*)*/,libusb_strerror((libusb_error)ret)));
            ETG_TRACE_ERRMEM(("[ERROR]: LibUSBGetString:ret=libusb_get_string_descriptor_ascii(...) ret=%d (%s)",/*ETG_CENUM(libusb_error,*/(int)ret/*)*/,libusb_strerror((libusb_error)ret)));
#endif //USE_LIBUSBMTP_ERRMEM
        }
    }
    else
    {
        ETG_TRACE_COMP(("[ERROR]LibUSBGetString: %d",f_iId));
        if(f_poDev == NULL)
        {
            ETG_TRACE_COMP(("[ERROR]LibUSBGetString: f_poDev == NULL"));
        }
    }

    //fill end of string to buffer if possible otherwhie don't touch it
    if(f_cBuffer) //buffer exists
    {
        if(f_iBufSize>0) //buffersize > 0
        {
            f_cBuffer[iIndex] = '\0';
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR] LibUSBGetString:f_iBufSize <=0"));
        }
    }
    else
    {
        ETG_TRACE_COMP(("[ERROR] LibUSBGetString:f_cBuffer == NULL"));
    }

    return (ret);
}







/*-----------------------------------------------------------------------------*
 * static int DumpDetails(struct libusb_device *f_poDev,                       *
 *              struct libusb_device_handle *f_poDevHdl,                       *
 *              const struct libusb_device_descriptor *l_poDevDesc)            *
 *-----------------------------------------------------------------------------*/
static int DumpDetails(struct libusb_device *f_poDev,
                       struct libusb_device_handle *f_poDevHdl,
                       const struct libusb_device_descriptor *l_poDevDesc)
{
    ETG_TRACE_USR3 (("DumpDetails: Begin"));

    ETG_TRACE_USR4 (("------------------------------------------------------"));
    ETG_TRACE_USR4 (("%04x:%04x ",
                     l_poDevDesc->idVendor, l_poDevDesc->idProduct));

    unsigned char mfg[128], prod[128], serial[128];

    LibUSBGetString(f_poDevHdl, l_poDevDesc->iManufacturer, mfg, sizeof(mfg));   //saw in tests at startup that l_poDevDesc->iManufacturer to be 0
    LibUSBGetString(f_poDevHdl, l_poDevDesc->iProduct, prod, sizeof(prod));
    LibUSBGetString(f_poDevHdl, l_poDevDesc->iSerialNumber, serial, sizeof(serial));

    ETG_TRACE_USR4 (("bcdUSB             %2x.%02x", l_poDevDesc->bcdUSB >> 8,
                     l_poDevDesc->bcdUSB & 0xff));

    ETG_TRACE_USR4 (("Vendor ID          0x%04x", l_poDevDesc->idVendor));
    ETG_TRACE_USR4 (("Product ID         0x%04x", l_poDevDesc->idProduct));
    ETG_TRACE_USR4 (("Manufacturer       %s", mfg));
    ETG_TRACE_USR4 (("Product            %s", prod));
    ETG_TRACE_USR4 (("Serial             %s", serial));

    if (l_poDevDesc->bDeviceClass == 0)
    {
        for (unsigned int i = 0; i < l_poDevDesc->bNumConfigurations; i++)
        {
            struct libusb_config_descriptor *l_poConfig;
            int iRes = libusb_get_config_descriptor (f_poDev, (uint8_t)i, &l_poConfig);
            if (!iRes)
            {
                // print configuration details
                for (int j = 0; j < l_poConfig->bNumInterfaces; j++)
                {
                    const struct libusb_interface *l_poIfce = &l_poConfig->interface[j];

                    // print interface details
                    for (int k = 0; k < l_poIfce->num_altsetting; k++)
                    {
                        // print interace setting
                        const struct libusb_interface_descriptor *l_poIfceDesc =
                                &l_poIfce->altsetting[k];

                        ETG_TRACE_USR4 (("Interface Class    %d",
                                         l_poIfceDesc->bInterfaceClass));
                    }
                }

                libusb_free_config_descriptor (l_poConfig);
            }
            else
            {

                if(iRes < 0)
                {
                    ETG_TRACE_FATAL(("[ERROR] iRes=libusb_get_config_descriptor(...) iRes=%d (%s)",(int)iRes,libusb_strerror((libusb_error)iRes)));
#ifdef USE_LIBUSBMTP_ERRMEM
                    ETG_TRACE_ERRMEM(("[ERROR] iRes=libusb_get_config_descriptor(...) iRes=%d (%s)",(int)iRes,libusb_strerror((libusb_error)iRes)));
#endif //USE_LIBUSBMTP_ERRMEM
                }
                else
                {
                    ETG_TRACE_FATAL(("[ERROR] iRes=%d",iRes));
#ifdef USE_LIBUSBMTP_ERRMEM
                    ETG_TRACE_ERRMEM(("[ERROR] iRes=%d",iRes));
#endif

                }

            }

        }
    }

    ETG_TRACE_USR4 (("------------------------------------------------------"));

    ETG_TRACE_USR3 (("DumpDetails: End"));
    return (0);
}

/*-----------------------------------------------------------------------------*
 * int LibUSBShowDeviceDetails (libusb_device *f_poDev)                        *
 *-----------------------------------------------------------------------------*/
int LibUSBShowDeviceDetails (libusb_device *f_poDev)
{
    ETG_TRACE_USR3 (("LibUSBShowDeviceDetails: Begin"));

    if (NULL == f_poDev)
    {
        ETG_TRACE_ERR (("LibUSBShowDeviceDetails: [ERROR] null device"));
    }
    else
    {
        libusb_device_descriptor l_poDesc;

        int r = libusb_get_device_descriptor(f_poDev, &l_poDesc);

        if (r < 0)
        {
            ETG_TRACE_ERR (("LibUSBShowDeviceDetails: [ERROR] r=libusb_get_device_descriptor(...) r:%d (%s)",(int)r,libusb_strerror((libusb_error)r)));
#ifdef USE_LIBUSBMTP_ERRMEM
            ETG_TRACE_ERRMEM (("LibUSBShowDeviceDetails: [ERROR] r=libusb_get_device_descriptor(...) r=%d (%s)",(int)r,libusb_strerror((libusb_error)r)));
#endif //USE_LIBUSBMTP_ERRMEM
        }
        else
        {
            libusb_device_handle *l_poHandle = NULL;

            int iRet = libusb_open (f_poDev, &l_poHandle);
            if (iRet >= 0)
            {
                DumpDetails(f_poDev, l_poHandle, &l_poDesc);

                libusb_close (l_poHandle);
            }
            else
            {
                ETG_TRACE_ERR (("LibUSBShowDeviceDetails: [ERROR] iRet=libusb_open  iRet:%d (%s)",(int)iRet,libusb_strerror((libusb_error)iRet)));
#ifdef USE_LIBUSBMTP_ERRMEM
                ETG_TRACE_ERRMEM (("LibUSBShowDeviceDetails: [ERROR] iRet=libusb_open  iRet:%d (%s)",(int)iRet,libusb_strerror((libusb_error)iRet)));
#endif //USE_LIBUSBMTP_ERRMEM
            }
        }
    }

    ETG_TRACE_USR3 (("LibUSBShowDeviceDetails: End"));

    return (0);
}

/*-----------------------------------------------------------------------------*
 * int LibUSBShowAllDevices ()                                                 *
 *-----------------------------------------------------------------------------*/
int LibUSBShowAllDevices ()
{
    ETG_TRACE_USR3 (("LibUSBShowAllDevices: Begin"));

    ssize_t cnt = 0;

    if(DVM_ON == ConfigurationManager::GetInstance()->u16GetConfigurationValue(eCONF_ENABLE_LIBUSB_ATSTARTUP_SHOWALLDEVICES))
    {


    if (NULL != ctx)
    {
        libusb_device **l_ppDevs;

        // get device list
        cnt =  libusb_get_device_list (ctx, &l_ppDevs);

        if (cnt < 0)
        {
            ETG_TRACE_ERR (("LibUSBShowAllDevices: [ERROR] cnt=libusb_get_device_list(...) %d (%s)",(int)cnt,libusb_strerror((libusb_error)cnt)));
#ifdef USE_LIBUSBMTP_ERRMEM
            ETG_TRACE_ERRMEM (("LibUSBShowAllDevices: [ERROR] cnt=libusb_get_device_list(...) %d (%s)",(int)cnt,libusb_strerror((libusb_error)cnt)));
#endif //USE_LIBUSBMTP_ERRMEM
        }
        else
        {
            ETG_TRACE_USR4 (("LibUSBShowAllDevices: %d devices found", cnt));

            libusb_device *l_poDev;
            int i = 0;

            while ((l_poDev = l_ppDevs[i++]) != NULL)
            {
                LibUSBShowDeviceDetails (l_poDev);
            }

            libusb_free_device_list (l_ppDevs, 1);
        }
    }
    else
    {
        ETG_TRACE_ERR (("LibUSBShowAllDevices: [ERROR] libusb not initialized"));
    }

    }
    else
    {
        ETG_TRACE_USR4(("LibUSBShowAllDevices: [INFO] configuration eCONF_ENABLE_LIBUSB_ATSTARTUP_SHOWALLDEVICES is DVM_OFF"))
    }
    ETG_TRACE_USR3 (("LibUSBShowAllDevices: End"));

    return (cnt);
}

/*-----------------------------------------------------------------------------*
 * static CGlobalEnumerations::DEVICE_TYPE_Type GetDeviceTypeAndDeviceClass(                 *
 *              struct libusb_device *f_poDev,                                 *
 *              struct libusb_device_handle *f_poDevHdl,                       *
 *              const struct libusb_device_descriptor *l_poDesc)               *
*   bSuppressMassStorage: scenario smartphone connected but has severall interface it this function is not triggered
*   by automounter then MASSStorage devices are not expected as an output.
 *-----------------------------------------------------------------------------*/
static CGlobalEnumerations::DEVICE_TYPE_Type GetDeviceTypeAndDeviceClass(
        struct libusb_device *f_poDev,
        struct libusb_device_handle *f_poDevHdl,
        const struct libusb_device_descriptor *l_poDesc,
        tBool bSuppressMassStorage,
        libusb_class_code &f_DeviceClass,
        GENSTRING &f_sInterfacestr)
{
    ETG_TRACE_USR3 (("GetDeviceTypeAndDeviceClass: Begin"));

    CGlobalEnumerations::DEVICE_TYPE_Type l_eType = CGlobalEnumerations::DTY_UNKNOWN;
    f_DeviceClass = LIBUSB_CLASS_HID;
    if (NULL != f_poDev
            && NULL != l_poDesc)
    {

        if(ConfigurationManager::GetInstance()->bFilterIsDigitalAuxInDevice(l_poDesc->idVendor,l_poDesc->idProduct))
        {
            //get Product string: Example: Product: 'Unwired Hub' or  'Audio with HID' or 'Host to Host Bridge'
            char l_sProduct[128];
            LibUSBGetString(f_poDevHdl, l_poDesc->iProduct,(unsigned char*)l_sProduct, sizeof(l_sProduct));

            //--------------------------
            //check for HUB and DTY_AUXIN
            //--------------------------
            GENSTRING strName = GENSTRING(l_sProduct);
            ETG_TRACE_USR4(("GetDeviceTypeAndDeviceClass: Unwired HUB check: %s",strName.toStdString().c_str()));
            if(strName.contains("Audio with HID"))
            {
                ETG_TRACE_USR1(("GetDeviceTypeAndDeviceClass: DIGIAUX  device found"));
                l_eType = CGlobalEnumerations::DTY_DIGIAUX;
            }
            else if(strName.contains("Unwired Hub"))
            {
                ETG_TRACE_USR1(("GetDeviceTypeAndDeviceClass: DTY_HUB"));
                if (0 != l_poDesc->bDeviceClass)
                {
                    if(l_poDesc->bDeviceClass == LIBUSB_CLASS_HUB)
                    {
                        ETG_TRACE_USR1(("GetDeviceTypeAndDeviceClass: DTY_HUB"));
                        l_eType = CGlobalEnumerations::DTY_HUB;
                        f_DeviceClass = LIBUSB_CLASS_HUB;
                    }
                }
            }
            else
            {
                if(strName.contains("Host to Host Bridge"))
                {
                    ETG_TRACE_USR1(("GetDeviceTypeAndDeviceClass: Host to Host Bridge - ignored (side info:used for carplay)"));
                }
                else
                {
                    ETG_TRACE_USR1(("GetDeviceTypeAndDeviceClass: unknown)"));
                }
            }

            return l_eType;
        }

        //check for BTA devices
        if ( ConfigurationManager::GetInstance()->bFilterIsTelematicsBox(l_poDesc->idVendor,l_poDesc->idProduct) )
        {
            ETG_TRACE_USR1(("GetDeviceTypeAndDeviceClass: BTA('Box Telematic Autonomous') device found"));
            l_eType = CGlobalEnumerations::DTY_BTA;
            return l_eType;
        }
        // check for apple devices
        if ((AppleVendorID == l_poDesc->idVendor)
                && 0x12 == (l_poDesc->idProduct >> 8))
        {
            ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: Apple device found"));
            l_eType = CGlobalEnumerations::DTY_IPOD;
        }
#ifdef USE_EXTERNAL_UNSUPPORTED_DEVICES
        else if( (l_poDesc->idVendor == AppleVendorID) && ((l_poDesc->idProduct >> 8) == 0x13) )
        {
            ETG_TRACE_USR1(( "GetDeviceTypeAndDeviceClass: not supported Apple device found" ));
            l_eType = CGlobalEnumerations::DTY_IPOD;
        }
#endif // USE_EXTERNAL_UNSUPPORTED_DEVICES
        else
        {
            // If bDeviceClass is 0 then device type is set at interface level
            if (0 != l_poDesc->bDeviceClass)
            {
                ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: Device class found is %d",l_poDesc->bDeviceClass));


                switch (l_poDesc->bDeviceClass)
                {
                case  LIBUSB_CLASS_MASS_STORAGE:
                    ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: Storage device found"));
                    l_eType = CGlobalEnumerations::DTY_USB;
                    f_DeviceClass = LIBUSB_CLASS_MASS_STORAGE;
                    break;
                case LIBUSB_CLASS_HUB:
                    ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: Hub device found"));
                    l_eType = CGlobalEnumerations::DTY_HUB;
                    f_DeviceClass = LIBUSB_CLASS_HUB;
                    break;
                case LIBUSB_CLASS_PTP:
                    ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: MTP device found"));
                    l_eType = CGlobalEnumerations::DTY_MTP;
                    f_DeviceClass = LIBUSB_CLASS_PTP;
                    break;
#ifdef USE_MYSPIN_CHANGE
                case _LIBUSB_CLASS_MISCELLANEUS: /* 239 Miscellaneous devices for MYSPIN*/

                    ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: MISCELLANEUS: device found might be MYSPIN"));
                    l_eType = CGlobalEnumerations::DTY_MISCELLANEOUS;
                    break;
#endif
                default:
                    ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: UNKNOWN device"));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_PER_INTERFACE        : 0x%x",LIBUSB_CLASS_PER_INTERFACE));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_AUDIO                : 0x%x",LIBUSB_CLASS_AUDIO));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_COMM                 : 0x%x",LIBUSB_CLASS_COMM));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_HID                  : 0x%x",LIBUSB_CLASS_HID));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_PHYSICAL             : 0x%x",LIBUSB_CLASS_PHYSICAL));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_PRINTER              : 0x%x",LIBUSB_CLASS_PRINTER));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_PTP                  : 0x%x",LIBUSB_CLASS_PTP));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_IMAGE                : 0x%x",LIBUSB_CLASS_IMAGE));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_MASS_STORAGE         : 0x%x",LIBUSB_CLASS_MASS_STORAGE));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_HUB                  : 0x%x",LIBUSB_CLASS_HUB));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_DATA                 : 0x%x",LIBUSB_CLASS_DATA));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_SMART_CARD           : 0x%x",LIBUSB_CLASS_SMART_CARD));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_CONTENT_SECURITY     : 0x%x",LIBUSB_CLASS_CONTENT_SECURITY));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_VIDEO                : 0x%x",LIBUSB_CLASS_VIDEO));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_PERSONAL_HEALTHCARE  : 0x%x",LIBUSB_CLASS_PERSONAL_HEALTHCARE));
                    ETG_TRACE_USR1 (("_LIBUSB_CLASS_AUDIOVIDEODEVICES   : 0x%x",_LIBUSB_CLASS_AUDIOVIDEODEVICES));
                    ETG_TRACE_USR1 (("_LIBUSB_CLASS_DILLBOARDDEVICE     : 0x%x",_LIBUSB_CLASS_DILLBOARDDEVICE));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_DIAGNOSTIC_DEVICE    : 0x%x",LIBUSB_CLASS_DIAGNOSTIC_DEVICE));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_WIRELESS             : 0x%x",LIBUSB_CLASS_WIRELESS));
                    ETG_TRACE_USR1 (("_LIBUSB_CLASS_MISCELLANEUS         : 0x%x",_LIBUSB_CLASS_MISCELLANEUS));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_APPLICATION          : 0x%x",LIBUSB_CLASS_APPLICATION));
                    ETG_TRACE_USR1 (("LIBUSB_CLASS_VENDOR_SPEC          : 0x%x",LIBUSB_CLASS_VENDOR_SPEC));

                    l_eType = CGlobalEnumerations::DTY_UNKNOWN;
                    break;
                }
            }
            else
            {
                bool l_cDevTypeFound = false;

                // check for supported MTP devices
                LIBMTP_device_entry_t *l_pDevices;
                int l_iCount = 0;

                LIBMTP_Get_Supported_Devices_List(&l_pDevices, &l_iCount);

                if (l_iCount)
                {
                    for (int itr = 0;itr < l_iCount; itr++)
                    {
                        if ((l_poDesc->idVendor == l_pDevices[itr].vendor_id)
                                && (l_poDesc->idProduct == l_pDevices[itr].product_id))
                        {
                            ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: Found in supported MTP device list"));
                            l_eType = CGlobalEnumerations::DTY_MTP;
                            l_cDevTypeFound = true;
                            break;
                        }
                    }
                }

                if (!l_cDevTypeFound)
                {
                    // device type defined at interface level
                    for (uint8_t i = 0; i < l_poDesc->bNumConfigurations; i++)
                    {
                        struct libusb_config_descriptor *l_poConfig;

                        // get the configuration
                        if (!libusb_get_config_descriptor (f_poDev, i, &l_poConfig))
                        {
                            // hack for USB with INTERRUPT interface
                            if (128 == l_poConfig->bConfigurationValue)
                            {
                                ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: MTP device found"));
                                l_eType = CGlobalEnumerations::DTY_MTP;
                            }
                            else
                            {
                                // get the interfaces
                                for (int j = 0; j < l_poConfig->bNumInterfaces; j++)
                                {
                                    const struct libusb_interface *l_poIfce = &l_poConfig->interface[j];

                                    // get the interface settings
                                    for (int k = 0; k < l_poIfce->num_altsetting; k++)
                                    {
                                        const struct libusb_interface_descriptor *l_poIfceDesc =
                                                &l_poIfce->altsetting[k];


                                        ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: bInterfaceClass %d ",l_poIfceDesc->bInterfaceClass));
                                        // check the endpoints (for MTP)
                                        if ((uint8_t)LIBUSB_CLASS_MASS_STORAGE == l_poIfceDesc->bInterfaceClass)
                                        {
                                            ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: Storage device found"));
                                            ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: bSuppressMassStorage: 0x%x",bSuppressMassStorage));
                                            f_DeviceClass = LIBUSB_CLASS_MASS_STORAGE;
                                            ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: f_DeviceClass: %d",f_DeviceClass));
                                            if(FALSE == bSuppressMassStorage)
                                            {
                                                l_eType = CGlobalEnumerations::DTY_USB;
                                            }
#if 0
                                            for (int m = 0; m < l_poIfceDesc->bNumEndpoints; m++)
                                            {
                                                const struct libusb_endpoint_descriptor *l_poEndpoint =
                                                        &l_poIfceDesc->endpoint[m];

                                                /* If the endpoint for a storage
                                                 * device supports INTERRUPT type
                                                 * then the device is assumed to
                                                 * be a MTP device
                                                 */
                                                if (3 == (l_poEndpoint->bmAttributes & 0x3))
                                                {
                                                    ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: Device is MTP"));
                                                    l_eType = CGlobalEnumerations::DTY_MTP;
                                                }
                                            }
#endif
                                        }
                                        else if ((uint8_t)LIBUSB_CLASS_PTP == l_poIfceDesc->bInterfaceClass)
                                        {
                                            ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: MTP device found"));
                                            l_eType = CGlobalEnumerations::DTY_MTP;
                                            f_DeviceClass = LIBUSB_CLASS_PTP;
                                        }
                                        else
                                        {
                                            /* hack for SanDisk Sansa Clip+
                                             * SanDisk Sansa Clip+ shows interface name with "PIMA"
                                             * PIMA corresponds to "Picture Transfer Protocol"
                                             */
                                            // get the interface string
                                            if (0 != l_poIfceDesc->iInterface)
                                            {
                                                char l_sInterface[128];
                                                LibUSBGetString (f_poDevHdl, l_poIfceDesc->iInterface,
                                                                 (unsigned char*)l_sInterface, sizeof(l_sInterface));

                                                ETG_TRACE_USR4 (("GetDeviceTypeAndDeviceClass: Interface string: %s",
                                                                 l_sInterface));

                                                f_sInterfacestr = l_sInterface;

                                                if (strstr (l_sInterface, "PIMA"))
                                                {
                                                    ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: MTP device found (PIMA)"));
                                                    l_eType = CGlobalEnumerations::DTY_MTP;
                                                }
                                                else if (strstr (l_sInterface, "MTP")) //ticket:
                                                {
                                                    ETG_TRACE_USR1 (("GetDeviceTypeAndDeviceClass: MTP device found (MTP)")); //Ticket: GMMY16-16072[C3_FINAL] MotoX phone not shown as a media source when connected through USB - BLOCKER #5
                                                    l_eType = CGlobalEnumerations::DTY_MTP;
                                                }
                                            }
                                        }
                                    }
                                }

                                libusb_free_config_descriptor (l_poConfig);
                            }
                        }
                    }
                }
            }
        }
    }
    else
    {
        ETG_TRACE_ERR (("GetDeviceTypeAndDeviceClass: [ERROR] null arguments"));
    }

    ETG_TRACE_USR3 (("GetDeviceTypeAndDeviceClass: End"));

    return (l_eType);
}

#if 0
//seems to be unused code
/*-----------------------------------------------------------------------------*
 * CGlobalEnumerations::DEVICE_TYPE_Type LibUSBGetDeviceType (                 *
 *              libusb_device *f_poDev)                                        *
 *-----------------------------------------------------------------------------*/
CGlobalEnumerations::DEVICE_TYPE_Type LibUSBGetDeviceType (libusb_device *f_poDev)
{
    ETG_TRACE_USR3 (("LibUSBGetDeviceType: Begin"));

    CGlobalEnumerations::DEVICE_TYPE_Type l_iType = CGlobalEnumerations::DTY_UNKNOWN;

    if (NULL == f_poDev)
    {
        ETG_TRACE_ERR (("LibUSBGetDeviceType: [ERROR] f_poDev is null"));
    }
    else
    {
        libusb_device_descriptor l_poDesc;

        int r = libusb_get_device_descriptor(f_poDev, &l_poDesc);

        if (r < 0)
        {
            ETG_TRACE_ERR (("LibUSBGetDeviceType: [ERROR] failed to get device descriptor"));
        }
        else
        {
            libusb_device_handle *l_poHandle = NULL;
            libusb_class_code l_DeviceClass = LIBUSB_CLASS_HID;
            int iRet = libusb_open (f_poDev, &l_poHandle);
            if (iRet >= 0)
            {
                l_iType = GetDeviceTypeAndDeviceClass(f_poDev, l_poHandle, &l_poDesc, &l_DeviceClass);

                // cleanup
                libusb_close (l_poHandle);
            }
            else
            {
                ETG_TRACE_ERR(("LibUSBGetDeviceType: [ERROR] iRes=libusb_open(...) %d (%s)",(int)iRes,libusb_strerror((libusb_error)iRes)));
#ifdef USE_LIBUSBMTP_ERRMEM
                ETG_TRACE_ERRMEM(("LibUSBGetDeviceType: [ERROR] iRes=libusb_open(...) %d (%s)",(int)iRes,libusb_strerror((libusb_error)iRes)));
#endif //USE_LIBUSBMTP_ERRMEM
            }
        }
    }

    ETG_TRACE_USR1 (("LibUSBGetDeviceType: Device type = %d", l_iType));

    ETG_TRACE_USR3 (("LibUSBGetDeviceType: End"));

    return (l_iType);
}
#endif

/*-----------------------------------------------------------------------------*
 * int LibUSBFillDeviceDetails (libusb_device *f_poDev, CDevice& f_poDevice)   *
 *-----------------------------------------------------------------------------*/
int LibUSBFillDeviceDetails (libusb_device *f_poDev, CDevice& f_poDevice)
{
    ETG_TRACE_USR3 (("LibUSBFillDeviceDetails: Begin"));

    int l_iError = DEVICEMANAGER_OK;

    if (NULL == f_poDev)
    {
        ETG_TRACE_ERR (("LibUSBFillDeviceDetails: [ERROR] null device"));
        l_iError = DEVICEMANAGER_ERROR_NULL_POINTER;
    }
    else
    {
        libusb_device_descriptor l_poDesc;

        int r = libusb_get_device_descriptor(f_poDev, &l_poDesc);

        if (r < 0)
        {
            ETG_TRACE_ERR (("[ERROR] LibUSBShowDeviceDetails: [ERROR] r=libusb_get_device_descriptor(...) r:%d (%s)",(int)r,libusb_strerror((libusb_error)r)));
#ifdef USE_LIBUSBMTP_ERRMEM
            ETG_TRACE_ERRMEM (("[ERROR] LibUSBShowDeviceDetails: [ERROR] r=libusb_get_device_descriptor(...) r=%d (%s)",(int)r,libusb_strerror((libusb_error)r)));
#endif //USE_LIBUSBMTP_ERRMEM
            ETG_TRACE_ERR (("LibUSBFillDeviceDetails: [ERROR] failed to get device descriptor %d ",r));
            l_iError = DEVICEMANAGER_ERROR_UNKNOWN;
        }
        else
        {
            libusb_device_handle *l_poHandle = NULL;
            int l_iexitstatus = libusb_open (f_poDev, &l_poHandle);
            if (l_iexitstatus >= 0)
            {
                // fill vendor/product id
                f_poDevice.m_iVendorID  = l_poDesc.idVendor;
                f_poDevice.m_iProductID = l_poDesc.idProduct;

                // fill device num
                f_poDevice.m_iDevNUM = libusb_get_device_address (f_poDev);

                char l_sMfg[128], l_sProduct[128], l_sShortSerial[128],
                        l_sSerial[256], l_sBcdUSB[32], l_sBcdDevice[32];;

                // fill serial id
                LibUSBGetString(l_poHandle, l_poDesc.iManufacturer,
                                (unsigned char*)l_sMfg, sizeof(l_sMfg));
                LibUSBGetString(l_poHandle, l_poDesc.iProduct,
                                (unsigned char*)l_sProduct, sizeof(l_sProduct));
                LibUSBGetString(l_poHandle, l_poDesc.iSerialNumber,
                                (unsigned char*)l_sShortSerial, sizeof(l_sShortSerial));

                f_poDevice.m_cShortSerial = GENSTRING(l_sShortSerial);
                snprintf (l_sSerial, sizeof(l_sSerial), "%s %s %s", l_sMfg,
                          l_sProduct, l_sShortSerial);
                f_poDevice.m_cSerialID = GENSTRING(l_sSerial).trimmed();
                f_poDevice.m_cSerialID.replace(" ", "_");

                f_poDevice.m_cManufacturer = GENSTRING(l_sMfg).trimmed();

                // fill device version
                snprintf (l_sBcdUSB, sizeof(l_sBcdUSB), "%2x.%02x",
                          l_poDesc.bcdUSB >> 8, l_poDesc.bcdUSB & 0xff);
                f_poDevice.m_cDeviceVersion = GENSTRING(l_sBcdUSB);

                // fill binary coded value which means each 4 bits give a decimal value
                //see https://en.wikipedia.org/wiki/Binary-coded_decimal
                //Decimal:            1        2      3
                // Binary :  0000 0001  0010 0011
                tU8 Dig4,Dig3,Dig2,Dig1; /*each of ot is 4bit*/
                Dig4 = (l_poDesc.bcdDevice & 0xF000) >> 12;
                Dig3 = (l_poDesc.bcdDevice & 0x0F00) >>  8;
                Dig2 = (l_poDesc.bcdDevice & 0x00F0) >>  4;
                Dig1 = (l_poDesc.bcdDevice & 0x000F) >>  0;
                snprintf (l_sBcdDevice, sizeof(l_sBcdDevice), "%1d%1d%1d%1d", Dig4,Dig3,Dig2,Dig1);
                f_poDevice.m_cBcdDevice = GENSTRING(l_sBcdDevice);

                // fill device type
                tBool bSupressMassStorageOutput = FALSE;
                if(f_poDevice.m_eTypeOfNotify == CGlobalEnumerations::enUnotify) //i.e. not called by automounter
                {
                    bSupressMassStorageOutput = TRUE;
                }
                libusb_class_code l_DeviceClass = LIBUSB_CLASS_HID;
                GENSTRING l_sInterfacestr;
                f_poDevice.m_eDeviceType = GetDeviceTypeAndDeviceClass(f_poDev, l_poHandle, &l_poDesc,IN bSupressMassStorageOutput, OUT l_DeviceClass,OUT l_sInterfacestr);
                if(l_sInterfacestr == "Android Accessory Interface")
                {
                    f_poDevice.m_eInterfaceDeviceClass = CGlobalEnumerations::INTERFACE_DEVICE_CLASS_MTP;
                }
                else if (l_sInterfacestr == "MIDI function")
                {
                    f_poDevice.m_eInterfaceDeviceClass = CGlobalEnumerations::INTERFACE_DEVICE_CLASS_AUDIO;
                }

                // fill device name
                f_poDevice.m_cDeviceName = GENSTRING(l_sProduct);

                // fill device speed
                int iDeviceSpeed = libusb_get_device_speed(f_poDev);
                if (iDeviceSpeed < 0)
                {
                     ETG_TRACE_ERR(("[[ERROR] LibUSBFillDeviceDetails:iDeviceSpeed=libusb_get_device_speed(...) iDeviceSpeed:%d (%s)",(int)iDeviceSpeed,libusb_strerror((libusb_error)iDeviceSpeed)));
#ifdef USE_LIBUSBMTP_ERRMEM
                     ETG_TRACE_ERRMEM (("[[ERROR] LibUSBFillDeviceDetails:iDeviceSpeed=libusb_get_device_speed(...) iDeviceSpeed:%d (%s)",(int)iDeviceSpeed,libusb_strerror((libusb_error)iDeviceSpeed)));
#endif //USE_LIBUSBMTP_ERRMEM
                    l_iError = DEVICEMANAGER_ERROR_UNKNOWN;
                }
                else
                {
                    f_poDevice.m_DeviceSpeedUSB = (statetbl::tenDeviceSpeedUSB)iDeviceSpeed;
                }
                ETG_TRACE_USR4 (("LibUSBFillDeviceDetails:  l_DeviceClass)            %d", l_DeviceClass));
                /*Fill Device class*/
                switch (l_DeviceClass)
                {
                    case  LIBUSB_CLASS_HID:
                        ETG_TRACE_USR1 (("LibUSBFillDeviceDetails: HID device found"));
                        f_poDevice.m_eDeviceClass = CGlobalEnumerations::DEVICE_CLASS_HID;
                        break;
                    case  LIBUSB_CLASS_MASS_STORAGE:
                        ETG_TRACE_USR1 (("LibUSBFillDeviceDetails: Storage device found"));
                        f_poDevice.m_eDeviceClass = CGlobalEnumerations::DEVICE_CLASS_MASS_STORAGE;
                        break;
                    case LIBUSB_CLASS_HUB:
                        ETG_TRACE_USR1 (("LibUSBFillDeviceDetails: Hub device found"));
                        f_poDevice.m_eDeviceClass = CGlobalEnumerations::DEVICE_CLASS_HUB;
                        break;
                    case LIBUSB_CLASS_PTP:
                        ETG_TRACE_USR1 (("LibUSBFillDeviceDetails"));
                        f_poDevice.m_eDeviceClass = CGlobalEnumerations::DEVICE_CLASS_MTP;
                        break;
                    default:
                        ETG_TRACE_USR4 (("LibUSBFillDeviceDetails:Default.. Set to DEVICE_CLASS_HID"));
                        f_poDevice.m_eDeviceClass = CGlobalEnumerations::DEVICE_CLASS_HID;
                        break;
                }

                ETG_TRACE_USR4 (("----------- USB details ----------------------------------------------------"));
                ETG_TRACE_USR4 (("LibUSBFillDeviceDetails:  bcdUSB        %s", l_sBcdUSB));
                ETG_TRACE_USR4 (("LibUSBFillDeviceDetails:  Vendor ID     0x%04x", f_poDevice.m_iVendorID));
                ETG_TRACE_USR4 (("LibUSBFillDeviceDetails:  Product ID    0x%04x", f_poDevice.m_iProductID));
                ETG_TRACE_USR4 (("LibUSBFillDeviceDetails:  Manufacturer  %s", l_sMfg));
                ETG_TRACE_USR4 (("LibUSBFillDeviceDetails:  Product       %s", l_sProduct));
                ETG_TRACE_USR4 (("LibUSBFillDeviceDetails:  Serial        %s", f_poDevice.m_cSerialID.toStdString().c_str()));
                ETG_TRACE_USR4 (("LibUSBFillDeviceDetails:  DeviceSpeed   %d", f_poDevice.m_DeviceSpeedUSB));
                ETG_TRACE_USR4 (("LibUSBFillDeviceDetails:  DeviceClass   %d", f_poDevice.m_eDeviceClass));
                ETG_TRACE_USR4 (("LibUSBFillDeviceDetails:  BcdDevice     %s", f_poDevice.m_cBcdDevice.toStdString().c_str()));
                ETG_TRACE_USR4 (("----------------------------------------------------------------------------"));

                libusb_close (l_poHandle);
            }
            else
            {
                ETG_TRACE_ERR (("LibUSBFillDeviceDetails: [WARNING] not getting libusb_device_handle %d ",l_iexitstatus));
                l_iError = DEVICEMANAGER_ERROR_UNKNOWN;
            }
        }
    }

    ETG_TRACE_USR3 (("LibUSBFillDeviceDetails: End"));

    return (l_iError);
}


/*-----------------------------------------------------------------------------*
 * void ChargeAppleDevice(uint16_t f_iVendorID, uint16_t f_iProductID)         *
 *-----------------------------------------------------------------------------*/
void ChargeAppleDevice(uint16_t f_iVendorID, uint16_t f_iProductID)
{
    ETG_TRACE_USR3 (("ChargeAppleDevice: Begin"));

    if ((0x05ac == f_iVendorID) && (0x12 == (f_iProductID >> 8)))
    {
        ETG_TRACE_USR4(("ChargeAppleDevice: valid Apple device"));

        // check libusb is initialized
        if (NULL != ctx)
        {
            //open usb device and raise current to 500mA (default) to +500mA = 1a for selected Apple device
            libusb_device_handle* l_poHdl = libusb_open_device_with_vid_pid(
                        ctx, f_iVendorID, f_iProductID);

            if (NULL != l_poHdl)
            {
                ETG_TRACE_USR4 (("ChargeAppleDevice: calling libusb_control_transfer()"));

                int transfer_ret = libusb_control_transfer(l_poHdl,
                                                           0x40 /*RequestType*/,
                                                           0x40,/*Request*/
                                                           500, /*value*/
                                                           500, /*Index*/
                                                           NULL,/*data*/
                                                           0, /*length*/
                                                           CHARGE_APPLE_TIMOUT /*timout*/);

                ETG_TRACE_USR4 (("ChargeAppleDevice: libusb_control_transfer() returned %d", transfer_ret));

                if ( (int)LIBUSB_SUCCESS == transfer_ret)
                {
                    ETG_TRACE_USR4 (("ChargeAppleDevice: SUCCESS"));
                }
                else
                {
                    if(transfer_ret < 0)
                    {
                        ETG_TRACE_ERR (("[[ERROR] ChargeAppleDevice:iResult=libusb_control_transfer(...) iResul:%d (%s)",(int)transfer_ret,libusb_strerror((libusb_error)transfer_ret)));
#ifdef USE_LIBUSBMTP_ERRMEM
                        ETG_TRACE_ERRMEM (("[[ERROR] ChargeAppleDevice:iResult=libusb_control_transfer(...) iResul:%d (%s)",(int)transfer_ret,libusb_strerror((libusb_error)transfer_ret)));
#endif //USE_LIBUSBMTP_ERRMEM
                     }
                     else
                     {
                        ETG_TRACE_USR3 (("[INFO] ChargeAppleDevice:libusb_control_transfer failed Error code iResult:%d",(int)transfer_ret));
                     }
                    ETG_TRACE_ERR (("ChargeAppleDevice: FAILED"));
                }

                // cleanup
                libusb_close (l_poHdl);
            }
            else
            {
                ETG_TRACE_ERR (("ChargeAppleDevice: [ERROR] libusb_open_device_with_vid_pid() failed"));
            }
        }
        else
        {
            ETG_TRACE_FATAL (("ChargeAppleDevice: [ERROR] libusb not initialized"));
        }
    }
    else
    {
        ETG_TRACE_ERR (("ChargeAppleDevice: [ERROR] invalid Apple device"));
    }

    ETG_TRACE_USR3 (("ChargeAppleDevice: End"));
}



#ifdef USE_EXPERIMENTAL_PSA_HUBCOMMUNICATION

//get device list and check for HUB - store device in global variable
//adjust this function to work with
// libusb_open_device_with_vid_pid could be used here but I don' know yet if there are more generic parameters for the internla HUB


//lsusb gives this output:
//Bus 002 Device 002: ID 0424:2514 Standard Microsystems Corp. USB 2.0 Hub <-------------------this is the internal HUB we are looking fore
//Bus 002 Device 001: ID 1d6b:0002 Linux Foundation 2.0 root hub
//Bus 001 Device 002: ID 2001:3c05 D-Link Corp. DUB-E100 Fast Ethernet Adapter(rev.B1) [ASIX AX88772] (Adapter connected ot PC used for Trace)
//Bus 001 Device 001: ID 1d6b:0002 Linux Foundation 2.0 root hub ()


//using VD_DVM_HUBINTERNAL_TEST these is the one we are looking for
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :----------- LibUsbScanForInternalHUB (device) -------------------------------------- [usbutils.cpp(1082)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  Vendor ID          0x0424 [usbutils.cpp(1083)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  Product ID         0x2514 [usbutils.cpp(1084)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  Address            0x0002 [usbutils.cpp(1085)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  Bus Number         0x0002 [usbutils.cpp(1086)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  Device Speed       0x0003 [usbutils.cpp(1087)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  bcdUSB              2.00 [usbutils.cpp(1088)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  Manufacturer        [usbutils.cpp(1089)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  Product             [usbutils.cpp(1090)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  Serial              [usbutils.cpp(1091)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  Class           9 [usbutils.cpp(1092)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  SubClass           0 [usbutils.cpp(1093)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  USB specification release number         512 [usbutils.cpp(1094)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  Maximum packet size          64 [usbutils.cpp(1095)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :GetInternalHubHandle:  Device release number in binary-coded decimal       80a0 [usbutils.cpp(1096)]
//<gen3flex@dlt>(core0)VD_DVM_MGR_UDEV_US4 :---------------------------------------------------------------------------- [usbutils.cpp(1097)]


libUsbHandle* GetInternalHubHandle()
{
    ETG_TRACE_USR4(("GetInternalHubHandle: Begin"));
    libUsbHandle *l_poHandleInternalHub = NULL;
    tBool l_bFound = FALSE;

    if (NULL == ctx)
    {
        ETG_TRACE_ERR (("GetInternalHubHandle: [ERROR] libusb not initialized"));
    }
    else
    {
        libusb_device **l_ppDevList,*l_poDev;
        ssize_t         l_iNumDevs, itr;



        //get list of all connected usb devices
        l_iNumDevs = libusb_get_device_list (ctx, &l_ppDevList);
        if(l_iNumDevs < 0)
        {
            ETG_TRACE_ERR(("GetInternalHubHandle: [ERROR] l_iNumDevs=libusb_get_device_list(...) %d (%s)",(int)l_iNumDevs,libusb_strerror((libusb_error)l_iNumDevs)));
#ifdef USE_LIBUSBMTP_ERRMEM
            ETG_TRACE_ERRMEM (("GetInternalHubHandle: [ERROR] l_iNumDevs=libusb_get_device_list(...) %d (%s)",(int)l_iNumDevs,libusb_strerror((libusb_error)l_iNumDevs)));
#endif //USE_LIBUSBMTP_ERRMEM
        }
        else
        {
            ETG_TRACE_USR4(("GetInternalHubHandle: l_iNumDevs: %d",l_iNumDevs));

            //iterate through the devices
            for (itr = 0; itr < l_iNumDevs; itr++)
            {
                l_poDev = l_ppDevList[itr];

                if (NULL == l_poDev)
                {
                    ETG_TRACE_ERR (("GetInternalHubHandle: [ERROR] null device: itr:%d",itr));
                }
                else
                {
                    //get device descriptor of device
                    libusb_device_descriptor l_poDesc;
                    /*
                    please check how to output also this information
                    uint8_t     bLength
                        Size of this descriptor (in bytes)
                    uint8_t     bDescriptorType
                        Descriptor type.
                    uint16_t     bcdUSB
                        USB specification release number in binary-coded decimal.
                    uint8_t     bDeviceClass
                        USB-IF class code for the device.
                    uint8_t     bDeviceSubClass
                        USB-IF subclass code for the device, qualified by the bDeviceClass value.
                    uint8_t     bDeviceProtocol
                        USB-IF protocol code for the device, qualified by the bDeviceClass and bDeviceSubClass values.
                    uint8_t     bMaxPacketSize0
                        Maximum packet size for endpoint 0.
                    uint16_t     idVendor
                        USB-IF vendor ID.
                    uint16_t     idProduct
                        USB-IF product ID.
                    uint16_t     bcdDevice
                        Device release number in binary-coded decimal.
                    uint8_t     iManufacturer
                        Index of string descriptor describing manufacturer.
                    uint8_t     iProduct
                        Index of string descriptor describing product.
                    uint8_t     iSerialNumber
                        Index of string descriptor containing device serial number.
                    uint8_t     bNumConfigurations
                        Number of possible configurations.
    */
                    int r = libusb_get_device_descriptor(l_poDev, &l_poDesc);

                    if (r < 0)
                    {
                        ETG_TRACE_ERR (("GetInternalHubHandle: [ERROR] failed to get device descriptor"));
#ifdef USE_LIBUSBMTP_ERRMEM
                        ETG_TRACE_ERRMEM(("GetInternalHubHandle: [ERROR] r=libusb_get_device_descriptor(...) %d (%s)",(int)r,libusb_strerror((libusb_error)r)));
#endif //USE_LIBUSBMTP_ERRMEM
                    }
                    else
                    {
                        //open device
                        libUsbHandle *l_poHandle = NULL;
                        int iRes = libusb_open (l_poDev, &l_poHandle);
                        if(iRes<0)
                        {
                            ETG_TRACE_ERR(("GetInternalHubHandle: [ERROR] iRes=libusb_open(...) %d (%s)",(int)iRes,libusb_strerror((libusb_error)iRes)));
#ifdef USE_LIBUSBMTP_ERRMEM
                            ETG_TRACE_ERRMEM(("GetInternalHubHandle: [ERROR] iRes=libusb_open(...) %d (%s)",(int)iRes,libusb_strerror((libusb_error)iRes)));
#endif //USE_LIBUSBMTP_ERRMEM
                        }
                        else /* if >= 0*/
                        {
                            int idVendor = 0;
                            int idProduct = 0;
                            libusb_class_code l_DeviceClass = LIBUSB_CLASS_HID;
                            // fill device type and Device Class
                            GENSTRING l_sInterfacestr;
                            if(CGlobalEnumerations::DTY_HUB == GetDeviceTypeAndDeviceClass(l_poDev, l_poHandle, &l_poDesc, FALSE, l_DeviceClass,l_sInterfacestr))
                            {
                                int iAddress                 = libusb_get_device_address(l_poDev);
                                int iBusNumber               = libusb_get_bus_number(l_poDev);
                                int iDeviceSpeed             = libusb_get_device_speed(l_poDev);
                                int iDeviceClass             = l_poDesc.bDeviceClass;
                                int iDevicecdUSB             = l_poDesc.bcdUSB;
                                int iDeviceSubClass          = l_poDesc.bDeviceSubClass;
                                int iDeviceMaxPacketSize     = l_poDesc.bMaxPacketSize0;
                                int iDevicecdDevice          = l_poDesc.bcdDevice;


                                // fill vendor/product id
                                idVendor  = l_poDesc.idVendor;
                                idProduct = l_poDesc.idProduct;


                                char l_sMfg[128], l_sProduct[128], l_sShortSerial[128],l_sBcdUSB[32];
                                snprintf (l_sBcdUSB, sizeof(l_sBcdUSB), "%2x.%02x",l_poDesc.bcdUSB >> 8, l_poDesc.bcdUSB & 0xff);
                                LibUSBGetString(l_poHandle, l_poDesc.iManufacturer,(unsigned char*)l_sMfg, sizeof(l_sMfg));
                                LibUSBGetString(l_poHandle, l_poDesc.iProduct,(unsigned char*)l_sProduct, sizeof(l_sProduct));
                                LibUSBGetString(l_poHandle, l_poDesc.iSerialNumber,(unsigned char*)l_sShortSerial, sizeof(l_sShortSerial));

                                //show result
                                ETG_TRACE_USR4 (("----------- GetInternalHub (device) --------------------------------------"));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  Vendor ID          0x%04x", idVendor));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  Product ID         0x%04x", idProduct));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  Address            0x%04x", iAddress));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  Bus Number         0x%04x", iBusNumber));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  Device Speed       0x%04x", iDeviceSpeed));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  bcdUSB             %s", l_sBcdUSB));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  Manufacturer       %s", l_sMfg));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  Product            %s", l_sProduct));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  Serial             %s", l_sShortSerial));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  Class       %5u",iDeviceClass));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  SubClass       %5u",iDeviceSubClass));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  USB specification release number       %5u",iDevicecdUSB));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  Maximum packet size       %5u",iDeviceMaxPacketSize));
                                ETG_TRACE_USR4 (("GetInternalHubHandle:  Device release number in binary-coded decimal       %x",iDevicecdDevice));
                                ETG_TRACE_USR4 (("----------------------------------------------------------------------------"));


                            }



                            if ( ConfigurationManager::GetInstance()->bFilterIsInternalHub(IN idVendor,IN idProduct ) )
                            {

                                ETG_TRACE_USR4 (("GetInternalHubHandle: Internal HUB found"));
                                l_bFound = TRUE;
                                l_poHandleInternalHub = l_poHandle;
                                break;

                            }
                            else
                            {
                                ETG_TRACE_USR4 (("GetInternalHubHandle: Close usb handle"));
                                libusb_close (l_poHandle);
                            }

                        }
                    }
                }
            }
        }
        //free lists
        if(l_iNumDevs >=0)
        {
            ETG_TRACE_USR3 (("GetInternalHubHandle: free devicelist"));
            libusb_free_device_list (l_ppDevList, 1);
        }
    }

    ETG_TRACE_USR4(("GetInternalHubHandle: End"));
    if(l_bFound)
    {
        return l_poHandleInternalHub;
    }
    else
    {
        return NULL;
    }

}


tBool isDeviceConnected(uint16_t f_iVendorID, uint16_t f_iProductID)
{
    ETG_TRACE_USR4(("isDeviceConnected: Begin"));
    ETG_TRACE_USR4(("isDeviceConnected: Vendor ID 0x%x",f_iVendorID));
    ETG_TRACE_USR4(("isDeviceConnected: Product ID 0x%x",f_iProductID));
    tBool bFound = FALSE;
    // check libusb is initialized
    if (NULL != ctx)
    {
       //open usb device and raise current to 500mA (default) to +500mA = 1a for selected Apple device
       libusb_device_handle* l_poHdl = libusb_open_device_with_vid_pid(
                   ctx, f_iVendorID, f_iProductID);
       if (NULL != l_poHdl)
       {
           ETG_TRACE_USR4(("isDeviceConnected: Device connected"));
           bFound = TRUE;
           // cleanup
           libusb_close (l_poHdl);
       }
       else
       {
#ifdef USE_LIBUSBMTP_ERRMEM
           ETG_TRACE_ERRMEM (("[[ERROR] isDeviceConnected: NULL=libusb_open_device_with_vid_pid(...)"));
#endif //USE_LIBUSBMTP_ERRMEM

           ETG_TRACE_USR4(("isDeviceConnected: Device not connected"));
       }
    }
    else
    {
        ETG_TRACE_FATAL (("isDeviceConnected: [ERROR] libusb not initialized"));
    }
    ETG_TRACE_USR4(("isDeviceConnected: End"));
    return bFound;
}





void LibUsbScanForInternalHUB()
{
    ETG_TRACE_USR4(("LibUsbScanForInternalHUB: Begin"));


    libUsbHandle *l_poHandle = NULL;
    l_poHandle = GetInternalHubHandle();
    if (NULL != l_poHandle)
    {
        ETG_TRACE_USR4 (("LibUsbScanForInternalHUB: Internal HUB found"));
        GetHubStatus(l_poHandle);
        GetPortStatus(l_poHandle,eUSB2);
        GetPortStatus(l_poHandle,eUSB3);
    }
    else
    {
        ETG_TRACE_USR4 (("!!!!!!!LibUsbScanForInternalHUB: Internal HUB Not found!!!!!!!"));

    }
    libusb_close (l_poHandle);
    ETG_TRACE_USR3 (("LibUsbScanForInternalHUB: End"));

    return ;
}


tBool bIsDeviceAtFullSpeed(tenDevManagerUSBHost eUsb,tBool &bIsDevicePresent)
{
    libUsbHandle *l_poHandle = NULL;
    l_poHandle = GetInternalHubHandle();
    tBool l_bFullSpeedDeviceFound = FALSE;
    if (NULL != l_poHandle)
    {
        ETG_TRACE_USR3 (("bIsDeviceAtFullSpeed: Begin"));

        int l_iPortNumber = -1;

        //TODO->Note this part has tobe made configurable dependent
        if(eUsb == eUSB2)
        {
            l_iPortNumber = 1;
        }
        else if(eUsb == eUSB3)
        {
            l_iPortNumber = 2;
        }
        //-------------------------<-

        unsigned char data[4];
        if(l_iPortNumber > 0)
        {
            int iResult = libusb_control_transfer(l_poHandle,
                                                  0xA3 /*GetHubStatus see  11.24.2: looking still for a define from libusb for this*/,
                                                  LIBUSB_REQUEST_GET_STATUS,
                                                  0,
                                                  l_iPortNumber,
                                                  data,
                                                  4, //number of bytes returned  consists of 2 'words' i.e. wHubStatus then wHubChange
                                                  1000 /*ms timout*/);


            ETG_TRACE_USR4(("bIsDeviceAtFullSpeed: iResult: %d",iResult));
            if(iResult == 4)
            {
                //taken from  'Universal Serial Bus Specification Revision 2.0' 11.24.2.7 Get Port Status
                //expected 2 words: 1st: wPortStatus and   2nd: wPortChange

                tU16 wPortStatus = (data[1] <<8) + data[0];
                std::bitset<16> wPortStatusBits(wPortStatus);
                ETG_TRACE_USR3 (("bIsDeviceAtFullSpeed: wPortStatusBits  = %s (Binary)", wPortStatusBits.to_string().c_str()));


                //==========================================================
                // evaluate bits  of wPortStatus
                //==========================================================

                //-------------------------------
                // evaluate wPortStatusBits Bit 0
                //-------------------------------
                //Bit 0: Current Connect Status: (PORT_CONNECTION) This field reflects whether or not a device is currently connected to this port.
                //         0 = No device is present.
                //         1 = A device is present on this port.
                unsigned int i = 0;
                //TODO check for the device type
                if(false == wPortStatusBits.test(i))
                {
                    ETG_TRACE_USR4(("bIsDeviceAtFullSpeed: wPortStatusBits[%2d]: 0: No device is present ",i));
                }
                else
                {
                    bIsDevicePresent = TRUE;
                    ETG_TRACE_USR4(("bIsDeviceAtFullSpeed: wPortStatusBits[%2d]: 1: A device is present on this port",i));
                }


                if(bIsDevicePresent)
                {
                    //-------------------------------
                    // evaluate wPortStatusBits Bit 10
                    //-------------------------------
                    //Bit 10: High-speed Device Attached: (PORT_HIGH_SPEED) This is relevant only if a device is attached.
                    //         0 = Full-speed device attached to this port.
                    //         1 = High-speed device attached to this port.
                    i = 10;
                    if( false == wPortStatusBits.test(i))
                    {
                        ETG_TRACE_USR4(("bIsDeviceAtFullSpeed: wPortStatusBits[%2d]: 0: Full-speed device attached to this port.",i));
                    }
                    else
                    {
                        l_bFullSpeedDeviceFound = TRUE;
                        ETG_TRACE_USR4(("bIsDeviceAtFullSpeed: wPortStatusBits[%2d]: 1: High-speed device attached to this port.",i));
                    }
                }
                //bit 5-15 are reserved and set to 0
            }
            else
            {
                if(iResult < 0)
                {
                     ETG_TRACE_ERR (("[[ERROR] bIsDeviceAtFullSpeed:iResult=libusb_control_transfer(...) iResul:%d (%s)",(int)iResult,libusb_strerror((libusb_error)iResult)));
#ifdef USE_LIBUSBMTP_ERRMEM
                     ETG_TRACE_ERRMEM (("[[ERROR] bIsDeviceAtFullSpeed:iResult=libusb_control_transfer(...) iResul:%d (%s)",(int)iResult,libusb_strerror((libusb_error)iResult)));
#endif //USE_LIBUSBMTP_ERRMEM
                }
                else
                {
                     ETG_TRACE_USR3 (("[INFO] bIsDeviceAtFullSpeed:libusb_control_transfer failed Error code iResult:%d",iResult));
                }
            }
        }
        else
        {
            ETG_TRACE_USR3 (("bIsDeviceAtFullSpeed:invalid port number!!!!!!!!!  got the port number %d",l_iPortNumber));
        }
        ETG_TRACE_USR3 (("bIsDeviceAtFullSpeed:Closing the libusb handle"));
        libusb_close (l_poHandle);
        ETG_TRACE_USR3 (("bIsDeviceAtFullSpeed: End"));

    }
    return l_bFullSpeedDeviceFound;
}





//please refer to USBSpec
//file://\\bosch.com\dfsrb\DfsDE\DIV\CM\DI\Projects\Common\CarNetworks\USB\Specs\USB_2.0\usb_20.pdf
//look for 11.24.2:
//please refer to http://libusb.sourceforge.net/api-1.0/group__syncio.html to care for return values etc. libusb_control_transfer


//next label will contain this
//see 11.24.2.7 in //file://\\bosch.com\dfsrb\DfsDE\DIV\CM\DI\Projects\Common\CarNetworks\USB\Specs\USB_2.0\usb_20.pdf
bool GetHubStatus(libUsbHandle* f_poHandle)
{
    ETG_TRACE_USR3 (("GetHubStatus: Begin"));

    unsigned char data[4];
    tBool l_bHasOC = FALSE;
    int iResult = libusb_control_transfer(f_poHandle,
                                          0xA0  /*GetHubStatus see  11.24.2: looking still for a define from libusb for this*/,
                                          LIBUSB_REQUEST_GET_STATUS,
                                          0,
                                          0,
                                          data,
                                          4, //number of bytes returned  consists of 2 'words' i.e. wHubStatus then wHubChange
                                          1000 /*ms*/);

    ETG_TRACE_USR4(("GetHubStatus: iResult: %d",iResult));
    if(iResult == 4)
    {
        //taken from  'Universal Serial Bus Specification Revision 2.0' 11.24.2.6 Get Hub Status
        //expected 2 words: 1st: wHubStatus  2nd: wHubChange
        //=============================================================
        //wHubStatus consists of data[0] and data[1] - Little endian, aus USB Spec 2.0 is used
        //=============================================================
        // e.g. data[0]=  00000011    (Bit index 0-7) ('LocalPower Supply lost', 'A hub over current condition exists')
        //        data[1] = 00000000    (Bit index 8-15)
        //wHubStatus =  00000000 00000011

        //wHubStatusBits : 00001100
        tU16 wHubStatus = (data[1] <<8) + data[0];
        std::bitset<16> wHubStatusBits(wHubStatus);
        ETG_TRACE_USR3 (("GetHubStatus: wHubStatusBits  = %s (Binary)", wHubStatusBits.to_string().c_str()));

        //=============================================================
        //wHubChange consists of data[2] and data[3] - Little endian, aus USB Spec 2.0 is used
        //=============================================================
        // e.g. data[0]   =  00000011    (Bit index 0-7) ('LocalPower Status has changed', 'Overcurrent Status has changed')
        //        data[1]   = 00000000    (Bit index 8-15)
        //wHubChange =  00000000 00000011

        tU16 wHubChange = (data[3] <<8) + data[2];
        std::bitset<16> wHubChangeBits(wHubChange);
        ETG_TRACE_USR3 (("GetHubStatus: wHubChangeBits = %s (Binary)", wHubChangeBits.to_string().c_str()));


        //-------------------------------
        // evaluate wHubStatusBits
        //-------------------------------
        //Bit 0:
        //           1 = Local power supply lost (inactive)
        //           0 = Local power supply good,
        //Bit 1:
        //           1 = A hub over-current condition exists.
        //           0 = No over-current condition currently exists.

        if(wHubStatusBits.test(0))
        {
            ETG_TRACE_USR4(("GetHubStatus: wHubStatusBits[0]: 1: Local power supply lost (inactive) "));
        }
        else
        {
            ETG_TRACE_USR4(("GetHubStatus: wHubStatusBits[0]: 0: Local power supply good) "));
        }

        if(wHubStatusBits.test(1))
        {
            ETG_TRACE_USR4(("GetHubStatus: wHubStatusBits[1]: 1: A hub over-current condition exists"));
        }
        else
        {
            ETG_TRACE_USR4(("GetHubStatus: wHubStatusBits[1]: 0: No over-current condition currently exists. "));
        }


        //-------------------------------
        // evaluate wHubChangeBits
        //-------------------------------

        //Bit 0:
        //           1 = Local Power Status has changed.
        //           0 = No change has occurred to Local Power Status.
        //Bit 1:
        //           1 = Over-Current Status has changed.
        //           0 = No change has occurred to the Over-Current Status.

        if(wHubStatusBits.test(0))
        {
            ETG_TRACE_USR4(("GetHubStatus: wHubChangeBits[0]: 1: Local Power Status has changed. "));
        }
        else
        {
            ETG_TRACE_USR4(("GetHubStatus: wHubChangeBits[0]: 0: No change has occurred to Local Power Status.  "));
        }

        if(wHubStatusBits.test(1))
        {
            ETG_TRACE_USR4(("GetHubStatus: wHubChangeBits[1]: 1: Over-Current Status has changed."));
        }
        else
        {
            ETG_TRACE_USR4(("GetHubStatus: wHubChangeBits[1]: 0: No change has occurred to the Over-Current Status. "));
        }




#if 0
        ETG_TRACE_USR3 (("GetPortStatus:  Got data"));
        //please now extract the bits from data fro wHubStatus and wHubChange and write these to bits_of_wHubStatus and bits_of_wHubChanged
        //from these outputs it is also possible to check if it is big or little endian
        for(int i=0;i<iResult;i++)
        {
            ETG_TRACE_USR3 (("GetHubStatus:bits_of_wHubStatus:  data[%d]=%x,",i,data[i]));
            std::bitset<8> bits(data[i]);
            //The first word of data contains HubStatus
            //refer the section 11.24.2.6
            if(2 == i)
            {
                //check for the second bit for the OC status
                if(bits.test(1))
                {
                    ETG_TRACE_USR3 (("!!!!!!!!!!GetHubStatus:Over current detected!!!!!!!!!!!!!"));
                    l_bHasOC = TRUE;
                }
                else
                {
                    ETG_TRACE_USR3 (("GetHubStatus:Over Current=FALSE"));
                }
            }
            ETG_TRACE_USR3 (("GetHubStatus:bits_of_wHubStatus:  data[%d](in binary form)=%s",i,bits.to_string().c_str()));
        }
#endif
    }
    else
    {
        if(iResult < 0)
        {
             ETG_TRACE_ERR (("[[ERROR] GetHubStatus:iResult=libusb_control_transfer(...) iResul:%d (%s)",(int)iResult,libusb_strerror((libusb_error)iResult)));
#ifdef USE_LIBUSBMTP_ERRMEM
             ETG_TRACE_ERRMEM (("[[ERROR] GetHubStatus:iResult=libusb_control_transfer(...) iResul:%d (%s)",(int)iResult,libusb_strerror((libusb_error)iResult)));
#endif //USE_LIBUSBMTP_ERRMEM
        }
        else
        {
             ETG_TRACE_USR3 (("[INFO] GetHubStatus:libusb_control_transfer failed Error code iResult:%d",iResult));
        }


    }
    ETG_TRACE_USR3 (("GetHubStatus: End"));

    return l_bHasOC;
}


//see 11.24.2.7 in //file://\\bosch.com\dfsrb\DfsDE\DIV\CM\DI\Projects\Common\CarNetworks\USB\Specs\USB_2.0\usb_20.pdf
bool GetPortStatus(libUsbHandle* f_poHandle,tenDevManagerUSBHost eUsb)//<------f_iPortNumber is not eUSB2 or eUSB3 a mapping function is necessary here
{


    ETG_TRACE_USR3 (("GetPortStatus: Begin"));

    int l_iPortNumber = -1;

    //->Note this part has tobe made configurable dependent
    if(eUsb == eUSB2)
    {
        l_iPortNumber = 1;
    }
    else if(eUsb == eUSB3)
    {
        l_iPortNumber = 2;
    }
    //-------------------------<-

    unsigned char data[4];
    tBool l_bHasOC = FALSE;
    if(l_iPortNumber > 0)
    {
        int iResult = libusb_control_transfer(f_poHandle,
                                              0xA3 /*GetHubStatus see  11.24.2: looking still for a define from libusb for this*/,
                                              LIBUSB_REQUEST_GET_STATUS,
                                              0,
                                              l_iPortNumber,
                                              data,
                                              4, //number of bytes returned  consists of 2 'words' i.e. wHubStatus then wHubChange
                                              1000 /*ms timout*/);


        ETG_TRACE_USR4(("GetPortStatus: iResult: %d",iResult));
        if(iResult == 4)
        {
            //taken from  'Universal Serial Bus Specification Revision 2.0' 11.24.2.6 Get Hub Status
            //expected 2 words: 1st: wHubStatus  2nd: wHubChange
            //=============================================================
            //wHubStatus consists of data[0] and data[1] - Little endian, aus USB Spec 2.0 is used
            //=============================================================
            // e.g. data[0]=  00000011    (Bit index 0-7)
            //        data[1] = 00000000    (Bit index 8-15)
            //wHubStatus =  00000000 00000011

            //wHubStatusBits : 00001100
            tU16 wPortStatus = (data[1] <<8) + data[0];
            std::bitset<16> wPortStatusBits(wPortStatus);
            ETG_TRACE_USR3 (("GetPortStatus: wPortStatusBits  = %s (Binary)", wPortStatusBits.to_string().c_str()));

            //=============================================================
            //wHubChange consists of data[2] and data[3] - Little endian, aus USB Spec 2.0 is used
            //=============================================================
            // e.g. data[0]   =  00000011    (Bit index 0-7)
            //        data[1]   = 00000000     (Bit index 8-15)
            //wHubChange =  00000000 00000011

            tU16 wPortChange = (data[3] <<8) + data[2];
            std::bitset<16> wPortChangeBits(wPortChange);
            ETG_TRACE_USR3 (("GetPortStatus: wPortChangeBits = %s (Binary)", wPortChangeBits.to_string().c_str()));

            //==========================================================
            // evaluate bits  of wPortStatus
            //==========================================================

            //-------------------------------
            // evaluate wPortStatusBits Bit 0
            //-------------------------------
            //Bit 0: Current Connect Status: (PORT_CONNECTION) This field reflects whether or not a device is currently connected to this port.
            //         0 = No device is present.
            //         1 = A device is present on this port.
            unsigned int i = 0;
            if(false == wPortStatusBits.test(i))
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 0: No device is present ",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 1: A device is present on this port",i));
            }

            //-------------------------------
            // evaluate wPortStatusBits Bit 1
            //-------------------------------
            //Bit 1: Port Enabled/Disabled: (PORT_ENABLE) Ports can be enabled by the USB System Software only. Ports
            //         can be disabled by either a fault condition (disconnect event or other fault condition) or by the USB System Software.
            //         0 = Port is disabled.
            //         1 = Port is enabled.
            i++;
            if(false == wPortStatusBits.test(i))
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 0: Port is disabled",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 1: Port is enabled",i));
            }

            //-------------------------------
            // evaluate wPortStatusBits Bit 2
            //-------------------------------
            //Bit 2: Suspend: (PORT_SUSPEND) This field indicates whether or not the device on this port is suspended.
            //         Setting this field causes the device to suspend by not propagating bus traffic downstream. This field may be
            //         reset by a request or by resume signaling from the device attached to the port.
            //         0 = Not suspended.
            //         1 = Suspended or resuming.
            i++;
            if(false == wPortStatusBits.test(i))
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 0: Not suspended",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 1: Suspended or resuming.",i));
            }

            //-------------------------------
            // evaluate wPortStatusBits Bit 3
            //-------------------------------
            //Bit 3: Over-current: (PORT_OVER_CURRENT) If the hub reports over-current conditions on a per-port basis, this field will indicate that the current drain on the
            //          port exceeds the specified maximum. For more details, see Section 7.2.1.2.1.
            //         0 = All no over-current condition exists on this port.
            //         1 = An over-current condition exists on this port.
            i++;
            if(false == wPortStatusBits.test(i))
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 0: All no over-current condition exists on this port. ",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 1: An over-current condition exists on this port.",i));
            }

            //-------------------------------
            // evaluate wPortStatusBits Bit 4
            //-------------------------------
            //Bit 4: Reset: (PORT_RESET) This field is set when the host wishes to reset the attached device. It remains set
            //         until the reset signaling is turned off by the hub.
            //         0 = Reset signaling not asserted.
            //         1 = Reset signaling asserted.
            i++;
            if(false == wPortStatusBits.test(i))
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 0: Reset signaling not asserted",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 1: Reset signaling asserted",i));
            }

            //Bit 5-7 are reserved these bits are 0

            //-------------------------------
            // evaluate wPortStatusBits Bit 8
            //-------------------------------
            //Bit 8: Port Power: (PORT_POWER) This field reflects a port�s logical, power control state. Because hubs can
            // implement different methods of port power switching, this field may or may not represent whether power is
            // applied to the port. The device descriptor reports the type of power switching implemented by the hub.
            //         0 = This port is in the Powered-off state.
            //         1 = This port is not in the Powered-off state.

            i = 8;
            if(false == wPortStatusBits.test(i))
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 0: This port is in the Powered-off state.",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 1: This port is not in the Powered-off state.",i));
            }

            //-------------------------------
            // evaluate wPortStatusBits Bit 9
            //-------------------------------
            //Bit 9: Low- Speed Device Attached: (PORT_LOW_SPEED) This is relevant only if a device is attached.
            //         0 = Full-speed or High-speed device attached to this port (determined by bit 10).
            //         1 = Low-speed device attached to this port.
            i++;
            if(false == wPortStatusBits.test(i))
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 0: Full-speed or High-speed device attached to this port (determined by bit 10).",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 1: Low-speed device attached to this port.",i));
            }

            //-------------------------------
            // evaluate wPortStatusBits Bit 10
            //-------------------------------
            //Bit 10: High-speed Device Attached: (PORT_HIGH_SPEED) This is relevant only if a device is attached.
            //         0 = Full-speed device attached to this port.
            //         1 = High-speed device attached to this port.
            i++;
            if(false == wPortStatusBits.test(i))
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 0: Full-speed device attached to this port.",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 1: High-speed device attached to this port.",i));
            }

            //-------------------------------
            // evaluate wPortStatusBits Bit 11
            //-------------------------------
            //Bit 11: Port Test Mode : (PORT_TEST) This field reflects the status of the port's test mode. Software uses the
            //          SetPortFeature() and ClearPortFeature() requests to manipulate the port test mode.
            //         0 = This port is not in the Port Test Mode.
            //         1 = This port is in Port Test Mode.
            i++;
            if(false == wPortStatusBits.test(i))
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 0: This port is not in the Port Test Mode.",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 1: This port is in Port Test Mode.",i));
            }


            //-------------------------------
            // evaluate wPortStatusBits Bit 12
            //-------------------------------
            //Bit 12: Port Indicator Control: (PORT_INDICATOR) This field is set to reflect software control of the port indicator.
            //           For more details see Sections 11.5.3, 11.24.2.7.1.10, and 11.24.2.13.
            //         0 = Port indicator displays default colors.
            //         1 = Port indicator displays software controlled color.
            i++;
            if(false == wPortStatusBits.test(i))
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 0: Port indicator displays default colors.",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetPortStatus: wPortStatusBits[%2d]: 1: Port indicator displays software controlled color.",i));
            }

            //Bit 13-15 are reserved and are 0

            //==========================================================
            // evaluate bits  of wPortChangeBits
            //==========================================================

            //-------------------------------
            // evaluate wPortChangeBits Bit 0
            //-------------------------------
            //Bit 0: Connect Status Change: (C_PORT_CONNECTION) Indicates a change has occurred in the port�s Current
            //          Connect Status. The hub device sets this field as described in Section 11.24.2.7.2.1.
            //         0 = No change has occurred to Current Connect status.
            //         1 = Current Connect status has changed.
            i = 0;
            if(false == wPortChangeBits.test(i))
            {
                ETG_TRACE_USR4(("GetHubStatus: wPortChangeBits[%2d]: 0: No change has occurred to Current Connect status.",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetHubStatus: wPortChangeBits[%2d]: 1: Current Connect status has changed.",i));
            }

            //-------------------------------
            // evaluate wPortChangeBits Bit 1
            //-------------------------------
            //Bit 1: Port Enable/Disable Change: (C_PORT_ENABLE) This field is set to one when a port is disabled because
            //          of a Port_Error condition (see Section 11.8.1).
            //         0 =
            //         1 = port disabled
            i++;
            if(false == wPortChangeBits.test(i))
            {
                ETG_TRACE_USR4(("GetHubStatus: wPortChangeBits[%2d]: 0: ok",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetHubStatus: wPortChangeBits[%2d]: 1: port disabled",i));
            }

            //-------------------------------
            // evaluate wPortChangeBits Bit 2
            //-------------------------------
            //Bit 2: Suspend Change: (C_PORT_SUSPEND) This field indicates a change in the host-visible suspend state of
            // the attached device. It indicates the device has transitioned out of the Suspend state. This field is set only
            // when the entire resume process has completed. That is, the hub has ceased signaling resume on this port.
            //         0 = No change.
            //         1 = Resume complete.
            i++;
            if(false == wPortChangeBits.test(i))
            {
                ETG_TRACE_USR4(("GetHubStatus: wPortChangeBits[%2d]: 0: No change.",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetHubStatus: wPortChangeBits[%2d]: 1: Resume complete.",i));
            }

            //-------------------------------
            // evaluate wPortChangeBits Bit 3
            //-------------------------------
            //Bit 3: Over-Current Indicator Change: (C_PORT_OVER_CURRENT) This field applies only to hubs that report
            //over-current conditions on a per-port basis (as reported in the hub descriptor).
            //         0 = No change has occurred to Over-Current Indicator
            //         1 = Over-Current Indicator has changed.
            i++;
            if(false == wPortChangeBits.test(i))
            {
                ETG_TRACE_USR4(("GetHubStatus: wPortChangeBits[%2d]: 0: No change has occurred to Over-Current Indicator",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetHubStatus: wPortChangeBits[%2d]: 1: Over-Current Indicator has changed.",i));
            }

            //-------------------------------
            // evaluate wPortChangeBits Bit 4
            //-------------------------------
            //Bit 4: Reset Change: (C_PORT_RESET) This field is set when reset processing on this port is complete.
            //         0 = No change.
            //         1 = Reset complete.
            i++;
            if(false == wPortChangeBits.test(i))
            {
                ETG_TRACE_USR4(("GetHubStatus: wPortChangeBits[%2d]: 0: No change.",i));
            }
            else
            {
                ETG_TRACE_USR4(("GetHubStatus: wPortChangeBits[%2d]: 1: Reset complete.",i));
            }

            //bit 5-15 are reserved and set to 0

#if 0
            ETG_TRACE_USR3 (("GetPortStatus:  Got data"));
            for(int i=0;i<iResult;i++)
            {
                ETG_TRACE_USR3 (("GetPortStatus:  data[%d]=%x,",i,data[i]));
                std::bitset<8> bits(data[i]);
                //The first word of data contains HubStatus
                //refer the section 11.24.2.7.1
                if(2 == i)
                {
                    //check for the third bit for the OC status
                    if(bits.test(2))
                    {
                        ETG_TRACE_USR3 (("!!!!!!!!!!GetPortStatus:Over current detected for port %d!!!!!!!!!!!!!",f_iPortNumber));
                        l_bHasOC = TRUE;
                    }
                    else
                    {
                        ETG_TRACE_USR3 (("GetPortStatus:Over Current=FALSE for the Port number %d",f_iPortNumber));
                    }
                }

                ETG_TRACE_USR3 (("GetPortStatus:  data[%d](in binary form)=%s",i,bits.to_string().c_str()));
            }
#endif

        }
        else
        {
            if(iResult < 0)
            {
                 ETG_TRACE_ERR (("[[ERROR] GetPortStatus:iResult=libusb_control_transfer(...) iResul:%d (%s)",(int)iResult,libusb_strerror((libusb_error)iResult)));
#ifdef USE_LIBUSBMTP_ERRMEM
                 ETG_TRACE_ERRMEM (("[[ERROR] GetPortStatus:iResult=libusb_control_transfer(...) iResul:%d (%s)",(int)iResult,libusb_strerror((libusb_error)iResult)));
#endif
            }
            else
            {
                ETG_TRACE_USR3 (("GetPortStatus:libusb_control_transfer failed Error code iResult:%d",iResult));
            }

        }
    }
    else
    {
        ETG_TRACE_USR3 (("GetPortStatus:invalid port number!!!!!!!!!  got the port number %d",l_iPortNumber));
    }

    ETG_TRACE_USR3 (("GetPortStatus: End"));
    return l_bHasOC;

}




#endif //USE_EXPERIMENTAL_PSA_HUBCOMMUNICATION


////////////////////////////////////////////////////////////////////////////////
// <EOF>
