/*-----------------------------------------------------------------------------*
 * MessageQueueDVM.cpp                                                         *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : G3G                                                           *
 * COPYRIGHT   : (c) 2012 - 2020 Robert Bosch GmbH, Hildesheim                 *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file MessageQueueDVM.cpp
 *
 * \brief Adapter class (Wrapper) to ease replacing message queue  functions it uses gmp frame work message queue
 *
 * \version 29.01.2015, Rajeev Narayanan Sambhu (RBEI/ECS1), version 1.0 - first implementation
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2020
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/


#include "Config.h"

#include "MessageQueueDVM.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_UDEVMANAGER
#include "trcGenProj/Header/MessageQueueDVM.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

#include <string.h>
#include <stdio.h>


#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"
#include "Enums.h"
using namespace statetbl;


/*-----------------------------------------------------------------------------*
 * define                                                                 *
 *-----------------------------------------------------------------------------*/



//==========================public functions of class===========================

/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/

MessageQueueDVM::MessageQueueDVM()
{
    m_pMsgQueue    = NULL;
    m_pMsgQName    = NULL;
    m_CountMsges   = 0;
    m_MaxMessages  = 0;
    m_MaxLength    = 0;
    m_AccessType   = 0;
}

MessageQueueDVM::~MessageQueueDVM()
{
    iDestroy();
}

/*-----------------------------------------------------------------------------*
 * iInit(...)                                                                  *
 *-----------------------------------------------------------------------------*/
//Maintenance: Function is not necessary is OSAL will have been replaced - cleanup later from code
int MessageQueueDVM::iInit(const char* pMsgQName, int u32MaxMessages, int u32MaxLength,int enAccess)
{
    int iRet = MessageQueueDVM::ERROR;
    m_pMsgQName   = pMsgQName;
    m_MaxMessages = u32MaxMessages;
    m_MaxLength   = u32MaxLength;
    m_AccessType  = enAccess;

    if(NULL == m_pMsgQueue)
    {
        if(NULL !=m_pMsgQName)
        {
            m_pMsgQueue = new MessageQueue(m_pMsgQName);
            if(m_pMsgQueue)
            {
                ETG_TRACE_USR4(("iInit:OK"));
                iRet = MessageQueueDVM::OK;
            }
            else
            {
                ETG_TRACE_FATAL(("iInit:NOK"));
            }
        }
        else
        {
            ETG_TRACE_FATAL(("iInit:NOK - invalid Name given"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("iInit:NOK - already exists"));
    }
    //printf("GMP Message Queue %s\n",pMsgQName);

    return iRet;
}

/*-----------------------------------------------------------------------------*
 * iDestroy(...)                                                     *
 *-----------------------------------------------------------------------------*/

int MessageQueueDVM::iDestroy()
{
    int iRet = MessageQueueDVM::ERROR;
    if(m_pMsgQueue)
    {
#ifdef WORKAROUND_EMPTYMESSAG
       unsigned int uNumOfMessage = (unsigned int)m_pMsgQueue->GetCurMessagesCount();
       unsigned int uNumbytesReceived;
       char receivemsg[100];
       for(unsigned int i=0;i<uNumOfMessage;i++)
       {
           iMessageQueueWait((void*)receivemsg , uNumbytesReceived,0,10);
       }
#endif //WORKAROUND_EMPTYMESSAG
        
        m_pMsgQueue->Flush();
        if(NULL !=m_pMsgQName)
        {
            m_pMsgQName = NULL;
            delete m_pMsgQueue;
            m_pMsgQueue = NULL; //check fro try here
            ETG_TRACE_USR4(("iDestroy:OK"));
            iRet = MessageQueueDVM::OK;
        }
        else
        {
            ETG_TRACE_FATAL(("iDestroy:NOK: NULL ==m_pMsgQName"));
        }

    }
    else
    {
        ETG_TRACE_FATAL(("iDestroy:NOK not yet created!"));
    }

    return iRet;
}


/*-----------------------------------------------------------------------------*
 * iMessageQueuePost(...)                                                      *
 *-----------------------------------------------------------------------------*/
int MessageQueueDVM::iMessageQueuePost(const void* pMsg, int iLength, int iPrio)

{
    ETG_TRACE_USR4(("Entering iMessageQueuePost"));
    int iRet = MessageQueueDVM::ERROR;

    if(m_pMsgQueue)
    {
        if((m_CountMsges<=m_MaxMessages) &&(iLength<=m_MaxLength))
        {
            //Bugfix: Without this  SegmentationFault happens using framework_so
            void* Sendbuf = m_pMsgQueue->CreateMessageBuffer(iLength);
            memcpy(Sendbuf,pMsg,iLength);
            if(-1 != m_pMsgQueue->Push(Sendbuf, iLength, iPrio))
            {
                m_CountMsges++;
                ETG_TRACE_USR4(("iMessageQueuePost:OK (m_CountMsges:%d)",m_CountMsges));
                iRet = MessageQueueDVM::OK;
            }
            else
            {
                ETG_TRACE_FATAL(("iMessageQueuePost:NOK"));
            }
        }
        else
        {
            ETG_TRACE_FATAL(("iMessageQueuePost:NOK Parameters"));
            ETG_TRACE_FATAL(("iMessageQueuePost:m_CountMsges:%d , m_MaxMessages:%d",m_CountMsges,m_MaxMessages));
            ETG_TRACE_FATAL(("iMessageQueuePost:u32Length:%d , m_MaxLength:%d",iLength,m_MaxLength));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("iMessageQueuePost:NOK not yet created!"));
    }

    ETG_TRACE_USR4(("Leaving iMessageQueuePost"));
    return iRet;
}
/*-----------------------------------------------------------------------------*
 * iMessageQueueWait(...)                                                      *
 *-----------------------------------------------------------------------------*/

int MessageQueueDVM::iMessageQueueWait(void* pu8Buffer, unsigned int &uLength,int pu32Prio, unsigned int time_out_in_ms)

{
    ETG_TRACE_USR4(("Entering iMessageQueueWait"));
    int iRet = MessageQueueDVM::ERROR;
    uLength = 0;
    (void)pu32Prio;
    if(m_pMsgQueue)
    {
        if(NULL != pu8Buffer)
        {
            size_t msgSize;
            void*  ReceiveBuf = m_pMsgQueue->WaitForMessage(&msgSize,time_out_in_ms);
            if(ReceiveBuf)
            {
                if(msgSize <= m_MaxLength)
                {
                    //copy to output buffer and drop ReceiveBuf
                    uLength = msgSize;
                    memcpy(pu8Buffer,(const void*)ReceiveBuf,msgSize);
                    m_pMsgQueue->DropMessage(ReceiveBuf);
                
                }
            }

            if(uLength > 0)
            {
                //printf("Message Queue Receive Successful. Received Data %s\n",pu8Buffer);

                ETG_TRACE_USR4(("iMessageQueueWait:Received OK (u32Length:%d)",uLength));
                ETG_TRACE_USR4(("iMessageQueueWait:Received OK (m_CountMsges:%d)",m_CountMsges));
                m_CountMsges--;
                iRet = MessageQueueDVM::OK;
            }
            else
            {
                ETG_TRACE_USR4(("[INFO]iMessageQueueWait:NOK: u32Length:%d (time_out_in_ms:%d)",uLength,time_out_in_ms));
                uLength = UTIL_LENOFQMSG;
                util_StructMsg* l_pMsgstruct = (util_StructMsg*)pu8Buffer;
                l_pMsgstruct->eMsgType = statetbl::enTimout;

            }
        }
        else
        {
            ETG_TRACE_FATAL(("iMessageQueueWait:NOK: pu8Buffer is NULL"));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("iMessageQueueWait:NOK not yet created!"));
    }

    ETG_TRACE_USR4(("Leaving iMessageQueueWait"));
    return iRet;
}


unsigned int MessageQueueDVM::iGetCurMessagesCount()
{
      unsigned int uRes=0;
      if(m_pMsgQueue)
      {
          uRes = (unsigned int)m_pMsgQueue->GetCurMessagesCount();
      }
      return uRes;
}

