
/*-----------------------------------------------------------------------------*
 * GenString.cpp                                                                  *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM Gen3                                                   *
 * COPYRIGHT   : (c) 2013 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/

/*!
 * \file GenString.cpp
 *
 * helper class to replace dependency from QStrings
 *
 * \version Initial Version
 * \version 23.10.2013, Koechling, Christian (Bosch)
 *
 *-----------------------------------------------------------------
 *                                  development for Gen3:
 *-----------------------------------------------------------------
 *\version 23.10.2013, Christian Koechling (Bosch)
 *        -# start tp replace QStrings by replacing QString by define GENSTRING
 *
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include "Enums.h"
#include "GenString.h"

#include <sstream> //see template below
#include <ctype.h> //for whitespace check
#include <algorithm>    // std::transform
#include <math.h>    //pow
#include <locale>

 /*-----------------------------------------------------------------------------*
  * ETG Tracing                                                                 *
  *-----------------------------------------------------------------------------*/

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_UDEVMANAGER
#include "trcGenProj/Header/GenString.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * templates                                                               *
 *-----------------------------------------------------------------------------*/
/*e.g. used to convert int ot string since it is a function it could be placed in cpp file*/

template <typename T> string toStr(T tmp)
{
    ostringstream out;
    out << tmp;
    return out.str();
}

/* did not work yet
template <typename T> T strTo(string tmp)
{
    T output;
    istringstream in(tmp);
    in >> output;
    return output;
}
*/


static unsigned short m_utf16[100];
static char m_data[100]; //intermediate workaround until function will have been fully defined


/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
//NOTE: SINCE THE CODE IS UNDERCONSTRUCTION a lot of elints have been added to keep overall code intermediately clean

GenString::GenString()
{
    m_String.clear(); //Erases the contents of the string, which becomes an empty string (with a length of 0 characters).
    memset(m_utf16,0,sizeof(m_utf16));
    memset(m_data,0,sizeof(m_data));
}
GenString::GenString(const char* pString)//lint !e830
{
    if(pString)
    {
        m_String = pString;
    }
    else
    {
         ETG_TRACE_SYS(("[WARNING]Constructor GenString(const char* pString) pString==NULL"));
         //DVM_NORMAL_M_ASSERT_ALWAYS(); //to check where this happens
         m_String.clear();
    }

    memset(m_utf16,0,sizeof(m_utf16));
    memset(m_data,0,sizeof(m_data));
}

//Example: expected to be used this way: GenString('d')
GenString::GenString(char cChar)//lint !e830
{
    ETG_TRACE_USR4(("GenString() cChar=%d",cChar));
    //printf("GenString(char cChar)\n");
    m_String = cChar;
    memset(m_utf16,0,sizeof(m_utf16));
    memset(m_data,0,sizeof(m_data));
    ETG_TRACE_USR4(("GenString() m_String=%s",m_String.c_str()));
}

GenString::GenString(int iNumber)//lint !e830
{
    ETG_TRACE_USR4(("GenString() iNumber=%d",iNumber));

    //m_String = std::to_string(iNumber); //C++11
    m_String = toStr(iNumber);         //use above template instead
    memset(m_utf16,0,sizeof(m_utf16));
    memset(m_data,0,sizeof(m_data));

    ETG_TRACE_USR4(("GenString() m_String=%s",m_String.c_str()));

/* with label 142 I had fails thus I went to the previous version
    char ichar = iNumber;
    m_String+= ichar;
    memset(m_utf16,0,sizeof(m_utf16));
    memset(m_data,0,sizeof(m_data));
*/
}//lint !e715, //lint !e72



GenString::GenString(std::string strString)
{
    m_String = strString;
    memset(m_utf16,0,sizeof(m_utf16));
    memset(m_data,0,sizeof(m_data));
}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
GenString::~GenString(){}

/*-----------------------------------------------------------------------------*
 * overloaded operators                                                                  *
 *-----------------------------------------------------------------------------*/
GenString& GenString::operator=(const GenString& oGenStringToSetThisOne)//lint !e830
{
    if( &oGenStringToSetThisOne == this || oGenStringToSetThisOne == *this)
    {
        ETG_TRACE_USR4(("GenString operator = fed with this pointer"));
    }
    else
    {
        m_String = oGenStringToSetThisOne.toStdString();
    }

   return *this; //this is pointer *this is the object i.e. used as reference
}
GenString& GenString::operator=(const char* charToSetThisOne) //lint !e830
{
    if(charToSetThisOne)
    {
        m_String = charToSetThisOne;
    }
    else
    {
        m_String = "";
        ETG_TRACE_ERR(( "GenString:(= operator) ERROR  Trying to copy NULL pointer to Genstring.. Storing empty string to avoid fault" ));
    }
    return *this;
}

GenString& GenString::operator=(const std::string& stdStringToSetThisOne)
{
    m_String = stdStringToSetThisOne;
    return *this;
}

GenString  GenString::operator+(const GenString& toAdd)
{
    GenString tempGenString(m_String + toAdd.toStdString());
    return tempGenString;
}

GenString&  GenString::operator+=(const GenString& toAdd)
{
    m_String.append(toAdd.toStdString());
    return *this; //tmp to make it compilable
}

bool  GenString::operator==(GenString const& arg2toCompareWith1) const//lint !e830
{
    bool bRet = false;
    int iRes = 1;
    if(!(arg2toCompareWith1.isEmpty()))
    {
        iRes = m_String.compare(arg2toCompareWith1.toStdString());
    }
    else
    {
        ETG_TRACE_ERR(( "GenString: (== operator) ERROR  Trying to compare NULL to Genstring.. Returning false" ));
    }

    if(iRes == 0) //is equal
    {
        bRet = true;
    }
    return bRet;
}

bool  GenString::operator==(const char* arg2toCompareWith1) const//lint !e830
{
    bool bRet = false;
    int iRes = 1;
    if(arg2toCompareWith1)
    {
        iRes = m_String.compare(arg2toCompareWith1);
    }
    else
    {
        ETG_TRACE_ERR(( "GenString: (== operator) ERROR  Trying to compare NULL pointer to Genstring.. Returning false" ));
    }

    if(iRes == 0) //is equal
    {
        bRet = true;
    }
    return bRet;
}



bool  GenString::operator!=(GenString const& arg2toCompareWith1) const //lint !e830
{
    bool bRet = true;
    int iRes = m_String.compare(arg2toCompareWith1.toStdString());

    if(iRes == 0) //is equal
    {
        bRet = false;
    }
    return bRet;
}


/*-----------------------------------------------------------------------------*
 * get a new GenString due to manipulation of an existing one                                                                  *
 *-----------------------------------------------------------------------------*/
GenString GenString::trimmed() const
{
    std::string l_String = m_String;

    //-----------------------------
    //erase spaces at the beginning
    //-----------------------------
    unsigned int uiElemsToEraseFromBegin = 0;
    for(std::string::iterator it=l_String.begin(); it!=l_String.end(); ++it)
    {
        if(isspace(*it)) //e.g. ' '-space  '\t'-Tab  '\n'-newline '\v-vertical tab' '\f'-carriage return
        {
            uiElemsToEraseFromBegin++;
        }
        else
        {
            break; //found none space character i.e. all spaces at the beginning have been found
        }

    }
    if(uiElemsToEraseFromBegin>0)
    {
        l_String.erase(0,uiElemsToEraseFromBegin);
    }

    //-----------------------------
    //erase spaces at the endof string
    //-----------------------------
    unsigned int uiElemsToEraseFromEnd = 0;
    for(std::string::reverse_iterator rit= l_String.rbegin(); rit!=l_String.rend(); ++rit)
    {
        if(isspace(*rit))
        {
            uiElemsToEraseFromEnd++;
            //l_String.erase(rit); //e.g. ' '-space  '\t'-Tab  '\n'-newline '\v-vertical tab' '\f'-carriage return   //does not work erase works with iterator only c++11 has a solution
        }
        else
        {
            break; //found none space character i.e. all spaces at the beginning have been found
        }

    }
    if(uiElemsToEraseFromEnd>0)
    {
        unsigned int uiPos = l_String.length() - uiElemsToEraseFromEnd;   //"12345   " 3 spaces at end length is 8 uiPos = 8-3 = 5 uiPos starts from 0 i.e. at 5 is the first space to be deleted
        l_String.erase(uiPos,string::npos /*until end of string*/);
    }

    GenString strTrimmed(l_String);
    return strTrimmed;
}//lint !e1762

GenString GenString::toLower() const
{
  std::locale loc;
  std::string strToLower = m_String; //now both have the same size

  for (std::string::size_type i=0; i<m_String.length(); ++i)
  {
      strToLower[i] = std::tolower(m_String[i],loc);
  }

/* alternative:
  std::transform(m_String.begin(),m_String.end(),std::back_inserter(strToLower), ::tolower);
*/

  GenString oGenStringToLower(strToLower);
  return oGenStringToLower;
}

GenString GenString::left(const int num) const
{
    std::string strLeft;

    //nok
    if(num < 0)
    {
        strLeft = m_String;
    }
    else
    {
        //nok
        if( ((unsigned int)num > m_String.length()))
        {
            strLeft = m_String;
        }
        //ok
        else
        {
            strLeft = m_String.substr(0,(unsigned int)num);
        }
    }

    GenString oGenString(strLeft);
    return oGenString;
}
GenString GenString::right(const int num) const
{
    std::string strRight;

    //nok
    if(num < 0)
    {
        strRight = m_String;
    }
    else
    {
        //nok
        if( ((unsigned int)num > m_String.length()))
        {
            strRight = m_String;
        }
        //ok
        else
        {
            strRight = m_String.substr((m_String.length()-(unsigned int)num), string::npos);
        }
    }
    GenString oGenString(strRight);
    return oGenString;
}
GenString GenString::mid ( int position, int num) const   //unit test this function
{
    std::string strMid;

    if( (position >= 0) && (num >=0))
    {
        if((unsigned int)position > m_String.length())
        {
            strMid ==""; //emtpy String
        }
        else
        {
            if( (unsigned int)num > m_String.length())
            {
                strMid = m_String.substr((unsigned int)position, string::npos);
            }
            else
            {
                strMid = m_String.substr((unsigned int)position, (unsigned int)num);
            }

        }
    }
    else
    {
        ETG_TRACE_USR2(("mid: parameter <0  position=%d, num=%d",position,num));

        if( (num < 0) && (position >= 0))
        {
            strMid = m_String.substr((unsigned int)position, string::npos);
        }
        else if(position < 0)
        {
            strMid ==""; //emtpy String
        }
        else
        {
            strMid ==""; //emtpy String   //check these cases!!
        }
    }


    GenString oGenString(strMid);
    return oGenString;
}

/*-----------------------------------------------------------------------------*
 * convert GenString into other format
 *-----------------------------------------------------------------------------*/
std::string GenString::toStdString() const
{
        return m_String;
}

int GenString::toInt() const
{
    int iRet;


    //iRet = strTo(m_String);         //use above template instead //need more error robustness here!!

    iRet = atoi(m_String.c_str());
    return iRet;
}//lint !e1762



tU16 GenString::Hex2UShort() const  //replaces toUshort
{
    tU16 u16Result = 0;
    string::iterator it;
    tU16 val,n;
    char c;


    //check if all letters are 0-F
    //'0'.....'9' i.e. 048....057
    //'A'.....'F' i.e. 065....070
    //'a'....'f'  i.e. 097...102
    for(int i=0; i<static_cast<int>(m_String.length()); ++i)
    {
        //printf("Hex2UShort: %d\n",i);
        c = m_String[i];
        n = (m_String.length()-1) -i;
        //lint !e747
        tU16 temp = static_cast<tU16>(pow(16,n));
        if((c >= '0') && (c <= '9'))
        {
            val = c-48;
            //printf("Hex2UShort: val=%d n:%d\n",val,n);
            u16Result+=  (val * temp);
            //printf("Hex2UShort: u16Result=%d\n",u16Result);
        }
        else if((c >= 'a') && (c <= 'f') )
        {
            val = (c-97) + 10;
            u16Result+= (val * temp);
        }
        else if((c >= 'A') && (c <= 'F'))
        {
            val = (c-65) + 10;
            u16Result+= (val * temp);
        }
        else
        {
            u16Result = 0;
            ETG_TRACE_FATAL(("Hex2UShort:String not convertable"));
            break;
        }
    }

    return u16Result;
}

/*-----------------------------------------------------------------------------*
 * manipulate existing string                                                                  *
 *-----------------------------------------------------------------------------*/

GenString& GenString::append(const GenString& oGenStringToappend) //add unit
{
    m_String = m_String + oGenStringToappend.toStdString();
    return *this;
}

//warning has to be tested and error condiction are missing
/*
Removes n characters from the string, starting at the given position index, and returns a reference to the string.

If the specified position index is within the string, but position + n is beyond the end of the string, the string is truncated at the specified position.

//it is thought to remove ascii of value <128 not utf8 values - for details please refer to code which uses this
*/
GenString & GenString::remove ( int position, int n )
{
    if(position >=0)
    {
        if(n>0)
        {
            if((unsigned int)n> m_String.size())
            {
                //ok
                m_String = m_String.erase((unsigned int)position,string::npos);
            }
            else
            {
                if((unsigned int)position<=m_String.size())
                {
                    //ok
                    m_String = m_String.erase((unsigned int)position,(unsigned int)n);
                }
                else
                {
                    ETG_TRACE_FATAL(("remove: not precisely defined in interface"));
                }
            }
        }
        else
        {
            ETG_TRACE_FATAL(("remove: n is negative: n=%d",n));
        }
    }
    else
    {
        ETG_TRACE_FATAL(("remove: position is negative: position=%d",position));
    }

    return *this;
}

//!!!!!!!!!!!!!!!!!!!TBD!!!!!!!!!!!
GenString& GenString::remove(const GenString& oGenStringToRemove)//okl: used with ascii 0...127 only
{
    (void)oGenStringToRemove;
    return *this;
}

/*-----------------------------------------------------------------------------*
 *  get info about the string
 *-----------------------------------------------------------------------------*/
int GenString::size() const
{
    int iRet = (int)m_String.length();
    return iRet;
}//lint !e1762

int GenString::length() const
{
    int iRet = (int)m_String.length();
    return iRet;
}
bool GenString::isEmpty() const
{
    bool bRet = m_String.empty();
     return bRet;
}

bool GenString::contains(const GenString& oGenStringToSearch,int iMode) const//lint !e830
{
    bool bRet = false;

    if((false == m_String.empty()) && (false == oGenStringToSearch.toStdString().empty()))
    {
        if(iMode == (int)CASE_SENSITIVE)
        {
            bRet = this->contains(oGenStringToSearch);
        }
        else if(iMode == (int)CASE_INSENSITIVE)
        {
            //make them all lower to have basis to compare
            GenString oStrLowered  = this->toLower();
            GenString oStrToSearch = oGenStringToSearch.toLower();

            bRet = oStrLowered.contains(oStrToSearch);
        }
    }

    return bRet;
}

bool GenString::contains(const GenString& oGenStringToSearch) const //lint !e830
{
    bool bRet = false;
    if(!(oGenStringToSearch.toStdString().empty()))
    {
        std::size_t found = m_String.find(oGenStringToSearch.toStdString());
        if(found!=std::string::npos)
        {
            bRet = true;
        }
    }
    return bRet;
}

int GenString::compare(const GenString& oGenStringToSearch) const //lint !e830
{
    int iRet = 1;
    if(*this == oGenStringToSearch)
    {
        iRet = 0;
    }
    return iRet;
}

int GenString::indexOf(const char* charToSearch) const
{
    int iRet = -1;

    std::size_t found;
    std::string strToSearch = charToSearch;

    if(!(m_String.empty()))
    {
        found = m_String.find(strToSearch);
        if(found!=std::string::npos)
        {
            iRet = (int)found;
        }
    }
    return iRet;
}

int GenString::indexOf(const char* charToSearch,int from) const
{
    int iRet = -1;

    std::size_t found;
    std::string strToSearch = charToSearch;

    if(!(m_String.empty()))
    {
        if(from>=0)
        {
            if((unsigned int)from <= m_String.size())
            {
                found = (std::size_t)m_String.find(strToSearch,(unsigned int)from);
                if(found!=std::string::npos)
                {
                    //ok
                    iRet = (int)found;
                }
            }
        }
        else
        {
            ETG_TRACE_FATAL(("indexOf: from <0 : %d",from));
        }
    }
    return iRet;
}

int GenString::lastIndexOf ( const GenString & oGenStringToFind, int from) const
{

    int iRet = -1;
    (void)from;
    std::size_t found = m_String.rfind(oGenStringToFind.toStdString());
    if (found!= std::string::npos)
    {
        //ok
        iRet = (int)found;
        //ETG_TRACE_FATAL(("lastIndexOf: from <0 : %d",iRet));
    }
    return iRet;
}


GenString& GenString::replace(const GenString& oBefore,const GenString& oAfter)
{
    //search before

    int iRet;
    int iCount= 0;
    int iFromPosition = 0;

    //printf("m_String:%s\n",m_String.c_str());
    //printf("oBefore.length():%d\n",oBefore.length());

    while(iFromPosition < (int)m_String.length())
    {
        iRet = indexOf(oBefore.toStdString().c_str(),iFromPosition);
        //printf("iRet:%d\n",iRet);
        //ok - found
        if(iRet >=0)
        {
            //replace
            m_String.replace(iRet,oAfter.length(),oAfter.toStdString());
            //printf("m_String:%s\n",m_String.c_str());
            iCount++;
            //new from position
            iFromPosition = iRet;
            //printf("iFromPosition:%d\n",iFromPosition);
        }
        //not found
        else
        {
            ETG_TRACE_USR4(("replace: finished, replaced %d times\n",iCount));
            break;
        }

    }
    return *this;
}

#ifdef USE_VALIDATE_AND_CONVERT2REGION_LANGUAGE
/*
int GenString::toAsciiSize() const
{
    return 10;
}
char GenString::toAsciiAt(unsigned int iIndex) const
{
    (void)iIndex;
    return 100;
}
GenString GenString::normalized ( int iNormalizationForm ) const
{
    (void)iNormalizationForm;
    GenString strNormalized;
    return strNormalized;
}

GenString GenString::fromUtf8 ( const char * str) const
{
    (void)str;
    GenString strfromUtf8;
    return strfromUtf8;
}

GenString& GenString::replace(unsigned int uiPos,unsigned int uiLen, char cReplacer)
{
    (void)uiPos;
    (void)uiLen;
    (void)cReplacer;
    return *this;
}
const char* GenString::toUtf8constData() const
{

    return m_data;
}
const char* GenString::toAscii() const
{

    return m_data;
}
*/
#endif



/*-----------------------------------------------------------------------------*
 * protected functions
 *-----------------------------------------------------------------------------*/
bool GenString::bTestProtected(IN int iVar) const
{
    bool bRet = false;
    if(iVar>=0)
    {
        bRet = true;
    }
    return bRet;
}









////////////////////////////////////////////////////////////////////////////////
// <EOF>

