/*-----------------------------------------------------------------------------*
 * Enums.h                                                                     *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file Enums.h
 *
 * \brief This file holds all the user defined enumerations that will be used
 *        across the project of the allocated memory
 *
 * \version Initial Version
 * \version 05.08.2011, Koechling, Christian (Bosch), Interface VD_DeviceManager changed
 * \version 11.11.2011, Koechling, Christian (Bosch), cleanup, state machine
 *          cares for several devs at a hub
 * \version 21.11.2011, Koechling, Christian (Bosch), cleanup includes and
 *          intermediate defines which marked bugfixes
 * \version 17.01.2012, Koechling, Christian (Bosch), adjusted
 *          HUB_SEARCHRESULT_Type used for DIAG
 * \version 18.01.2012, Sunder Negi (Montavista), add helper function to detect
 *          if HUB is connected
 * \version 07.03.2012, Koechling, Christian (Bosch), cleanup enums for
 *          mechanism to observe permanent overcurrent
 * \version 07.03.2012, Koechling, Christian (Bosch), merged MV-code Feb27 -
 *          contains issue for startup synchronisation
 * \version 02.06.2012, Koechling, Christian (Bosch), added mechanisms to use
 *          central device list manager, improved critical voltage behavior
 * \version 27.07.2012, Negi, Sunder (MontaVista), Added DTY_HUB as device type
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 24.08.2012, Negi, Sunder (MontaVista), cleanup HANLDE_GPT
 * \version 27.08.2012, Koechling, Christian (Bosch), trigger FillStorage fct. by PDT-diag
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

#ifndef __ENUMS_H__
#define __ENUMS_H__

#include "Config.h"


// to have one define for HUB_NUM_t and tenDevManagerUSBHost.
#define USB_CONNECTOR_UNDEFINED 0x00
#define USB_CONNECTOR_1   0x01
#define USB_CONNECTOR_2   0x02
#define USB_CONNECTOR_3   0x03//TODO check the value for the third USB connecter.
#define USB_CONNECTOR_4   0x04//TODO check the value for the Fourth USB connecter.

#define STRING_LENGTH 20
#define STRING_MAXINDEX (STRING_LENGTH-1)

#define U64TIMESTAMP_NOTSET 0


/*!
 * \class CGlobalEnumerations
 *
 * \brief This class contains user defined enumerations for device type
 */
class CGlobalEnumerations
{
public:

    /*!
         * \enum DEVICE_TYPE_E
         *
         * \brief Different device types e.g. if a USB-stick is connected
         *        DTY_USB is used DTY_IPOD for the corresponding apple device
         * !!! if you extend this array - please be aware to extend corresponding array in
         *     CDeviceProcessorHistory::TraceHistory ()!!!
         * ... and check the FI definitions of .../midw_fi/midw_devicemanager_fi/midw_devicemanager_fi.pdf!!!
         */
    typedef enum DEVICE_TYPE_E
    {
        DTY_UNKNOWN        = 0,
        DTY_USB            = 1,
        DTY_SD             = 2, //@HUB
        DTY_IPOD           = 3,
        DTY_IPHONE         = 4,
        DTY_MTP            = 5,
        DTY_BLUETOOTH      = 6,
        DTY_MSZUNE         = 7,
        DTY_IPAD           = 8,
        DTY_HUB            = 9,
        DTY_NOT_SUPPORTED  = 10,
        DTY_CDROM          = 11,
        DTY_SD_INTERNAL    = 12,
        DTY_DIGIAUX        = 13,
        DTY_BTA            = 14,
        DTY_MISCELLANEOUS  = 15,
        DTY_CDDA           = 16,//Roadmap 15002 CDaudio Detection
        DTY_DVD_DRIVE      = 17,//Roadmap 1615 Internal DVD Drive - connected through usb
        DTY_LASTELEM       = 18
    }DEVICE_TYPE_Type;

    /*!
         * \enum REMOVED_DEVICE_TYPE_E
         *
         * \brief Type of the removed device
         */
    typedef enum REMOVED_DEVICE_TYPE_E
    {
        SINGLE = 0,
        HUB1,
        HUB2,
        HUB3
#ifdef USE_FOURTH_USB_CONNECTOR
        ,HUB4
#endif//USE_FOURTH_USB_CONNECTOR
    }REMOVED_DEVICE_TYPE_Type;

    /*!
         * \enum HUB_ENABLED_TYPE_E
         *
         * \brief Used in UdevManager to store if HUB has been found or not
         *        connected to on of the usb connectors
         */
    typedef enum HUB_ENABLED_TYPE_E
    {
        HUB_FOUND_NO           = 0x00,
        HUB_FOUND_YES          = 0x01
    }HUB_SEARCHRESULT_Type;

    /*!
         * \enum HUB_NUM_E
         *
         * \brief Similar to REMOVED_DEVICE_TYPE_E. Can be removed later.
         *
         * @todo: cleanup possible - could be removed
         */
    typedef enum HUB_NUM_E
    {
        HUB_UNDEFINED = USB_CONNECTOR_UNDEFINED,
        HUB_1 = USB_CONNECTOR_1,
        HUB_2 = USB_CONNECTOR_2,
        HUB_3 = USB_CONNECTOR_3
#ifdef USE_FOURTH_USB_CONNECTOR
        ,HUB_4 = USB_CONNECTOR_4
#endif//USE_FOURTH_USB_CONNECTOR
    } HUB_NUM_t;

    typedef enum DeviceRecognitionStatusType
    {
        DEVICE_INVALID   = -1,
        DEVICE_SEEN      = 0,
        DEVICE_CONNECTED = 1,
        DEVICE_DISABLED  = 2
    } eDeviceRecognitionStatusType_t;


    /*!
         * \enum NOTIFICATION_E
         *
         * \brief used to differentiate if notified by udev or /tmp/.autmount - mechanism
         */
    typedef enum NOTIFICATION_E
    {
        enInotify     = 0,  ///< /tmp/.automount mechanism of platform
        enInMscDev    = 1,  ///< /tmp/.mountdb mechanism of platform
        enInUnsupDev  = 2,  ///< /tmp/.unsupported mechanism of platform
        enInAppleDev  = 3,  ///< /tmp/.appledev mechanism of platform
        enInMtpDev    = 4,  ///< /tmp/.mtpdev mechanism of platform
        enUnotify     = 5,  ///< notification mechanism of UDEV
        enNotifyAll   = 6   // add new elements before enNotifyAll
    } NOTIFICATION_Type;


    /*!
          * \enum DETECTION_E
          *
          * \brief used with IDeviceRecognizer
          */
     typedef enum DETECTION_E
     {
        enMTPDevice   = 0,
        enAppleDev    = 1,
        enBlockDevice = 2
     } DETECTION_Type;

    /*!
         * \enum  PartitionTableType
         *
         * \brief Used in context of reading the partition table
         */
    typedef enum PartitionTableType {
        PT_MSDOS,    ///< IBM based partition table
        PT_GPT,      ///< GPT based partition table
        PT_GUID,     ///< GUID based partition table
        PT_UNKNOWN   ///< unknown partition table
    } PT_Type_e;

    /*!
         * \enum KeyFile_Type_e
         *
         * \brief Used in context of reading the partition table
         */
    typedef enum KeyFileType {
        KEY_LOGDGBINFO       = 0,  ///< IBM based partition table
        Key_LSIM_SDCARD_FAKE = 1,
        KEY_LAST_VAL         = 2
    } KeyFile_Type_e;

    /*!
         * \enum Diag_Type_e
         *
         * \brief diagnosis type
         */
    typedef enum DiagType {
        DIAG_DTC_CYCLIC   = 0,      ///< DTC: for customer diagnosis to write to error memory
        DIAG_PDT_ONDEMAND = 1,      ///< PDT: production diagnosis test used to check if HW has produced ok
        DIAG_LAST_VAL     = 2
    } Diag_Type_e;


#ifdef USE_TOTAL_USED_FREE_SIZE

    /*!
      * \enum FILESYSTEM_TYPE_E
      *
      * \brief different filesystem types are supported by the system these are notified by the FI
      * ... and check the FI definitions of .../midw_fi/midw_devicemanager_fi/midw_devicemanager_fi.pdf!!!
      */
    typedef enum FILESYSTEM_TYPE_E
    {
        FSTY_UNKNOWN        = 0,
        FSTY_FAT            = 1,
        FSTY_NTFS           = 2, //@HUB
        FSTY_EXT3_4         = 3,
        FSTY_HFS            = 4,
        FSTY_EXFAT          = 5,
        FSTY_ISO9660        = 6,
        FSTY_UDF            = 7,
        FSTY_LASTELEM       = 8
    }FILESYSTEM_TYPE_Type;
#endif


    typedef struct hub_details
    {
        GENSTRING sSysName;
        CGlobalEnumerations::HUB_NUM_t eHubNo;
        CGlobalEnumerations::HUB_SEARCHRESULT_Type eHUBFound;
        unsigned int uiPortCount;
        bool         bHasCardReader;
    } HubDetails_t;

    /*!
         * \enum tenDeviceClass-
         * Device Class of device connected as per libusb
         */
    typedef enum
    {
        DEVICE_CLASS_HID              = 0x00, //Human Interface Device class.
        DEVICE_CLASS_MTP              = 0x01, //MTP storage class
        DEVICE_CLASS_MASS_STORAGE     = 0x02, // Mass storage class.
        DEVICE_CLASS_HUB              = 0x03  // Hub class
    }tenDeviceClass;

    typedef enum
    {
        INTERFACE_DEVICE_CLASS_OTHERS           = 0x00, //all other device classes.
        INTERFACE_DEVICE_CLASS_MTP              = 0x01, //255 (Android Accessory Interface)
        INTERFACE_DEVICE_CLASS_AUDIO            = 0x02  //MIDI function
    }tenInterfaceDeviceClass;

};

/*!
 * \namespace devlistmgr
 *
 * \brief enums and structs provided in context of UdevManager
 */
//*********************************
namespace devlistmgr
//---------------------------------
{
/*!
     * \enum DEVICELIST_TYPE
     *
     * \brief central device list manager has different lists  this enum is used
     *        to differentiate between these lists
     *
     * \todo: cleanup possible: investigate to reduce the number of lists to a single one
     */
typedef enum DEVICELIST_TYPE
{
    DEVLIST_UNDEFINED          = 0x00,
    DEVLIST_UDEVMGR            = 0x01,
    DEVLIST_STATETABLE         = 0x02,
    DEVLIST_DEVICEMGRSERVICE   = 0x03,
    DEVLIST_ALL                = 0x04
}DEVICELIST_Type;
}


/*!
 * \namespace statetbl
 *
 * \brief enums and structs provided in context of StateTable
 */
//*********************************
namespace udevmgr
//-------------------------------------
{
/*!
     * \enum DEVICE_CONNECTSTATUS_E
     *
     * \brief The state machine uses this enum to calculate its output. Apart
     *        from the entries named with 'APPLY' these
     */

typedef struct
{
    tBool bMount;
    tChar cDeviceName[80];
    tChar cMountpoint[80];
} trPipeParmsSendMountDev;

}


/*!
 * \namespace statetbl
 *
 * \brief enums and structs provided in context of StateTable
 */
//*********************************
namespace statetbl
//-------------------------------------
{
/*!
     * \enum DEVICE_CONNECTSTATUS_E
     *
     * \brief The state machine uses this enum to calculate its output. Apart
     *        from the entries named with 'APPLY' these
     */
typedef enum DEVICE_CONNECTSTATUS_E   //Roadmap13035_Overtemp  revname USB_DEV to DEVICE_
{
    /*external notifieable state for clients - see FI of vd_device  manager*/
    USB_DEV_UNDEFINED                     = 0x00,
    USB_DEV_WARNING                       = 0x01, //@todo obsolete case fully remove this
    USB_DEV_CONNECTED                     = 0x02,
    USB_DEV_REMOVED_BY_USR                = 0x03,
    USB_DEV_UNAVAIL_BAT_LOWVOLT           = 0x04,
    USB_DEV_UNAVAIL_HW_MALFUNC            = 0x05, //temporarily not available
    USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT  = 0x06, //if temporarily not available lasts to long
    /*-->used internally to care if e.g. second device bound to HUB connected to wither USB1-1 or USB1-2 s is managed*/
    USB_DEV_INTERNAL_APPLY_CONNECT        = 0x07,
    USB_DEV_INTERNAL_APPLY_REMOVED_BY_USR = 0x08,
    /*<--end internall used*/
    USB_DEV_UNAVAIL_CDDRIVE_TEMPERATURE   = 0x09 //Roadmap13035_Overtemp - rename function to SetConnectStatus
#ifdef  MSD_BIGGEST_PARTITON_FEATURE
    ,USB_DEV_INTERNAL_APPLY_CONNECT_SUPPRESSED = 0x0A
#endif
#ifdef USE_HIGHRISK4USB_HWMALFUNCTION
    ,USB_DEV_INTERNAL_APPLY_CONNECT_AFTER_MALFUNC          = 0x0B
#endif
}DEVICE_CONNECTSTATUS_Type;

//------------------------------
//list of signals
//------------------------------

/*!
     * \enum tenDevManagerUSBHost
     *
     * The number of usb connectors is specific to the HW here we offer to use
     * USB1 and USB2 in other projects these might be more. eUSBndef is used if
     * the connector is not yet recognized. eUSBAll is used e.g. with ttfis
     * command 'VD_DVM_GET_CONNECTED_DEVICES' to get the whole list of devices
     * nevertheless if these a are conencted to eUSB1 or eUSB2
     */
typedef enum
{
    eUSBUndef = 0x00,
    eUSB1     = USB_CONNECTOR_1,
    eUSB2 = USB_CONNECTOR_2,
    eUSB3 = USB_CONNECTOR_3
#ifdef USE_FOURTH_USB_CONNECTOR
    ,eUSB4 = USB_CONNECTOR_4,
    eUSBAll = 0x05
#else
    ,eUSBAll = 0x04
#endif//USE_FOURTH_USB_CONNECTOR
}tenDevManagerUSBHost;

#define ARRAYSIZEFORUSB    (tInt)(eUSBAll)
#define NUMOFUSBCONNECTORS (tInt)(ARRAYSIZEFORUSB -1)

/*!
     * \enum tenDevManagerSignals
     *
     * The state table has different inputs these enums help to differentiate
     * the inputs. The table has to be adjusted to the HW which is used. E.g.
     * in GM we have 2 connectors and for each those signals are possible.
     */
typedef enum
{
    U60_drop                     = 0x00,
    U50_USB_Reset                = 0x01,
    OC1                          = 0x02,
    OC2                          = 0x03,
    OC3                          = 0x04,
#ifdef USE_FOURTH_USB_CONNECTOR
    OC4                          = 0x05,
#endif//USE_FOURTH_USB_CONNECTOR
    PPON1                        = 0x06,
    PPON2                        = 0x07,
    PPON3                        = 0x08,
#ifdef USE_FOURTH_USB_CONNECTOR
    PPON4                        = 0x09,
#endif//USE_FOURTH_USB_CONNECTOR
    mountedUSB1_prm              = 0x0A,
    mountedUSB2_prm              = 0x0B,
    mountedUSB3_prm              = 0x0C,
#ifdef USE_FOURTH_USB_CONNECTOR
    mountedUSB4_prm              = 0x0D,
#endif//USE_FOURTH_USB_CONNECTOR
    mountedUSB1_udev             = 0x0E,
    mountedUSB2_udev             = 0x0F,
    mountedUSB3_udev             = 0x10
#ifdef USE_FOURTH_USB_CONNECTOR
    ,mountedUSB4_udev             = 0x11
#endif//USE_FOURTH_USB_CONNECTOR
}tenDevManagerSignals;

/*!
     * \struct trPortStates
     *
     * holds information about the USB connectors. It is filled in the state table. will be send to diagnosis
     *
     */
typedef struct{
    tU8   u8USBPortNo; //if set to eUSBUndef other elements are marked herby invalid as well
    tBool bOverCurrent;//Deprecated
    tBool bUndervoltage;
    tBool bHubConnected;
    tBool bOpenCircuit;
    tBool bElectricalFailure;
    tBool bUSBPowerON;
    tBool bUSBPortConfiguredUsed;
}trPortStates;


/*!
     * the state table has different inputs these enums help to differentiate
     * the inputs these enums are used afterwards. The code has been implemented
     * in a generic way to ease further extension. Hence these enums will be used
     * with tenDevManagerUSBHost - i.e. it is a more generic approach for
     * code parts executed after usage of state table
     */
enum
{
    USBNo                       = 0x01,
    U60_drop_                   = 0x02,
    U50_USB_Reset_              = 0x03,
    OC                          = 0x04,
    PPON                        = 0x05,
    mountedUSB                  = 0x06,
    mountedUSBref               = 0x07
};

/*!
     * \enum tenSignalValue
     *
     * \brief values of signals - e.g. for those in 'tenDevManagerSignals' -
     *        see above in code
     */
typedef enum
{
    SIG_UNDEF   = 0x00,
    SIG_FALSE   = 0x01,
    SIG_TRUE    = 0x02,
    SIG_UNUSED  = 0xAA,
    SIG_ANY     = 0xFF
}tenSignalValue;

/*!
     * \struct voltageFailure_HWSignalsAll_t
     *
     * \brief 'voltageFailure_Result_t' plus 'voltageFailure_HWSignalsAll_t'
     *        describe an input row of the state
     */
typedef struct {
    tenDevManagerUSBHost usb;
    tenSignalValue       U60_drop;
    tenSignalValue       U50_USB_Reset;
    tenSignalValue       OC;
    tenSignalValue       PPON;
    tenSignalValue       mountedUSB;      ////@todo is obsolete
    tenSignalValue       mountedUSBref;   //send by udev (reference)
}voltageFailure_HWSignalsAll_t;


#define NOTLOCKED_UDEV 0

/*!
 * \struct voltageFailure_Result_t
 *
 * 'voltageFailure_Result_t' plus 'voltageFailure_HWSignalsAll_t' describe an
 * input row of the state
 *
 * \todo cleanup: directly use DEVICE_CONNECTSTATUS_Type instead of voltageFailure_Result_t
 */
typedef struct {
    DEVICE_CONNECTSTATUS_Type                      resultUSB;
}voltageFailure_Result_t;

/*!
 * \struct voltageFailure_IFOutput_t
 *
 * 'voltageFailure_Result_t' plus 'voltageFailure_HWSignalsAll_t' describe an
 * input row of the state
 */
typedef struct {
    voltageFailure_Result_t        result4Client;
    voltageFailure_HWSignalsAll_t  signalsHW;
}voltageFailure_IFOutput_t;


#define MAXDEMANDED /*USB_DEV_UNAVAIL_HW_MALFUNC_PERMANENT=*/(6 + 1)

/*!
     * \enum tenMsgList_index
     */
typedef enum
{
    eNo               = 0x00,
    eWARN             = 0x01,
    eCONNECT          = 0x02,
    eREMOVE           = 0x03,
    eLOWVOLT          = 0x04,
    eLOWVOLT_CONNECT  = 0x05,
    eMALFUNC          = 0x06,
    eMALFUNC_CONNECT  = 0x07
}tenMsgList_index;

/*!
     * \struct StateTableElemInformClients_t
     */
typedef struct {
    DEVICE_CONNECTSTATUS_Type      currentState;
    tenMsgList_index               demandedState[MAXDEMANDED];
}StateTableElemInformClients_t;

/*!
     * \struct MsgListElemInformClients_t
     */
typedef struct {
    tU8        u8NumOfMsgs;
    tInt       msgList[3];
}MsgListElemInformClients_t;

/*!
     * \struct util_StructQInfo
     */
typedef struct
{
    char sMsgQName[20];
    tU16 u16SizeOfElement;
    tU32 u32MaxElementsInQueue;
} util_StructQInfo;

/*!
     * \enum tenDevice_UnsupportedReason -
     * USB Compliance - UnsupportedReason for device connected
     */
typedef enum
{
    OK                                                     = 0x00,
    NotMountedbyAutomounter                               = 0x01,  //Note: Device might have different partitions
    DisabledInConfiguration                                = 0x02,
    NotSupportedByDVM                                        = 0x03,
    HubButNotCustomerHub                                   = 0x04,
    NotMountedbyAutomounter_UNSUPPORTED_FILESYSTEM       = 0x05,  //NotMountedAutomounter_... information given for partitions by automounter
    NotMountedbyAutomounter_UNSUPPORTED_NO_FILESYSTEM    = 0x06,
    NotMountedbyAutomounter_UNSUPPORTED_AUDIO_CD         = 0x07,
    NotMountedbyAutomounter_UNSUPPORTED_AUDIO_MIXED_MODE = 0x08,
    NotMountedbyAutomounter_UNSUPPORTED_BLACKLISTED      = 0x09,
    PossiblySupportedByOtherClient                       = 0xA, // Device can be used by other applications like SPI
    Multipart_UnSupportedPartition                       = 0xB, // Device might have other valid partitions
    AppleDeviceNotConnectedOnCustomerHub                 = 0xC // Apple device is not connected on customer hub

}tenDevice_UnsupportedReason;


/*!
     * \enum tenMsgTyp
     */
typedef enum
{
    enMountUdev                              = 0x01,
    enTimerExp                               = 0x02,
    enHWSignals                              = 0x03,
    enVoltWarning                            = 0x04, //Roadmap 13024_Undervoltage_2
    enTimout                                 = 0x05,
    enShutdown                               = 0x06,
    enTestSend2Client                        = 0x07,
    enTriggerTimer                           = 0x08,
    enCardSignals                            = 0x0A, //bound to eConf_ENABLE_USE_PRM_SDCARDSIGNALS
    enCardOcStatus                           = 0x0B,
    enDevRecAfterClv                         = 0x0C, //bound to USE_SENT_ALL_CHANGES
    enCDDriveOverTemp                        = 0x0D, //Roadmap 13035_OverTemperature
    enDiagMountCDRom                         = 0x0E, //Roadmap 14002 MountCDROM (MixedModeDiag)
    enCDDANotified                           = 0x0F, //Roadmap 15002 CDaudio Detection
    enOptDiscSlotStateNotified               = 0x10, //Roadmap 15002 CDaudio Detection
    enMalfunctionUsbConnector                = 0x11,
    enMalfunctionOpticalDisc                 = 0x12,
    enMalfunctionSDCardInternal              = 0x13,
    enOptDiscCDInfoNotified                  = 0x14,  //CD Info
    enPortPowerInfoNotified                  = 0x15,  //Port Info

    enUndefined                = 0xFF
}tenMsgTyp;

// DeviceSpeed in USB
typedef enum{
    UNKNOWN = 0UL,
    LOW_SPEED,
    FULL_SPEED,
    HI_SPEED,
    SUPER_SPEED
}tenDeviceSpeedUSB;

// Retrial Status for Multipartition device
typedef enum{
    RETRIAL_NOTDONE = 0UL,
    RETRIAL_NEEDED,
    RETRIAL_DONE
}tenMultiPartRetrialStatus;

//@todo think about dynamic allocation of memory
#define LEN_cSerialID        256
#define LEN_cUSBPort         10
#define LEN_cDeviceVersion   80
#define LEN_cDeviceName      80
#define LEN_cAccessoryName   256
#define LEN_cMountPoint      100
#define LEN_cBlocknode       100
#define LEN_cManufacturer    80
#define LEN_cDevPath         256
#define LEN_cDevNode         256

/*!
     * \struct structNotifyClients
     *
     * similar to CDevice but used with state table only and reduced to needs
     * of what will be send to clients
     */
typedef struct
{
    tChar                                 cSerialID[LEN_cSerialID];
    tChar                                 cUSBPort[LEN_cUSBPort];
    CGlobalEnumerations::DEVICE_TYPE_Type eDeviceType;
    tChar                                 cDeviceVersion[LEN_cDeviceVersion];
    tChar                                 cDeviceName[LEN_cDeviceName];
    tChar                                 cAccessoryName[LEN_cAccessoryName];
    tChar                                 cMountPoint[LEN_cMountPoint];
    tChar                                 cDevPath[LEN_cDevPath]; //DeviceSysPath  CRQ     CMG3G-12660
    tChar                                 cDevNode[LEN_cDevNode];
    DEVICE_CONNECTSTATUS_Type             eConnectStatus;
    uint16_t                              iVendorID;
    uint16_t                              iProductID;
    tU64                                  u64Size;
    tChar                                 cManufacturer[LEN_cManufacturer];
#ifdef USE_FILESYSTEMTYPE_ALIAS_FSTY
    CGlobalEnumerations::FILESYSTEM_TYPE_Type    eFSType;
#endif
#ifdef USE_TOTAL_USED_FREE_SIZE
    tU64                                  u64TotalSize_KB;
    tU64                                  u64UsedSize_KB;
    tU64                                  u64FreeSize_KB;
#endif
#ifdef USE_PARITIONNUMBER
    tU8                                   u8ValidPartitionNo;
#endif
#ifdef  MSD_BIGGEST_PARTITON_FEATURE
    tU8                                   u8TotalValidPartitons;//bound to eConf_ENABLE_MSD_WAIT_FOR_PARTITIONS_ALL
    tChar                                 cAccessoryName2[LEN_cSerialID];//this will be used to identify all the partitions of multipartition  stick belongs to same device
#endif

    tS32                                  s32MalfunctionElapsedTime_sec;//Store the elapsed time of malfunction. This will be used to monitor the hw malfunction permanent
#ifdef TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION
    //this is stored to compare with the malfunction occurred at the usb port cUSBPort
    //if u64MalfunctionTimestamp > u64ConnectTimestamp and > u64RemoveTimestamp but falls with in eCONF_USBCOMPARETIME_MALFUNCTION_OR_REMOVE then we have temporary not available
    tU64                                  u64TimeStamp_ApplyForConnect; //timestamp when a connect of device has happened
    tU64                                  u64TimeStamp_ApplyForRemove; //timstamp when a remove of device has happened
    tU64                                  u64TimeStamp_Malfunction;    //due to HW-signals this makrs a real malfunction
#ifdef USE_HIGHRISK4USB_HWMALFUNCTION
    tU64                                  u64TimeStamp_HighRiskForMalfunction;   //e.g. due to crit voltage a malfunction of USB could happen but might not happen that's why this is called 'HighRiskForMalfunction'
#endif
#endif//TEMPORARY_UNAVAILABLE_INTERMEDIATE_SOLUTION

    tenDevManagerUSBHost                  eUSB;      /*Connector - same as cUSBPort but as enum*/
    tenDeviceSpeedUSB                     eDeviceSpeedUSB; /* Enum - Device Speed of USB*/
    tenDevice_UnsupportedReason           eDeviceUnsupportedReason;      /* USB Compliance - UnsupportedReason for device connected*/
    tenMultiPartRetrialStatus             eMultiPartRetrialStatus;  /*Enum -Retrial Status for Multipartition device */
    tU64                                  u64TimeStamp_ConnectTimeStampForRetry; // This will be used for retrying the device status after a timeout
} structNotifyClients;



typedef enum
{
   eIsMalfunction             = 0,
   eIsHighRiskForMalfunction =  1
}tenTypeOFMalfunction;


/*!
     * \struct structNotifyMalfunction
     *
     * holds the information about malfunction of usbconnector
     */
typedef struct
{
    tBool                      bIsMalfunction = false;
    tenDevManagerUSBHost       eUSB = eUSBUndef;     /*Connector - same as cUSBPort but as enum*/
    tenTypeOFMalfunction       eTypeOFMalfunction = eIsMalfunction;

}structNotifyMalfunction;

typedef enum
{
    MALFUNCTION_NO            = -1,
    MALFUNCTION_YES           =  0,
    MALFUNCTIONUPDATEINTERVAL =  STATETABLE_MESSAGE_QUEUE_TIMEOUT_SEC
}MalfunctionInfo;







#define UTIL_MAX_CONTENT_LEN  (sizeof(tenDevManagerUSBHost)+sizeof(tenSignalValue)+sizeof(structNotifyClients))
//largest entity to be send in a message also depends on
//sizeof(UsbPortState) and
// sizeof(tenDevManagerUSBHost/*eUsb*/)
// sizeof(tenDevManagerUSBHost/*eU60_drop*/)

/*!
     * \struct util_StructMsg
     *
     * inputs to the state machine will be send in a message like this. In
     * class 'StateTable' function 'StartMonitorLooseBlocking' listens to those
     * messages and triggers corresponding functionalities
     */
typedef struct
{
    tenMsgTyp eMsgType;
    tU16 u16LenCont;
    tU8 pu8Content[UTIL_MAX_CONTENT_LEN]; //size must fit into unsigned short
}util_StructMsg;


#define UTIL_LENOFQMSG  sizeof(util_StructMsg)
//#define UTIL_USED_SIZEOF_WM(poWm) (sizeof(util_StructMsg) + (poWm)->u16LenCont - _UTIL_MAX_CONTENT_LEN )
#define UTIL_SIZEOFWIREMSG sizeof(util_StructMsg)
}


typedef enum
{
    eCard0     = 0x00,       // "/dev/card"  in Gen3 /media/....
    eCardAll   = 0x01        // insert additional SD Cards before eCardAll
    // Use eCardAll to access "/dev/cryptcard" in PRM. "/dev/crytcard" and "/dev/card" are for the same SD card slot. CF made two OSAL independet devices.
} tenSlot;

#define ARRAYSIZEFORCARD (tInt)(eCardAll)



/*!
 * \namespace servicedvm
 *
 * \brief enums and structs provided in context of VD_DeviceManagerService
 */

namespace servicedvm
{

/*!
      * \struct ElemOfProperty_Type
      *
      * used to map strings from CDevice-property to FI
      */
typedef enum
{
    eSerialID        = 1,
    ecUSBPort        = 2,
    ecDeviceVersion  = 3,
    eDeviceName      = 4,
    eAccessoryName   = 5,
    eMountPoint      = 6,
    eManufacturer    = 7,
    eAccessoryName2  = 8,
    eDevicePath      = 9,
    eDeviceNode      = 10

}tenElemOfProperty;

/*!
      * \struct trReceiverInfoContainer
      *
      * used to to store data to create return cca message with methods
      */
typedef struct
{
    tBool bValid;            //marks if valid values are set
    tU16  u16GetSourceAppID; // Target AppID
    tU16  u16GetRegisterID;  // RegisterID
    tU16  u16GetCmdCounter;  // CmdCounter,
    tU16  u16GetFunctionID;  // Function ID
} trReceiverInfo;

}

/*!
 * \namespace prmmgr
 *
 * \brief enums and structs provided in context of PrmManager
 */
namespace dvmDiaglog
{

#define NUM_OF_SDC_DTC_ERRORS       2
//#define NUM_OF_USB_DTC_ERRORS       9 //USB3 //USE_THIRD_USB_CONNECTOR - see config.h
#define NUM_OF_USB_DTC_ERRORS         33 //enITC_LASTVALUE - check how to get this values dependend to make is maintenenable
#define NUM_OF_DTC_ERRORS           (NUM_OF_USB_DTC_ERRORS + NUM_OF_SDC_DTC_ERRORS)



/*!
     * \struct diaglog_m_rDtc_error
     *
     * \brief Stores DTC test result.
     */

struct diaglog_rDtc_error
{
    tU16 u16ErrorCode; ///< DTC error code for USB Circuit 1 and 2
    tS8  s8DtcResult;  ///< DTC test result for USB Circuit 1 and 2
};


typedef enum
{
    /*USB1*/
    enITC_USB1_SIGNAL_OVERCURRENT           = 0,
    enITC_USB1_HUB_NOT_DETECTED             = 1,   ///USB HUB 1 Open Circuit
    enITC_USB1_HUB_NUMBER_OF_PORT_MISMATCH  = 2,   ///USB HUB 1 Above Maximum Threshold
    /*USB2*/
    enITC_USB2_SIGNAL_OVERCURRENT           = 3,
    enITC_USB2_HUB_NOT_DETECTED             = 4,   ///USB HUB 2 Open Circuit
    enITC_USB2_HUB_NUMBER_OF_PORT_MISMATCH  = 5,   ///USB HUB 2 Above Maximum Threshold
    /*SDCard*/
    enITC_SDC1_SIGNAL_OVERCURRENT           = 6,

    /*USB3*/ //extension for PSA  //Note: added this at the end since dependency in diaglog service looks complex
    enITC_USB3_SIGNAL_OVERCURRENT           = 7,
    enITC_USB3_HUB_NOT_DETECTED             = 8,   ///USB HUB 3 Open Circuit
    enITC_USB3_HUB_NUMBER_OF_PORT_MISMATCH  = 9,   ///USB HUB 3 Above Maximum Threshold

    enITC_USB4_SIGNAL_OVERCURRENT          = 10,
    enITC_USB4_HUB_NOT_DETECTED            = 11,   ///USB HUB 4 FOR CHERY Open Circuit
    enITC_USB4_HUB_NUMBER_OF_PORT_MISMATCH = 12,   ///USB HUB 4 FOR CHERY Above Maximum Threshold
    enITC_USB1_POWER_SUPPLY_FAULT           = 13,
    enITC_USB2_POWER_SUPPLY_FAULT           = 14,
    enITC_USB3_POWER_SUPPLY_FAULT           = 15,
    enITC_USB4_POWER_SUPPLY_FAULT           = 16,  ///USB HUB 4 FOR CHERY
    enITC_USB1_OPEN                         = 17,
    enITC_USB2_OPEN                         = 18,
    enITC_USB3_OPEN                         = 19,
    enITC_USB4_OPEN                         = 20,  ///USB HUB 4 FOR CHERY


    //used with BTA - Telematics Box in PSARCC  first
    enITC_USB1_LINK_DISABLED                = 21,
    enITC_USB2_LINK_DISABLED                = 22,
    enITC_USB3_LINK_DISABLED                = 23,
    enITC_USB4_LINK_DISABLED                = 24,  ///USB HUB 4 FOR CHERY
    enITC_USB1_SPEED_REDUCED                = 25,
    enITC_USB2_SPEED_REDUCED                = 26,
    enITC_USB3_SPEED_REDUCED                = 27,
    enITC_USB4_SPEED_REDUCED                = 28,  ///USB HUB 4 FOR CHERY
    enITC_USB1_DATA_LINE_FAULT              = 29,
    enITC_USB2_DATA_LINE_FAULT              = 30,
    enITC_USB3_DATA_LINE_FAULT              = 31,
    enITC_USB4_DATA_LINE_FAULT              = 32,  ///USB HUB 4 FOR CHERY

    //--------------------------------------------
    enITC_LASTVALUE                         = 33
}tenIndexDiagEvt;





typedef enum
{
    enUSB_OVERCURRENT                 = 0,
    enUSB_HUB_NOT_DETECTED            = 1,
    enUSB_HUB_NUMBER_OF_PORT_MISMATCH = 2,
#ifdef USE_PSA_DIAGLOG_ADDITIONAL_ITCS_FOR_MALFUNCTION
    enUSB_POWER_SUPPLY_FAULT          = 3,
    enUSB_OPEN                        = 4, //first usage with Telematics Box of PSA
    enUSB_LINK_DISABLED               = 5, //first usage with Telematics Box of PSA
    enUSB_SPEED_REDUCED               = 6,  //first usage with Telematics Box of PSA
    enITC_DATA_LINE_FAULT             = 7  //first usage with Telematics Box of PSA
#endif

} tenUsbTestType ;

}






/*!
 * \namespace prmmgr
 *
 * \brief enums and structs provided in context of PrmManager
 */
namespace prmmgr
{
/*!
     * \struct UsbPortState
     *
     * prm sends a callback within this callback these signals can be seen
     */
typedef struct
{
    tU8  u8PortNr;            // 1. Byte Wert = 1 oder 2
    tU8  u8OC;
    tU8  u8UV;
    tU8  u8PPON;              // 4 bytes

    tU32 u32OCStartTime;      // 8
    tU32 u32OCEndTime;        // 12
    tU32 u32UVStartTime;
    tU32 u32UVEndTime;
    tU32 u32PPONStartTime;
    tU32 u32PPONEndTime;      // 28 bytes
}UsbPortState;

typedef struct
{
    tU32 PortMask;
    tU32 ValueMask;
}usb_port_control;

/*!
     * \enum tenSignalValuePRM
     *
     * UsbPortState - signals u8OC,u8UV,u8PPON can have these states
     */
typedef enum
{
    SIGNAL_UNDEF = 0x00,
    SIGNAL_FALSE = 0x01,
    SIGNAL_TRUE  = 0x02
}tenSignalValuePRM;
}

typedef struct
{
    tenSlot eSlot;      // 1. Byte Card slot; starts with 0 for first card
    tU16    u16Type;
    tU16    u16Data;
} CardNotInfo;         // holds Notification info (media change, media state)

typedef struct
{
    tenSlot eSlot;      // 1. Byte Card slot; starts with 0 for first card
    tBool   bOcActive;
} CardNotInfoOcStatus;         // holds Notification info (media change, media state)


//#ifdef USE_DECOUPLE_CYCLICDIAGCCA
typedef struct  //this could be shifted to struc below
{
    tU8  u8Result;
    tU16 u16DTroubleCode;
}trDtcCodeResult;
//#endif


typedef struct  //this could be shifted to struc below
{
    tS8 s8DtcResult;
}trDtc_error;

typedef struct
{
    unsigned int send_nr;
    tU32         sent_time;
    trDtc_error  last_msg[NUM_OF_DTC_ERRORS]; //for analysis
}trDiagEvt;

typedef struct
{
    uint16_t iVendorID;
    uint16_t iProductID;
}trDigitalAuxinDeviceInfo;


typedef struct
{
    uint16_t iVendorID;
    uint16_t iProductID;
    tBool bIsHighSpeedExpected;
    statetbl::tenDevManagerUSBHost eUsb;
}trTelematicsBoxDeviceInfo;

typedef struct
{
    tBool b_DeviceSpeed;//to store the device speed obtained after the scanning
    statetbl::tenDevManagerUSBHost eUsb;//to store the port number in which the BTA device is connected to
    tBool bIsDevicePresent;//to store the connectio status, whether it is connected or not connected
}TelematicsBoxSpeedAndPortInfo;

typedef enum
{
    BTA_MONITOR_NOT_STARTED = -1,
    BTA_MONITOR_STARTED = 0,
    BTA_MONITOR_FINISHED  = 1
}BTAMonitorStates;

//to hold the result of CDDA Eject
typedef enum
{
    ACCEPTED,
    NOTACCEPTED,
    DEVICENOTREADY
}tOpticalDiscEjectMethodResult;

//to store the optical disc types
typedef enum
{
    MMGR_INIT = 0,
    MMGR_NO_MEDIA,
    MMGR_INSERTION,
    MMGR_INCORRECT,
    MMGR_AUDIO,
    MMGR_DATA,
    MMGR_EJECTING = 17,
    MMGR_MEDIA_IN_SLOT
}CDDATypes;



//to store the optical disc media state
typedef enum
{
    MMGR_MEDIA_ANY,
    MMGR_MEDIA_NOT_READY,
    MMGR_MEDIA_READY
}CDDAMediaState;

//to store the optical disc insert state
typedef enum
{
    MMGR_INSERT_ANY,
    MMGR_INSERT_BEFOREON,
    MMGR_INSERT_AFTERON,
    MMGR_INSERT_AUTOMATIC

}CDDAInsertState;


//Struct contains all of the cd information
typedef struct {

    CDDATypes        e8CDDATypes;
    CDDAInsertState  e8CDDAInsertState;
    CDDAMediaState   e8CDDAMediaState;
}CDDA_InfoAll_t;


//to hold the cd slot state
typedef enum
{
    INITIALISING                = 1,
    INITIALISED                 = 2,
    LASTMODE_EMTPY              = 3,
    LASTMODE_INSERTED_CDDA      = 4,
    LASTMODE_INSERTED_CDROM     = 5,
    LASTMODE_INSERTED_CDERROR   = 6,
    INSERTING                   = 7,
    INSERTED_CDAUDIO            = 8,
    INSERTED_CDROM              = 9,
    INSERTED_CDERROR            = 10,
    INSERTED_AUTOMATIC_CDAUDIO  = 11,
    INSERTED_AUTOMATIC_CDROM    = 12,
    INSERTED_AUTOMATIC_CDERROR  = 13,
    EJECTING                    = 14,
    EJECTED_READY_TO_REMOVE     = 15,
    EJECTED_EMPTY               = 16,
    UNDEFINED_STATE = 0xFF
}tDMOpticalDiscSlotState;

typedef enum{
    DM_OPTDISC_CMD_EJECT                       = 0UL, //used to eject optical disc
    DM_OPTDISC_CMD_INSERT                      = 1UL  //used to force insertion of optical disc
}tDMOpticalDiscEjectCmd;

/**************************************
 * CDInfo- Device State for RNAIVI
 **************************************/
typedef enum
{
    DEVICE_NOT_READY = 0L,
    DEVICE_READY
} tenOpticalDeviceState;

#define DVMGR_MAX_DRIVEVERSIONLENGTH 40
/**
 * Type declaration - context CD drive Version
 */
typedef char tOpticalDriveVersion[DVMGR_MAX_DRIVEVERSIONLENGTH];

typedef struct {
    tenOpticalDeviceState  OpticalDeviceState;
    tOpticalDriveVersion   OpticalDriveVersion;
}trOpticalDiskCDInfo;

/**************************************/

/*!
     * \struct CDDA_IFOutput_t
     *
     * 'tDMOpticalDiscSlotState' plus 'CDDA_InfoAll_t' describe an
     * input row of the state
     */
typedef struct {
    tDMOpticalDiscSlotState  result4Client;
    CDDA_InfoAll_t           CDDAinfo;
}CDDA_IFOutput_t;

typedef enum{
    DEFAULT,
    TEMPORARY,
    PERMANENT
}tenSettingsMode;



typedef struct
{
    uint16_t iVendorID;
    uint16_t iProductID;
    statetbl::tenDevManagerUSBHost eUsb;

}trExternalHubInfo;

/*RTC-816210: Customer Hubs come with internal USB devices. To avoid Devicemanager counting such devices, the vid,pid of such devices need to be maintained
 *Ex:The USB4914(Molex hub) has two internal USB devices. The Multi-Host Endpoint Reflector device is a Composite WinUSB and NCM device with Product ID 0x4910.
 *The Hub Feature Controller device which enables USB bridging function is a WinUSB device with Product ID 0x4940.
 */
struct UsbDeviceInfo
{
    uint16_t iVendorID{0};
    uint16_t iProductID{0};
};

typedef enum{
    NON_DATA_DISC = 0UL,
    DATA_DISC = 1UL
}tenDVDDiscType;

typedef enum{
    MASS_STORAGE_MODE                  = 0UL,
    BACKEND_MODE                       = 1UL,
    TRANSITION_TO_MASS_STORAGE_MODE    = 16UL,
    TRANSITION_TO_BACKEND_MODE         = 17UL,
    INVALID_MODE                       = 255UL
}tenDVDDriveMode;

typedef struct
{
    tenDVDDiscType DVDDiscType;
    tenDVDDriveMode DVDDriveMode;
}trDVDDriveInfo;

#endif // __ENUMS_H__

////////////////////////////////////////////////////////////////////////////////
// <EOF>
